# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArPyUtils
import CliModel
from CliModel import Dict, Float

MAX_POWER_BUDGET = 100000

def _powerMngString( name, consumedPower ):
   return f"{name:<23} {consumedPower:<23}"

class PowerConsumer( CliModel.Model ):
   amount = Float( help="Amount of power the device is consuming in watts" )

class PowerDomainStatus( CliModel.Model ):
   totalAvailablePower = Float( help="Total available power in watts" )
   consumedPower = Float( help="Consumed power in watts" )

class PowerManager( CliModel.Model ):
   powerConsumers = Dict( keyType=str, valueType=PowerConsumer,
                          help="Devices consuming power" )
   totalAvailablePower = Float( help="Total available power in watts" )
   powerBudget = Float( help="Budgeted power in watts" )
   consumedPower = Float( help="Total consumed power in watts" )
   powerDomains = Dict( optional=True, keyType=str, valueType=PowerDomainStatus,
                        help="Power information per domain, keyed by domain name" )

   def render( self ):
      if self.powerBudget > MAX_POWER_BUDGET:
         defaultBudgetStr = "Budget is not set, using all of {0:.1f}W of total power"
         print( defaultBudgetStr.format( self.totalAvailablePower ) )
      else:
         budgetStr = "Budget is {0:.1f}W out of {1:.1f}W of total power"
         print( budgetStr.format( self.powerBudget, self.totalAvailablePower ) )
      if self.powerDomains:
         for name in ArPyUtils.naturalsorted( self.powerDomains ):
            domain = self.powerDomains[ name ]
            domainAvailPowerStr = "Power domain {0} has {1:.1f}W of total power"
            print( domainAvailPowerStr.format( name, domain.totalAvailablePower ) )
      print( f"Active devices using up to {self.consumedPower:.1f}W" )
      if self.powerDomains is None:
         print()
      elif self.powerDomains:
         print( "System currently has split power domains\n" )
      else:
         print( "System currently does not have split power domains\n" )

      print( _powerMngString( "Device", "Consumed" ) )
      print( _powerMngString( "Name", "Power" ) )
      print( _powerMngString( "-" * 23, "-" * 23 ) )
      # pylint: disable=W1633, round-builtin # pylint: disable=bad-option-value
      for name in ArPyUtils.naturalsorted( self.powerConsumers ):
         consumer = self.powerConsumers[ name ]
         print( _powerMngString( name, str( round( consumer.amount, 1 ) ) + "W" ) )
      if self.powerDomains:
         if self.powerConsumers:
            print()
         for name in ArPyUtils.naturalsorted( self.powerDomains ):
            domain = self.powerDomains[ name ]
            print( _powerMngString( f"Power domain {name} total",
                                    f"{str( round( domain.consumedPower, 1 ) )}W" ) )
         print( _powerMngString( "System total",
                                 f"{str( round( self.consumedPower, 1 ) )}W" ) )
      else:
         print( _powerMngString( "Total",
                                 f"{str( round( self.consumedPower, 1 ) )}W" ) )

class PowerHistogram( CliModel.Model ):
   buckets = CliModel.Dict( keyType=int, valueType=int,
                            help=( "Mapping of power ranges (in percent) to "
                                   "uptime (seconds) in the range" ) )
   sampleRate = Float( help=( "The rate (in samples per second) at which "
                              "samples are collected" ) )

   def render( self ):
      sortedBuckets = sorted( self.buckets.items(), key=lambda kv: kv[ 0 ] )
      for percent, uptime in sortedBuckets:
         print( f"{percent: >3}%: {uptime}" )
