# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# Cli Plugin base mode for profile handling
#
# Provides
#   group-change with commit and abort for profile handling and profile copy
#
# In order to use it, the child mode CLI plugin class should inherit from
# ProfileConfigMode (this class) and from CliMode.ProfileConfig.ProfileConfigBaseMode
# (or a subclass).
#
# See also AleTcam/AleTcamProfileConfigMode for usage example.

import BasicCli
from CliMode.ProfileConfig import ProfileConfigBaseMode

class ProfileConfigMode( BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, profileName, srcProfileName ):
      assert isinstance( self, ProfileConfigBaseMode )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.pendingProfile = self.newProfile( profileName )
      if srcProfileName:
         self.copyProfile( self.getProfile( srcProfileName ), self.pendingProfile )
      else:
         existingProfile = self.getProfile( profileName )
         if existingProfile is not None:
            self.copyProfile( existingProfile, self.pendingProfile )

   def commit( self ):
      # self.profileName comes from CliMode.ProfileConfigBaseMode
      profileName = self.profileName # pylint: disable=no-member
      if self.pendingProfile is None:
         # Pending profile is aborted
         return
      existingProfile = self.getProfile( profileName )
      if existingProfile is None:
         newProfile = self.addProfile( profileName )
         self.copyProfile( self.pendingProfile, newProfile, update=True )
      else:
         if ( not self.identicalProfiles( existingProfile, self.pendingProfile ) and
              self.confirmUpdate() ):
            lastProfile = self.newProfile( profileName )
            self.copyProfile( existingProfile, lastProfile )
            self.copyProfile( self.pendingProfile, existingProfile, update=True )
            if not self.verify():
               self.copyProfile( lastProfile, existingProfile, update=True )

   def onExit( self ):
      self.commit()
      BasicCli.ConfigModeBase.onExit( self )
      self.pendingProfile = None

   def abort( self ):
      self.pendingProfile = None
      self.session_.gotoParentMode()

   def activeProfile( self ):
      return self.getProfile( self.profileName ) # pylint: disable=no-member

   # To implement or override
   def newProfile( self, profileName ):
      """Create a new profile instance."""
      raise NotImplementedError()

   def getProfile( self, profileName ):
      """Get profile from config. Return None if the profile is not found."""
      raise NotImplementedError()

   def addProfile( self, profileName ):
      """Add a new profile in the config, and return it."""
      raise NotImplementedError()

   def copyProfile( self, srcProfile, dstProfile, update=False ):
      """Copy srcProfile into dstProfile. Update is True when this function is called
      from the commit function."""
      raise NotImplementedError()

   def identicalProfiles( self, profile1, profile2 ):
      """Return True if profile1 and profile2 are exactly the same. Used during
      commit before copying profile1 to profile2."""
      return False

   def verify( self ):
      """Verify that the profile was applied successfully."""
      return True

   def confirmUpdate( self ):
      """Ask the user if he wants to update the profile when he is trying to modify
      an active profile"""
      return True
