# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import ArPyUtils
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import LazyMount
from CliModel import Model, Dict, Str, Float, Bool
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from TableOutput import TableFormatter, Headings, Format
import Intf.IntfRange
import ShowCommand
import Tac

hwLedConfigInit = None
hwLedConfig = None
ledConfig = None

fl = Format( justify="left", wrap=True, minWidth=1, padding=1 )
fc = Format( justify="center", wrap=True, minWidth=1, padding=1 )
fl.noPadLeftIs( True )

def ledStateToDashCase( stateName ):
   stateValueName = re.sub( '([A-Z]+|(?:[0-9]+[A-Z][a-z]+))', r'-\1',
                            stateName ).lower()
   stateValueName = re.sub( r'mbps-|bps-', r'', stateValueName )
   return stateValueName

def isUserConfigurableLed( state, token ):
   if hwLedConfigInit.palette is not None:
      return None
   return CliParser.guardNotThisPlatform

matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface LED' )
matcherIntfRange = Intf.IntfRange.IntfRangeMatcher(
      explicitIntfTypes=( EthPhyAutoIntfType, ) )
ledMatcher = CliMatcher.KeywordMatcher( 'led',
      helpdesc='Show LED Information' )

class LedState( Model ):
   name = Str( help='The name of the LED state (e.g. linkUpActivity400G, etc.)' )
   available = Bool( help='Indicate whether the state is available for the '
                          'hardware' )

   def renderLedState( self, prefix, table ):
      if self.available:
         table.newRow( prefix, ledStateToDashCase( self.name ) )

class LedStateDict( Model ):
   name = Str( help='The name of the LED' )
   state = Dict( keyType=str, valueType=LedState,
                help='All LED states; key is the name of the state' )

   def renderLedStateDict( self, table ):
      padded = False
      for _, stateValue in sorted( self.state.items() ):
         if not padded:
            padded = True
            prefix = self.name
         else:
            prefix = " " * len( self.name )
         stateValue.renderLedState( prefix, table )

class LedConfigState( Model ):
   leds = Dict( keyType=str, valueType=LedStateDict,
                help='All LEDs with their states, key is the name of the LED' )

   def render( self ):
      table = TableFormatter()
      th = Headings( ( ( "LED Name", "c" ), ( "LED State", "r" ) ) )
      th.doApplyHeaders( table )

      newline = False
      for led in ArPyUtils.naturalsorted( self.leds ):
         if newline:
            table.newRow( "", "" )
         if not newline:
            newline = True
         self.leds.get( led ).renderLedStateDict( table )
      table.formatColumns( fc, fl )
      print( table.output() )

class LightSetting( Model ):
   color = Str( help="The color of the LightSetting" )
   ledFlashRate = Float( help='Duration per flash (in seconds) of the LED.'
                              'An LED flash-rate of 0.0 means no flash' )

   def renderLightSetting( self, vertical=True, prefix=None, table=None ):
      if vertical:
         table.newRow( prefix[ 0 ], self.color.lower(), self.ledFlashRate, 0.0,
                       60.0 )
      else:
         prefix += [ self.color.lower(), self.ledFlashRate ]
         table.newRow( *prefix )

class LedCapability( Model ):
   name = Str( help='The name of the LED (e.g. Ethernet1/1, Fan1/1, etc.)' )
   lightSetting = Dict( keyType=str, valueType=LightSetting,
                        help='All LedCapability LightSettings; key is name of ' + \
                             'the color' )

   def renderLedCapability( self, table ):
      padded = False
      for ls in self.lightSetting.values():
         if not padded:
            padded = True
            prefix = self.name
         else:
            prefix = " " * len( self.name )
         ls.renderLightSetting( True, [ prefix ], table )

class LedPalette( Model ):
   ledCapability = Dict( keyType=str, valueType=LedCapability,
                         help='All LED capabilities; key is name of the LED' )

   def render( self ):
      table = TableFormatter()
      th = Headings( ( ( "LED Name", "c" ), ( "Color", "c" ),
                       ( "Flash Rate", "c" ), ( "Flash Rate Min", "c" ),
                       ( "Flash Rate Max", "c" ) ) )
      th.doApplyHeaders( table )

      newline = False
      for led in ArPyUtils.naturalsorted( self.ledCapability ):
         if newline:
            table.newRow( "", "", "", "", "" )
         else:
            newline = True
         self.ledCapability.get( led ).renderLedCapability( table )
      table.formatColumns( fc, fl, fc, fc, fc )
      print( table.output() )

class LightSettingTheme( Model ):
   name = Str( help='The name of the theme' )
   setting = Dict( keyType=str, valueType=LightSetting,
                   help='All theme settings; the key is led state' )

   def renderThemeWithLed( self, led, table ):
      padded = False
      settings = hwLedConfig.theme[ self.name ].setting
      for settingName in ArPyUtils.naturalsorted( settings ):
         if settingName in hwLedConfig.leds[ led ].state \
               and hwLedConfig.leds[ led ].state[ settingName ].available:
            if not padded:
               padded = True
               prefix = [ led, self.name, ledStateToDashCase( settingName ) ]
            else:
               prefix = [ " " * len( led ), " " * len( self.name ),
                          ledStateToDashCase( settingName ) ]
            self.setting.get( settingName ).renderLightSetting( False, prefix,
                                                                table )

   def renderTheme( self, table ):
      padded = False
      for settingName in ArPyUtils.naturalsorted( self.setting ):
         if not padded:
            padded = True
            prefix = [ self.name, ledStateToDashCase( settingName ) ]
         else:
            prefix = [ " " * len( self.name ), ledStateToDashCase( settingName ) ]
         self.setting.get( settingName ).renderLightSetting( False, prefix,
                                                             table )

class LedConfigDirTheme( Model ):
   theme = Dict( keyType=str, valueType=LightSettingTheme,
                 help='All themes created, the key is names of the themes' )
   ledsTheme = Dict( keyType=str, valueType=str,
                     help='Mapping of which LEDs have which themes' )

   def render( self ):
      table = TableFormatter()
      th = Headings( ( ( "LED Name", "c" ), ( "Current Theme", "c" ),
                       ( "Theme LED State", "c" ), ( "Color", "c" ),
                       ( "Flash Rate", "c" ) ) )
      th.doApplyHeaders( table )

      newline = False
      for led in ArPyUtils.naturalsorted( self.ledsTheme ):
         if newline:
            table.newRow( "", "", "", "", "" )
         else:
            newline = True
         self.theme.get( self.ledsTheme.get( led ) ).renderThemeWithLed( led,
                                                                         table )
      table.formatColumns( fc, fc, fl, fl, fc )
      print( table.output() )

class LightSettingThemeCollection( Model ):
   theme = Dict( keyType=str, valueType=LightSettingTheme,
                 help='All themes created, the key is names of the themes' )

   def render( self ):
      table = TableFormatter()
      th = Headings( ( ( "Theme Name", "c" ), ( "Theme LED State", "c" ),
                       ( "Color", "c" ), ( "Flash Rate", "c" ) ) )
      th.doApplyHeaders( table )

      newline = False
      for themeName in ArPyUtils.naturalsorted( self.theme ):
         if newline:
            table.newRow( "", "", "", "" )
         else:
            newline = True
         self.theme.get( themeName ).renderTheme( table )
      table.formatColumns( fc, fl, fl, fc )
      print( table.output() )

#--------------------------------------------------------------------------------
# show led palette [ interface ] [ INTFS ]
#--------------------------------------------------------------------------------
class LedPaletteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led palette [ interface ] [ INTFS ]'
   data = {
      'led' : ledMatcher,
      'palette' : CliCommand.guardedKeyword( 'palette', helpdesc='Show LED Palette',
         guard=isUserConfigurableLed ),
      'interface' : matcherInterface,
      'INTFS' : matcherIntfRange,
   }
   cliModel = LedPalette

   @staticmethod
   def handler( mode, args ):
      intfs = args.get( 'INTFS', hwLedConfig.leds )
      leds = {}

      for led in intfs:
         if led not in hwLedConfig.leds:
            continue
         if hwLedCap := hwLedConfigInit.palette.ledCapability.get( led ):
            leds[ led ] = {}
            for hwLightSetting in hwLedCap.lightSetting.values():
               lightSetting = LightSetting()
               lightSetting.color = hwLightSetting.color
               lightSetting.ledFlashRate = hwLightSetting.ledFlashRate
               leds[ led ][ lightSetting.color ] = lightSetting
            ledCap = LedCapability( lightSetting=leds[ led ] )
            ledCap.name = led
            leds[ led ] = ledCap
      return LedPalette( ledCapability=leds )

BasicCli.addShowCommandClass( LedPaletteCmd )

#--------------------------------------------------------------------------------
# show led state [ interface ] [ INTFS ]
#--------------------------------------------------------------------------------
class LedStateCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led state [ interface ] [ INTFS ]'
   data = {
      'led' : ledMatcher,
      'state' : CliCommand.guardedKeyword( 'state', helpdesc='Show LED State',
         guard=isUserConfigurableLed ),
      'interface' : matcherInterface,
      'INTFS' : matcherIntfRange,
   }
   cliModel = LedConfigState

   @staticmethod
   def handler( mode, args ):
      intfs = args.get( 'INTFS', hwLedConfig.leds )
      leds = {}

      for led in intfs:
         if led not in hwLedConfig.leds:
            continue
         leds[ led ] = {}
         for stateKey, state in hwLedConfig.leds[ led ].state.items():
            ledState = LedState()
            ledState.name = stateKey
            ledState.available = state.available
            leds[ led ][ stateKey ] = ledState
         ledStateDict = LedStateDict( state=leds[ led ] )
         leds[ led ] = ledStateDict
         leds[ led ].name = led
      return LedConfigState( leds=leds )

BasicCli.addShowCommandClass( LedStateCmd )

#--------------------------------------------------------------------------------
# show led theme
#--------------------------------------------------------------------------------
nodeTheme = CliCommand.guardedKeyword( 'theme', helpdesc='Show LED Theme',
      guard=isUserConfigurableLed )

class LedThemeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led theme'
   data = {
      'led' : ledMatcher,
      'theme' : nodeTheme,
   }
   cliModel = LightSettingThemeCollection

   @staticmethod
   def handler( mode, args ):
      theme = {}
      for themeName in hwLedConfig.theme:
         theme[ themeName ] = {}
         for settingName, setting in hwLedConfig.theme[ themeName ].setting.items():
            lsThemeSetting = setting.value
            lightSetting = LightSetting()
            lightSetting.color = lsThemeSetting.color
            lightSetting.ledFlashRate = lsThemeSetting.ledFlashRate
            theme[ themeName ][ settingName ] = lightSetting
         lightSettingTheme = LightSettingTheme( name=themeName,
                                                setting=theme[ themeName ] )
         theme[ themeName ] = lightSettingTheme
      return LightSettingThemeCollection( theme=theme )

BasicCli.addShowCommandClass( LedThemeCmd )

#--------------------------------------------------------------------------------
# show led theme interface [ INTFS ]
#--------------------------------------------------------------------------------
class LedThemeInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show led theme interface [ INTFS ]'
   data = {
      'led' : ledMatcher,
      'theme' : nodeTheme,
      'interface' : matcherInterface,
      'INTFS' : matcherIntfRange,
   }
   cliModel = LedConfigDirTheme

   @staticmethod
   def handler( mode, args ):
      intfs = ( led for led in hwLedConfig.leds if led.startswith( "Ethernet" ) )
      intfs = args.get( 'INTFS', intfs )
      theme = {}
      ledsTheme = {}
      for led in intfs:
         if led not in hwLedConfig.leds:
            continue
         themeName = hwLedConfig.leds[ led ].theme
         theme[ themeName ] = {}
         ledsTheme[ led ] = themeName

         for settingName, setting in hwLedConfig.theme[ themeName ].setting.items():
            lsThemeSetting = setting.value
            lightSetting = LightSetting()
            lightSetting.color = lsThemeSetting.color
            lightSetting.ledFlashRate = lsThemeSetting.ledFlashRate
            theme[ themeName ][ settingName ] = lightSetting
         lightSettingTheme = LightSettingTheme( name=themeName,
                                                setting=theme[ themeName ] )
         theme[ themeName ] = lightSettingTheme
      return LedConfigDirTheme( theme=theme, ledsTheme=ledsTheme )

BasicCli.addShowCommandClass( LedThemeInterfaceCmd )

# -----------------------------------------------------------------------------
# Cli Plugin initialization
# -----------------------------------------------------------------------------
def Plugin( entityManager ):
   global hwLedConfigInit, hwLedConfig, ledConfig

   hwLedConfigInit = LazyMount.mount( entityManager, "hardware/led/configInit",
                                  "Hardware::Led::LedSystemConfigDir", "r" )
   hwLedConfig = LazyMount.mount( entityManager, "hardware/led/config",
                                  "Hardware::Led::LedSystemConfigDir", "r" )
   ledConfig = LazyMount.mount( entityManager, "led/config", "Tac::Dir", "w" )
