# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet.MplsLib
import BasicCli
from BgpLib import getVpwsName, pwNamePattern
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
import CliMatcher
import CliParser
from CliPlugin import IpAddrMatcher
from CliPlugin.BgpMacVrfConfigCli import BgpMacVrfVpwsMode
from CliPlugin.LdpCli import (
   LdpConfigMode,
   getLdpConfigPseudowireCleanupHook,
)
import LazyMount

bgpMacVrf = None
bridgingHwCapabilities = None
pwHwCapability = None

def pwSupportedGuard( mode, token ):
   if pwHwCapability.mplsPseudowireSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsStaticPwSupportedGuard( mode, token ):
   if pwHwCapability.mplsStaticPseudowireSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def getPseudowireNames( mode, ctx ):
   vpwsName = getVpwsName( ctx.sharedResult[ "VPWS_NAME" ] )
   bgpMacVrfConfig = bgpMacVrf.config.get( vpwsName )
   return sorted( bgpMacVrfConfig.pseudowireConfig ) if bgpMacVrfConfig else []

# Our l2MtuMin and l2MtuMax size include the ethernet header and 802.1Q tag
# (i.e., 18 bytes). However, RFC4447 requires the MTU used in LDP to exclude
# the encapsulation overhead.
def mtuRangeFn( mode, context ):
   encapOctets = 18
   return ( bridgingHwCapabilities.l2MtuMin - encapOctets,
            bridgingHwCapabilities.l2MtuMax - encapOctets )

# Common tokens
bgpPseudowireNameMatcher = CliMatcher.DynamicNameMatcher(
      getPseudowireNames,
      "EVPN VPWS pseudowire name",
      pattern=pwNamePattern,
      passContext=True )
controlWordMatcher = CliMatcher.KeywordMatcher( 'control-word',
      helpdesc='Enable control word' )
labelMatcher = CliMatcher.IntegerMatcher( Arnet.MplsLib.labelMin,
                                          Arnet.MplsLib.labelMax,
                                          helpdesc='MPLS label' )
mtuMatcher = CliMatcher.KeywordMatcher( 'mtu', helpdesc='MTU' )
mtuNumMatcher = CliMatcher.DynamicIntegerMatcher(
                                             rangeFn=mtuRangeFn,
                                             helpdesc='MTU for pseudowire' )

mplsKeywordForConfig = CliCommand.guardedKeyword( 'mpls',
      helpdesc='Global MPLS configuration commands', guard=pwSupportedGuard )

#--------------------------------------------------------------------------------
# [ no | default ] patch panel
#--------------------------------------------------------------------------------
class PatchPanelCmd( CliCommand.CliCommandClass ):
   syntax = 'patch panel'
   noOrDefaultSyntax = syntax
   data = {
      'patch': CliCommand.guardedKeyword( 'patch',
         helpdesc='Configure pseudowire patch',
         guard=pwSupportedGuard ),
      'panel': 'Configure pseudowire patches',
   }

   handler = 'Pseudowire.gotoPatchPanelMode'
   noOrDefaultHandler = 'Pseudowire.noPatchPanel'

BasicCli.GlobalConfigMode.addCommandClass( PatchPanelCmd )

# --------------------------------------------------------------------------------
# [ no | default ] static pseudowires
# --------------------------------------------------------------------------------
mplsStaticPseudowiresNode = CliCommand.guardedKeyword( 'pseudowires',
      helpdesc='Configure MPLS static pseudowires',
      guard=mplsStaticPwSupportedGuard )

class MplsStaticPseudowiresCmd( CliCommand.CliCommandClass ):
   syntax = 'static pseudowires'
   noOrDefaultSyntax = syntax
   data = {
      'static': 'Configure MPLS static pseudowires',
      'pseudowires': mplsStaticPseudowiresNode
   }

   handler = 'Pseudowire.gotoMplsStaticPseudowiresMode'
   noOrDefaultHandler = 'Pseudowire.noMplsStaticPseudowires'

#--------------------------------------------------------------------------------
# [ no | default ] mpls ldp pseudowires
#--------------------------------------------------------------------------------
mplsLdpPseudowiresNode = CliCommand.guardedKeyword( 'pseudowires',
      helpdesc='Configure MPLS LDP pseudowires', guard=pwSupportedGuard )

class MplsLdpPseudowiresCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp pseudowires'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsKeywordForConfig,
      'ldp': 'Configure MPLS LDP pseudowires',
      'pseudowires': mplsLdpPseudowiresNode
   }
   hidden = True

   handler = 'Pseudowire.gotoMplsLdpPseudowiresMode'
   noOrDefaultHandler = 'Pseudowire.noMplsLdpPseudowires'

BasicCli.GlobalConfigMode.addCommandClass( MplsLdpPseudowiresCmd )

# --------------------------------------------------------------------------------
# [ no | default ] mpls pseudowires
# --------------------------------------------------------------------------------
class MplsPseudowiresCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls pseudowires'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': mplsKeywordForConfig,
      'pseudowires': 'MPLS pseudowire configuration',
   }
   handler = 'Pseudowire.gotoMplsPseudowiresMode'
   noOrDefaultHandler = 'Pseudowire.noMplsPseudowiresMode'

BasicCli.GlobalConfigMode.addCommandClass( MplsPseudowiresCmd )

#--------------------------------------------------------------------------------
# [ no | default ] pseudowires
#--------------------------------------------------------------------------------
class PseudowiresCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowires'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowires': mplsLdpPseudowiresNode
   }

   handler = 'Pseudowire.gotoMplsLdpPseudowiresMode'
   noOrDefaultHandler = 'Pseudowire.noMplsLdpPseudowires'

LdpConfigMode.addCommandClass( PseudowiresCmd )

#-------------------------------------------------------------------------------
# The "[no|default] debug pseudowire PWNAME neighbor PWPEER peer-label LABEL
# [mtu MTU] [control-word]" hidden command under "vpws EVINAME" config mode
#-------------------------------------------------------------------------------
class DebugPwBgpCmd( CliCommand.CliCommandClass ):
   syntax = '''debug pseudowire PWNAME neighbor PWPEER peer-label LABEL
               [ local [ mtu-l LOCALMTU ] [ control-word-l ] ]
               [ peer [ mtu-p PEERMTU ] [ control-word-p ] ]
            '''
   noOrDefaultSyntax = '''debug pseudowire [ PWNAME ] ...'''
   data = {
      'debug': 'BGP pseudowire debug',
      'pseudowire': 'EVPN VPWS pseudowire name',
      'PWNAME': bgpPseudowireNameMatcher,
      'neighbor': 'Pseudowire neighbor',
      'PWPEER': IpAddrMatcher.IpAddrMatcher( helpdesc='Pseudowire neighbor' ),
      'peer-label': 'Peer label',
      'LABEL': labelMatcher,
      'local': 'Local options',
      'peer': 'Peer options',
      'mtu-l': mtuMatcher,
      'LOCALMTU': mtuNumMatcher,
      'control-word-l': controlWordMatcher,
      'mtu-p': mtuMatcher,
      'PEERMTU': mtuNumMatcher,
      'control-word-p': controlWordMatcher,
   }
   hidden = True
   handler = 'Pseudowire.debugPwBgpCmdHandler'
   noOrDefaultHandler = 'Pseudowire.debugPwBgpCmdNoHandler'

BgpMacVrfVpwsMode.addCommandClass( DebugPwBgpCmd )

# -------------------------------------------------------------------------------
# Cleanup
#-------------------------------------------------------------------------------

getLdpConfigPseudowireCleanupHook().addExtension(
   lambda mode: CliDynamicPlugin( 'Pseudowire' ).noMplsLdpPseudowires( mode, None ) )

def Plugin( entityManager ):
   global bgpMacVrf
   global bridgingHwCapabilities
   global pwHwCapability

   bgpMacVrf = LazyMount.mount( entityManager, "routing/bgp/macvrf",
                                "Routing::Bgp::MacVrfConfigDir", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   pwHwCapability = LazyMount.mount( entityManager,
                                     "routing/hardware/pseudowire/capability",
                                     "Pseudowire::Hardware::Capability", "r" )
