# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#-------------------------------------------------------------------------------
# This module implements PTP configuration.  In particular, it provides:
#
# Global Configuration
# [no|default] ptp mode disabled|boundary|e2etransparent|p2ptransparent|gptp
# [no|default] ptp forward-v1
# [no|default] ptp priority1
# [no|default] ptp priority2
# [no|default] ptp clock-identity <eui-64 number>|<mac address>
# [no|default] ptp clock-accuracy < value >
# [no|default] ptp hold-ptp-time <interval>
# [no|default] ptp monitor
# [no|default] ptp monitor threshold offset-from-master <threshold>
# [no|default] ptp monitor threshold mean-path-delay <threshold>
# [no|default] ptp monitor threshold skew <threshold>
#
# Ethernet/Port-Channel Interface Configuration
# [no|default] ptp announce interval <interval>
# [no|default] ptp announce timeout <timeout>
# [no|default] ptp sync-message interval <interval>
# The command sync timeout is used only in gPTP mode
# [no|default] ptp sync timeout <timeout>
# [no|default] ptp sync limit <limit>
# [no|default] ptp delay-req interval <interval>
# [no|default] ptp pdelay-req interval <interval>
# [no|default] ptp pdelay-neighbor-threshold <threshold in nanoseconds>
# [no|default] ptp delay-mechanism p2p|e2e|disabled
# [no|default] ptp enable
# [no|default] ptp mpass
# [no|default] ptp debug [message {(announce|sync|follow-up|delay-req|delay-resp)}]
#              tx-interface INTF
# [no|default] ptp mpass message sync timeout VALUE messages
# [no|default] ptp mpass message delay-req timeout VALUE seconds
#-------------------------------------------------------------------------------

# pylint: disable=consider-using-f-string

import datetime
from functools import partial
import glob
import io
import math
from operator import attrgetter

import AgentCommandRequest
from AgentDirectory import agentIsRunning
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
import CliPlugin.TechSupportCli
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import
from CliPlugin.PtpCliModel import (
   PtpHoldPtpTimeForShow, PtpClockQuality, PtpClock,
   PtpParent, PtpParentWrapper, PtpTimeProperty, PtpTimePropertyWrapper,
   PtpIntfCounters, PtpIntfMpassCounters,
   PtpIntfCountersWrapper, PtpCounters, PtpIntfVlansCounters,
   PtpIntf, PtpMpassIntf, PtpIntfWrapper, PtpIntfVlans, PtpInterfaces, PtpSummary,
   PtpMpassSummary, PtpIntfSummary, PtpIntfVlanSummary, PtpPortIdentity,
   PtpForeignMasterDS, PtpIntfForeignMasterRecords, PtpForeignMasterRecords,
   PtpIntfVlanForeignMasterRecords, PtpSourceIp,
   PtpMonitorDataEntry, PtpMonitorData,
   PtpVlans, PtpVlansExtendedWrapper, PtpDomainNumbersSummary,
   UcastNegProfiles, UcastNegProfile,
   UcastNegCandidateGrantor, UcastNegCandidateGrantorEntry,
   UcastNegRemoteGrantee, UcastNegRemoteGranteeEntry,
   UcastNegMessage, UcastNegMessages, UcastNegStatus )
from CliPlugin.PtpDebugCounterModel import PtpDebugCounters, PtpDebugIntfCounters, \
   PtpDebugCounter, PtpEndpoint, PtpDebugCounterDropReason
import CliToken.Clear
import CliToken.Ip
import CliToken.Ipv6
import ConfigMount
import Ethernet
from IpLibConsts import DEFAULT_VRF
import Ptp
import LazyMount
import SharedMem
import Smash
import Tac
from TypeFuture import TacLazyType
from Vlan import computeVlanRangeSet
from Toggles.PtpLibToggleLib import (
   togglePtpIsolatedSlaveEnabled )

modeSpecificIntervalAttrs = [ 'logSyncInterval',
                              'logAnnounceInterval',
                              'logMinPdelayReqInterval' ]
ptpTwoStepBoundaryClockMode = "ptpBoundaryClock"
ptpBoundaryModes = [ ptpTwoStepBoundaryClockMode, "ptpOneStepBoundaryClock" ]
ptpTransparentModes = [ "ptpEndToEndTransparentClock",
                        "ptpOneStepEndToEndTransparentClock" ]

PtpMode = TacLazyType( 'Ptp::PtpMode' )
PtpProfile = TacLazyType( 'Ptp::PtpProfile' )
CliConstants = TacLazyType( 'Ptp::Constants' )
CliRegionConfigConstants = TacLazyType( 'Ptp::CliRegionConfigConstants' )
UcastNegGrantorState = TacLazyType( 'Ptp::UcastNegGrantorState' )
CounterDir = TacLazyType( "Ptp::DebugCounter::Direction::Constant" )
CounterType = TacLazyType( "Ptp::DebugCounter::Type::Constant" )
CounterReason = TacLazyType( "Ptp::DebugCounter::Reason::Constant" )
CounterPortKeyConstant = TacLazyType( "Ptp::DebugCounter::CounterPortKeyConstant" )
MTN = TacLazyType( "Ptp::MessageTypeNumber" )

ptpConfig = None
ptpStatus = None
ptpHwCapabilityDir = None
ptpHwStatusDir = None
brConfig = None
bridgingSic = None
serviceName = 'ptp'
aclCheckpoint = None
ptpUcastNegProfiles = None
ucastNegStatus = None
debugCounter = None
pktCounter = None
ptpModeDict = {
   'disabled': 'Disable PTP and forward PTP messages normally',
   'boundary': 'Boundary Clock Mode',
   'e2etransparent': 'End-to-End Transparent Clock',
   'p2ptransparent': 'Peer-to-Peer Transparent Clock',
   'gptp': 'Generalized PTP Clock',
   'ordinarymaster': 'Ordinary Master Clock',
}

def checkVlanLimit( mode, logSyncInterval, configuredVlans ):
   rate = ( 1 / float( 2**logSyncInterval ) ) * configuredVlans
   maxRate = 1 / CliConstants.minEgressTsFifoPollInterval # = 200
   if rate > maxRate:
      mode.addError( "Number of PTP configured VLANs on a trunk is too high."
                     " Please consider increasing the sync messages interval or"
                     " configuring less VLANs" )
      return False
   return True

def getBounds( attr, ptpMode=None ):
   if not ptpMode:
      ptpMode = '802Dot1AS' if ptpConfig.ptpMode == 'ptpGeneralized' else '1588V2'
   return tuple( getattr( CliConstants, bound + attr + ptpMode )
                 for bound in ( 'min', 'max' ) )

def featureSupported( feature: str ):
   getter = attrgetter( feature )
   # XXX BUG792440 remove ptpHwStatusDir once all platforms migrate
   for ptpDir in ( ptpHwStatusDir, ptpHwCapabilityDir ):
      if any( getter( dirEntry ) for dirEntry in ptpDir.values() ):
         return True
   return False

def ptpSupported():
   return featureSupported( "ptpSupported" )

def ptpSupportedGuard( mode, token ):
   if ptpSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpForwardV1Supported():
   return featureSupported( "ptpForwardV1Supported" )

def ptpForwardV1SupportedGuard( mode, token):
   if ptpForwardV1Supported():
      return None
   return CliParser.guardNotThisPlatform

def ptpOneStepBoundaryClockSupported():
   return featureSupported( "ptpOneStepBoundaryClockSupported" )

def ptpOneStepBoundaryClockSupportedGuard( mode, token ):
   if ptpOneStepBoundaryClockSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpBoundaryClockSupported():
   return featureSupported( "ptpBoundaryClockSupported" )

def ptpBoundaryClockSupportedGuard( mode, token ):
   if ptpBoundaryClockSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpE2ETransparentClockSupported():
   return featureSupported( "ptpE2ETransparentClockSupported" )

def ptpE2ETransparentClockSupportedGuard( mode, token ):
   if ptpE2ETransparentClockSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpP2PTransparentClockSupported():
   return featureSupported( "ptpP2PTransparentClockSupported" )

def ptpOneStepTransparentClockSupported():
   return featureSupported( "ptpOneStepTransparentClockSupported" )

# guard for the `e2etransparent` keyword in the `e2etransparent one-step` command
# needed to maintain giving a GuardError when 1 step tc is not supported but 2 step
# is, otherwise we get InvalidInputError for `e2etransparent one-step`
def ptpE2ETransparentKeywordForOneStepGuard( mode, token ):
   if ptpE2ETransparentClockSupported() or ptpOneStepTransparentClockSupported():
      return None
   return CliParser.guardNotThisPlatform

# guard for the `one-step` keyword at the end of the command
def ptpOneStepTransparentClockSupportedGuard( mode, token ):
   if ptpOneStepTransparentClockSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpGptpClockSupported():
   return featureSupported( "ptpGptpClockSupported" )

def ptpOrdinaryMasterClockSupported():
   return featureSupported( "ptpOrdinaryMasterClockSupported" )

def getSupportedModes():
   supportedModes = set( ptpModeDict.keys() )
   if not ptpBoundaryClockSupported():
      supportedModes.remove( 'boundary' )
   if not ptpE2ETransparentClockSupported():
      supportedModes.remove( 'e2etransparent' )
   if not ptpP2PTransparentClockSupported():
      supportedModes.remove( 'p2ptransparent' )
   if not ptpGptpClockSupported():
      supportedModes.remove( 'gptp' )
   if not ptpOrdinaryMasterClockSupported():
      supportedModes.remove( 'ordinarymaster' )
   return supportedModes

def ptpModeGuard( mode, token ):
   if token in getSupportedModes():
      return None
   return CliParser.guardNotThisPlatform

def ptpG82751BCSupported():
   return featureSupported( "ptpG82751BCSupported" )

def ptpG82751TCSupported():
   return featureSupported( "ptpG82751TCSupported" )

def ptpG82751Guard( mode, token ):
   if ptpG82751BCSupported() or ptpG82751TCSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpG82751Active():
   if ptpConfig.ptpProfile != 'ptpG8275_1':
      return False
   if ptpConfig.ptpMode in ptpBoundaryModes and ptpG82751BCSupported():
      return True
   if ptpConfig.ptpMode in ptpTransparentModes and ptpG82751TCSupported():
      return True
   return False

def ptpForwardUnicastSupported():
   return featureSupported( "ptpForwardUnicastSupported" )

def ptpForwardUnicastGuard( mode, token ):
   if ptpForwardUnicastSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpSyncFromSystemClockSupported():
   return featureSupported( "ptpSyncFromSystemClockSupported" )

def ptpSyncFromSystemClockGuard( mode, token ):
   if ptpSyncFromSystemClockSupported():
      return None
   return CliParser.guardNotThisPlatform

def getDsPorts( intfs=None, vlanId=None ):
   ports = ptpStatus.portDS

   if intfs: # Filter by interface?
      intfs = set( intfs )
      ports = ( port for port in ports if port.intf in intfs )

   if vlanId: # Filter by VLAN ID?
      ports = ( port for port in ports if port.vlanId == vlanId )

   return ports

def createDefaultCliIntfConfig( intf, createEnabled=False ):
   intfConfig = ptpConfig.newIntfConfig( intf, createEnabled, False )
   # Add default portDS vlan to newly created interface, to only send VLAN untagged
   # packet by default
   intfConfig.vlanConfig.newMember( CliConstants.defaultPortDSVlanId )
   intfConfig.vlanIds[ CliConstants.defaultPortDSVlanId ] = True
   return intfConfig

def getConfigIntfs( intfs=None ):
   return intfs or ptpConfig.intfConfig

def setIntfDefaults( intf, createEnabled=False ):
   ptpMode = ''
   if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
      ptpMode = '802Dot1AS'
   else:
      ptpMode = '1588V2'

   intfConfig = ptpConfig.intfConfig.get( intf )
   if not intfConfig:
      intfConfig = createDefaultCliIntfConfig( intf, createEnabled )
      for intervalAttr in modeSpecificIntervalAttrs:
         intervalValue = getattr( CliConstants, intervalAttr + 'Default' + ptpMode )
         setattr( intfConfig, intervalAttr, intervalValue )
   return ( intfConfig, ptpMode )

attrsUsingDeprecatedCmds = {
   'logSyncInterval' : 'syncIntervalUseDeprecatedCmd'
}

def setIntfAttr( mode, attr, value=None, no=None, deprecatedCmd=False ):
   ( intfConfig, ptpMode ) = setIntfDefaults( mode.intf.name )
   assert hasattr( intfConfig, attr )
   deprecatedAttr = attrsUsingDeprecatedCmds.get( attr )
   if deprecatedAttr:
      setattr( intfConfig, deprecatedAttr, deprecatedCmd )
   if value is None or no is not None:
      if attr in modeSpecificIntervalAttrs:
         value = getattr( CliConstants, attr + 'Default' + ptpMode )
      else:
         value = getattr( CliConstants, attr + 'Default' )
   setattr( intfConfig, attr, value )

def regionConfigIsDefault( regionConfig ):
   for attrDefault in CliRegionConfigConstants.attributes:
      assert attrDefault.endswith( 'Default' )
      attr = attrDefault.removesuffix( 'Default' )
      current = getattr( regionConfig, attr )
      default = getattr( CliRegionConfigConstants, attrDefault )
      if current != default:
         return False
   return True

def setRegionIntfVlanAttr( mode, vlans, attr, value, no ):
   # get or initialize per intf,vlan region configs
   if no:
      value = getattr( CliRegionConfigConstants, attr + 'Default' )

   PortDSKey = TacLazyType( "Ptp::PortDSKey" )
   intf = mode.intf.name
   if vlans is None:
      vlans = [ CliConstants.defaultPortDSVlanId ]
   for vlan in vlans:
      key = PortDSKey( intf, vlan )
      if key not in ptpConfig.intfRegionConfig:
         if no:
            # the config is already cleaned up, don't create it just to
            # remove it below.
            continue
         # Create a new config for the non-no command.
         ptpConfig.intfRegionConfig.newMember( key )
      regionConfig = ptpConfig.intfRegionConfig[ key ]
      setattr( regionConfig, attr, value )
      if regionConfigIsDefault( regionConfig ):
         # The config just became all default values, clean it up.
         # Software should already be treating lack of a region config as no config.
         # Otherwise we end up with up to 4095 entities per interface sitting around.
         del regionConfig
         del ptpConfig.intfRegionConfig[ key ]

def getVlanId( portDS ):
   vlanId = 0
   if ptpStatus.defaultDS.ptpMode in ptpBoundaryModes and portDS.vlanId != 0:
      switchIntfConfig = bridgingSic.switchIntfConfig.get( portDS.intf )
      isIntfTrunkMode = switchIntfConfig is not None and \
                        switchIntfConfig.switchportMode == 'trunk'
      if isIntfTrunkMode:
         vlanId = portDS.vlanId
   return vlanId

guardedPtpConfigPartial = partial( CliCommand.guardedKeyword, 'ptp',
                          helpdesc='Precision Time Protocol' )
ptpMatcherForConfig = guardedPtpConfigPartial ( guard=ptpSupportedGuard )

#-------------------------------------------------------------------------------
# [no|default] ptp mode disabled|boundary|e2etransparent|p2ptransparent|gptp
#-------------------------------------------------------------------------------
def setGlobalAttr( mode, attr, value=None, no=None ):
   assert hasattr( ptpConfig, attr )
   if value is None or no is not None:
      value = getattr( CliConstants, attr + 'Default' )
   setattr( ptpConfig, attr, value )

def setGlobalMapAttr( mode, attr, key, value=None, no=None ):
   assert hasattr( ptpConfig, attr )
   attrMap = getattr( ptpConfig, attr )
   attrMap[ key ] = value

ptpModeMapping = {
   'disabled': 'ptpDisabled',
   'boundary': 'ptpBoundaryClock',
   'e2etransparent': 'ptpEndToEndTransparentClock',
   'p2ptransparent': 'ptpPeerToPeerTransparentClock',
   'gptp': 'ptpGeneralized',
   'ordinarymaster': 'ptpOrdinaryMasterClock',
   'ptpOneStepEndToEndTransparentClock': 'ptpOneStepEndToEndTransparentClock',
   'ptpOneStepBoundaryClock': 'ptpOneStepBoundaryClock'
}

def setPtpMode( mode, ptpMode ):
   ptpMode = ptpModeMapping[ ptpMode ] #translate the mode
   oldPtpMode = ptpConfig.ptpMode
   if ptpMode == PtpMode.ptpGeneralized:
      lagIntfs = []
      for intf in ptpConfig.intfConfig:
         if Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( intf ) \
            and ptpConfig.intfConfig[ intf ].enabled:
            lagIntfs.append( intf )
      if lagIntfs:
         mode.addError( "Gptp is not supported on port-channels." )
         mode.addError( "Please disable ptp on the following port-channels "
                        "before attempting to enable gptp:" )
         for lagIntf in lagIntfs:
            mode.addError( lagIntf )
         return
   elif ptpMode == ptpTwoStepBoundaryClockMode:
      intfConfigs = ptpConfig.intfConfig
      if intfConfigs:
         for intfConfig in intfConfigs.values():
            if not checkVlanLimit( mode, intfConfig.logSyncInterval,
                  len( intfConfig.vlanConfig ) ):
               return
   ptpConfig.ptpMode = ptpMode
   ptpConfig.enabled = ( ptpMode != 'ptpDisabled' )
   if PtpMode.ptpGeneralized in ( oldPtpMode, ptpMode ):
      if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
         ptpType = '802Dot1AS'
         oldPtpType = '1588V2'
      else:
         ptpType = '1588V2'
         oldPtpType = '802Dot1AS'
      outOfIntervalAttrWarning = { }
      attrOutputMap = { 'logSyncInterval':'sync-interval',
                        'logAnnounceInterval':'announce-interval',
                        'logMinPdelayReqInterval':'peer-delay-req-interval' }
      for intfConfig in ptpConfig.intfConfig.values():
         for attr in modeSpecificIntervalAttrs:
            oldDefaultValue = getattr( CliConstants, attr + 'Default' + oldPtpType )
            if getattr( intfConfig, attr ) == oldDefaultValue:
               newDefaultValue = getattr( CliConstants, attr + 'Default' + ptpType )
               setattr( intfConfig, attr, newDefaultValue )
               continue
            lowerLimit = getattr( CliConstants, 'min' + attr + ptpType )
            upperLimit = getattr( CliConstants, 'max' + attr + ptpType )
            oldValue = getattr( intfConfig, attr )
            # Reset to default only if oldValue does not lie in the new mode's
            # allowed interval
            if oldValue < lowerLimit or oldValue > upperLimit:
               outOfIntervalAttrWarning[ attr ] = True
               defaultValue = getattr( CliConstants, attr + 'Default' + ptpType )
               setattr( intfConfig, attr, defaultValue )
      for attr in outOfIntervalAttrWarning:
         mode.addWarning( "Unsupported Value of %s on some interfaces "
                          "in %s mode" % ( attrOutputMap[ attr ] , ptpMode ) )

#-------------------------------------------------------------------------------
# ptp profile ( g8275.2 | g8275.1 )
#
# [no|default] ptp profile
#-------------------------------------------------------------------------------
ucastNegRequiredProfiles = { PtpProfile.ptpG8275_2 }

def setPtpProfile( mode, attr, value=None, no=None ):
   if value == 'g8275.2':
      value = PtpProfile.ptpG8275_2
   elif value == 'g8275.1':
      value = PtpProfile.ptpG8275_1
      mode.addWarning( "Announce message rate will be fixed to 8/s" )
      mode.addWarning( "Sync/FollowUp message rate will be fixed to 16/s" )
      mode.addWarning( "Delay-req message rate will be fixed to 16/s" )
      mode.addWarning( "Only E2E delay mechanism will be supported" )

   setGlobalAttr( mode, attr, value, no )
   if value in ucastNegRequiredProfiles and no is None:
      # ptp source ip is not supported with unicast negotiation
      # Reset to default values
      setGlobalAttr( mode, 'srcIp4', value=None, no=True )
      setGlobalAttr( mode, 'srcIp6', value=None, no=True )
      ptpConfig.unicastNegotiation = True
   else:
      ptpConfig.unicastNegotiation = False

#-------------------------------------------------------------------------------
# [no|default] ptp management all
#-------------------------------------------------------------------------------
def setManagementEnabled( mode, no=False ):
   if ptpConfig.ptpMode not in ptpBoundaryModes:
      mode.addWarning( "PTP Management Messages configuration will be ignored while "
                       "not in Boundary mode" )
   ptpConfig.managementEnabled = not no

#-------------------------------------------------------------------------------
# [no|default] ptp netsync-monitor delay-request
#-------------------------------------------------------------------------------
def setNetSyncMonitor( mode, no=False ):
   if ptpConfig.ptpMode not in ptpBoundaryModes:
      mode.addWarning( "PTP NetSync Monitor configuration will be ignored while not "
                       "in Boundary mode" )
   ptpConfig.netSyncMonitor = not no

#-------------------------------------------------------------------------------
# [no|default] ptp forward-v1
#-------------------------------------------------------------------------------
def forwardPtpV1( mode, no=False ):
   ptpConfig.forwardPtpV1 = not no

def forwardPtpUnicast( mode, no=False ):
   ptpConfig.forwardUnicast = not no

#-------------------------------------------------------------------------------
# [no|default] ptp domain <domain>
#-------------------------------------------------------------------------------
def domainNumberRange( mode, context ):
   ptpProfile = '1588V2'
   if ptpConfig.ptpMode in ptpBoundaryModes:
      if ptpConfig.ptpProfile in ( 'ptpG8275_2', 'ptpG8275_1' ):
         ptpProfile = ptpConfig.ptpProfile[ 3: ] # Remove leading 'ptp'.
   return getBounds( 'DomainNumber', ptpProfile )

#-------------------------------------------------------------------------------
# [no|default] ptp priority1 <priority1>
#-------------------------------------------------------------------------------
def setPriority1( mode, attr, value=None, no=None ):
   if ptpConfig.unicastNegotiation:
      mode.addWarning( "Priority1 is not used in PTP G8275.2 profile and has a "
                       "static value of 128" )
   setGlobalAttr( mode, 'priority1', value, no )

#-------------------------------------------------------------------------------
# [no|default] ptp clock-identity <eui-64 number>|<mac address>
#-------------------------------------------------------------------------------
colonPattern = ':'.join( [ Ethernet.pair for x in range( 8 ) ] ) + r'$'
dashPattern = '\\-'.join( [ Ethernet.pair for x in range( 8 ) ] ) + r'$'
dotPattern = '\\.'.join( [ Ethernet.quad for x in range( 4 ) ] ) + r'$'
eui64Pattern = '(%s|%s|%s)' % ( colonPattern, dashPattern, dotPattern )

def setPtpClockIdentity( mode, args ):
   # Parse the given EUI string, and create an instance of ClockIdentity.
   def createClockIdentity( euiStr ):
      # Split raw eui string by its delimeters and place leading 0's in each octet
      if '.' in euiStr:
         euiStr = [ '%04x' % int( n, 16 ) for n in euiStr.split('.') ]
      elif ':' in euiStr:
         euiStr = [ '%02x' % int( n, 16 ) for n in euiStr.split(':') ]
      elif '-' in euiStr:
         euiStr = [ '%02x' % int( n, 16 ) for n in euiStr.split('-') ]
      else:
         # We should never hit here, but just in case
         mode.addError( "Invalid character found in EUI number" )
         return None

      # Join the all octet to get a hexadecimal string of the eui number
      euiStr = ''.join( euiStr )

      isMacAddress = ( len( euiStr ) == 12 )
      isEui64Number = ( len( euiStr ) == 16 )
      if not ( isMacAddress or isEui64Number ):
         # We should never hit here, but just in case
         mode.addError( "input was in neither MAC address nor EUI-64 bit format: " +
                        euiStr )
         return None

      # If it's in MAC address format, convert it to EUI-64
      if isMacAddress:
         # from IEEE 1588 - 7.5.2.2.2
         # When using an EUI-48, the first 3 octets, ... of the IEEE EUI-48 are
         # assigned in order to the first 3 octets of the clockIdentity with the
         # most significant octet of the IEEE EUI-64. ... Octets with index 3 and
         # 4 have hex values FF and FE, respectively. The remaining 3 octets of
         # the IEEE EUI-48 are assigned in order to the last 3 octets of the
         # clockIdentity
         # But we are using FFFF to follow the current codebase practice. Once
         # our codebase changes back to follow the spec, we should chnage this to
         # FFFE as well.
         euiStr = euiStr[ : 6 ] + 'ffff' + euiStr[ 6 : ]

      # according to IEEE 1588 7.5.2.2 the least significant two bits of the most
      # significant octet of the OUI shall both be 0. However, we omit this check
      # since the Ptp agent itself never checks for this restriction. And upon
      # inspecting the spec, not following it will not result in unexpected
      # errors.

      # Convert to integer
      eui64 = int( euiStr, 16 )

      return Tac.newInstance( "Ptp::ClockIdentity",
                              Tac.newInstance( "Arnet::NetU64", eui64 ) )

   ptpConfig.clockIdentityConfigured = True
   ptpConfig.clockIdentity = createClockIdentity( args[ 'CLOCK_IDENTITY' ] )

def unsetPtpClockIdentity( mode, args ):
   ptpConfig.clockIdentityConfigured = False

#-------------------------------------------------------------------------------
# [no|default] ptp clock-accuracy VALUE
#-------------------------------------------------------------------------------
def newClockQuality( clkQual ):
   return Tac.Value( "Ptp::ClockQuality", clkQual.clockClass,
                     clkQual.clockAccuracy, clkQual.offsetScaledLogVariance )

def setPtpClockAccuracy( mode, args ):
   value = args.get( 'VALUE' )
   ptpConfig.clockQualityConfigured = value is not None


   clkQual = newClockQuality( ptpConfig.clockQuality )

   if ptpConfig.clockQualityConfigured:
      clkQual.clockAccuracy = value
   else:
      clkQual.clockAccuracy = ptpStatus.localClockQuality.clockAccuracy

   ptpConfig.clockQuality = clkQual

#-------------------------------------------------------------------------------
# [no|default] ptp source ip address <address>
#-------------------------------------------------------------------------------
def setPtpSrcIp4( mode, ipAddr=None, no=None ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return

   if ipAddr is not None:
      ip = Tac.Value( "Arnet::IpGenAddr", ipAddr )
      if not ip.isUnicast:
         mode.addError( "Source IP must be an unicast address" )
         return
   setGlobalAttr( mode, 'srcIp4', value=ipAddr, no=no )

#-------------------------------------------------------------------------------
# [no|default] ptp local-interface <INTERFACE>
#-------------------------------------------------------------------------------
def setPtpSrcIntf( mode, sourceIntf=None, no=None ):
   loopbackName = sourceIntf.name if sourceIntf else None
   setIntfAttr( mode, 'loopbackSourceIntf', loopbackName, no=no )

#-------------------------------------------------------------------------------
# [no|default] ptp source ipv6 address <address>
#-------------------------------------------------------------------------------
def setPtpSrcIp6( mode, ip6Addr=None, no=None ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return
   setGlobalAttr( mode, 'srcIp6', value=ip6Addr, no=no )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor
#-------------------------------------------------------------------------------
def setPtpMonitor( mode, no=None ):
   setGlobalAttr( mode, 'monitorEnabled', value=not no )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor sequence-id
#-------------------------------------------------------------------------------
def setLogMissingMsgs( mode, no=None ):
   # Enable/disable all missing message type
   seqIdMap = 'logMissingMessageSeqId'
   thresholdMap = 'missingSequenceThresholds'
   setGlobalAttr( mode, 'logMissingMessages', value=not no)
   setGlobalMapAttr( mode, seqIdMap, MTN.messageSync, value=not no )
   setGlobalMapAttr( mode, seqIdMap, MTN.messageFollowUp, value=not no )
   setGlobalMapAttr( mode, seqIdMap, MTN.messageDelayResp, value=not no )
   setGlobalMapAttr( mode, seqIdMap, MTN.messageAnnounce, value=not no )

   # Default threshold 1
   setGlobalMapAttr( mode, thresholdMap, MTN.messageSync, value=1 )
   setGlobalMapAttr( mode, thresholdMap, MTN.messageFollowUp, value=1 )
   setGlobalMapAttr( mode, thresholdMap, MTN.messageDelayResp, value=1 )
   setGlobalMapAttr( mode, thresholdMap, MTN.messageAnnounce, value=1 )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold missing-message TYPE INTERVAL intervals
#-------------------------------------------------------------------------------
def setMissingMsgTimer( mode, messageType, threshold=2, no=None ):
   timerMap = 'logMissingMessageTimer'
   thresholdMap = 'missingTimerThresholds'
   if messageType == 'sync':
      setGlobalMapAttr( mode, timerMap, MTN.messageSync, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageSync, threshold )
   if messageType == 'follow-up':
      setGlobalMapAttr( mode, timerMap, MTN.messageFollowUp, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageFollowUp, threshold )
   if messageType == 'announce':
      setGlobalMapAttr( mode, timerMap, MTN.messageAnnounce, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageAnnounce, threshold )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold missing-message TYPE VALUE sequence-ids
#-------------------------------------------------------------------------------
def checkIfMsgsFalse( mode ):
   if not ptpConfig.logMissingMessageSeqId.get( MTN.messageSync, False ) and \
      not ptpConfig.logMissingMessageSeqId.get( MTN.messageAnnounce, False ) and \
      not ptpConfig.logMissingMessageSeqId.get( MTN.messageDelayResp, False ) and \
      not ptpConfig.logMissingMessageSeqId.get( MTN.messageFollowUp, False ):
      setGlobalAttr( mode, 'logMissingMessages', value=False )

def setMissingMsgSeqId( mode, messageType, threshold=2, no=None ):
   setGlobalAttr( mode, 'logMissingMessages', value=True )
   seqIdMap = 'logMissingMessageSeqId'
   thresholdMap = 'missingSequenceThresholds'
   if messageType == 'sync':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageSync, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageSync, threshold )
   if messageType == 'follow-up':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageFollowUp, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageFollowUp, threshold )
   if messageType == 'delay-resp':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageDelayResp, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageDelayResp, threshold )
   if messageType == 'announce':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageAnnounce, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageAnnounce, threshold )
   checkIfMsgsFalse( mode )

def setMissingMsgTimerAndSeqId(
      mode, messageType, timerThreshold=2, seqIdThreshold=2, no=None ):
   setMissingMsgTimer( mode, messageType, timerThreshold, no )
   setMissingMsgSeqId( mode, messageType, seqIdThreshold, no )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold offset-from-master <threshold>
# [no|default] ptp monitor threshold mean-path-delay <threshold>
# [no|default] ptp monitor threshold skew <threshold>
#-------------------------------------------------------------------------------
def setPtpMonitorOffsetThreshold( mode, val=None, no=None ):
   value = val
   if no or value is None:
      value = CliConstants.invalidOffsetFromMasterThreshold

   setGlobalAttr( mode, 'offsetFromMasterThreshold', value=value )

def setPtpMonitorDelayThreshold( mode, val=None, no=None ):
   value = val
   if no or value is None:
      value = CliConstants.invalidMeanPathDelayThreshold

   setGlobalAttr( mode, 'meanPathDelayThreshold', value=value )

def setPtpMonitorSkewThreshold( mode, val=None, no=None ):
   value = val
   if no or value is None:
      value = CliConstants.invalidSkewThreshold

   setGlobalAttr( mode, 'skewThreshold', value=value )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold offset-from-master <threshold> nanoseconds drop
# [no|default] ptp monitor threshold mean-path-delay <threshold> nanoseconds drop
# [no|default] ptp monitor threshold skew <threshold> drop
#-------------------------------------------------------------------------------
def setPtpMonitorOffsetDropThreshold( mode, args ):
   constant = CliConstants.invalidOffsetFromMasterDropThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'offsetFromMasterDropThreshold', value=value )

def setNoPtpMonitorOffsetDropThreshold( mode, args ):
   args[ 'VALUE' ] = CliConstants.invalidOffsetFromMasterDropThreshold
   setPtpMonitorOffsetDropThreshold( mode, args )

def setPtpMonitorDelayDropThreshold( mode, args ):
   constant = CliConstants.invalidMeanPathDelayDropThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'meanPathDelayDropThreshold', value=value )

def setNoPtpMonitorDelayDropThreshold( mode, args ):
   args[ 'VALUE' ] = CliConstants.invalidMeanPathDelayDropThreshold
   setPtpMonitorDelayDropThreshold( mode, args )

def setPtpMonitorSkewDropThreshold( mode, args ):
   constant = CliConstants.invalidSkewDropThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'skewDropThreshold', value=value )

def setNoPtpMonitorSkewDropThreshold( mode, args ):
   args[ 'VALUE' ] = CliConstants.invalidSkewDropThreshold
   setPtpMonitorSkewDropThreshold( mode, args )

#-------------------------------------------------------------------------------
# [no|default] ptp ip access-group <acl-name> in
#-------------------------------------------------------------------------------
def aclNameIs( mode, aclType, aclName ):
   if ptpConfig.serviceAclTypeVrfMap is None:
      ptpConfig.serviceAclTypeVrfMap = ( serviceName, )
   aclKey = Tac.Value( "Acl::AclTypeAndVrfName", aclType, DEFAULT_VRF )
   ptpConfig.serviceAclTypeVrfMap.aclName[ aclKey ] = aclName

def aclNameDel( mode, aclType ):
   if ptpConfig.serviceAclTypeVrfMap is None:
      return
   aclKey = Tac.Value( "Acl::AclTypeAndVrfName", aclType, DEFAULT_VRF )
   del ptpConfig.serviceAclTypeVrfMap.aclName[ aclKey ]

def setIntfDscpGeneral( mode, value=None, no=None ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if value is None or no is not None:
      value = getattr( CliConstants, 'globalDscpGeneralDefault' )
   setattr( intfConfig, 'dscpGeneralConfigured', not no )
   setattr( intfConfig, 'dscpGeneral', value )

def setIntfDscpEvent( mode, value=None, no=None ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if value is None or no is not None:
      value = getattr( CliConstants, 'globalDscpEventDefault' )
   setattr( intfConfig, 'dscpEventConfigured', not no )
   setattr( intfConfig, 'dscpEvent', value )

#-------------------------------------------------------------------------------
# [no|default] ptp announce interval <interval>
#-------------------------------------------------------------------------------
def announceIntervalRange( mode, context ):
   return getBounds( 'logAnnounceInterval' )

#-------------------------------------------------------------------------------
# [no|default] ptp sync-message interval <interval>
#-------------------------------------------------------------------------------
def syncIntervalRange( mode, context ):
   return getBounds( 'logSyncInterval' )

def setSyncInterval( mode, attr, value=None, no=None, deprecatedCmd=False ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )

   # We are limited in the number of timestamped packet we can send (200 pkts/sec)
   # If we have configured PTP to run per VLAN on a trunk interface, we need to
   # prevent that the Sync Interval * Configured VLANs exceeds this limit.
   if value is None:
      setIntfAttr( mode, attr, value, no, deprecatedCmd )
      return

   configuredVlans = len( intfConfig.vlanConfig )
   if ptpConfig.ptpMode == ptpTwoStepBoundaryClockMode and \
         not checkVlanLimit( mode, value, configuredVlans ):
      return

   if ptpG82751Active():
      mode.addWarning( "Sync message rate is fixed to 16/s in PTP profile g8275.1" )

   setIntfAttr( mode, attr, value, no, deprecatedCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp pdelay-req interval <interval>
#-------------------------------------------------------------------------------
def pDelayIntervalRange( mode, context ):
   return getBounds( 'logMinPdelayReqInterval' )

#-------------------------------------------------------------------------------
# [no|default] ptp delay-mechanism p2p|e2e|disabled
#-------------------------------------------------------------------------------
def setDelayMechanism( mode, attr, value=None, no=None, deprecatedCmd=False ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )

   def _isTrunkWithPtpVlan():
      switchIntfConfig = bridgingSic.switchIntfConfig.get( mode.intf.name )
      isIntfTrunkMode = switchIntfConfig is not None and \
                        switchIntfConfig.switchportMode == 'trunk'
      return isIntfTrunkMode and \
             CliConstants.defaultPortDSVlanId not in intfConfig.vlanConfig

   if not no and value == 'p2p' and _isTrunkWithPtpVlan():
      mode.addWarning( "P2P delay mechanism is not supported with PTP configured to"
                       " run per VLANs on a trunk interface" )

   if ptpStatus.unicastNegotiation and value == 'p2p':
      mode.addWarning( "P2P delay mechanism is not supported in PTP G8275.2 "
                       "profile" )

   if ptpG82751Active() and value != 'e2e':
      mode.addWarning( "Only E2E delay mechanism is supported in PTP G8275.1 "
                       "profile" )

   setIntfAttr( mode, attr, value, no, deprecatedCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp transport layer2|ipv4|ipv6
#-------------------------------------------------------------------------------
def setTransportMode( mode, attr, value=None, no=None ):
   # Do nothing if we're already in the transport mode
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if intfConfig.transportMode == value:
      return

   setIntfAttr( mode, attr, value, no )
   # Removing all unicast-negotiation profiles as they are related to the
   # transport mode
   intfConfig.ucastNegGrantorIpToProfileMap.clear()
   intfConfig.ucastNegGranteeIpToProfileMap.clear()

#-------------------------------------------------------------------------------
# [no|default] ptp enable
#-------------------------------------------------------------------------------
def enablePtp( mode, no=False ):
   if ptpConfig.ptpMode == PtpMode.ptpGeneralized and not no and \
      Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( mode.intf.name ):
      mode.addError( "gPTP is not supported on port-channels" )
      return

   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name, createEnabled=not no )
   intfConfig.enabled = not no

#-------------------------------------------------------------------------------
# [no|default] ptp enable synctest (hidden)
#-------------------------------------------------------------------------------
def enableSyncTest( mode, no=False ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if not no:
      synctestPorts = 0
      for portDS in ptpStatus.portDS.values():
         # TODO BUG344577
         if portDS.vlanId != 0:
            continue
         if not portDS.syncTestRole == "syncTestDisabled":
            synctestPorts += 1
      portsAvailable = synctestPorts < 2
      intfConfig.syncTestEnabled = portsAvailable
      if not portsAvailable:
         print( "Reached limit of interfaces that can be configured in synctest "
                "mode" )
         return
   else:
      intfConfig.syncTestEnabled = False

   intfConfig.enabled = not no

def getClockIdentityString( clockIdentity ):
   return Tac.Type( 'Arnet::Eui64' ).eui64( clockIdentity.value.value )

#-------------------------------------------------------------------------------
# [no|default] ptp vlan [ ( all | <vlan-range> ) ]
#-------------------------------------------------------------------------------
def allAllowedVlanIds( mode ):
   switchIntfConfig = bridgingSic.switchIntfConfig.get( mode.intf.name )
   if switchIntfConfig is None:
      return set()
   return computeVlanRangeSet( switchIntfConfig.trunkAllowedVlans )

def allVlanIds():
   # TODO BUG813920 - currently lots of people are reproducing
   # ways of getting 1-4094 as a string or step.
   vlanRanges = VlanCli.allVlanIds( args=[] )
   return computeVlanRangeSet( str( vlanRanges ) )

def getPtpCommandVlanSet( args, mode, allowedOnly, default=None ):
   if 'all' in args:
      if allowedOnly:
         return allAllowedVlanIds( mode )
      else:
         return allVlanIds()
   return args.get( 'VLAN_SET', default )

def setPtpVlan( mode, args ):
   vlanSet = getPtpCommandVlanSet( args, mode, allowedOnly=True )
   no = CliCommand.isNoOrDefaultCmd( args )

   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )

   # By default, when `ptp vlan` is not configured, only one PortDS with vlanId=0 as
   # key is created. This is the defaultVlanId.
   defaultVlanId = CliConstants.defaultPortDSVlanId

   if no and vlanSet is None:
      # `no ptp vlan` revert to the initial state, i.e. only vlanConfig[ 0 ] exists
      intfConfig.vlanConfig.clear()
      intfConfig.vlanIds.clear()
      intfConfig.vlanConfig.newMember( defaultVlanId )
      intfConfig.vlanIds[ defaultVlanId ] = True
      return
   elif no and defaultVlanId in intfConfig.vlanConfig:
      # `no ptp vlan <vlanId>` without previous `ptp vlan` configuration, skipping
      return

   # We are limited in the number of timestamped packet we can send (200 pkts/sec)
   # We need to prevent that the Sync Interval * Configured VLANs exceeds this
   # limit.
   if defaultVlanId not in intfConfig.vlanConfig:
      configuredVlans = len( set( intfConfig.vlanConfig ) | set( vlanSet ) )
   else:
      # We will delete the defaultVlanId entry so just need to count the number
      # of vlans in vlanSet
      configuredVlans = len( vlanSet )

   if ptpConfig.ptpMode == ptpTwoStepBoundaryClockMode and \
         not checkVlanLimit( mode, intfConfig.logSyncInterval, configuredVlans ):
      return

   if defaultVlanId in intfConfig.vlanConfig:
      # `ptp vlan <vlan-range>`

      # Once the user configure with the `ptp vlan` CLI cmd, we are removing the old
      # default behaviour to only send untagged packets. If the user wants to send
      # untagged packets, he'll have to configure `ptp vlan <nativeVlan>` manually
      del intfConfig.vlanConfig[ defaultVlanId ]
      del intfConfig.vlanIds[ defaultVlanId ]

   for vlanId in vlanSet:
      vlan = VlanCli.Vlan( vlanId )
      if no and vlan.id in intfConfig.vlanConfig: # Deleting
         del intfConfig.vlanConfig[ vlan.id ]
         del intfConfig.vlanIds[ vlan.id ]
      elif not no and not vlan.id in intfConfig.vlanConfig: # Adding
         intfConfig.vlanConfig.newMember( vlan.id )
         intfConfig.vlanIds[ vlan.id ] = True

   # residual case of using `no ptp vlan <vlanList>` that deletes all of the vlans,
   # this should revert to normal behavior with the default vlan.
   if not intfConfig.vlanConfig.members():
      intfConfig.vlanConfig.newMember( defaultVlanId )
      intfConfig.vlanIds[ defaultVlanId ] = True

#-------------------------------------------------------------------------------
# ptp boundary ( unicast-negotiation | multicast )
#
# BUG284794 - This is not something needed for now, just comment this out.
#-------------------------------------------------------------------------------
def boundaryUnicastNegotiation( mode, ptpBoundaryModeType ):
   if ptpConfig.ptpMode in ptpBoundaryModes:
      ptpConfig.unicastNegotiation = ptpBoundaryModeType == "unicast-negotiation"
   else:
      mode.addWarning( "In order to configure to unicast-negotiation or multicast, "
                       "you need to be in boundary mode first." )

#-------------------------------------------------------------------------------
# [no|default] ptp mpass
#-------------------------------------------------------------------------------
def setPtpMpass( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   intfConfig.mpassEnabled = not no

#-------------------------------------------------------------------------------
# [no|default] ptp debug [message {(announce|sync|follow-up|delay-req|delay-resp)}]
#              tx-interface INTF
#-------------------------------------------------------------------------------
def setPtpTxIntf( mode, args ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   intf = str( args[ 'INTF' ] )
   for msg in args[ 'MESSAGES' ]:
      intfConfig.txMessageToIntfMap[ msg ] = intf

def unsetPtpTxIntf( mode, args ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   for msg in args[ 'MESSAGES' ]:
      del intfConfig.txMessageToIntfMap[ msg ]

#-------------------------------------------------------------------------------
# [no|default] ptp mpass message sync timeout VALUE messages
#-------------------------------------------------------------------------------
def setMpassSyncTimeout( mode, args ):
   messages = args.get( 'VALUE', CliConstants.mpassDefaultSyncTimeout )
   ptpConfig.mpassSyncTimeout = messages

#-------------------------------------------------------------------------------
# [no|default] ptp mpass message delay-req timeout VALUE seconds
#-------------------------------------------------------------------------------
def setMpassDelayReqTimeout( mode, args ):
   seconds = args.get( 'VALUE', CliConstants.mpassDefaultDelayReqTimeout )
   ptpConfig.mpassDelayReqTimeout = seconds

#-------------------------------------------------------------------------------
# [no|default] ptp diag role isolated-follower
#-------------------------------------------------------------------------------
def enableIsolatedFollower( mode, no=False ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if not no:
      intfConfig.isolatedSlaveEnabled = True
      if not intfConfig.isolatedSlaveVlanId:
         intfConfig.isolatedSlaveVlanId[ CliConstants.defaultPortDSVlanId ] = True
   else:
      intfConfig.isolatedSlaveEnabled = False

#-------------------------------------------------------------------------------
# [no|default] ptp diag isolated-follower vlan VLAN_SET
#-------------------------------------------------------------------------------
def enableIsolatedFollowerVlan( mode, args ):
   vlanSet = args.get( 'VLAN_SET' )
   no = CliCommand.isNoOrDefaultCmd( args )
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )

   defaultVlanId = CliConstants.defaultPortDSVlanId

   if no and not vlanSet:
      intfConfig.isolatedSlaveVlanId.clear()
      intfConfig.isolatedSlaveVlanId[ defaultVlanId ] = True
      return

   for vlanId in vlanSet:
      vlan = VlanCli.Vlan( vlanId )
      if no:
         del intfConfig.isolatedSlaveVlanId[ vlan.id ]
      else:
         intfConfig.isolatedSlaveVlanId[ vlan.id ] = True

   # restore the default vlanId if isolatedSlave is still enabled
   if intfConfig.isolatedSlaveEnabled:
      intfConfig.isolatedSlaveVlanId[ defaultVlanId ] = True

#-------------------------------------------------------------------------------
# [no|default] ptp free-running time-source SOURCE
#-------------------------------------------------------------------------------
def setFreeRunningTimeSource( mode, args ):
   ptpConfig.syncFromSystemTime = 'system' in args

#===============================================================================
#   Show Commands
#===============================================================================
guardedPtpShowPartial = partial( CliCommand.guardedKeyword, 'ptp',
                           helpdesc='Show Precision Time Protocol information' )
ptpMatcherForShow = guardedPtpShowPartial( guard=ptpSupportedGuard )

# This is for timesync CLI which do not need to depend on PTP supported guard.
# They will be guarded at the next token by timeSync guard
ptpMatcherForPlatformShow = CliMatcher.KeywordMatcher( 'ptp',
      helpdesc='Show platform specific Precision Time Protocol information' )

ptpMatcherForDomainNumber = CliMatcher.IntegerMatcher( 0, 255,
   helpdesc='PTP Domain Number' )

#-------------------------------------------------------------------------------
# show ptp hold ptp time interval
#-------------------------------------------------------------------------------
def showHoldPtpTime( mode, args ):
   model = PtpHoldPtpTimeForShow()
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return model
   model.ptpMode = ptpStatus.defaultDS.ptpMode
   model.holdPtpTimeInterval = None
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return model
   model.holdPtpTimeInterval = ptpStatus.defaultDS.holdPtpTimeInterval
   model.holdPtpTimeState = ptpStatus.holdoverState
   model.holdPtpTimeSinceEpoch = ptpStatus.holdoverSinceEpoch
   model.holdPtpTimeExpiry = ptpStatus.holdoverExpiry
   return model

#-------------------------------------------------------------------------------
# show ptp local-clock
#-------------------------------------------------------------------------------
def showClock( mode ):
   ptpClockModel = PtpClock()
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return ptpClockModel
   ptpClockModel.ptpMode = ptpStatus.defaultDS.ptpMode
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return ptpClockModel
   ptpOrdinaryOrBC = None
   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock',
                                       'ptpOneStepBoundaryClock',
                                       'ptpOrdinaryMasterClock',
                                       'ptpGeneralized' ]:
      ptpOrdinaryOrBC = PtpClock.PtpOrdinaryOrBoundaryClock()
      ptpOrdinaryOrBC.clockIdentity = getClockIdentityString(
            ptpStatus.defaultDS.clockIdentity )
      ptpOrdinaryOrBC.domainNumber = ptpStatus.defaultDS.domainNumber
      ptpOrdinaryOrBC.numberPorts = ptpStatus.defaultDS.numberPorts
      ptpOrdinaryOrBC.priority1 = ptpStatus.defaultDS.priority1
      ptpOrdinaryOrBC.priority2 = ptpStatus.defaultDS.priority2
      if ptpStatus.defaultDS.ptpMode in ptpBoundaryModes and \
         ptpConfig.ptpProfile in [ PtpProfile.ptpG8275_1,
                                   PtpProfile.ptpG8275_2 ]:
         ptpOrdinaryOrBC.localPriority = ptpStatus.defaultDS.localPriority
      clockClass = ptpStatus.defaultDS.clockQuality.clockClass
      accuracy = ptpStatus.defaultDS.clockQuality.clockAccuracy
      offsetScaledLogVariance = \
          ptpStatus.defaultDS.clockQuality.offsetScaledLogVariance.value
      clockQuality = PtpClockQuality()
      clockQuality.clockClass = clockClass
      clockQuality.accuracy = accuracy
      clockQuality.offsetScaledLogVariance = offsetScaledLogVariance
      ptpOrdinaryOrBC.clockQuality = clockQuality
      if ptpStatus.defaultDS.ptpMode == 'ptpOrdinaryMasterClock':
         ptpClockModel.ptpOrdinaryOrBoundaryClock = ptpOrdinaryOrBC
         return ptpClockModel
      bcProperties = PtpClock.PtpOrdinaryOrBoundaryClock.PtpBoundaryClockProperties()
      bcProperties.offsetFromMaster = int( ptpStatus.currentDS.offsetFromMaster )
      bcProperties.meanPathDelay = int( ptpStatus.currentDS.meanPathDelay )
      bcProperties.stepsRemoved = ptpStatus.currentDS.stepsRemoved.value
      if ptpStatus.defaultDS.ptpMode != 'ptpGeneralized':
         bcProperties.skew = ptpStatus.currentDS.skew
      else:
         bcProperties.neighborRateRatio = ptpStatus.currentDS.neighborRateRatio
         bcProperties.gptpRateRatio = ptpStatus.currentDS.gptpRateRatio
         bcProperties.gptpCumulativeRateOffset = \
             ptpStatus.currentDS.gptpCumulativeRateOffset
      if ptpStatus.currentDS.lastSyncLocalTime:
         masterTime = int( ptpStatus.currentDS.lastSyncLocalTime -
                           ptpStatus.currentDS.offsetFromHardware )
         seconds = masterTime // 1000000000
         if ptpStatus.timePropertiesDS.ptpTimescale and \
                ptpStatus.timePropertiesDS.currentUtcOffsetValid:
            seconds = seconds - ptpStatus.timePropertiesDS.currentUtcOffset.value
         bcProperties.lastSyncTime = int( seconds )
         estimatedTime = seconds + Tac.now() - ptpStatus.currentDS.lastSyncSystemTime
         bcProperties.currentPtpSystemTime = int( estimatedTime )
      ptpOrdinaryOrBC.boundaryClockProperties = bcProperties
   ptpClockModel.ptpOrdinaryOrBoundaryClock = ptpOrdinaryOrBC
   return ptpClockModel

#-------------------------------------------------------------------------------
# show ptp ip access-list
#-------------------------------------------------------------------------------
def showPtpIpAcl( mode, args ):
   return AclCli.showServiceAcl( mode, ptpConfig.serviceAclTypeVrfMap,
                                 ptpStatus.aclStatusService, aclCheckpoint, 'ip',
                                 ( None, None ), supressVrf=True,
                                 vrfUnaware=True )

#-------------------------------------------------------------------------------
# show ptp masters
#-------------------------------------------------------------------------------
def showParent( mode, args ):
   # if the PTP agent never ran, defaultDs is None
   parentWrapper = PtpParentWrapper()
   if not ptpStatus.defaultDS:
      parentWrapper.message = 'PTP is not configured'
      return parentWrapper

   if ptpStatus.defaultDS.ptpMode not in [ 'ptpBoundaryClock',
                                           'ptpOneStepBoundaryClock',
                                           'ptpGeneralized' ]:
      parentWrapper.message = 'Not a boundary clock, no parent clock information.'
      return parentWrapper

   parent = PtpParent()
   clockIdentity = \
         ptpStatus.parentDS.parentSourceAddrs.parentPortIdentity.clockIdentity
   parent.parentClockIdentity = getClockIdentityString( clockIdentity )
   parent.parentPortNumber = \
         ptpStatus.parentDS.parentSourceAddrs.parentPortIdentity.portNumber
   if ptpStatus.parentDS.parentSourceAddrs.parentIpAddr != \
         CliConstants.parentIpAddrDefault:
      if ptpStatus.parentDS.parentSourceAddrs.parentIpAddr.af == 'ipv4':
         parent.parentIpAddr = \
               ptpStatus.parentDS.parentSourceAddrs.parentIpAddr.v4Addr
      else:
         parent.parentIp6Addr = \
               ptpStatus.parentDS.parentSourceAddrs.parentIpAddr.v6Addr
   parent.parentTwoStepFlag = ptpStatus.parentTwoStepFlag
   clockIdentity = ptpStatus.parentDS.grandmasterIdentity
   parent.gmClockIdentity = getClockIdentityString( clockIdentity )
   gmClockQuality = PtpClockQuality()
   gmClockQuality.clockClass = ptpStatus.parentDS.grandmasterClockQuality.clockClass
   gmClockQuality.accuracy = ptpStatus.parentDS.grandmasterClockQuality.clockAccuracy
   gmClockQuality.offsetScaledLogVariance = \
       ptpStatus.parentDS.grandmasterClockQuality.offsetScaledLogVariance.value
   parent.gmClockQuality = gmClockQuality
   parent.gmPriority1 = ptpStatus.parentDS.grandmasterPriority1
   parent.gmPriority2 = ptpStatus.parentDS.grandmasterPriority2
   parentWrapper.parent = parent
   return parentWrapper

#-------------------------------------------------------------------------------
# show ptp unicast-negotiation candidate-grantor|remote-grantee
#-------------------------------------------------------------------------------
def showCandidateGrantors( mode, args ):
   ucastNegCandidateGrantors = UcastNegCandidateGrantor()
   for intf in sorted( getConfigIntfs() ):
      intfConfig = ptpConfig.intfConfig.get( intf )
      key = Tac.newInstance( "Ptp::PortDSKey", intf,
                             CliConstants.defaultPortDSVlanId )
      portDS = ptpStatus.portDS.get( key )
      if not intfConfig or not portDS:
         continue

      profileMap = intfConfig.ucastNegGrantorIpToProfileMap
      for ip, profile in sorted( profileMap.items() ):
         entry = UcastNegCandidateGrantorEntry()
         entry.intf = intf
         entry.ipAddr = ip
         entry.profileName = profile
         if ip in portDS.g8275_2.signalFail:
            entry.grantorState = UcastNegGrantorState.ucastNegBlacklisted
         elif ( portDS.portState == 'psSlave' and
                ptpStatus.parentDS.parentSourceAddrs.parentIpAddr == ip ):
            entry.grantorState = UcastNegGrantorState.ucastNegMaster
         else:
            entry.grantorState = UcastNegGrantorState.ucastNegCandidateMaster
         ucastNegCandidateGrantors.candidateGrantors.append( entry )
   return ucastNegCandidateGrantors

def showRemoteGrantees( mode, args ):
   ucastNegRemoteGrantees = UcastNegRemoteGrantee()
   for intf in sorted( getConfigIntfs() ):
      intfConfig = ptpConfig.intfConfig.get( intf )
      if not intfConfig:
         continue

      profileMap = intfConfig.ucastNegGranteeIpToProfileMap
      for ipAndMask, profile in sorted( profileMap.items() ):
         entry = UcastNegRemoteGranteeEntry()
         entry.intf = intf
         entry.ipAddrAndMask = ipAndMask
         entry.profileName = profile
         ucastNegRemoteGrantees.remoteGrantees.append( entry )
   return ucastNegRemoteGrantees

#-------------------------------------------------------------------------------
# show ptp unicast-negotiation profile [<name>]
#-------------------------------------------------------------------------------
def showUcastNegProfiles( mode, args ):
   profileName = args.get( 'PROFILE' )
   profileKeys = None
   profileList = {}
   if profileName is None:
      profileKeys = list( ptpConfig.ucastNegProfile )
   else:
      profileKeys = profileName.split()

   profiles = ptpConfig.ucastNegProfile
   for profileKey in profileKeys:
      tmp = UcastNegProfile()
      if profileKey in profiles:
         profile = profiles.get( profileKey )
         tmp.announceInterval = profile.announceProfile.logInterval
         tmp.announceDuration = profile.announceProfile.duration
         tmp.syncInterval = profile.syncProfile.logInterval
         tmp.syncDuration = profile.syncProfile.duration
         tmp.delayResponseInterval = profile.delayRespProfile.logInterval
         tmp.delayResponseDuration = profile.delayRespProfile.duration
         profileList[ profileKey ] = tmp

   return UcastNegProfiles( profiles=profileList )

#-------------------------------------------------------------------------------
# show ptp unicast-negotiation granted|requested
#-------------------------------------------------------------------------------
def getUcastGrantsAndRequests( granted ):
   interfaces = {}
   for intf in getConfigIntfs():
      key = Tac.newInstance( "Ptp::PortDSKey", intf,
                             CliConstants.defaultPortDSVlanId )
      portDS = ptpStatus.portDS.get( key )
      if not portDS or portDS.g8275_2 is None:
         continue

      ucastNegMessages = UcastNegMessages()
      messagesIpv4 = []
      messagesIpv6 = []
      if granted:
         messageStatus = { ucastKey:ucastStatus for ucastKey, ucastStatus in
                           ucastNegStatus.grant.items()
                           if ucastKey.intf == intf }
      else:
         messageStatus = { ucastKey:ucastStatus for ucastKey, ucastStatus in
                           ucastNegStatus.request.items()
                           if ucastKey.intf == intf }
      sigFail = portDS.g8275_2.signalFail
      for ucastKey, ucastStatus in messageStatus.items():
         if ucastKey.ipAddr in sigFail:
            continue
         tmp = UcastNegMessage()
         tmp.ipAddr = ucastKey.ipAddr
         tmp.messageType = ucastKey.messageType
         tmp.logInterval = ucastStatus.logInterval
         tmp.duration = ucastStatus.duration
         tmp.expirationTime = ucastStatus.expirationTime - Tac.now()
         if tmp.ipAddr.af == "ipv4":
            messagesIpv4.append( tmp )
         else:
            messagesIpv6.append( tmp )

      key = attrgetter( "ipAddr" )
      messagesIpv4 = sorted( messagesIpv4, key=key )
      messagesIpv6 = sorted( messagesIpv6, key=key )
      messages = messagesIpv4 + messagesIpv6

      ucastNegMessages.messages = messages
      interfaces[ intf ] = ucastNegMessages
   return UcastNegStatus( interfaces=interfaces )

def showGrantedSlaves( mode, args ):
   return getUcastGrantsAndRequests( True )

def showRequestedMasters( mode, args ):
   return getUcastGrantsAndRequests( False )

#-------------------------------------------------------------------------------
# show ptp time-property
#-------------------------------------------------------------------------------
def showTimeProperty( mode, args ):
   # if the PTP agent never ran, defaultDs is None
   timePropertyWrapper = PtpTimePropertyWrapper()
   if not ptpStatus.defaultDS:
      timePropertyWrapper.message = 'PTP is not configured'
      return timePropertyWrapper

   if ptpStatus.defaultDS.ptpMode not in [ 'ptpBoundaryClock',
                                           'ptpOneStepBoundaryClock',
                                           'ordinaryMaster', 'ptpGeneralized' ]:
      timePropertyWrapper.message = \
          "Not a boundary clock, no time-property information."
      return timePropertyWrapper

   timePropertiesDS = ptpStatus.timePropertiesDS
   timeProperty = PtpTimeProperty()
   timeProperty.currentUtcOffsetValid =  timePropertiesDS.currentUtcOffsetValid
   timeProperty.currentUtcOffset = timePropertiesDS.currentUtcOffset.value
   timeProperty.leap59 = timePropertiesDS.leap59
   timeProperty.leap61 = timePropertiesDS.leap61
   timeProperty.timeTraceable = timePropertiesDS.timeTraceable
   timeProperty.frequencyTraceable = timePropertiesDS.frequencyTraceable
   timeProperty.ptpTimescale = timePropertiesDS.ptpTimescale
   timeProperty.timeSource = timePropertiesDS.timeSource
   timePropertyWrapper.timeProperty = timeProperty
   return timePropertyWrapper


#---------------------------------------------------------------------------------
# show ptp interface [ INTFS ] counters drop
#---------------------------------------------------------------------------------
def showIntfDropCounters( mode, args ):

   disabled = not ptpStatus.defaultDS or ptpStatus.defaultDS.ptpMode == 'ptpDisabled'

   if disabled:
      return PtpDebugCounters( disabled=disabled )

   intfs = args.get( 'INTFS', [] )
   intfsAll = not intfs

   counters = { intf : PtpDebugIntfCounters() for intf in intfs }
   for key, value in debugCounter.counter.items():
      # retrive the PTP port from the portKeyMap
      portHashKey = Tac.Value( "Ptp::DebugCounter::CounterPortHashKey",
                               key.counterPortKeyHash )
      port = debugCounter.counterPortKeyMap[ portHashKey ].counterPortKey
      # only accept this counter if interface is specified
      # or when no interface is provided ( default to all interfaces)
      # no need to show sent/recv counter
      if port.intfId in intfs or intfsAll:
         endpoint = PtpEndpoint()
         endpoint.intf = port.intfId
         endpoint.vlanId = port.vlanId
         endpoint.domain = port.domain
         endpoint.ip = port.ip
         dropReason = PtpDebugCounterDropReason()
         dropReason.debugType = key.type
         dropReason.debugReason = key.reason
         direction = key.direction
         lastSeen = value.lastSeen
         counter = PtpDebugCounter( endpoint=endpoint, dropReason=dropReason,
                                    direction=direction, lastSeen=lastSeen,
                                    count=value.count )
         if intfsAll:
            value = counters.setdefault( port.intfId, PtpDebugIntfCounters() )
         else:
            value = counters[ port.intfId ]
         value.intfCounters.append( counter )

   return PtpDebugCounters( counters=counters )

#-------------------------------------------------------------------------------
# show ptp interface [<interface list>] counters
#-------------------------------------------------------------------------------
def getPktCounter( portDS, messageType, sentOrRecv ):
   # pkt counters should be saved with counterPortKeyConstant domain and Ip,
   # modify to support showing different domain/ip
   intf = Tac.Value( "Arnet::IntfId", portDS.intf )
   vlanId = Tac.Value( "Bridging::VlanIdOrNone", portDS.vlanId )
   ip = CounterPortKeyConstant.ip()
   domain = CounterPortKeyConstant.domain
   portKeyHash = Tac.hashOf( Tac.Value( "Ptp::DebugCounter::CounterPortKey",
                                        intf, vlanId, domain, ip ) )

   direction = CounterDir.rx
   reason = CounterReason.received
   if sentOrRecv == 'Sent':
      direction = CounterDir.tx
      reason = CounterReason.sent

   key = Tac.Value( 'Ptp::DebugCounter::CounterReasonKey', portKeyHash,
                    direction, messageType, reason )
   counterValue = pktCounter.counter.get( key )
   return counterValue.count if counterValue else 0

def showIntfCounters( mode, portDS ):
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return PtpIntfCountersWrapper()

   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpIntfCountersWrapper()

   if not portDS.enabled:
      if ptpStatus.defaultDS.ptpMode != PtpMode.ptpGeneralized:
         return PtpIntfCountersWrapper()
      # In gptp, port is disabled if asCapable is false, but we
      # still want to show the counters
      for disabledReason in portDS.disabledReason:
         if( disabledReason != 'asCapableFalse' and
             portDS.disabledReason[disabledReason] ):
            return PtpIntfCountersWrapper()


   intfCounters = PtpIntfCounters()
   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock',
                                       'ptpGeneralized' ]:
      intfCounters.announceTx = \
         getPktCounter( portDS, CounterType.messageAnnounce, 'Sent' )
      intfCounters.announceRx = \
         getPktCounter( portDS, CounterType.messageAnnounce, 'Received' )
   intfCounters.syncTx = getPktCounter( portDS, CounterType.messageSync, 'Sent' )
   intfCounters.syncRx = getPktCounter( portDS, CounterType.messageSync, 'Received' )
   intfCounters.followUpTx = \
      getPktCounter( portDS, CounterType.messageFollowUp, 'Sent' )
   intfCounters.followUpRx = \
      getPktCounter( portDS, CounterType.messageFollowUp, 'Received' )
   intfCounters.delayReqTx = \
      getPktCounter( portDS, CounterType.messageDelayReq, 'Sent' )
   intfCounters.delayReqRx = \
      getPktCounter( portDS, CounterType.messageDelayReq, 'Received' )
   intfCounters.delayRespTx = \
      getPktCounter( portDS, CounterType.messageDelayResp, 'Sent' )
   intfCounters.delayRespRx = \
      getPktCounter( portDS, CounterType.messageDelayResp, 'Received' )
   intfCounters.pDelayReqTx = \
      getPktCounter( portDS, CounterType.messagePDelayReq, 'Sent' )
   intfCounters.pDelayReqRx = \
      getPktCounter( portDS, CounterType.messagePDelayReq, 'Received' )
   intfCounters.pDelayRespTx = \
      getPktCounter( portDS, CounterType.messagePDelayResp, 'Sent' )
   intfCounters.pDelayRespRx = \
      getPktCounter( portDS, CounterType.messagePDelayResp, 'Received' )
   intfCounters.pDelayRespFollowUpTx = \
       getPktCounter( portDS, CounterType.messagePDelayRespFollowup, 'Sent' )
   intfCounters.pDelayRespFollowUpRx = \
       getPktCounter( portDS, CounterType.messagePDelayRespFollowup, 'Received' )
   if ptpStatus.defaultDS.ptpMode in ptpBoundaryModes:
      intfCounters.managementTx = \
         getPktCounter( portDS, CounterType.messageManagement, 'Sent' )
      intfCounters.managementRx = \
         getPktCounter( portDS, CounterType.messageManagement, 'Received' )
      intfCounters.signalingTx = \
         getPktCounter( portDS, CounterType.messageSignaling, 'Sent' )
      intfCounters.signalingRx = \
         getPktCounter( portDS, CounterType.messageSignaling, 'Received' )


   if ptpStatus.defaultDS.ptpMode == 'ptpGeneralized':
      intfCounters.gptpFollowUpMessagesTxMissed = \
         portDS.gptpFollowUpMessagesSentMissed

   return PtpIntfCountersWrapper( interfaceCounters=intfCounters,
                                  vlanId=getVlanId( portDS ) )

def showIntfMpassCounters( mode, portDS ):
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return PtpIntfCountersWrapper()

   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpIntfCountersWrapper()

   intfCounters = PtpIntfMpassCounters()
   intfCounters.announceLikeTx = (
      getPktCounter( portDS, CounterType.messageAnnounceLike, 'Sent' ) )
   intfCounters.announceLikeRx = (
      getPktCounter( portDS, CounterType.messageAnnounceLike, 'Received' ) )
   intfCounters.syncLikeTx = (
      getPktCounter( portDS, CounterType.messageSyncLike, 'Sent' ) )
   intfCounters.syncLikeRx = (
      getPktCounter( portDS, CounterType.messageSyncLike, 'Received' ) )
   intfCounters.followUpLikeTx = (
      getPktCounter( portDS, CounterType.messageFollowUpLike, 'Sent' ) )
   intfCounters.followUpLikeRx = (
      getPktCounter( portDS, CounterType.messageFollowUpLike, 'Received' ) )
   intfCounters.delayReqLikeTx = (
      getPktCounter( portDS, CounterType.messageDelayReqLike, 'Sent' ) )
   intfCounters.delayReqLikeRx = (
      getPktCounter( portDS, CounterType.messageDelayReqLike, 'Received' ) )
   intfCounters.delayRespLikeTx = (
      getPktCounter( portDS, CounterType.messageDelayRespLike, 'Sent' ) )
   intfCounters.delayRespLikeRx = (
      getPktCounter( portDS, CounterType.messageDelayRespLike, 'Received' ) )

   return PtpIntfCountersWrapper( interfaceCounters=intfCounters,
                                  vlanId=getVlanId( portDS ) )

def showIntfListCounters( mode, args, mpass=False ):
   intfs = args.get( 'INTFS' )
   if not ptpStatus.defaultDS:
      return PtpCounters( interfaceCounters={} )
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpCounters( interfaceCounters={} )

   vlan = getattr( args.get( 'VLAN_ID' ), 'id', None )
   ports = getDsPorts( intfs, vlan )
   intfCounters = {}
   for port in ports:
      portDS = ptpStatus.portDS.get( port )
      if not portDS:
         continue
      intf = portDS.intf
      if mpass:
         intfConfig = ptpConfig.intfConfig.get( intf )
         if not intfConfig or not intfConfig.mpassEnabled:
            continue
         if not ptpStatus.intfStatus.get( port ):
            continue
         intfCounter = showIntfMpassCounters( mode, portDS )
      else:
         intfCounter = showIntfCounters( mode, portDS )
      if intfCounter.interfaceCounters is not None:
         if intf not in intfCounters:
            intfCounters[ intf ] = PtpIntfVlansCounters( interface=intf )
         intfCounters[ intf ].ptpInterfaceVlansCounters.append( intfCounter )
   return PtpCounters( interfaceCounters=intfCounters )

#-------------------------------------------------------------------------------
# show ptp mpass interface [<interface list>] counters
#-------------------------------------------------------------------------------
def showIntfListMpassCounters( mode, args ):
   return showIntfListCounters( mode, args, mpass=True )

#-------------------------------------------------------------------------------
# show ptp interface [<interface list>] vlans
#-------------------------------------------------------------------------------

def _getConfiguredVlans( intf ):
   ''' This function returns 3 lists and 1 boolean
   vlans: Actual PTP configured vlans, where ptp packets will be sent or received
   notCreatedVlans: Vlans that are not created ( received pkts will be dropped )
   notAllowedVlans: Vlans not allowed in trunk port
   '''
   ( vlans, notCreatedVlans, notAllowedVlans )  = ( [], [], [] )
   for portDSKey in ptpStatus.vlanStatus:
      if portDSKey.intf != intf:
         continue

      # If the intf is not in trunk mode, just don't display anything
      switchIntfConfig = bridgingSic.switchIntfConfig.get( portDSKey.intf )
      isIntfTrunkMode = switchIntfConfig is not None and \
                        switchIntfConfig.switchportMode == 'trunk'
      if not isIntfTrunkMode:
         continue

      vs = ptpStatus.vlanStatus.get( portDSKey )
      vlanId = portDSKey.vlanId
      # If vlanId == 0, we have not configured any PTP VLAN to run and no other
      # PortDSKey of this interface exist.
      if vlanId == 0: # pylint: disable=no-else-continue
         continue
      elif vs.trunkAllowed and vs.vlanCreated:
         vlans.append( vlanId )
      else:
         if not vs.vlanCreated:
            notCreatedVlans.append( vlanId )
         if not vs.trunkAllowed:
            notAllowedVlans.append( vlanId )
   return ( vlans, notCreatedVlans, notAllowedVlans )

def showIntfListVlans( mode, args ):
   intfs = args.get( 'INTFS', { key.intf for key in ptpStatus.vlanStatus } )
   if not ptpStatus.defaultDS:
      return PtpVlans( interfaces={}, message="PTP is not configured" )
   if ptpStatus.defaultDS.ptpMode not in ptpBoundaryModes:
      return PtpVlans( interfaces={},
                       message="Not a boundary clock, no PTP vlan information" )

   ptpIntfVlans = {}
   for intf in intfs:
      ( vlans, notCreatedVlans, notAllowedVlans ) = _getConfiguredVlans( intf )
      if not vlans and not notCreatedVlans and not notAllowedVlans:
         continue

      ptpIntfVlans[ intf ] = PtpVlansExtendedWrapper(
                                vlans=vlans,
                                notCreatedVlans=notCreatedVlans,
                                notAllowedVlans=notAllowedVlans )
   return PtpVlans( interfaces=ptpIntfVlans )

#-------------------------------------------------------------------------------
# show ptp interface [ INTFS ] vlan <vlanId>
#-------------------------------------------------------------------------------
def showIntfListVlan( mode, args ):
   intfs = args[ 'INTFS' ]
   vlanId = args[ 'VLAN_ID' ]
   if not ptpStatus.defaultDS:
      return PtpInterfaces( interfaces={} )

   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpInterfaces( interfaces={} )

   ports = getDsPorts( intfs, vlanId.id )
   ptpIntfs = {}
   for port in ports:
      portDS = ptpStatus.portDS.get( port )
      if not portDS:
         continue
      intf = portDS.intf
      ptpIntf = showIntf( mode, portDS, None )
      if ptpIntf.interface is not None:
         if intf not in ptpIntfs:
            ptpIntfs[ intf ] = PtpIntfVlans( interface=intf )
         ptpIntfs[ intf ].ptpInterfaceVlansStates.append( ptpIntf )
   return PtpInterfaces( interfaces=ptpIntfs )

#-------------------------------------------------------------------------------
# show ptp interface [<interface list>]
#-------------------------------------------------------------------------------
def getAsCapableLastChanged( lastUpdateTime ):
   if lastUpdateTime == 0:
      return 'Never'
   lastMoveDelta = int( Tac.utcNow() - lastUpdateTime )
   return str( datetime.timedelta( seconds=lastMoveDelta ) )

def showIntf( mode, portDS, enabled, intfStatus=None ):
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return PtpIntfWrapper()

   if portDS.adminDisabled and enabled:
      return PtpIntfWrapper()

   ptpIntf = PtpIntf()
   ptpIntf.adminDisabled = portDS.adminDisabled
   ptpIntf.syncTestRole = portDS.syncTestRole
   ptpIntf.ptpMode = ptpStatus.defaultDS.ptpMode
   ptpIntf.counters = showIntfCounters( mode, portDS )
   if ptpStatus.defaultDS.ptpMode == 'ptpEndToEndTransparentClock':
      return PtpIntfWrapper( interface=ptpIntf )

   # The 'ptp vlan' cmd only works in boundary mode
   # According to gPTP spec ( 11.3.3 ) IEEE 802.1AS message shall not have a VLAN tag
   ptpIntf.vlanId = getVlanId( portDS )

   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock',
                                       'ptpOneStepBoundaryClock',
                                       'ptpGeneralized' ]:
      ptpIntf.portState = portDS.portState
      if ptpStatus.defaultDS.ptpMode == PtpMode.ptpGeneralized:

         ptpIntf.asCapable = portDS.asCapable
         ptpIntf.timeSinceAsCapableLastChanged = \
             getAsCapableLastChanged( portDS.lastUpdateTimeAsCapable )
         if ptpIntf.timeSinceAsCapableLastChanged != 'Never':
            ptpIntf.timeSinceAsCapableLastChanged += ' ago'

         if portDS.portState == 'psMaster':
            ptpIntf.lastGptpResidenceTime = portDS.lastGptpResidenceTime
            ptpIntf.minGptpResidenceTime = portDS.minGptpResidenceTime
            ptpIntf.maxGptpResidenceTime = portDS.maxGptpResidenceTime
      if not ptpStatus.unicastNegotiation:
         ptpIntf.syncInterval =  math.pow( 2, portDS.logSyncInterval )
         ptpIntf.announceInterval = math.pow( 2, portDS.logAnnounceInterval )
         ptpIntf.announceReceiptTimeout = portDS.announceReceiptTimeout
      if ptpStatus.defaultDS.ptpMode == 'ptpGeneralized':
         ptpIntf.syncReceiptTimeout = portDS.syncReceiptTimeout
   ptpIntf.delayMechanism = portDS.delayMechanism
   if portDS.delayMechanism == 'e2e' and \
      ptpStatus.defaultDS.ptpMode in ptpBoundaryModes and \
      not ptpStatus.unicastNegotiation:
      ptpIntf.delayReqInterval = math.pow(
         2, portDS.logMinDelayReqInterval )
   elif portDS.delayMechanism == 'p2p':
      ptpIntf.peerDelayReqInterval = math.pow(
         2, portDS.logMinPdelayReqInterval )
      ptpIntf.peerPathDelay = portDS.peerMeanPathDelay

   if Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( portDS.intf ):
      ptpIntf.mpassEnabled = intfStatus.mpassEnabled if intfStatus else False
      ptpIntf.mpassStatus = intfStatus.mpassStatus == 'active' if intfStatus \
            else True
      ptpIntf.mpassDisabledReason = intfStatus.mpassDisabledReason if intfStatus \
            else 'notConfigured'

   if ptpConfig.ptpProfile in [ PtpProfile.ptpG8275_1, PtpProfile.ptpG8275_2 ] and \
      ptpStatus.defaultDS.ptpMode in ptpBoundaryModes:
      ptpIntf.localPriority = portDS.localPriority
      if ptpG82751Active():
         intfConfig = ptpConfig.intfConfig.get( portDS.intf )
         if intfConfig is not None:
            ptpIntf.mcastTxAddr = intfConfig.pktTxMacAddress

   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock',
                                       'ptpOneStepBoundaryClock',
                                       'ptpGeneralized' ]:
      ptpIntf.transportMode = portDS.transportMode

   if Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( portDS.intf ) and \
         portDS.txLagMember is not None:
      ptpIntf.txLagMember = portDS.txLagMember

   return PtpIntfWrapper( interface=ptpIntf )

def showMpassIntf( mode, portDS, intfStatus ):
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return PtpIntfWrapper()

   ptpIntf = PtpMpassIntf()
   ptpIntf.syncTestRole = portDS.syncTestRole
   ptpIntf.counters = showIntfMpassCounters( mode, portDS )
   ptpIntf.syncExpiry = intfStatus.mpassSyncExpiry
   ptpIntf.delayReqExpiry = intfStatus.mpassDelayReqExpiry
   # The 'ptp vlan' cmd only works in boundary mode
   # According to gPTP spec ( 11.3.3 ) IEEE 802.1AS message shall not have a VLAN tag
   ptpIntf.vlanId = getVlanId( portDS )
   ptpIntf.portState = portDS.portState
   ptpIntf.mpassEnabled = intfStatus.mpassEnabled
   ptpIntf.mpassStatus = ( intfStatus.mpassStatus == 'active' )
   ptpIntf.mpassDisabledReason = intfStatus.mpassDisabledReason

   return PtpIntfWrapper( interface=ptpIntf )

def showIntfList( mode, args, mpass=False ):
   intfs = args.get( 'INTFS' )
   enabled = 'enabled' in args
   if not ptpStatus.defaultDS:
      return PtpInterfaces( interfaces={} )

   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpInterfaces( interfaces={} )

   ports = getDsPorts( intfs )
   ptpIntfs = {}
   for port in ports:
      portDS = ptpStatus.portDS.get( port )
      if not portDS:
         continue
      intf = portDS.intf
      if mpass:
         intfConfig = ptpConfig.intfConfig.get( intf )
         if not intfConfig or not intfConfig.mpassEnabled:
            continue
         intfStatus = ptpStatus.intfStatus.get( port )
         if not intfStatus:
            continue
         ptpIntf = showMpassIntf( mode, portDS, intfStatus )
      else:
         intfStatus = ptpStatus.intfStatus.get( port )
         ptpIntf = showIntf( mode, portDS, enabled, intfStatus )
      if ptpIntf.interface is not None:
         if intf not in ptpIntfs:
            ptpIntfs[ intf ] = PtpIntfVlans( interface=intf )
         ptpIntfs[ intf ].ptpInterfaceVlansStates.append( ptpIntf )
   return PtpInterfaces( interfaces=ptpIntfs )

#-------------------------------------------------------------------------------
# show mpass ptp interface [<interface list>]
#-------------------------------------------------------------------------------
def showMpassIntfList( mode, args ):
   return showIntfList( mode, args, mpass=True )

#-------------------------------------------------------------------------------
# show ptp
#-------------------------------------------------------------------------------
def showIntfSummary( ptpSummary, mode, portDS, mpass=True ):
   if not ptpStatus.defaultDS:
      return
   if portDS.adminDisabled:
      return
   intfSummary = PtpIntfSummary()
   intfSummary.transportMode = portDS.transportMode
   if mode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock', 'ptpGeneralized' ]:
      intfSummary.portState = portDS.portState
      intfSummary.delayMechanism = portDS.delayMechanism
      if mode == 'ptpGeneralized':
         intfSummary.asCapable = portDS.asCapable
         intfSummary.timeSinceAsCapableLastChanged = \
             getAsCapableLastChanged( portDS.lastUpdateTimeAsCapable )
         intfSummary.lastGptpResidenceTime = portDS.lastGptpResidenceTime
         intfSummary.peerPathDelay = portDS.peerMeanPathDelay
         intfSummary.neighborRateRatio = portDS.neighborRateRatio
      elif portDS.vlanId != 0:
         intfSummary.vlanId = portDS.vlanId
   if mpass:
      intfStatus = ptpStatus.intfStatus.get( portDS.key )
      if not intfStatus:
         return
      intfSummary.mpassEnabled = intfStatus.mpassEnabled
      intfSummary.mpassStatus = intfStatus.mpassStatus
      intfSummary.vlanId = portDS.vlanId

   intf = portDS.intf
   if intf not in ptpSummary.ptpIntfSummaries:
      ptpSummary.ptpIntfSummaries[ intf ] = PtpIntfVlanSummary(
         interface=intf )
   ptpSummary.ptpIntfSummaries[ intf ].ptpIntfVlanSummaries.append( intfSummary )

def showPtpSummaryInternals( mode, args ):
   ptpSummary = PtpSummary()
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return ptpSummary
   ptpSummary.ptpMode = ptpStatus.defaultDS.ptpMode
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return ptpSummary
   ptpClockSummary = None

   ptpProfile = 'ptpDefaultProfile'
   if ptpConfig.ptpProfile == 'ptpG8275_2':
      if ptpSummary.ptpMode in ptpBoundaryModes:
         ptpProfile = 'ptpG8275_2'
   elif ptpConfig.ptpProfile == 'ptpG8275_1':
      if ptpSummary.ptpMode in ptpBoundaryModes and ptpG82751BCSupported():
         ptpProfile = 'ptpG8275_1'
      elif ptpSummary.ptpMode in ptpTransparentModes and ptpG82751TCSupported():
         ptpProfile = 'ptpG8275_1'
   elif ptpConfig.ptpProfile == 'ptpDefaultProfile':
      ptpProfile = 'ptpDefaultProfile'

   # If the active profile is distinct from the selected one, note it
   if ptpProfile != ptpConfig.ptpProfile:
      ptpSummary.ptpSelectedProfile = ptpConfig.ptpProfile

   # Log the active PTP profile in modes that support multiple profiles
   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock',
                                    'ptpEndToEndTransparentClock',
                                    'ptpOneStepEndToEndTransparentClock' ]:
      ptpSummary.ptpProfile = ptpProfile

   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock',
                                       'ptpGeneralized' ]:
      ptpClockSummary = ptpSummary.PtpClockSummary()
      ptpClockSummary.numberOfSlavePorts = 0
      ptpClockSummary.numberOfMasterPorts = 0
      clock = showClock( ptpStatus.defaultDS.ptpMode ).ptpOrdinaryOrBoundaryClock
      ptpClockSummary.clockIdentity = clock.clockIdentity
      ptpClockSummary.gmClockIdentity = getClockIdentityString(
         ptpStatus.parentDS.grandmasterIdentity )
      ptpClockSummary.stepsRemoved = clock.boundaryClockProperties.stepsRemoved
      ptpClockSummary.meanPathDelay = clock.boundaryClockProperties.meanPathDelay
      if ptpSummary.ptpMode in ptpBoundaryModes:
         ptpClockSummary.skew = clock.boundaryClockProperties.skew
         ptpClockSummary.lastSyncTime = clock.boundaryClockProperties.lastSyncTime
         ptpClockSummary.currentPtpSystemTime = \
             clock.boundaryClockProperties.currentPtpSystemTime
         ptpClockSummary.offsetFromMaster = \
             clock.boundaryClockProperties.offsetFromMaster
      else:
         ptpClockSummary.gptpRateRatio = clock.boundaryClockProperties.gptpRateRatio
         ptpClockSummary.neighborRateRatio = \
             clock.boundaryClockProperties.neighborRateRatio
   ptpSummary.ptpClockSummary = ptpClockSummary

   ptpSummary.ptpIntfSummaries = {}
   for portDS in ptpStatus.portDS.values():
      intf = portDS.intf
      if ptpClockSummary:
         if portDS.portState == 'psMaster':
            ptpClockSummary.numberOfMasterPorts += 1
         elif portDS.portState == 'psSlave':
            ptpClockSummary.numberOfSlavePorts += 1
            ptpClockSummary.slavePort = intf
            ptpClockSummary.slaveVlanId = portDS.vlanId
      showIntfSummary( ptpSummary, ptpSummary.ptpMode, portDS )
   return ptpSummary

def showPtpSummary( mode, args ):
   # BUG749203/BUG749423: if PTP mode changes during the command, some data is
   # inaccessible, command fails. In this case we can just retry immediately.
   try:
      return showPtpSummaryInternals( mode, args )
   except AttributeError:
      return showPtpSummaryInternals( mode, args )

#-------------------------------------------------------------------------------
# show ptp mpass
#-------------------------------------------------------------------------------
def showPtpMpassSummary( mode, args ):
   ptpMpassSummary = PtpMpassSummary()
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS or ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return ptpMpassSummary

   ptpMpassSummary.ptpIntfSummaries = {}
   for portDS in ptpStatus.portDS.values():
      intf = portDS.intf
      intfConfig = ptpConfig.intfConfig.get( intf )
      if intfConfig and intfConfig.mpassEnabled:
         showIntfSummary(
            ptpMpassSummary, ptpStatus.defaultDS.ptpMode, portDS, mpass=True )
   if ptpMpassSummary.ptpIntfSummaries:
      ptpMpassSummary.syncTimeout = ptpConfig.mpassSyncTimeout
      ptpMpassSummary.delayReqTimeout = ptpConfig.mpassDelayReqTimeout
   return ptpMpassSummary

ptpMatcherForExec = CliCommand.guardedKeyword(
   'ptp',
   helpdesc='Precision Time Protocol',
   guard=ptpSupportedGuard )

# This is for timesync CLI which do not need to depend on PTP supported guard.
# They will be guarded at the next token by timeSync guard
ptpMatcherForReset = CliMatcher.KeywordMatcher(
   'ptp',
   helpdesc='Precision Time Protocol' )

#-------------------------------------------------------------------------------
# "clear ptp interface [<interface list>] counters" in enable mode
#-------------------------------------------------------------------------------
def clearCounters( mode, intfs=None, vlanId=None ):
   # if intfs is None, clean all intf
   # if vlanId is None clean all vlanId
   if intfs:
      intfs = set( intfs )
   if vlanId:
      vlanId = vlanId.id

   # fill clearLists with the pkt counter and drop counter to clear
   counters = [ debugCounter, pktCounter ]
   counterClearLists = [ ptpConfig.dropCounterClearList,
                         ptpConfig.pktCounterClearList ]

   for ( counter, counterClearList ) in zip( counters, counterClearLists ):
      for key in counter.counter:
         portHashKey = Tac.Value( "Ptp::DebugCounter::CounterPortHashKey",
                                  key.counterPortKeyHash )
         portKey = counter.counterPortKeyMap[ portHashKey ].counterPortKey
         if ( intfs is None or portKey.intfId in intfs ) and \
            ( vlanId is None or portKey.vlanId == vlanId ):
            counterClearList.addKey( key )

   # clear the counters
   ptpConfig.clearCountersRequest = True

   try:
      Tac.waitFor(
         lambda: ptpStatus.clearCountersRequestCompleted,
         description='PTP clear interface counter request to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      print( 'PTP counters may not have been reset yet' )

   for counterClearList in counterClearLists:
      counterClearList.clearList()
   ptpConfig.clearCountersRequest = False

def clearIntfCounters( mode, portDS ):
   intfConfig = ptpConfig.intfConfig.get( portDS.intf )
   if not intfConfig:
      return
   assert hasattr( intfConfig, 'clearCountersRequestTime' )
   intfConfig.clearCountersRequestTime = Tac.now()
   try:
      Tac.waitFor(
         lambda:
            portDS.lastClearTime >= intfConfig.clearCountersRequestTime,
         description='PTP clear interface counter request to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      print( 'PTP interface', portDS.intf, 'counters may not have been reset yet' )

def clearIntfListCounters( mode, args ):
   intfs = args.get( 'INTFS' )
   vlanId = args.get( 'VLAN_ID' )
   if not ptpStatus.defaultDS:
      return
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return

   vlan = getattr( args.get( 'VLAN_ID' ), 'id', None )
   ports = getDsPorts( intfs, vlanId=vlan )
   for port in ports:
      portDS = ptpStatus.portDS.get( port )
      if not portDS:
         continue
      clearIntfCounters( mode, portDS )
   # clear the drop counter and pkt counter
   clearCounters( mode, intfs=intfs, vlanId=vlanId )

class ClearIntfListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ptp interface [ INTFS ] [ vlan VLAN_ID ] counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ptp': ptpMatcherForExec,
      'interface': 'Details on PTP',
      'INTFS': IntfCli.Intf.rangeMatcher,
      'vlan': 'Show PTP VLAN information',
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'counters': 'Show PTP counter information',
   }
   handler = clearIntfListCounters

BasicCliModes.EnableMode.addCommandClass( ClearIntfListCountersCmd )

#-------------------------------------------------------------------------------
# clear ptp ip access-list counters
#-------------------------------------------------------------------------------
def clearPtpIpAclCounters( mode, args ):
   AclCli.clearServiceAclCounters( mode, ptpStatus.aclStatusService,
                                   aclCheckpoint, 'ip' )

class ClearPtpIpAccessListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ptp ip access-list counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ptp': ptpMatcherForExec,
      'ip': AclCli.ipKwForClearServiceAclMatcher,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
      'counters': AclCli.countersKwMatcher,
   }
   handler = clearPtpIpAclCounters

BasicCliModes.EnableMode.addCommandClass( ClearPtpIpAccessListCountersCmd )

#-------------------------------------------------------------------------------
# show ptp foreign-master-record
#-------------------------------------------------------------------------------
def showFMR( mode, args ):
   # if the PTP agent never ran, defaultDS is None
   if not ptpStatus.defaultDS:
      return PtpForeignMasterRecords( message='PTP is not configured',
                                      intfForeignMasterRecords={} )

   if ptpStatus.defaultDS.ptpMode not in ptpBoundaryModes:
      return PtpForeignMasterRecords(
         message='Not a boundary clock, no foreign master information.',
         intfForeignMasterRecords={} )

   keyList = []
   for key in ptpStatus.intfStatus:
      intfStatus = ptpStatus.intfStatus.get( key )
      if intfStatus.foreignMasterDS:
         keyList.append( key )

   if not keyList:
      return PtpForeignMasterRecords( intfForeignMasterRecords={} )

   keyList.sort()

   ptpForeignMasterRecords = {}
   for key in keyList:
      intfStatus = ptpStatus.intfStatus.get( key )
      ptpIntfForeignMasterRecords = PtpIntfForeignMasterRecords()
      ptpIntfForeignMasterRecords.maxForeignRecords = intfStatus.maxForeignRecords
      for record in intfStatus.foreignMasterDS.values():
         portIdentity = PtpPortIdentity()
         clockIdentity = record.foreignMasterPortIdentity.clockIdentity
         portIdentity.clockIdentity = getClockIdentityString( clockIdentity )
         portIdentity.portNumber = record.foreignMasterPortIdentity.portNumber
         foreignMasterRecord = PtpForeignMasterDS()
         foreignMasterRecord.foreignMasterPortIdentity = portIdentity
         foreignMasterRecord.foreignMasterIpAddr = None
         foreignMasterRecord.foreignMasterIp6Addr = None
         if record.foreignMasterIpAddr != CliConstants.foreignMasterIpAddrDefault:
            if record.foreignMasterIpAddr.af == 'ipv4':
               foreignMasterRecord.foreignMasterIpAddr = \
                  record.foreignMasterIpAddr.v4Addr
            else:
               foreignMasterRecord.foreignMasterIp6Addr = \
                  record.foreignMasterIpAddr.v6Addr
         msgs = record.foreignMasterAnnounceMessages
         foreignMasterRecord.foreignMasterAnnounceMessages = msgs
         foreignMasterRecord.timeAgo = record.updateTime
         ptpIntfForeignMasterRecords.foreignMasterRecords.append(
            foreignMasterRecord )
         if key.vlanId != 0:
            ptpIntfForeignMasterRecords.vlanId = key.vlanId
         if key.intf not in ptpForeignMasterRecords:
            ptpForeignMasterRecords[ key.intf ] = PtpIntfVlanForeignMasterRecords(
                  interface=key.intf )
      if key.intf in ptpForeignMasterRecords:
         ptpForeignMasterRecords[ key.intf ].intfForeignMasterRecords \
                                            .append( ptpIntfForeignMasterRecords )
   return PtpForeignMasterRecords( intfForeignMasterRecords=ptpForeignMasterRecords )

#-------------------------------------------------------------------------------
# show ptp monitor
# show ptp isolated-follower monitor interface INTF [ vlan VLAN_ID ]
#-------------------------------------------------------------------------------
def getPtpMonitorData( mode, isolatedFollowerIntf=None, isolatedFollowerVlan=0 ):
   ptpMonitorData = PtpMonitorData()

   if not ptpStatus.defaultDS:
      return ptpMonitorData

   ptpMonitorData.monitorEnabled = ptpConfig.monitorEnabled
   ptpMonitorData.ptpMode = ptpStatus.defaultDS.ptpMode
   if ( ptpConfig.offsetFromMasterThreshold !=
        CliConstants.invalidOffsetFromMasterThreshold ):
      ptpMonitorData.offsetFromMasterThreshold = ptpConfig.offsetFromMasterThreshold
   if ptpConfig.meanPathDelayThreshold != CliConstants.invalidMeanPathDelayThreshold:
      ptpMonitorData.meanPathDelayThreshold = ptpConfig.meanPathDelayThreshold
   if ptpConfig.skewThreshold != CliConstants.invalidSkewThreshold:
      ptpMonitorData.skewThreshold = ptpConfig.skewThreshold

   if ptpMonitorData.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock',
                                  'ptpGeneralized' ]:
      if not agentIsRunning( mode.sysname, 'Ptp' ):
         mode.addError( 'Ptp agent is not running' )
         return None

      output = io.StringIO()
      command = "BC_SHOW_MONITOR"
      if togglePtpIsolatedSlaveEnabled() and isolatedFollowerIntf is not None:
         command = "BC_SHOW_MONITOR_ISOLATEDFOLLOWER %s %d" % \
                   ( isolatedFollowerIntf, isolatedFollowerVlan )

      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            Ptp.agentName(), "ShowPtpCallback",
                                            command=command, timeout=30,
                                            stringBuff=output )

      # Parse the output from the agent
      # First line contains single integer repesenting the number of data, N
      # Each of the next N lines contains interface, real last sync time, offset
      # fro master, mean path delay and skew in order separated by whitespace.
      output.seek( 0, 0 )
      while True:
         line = output.readline()
         if not line or 'Cli connection' in line:
            break

         numberOfRecords = int( line.strip() )
         records = []
         for _ in range( numberOfRecords ):
            line = output.readline().strip()
            slave, timestamp, offset, delay, skew, seqId = line.split( ' ' )

            entry = PtpMonitorDataEntry()
            entry.intf = Tac.Value( 'Arnet::IntfId', slave )
            entry.realLastSyncTime = int( timestamp )
            entry.offsetFromMaster = int( offset )
            entry.meanPathDelay = int( delay )
            entry.skew = float( skew )
            entry.lastSyncSeqId = int( seqId )
            records.append( entry )
         records.sort( key=lambda entry: -entry.realLastSyncTime )
         ptpMonitorData.ptpMonitorData = records

   return ptpMonitorData

def showPtpMonitor( mode, args ):
   return getPtpMonitorData( mode )

def showIsolatedFollowerMonitor( mode, args ):
   intf = args[ "INTF" ]
   vlan = getattr( args.get( "VLAN_ID" ), "id", 0 )
   return getPtpMonitorData( mode, intf.name, vlan )

#-------------------------------------------------------------------------------
# show ptp source
# show ptp source ip
#-------------------------------------------------------------------------------
def showSrcIp( mode, args ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return None
   if not ptpStatus.defaultDS:
      return PtpSourceIp()
   return PtpSourceIp( sourceIp=ptpStatus.defaultDS.srcIp4,
                       sourceIp6=ptpStatus.defaultDS.srcIp6 )

# We are keeping the existing command `show ptp source ip` which only displays IPv4
# address
def showSrcIp4( mode, args ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return None
   if not ptpStatus.defaultDS:
      return PtpSourceIp()
   return PtpSourceIp( sourceIp=ptpStatus.defaultDS.srcIp4 )

# ---------------------------------------------------------------------------------
# show ptp region domain-numbers [ DOMAIN_NUMBER ] [ interface INTF ]
# ---------------------------------------------------------------------------------

def showPtpDomainNumbers( mode, args ):
   summary = PtpDomainNumbersSummary()

   if not ptpStatus.enabled:
      summary.setPtpConfigured( False )
      summary.clockDomain = None
      return summary

   summary.setPtpConfigured( True )

   targetIntfs = args.get( 'INTFS', [] )

   summary.clockDomain = ptpStatus.domainStatus.globalDomain

   targetDomainNumbers = args.get( 'DOMAIN_NUMBER', [] )
   if targetDomainNumbers:
      # pylint: disable-next=unnecessary-comprehension
      targetDomainNumbers = [ num for num in targetDomainNumbers.values() ]
      # Clock domain should match all interfaces
      # Make list empty to enforce no domain number filtering
      if summary.clockDomain in targetDomainNumbers:
         targetDomainNumbers = []

   intfDomains = ptpStatus.domainStatus.intfDomainNumberInfo

   for portDSKey, domainNumberInfo in intfDomains.items():
      intfName = portDSKey.intf
      domainNumber = domainNumberInfo.activeRegionDomain

      if not domainNumber:
         continue

      if targetIntfs and intfName not in targetIntfs:
         continue

      if targetDomainNumbers and domainNumber not in targetDomainNumbers:
         continue

      if intfName not in summary.interfaces:
         domainsVlans = PtpDomainNumbersSummary.DomainNumbersVlans()
         summary.interfaces[ intfName ] = domainsVlans

      domainsVlans = summary.interfaces[ intfName ]

      vlanId = portDSKey.vlanId

      if vlanId and vlanId not in domainsVlans.vlans:
         domainList = PtpDomainNumbersSummary.DomainNumbersVlans.DomainNumberList()
         domainsVlans.vlans[ vlanId ] = domainList

      # The model for domain-numbers is that if they are on the default vlanID 0,
      # they are not for a "ptp vlan" and apply to the native vlan only
      # Otherwise, they are only used for the particular "ptp vlan"
      if vlanId:
         vlanDomainNumbers = domainsVlans.vlans[ vlanId ].domainNumbers
         vlanDomainNumbers.append( domainNumber )
      else:
         domainsVlans.domainNumbers.append( domainNumber )

   if not summary.interfaces:
      summary.clockDomain = None

   return summary

#---------------------------------------------------------------------------------
# Register to show tech-support
#---------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-02-11 12:53:04',
   cmds=[ 'show ptp interface enabled' ],
   cmdsGuard=ptpSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2014-01-25 10:34:23',
   cmds=[ 'show ptp' ],
   cmdsGuard=ptpSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2016-12-19 09:21:00',
   cmds=[ 'show ptp local-clock', 'show ptp masters' ],
   cmdsGuard=ptpSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2018-02-20 10:38:54',
   cmds=[ 'show ptp monitor' ],
   cmdsGuard=ptpSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2019-04-15 14:08:40',
   cmds=[ 'show ptp unicast-negotiation profile',
          'show ptp unicast-negotiation candidate-grantor',
          'show ptp unicast-negotiation remote-grantee' ],
   cmdsGuard=ptpSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-05-26 22:15:47',
   cmds=[ 'show ptp interface counter drop' ],
   cmdsGuard=ptpSupported )

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2021-03-02 22:15:47',
   cmds=[ 'show ptp mpass',
          'show ptp mpass interface' ],
   cmdsGuard=ptpSupported )

# pkgdeps: import CliPlugin.ShellCli (for bash shell)
anyPtp4lLog = "/var/log/ptp4l*"
ptp4lActiveLog = "/var/log/ptp4l*.log"
ptp4lRotatedLog = "/var/log/ptp4l*.log*.gz"
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
    '2023-02-02 17:44:11',
    # include last 1000 lines, unzipping if required
    # ptp4l writes to the .log file, and logrotate daemon will occasionally
    # gz-and-rename when the file becomes large enough. For example:
    #   ptp4l@Requester.log
    #   ptp4l@Requester.log.date1.gz
    #   ptp4l@Requester.log.date2.gz
    # we would like to print the current .log file, then the latest log, 2nd latest
    # log, and so on..., capping it to 1000
    # older files will be printed at the end (and be filtered by tail)
    # we don't expect more than one requester to be in use at a time
    cmds=[ f'bash zcat -f {ptp4lRotatedLog} {ptp4lActiveLog} | tac | head -n 1000' ],
    # don't run command unless there is at least 1 log file
    cmdsGuard=lambda: bool( glob.glob( anyPtp4lLog ) ) )

def Plugin( entityManager ):
   global ptpConfig, ptpStatus, ptpHwStatusDir
   global ptpHwCapabilityDir
   global brConfig, bridgingSic, aclCheckpoint, ucastNegStatus
   global debugCounter, pktCounter
   sem = SharedMem.entityManager( sysdbEm=entityManager )
   ucastNegStatusInfo = Smash.mountInfo( 'reader' )
   ucastNegStatus = sem.doMount( "ptp/status/unicastnegotiation/",
                                 "Ptp::UcastNegStatusSmash",
                                 ucastNegStatusInfo )
   ptpConfig = ConfigMount.mount( entityManager, "ptp/config", "Ptp::Config", "w" )
   ptpStatus = LazyMount.mount( entityManager, "ptp/status", "Ptp::Status", "rO" )
   ptpHwCapabilityDir = LazyMount.mount(
         entityManager, "ptp/hwCapability", "Tac::Dir", "ri" )
   ptpHwStatusDir = LazyMount.mount( entityManager,
                                     "ptp/hwStatus", "Tac::Dir", "ri" )
   aclCheckpoint = LazyMount.mount( entityManager, "ptp/aclCheckpoint",
                                    "Acl::CheckpointStatus", "w" )
   brConfig = LazyMount.mount( entityManager, "bridging/config",
                               "Bridging::Config", "r" )
   bridgingSic = LazyMount.mount( entityManager, "bridging/switchIntfConfig",
                                  "Bridging::SwitchIntfConfigDir", "r" )
   debugCounterInfo = Smash.mountInfo( 'reader' )
   debugCounter = sem.doMount( "ptp/debug/counter",
                               "Ptp::DebugCounter::DebugCounter",
                               debugCounterInfo )
   pktCounterInfo = Smash.mountInfo( 'reader' )
   pktCounter = sem.doMount( "ptp/pkt/counter",
                             "Ptp::DebugCounter::DebugCounter",
                             pktCounterInfo )
