# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import math
from operator import attrgetter
from time import gmtime, strftime
from functools import reduce

import Arnet
import ArnetModel
import Ark
import CliModel
from IntfModels import Interface
import MultiRangeRule
import PtpLib
import TableOutput
import Tac

ptpModeToName = {
   'ptpDisabled' : 'Disabled',
   'ptpBoundaryClock' : 'Boundary Clock',
   'ptpEndToEndTransparentClock' : 'E2E Transparent Clock',
   'ptpPeerToPeerTransparentClock' : 'P2P Transparent Clock',
   'ptpOrdinaryMasterClock' : 'Ordinary Master Clock',
   'ptpGeneralized' : 'gPTP - Generalized PTP Clock',
   'ptpOneStepEndToEndTransparentClock' : 'One-Step E2E Transparent Clock',
   'ptpOneStepBoundaryClock' : 'One-Step Boundary Clock',
}

messageTypeToName = {
   'none' : 'None',
   'messageSync' : 'Sync',
   'messageDelayReq' : 'Delay Request',
   'messagePDelayReq' : 'Peer Delay Request',
   'messagePDelayResp' : 'Peer Delay Response',
   'messageFollowUp' : 'Follow up',
   'messageDelayResp' : 'Delay Response',
   'messagePDelayRespFollowup' : 'Peer Delay Response Follow up',
   'messageAnnounce' : 'Announce',
   'messageSignaling' : 'Signalling',
   'messageManagement' : 'Management',
   'messageSyncLike' : 'MPASS peer-forwarded Sync',
   'messageDelayReqLike' : 'MPASS peer-forwarded Delay Request',
   'messageFollowUpLike' : 'MPASS peer-forwarded Follow up',
   'messageDelayRespLike' : 'MPASS peer-forwarded Delay Response',
   'messageAnnounceLike' : 'MPASS peer-forwarded Announce',
}

ptpProfileToName = {
   'ptpDefaultProfile' : 'Default ( IEEE1588 )',
   'ptpG8275_1' : 'G8275.1',
   'ptpG8275_2' : 'G8275.2',
}

candidateTypes = [
   'candidate-grantor',
   'remote-grantee'
]

portStateToName = {
   'psInitializing' : 'Initializing',
   'psFaulty' : 'Faulty',
   'psDisabled' : 'Disabled',
   'psListening' : 'Listening',
   'psPreMaster' : 'PreMaster',
   'psMaster' : 'Master',
   'psPassive' : 'Passive',
   'psUncalibrated' : 'Uncalibrated',
   'psSlave' : 'Slave',
   'psIsolatedSlave' : 'Isolated Follower'
}

synctestStateToName = {
   'syncTestMaster' : 'master',
   'syncTestSlave' : 'slave',
   'syncTestDisabled' : 'disabled'
}

ucastNegGrantorState = Tac.Type( 'Ptp::UcastNegGrantorState' )
ucastNegGrantorStateToName = {
   ucastNegGrantorState.ucastNegMaster: 'Master',
   ucastNegGrantorState.ucastNegCandidateMaster: 'Candidate Master',
   ucastNegGrantorState.ucastNegBlacklisted: 'Blacklisted',
}

transportModeToDisplayStr = {
   'layer2' : 'layer 2',
   'ipv4' : 'ipv4',
   'ipv6' : 'ipv6',
}

timeSourceToName = {
   16 : 'ATOMIC_CLOCK',
   32 : 'GPS',
   48 : 'TERRESTRIAL_RADIO',
   64 : 'PTP',
   80 : 'NTP',
   96 : 'HAND_SET',
   112 : 'OTHER',
   160 : 'INTERNAL_OSCILLATOR'
}

mpassDisabledReasonToName = {
   'notConfigured' : 'MPASS not configured',
   'notMlagInterface' : 'Interface not part of MLAG',
   'ptpDisabledInterface' : 'PTP not enabled on interface',
   'ptpDisabledPeerLink' : 'PTP not enabled on MLAG peer-link',
   'notBoundaryMode' : 'PTP not in boundary mode',
   'delayMechanismNotE2e' : 'Delay mechanism not end to end',
   'ucastNegEnabled' : 'Unicast negotiation enabled',
   'mlagIntfDisabled' : 'MLAG interface disabled'
}

ptpHoldoverStates = [
   "holdoverMode",
   "holdoverInit",
   "holdoverMasterDetected",
   "holdoverTimerExpired",
]

MpassStatus = Tac.Type( "Ptp::Mpass::Status" )
MpassDisabledReason = Tac.Type( "Ptp::Mpass::DisabledReason" )

CliConstants = Tac.Type( 'Ptp::Constants' )

PTP_NOT_CONFIGURED = 'PTP is not configured'

class PtpHoldPtpTimeForShow( CliModel.Model ):
   ptpMode = CliModel.Enum( help="PTP mode",
                            values=list( ptpModeToName ),
                            optional=True )

   holdPtpTimeInterval = CliModel.Int( help="hold-ptp-time interval in seconds.",
                                       optional=True )
   holdPtpTimeState = CliModel.Enum( help="current Ptp hold-over mode.",
                                     values=ptpHoldoverStates,
                                     optional=True )
   holdPtpTimeSinceEpoch = CliModel.Float( help="Timestamp of when Ptp entered"
                                                " hold-over mode.",
                                           optional=True )
   holdPtpTimeExpiry = CliModel.Float( help="Timestamp of when Ptp is expected to"
                                            " exit hold-over mode.",
                                       optional=True )
   def render( self ):
      if self.ptpMode is None:
         print( PTP_NOT_CONFIGURED )
         return
      print( 'PTP Mode:', ptpModeToName[ self.ptpMode ] )

      holdoverActive =\
         self.holdPtpTimeState == Tac.Type("Ptp::HoldoverState").holdoverMode
      if holdoverActive:
         try:
            beginTime = strftime( '%Y-%m-%d UTC %H:%M:%S',
                                  gmtime( self.holdPtpTimeSinceEpoch ) )
         except OverflowError:
            beginTime = 'ERROR, timestamp too large'
         try:
            expiryTime = strftime( '%Y-%m-%d UTC %H:%M:%S',
                                   gmtime( self.holdPtpTimeExpiry ) )
         except OverflowError:
            expiryTime = 'ERROR, timestamp too large'
      else:
         beginTime = 'N/A'
         expiryTime = 'N/A'

      if self.holdPtpTimeInterval is not None:
         print( 'PTP hold-ptp-time interval:', self.holdPtpTimeInterval )
         print( 'PTP holdover begin time:', beginTime )
         print( 'PTP holdover expiry time:', expiryTime )

class PtpClockQuality( CliModel.Model ):
   clockClass = CliModel.Int(
      help="Clock class. See IEEE 1588-2008 for more information." )
   accuracy = CliModel.Int(
      help="Clock accuracy. See IEEE 1588-2008 for more information." )
   offsetScaledLogVariance = CliModel.Int(
      help="Clock stability. See IEEE 1588-2008 for more information." )

   def render( self ):
      print( '   Class:', self.clockClass )
      print( '   Accuracy: 0x%02x' % self.accuracy )
      print( '   OffsetScaledLogVariance: 0x%02x' % self.offsetScaledLogVariance )

class PtpClock( CliModel.Model ):
   class PtpOrdinaryOrBoundaryClock( CliModel.Model ):
      class PtpBoundaryClockProperties( CliModel.Model ):
         offsetFromMaster = CliModel.Int(
            help="CurrentDS offset from master in nanoseconds" )
         meanPathDelay = CliModel.Int(
            help="CurrentDS mean path delay to master in nanoseconds" )
         stepsRemoved = CliModel.Int( help="CurrentDS steps removed from GM" )
         skew = CliModel.Float( help="Ratio of master's seconds to local seconds",
                                optional=True,
                                default=None )
         lastSyncTime = CliModel.Int(
            help="Last time local clock synced from master",
            optional=True,
            default=None )
         currentPtpSystemTime = CliModel.Int(
            help="Current sytem time measured with PTP",
            optional=True,
            default=None )
         neighborRateRatio = CliModel.Float(
            help="Estimated gPTP neighbor rate ratio",
            optional = True,
            default = None )
         gptpRateRatio = CliModel.Float(
            help="Estimated ratio of gPTP grandmaster to local clock frequency",
            optional = True,
            default = None )
         gptpCumulativeRateOffset = CliModel.Int(
            help="Estimated rate ratio expressed as an integer",
            optional = True,
            default = None )
         def render( self ):
            print( 'Offset From Master (nanoseconds):', self.offsetFromMaster )
            print( 'Mean Path Delay: %d nanoseconds' % self.meanPathDelay )
            print( 'Steps Removed:', self.stepsRemoved )
            if self.skew is not None:
               print( 'Skew:', self.skew )
            if self.lastSyncTime is not None:
               try:
                  date = strftime( '%H:%M:%S UTC %b %d %Y',
                                   gmtime( self.lastSyncTime ) )
                  print( 'Last Sync Time:', date )
               except OverflowError:
                  print( 'Last Sync Time: ERROR, timestamp too large' )
            if self.currentPtpSystemTime is not None:
               try:
                  estimatedDate = strftime( '%H:%M:%S UTC %b %d %Y',
                                            gmtime( self.currentPtpSystemTime ) )
                  print( 'Current PTP System Time:', estimatedDate )
               except OverflowError:
                  print( 'Current PTP System Time: ERROR, timestamp too large' )

            if self.neighborRateRatio is not None:
               print( 'Neighbor Rate Ratio:', self.neighborRateRatio )
            if self.gptpRateRatio is not None:
               print( 'Rate Ratio:', self.gptpRateRatio )
            if self.gptpCumulativeRateOffset is not None:
               print( 'Cumulative Rate Ratio:', self.gptpCumulativeRateOffset )

      clockIdentity = CliModel.Str( help="DefaultDS clock identity" )
      domainNumber = CliModel.Int( help="DefaultDS domain number" )
      numberPorts = CliModel.Int( help="DefaultDS number ports" )
      priority1 = CliModel.Int( help="DefaultDS priority1" )
      priority2 = CliModel.Int( help="DefaultDS priority2" )
      localPriority = CliModel.Int( help="DefaultDS local priority. "
                                         "See ITU-T G8275 for more information.",
                                    optional=True,
                                    default=None )
      clockQuality = CliModel.Submodel( help="DefaultDS clock quality",
                                        valueType=PtpClockQuality )
      boundaryClockProperties = CliModel.Submodel(
         help="Boundary clock properties",
         optional=True,
         default=None,
         valueType=PtpBoundaryClockProperties )

      def render( self ):
         print( 'Clock Identity:', self.clockIdentity )
         print( 'Clock Domain:', self.domainNumber )
         print( 'Number of PTP ports:', self.numberPorts )
         print( 'Priority1:', self.priority1 )
         print( 'Priority2:', self.priority2 )
         if self.localPriority is not None:
            print( 'Local Priority:', self.localPriority )
         print( 'Clock Quality:' )
         self.clockQuality.render()
         if self.boundaryClockProperties is not None:
            self.boundaryClockProperties.render()

   ptpMode = CliModel.Enum( help="PTP mode",
                            values=list( ptpModeToName ),
                            optional=True,
                            default=None )
   ptpOrdinaryOrBoundaryClock = CliModel.Submodel(
      help="Information for ordinary or boundary clocks",
      optional=True,
      default=None,
      valueType=PtpOrdinaryOrBoundaryClock )

   def render( self ):
      ptpMode = 'ptpDisabled' if self.ptpMode is None else self.ptpMode
      print( 'PTP Mode:', ptpModeToName[ ptpMode ] )
      if self.ptpOrdinaryOrBoundaryClock is not None:
         self.ptpOrdinaryOrBoundaryClock.render()

class UcastNegCandidateGrantorEntry( CliModel.Model ):
   intf = Interface(
      help="Interface with which the candidate grantor is negotiating" )
   ipAddr = ArnetModel.IpGenericAddress(
      help="IP address of the candidate grantor" )
   profileName = CliModel.Str(
      help="Profile with which the candidate grantor is associated" )
   grantorState = CliModel.Enum(
      help="State of the candidate grantor.",
      values=list( ucastNegGrantorStateToName ) )

class UcastNegCandidateGrantor( CliModel.Model ):
   candidateGrantors = CliModel.List(
      help="List of candidate grantors",
      valueType=UcastNegCandidateGrantorEntry )

   def render( self ):
      headings = ( "Interface", "Address", "Profile", "Grantor Status" )
      table = TableOutput.createTable( ( h, 'l' ) for h in headings )
      colWidths = [ 10, 45, 15, 20 ]
      colFmt = [ TableOutput.Format( maxWidth=w, wrap=True ) for w in colWidths ]
      table.formatColumns( *colFmt )

      for candidateGrantor in self.candidateGrantors:
         intf = candidateGrantor.intf.shortName
         ipAddr = candidateGrantor.ipAddr
         profileName = candidateGrantor.profileName
         grantorState = ucastNegGrantorStateToName[ candidateGrantor.grantorState ]
         table.newRow( intf, ipAddr, profileName, grantorState )

      print( table.output() )

class UcastNegRemoteGranteeEntry( CliModel.Model ):
   intf = Interface(
      help="Interface with which the remote grantee can negotiate" )
   ipAddrAndMask = ArnetModel.IpGenericAddrAndMask(
      help="Allowed IP address range for the remote grantee" )
   profileName = CliModel.Str(
      help="Profile with which the remote grantee is associated" )

class UcastNegRemoteGrantee( CliModel.Model ):
   remoteGrantees = CliModel.List(
      help="List of remote grantees",
      valueType=UcastNegRemoteGranteeEntry )

   def render( self ):
      headings = ( "Interface", "Address Range", "Profile" )
      table = TableOutput.createTable( ( h, 'l' ) for h in headings )
      colWidths = [ 10, 50, 15 ]
      colFmt = [ TableOutput.Format( maxWidth=w, wrap=True ) for w in colWidths ]
      table.formatColumns( *colFmt )

      for remoteGrantee in self.remoteGrantees:
         intf = remoteGrantee.intf.shortName
         ipAddrAndMask = remoteGrantee.ipAddrAndMask
         profileName = remoteGrantee.profileName
         table.newRow( intf, ipAddrAndMask, profileName )

      print( table.output() )

class UcastNegProfile( CliModel.Model ):
   announceInterval = CliModel.Int( help="Announce interval of time of profile. "
      "See IEEE 1588-2008 for more information." )
   announceDuration = CliModel.Int( help="Announce duration of time of profile. "
      "See IEEE 1588-2008 for more information." )
   syncInterval = CliModel.Int( help="Sync interval of time of profile. "
      "See IEEE 1588-2008 for more information." )
   syncDuration = CliModel.Int( help="Sync duration of time of profile. "
      "See IEEE 1588-2008 for more information." )
   delayResponseInterval = CliModel.Int(
         help="Delay Response interval of profile"
      "See IEEE 1588-2008 for more information." )
   delayResponseDuration = CliModel.Int(
         help="Delay Response interval of profile"
      "See IEEE 1588-2008 for more information." )

class UcastNegProfiles( CliModel.Model ):
   profiles = CliModel.Dict(
         valueType=UcastNegProfile,
         keyType=str,
         help="Unicast Negotiation Profiles list." )

   def render( self ):
      for name, p in self.profiles.items():
         print( 'Unicast Negotiation Profile', name )
         print( 'Announce interval:', 2 ** p.announceInterval, 'seconds' )
         print( 'Announce duration:', p.announceDuration, 'seconds' )
         print( 'Sync interval:', 2 ** p.syncInterval, 'seconds' )
         print( 'Sync duration:', p.syncDuration, 'seconds' )
         print( 'Delay Response interval:',
                2 ** p.delayResponseInterval, 'seconds' )
         print( 'Delay Response duration:', p.delayResponseDuration, 'seconds' )
         print()

class UcastNegMessage( CliModel.Model ):
   expirationTime = CliModel.Float(
         help="Time, in seconds, at which request was sent from grantor/grantee to "
              "Slave/Master." )
   logInterval = CliModel.Int(
         help="log Interval of time, in seconds, agreed upon between the the "
              "and grantee." )
   duration = CliModel.Int(
         help="Duration of time, in seconds, agreed upon between the grantor and "
              "grantee." )
   ipAddr = ArnetModel.IpGenericAddress(
         help="Ip address of the Master/Slave from which we received a message." )
   messageType = CliModel.Enum(
         help="Type of Message sent from potential Master/Slave to Slave/Master.",
         values=list( messageTypeToName ) )

class UcastNegMessages( CliModel.Model ):
   messages = CliModel.List(
         valueType=UcastNegMessage,
         help="Unicast Negotiation request sent from master/slave." )

class UcastNegStatus( CliModel.Model ):
   interfaces = CliModel.Dict(
         valueType=UcastNegMessages,
         keyType=Interface,
         help="A mapping from the interface name to message infos." )

   def render( self ):
      interfaces = self.interfaces
      headings = ( "Interface", "Address", "Message", "Interval",
                   "Duration", "Expires in" )
      table = TableOutput.createTable( headings )

      fInterface = TableOutput.Format(
         justify="left", maxWidth=10, wrap=True )
      fIpAddress = TableOutput.Format(
         justify="left", maxWidth=13, wrap=True )
      fMessage = TableOutput.Format(
         justify="left", maxWidth=16, wrap=True )
      fInterval = TableOutput.Format(
         justify="left", maxWidth=20, wrap=True )
      fDuration = TableOutput.Format(
         justify="left", maxWidth=20, wrap=True )
      fExpirationTime = TableOutput.Format(
         justify="left", maxWidth=20, wrap=True )

      table.formatColumns( fInterface, fIpAddress, fMessage, fInterval, fDuration,
                           fExpirationTime )

      for intf in sorted( interfaces ):
         for msg in interfaces[ intf ].messages:
            if msg.expirationTime <= 0:
               expiresIn = "denied"
            else:
               expiresIn = str( int( msg.expirationTime ) ) + " seconds"
            interval = str( math.pow( 2, msg.logInterval ) ) + ' seconds'
            duration = str( msg.duration ) + ' seconds'
            table.newRow(
               Tac.Value( "Arnet::IntfId", intf ).shortName,
               msg.ipAddr,
               messageTypeToName[ msg.messageType ],
               interval,
               duration,
               expiresIn )
      print( table.output() )

class PtpParent( CliModel.Model ):
   parentClockIdentity = CliModel.Str( help="Parent clock identity. "
                                       "See IEEE 1588-2008 for more information." )
   parentPortNumber = CliModel.Int( help="Parent PTP port number. "
                                    "See IEEE 1588-2008 for more information." )
   parentIpAddr = ArnetModel.Ip4Address( help="IPv4 address of the parent",
                                         optional=True,
                                         default=None )
   parentIp6Addr = ArnetModel.Ip6Address( help="IPv6 address of the parent",
                                          optional=True,
                                          default=None )
   parentTwoStepFlag = CliModel.Bool( help="Parent PTP Boundary Clock "
                                       "Two-Step Flag." )
   gmClockIdentity = CliModel.Str( help="Grandmaster clock identity. "
                                   "See IEEE 1588-2008 for more information." )
   gmClockQuality = CliModel.Submodel( help="Grandmaster clock quality. "
                                       "See IEEE 1588-2008 for more information.",
                                       valueType=PtpClockQuality )
   gmPriority1 = CliModel.Int( help="Grandmaster priority1. "
                               "See IEEE 1588-2008 for more information." )
   gmPriority2 = CliModel.Int( help="Grandmaster priority2. "
                               "See IEEE 1588-2008 for more information." )

   def render( self ):
      print( 'Parent Clock:' )
      print( 'Parent Clock Identity:', self.parentClockIdentity )
      print( 'Parent Port Number:', self.parentPortNumber )
      ipAddr = 'N/A'
      if self.parentIpAddr:
         ipAddr = self.parentIpAddr
      elif self.parentIp6Addr:
         ipAddr = self.parentIp6Addr
      print( 'Parent IP Address:', ipAddr )
      print( 'Parent Two Step Flag:', self.parentTwoStepFlag )
      print( 'Observed Parent Offset (log variance): N/A' )
      print( 'Observed Parent Clock Phase Change Rate: N/A' )
      print()
      print( 'Grandmaster Clock:' )
      print( 'Grandmaster Clock Identity:', self.gmClockIdentity )
      print( 'Grandmaster Clock Quality:' )
      self.gmClockQuality.render()
      print( '   Priority1:', self.gmPriority1 )
      print( '   Priority2:', self.gmPriority2 )

class PtpParentWrapper( CliModel.Model ):
   message = CliModel.Str( help="Message to display in case of unexpected state",
                           optional=True,
                           default=None )
   parent = CliModel.Submodel( help="PTP parent information",
                              optional=True,
                              default=None,
                              valueType=PtpParent )
   def render( self ):
      if self.message is not None:
         print( self.message )
         return
      if self.parent is not None:
         self.parent.render()

class PtpTimeProperty( CliModel.Model ):
   currentUtcOffsetValid = CliModel.Bool(
      help="Current offset between PTP time and UTC time is valid. "
      "See IEEE 1588-2008 for more information." )
   currentUtcOffset = CliModel.Int(
      help="Current offset between PTP time and UTC time. "
      "See IEEE 1588-2008 for more information." )
   leap59 = CliModel.Bool( help="Last minute of today has 59 seconds. "
                           "See IEEE 1588-2008 for more information." )
   leap61 = CliModel.Bool( help="Last minute of today has 61 seconds. "
                           "See IEEE 1588-2008 for more information." )
   timeTraceable = CliModel.Bool( help="timePropertiesDS.timeTraceable. "
                                  "See IEEE 1588-2008 for more information." )
   frequencyTraceable = CliModel.Bool( help="timePropertiesDS.frequencyTraceable. "
                                       "See IEEE 1588-2008 for more information." )
   ptpTimescale = CliModel.Bool( help="GM is using PTP timescale. "
                                 "See IEEE 1588-2008 for more information." )
   timeSource = CliModel.Int( help="Time source GM is using. "
                              "See IEEE 1588-2008 for more information." )

   def render( self ):
      print( 'Current UTC offset valid:', self.currentUtcOffsetValid )
      print( 'Current UTC offset:', self.currentUtcOffset )
      print( 'Leap 59:', self.leap59 )
      print( 'Leap 61:', self.leap61 )
      print( 'Time Traceable:', self.timeTraceable )
      print( 'Frequency Traceable:', self.frequencyTraceable )
      print( 'PTP Timescale:', self.ptpTimescale )
      timeSource = self.timeSource
      if timeSource in timeSourceToName:
         print( 'Time Source:', timeSourceToName[ timeSource ] )
      else:
         print( 'Time Source:', hex( timeSource ) )

class PtpTimePropertyWrapper( CliModel.Model ):
   message = CliModel.Str( help="Message to display in case of unexpected state",
                           optional=True,
                           default=None )
   timeProperty = CliModel.Submodel( help="Time property data",
                                     valueType=PtpTimeProperty,
                                     optional=True,
                                     default=None )

   def render( self ):
      if self.message is not None:
         print( self.message )
         return
      if self.timeProperty is not None:
         self.timeProperty.render()

class PtpIntfCountersBase( CliModel.Model ):
   def _render( self, msgType, countTx, countRx, txRx="TxRx" ):
      suffix = ( "sent", "received" )
      count = ( "N/A" if countTx is None else countTx,
                "N/A" if countRx is None else countRx )
      start = 1 if txRx == "Rx" else 0
      stop = 1 if txRx == "Tx" else 2
      for i in range( start, stop ):
         print( f"{msgType} messages {suffix[ i ]}: {count[ i ]}" )

class PtpIntfCounters( PtpIntfCountersBase ):
   announceTx = CliModel.Int( help="Announce messages tx",
                              optional=True,
                              default=None )
   announceRx = CliModel.Int( help="Announce messages rx",
                              optional=True,
                              default=None )
   gptpFollowUpMessagesTxMissed = CliModel.Int(
      help="Follow up gPTP messages tx missed",
      optional=True,
      default=None )
   syncTx = CliModel.Int( help="Sync messages tx" )
   syncRx = CliModel.Int( help="Sync messages rx" )
   followUpTx = CliModel.Int( help="Follow up tx" )
   followUpRx = CliModel.Int( help="Follow up rx" )
   delayReqTx = CliModel.Int( help="Delay request tx" )
   delayReqRx = CliModel.Int( help="Delay request rx" )
   delayRespTx = CliModel.Int( help="Delay response tx" )
   delayRespRx = CliModel.Int( help="Delay response rx" )
   pDelayReqTx = CliModel.Int( help="Peer delay request tx" )
   pDelayReqRx = CliModel.Int( help="Peer delay request rx" )
   pDelayRespTx = CliModel.Int( help="Peer delay response tx" )
   pDelayRespRx = CliModel.Int( help="Peer delay response rx" )
   pDelayRespFollowUpTx = CliModel.Int( help="Peer delay response follow up tx" )
   pDelayRespFollowUpRx = CliModel.Int( help="Peer delay response follow up rx" )

   managementTx = CliModel.Int( help="Management tx", optional=True, default=None )
   managementRx = CliModel.Int( help="Management rx", optional=True, default=None )

   signalingTx = CliModel.Int( help="Signaling tx", optional=True, default=None )
   signalingRx = CliModel.Int( help="Signaling rx", optional=True, default=None )

   def render( self ):
      self._render( "Announce", self.announceTx, self.announceRx )
      self._render( "Sync", self.syncTx, self.syncRx )
      self._render( "Follow up", self.followUpTx, self.followUpRx )
      if self.gptpFollowUpMessagesTxMissed is None:
         self._render( "Delay request", self.delayReqTx, self.delayReqRx )
         self._render( "Delay response", self.delayRespTx, self.delayRespRx )
         self._render( "Peer delay request", self.pDelayReqTx, self.pDelayReqRx )
         self._render( "Peer delay response", self.pDelayRespTx, self.pDelayRespRx )
         self._render( "Peer delay response follow up", self.pDelayRespFollowUpTx,
                       self.pDelayRespFollowUpRx )
      else:
         print( 'Follow up messages skipped:', self.gptpFollowUpMessagesTxMissed )

         # Review: same info below as in `if` clause, but different order. Is it
         # what we want? Shall we keep the order consistent?
         self._render( "Peer delay request", self.pDelayReqTx, None, txRx="Tx" )
         self._render( "Peer delay response", None, self.pDelayRespRx, txRx="Rx" )
         self._render( "Peer delay response follow up", None,
                       self.pDelayRespFollowUpRx, txRx="Rx" )
         self._render( "Peer delay request", None, self.pDelayReqRx, txRx="Rx" )
         self._render( "Peer delay response", self.pDelayRespTx, None, txRx="Tx" )
         self._render( "Peer delay response follow up", self.pDelayRespFollowUpTx,
                       None, txRx="Tx" )

      self._render( "Management", self.managementTx, self.managementRx )
      self._render( "Signaling", self.signalingTx, self.signalingRx )

class PtpIntfMpassCounters( PtpIntfCountersBase ):
   announceLikeTx = CliModel.Int( help="Forwarded Announce tx" )
   announceLikeRx = CliModel.Int( help="Forwarded Announce rx" )
   syncLikeTx = CliModel.Int( help="Forwarded Sync tx" )
   syncLikeRx = CliModel.Int( help="Forwarded Sync rx" )
   followUpLikeTx = CliModel.Int( help="Forwarded Follow up tx" )
   followUpLikeRx = CliModel.Int( help="Forwarded Follow up rx" )
   delayReqLikeTx = CliModel.Int( help="Forwarded Delay request tx" )
   delayReqLikeRx = CliModel.Int( help="Forwarded Delay request rx" )
   delayRespLikeTx = CliModel.Int( help="Forwarded Delay response tx" )
   delayRespLikeRx = CliModel.Int( help="Forwarded Delay response rx" )

   def render( self ):
      self._render( "Forwarded Announce", self.announceLikeTx, self.announceLikeRx )
      self._render( "Forwarded Sync", self.syncLikeTx, self.syncLikeRx )
      self._render( "Forwarded Follow up", self.followUpLikeTx, self.followUpLikeRx )
      self._render( "Forwarded Delay request", self.delayReqLikeTx,
                    self.delayReqLikeRx )
      self._render( "Forwarded Delay response", self.delayRespLikeTx,
                    self.delayRespLikeRx )

class PtpIntfCountersWrapper( CliModel.Model ):
   interfaceCounters = CliModel.Submodel( help="PTP counters state",
                                          valueType=PtpIntfCountersBase,
                                          optional=True,
                                          default=None )
   vlanId = CliModel.Int(
      help="VLAN on which the PTP trunk interface is configured",
      optional=True )

   def render( self ):
      if self.interfaceCounters is not None:
         self.interfaceCounters.render()

class PtpIntfVlansCounters( CliModel.Model ):
   interface = Interface( help="PTP interface name" )
   ptpInterfaceVlansCounters = \
         CliModel.List( help="List of PTP counters on a same interface."
                             " There can be several of them when PTP is configured "
                             " to run on multiple VLANs on a trunk interface.",
                               valueType=PtpIntfCountersWrapper )

   def render( self ):
      ptpInterfaceVlansCounters = sorted( self.ptpInterfaceVlansCounters,
                                          key=attrgetter( 'vlanId' ) )
      for intfCounter in ptpInterfaceVlansCounters:
         line = f"Interface {self.interface.stringValue}"
         if intfCounter.vlanId is not None and intfCounter.vlanId != 0:
            vlanId = intfCounter.vlanId
            line += f", VLAN {vlanId}"
         print( line )
         intfCounter.render()

class PtpCounters( CliModel.Model ):
   __revision__ = 3
   interfaceCounters = CliModel.Dict( help="Interfaces to display counters.",
                                      valueType=PtpIntfVlansCounters,
                                      keyType=Interface )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         intfs = {}
         for ptpIntfVlanCounter in self.interfaceCounters.values():
            ptpIntfWrapper = \
               next( ( x for x in ptpIntfVlanCounter.ptpInterfaceVlansCounters
                               if not x.vlanId ), None )
            if ptpIntfWrapper is not None:
               intfName = ptpIntfVlanCounter.interface.stringValue
               intfs[ intfName ] = ptpIntfWrapper.toDict()
         dictRepr[ "interfaceCounters" ] = intfs
      elif revision < 3:
         # prior to revision 3, interfaceCounters is of type PtpIntfCounters
         # after revision 3, interfaceCounters is of type PtpIntfCountersBase
         intfs = {}
         for intf, ptpIntfVlanCounter in self.interfaceCounters.items():
            intfs[ intf ] = ptpIntfVlanCounter.toDict()
         dictRepr[ "interfaceCounters" ] = intfs
      return dictRepr

   def render( self ):
      for intf in Arnet.sortIntf( self.interfaceCounters ):
         self.interfaceCounters[ intf ].render()

class PtpVlansExtendedWrapper( CliModel.Model ):
   vlans = CliModel.List( valueType=int,
                          help="PTP configured vlans on trunk port" )
   notCreatedVlans = CliModel.List( valueType=int,
                                    help="PTP configured vlans but not created" )
   notAllowedVlans = CliModel.List(
         valueType=int,
         help="PTP configured vlans but not allowed on trunk port" )

   def render( self ):
      vlanStr = MultiRangeRule.multiRangeToCanonicalString( self.vlans )
      print( 'Active trunk VLANs running PTP:', vlanStr if vlanStr else 'None' )

      notCreatedVlansStr = \
            MultiRangeRule.multiRangeToCanonicalString( self.notCreatedVlans )
      if notCreatedVlansStr:
         label = 'VLANs allowed by trunk configuration but not configured:'
         print( label, notCreatedVlansStr )
      notAllowedVlansStr = \
            MultiRangeRule.multiRangeToCanonicalString( self.notAllowedVlans )
      if notAllowedVlansStr:
         print( 'VLANs unallowed by trunk configuration:', notAllowedVlansStr )

class PtpVlans( CliModel.Model ):
   interfaces = CliModel.Dict(
      help="A mapping of interfaces to ptp configured vlans on trunk port",
      valueType=PtpVlansExtendedWrapper,
      keyType=Interface )
   message = CliModel.Str( help="Message in case of unexpected state",
                           optional=True,
                           default=None )

   def render( self ):
      if self.message is not None:
         print( self.message )
         return

      for intf in Arnet.sortIntf( self.interfaces ):
         print( "Interface", intf )
         self.interfaces[ intf ].render()
         print()

class PtpIntfBase( CliModel.Model ):
   syncTestRole = CliModel.Enum(
      help="PTP synctest state",
      values=list( synctestStateToName ),
      optional=True,
      default=None )
   portState = CliModel.Enum(
      help="PTP port state. See IEEE 1588-2008 for more information.",
      values=list( portStateToName ),
      optional=True,
      default=None )
   vlanId = CliModel.Int(
      help="VLAN on which the PTP trunk interface is configured",
      optional=True )
   mpassEnabled = CliModel.Bool(
      help="Interface is PTP MPASS enabled",
      optional=True,
      default=None )
   mpassStatus = CliModel.Bool(
      help="Interface in MPASS active state",
      optional=True,
      default=None )
   mpassDisabledReason = CliModel.Enum(
      help="MPASS Disabled Reason",
      values=MpassDisabledReason.attributes,
      optional=True,
      default=None )

   def renderPortState( self ):
      onSynctest = ''
      if self.syncTestRole != 'syncTestDisabled':
         onSynctest = '(synctest)'
      print( 'Port state:', portStateToName[ self.portState ]
             if self.portState is not None else 'None', onSynctest )

   def renderMPASS( self ):
      if self.mpassEnabled is not None:
         mpassString = 'Enabled ' if self.mpassEnabled else 'Disabled '
         if self.mpassEnabled:
            mpassActive = 'active' if self.mpassStatus else 'inactive'
            mpassString += "(" + mpassActive + ")"
         else:
            mpassString += ( "(" +
                        mpassDisabledReasonToName[ self.mpassDisabledReason ] + ")" )
         print( 'MPASS:', mpassString )

class PtpIntf( PtpIntfBase ):
   adminDisabled = CliModel.Bool( help="Interface is PTP disabled" )
   counters = CliModel.Submodel( help="Interface PTP counters",
                                 valueType=PtpIntfCountersWrapper,
                                 optional=True,
                                 default=None )
   ptpMode = CliModel.Enum( help="PTP mode",
                            values=list( ptpModeToName ) )
   asCapable = CliModel.Bool(
      help="Interface is gptp capable",
      optional = True,
      default=None )
   lastGptpResidenceTime = CliModel.Int(
      help="Minimum measured gptp residence time",
      optional = True,
      default=None )
   minGptpResidenceTime = CliModel.Int(
      help="Maximum measured gptp residence time",
      optional = True,
      default=None )
   maxGptpResidenceTime = CliModel.Int(
      help="Last measured gptp residence time",
      optional = True,
      default=None )
   syncInterval = CliModel.Float(
      help="Sync interval in seconds. See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   syncReceiptTimeout = CliModel.Int(
      help="Sync receipt timeout, measured in sync intervals. "
      "See IEEE 802.1AS for more information.",
      optional=True,
      default=None )
   announceInterval = CliModel.Float(
      help="Announce interval in seconds. See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   announceReceiptTimeout = CliModel.Int(
      help="Announce receipt timeout, measured in announce intervals. "
      "See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   delayMechanism = CliModel.Str(
      help="PTP delay mechanism. See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   delayReqInterval = CliModel.Float(
      help="Delay request interval. See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   peerDelayReqInterval = CliModel.Float(
      help="Peer delay request interval in seconds. "
      "See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   peerPathDelay = CliModel.Int(
      help="Peer mean path delay in nanoseconds. "
      "See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   timeSinceAsCapableLastChanged = CliModel.Str(
      help="Time elasped since AS capable state of the port last changed",
      optional=True,
      default=None )
   transportMode = CliModel.Enum(
      help="Type of packet used to send PTP messages",
      values=list( transportModeToDisplayStr ),
      optional=True,
      default=None )
   dscpEvent = CliModel.Int(
      help="Dscp value to set for Ptp Ipv4 event packets",
      optional=True,
      default=None )
   dscpGeneral = CliModel.Int(
      help="Dscp value to set for Ptp Ipv4 general packets",
      optional=True,
      default=None )
   localPriority = CliModel.Int(
      help="Local Priority value. See ITU-T G8275 for more information.",
      optional=True,
      default=None )
   mcastTxAddr = ArnetModel.MacAddress(
      help="Multicast destination address under PTP profile g8275.1",
      optional=True,
      default=None )
   txLagMember = Interface(
      help="Member interface used for sending packets in a LAG",
      optional=True,
      default=None )

   def render( self ):
      if self.adminDisabled:
         print( 'PTP: Disabled' )
      else:
         print( 'PTP: Enabled' )

      # I don't think there's anything else to show for e2e transparent mode.
      if self.ptpMode == 'ptpEndToEndTransparentClock':
         if self.counters is not None:
            self.counters.render()
         return

      if self.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock',
                           'ptpGeneralized' ]:
         self.renderPortState()
         self.renderMPASS()
         if self.ptpMode == 'ptpGeneralized':
            print( 'Sync interval timeout multiplier:', self.syncReceiptTimeout )
            print( 'AS capable: %s, Last changed: %s' %
                   ( self.asCapable, self.timeSinceAsCapableLastChanged ) )
            if self.portState == 'psMaster':
               print( 'Last Residence Time:', self.lastGptpResidenceTime )
               print( 'Minimum Residence Time:', self.minGptpResidenceTime )
               print( 'Maximum Residence Time:', self.maxGptpResidenceTime )
         if self.syncInterval:
            print( 'Sync interval:', self.syncInterval, 'seconds' )
         if self.announceInterval:
            print( 'Announce interval:', self.announceInterval, 'seconds' )
         if self.announceReceiptTimeout:
            print( 'Announce interval timeout multiplier:',
                   self.announceReceiptTimeout )
      if self.delayMechanism == 'e2e' and \
         self.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock' ]:
         print( 'Delay mechanism: end to end' )
         if self.delayReqInterval:
            print( 'Delay request message interval:',
                   self.delayReqInterval, 'seconds' )
      elif self.delayMechanism == 'p2p':
         print( 'Delay mechanism: peer to peer' )
         print( 'Peer delay request message interval:',
                self.peerDelayReqInterval, 'seconds' )
         print( 'Peer Mean Path Delay:', self.peerPathDelay, 'nanoseconds' )
      if self.localPriority is not None:
         print( 'Local Priority:', self.localPriority )

      if self.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock',
                           'ptpGeneralized' ]:
         if self.transportMode is not None:
            print( 'Transport mode:',
                   transportModeToDisplayStr[ self.transportMode ] )
      if self.mcastTxAddr:
         print( 'Destination MAC address: ' + str( self.mcastTxAddr ) )
      if self.counters is not None:
         self.counters.render()
      if self.txLagMember is not None:
         print( 'Transmit member:', self.txLagMember.stringValue )

      # Print additional dynamic state here.
      print()

class PtpMpassIntf( PtpIntfBase ):
   syncExpiry = CliModel.Float( help="Forwarded Sync expiration timer." )
   delayReqExpiry = CliModel.Float(
      help="Forwarded Delay request expiration timer." )
   counters = CliModel.Submodel( help="Interface PTP MPASS counters",
                                 valueType=PtpIntfCountersWrapper,
                                 optional=True,
                                 default=None )
   def render( self ):
      self.renderPortState()
      self.renderMPASS()
      if self.syncExpiry:
         print( "Forwarded Sync expiration timer: %.2f seconds" % self.syncExpiry )
      if self.delayReqExpiry:
         print ( "Forwarded Delay request expiration timer: %.2f seconds" %
                 self.delayReqExpiry )
      if self.counters is not None:
         self.counters.render()

      print()

class PtpIntfWrapper( CliModel.Model ):
   interface = CliModel.Submodel( help="PTP interface state",
                                  valueType=PtpIntfBase,
                                  optional=True,
                                  default=None )

   def render( self ):
      if self.interface is not None:
         self.interface.render()

class PtpIntfVlans( CliModel.Model ):
   interface = Interface( help="PTP interface name" )
   ptpInterfaceVlansStates = \
         CliModel.List( help="List of PTP states on a same interface."
                             " There can be several of them when PTP is configured "
                             " to run on multiple VLANs on a trunk interface.",
                               valueType=PtpIntfWrapper )

   def render( self ):
      ptpInterfaceVlansStates = sorted( self.ptpInterfaceVlansStates,
                                        key=lambda e: e.interface.vlanId )
      for portDS in ptpInterfaceVlansStates:
         line = f"Interface {self.interface.stringValue}"
         if portDS.interface.vlanId is not None and portDS.interface.vlanId != 0:
            vlanId = portDS.interface.vlanId
            line += f", VLAN {vlanId}"
         print( line )
         portDS.render()

class PtpInterfaces( CliModel.Model ):
   __revision__ = 3

   interfaces = CliModel.Dict( help="Collection of interfaces to display",
                               valueType=PtpIntfVlans,
                               keyType=Interface )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         intfs = {}
         for ptpIntfVlan in self.interfaces.values():
            ptpIntfWrapper = next( ( x for x in ptpIntfVlan.ptpInterfaceVlansStates
                                     if x.interface and not x.interface.vlanId ),
                                   None )
            if ptpIntfWrapper is not None:
               intfs[ ptpIntfVlan.interface.stringValue ] = ptpIntfWrapper.toDict()
         dictRepr[ "interfaces" ] = intfs
      elif revision < 3:
         # prior to revision 3, interface is of type PtpIntf
         # after revision 3, interface is of type PtpIntfBase
         intfs = {}
         for intf, intfVlan in self.interfaces.items():
            intfs[ intf ] = intfVlan.toDict()
         dictRepr[ "interfaces" ] = intfs
      return dictRepr

   def render( self ):
      for intf in Arnet.sortIntf( self.interfaces ):
         self.interfaces[ intf ].render()

class PtpIntfSummary( CliModel.Model ):
   vlanId = CliModel.Int(
      help="VLAN on which the PTP trunk interface is configured",
      optional=True )
   portState = CliModel.Enum(
      help="PTP port state. See IEEE 1588-2008 for more information.",
      values=list( portStateToName ),
      optional=True,
      default=None )
   asCapable = CliModel.Bool(
      help="Interface is GPTP capable",
      optional=True,
      default=None )
   lastGptpResidenceTime = CliModel.Int(
      help="Minimum measured GPTP residence time",
      optional=True,
      default=None )
   delayMechanism = CliModel.Str(
      help="PTP delay mechanism. See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   peerPathDelay = CliModel.Int(
      help="Peer mean path delay in nanoseconds. "
      "See IEEE 1588-2008 for more information.",
      optional=True,
      default=None )
   timeSinceAsCapableLastChanged = CliModel.Str(
      help="Time elasped since AS capable state of the port last changed",
      optional=True,
      default=None )
   transportMode = CliModel.Enum(
      help="Type of packet used to send PTP messages",
      values=list( transportModeToDisplayStr ),
      optional=False,
      default=None )
   neighborRateRatio = CliModel.Float(
      help="Estimated gptp neighbor rate ratio",
      optional = True,
      default = None )
   mpassEnabled = CliModel.Bool(
      help="PTP MPASS operational state",
      optional=True,
      default=False )
   mpassStatus = CliModel.Enum(
      help="PTP MPASS status",
      values=[ PtpLib.MpassStatus.active, PtpLib.MpassStatus.inactive ],
      optional=True,
      default=PtpLib.MpassStatus.active )

class PtpIntfVlanSummary( CliModel.Model ):
   interface = Interface( help="PTP interface name" )
   ptpIntfVlanSummaries = CliModel.List(
         help="List of summaries of PTP enabled interface on different VLAN to"
              " display. There can be several of them when PTP is configured "
              " to run on multiple VLANs on a trunk interface.",
         valueType=PtpIntfSummary )

def renderIntfSummaries( table, ptpIntfSummaries, mode=None, mpass=False ):
   intfSummaries = Arnet.sortIntf( ptpIntfSummaries )
   for intfVlan in intfSummaries:
      intfVlanSummaries = ptpIntfSummaries[ intfVlan ] \
                          .ptpIntfVlanSummaries
      intf = ptpIntfSummaries[ intfVlan ].interface
      intfVlanSummaries = sorted( intfVlanSummaries,
                                  key=attrgetter( 'vlanId' ) )
      for intfSummary in intfVlanSummaries:
         intfName = intf.shortName
         if intfSummary.vlanId:
            intfName = f"{intfName}, VLAN {intfSummary.vlanId}"
         if mpass:
            mpassStatus = ( intfSummary.mpassStatus if intfSummary.mpassEnabled
                            else 'disabled' )
            table.newRow( intfName,
                          portStateToName.get( intfSummary.portState, '' ),
                          mpassStatus )
         elif mode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock' ]:
            table.newRow( intfName,
                          portStateToName[ intfSummary.portState ],
                          intfSummary.transportMode,
                          intfSummary.delayMechanism )
         elif mode == 'ptpGeneralized':
            table.newRow( intfName,
                          portStateToName[ intfSummary.portState ],
                          'Yes' if intfSummary.asCapable else 'No',
                          intfSummary.timeSinceAsCapableLastChanged,
                          str( intfSummary.neighborRateRatio )[ :10 ],
                          intfSummary.peerPathDelay,
                          intfSummary.lastGptpResidenceTime // 1000000 )
         else:
            table.newRow( intfName,
                          intfSummary.transportMode )

class PtpSummary( CliModel.Model ):
   __revision__ = 2

   class PtpClockSummary( CliModel.Model ):
      clockIdentity = CliModel.Str( help="DefaultDS clock identity" )
      gmClockIdentity = CliModel.Str( help="Grandmaster clock identity. "
                                      "See IEEE 1588-2008 for more information." )
      numberOfSlavePorts = CliModel.Int( help="Number of slave ports" )
      numberOfMasterPorts = CliModel.Int( help="Number of master ports" )
      slavePort = CliModel.Str(
         help="Interface name of slave port",
         optional=True,
         default = None )
      slaveVlanId = CliModel.Int(
         help="VLAN ID of slave port",
         optional=True,
         default=None )
      offsetFromMaster = CliModel.Int(
         help="CurrentDS offset from master in nanoseconds",
         optional=True,
         default = None )
      meanPathDelay = CliModel.Int(
         help="CurrentDS mean path delay to master in nanoseconds" )
      stepsRemoved = CliModel.Int( help="CurrentDS steps removed from GM" )
      skew = CliModel.Float(
         help="Ratio of master's seconds to local seconds",
         optional=True,
         default = None )
      lastSyncTime = CliModel.Int(
         help="Last time local clock synced from master",
         optional=True,
         default=None )
      currentPtpSystemTime = CliModel.Int(
         help="Current sytem time measured with PTP",
         optional=True,
         default=None )
      neighborRateRatio = CliModel.Float(
         help="Estimated gptp neighbor rate ratio",
         optional = True,
         default = None )
      gptpRateRatio = CliModel.Float(
         help="Estimated gptp rate ratio",
         optional = True,
         default = None )
      systemClockOffset = CliModel.Int(
         help="Current offset between system clock and PTP clock in nanoseconds",
         optional=True,
         default=None )


      def render( self ):
         print( 'Clock Identity:', self.clockIdentity )
         print( 'Grandmaster Clock Identity:', self.gmClockIdentity )
         print( 'Number of slave ports:', self.numberOfSlavePorts )
         print( 'Number of master ports:', self.numberOfMasterPorts )
         if self.slavePort:
            slavePortLine = f'Slave port: {self.slavePort}'
            defaultVlanId = CliConstants.defaultPortDSVlanId
            if self.slaveVlanId is not None and self.slaveVlanId != defaultVlanId:
               slavePortLine += f', VLAN {self.slaveVlanId}'
            print( slavePortLine )
         if self.offsetFromMaster is not None:
            print( 'Offset From Master (nanoseconds):', self.offsetFromMaster )
         print( 'Mean Path Delay (nanoseconds):', self.meanPathDelay )
         print( 'Steps Removed:', self.stepsRemoved )
         if self.skew:
            print( 'Skew (estimated local-to-master clock frequency ratio):',
                   self.skew )
         if self.lastSyncTime is not None:
            try:
               date = strftime( '%H:%M:%S UTC %b %d %Y',
                                         gmtime( self.lastSyncTime ) )
               print( 'Last Sync Time:', date )
            except OverflowError:
               print( 'Last Sync Time: ERROR, timestamp too large' )

         if self.currentPtpSystemTime is not None:
            try:
               estimatedDate = strftime( '%H:%M:%S UTC %b %d %Y',
                                         gmtime( self.currentPtpSystemTime ) )
               print( 'Current PTP System Time:', estimatedDate )
            except OverflowError:
               print( 'Current PTP System Time: ERROR, timestamp too large' )

         if self.neighborRateRatio:
            print( 'Neighbor Rate Ratio:', self.neighborRateRatio )
         if self.gptpRateRatio:
            print( 'Rate Ratio:', self.gptpRateRatio )
         if self.systemClockOffset is not None:
            print( 'System clock to PTP clock offset (nanoseconds):',
                   self.systemClockOffset )

   ptpMode = CliModel.Enum( help="PTP mode",
                            values=list( ptpModeToName ),
                            optional=True,
                            default=None )
   ptpProfile = CliModel.Enum( help="PTP profile",
                               values=list( ptpProfileToName ),
                               optional=True,
                               default=None )
   ptpSelectedProfile = CliModel.Enum( help="Selected (but not applied) PTP profile",
                                       values=list( ptpProfileToName ),
                                       optional=True,
                                       default=None )
   ptpClockSummary = CliModel.Submodel(
      help="Summary Information for boundary clocks",
      optional=True,
      default=None,
      valueType=PtpClockSummary )

   ptpIntfSummaries = CliModel.Dict(
      help="Collection of summaries of PTP enabled interfaces to display",
      valueType=PtpIntfVlanSummary,
      keyType=Interface )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         intfs = {}
         for ptpIntfVlanSummary in self.ptpIntfSummaries.values():
            ptpIntfSummary = \
               next( ( x for x in ptpIntfVlanSummary.ptpIntfVlanSummaries
                               if not x.vlanId ), None )
            if ptpIntfSummary is not None:
               intfName = ptpIntfVlanSummary.interface.stringValue
               intfs[ intfName ] = ptpIntfSummary.toDict()
         dictRepr[ "ptpIntfSummaries" ] = intfs
      return dictRepr

   def render( self ):
      # Print mode
      if self.ptpMode is None:
         print( PTP_NOT_CONFIGURED )
         return

      print( 'PTP Mode:', ptpModeToName[ self.ptpMode ] )
      # ptpClockSummary should be defined in boundary mode
      if self.ptpProfile:
         print( 'PTP Profile:', ptpProfileToName[ self.ptpProfile ] )
      if self.ptpSelectedProfile:
         print( 'Selected PTP Profile: {} configuration will be ignored in '
                '{} mode'.format( ptpProfileToName[ self.ptpSelectedProfile ],
                                  ptpModeToName[ self.ptpMode ] ) )
      if self.ptpClockSummary:
         self.ptpClockSummary.render()

      # Print interface summary
      if self.ptpIntfSummaries:
         fInterface = TableOutput.Format( justify="left", maxWidth=21, wrap=True )
         fState = TableOutput.Format( justify="left", maxWidth=17, wrap=True )
         fTransport = TableOutput.Format( justify="left", maxWidth=9, wrap=True )
         if self.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock' ]:
            # This is boundary mode
            tableHeadings = ( "Interface", "State", "Transport", "Delay Mechanism" )
         elif self.ptpMode == 'ptpGeneralized':
            # This is gptp mode
            tableHeadings = ( "Interface", "State", "AS Capable",
                              "Time Since Last Changed",
                              "Neighbor Rate Ratio", "Mean Path Delay (ns)",
                              "Residence Time (ms)" )
         else:
            # This is transparent mode
            tableHeadings = ( "Interface", "Transport" )

         table = TableOutput.createTable( tableHeadings )
         if self.ptpMode in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock' ]:
            fDelayMechanism = TableOutput.Format(
               justify="left", maxWidth=9, wrap=True )
            table.formatColumns( fInterface, fState, fTransport, fDelayMechanism )
         elif self.ptpMode == 'ptpGeneralized':
            fAsCapable = TableOutput.Format( justify="left", maxWidth=7, wrap=True )
            fAsCapableChanged = TableOutput.Format(
               justify="left", maxWidth=18, wrap=True )
            fNrr = TableOutput.Format( justify="left", maxWidth=10, wrap=True )
            fMeanPathDelay = TableOutput.Format(
               justify="left", maxWidth=10, wrap=True )
            fResidenceTime = TableOutput.Format(
               justify="left", maxWidth=9, wrap=True )
            table.formatColumns( fInterface, fState, fAsCapable, fAsCapableChanged,
                                 fNrr, fMeanPathDelay, fResidenceTime )
         else:
            table.formatColumns( fInterface, fTransport )
         renderIntfSummaries( table, self.ptpIntfSummaries, self.ptpMode )
         print( table.output() )

class PtpMpassSummary( CliModel.Model ):
   syncTimeout = CliModel.Int( help="Forwarded Sync expiration timeout in messages",
         optional=True )
   delayReqTimeout = CliModel.Int(
         help="Forwarded Delay request expiration timeout in seconds",
         optional=True )
   ptpIntfSummaries = CliModel.Dict(
      help="A mapping of interface to a summary of its MPASS configuration",
      valueType=PtpIntfVlanSummary,
      keyType=Interface )

   def render( self ):
      # Globally configured MPASS timeouts
      if self.syncTimeout and self.delayReqTimeout:
         print( "Configured sync timeout: %d messages" % self.syncTimeout )
         print( "Configured delay request timeout: %d seconds" %
                self.delayReqTimeout )
      # Print interface summary
      if self.ptpIntfSummaries:
         fInterface = TableOutput.Format( justify="left", maxWidth=21, wrap=True )
         fState = TableOutput.Format( justify="left", maxWidth=9, wrap=True )
         fMpass = TableOutput.Format( justify="left", maxWidth=15, wrap=True )
         tableHeadings = ( "Interface", "State", "MPASS Status" )

         table = TableOutput.createTable( tableHeadings )
         table.formatColumns( fInterface, fState, fMpass )
         renderIntfSummaries( table, self.ptpIntfSummaries, mpass=True )
         print( table.output() )

class PtpPortIdentity( CliModel.Model ):
   clockIdentity = CliModel.Str( help="DefaultDS clock identity." )
   portNumber = CliModel.Int( help="PTP port number." )

   def render( self ):
      print( '   Foreign master port id: clock id:', self.clockIdentity )
      print( '   Foreign master port id: port num:', self.portNumber )

class PtpForeignMasterDS( CliModel.Model ):
   foreignMasterPortIdentity = CliModel.Submodel(
      help="PTP port identity on which the foreign master was seen",
      valueType=PtpPortIdentity )
   foreignMasterIpAddr = ArnetModel.Ip4Address(
      help="IPv4 address of the foreign master",
      optional=True,
      default=None )
   foreignMasterIp6Addr = ArnetModel.Ip6Address(
      help="IPv6 address of the foreign master",
      optional=True,
      default=None )
   foreignMasterAnnounceMessages = CliModel.Int(
      help="Number of announce messages received from the foreign master" )
   timeAgo = CliModel.Float( help="Last time this record was updated in monotonic "
                             "seconds" )

   def render( self ):
      self.foreignMasterPortIdentity.render()
      ipAddr = 'N/A'
      if self.foreignMasterIpAddr:
         ipAddr = self.foreignMasterIpAddr
      elif self.foreignMasterIp6Addr:
         ipAddr = self.foreignMasterIp6Addr
      print( '   Foreign master ip address:', ipAddr )
      print( '   Number of announce messages:', self.foreignMasterAnnounceMessages )
      print( '   Last update time:', Ark.timestampToStr( self.timeAgo ) )

class PtpIntfForeignMasterRecords( CliModel.Model ):
   vlanId = CliModel.Int(
      help="VLAN on which the PTP trunk interface have receive foreign master"
           " records",
      optional=True )
   foreignMasterRecords = CliModel.List(
      help="List of foreign master records for this interface",
      valueType=PtpForeignMasterDS )
   maxForeignRecords = CliModel.Int(
      help="Most foreign masters heard from on this port" )

   def render( self ):
      print( '   Number of foreign records:', len( self.foreignMasterRecords ) )
      print( '   Max foreign records seen:', self.maxForeignRecords )
      rec = 0
      for record in self.foreignMasterRecords:
         print( '   Record:', rec )
         rec += 1
         record.render()

class PtpIntfVlanForeignMasterRecords( CliModel.Model ):
   interface = Interface( help="PTP interface name" )
   intfForeignMasterRecords = CliModel.List(
      help="List of foreign master records for this interface. They might be"
           " on different VLANs when ptp vlan is configured on a trunk interface",
      valueType=PtpIntfForeignMasterRecords )

   def render( self ):
      intfForeignMasterRecords = sorted( self.intfForeignMasterRecords,
                                         key=attrgetter( 'vlanId' ) )
      for fmr in intfForeignMasterRecords:
         line = f"Interface: {self.interface.stringValue}"
         if fmr.vlanId is not None and fmr.vlanId != 0:
            vlanId = fmr.vlanId
            line += f", VLAN: {vlanId}"
         print( line )
         fmr.render()

class PtpForeignMasterRecords( CliModel.Model ):
   __revision__ = 2

   message = CliModel.Str( help="Message to display in case of unexpected state",
                           optional=True,
                           default=None )
   intfForeignMasterRecords = CliModel.Dict(
      help="Collection of interface foreign master records",
      valueType=PtpIntfVlanForeignMasterRecords,
      keyType=Interface )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         intfs = {}
         for ptpIntfVlanFmr in self.intfForeignMasterRecords.values():
            ptpIntfFmr = \
               next( ( x for x in ptpIntfVlanFmr.intfForeignMasterRecords
                               if not x.vlanId ), None )
            if ptpIntfFmr is not None:
               intfs[ ptpIntfVlanFmr.interface.stringValue ] = ptpIntfFmr.toDict()
         dictRepr[ "intfForeignMasterRecords" ] = intfs
      return dictRepr

   def render( self ):
      if self.message is not None:
         print( self.message )
         return
      if not self.intfForeignMasterRecords:
         print( 'No Foreign Master Records' )
         return
      print( 'Foreign Master Records:' )
      intfs = list( self.intfForeignMasterRecords )
      intfs = Arnet.sortIntf( intfs )
      for intf in intfs:
         self.intfForeignMasterRecords[ intf ].render()

class PtpSourceIp( CliModel.Model ):
   sourceIp = ArnetModel.Ip4Address(
      help="Source IPv4 address to use when sending PTP messages.",
      optional=True,
      default=None )
   sourceIp6 = ArnetModel.Ip6Address(
      help="Source IPv6 address to use when sending PTP messages.",
      optional=True,
      default=None )

   def render( self ):
      if self.sourceIp is None and self.sourceIp6 is None:
         print( PTP_NOT_CONFIGURED )
         return

      if self.sourceIp:
         print( 'PTP source IP:', self.sourceIp )
      if self.sourceIp6:
         print( 'PTP source IPv6:', self.sourceIp6 )

class PtpMonitorDataEntry( CliModel.Model ):
   intf = Interface( help="Slave interface on which the PTP data has been "
                                    " recorded" )
   # This may be kind of confusing, but realLastSyncTime does not match the one in
   # PTP agent which is actually the hardware timestamp. On Alta, this is not Epoch
   # at all. The value should actually be lastSyncTime in PTP agent.
   realLastSyncTime = CliModel.Int( help="Epoch timestamp of when the PTP data has "
                                          "been detected" )
   lastSyncSeqId = CliModel.Int( help="Last sequenceId of sync / followUp PTP "
                                      "message processed" )
   offsetFromMaster = CliModel.Int( help="Offset from master value" )
   meanPathDelay = CliModel.Int( help="Mean path delay value" )
   skew = CliModel.Float( help="Ratio of master's second to local second" )

class PtpMonitorData( CliModel.Model ):
   monitorEnabled = CliModel.Bool( help="PTP monitor enabled",
                                   default=True )
   ptpMode = CliModel.Enum( help="PTP mode",
                            values=list( ptpModeToName ),
                            optional=True,
                            default=None )
   offsetFromMasterThreshold = CliModel.Int( help="Offset from master threshold",
                                             optional=True,
                                             default=None )
   meanPathDelayThreshold = CliModel.Int( help="Mean path delay threshold",
                                          optional=True,
                                          default=None  )
   skewThreshold = CliModel.Float( help="Skew threshold",
                                   optional=True,
                                   default=None  )
   ptpMonitorData = CliModel.List( help="Collection of PTP data from all interfaces",
                                   valueType=PtpMonitorDataEntry )

   def render( self ):
      if not self.ptpMode:
         print( PTP_NOT_CONFIGURED )
         return

      print( 'PTP Mode:', ptpModeToName[ self.ptpMode ] )
      if self.ptpMode not in [ 'ptpBoundaryClock', 'ptpOneStepBoundaryClock',
                               'ptpGeneralized' ]:
         return

      sortedMonitorData = sorted( self.ptpMonitorData,
                                  key=lambda x: -x.realLastSyncTime )

      print( 'Ptp monitoring: %s' %
             ( 'enabled' if self.monitorEnabled else 'disabled' ) )
      if not self.monitorEnabled:
         return

      print( 'Number of entries: %d' % len( sortedMonitorData ) )
      def thresholdToStr( threshold ):
         if threshold is not None:
            return str( threshold )
         return "not configured"
      print( 'Offset from master threshold: %s' % (
               thresholdToStr( self.offsetFromMasterThreshold ) ) )
      print( 'Mean path delay threshold: %s' % (
               thresholdToStr( self.meanPathDelayThreshold ) ) )
      print( 'Skew threshold: %s' % ( thresholdToStr( self.skewThreshold ) ) )

      # maxWidth is determined by the following rules:
      # 1. Sum of max width < 80
      # 2. Fit content in a single line
      # 3. Fit header in a single line
      # Rule 3 was not possible, so will fit in two lines
      fInterface = TableOutput.Format( justify="left", maxWidth=9, wrap=True )
      fTime = TableOutput.Format( justify="left", maxWidth=28, wrap=True )
      fOffset = TableOutput.Format( justify="left", maxWidth=11, wrap=True )
      fDelay = TableOutput.Format( justify="left", maxWidth=11, wrap=True )
      fSkew = TableOutput.Format( justify="left", maxWidth=12, wrap=True )
      fSeqId = TableOutput.Format( justify="left", maxWidth=8, wrap=True )
      tableHeadings = ( "Interface", "Time", "Offset from Master (ns)",
                        "Mean Path Delay (ns)", "Skew", "Seq Id" )
      table = TableOutput.createTable( tableHeadings, tableWidth=89 )
      table.formatColumns( fInterface, fTime, fOffset, fDelay, fSkew, fSeqId )

      for entry in sortedMonitorData:
         intf = entry.intf.shortName
         sec = entry.realLastSyncTime // ( 10 ** 9 )
         ns = ( entry.realLastSyncTime % ( 10 ** 9 ) ) / ( 10 ** 6 )
         try:
            date = strftime( '%H:%M:%S.%%03u UTC %b %d %Y', gmtime( sec ) ) % ns
         except OverflowError:
            date = "Time value is too large"
         offset = entry.offsetFromMaster
         delay = entry.meanPathDelay
         skew = entry.skew
         seqId = entry.lastSyncSeqId
         table.newRow( intf, date, offset, delay, skew, seqId )

      print( table.output() )

class PtpDomainNumbersSummary( CliModel.Model ):

   class DomainNumbersVlans( CliModel.Model ):

      # CliModel.Dict in DomainNumbersVlans must have a value type that is a model
      # and not just a CliModel.List like we have used for domainNumbers
      class DomainNumberList( CliModel.Model ):
         domainNumbers = CliModel.List( valueType=int, help='Active domains' )

      vlans = CliModel.Dict( keyType=int, valueType=DomainNumberList,
                             help='Active domains per PTP VLAN' )

      domainNumbers = CliModel.List( valueType=int,
                                     help='PTP domains connected to this interface' )

   interfaces = CliModel.Dict( keyType=Interface,
                               valueType=DomainNumbersVlans,
                               help='Active domains per interface per PTP VLAN' )

   clockDomain = CliModel.Int( help='PTP clock global domain number', optional=True )

   _ptpConfigured = CliModel.Bool( help='Is PTP configured' )

   def setPtpConfigured( self, isConfigured ):
      self._ptpConfigured = isConfigured

   def populateTable( self, table, intfId, intfDomainNumbers, vlanDomainInfo ):
      if intfDomainNumbers:
         domainNumbers = reduce( lambda prev, curr: str( prev ) + ", " + str( curr ),
                                 intfDomainNumbers )
         table.newRow( intfId, domainNumbers )

      for vlanId, activeDomains in sorted( vlanDomainInfo.items() ):
         table.newRow( f'VLAN {vlanId}', activeDomains.domainNumbers[ 0 ] )

   def render( self ):
      '''
      We handle three text rendering cases:
      1. PTP is not configued (ptpModeDisabled):
         -> we print the common not configured message only

      2. PTP is configured but no interfaces have domain-number configurations
         or all interfaces have been filtered out by command parameters:
         -> we do not print anything

      3. PTP is configured & some interfaces/ptpvlans have configured domain-number
         handling:
         -> we print a table per interface with entries for the interface name and
            the ptpvlans it handles and the domain number(s) associated with each.
            Individual entries or entire tables may be filtered out depending on
            command parameters
      '''
      if not self._ptpConfigured:
         print( PTP_NOT_CONFIGURED )
         return

      if not self.interfaces:
         return

      print( f'Clock domain: {self.clockDomain}\n' )

      header = ( ( 'Region Scope', 'l' ), ( 'Active Domain Numbers', 'r' ) )
      table = TableOutput.createTable( header )

      for interface, domainInfo in sorted( self.interfaces.items() ):
         print( f'Interface {interface}' )
         table = TableOutput.createTable( header )
         fInterface = TableOutput.Format( justify="left", maxWidth=25, wrap=True )
         fInterface.noPadLeftIs( True )
         fDomainNumbers = TableOutput.Format( justify="right", maxWidth=54,
                                              wrap=True )
         table.formatColumns( fInterface, fDomainNumbers )
         self.populateTable( table, interface, domainInfo.domainNumbers,
                             domainInfo.vlans )
         print( table.output() )
