#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.MacAddr as MacAddr # pylint: disable=consider-using-from-import
import CliPlugin.PtpCli as PtpCli # pylint: disable=consider-using-from-import
import CliToken.Ip
import CliToken.Ipv6
from functools import partial

matcherDscp = CliMatcher.KeywordMatcher( 'dscp',
      helpdesc='Set DSCP value for PTP Ipv4 packets' )
matcherDscpValue = CliMatcher.IntegerMatcher(
      PtpCli.CliConstants.minDscp, PtpCli.CliConstants.maxDscp,
      helpdesc='DSCP value for PTP Ipv4 packets' )
matcherMessageType = CliMatcher.KeywordMatcher( 'message-type',
      helpdesc='Configure message-type specific attributes' )
matcherMode = CliMatcher.KeywordMatcher( 'mode',
      helpdesc='Configure dataplane PTP operating mode' )
matcherMonitor = CliMatcher.KeywordMatcher( 'monitor',
      helpdesc='Record dataplane PTP timing data' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Set the source IP for dataplane PTP' )
matcherThreshold = CliMatcher.KeywordMatcher( 'threshold',
      helpdesc='Set thresholds for PTP data' )
matcherSkew = CliMatcher.KeywordMatcher( 'skew',
      helpdesc='Set PTP monitor threshold for skew' )
matcherMPD = CliMatcher.KeywordMatcher( 'mean-path-delay',
      helpdesc='Set PTP monitor threshold for mean path delay' )
matcherOffset = CliMatcher.KeywordMatcher( 'offset-from-master',
      helpdesc='Set PTP monitor threshold for offset from master' )
matcherNanosec = CliMatcher.KeywordMatcher( 'nanoseconds',
      helpdesc='Unit for threshold value' )
matcherDrop = CliMatcher.KeywordMatcher( 'drop',
      helpdesc='Enable drop for PTP monitor threshold' )
matcherFreeRunningPartial = partial( CliCommand.guardedKeyword, 'free-running',
                                     helpdesc='Configure properties for PTP when '
                                              'free running' )
matcherFreeRunningQuiet = \
      matcherFreeRunningPartial( guard=None )
matcherFreeRunning = matcherFreeRunningPartial( guard=None )

class PtpClockAccuracyCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp clock-accuracy VALUE'
   noOrDefaultSyntax = 'ptp clock-accuracy ...'
   data = {
         'ptp' : PtpCli.ptpMatcherForConfig,
         'clock-accuracy' : 'Set dataplane PTP clock accuracy',
         'VALUE' :  CliMatcher.IntegerMatcher( 0x00, 0xff,
                                               helpdesc="Clock accuracy value" )
         }
   handler = PtpCli.setPtpClockAccuracy
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PtpClockAccuracyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp clock-identity MAC_CLOCK_IDENTITY | EUI_CLOCK_IDENTITY
#--------------------------------------------------------------------------------
class PtpClockIdentityCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp clock-identity MAC_CLOCK_IDENTITY | EUI_CLOCK_IDENTITY'
   noOrDefaultSyntax = 'ptp clock-identity ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'clock-identity': 'Set dataplane PTP Clock Identity',
      'MAC_CLOCK_IDENTITY': CliCommand.Node(
         matcher=MacAddr.MacAddrMatcher(
            helpdesc='Clock Identity in MAC address format' ),
         alias='CLOCK_IDENTITY' ),
      'EUI_CLOCK_IDENTITY': CliCommand.Node(
         matcher=CliMatcher.PatternMatcher(
            PtpCli.eui64Pattern,
            helpname='EUI-64 bit number',
            helpdesc='clockIdentity in EUI-64 bit format' ),
         alias='CLOCK_IDENTITY' )
   }
   handler = PtpCli.setPtpClockIdentity
   noOrDefaultHandler = PtpCli.unsetPtpClockIdentity

BasicCliModes.GlobalConfigMode.addCommandClass( PtpClockIdentityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp domain VALUE
#--------------------------------------------------------------------------------
class PtpDomainValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp domain VALUE'
   noOrDefaultSyntax = 'ptp domain ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'domain': 'Set the dataplane PTP domain',
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.domainNumberRange,
         helpdesc='The PTP domain number' ),
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'domainNumber',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                               'domainNumber',
                                                                no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpDomainValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp forward-unicast
#--------------------------------------------------------------------------------
class PtpForwardUnicastCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp forward-unicast'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'forward-unicast': CliCommand.guardedKeyword( 'forward-unicast',
         helpdesc=( 'Configure switch to forward dataplane PTP unicast packets in '
                    'PTP boundary mode' ),
         guard=PtpCli.ptpForwardUnicastGuard ),
   }
   handler = lambda mode, args: PtpCli.forwardPtpUnicast( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.forwardPtpUnicast( mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpForwardUnicastCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp forward-v1
#--------------------------------------------------------------------------------
class PtpForwardV1Cmd( CliCommand.CliCommandClass ):
   syntax = 'ptp forward-v1'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'forward-v1': CliCommand.guardedKeyword( 'forward-v1',
         helpdesc='Configure switch to forward dataplane PTP V1 packets',
         guard=PtpCli.ptpForwardV1SupportedGuard ),
   }
   handler = lambda mode, args: PtpCli.forwardPtpV1( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.forwardPtpV1( mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpForwardV1Cmd )

#--------------------------------------------------------------------------------
# ptp hold-ptp-time VALUE
#--------------------------------------------------------------------------------
class PtpHoldPtpTimeValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp hold-ptp-time VALUE'
   noOrDefaultSyntax = 'ptp hold-ptp-time ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'hold-ptp-time': 'Hold dataplane PTP offset and skew',
      'VALUE': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.minHoldPtpTimeInterval,
         PtpCli.CliConstants.maxHoldPtpTimeInterval,
         helpdesc='Hold PTP Time in seconds' ),
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'holdPtpTimeInterval',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                               'holdPtpTimeInterval',
                                                                no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpHoldPtpTimeValueCmd )

#--------------------------------------------------------------------------------
# ptp skew sample size VALUE
#--------------------------------------------------------------------------------
class NumSkewSamplesToCalibrated( CliCommand.CliCommandClass ):
   syntax = 'ptp skew sample size VALUE'
   noOrDefaultSyntax = 'ptp skew sample size ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'skew': CliCommand.guardedKeyword( 'skew',
         helpdesc='Skew config commands',
         guard=None ),
      'sample': CliMatcher.KeywordMatcher( 'sample',
         helpdesc='Skew sample config commands' ),
      'size': CliMatcher.KeywordMatcher( 'size',
         helpdesc='Skew sample size config commands' ),
      'VALUE': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.skewMinQueueSize,
         PtpCli.CliConstants.skewMaxQueueSize,
         helpdesc='Number of skew samples before transitioning from uncalibrated '
                  'to slave port state' ),
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                      'numSkewSamplesToCalibrated',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, \
                        args: PtpCli.setGlobalAttr( mode,
                                                    'numSkewSamplesToCalibrated',
                                                    no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( NumSkewSamplesToCalibrated )

#--------------------------------------------------------------------------------
# [ no | default ] ptp local-priority VALUE
#--------------------------------------------------------------------------------
class PtpLocalPriorityValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp local-priority VALUE'
   noOrDefaultSyntax = 'ptp local-priority ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'local-priority': ( 'Set the value of the dataplane PTP clock local priority '
                          '( G8275 profile BMCA )' ),
      'VALUE': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.minLocalPriority, 255, helpdesc='Local Priority' ),
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'localPriority',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                                 'localPriority',
                                                                 no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpLocalPriorityValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp management all
#--------------------------------------------------------------------------------
class PtpManagementAllCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp management all'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'management': 'Configure dataplane PTP Management messages processing',
      'all': 'Process all Management messages',
   }
   handler = lambda mode, args: PtpCli.setManagementEnabled( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.setManagementEnabled( mode,
                                                                        no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpManagementAllCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp message-type event dscp VALUE default
#--------------------------------------------------------------------------------
class GlobalDscpEventCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp message-type event dscp VALUE default'
   noOrDefaultSyntax = 'ptp message-type event dscp [ VALUE ] default ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'message-type': matcherMessageType,
      'event': 'Configure dataplane ptp event message attributes',
      'dscp': matcherDscp,
      'VALUE': matcherDscpValue,
      'default': 'Set for all interfaces',
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'globalDscpEvent',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                                 'globalDscpEvent',
                                                                 no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( GlobalDscpEventCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp message-type general dscp VALUE default
#--------------------------------------------------------------------------------
class GlobalDscpGeneralCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp message-type general dscp VALUE default'
   noOrDefaultSyntax = 'ptp message-type general dscp [ VALUE ] default ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'message-type': matcherMessageType,
      'general': 'Configure dataplane ptp general message attributes',
      'dscp': matcherDscp,
      'VALUE': matcherDscpValue,
      'default': 'Set for all interfaces',
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'globalDscpGeneral',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                                 'globalDscpGeneral',
                                                                 no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( GlobalDscpGeneralCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp mode MODE
#--------------------------------------------------------------------------------
class PtpModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp mode MODE'
   noOrDefaultSyntax = 'ptp mode ...'

   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'mode': matcherMode,
      'MODE': CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( PtpCli.ptpModeDict ),
         guard=PtpCli.ptpModeGuard
      )
   }
   handler = lambda mode, args: PtpCli.setPtpMode( mode, args[ 'MODE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPtpMode( mode, 'disabled' )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpModeCmd )

#--------------------------------------------------------------------------------
# ptp mode e2etransparent one-step
#--------------------------------------------------------------------------------
class PtpModeE2EtransparentOneStepCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp mode e2etransparent one-step'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'mode': matcherMode,
      'e2etransparent': CliCommand.guardedKeyword(
         'e2etransparent',
         helpdesc='End-to-End Transparent Clock',
         guard=PtpCli.ptpE2ETransparentKeywordForOneStepGuard ),
      'one-step': CliCommand.guardedKeyword( 'one-step',
         helpdesc='Configure one-step variant for this PTP mode',
         guard=PtpCli.ptpOneStepTransparentClockSupportedGuard ),
   }
   handler = lambda mode, args: PtpCli.setPtpMode( mode,
      'ptpOneStepEndToEndTransparentClock' )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpModeE2EtransparentOneStepCmd )

#--------------------------------------------------------------------------------
# ptp mode boundary one-step
#--------------------------------------------------------------------------------
class PtpModeBoundaryOneStepCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp mode boundary one-step'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'mode': matcherMode,
      'boundary': CliCommand.guardedKeyword(
         'boundary',
         helpdesc='Boundary Clock Mode',
         guard=PtpCli.ptpBoundaryClockSupportedGuard ),
      'one-step': CliCommand.guardedKeyword( 'one-step',
         helpdesc='Configure one-step variant for this PTP mode',
         guard=PtpCli.ptpOneStepBoundaryClockSupportedGuard ),
   }
   handler = lambda mode, args: PtpCli.setPtpMode( mode,
      'ptpOneStepBoundaryClock' )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpModeBoundaryOneStepCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor
#--------------------------------------------------------------------------------
class PtpMonitorCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
   }
   handler = lambda mode, args: PtpCli.setPtpMonitor( mode )
   noHandler = lambda mode, args: PtpCli.setPtpMonitor( mode, no=True )
   defaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor sequence-id
#--------------------------------------------------------------------------------
class PtpMonitorSequenceIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor sequence-id'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'sequence-id': 'Turn on/off logging of missing messages',
   }

   handler = lambda mode, args: PtpCli.setLogMissingMsgs( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.setLogMissingMsgs( mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorSequenceIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold missing-message TYPE VALUE sequence-ids
#--------------------------------------------------------------------------------
class PtpMonitorSpecificMsgsSeqIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold missing-message TYPE VALUE sequence-ids'
   noOrDefaultSyntax =\
         'ptp monitor threshold missing-message TYPE [ VALUE ] sequence-ids ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'missing-message': 'Turn on/off logging of missing messages',
      'TYPE': CliMatcher.EnumMatcher( {
         'sync': '',
         'follow-up': '',
         'delay-resp': '',
         'announce': '', } ),
      'VALUE': CliMatcher.IntegerMatcher( 2, 255, helpdesc='' ),
      'sequence-ids': '',
   }
   handler = lambda mode, args: PtpCli.setMissingMsgSeqId( mode, args[ 'TYPE' ],\
         args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setMissingMsgSeqId(
         mode, args[ 'TYPE' ], no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorSpecificMsgsSeqIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold missing-message TYPE INTERVAL intervals
#--------------------------------------------------------------------------------
class PtpMonitorSpecificMsgsTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold missing-message TYPE INTERVAL intervals'
   noOrDefaultSyntax =\
         'ptp monitor threshold missing-message TYPE [ INTERVAL ] intervals'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'missing-message': 'Turn on/off logging of missing messages',
      'TYPE': CliMatcher.EnumMatcher( {
         'sync': '',
         'follow-up': '',
         'announce': '', } ),
      'INTERVAL': CliMatcher.IntegerMatcher( 2, 255, helpdesc='' ),
      'intervals': '',
   }
   handler = lambda mode, args: PtpCli.setMissingMsgTimer( mode, args[ 'TYPE' ],\
         args[ 'INTERVAL' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setMissingMsgTimer(
         mode, args[ 'TYPE' ], no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorSpecificMsgsTimerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold missing-message
#                                     TYPE INTERVAL intervals VALUE sequence-ids
#--------------------------------------------------------------------------------
class PtpMonitorSpecificMsgsTimerSeqIdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold missing-message TYPE INTERVAL intervals VALUE\
         sequence-ids'
   noOrDefaultSyntax = 'ptp monitor threshold missing-message TYPE [ INTERVAL ]\
         intervals [ VALUE ] sequence-ids'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'missing-message': 'Turn on/off logging of missing messages',
      'TYPE': CliMatcher.EnumMatcher( {
         'sync': '',
         'follow-up': '',
         'announce': '', } ),
      'INTERVAL': CliMatcher.IntegerMatcher( 2, 255, helpdesc='' ),
      'intervals': '',
      'VALUE': CliMatcher.IntegerMatcher( 2, 255, helpdesc='' ),
      'sequence-ids': '',
   }
   handler = lambda mode, args: PtpCli.setMissingMsgTimerAndSeqId(
         mode, args[ 'TYPE' ], args[ 'INTERVAL' ], args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setMissingMsgTimerAndSeqId(
         mode, args[ 'TYPE' ], no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorSpecificMsgsTimerSeqIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold mean-path-delay VALUE
#--------------------------------------------------------------------------------
class PtpMonitorDelayThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold mean-path-delay VALUE'
   noOrDefaultSyntax = 'ptp monitor threshold mean-path-delay ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'mean-path-delay': matcherMPD,
      'VALUE': CliMatcher.IntegerMatcher( 0, 1000000000,
         helpdesc='Offset from 0 (ns)' ),
   }
   handler = lambda mode, args: PtpCli.setPtpMonitorDelayThreshold(
         mode, args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPtpMonitorDelayThreshold(
         mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorDelayThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold offset-from-master VALUE
#--------------------------------------------------------------------------------
class PtpMonitorOffsetThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold offset-from-master VALUE'
   noOrDefaultSyntax = 'ptp monitor threshold offset-from-master ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'offset-from-master': matcherOffset,
      'VALUE': CliMatcher.IntegerMatcher( 0, 1000000000,
         helpdesc='+/- of the offset entered (ns)' ),
   }
   handler = lambda mode, args: PtpCli.setPtpMonitorOffsetThreshold(
         mode, args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPtpMonitorOffsetThreshold(
         mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorOffsetThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold skew VALUE
#--------------------------------------------------------------------------------
class PtpMonitorThresholdSkewValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold skew VALUE'
   noOrDefaultSyntax = 'ptp monitor threshold skew ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'skew': matcherSkew,
      'VALUE': CliMatcher.FloatMatcher( 0, 10,
         helpdesc='Percentage offset from 1.0 (1 = 100%).',
         precisionString='%.25g' ),
   }
   handler = lambda mode, args: PtpCli.setPtpMonitorSkewThreshold(
         mode, args[ 'VALUE' ]  )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPtpMonitorSkewThreshold(
         mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorThresholdSkewValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold mean-path-delay VALUE nanoseconds drop
#--------------------------------------------------------------------------------
class PtpMonitorDelayDropThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold mean-path-delay VALUE nanoseconds drop'
   noOrDefaultSyntax = syntax.replace( 'VALUE', '[ VALUE ]' )
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'mean-path-delay': matcherMPD,
      'VALUE': CliMatcher.IntegerMatcher( 0, 1000000000,
         helpdesc='Offset from 0 (ns)' ),
      'nanoseconds' : matcherNanosec,
      'drop' : matcherDrop,
   }
   handler = PtpCli.setPtpMonitorDelayDropThreshold
   noOrDefaultHandler = PtpCli.setNoPtpMonitorDelayDropThreshold

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorDelayDropThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold offset-from-master VALUE nanoseconds drop
#--------------------------------------------------------------------------------
class PtpMonitorOffsetDropThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold offset-from-master VALUE nanoseconds drop'
   noOrDefaultSyntax = syntax.replace( 'VALUE', '[ VALUE ]' )
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'offset-from-master': matcherOffset,
      'VALUE': CliMatcher.IntegerMatcher( 0, 1000000000,
         helpdesc='+/- of the offset entered (ns)' ),
      'nanoseconds' : matcherNanosec,
      'drop' : matcherDrop,
   }
   handler = PtpCli.setPtpMonitorOffsetDropThreshold
   noOrDefaultHandler = PtpCli.setNoPtpMonitorOffsetDropThreshold

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorOffsetDropThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp monitor threshold skew VALUE drop
#--------------------------------------------------------------------------------
class PtpMonitorThresholdSkewDropValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp monitor threshold skew VALUE drop'
   noOrDefaultSyntax = syntax.replace( 'VALUE', '[ VALUE ]' )
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'monitor': matcherMonitor,
      'threshold': matcherThreshold,
      'skew': matcherSkew,
      'VALUE': CliMatcher.FloatMatcher( 0, 10,
         helpdesc='Percentage offset from 1.0 (1 = 100%).',
         precisionString='%.25g' ),
      'drop' : matcherDrop,
   }
   handler = PtpCli.setPtpMonitorSkewDropThreshold
   noOrDefaultHandler = PtpCli.setNoPtpMonitorSkewDropThreshold

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMonitorThresholdSkewDropValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp netsync-monitor delay-request
#--------------------------------------------------------------------------------
class PtpNetsyncMonitorDelayRequestCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp netsync-monitor delay-request'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'netsync-monitor': 'Configure dataplane PTP NetSync Monitor',
      'delay-request': 'Use DelayRequest message mechanism',
   }

   handler = lambda mode, args: PtpCli.setNetSyncMonitor( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.setNetSyncMonitor( mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpNetsyncMonitorDelayRequestCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp priority1 VALUE
#--------------------------------------------------------------------------------
class PtpPriority1ValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp priority1 VALUE'
   noOrDefaultSyntax = 'ptp priority1 ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'priority1': 'Set the value of priority1 dataplane PTP',
      'VALUE': CliMatcher.IntegerMatcher( 0, 255, helpdesc='Priority' ),
   }
   handler = lambda mode, args: PtpCli.setPriority1( mode, 'priority1',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPriority1( mode, 'priority1',
                                                                 no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpPriority1ValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp priority2 VALUE
#--------------------------------------------------------------------------------
class PtpPriority2ValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp priority2 VALUE'
   noOrDefaultSyntax = 'ptp priority2 ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'priority2': 'Set the value of priority2 for dataplane PTP',
      'VALUE': CliMatcher.IntegerMatcher( 0, 255, helpdesc='Priority' ),
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'priority2',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'priority2',
                                                                 no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpPriority2ValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp profile PROFILE
#--------------------------------------------------------------------------------
class PtpProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp profile ( g8275.1 | g8275.2 )'
   noOrDefaultSyntax = 'ptp profile ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'profile': 'Configure dataplane PTP in a specific profile',
      'g8275.1': CliCommand.guardedKeyword( 'g8275.1',
         helpdesc='Use G8275.1 profile',
         guard=PtpCli.ptpG82751Guard ) ,
      'g8275.2': CliMatcher.KeywordMatcher( 'g8275.2',
         helpdesc='Use G8275.2 profile' ),
   }
   handler = lambda mode, args: PtpCli.setPtpProfile( mode, 'ptpProfile',
         'g8275.1' if 'g8275.1' in args else 'g8275.2' )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPtpProfile( mode, 'ptpProfile',
         no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpProfileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp source ip IP_ADDR
#--------------------------------------------------------------------------------
class PtpSourceIpIpaddrCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp source ip IP_ADDR'
   noOrDefaultSyntax = 'ptp source ip ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'source': matcherSource,
      'ip': CliToken.Ip.ipMatcherForConfig,
      'IP_ADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='IP address' ),
   }
   handler = lambda mode, args: PtpCli.setPtpSrcIp4( mode, args[ 'IP_ADDR' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPtpSrcIp4( mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpSourceIpIpaddrCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp source ipv6 IP6_ADDR
#--------------------------------------------------------------------------------
class PtpSourceIpv6Ip6AddrCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp source ipv6 IP6_ADDR'
   noOrDefaultSyntax = 'ptp source ipv6 ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'source': matcherSource,
      'ipv6': CliToken.Ipv6.ipv6MatcherForConfig,
      'IP6_ADDR': Ip6AddrMatcher.Ip6AddrMatcher( helpdesc='IPv6 address' ),
   }
   handler = lambda mode, args: PtpCli.setPtpSrcIp6( mode, args[ 'IP6_ADDR' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setPtpSrcIp6( mode, no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpSourceIpv6Ip6AddrCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp ttl VALUE
#--------------------------------------------------------------------------------
class PtpTtlCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp ttl VALUE'
   noOrDefaultSyntax = 'ptp ttl ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'ttl': 'Set the TTL of dataplane PTP packets',
      'VALUE': CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL of PTP packets' ),
   }

   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'ttl', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'ttl',
                                                                 no=True )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpTtlCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp sync limit VALUE
#-------------------------------------------------------------------------------
class PtpSyncLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp sync limit VALUE'
   noOrDefaultSyntax = 'ptp sync limit ..'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'sync': 'Master sync options',
      'limit': 'Maximum time drift from master before resync',
      'VALUE': CliMatcher.IntegerMatcher( 50, int( 5e8 ), helpdesc='Limit in ns' ),
   }

   handler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'driftLimit',
                                                      args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode, 'driftLimit',
                                                                 no=True )

#BasicCliModes.GlobalConfigMode.addCommandClass( PtpSyncLimitCmd )

#-------------------------------------------------------------------------------
# ptp boundary ( unicast-negotiation | multicast )
#-------------------------------------------------------------------------------
class PtpBoundaryCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp boundary VALUE'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'boundary': 'Configuring Boundary mode',
      'VALUE': CliMatcher.EnumMatcher( {
         'multicast': 'Use multicast variant',
         'unicast-negotiation': 'Use unicast-negotiation variant'
      } ),
   }

   handler = lambda mode, args: PtpCli.boundaryUnicastNegotiation( mode,
                                                                   args[ 'VALUE' ] )

# BUG284794 - This is not something needed for now, just comment this out.
#BasicCliModes.GlobalConfigMode.addCommandClass( PtpBoundaryCmd )

#--------------------------------------------------------------------------------
# ptp ip access-group ACL_NAME in
#--------------------------------------------------------------------------------
class PtpIpAccessGroupAclnameInCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp ip access-group ACL_NAME in'
   noOrDefaultSyntax = 'ptp ip access-group ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'ip': AclCli.ipKwForServiceAclMatcher,
      'access-group': AclCli.accessGroupKwMatcher,
      'ACL_NAME': AclCli.ipAclNameMatcher,
      'in': AclCli.inKwMatcher,
   }
   handler = lambda mode, args: PtpCli.aclNameIs( mode, 'ip', args[ 'ACL_NAME' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.aclNameDel( mode, 'ip' )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpIpAccessGroupAclnameInCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp mpass message sync timeout VALUE messages
#-------------------------------------------------------------------------------
class PtpMpassSyncTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp mpass message sync timeout VALUE messages'
   noOrDefaultSyntax = 'ptp mpass message sync timeout ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'mpass': 'MPASS protocol (dataplane only)',
      'message': 'Configure message information',
      'sync': 'sync message',
      'timeout': 'Message timeout',
      'VALUE': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.mpassMinTimeout,
         PtpCli.CliConstants.mpassMaxTimeout,
         helpdesc='Number of messages until timeout' ),
      'messages': 'Messages'
   }
   handler = PtpCli.setMpassSyncTimeout
   noOrDefaultHandler = PtpCli.setMpassSyncTimeout

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMpassSyncTimeoutCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp mpass message delay-req timeout VALUE seconds
#-------------------------------------------------------------------------------
class PtpMpassDelayReqTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp mpass message delay-req timeout VALUE seconds'
   noOrDefaultSyntax = 'ptp mpass message delay-req timeout ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'mpass': 'MPASS protocol (dataplane only)',
      'message': 'Configure message information',
      'delay-req': 'delay-req message',
      'timeout': 'Message timeout',
      'VALUE': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.mpassMinTimeoutSec,
         PtpCli.CliConstants.mpassMaxTimeoutSec,
         helpdesc='Number of seconds until timeout' ),
      'seconds': 'Seconds'
   }
   handler = PtpCli.setMpassDelayReqTimeout
   noOrDefaultHandler = PtpCli.setMpassDelayReqTimeout

BasicCliModes.GlobalConfigMode.addCommandClass( PtpMpassDelayReqTimeoutCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp free-running
#-------------------------------------------------------------------------------
class PtpFreeRunningQuiet( CliCommand.CliCommandClass ):
   syntax = 'ptp free-running'
   noOrDefaultSyntax = 'ptp free-running ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'free-running': matcherFreeRunningQuiet,
   }
   handler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                      'freeRunningQuiet',
                                                      False )
   noHandler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                        'freeRunningQuiet',
                                                        True )
   defaultHandler = lambda mode, args: PtpCli.setGlobalAttr( mode,
                                                             'freeRunningQuiet',
                                                             False )

BasicCliModes.GlobalConfigMode.addCommandClass( PtpFreeRunningQuiet )

#-------------------------------------------------------------------------------
# [no|default] ptp free-running source clock (hardware|system)
#-------------------------------------------------------------------------------
class PtpFreeRunningTimeSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp free-running source clock ( hardware | system )'
   noOrDefaultSyntax = 'ptp free-running source clock ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'free-running': matcherFreeRunning,
      'source': 'Configure clock source when free running',
      'clock': 'Clock source',
      'hardware': CliMatcher.KeywordMatcher( 'hardware',
         helpdesc='Use hardware clock for PTP time when free-running' ),
      'system': CliCommand.guardedKeyword( 'system',
         helpdesc=( 'Use system time for PTP time when free-running' ),
         guard=PtpCli.ptpSyncFromSystemClockGuard ),
   }
   handler = PtpCli.setFreeRunningTimeSource
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PtpFreeRunningTimeSourceCmd )
