#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliModel as AclCliModel
import CliPlugin.IntfCli as IntfCli
import CliPlugin.PtpCli as PtpCli
import CliPlugin.PtpCliModel as PtpCliModel
import CliPlugin.PtpDebugCounterModel as PtpDebugCounterModel
import CliPlugin.VlanCli as VlanCli
import CliToken.Ip
import ShowCommand
from Toggles.PtpLibToggleLib import togglePtpIsolatedSlaveEnabled
from MultiRangeRule import MultiRangeMatcher

matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Details on PTP' )
matcherLocalClock = CliMatcher.KeywordMatcher( 'local-clock',
      helpdesc='Show PTP local clock information' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Show the source IP for PTP' )
matcherUnicastNegotiation = CliMatcher.KeywordMatcher( 'unicast-negotiation',
      helpdesc='Show Unicast Negotiation information' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Show PTP VLAN information' )
matcherMpass = CliMatcher.KeywordMatcher( 'mpass',
      helpdesc='Show PTP MPASS information' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='Show PTP counter information' )
matcherIsolatedFollower = CliMatcher.KeywordMatcher( 'isolated-follower',
      helpdesc='Show PTP isolated follower information' )
matcherRegion = CliMatcher.KeywordMatcher( 'region',
      helpdesc='Show PTP region information' )
matcherDomainNumber = CliMatcher.KeywordMatcher( 'domain-number',
      helpdesc='Show PTP domain-number information' )

#--------------------------------------------------------------------------------
# show ptp
#--------------------------------------------------------------------------------
class PtpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
   }
   handler = PtpCli.showPtpSummary
   cliModel = PtpCliModel.PtpSummary

BasicCli.addShowCommandClass( PtpCmd )

#--------------------------------------------------------------------------------
# show ptp mpass
#--------------------------------------------------------------------------------
class PtpMpassCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp mpass'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'mpass': matcherMpass,
   }
   handler = PtpCli.showPtpMpassSummary
   cliModel = PtpCliModel.PtpMpassSummary

BasicCli.addShowCommandClass( PtpMpassCmd )

#--------------------------------------------------------------------------------
# show ptp foreign-master-record
#--------------------------------------------------------------------------------
class PtpForeignMasterRecordCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp foreign-master-record'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'foreign-master-record': 'Show PTP foreign master information',
   }
   handler = PtpCli.showFMR
   cliModel = PtpCliModel.PtpForeignMasterRecords

BasicCli.addShowCommandClass( PtpForeignMasterRecordCmd )

#--------------------------------------------------------------------------------
# show ptp hold-ptp-time
#--------------------------------------------------------------------------------
class PtpHoldPtpTimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp hold-ptp-time'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'hold-ptp-time': 'Show PTP hold ptp time interval information',
   }
   handler = PtpCli.showHoldPtpTime
   cliModel = PtpCliModel.PtpHoldPtpTimeForShow

BasicCli.addShowCommandClass( PtpHoldPtpTimeCmd )

#--------------------------------------------------------------------------------
# show ptp interface [ INTFS ] [ enabled ]
#--------------------------------------------------------------------------------
class PtpInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTFS ] [ enabled ]'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTFS': IntfCli.Intf.rangeMatcher,
      'enabled': 'Displays only ptp enabled interfaces',
   }
   handler = PtpCli.showIntfList
   cliModel = PtpCliModel.PtpInterfaces

BasicCli.addShowCommandClass( PtpInterfaceCmd )

#--------------------------------------------------------------------------------
# show ptp mpass interface [ INTFS ]
#--------------------------------------------------------------------------------
class PtpMpassInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp mpass interface [ INTFS ]'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'mpass': matcherMpass,
      'interface': matcherInterface,
      'INTFS': IntfCli.Intf.rangeMatcher,
   }
   handler = PtpCli.showMpassIntfList
   cliModel = PtpCliModel.PtpInterfaces

BasicCli.addShowCommandClass( PtpMpassInterfaceCmd )

#--------------------------------------------------------------------------------
# show ptp interface INTFS vlan VLAN_ID
#--------------------------------------------------------------------------------
class PtpInterfaceIntfVlanVlanidCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface INTFS vlan VLAN_ID'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTFS': IntfCli.Intf.rangeMatcher,
      'vlan': matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
   }
   handler = PtpCli.showIntfListVlan
   cliModel = PtpCliModel.PtpInterfaces

BasicCli.addShowCommandClass( PtpInterfaceIntfVlanVlanidCmd )

#--------------------------------------------------------------------------------
# show ptp interface [ INTFS ] [ vlan VLAN_ID ] counters
#--------------------------------------------------------------------------------
class PtpInterfaceVlanVlanidCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTFS ] [ vlan VLAN_ID ] counters'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTFS': IntfCli.Intf.rangeMatcher,
      'vlan': matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'counters': matcherCounters,
   }
   handler = PtpCli.showIntfListCounters
   cliModel = PtpCliModel.PtpCounters

BasicCli.addShowCommandClass( PtpInterfaceVlanVlanidCountersCmd )

#--------------------------------------------------------------------------------
# show ptp mpass interface [ INTFS ] counters
#--------------------------------------------------------------------------------
class PtpInterfaceMpassCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp mpass interface [ INTFS ] counters'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'mpass': matcherMpass,
      'interface': matcherInterface,
      'INTFS': IntfCli.Intf.rangeMatcher,
      'counters': matcherCounters,
   }
   handler = PtpCli.showIntfListMpassCounters
   cliModel = PtpCliModel.PtpCounters

BasicCli.addShowCommandClass( PtpInterfaceMpassCountersCmd )

#--------------------------------------------------------------------------------
# show ptp interface [ INTFS ] counters drop
#--------------------------------------------------------------------------------
class PtpShowIntfCounterDrop( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTFS ] counters drop'
   data = { 'ptp' : PtpCli.ptpMatcherForShow,
            'interface' : matcherInterface,
            'INTFS' : IntfCli.Intf.rangeMatcher,
            'counters' : matcherCounters,
            'drop' : 'Show PTP drop counter information'
         }
   handler = PtpCli.showIntfDropCounters
   cliModel = PtpDebugCounterModel.PtpDebugCounters

BasicCli.addShowCommandClass( PtpShowIntfCounterDrop )

#--------------------------------------------------------------------------------
# show ptp interface [ INTFS ] vlans
#--------------------------------------------------------------------------------
class PtpInterfaceIntfVlansCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTFS ] vlans'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTFS': IntfCli.Intf.rangeMatcher,
      'vlans': 'Show PTP VLANs information',
   }
   handler = PtpCli.showIntfListVlans
   cliModel = PtpCliModel.PtpVlans

BasicCli.addShowCommandClass( PtpInterfaceIntfVlansCmd )

#--------------------------------------------------------------------------------
# show ptp ( clock | local-clock )
#--------------------------------------------------------------------------------
class PtpClockCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ( clock | local-clock )'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'clock': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'clock',
            helpdesc='Show PTP local clock information' ),
         deprecatedByCmd='show ptp local-clock' ),
      'local-clock': matcherLocalClock,
   }
   handler = lambda mode, args: PtpCli.showClock( mode )
   cliModel = PtpCliModel.PtpClock

BasicCli.addShowCommandClass( PtpClockCmd )

#--------------------------------------------------------------------------------
# show ptp ( masters | parent )
#--------------------------------------------------------------------------------
class PtpMastersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ( masters | parent )'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'masters': 'Show PTP master properties',
      'parent': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'parent',
            helpdesc='Show PTP parent properties' ),
         deprecatedByCmd='show ptp masters' ),
   }
   handler = PtpCli.showParent
   cliModel = PtpCliModel.PtpParentWrapper

BasicCli.addShowCommandClass( PtpMastersCmd )

#--------------------------------------------------------------------------------
# show ptp monitor
#--------------------------------------------------------------------------------
class PtpMonitorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp monitor'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'monitor': 'Show recent PTP data',
   }
   handler = PtpCli.showPtpMonitor
   cliModel = PtpCliModel.PtpMonitorData

BasicCli.addShowCommandClass( PtpMonitorCmd )

#--------------------------------------------------------------------------------
# show ptp source
#--------------------------------------------------------------------------------
class PtpSourceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp source'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'source': matcherSource,
   }
   handler = PtpCli.showSrcIp
   cliModel = PtpCliModel.PtpSourceIp

BasicCli.addShowCommandClass( PtpSourceCmd )

#--------------------------------------------------------------------------------
# show ptp source ip
#--------------------------------------------------------------------------------
class PtpSourceIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp source ip'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'source': matcherSource,
      'ip': CliToken.Ip.ipMatcherForShow,
   }
   handler = PtpCli.showSrcIp4
   cliModel = PtpCliModel.PtpSourceIp

BasicCli.addShowCommandClass( PtpSourceIpCmd )

#--------------------------------------------------------------------------------
# show ptp local-clock time-properties
#--------------------------------------------------------------------------------
class PtpLocalClockTimePropertiesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ( ( local-clock time-properties ) | time-property )'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'local-clock': matcherLocalClock,
      'time-properties': 'Show PTP clock time properties',
      'time-property': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'time-property',
            helpdesc='Show PTP clock time property' ),
         deprecatedByCmd='show ptp local-clock time-properties' ),
   }
   handler = PtpCli.showTimeProperty
   cliModel = PtpCliModel.PtpTimePropertyWrapper

BasicCli.addShowCommandClass( PtpLocalClockTimePropertiesCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation candidate-grantor
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationCandidateGrantorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation candidate-grantor'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'candidate-grantor': 'Show candidate masters of an interface',
   }
   handler = PtpCli.showCandidateGrantors
   cliModel = PtpCliModel.UcastNegCandidateGrantor

BasicCli.addShowCommandClass( PtpUnicastNegotiationCandidateGrantorCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation granted
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationGrantedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation granted'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'granted': 'Show unicast negotiation contracts granted',
   }
   handler = PtpCli.showGrantedSlaves
   cliModel = PtpCliModel.UcastNegStatus

BasicCli.addShowCommandClass( PtpUnicastNegotiationGrantedCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation profile
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation profile [ PROFILE ]'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'profile': 'Show Unicast Negotiation profiles',
      'PROFILE': CliMatcher.DynamicNameMatcher(
         lambda mode: PtpCli.ptpConfig.ucastNegProfile,
         'Unicast Negotiation profile name' ),
   }
   handler = PtpCli.showUcastNegProfiles
   cliModel = PtpCliModel.UcastNegProfiles

BasicCli.addShowCommandClass( PtpUnicastNegotiationProfileCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation remote-grantee
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationRemoteGranteeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation remote-grantee'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'remote-grantee': 'Show remote grantees of an interface',
   }
   handler = PtpCli.showRemoteGrantees
   cliModel = PtpCliModel.UcastNegRemoteGrantee

BasicCli.addShowCommandClass( PtpUnicastNegotiationRemoteGranteeCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation requested
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationRequestedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation requested'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'requested': 'Show unicast negotiation contracts requested',
   }
   handler = PtpCli.showRequestedMasters
   cliModel = PtpCliModel.UcastNegStatus

BasicCli.addShowCommandClass( PtpUnicastNegotiationRequestedCmd )

#--------------------------------------------------------------------------------
# show ptp ip access-list
#--------------------------------------------------------------------------------
class PtpIpAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ip access-list'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'ip': AclCli.ipKwForShowServiceAcl,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
   }
   handler = PtpCli.showPtpIpAcl
   cliModel = AclCliModel.AllAclList

BasicCli.addShowCommandClass( PtpIpAccessListCmd )

#--------------------------------------------------------------------------------
# show ptp isolated-follower monitor interface INTF [ vlan VLAN_ID ]
#--------------------------------------------------------------------------------
class PtpIsolatedFollowerMonitorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp isolated-follower monitor interface INTF [ vlan VLAN_ID ]'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'isolated-follower': matcherIsolatedFollower,
      'monitor': 'Show recent PTP data',
      'interface': matcherInterface,
      'INTF': IntfCli.Intf.matcher,
      'vlan': matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
   }
   handler = PtpCli.showIsolatedFollowerMonitor
   cliModel = PtpCliModel.PtpMonitorData

if togglePtpIsolatedSlaveEnabled():
   BasicCli.addShowCommandClass( PtpIsolatedFollowerMonitorCmd )

# --------------------------------------------------------------------------------
# show ptp region domain-number
# --------------------------------------------------------------------------------
class PtpRegionDomainNumberCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp region domain-number [ DOMAIN_NUMBER ] [ interface INTFS ]'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'region': matcherRegion,
      'domain-number': matcherDomainNumber,
      'DOMAIN_NUMBER': MultiRangeMatcher(
                          rangeFn=lambda: (
                             PtpCli.CliConstants.minDomainNumber1588V2,
                             PtpCli.CliConstants.maxDomainNumber1588V2 ),
                          noSingletons=False,
                          helpdesc='PTP domain number' ),
      'interface': CliMatcher.KeywordMatcher( 'interface',
                      'Show PTP region information for interfaces' ),
      'INTFS': IntfCli.Intf.rangeMatcher,
   }
   handler = PtpCli.showPtpDomainNumbers
   cliModel = PtpCliModel.PtpDomainNumbersSummary

BasicCli.addShowCommandClass( PtpRegionDomainNumberCmd )
