#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import Tac
# pylint: disable-next=consider-using-from-import
import CliPlugin.EthIntfCli as EthIntfCli
# pylint: disable-next=consider-using-from-import
import CliPlugin.LoopbackIntfCli as LoopbackIntfCli
import CliPlugin.PtpCli as PtpCli # pylint: disable=consider-using-from-import
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import
from CliPlugin.LagIntfCli import LagIntfConfigModelet
from CliPlugin.PtpGlobalConfigMode import ( matcherMessageType, matcherDscp,
                                            matcherDscpValue )
from CliPlugin.VirtualIntfRule import ( IntfMatcher, VirtualIntfMatcher )
from LoopbackIntfUtils import ( maxLoopbackIntfNum, minLoopbackIntfNum )
from PtpLib import MessageTypeNumber, lagTxMessageStr
from Toggles.PtpLibToggleLib import ( togglePtpIsolatedSlaveEnabled,
                                      togglePtpLearnIntfVlanDomainEnabled )

matcherAnnounce = CliMatcher.KeywordMatcher( 'announce',
      helpdesc='Configure announce message parameters' )
matcherDelayMechanism = CliMatcher.KeywordMatcher( 'delay-mechanism',
      helpdesc='Delay mechanism in boundary clock mode' )
matcherDelayReq = CliMatcher.KeywordMatcher( 'delay-req',
      helpdesc='Configure delay request message parameters' )
matcherInterval = CliMatcher.KeywordMatcher( 'interval',
      helpdesc='Interval between messages, base 2 log' )
matcherPdelayNeighborThreshold = CliMatcher.KeywordMatcher(
      'pdelay-neighbor-threshold',
      helpdesc='Configure link propagation threshold' )
matcherPdelayReq = CliMatcher.KeywordMatcher( 'pdelay-req',
      helpdesc='Configure peer delay request message parameters' )
matcherSync = CliMatcher.KeywordMatcher( 'sync',
      helpdesc='Configure sync message parameters' )
matcherSyncMessage = CliMatcher.KeywordMatcher( 'sync-message',
      helpdesc='Configure sync message parameters' )
matcherTimeout = CliMatcher.KeywordMatcher( 'timeout',
      helpdesc='Message interval timeout multiplier' )
matcherTransport = CliMatcher.KeywordMatcher( 'transport',
      helpdesc='Transport mode in boundary clock mode' )
matcherDebug = CliMatcher.KeywordMatcher( 'debug',
      helpdesc='Configure tx-interface for debugging' )
matcherTxMessage = CliMatcher.KeywordMatcher( 'message',
      helpdesc='Tx message type' )
matcherTxIntf = CliMatcher.KeywordMatcher( 'tx-interface',
      helpdesc='Port-Channel member interface to tx PTP message' )
matcherDiag = CliMatcher.KeywordMatcher( 'diag',
      helpdesc='Configure diagnostic and test tools' )
matcherIsolatedFollower = CliMatcher.KeywordMatcher( 'isolated-follower',
      helpdesc='Isolated follower in boundary clock mode' )

intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher

lagTxMessageType = {
  'announce': MessageTypeNumber.messageAnnounce,
  'sync': MessageTypeNumber.messageSync,
  'follow-up': MessageTypeNumber.messageFollowUp,
  'delay-req': MessageTypeNumber.messageDelayReq,
  'delay-resp': MessageTypeNumber.messageDelayResp
}

#--------------------------------------------------------------------------------
# [ no | default ] ptp enable
#--------------------------------------------------------------------------------
class PtpEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp enable'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'enable': 'Enable PTP on this port',
   }
   handler = lambda mode, args: PtpCli.enablePtp( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.enablePtp( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpEnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp enable
#--------------------------------------------------------------------------------
class PtpEnableSyncTestCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp enable synctest'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'enable': 'Enable PTP on this port',
      'synctest': 'Puts interface in synctest mode'
   }
   hidden = True
   handler = lambda mode, args: PtpCli.enableSyncTest( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.enableSyncTest( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpEnableSyncTestCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp delay-mechanism VALUE
#--------------------------------------------------------------------------------
class PtpDelayMechanismCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp delay-mechanism VALUE'
   noOrDefaultSyntax = 'ptp delay-mechanism ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'delay-mechanism': matcherDelayMechanism,
      'VALUE': CliMatcher.EnumMatcher( {
         'e2e': 'Delay req/resp mechanism',
         'p2p': 'Peer-to-peer mechanism',
      } )
   }
   handler = lambda mode, args: PtpCli.setDelayMechanism( mode,
         'delayMechanism', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setDelayMechanism( mode,
         'delayMechanism', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpDelayMechanismCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp announce interval VALUE
#--------------------------------------------------------------------------------
class PtpAnnounceIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp announce interval VALUE'
   noOrDefaultSyntax = 'ptp announce interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'announce': matcherAnnounce,
      'interval': matcherInterval,
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.announceIntervalRange,
         helpdesc='Log of announce message interval (secs)' ),
   }

   @staticmethod
   def handler( mode, args ):
      if PtpCli.ptpG82751Active():
         mode.addWarning( "Announce message rate is fixed to 8/s "
                          "in PTP profile g8275.1" )
      PtpCli.setIntfAttr( mode, 'logAnnounceInterval', args[ 'VALUE' ] )

   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logAnnounceInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpAnnounceIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp delay-req interval VALUE
#--------------------------------------------------------------------------------
class PtpDelayReqIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp delay-req interval VALUE'
   noOrDefaultSyntax = 'ptp delay-req interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'delay-req': matcherDelayReq,
      'interval': matcherInterval,
      'VALUE': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.minlogMinDelayReqInterval,
         PtpCli.CliConstants.maxlogMinDelayReqInterval,
         helpdesc='Log of delay req interval (secs)' ),
   }

   @staticmethod
   def handler( mode, args ):
      if PtpCli.ptpG82751Active():
         mode.addWarning( "Delay-req message rate is fixed to 16/s "
                          "in PTP profile g8275.1" )
      PtpCli.setIntfAttr( mode, 'logMinDelayReqInterval', args[ 'VALUE' ] )

   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logMinDelayReqInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpDelayReqIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp local-priority VALUE
#--------------------------------------------------------------------------------
class PtpLocalPriorityValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp local-priority VALUE'
   noOrDefaultSyntax = 'ptp local-priority ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'local-priority': ( 'Set the value of interface local priority '
                          '( G8275 profile BMCA )' ),
      'VALUE': CliMatcher.IntegerMatcher( 1, 255, helpdesc='Local Priority' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'localPriority', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'localPriority', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpLocalPriorityValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp message-type event dscp VALUE
#--------------------------------------------------------------------------------
class PtpMessageTypeEventDscpValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp message-type event dscp VALUE'
   noOrDefaultSyntax = 'ptp message-type event dscp ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'message-type': matcherMessageType,
      'event': 'Configure ptp event message attributes',
      'dscp': matcherDscp,
      'VALUE': matcherDscpValue
   }
   handler = lambda mode, args: PtpCli.setIntfDscpEvent( mode, args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfDscpEvent( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpMessageTypeEventDscpValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp message-type general dscp VALUE
#--------------------------------------------------------------------------------
class PtpMessageTypeGeneralDscpValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp message-type general dscp VALUE'
   noOrDefaultSyntax = 'ptp message-type general dscp ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'message-type': matcherMessageType,
      'general': 'Configure ptp general message attributes',
      'dscp': matcherDscp,
      'VALUE': matcherDscpValue
   }
   handler = lambda mode, args: PtpCli.setIntfDscpGeneral( mode, args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfDscpGeneral( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpMessageTypeGeneralDscpValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp management drop
#--------------------------------------------------------------------------------
class PtpMessageTypeManagementDropCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp management drop'
   noOrDefaultSyntax = 'ptp management drop'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'management': 'Configure PTP management message attributes',
      'drop': 'Set PTP management messages to drop'
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'disableManagementMessageForwarding', value=True )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'disableManagementMessageForwarding', value=False )

VlanCli.SwitchportModelet.addCommandClass( PtpMessageTypeManagementDropCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp pdelay-neighbor-threshold VALUE
#--------------------------------------------------------------------------------
class PtpPdelayNeighborThresholdValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp pdelay-neighbor-threshold VALUE'
   noOrDefaultSyntax = 'ptp pdelay-neighbor-threshold ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'pdelay-neighbor-threshold': matcherPdelayNeighborThreshold,
      'VALUE': CliMatcher.IntegerMatcher( 0, 10 **10,
         helpdesc='Threshold in nanoseconds' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'linkDelayThreshold', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'linkDelayThreshold', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpPdelayNeighborThresholdValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp pdelay-req interval VALUE
#--------------------------------------------------------------------------------
class PtpPdelayReqIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp pdelay-req interval VALUE'
   noOrDefaultSyntax = 'ptp pdelay-req interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'pdelay-req': matcherPdelayReq,
      'interval': matcherInterval,
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.pDelayIntervalRange,
         helpdesc='Log of peer delay req interval (secs)' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logMinPdelayReqInterval', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logMinPdelayReqInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpPdelayReqIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp role ROLE
#--------------------------------------------------------------------------------
class PtpRoleCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp role ROLE'
   noOrDefaultSyntax = 'ptp role ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'role': 'Role in boundary clock mode',
      'ROLE': CliMatcher.EnumMatcher( {
         'dynamic': 'PTP protocol determines role',
         'master': 'PTP master port',
      } )
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'role', args[ 'ROLE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'role', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpRoleCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp sync interval VALUE
#--------------------------------------------------------------------------------
class PtpSyncIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp sync interval VALUE'
   noOrDefaultSyntax = 'ptp sync interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'sync': matcherSync,
      'interval': CliCommand.Node( matcher=matcherInterval,
         deprecatedByCmd='[no] ptp sync-message interval' ),
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.syncIntervalRange,
         helpdesc='Log of sync interval (secs)' ),
   }
   handler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', args[ 'VALUE' ], deprecatedCmd=True )
   noOrDefaultHandler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', no=True, deprecatedCmd=True )

VlanCli.SwitchportModelet.addCommandClass( PtpSyncIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp sync-message interval VALUE
#--------------------------------------------------------------------------------
class PtpSyncMessageIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp sync-message interval VALUE'
   noOrDefaultSyntax = 'ptp sync-message interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'sync-message': matcherSyncMessage,
      'interval': matcherInterval,
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.syncIntervalRange,
         helpdesc='Log of sync interval (secs)' ),
   }
   handler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpSyncMessageIntervalValueCmd )


#--------------------------------------------------------------------------------
# [ no | default ] ptp sync timeout TIMEOUT
#--------------------------------------------------------------------------------
class PtpSyncTimeoutTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp sync timeout TIMEOUT'
   noOrDefaultSyntax = 'ptp sync timeout ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'sync': matcherSync,
      'timeout': matcherTimeout,
      'TIMEOUT': CliMatcher.IntegerMatcher( 2, 255,
         helpdesc='Sync interval timeout multiplier' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'syncReceiptTimeout', args[ 'TIMEOUT' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'syncReceiptTimeout', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpSyncTimeoutTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp announce timeout TIMEOUT
#--------------------------------------------------------------------------------
class PtpAnnounceTimeoutTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp announce timeout TIMEOUT'
   noOrDefaultSyntax = 'ptp announce timeout ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'announce': matcherAnnounce,
      'timeout': matcherTimeout,
      'TIMEOUT': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.minannounceReceiptTimeout,
         PtpCli.CliConstants.maxannounceReceiptTimeout,
         helpdesc='Announce interval timeout multiplier' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'announceReceiptTimeout', args[ 'TIMEOUT' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'announceReceiptTimeout', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpAnnounceTimeoutTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp transport TRANSPORT
#--------------------------------------------------------------------------------
class PtpTransportCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp transport TRANSPORT'
   noOrDefaultSyntax = 'ptp transport ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'transport': matcherTransport,
      'TRANSPORT': CliMatcher.EnumMatcher( {
         'layer2': 'Layer 2 messaging',
         'ipv4': 'Ipv4 messaging',
         'ipv6': 'Ipv6 messaging',
      } )
   }
   handler = lambda mode, args: PtpCli.setTransportMode( mode,
         'transportMode', args[ 'TRANSPORT' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setTransportMode( mode,
         'transportMode', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpTransportCmd )

#-------------------------------------------------------------------------------
# ptp profile g8275.1 destination mac-address ( forwardable | non-forwardable )
#
# [no|default] no ptp profile g8275.1 destination mac-address
#-------------------------------------------------------------------------------
class MacAddressCmd( CliCommand.CliCommandClass ):
   syntax = "ptp profile g8275.1 destination mac-address \
             ( forwardable | non-forwardable )"
   noOrDefaultSyntax = "ptp profile g8275.1 destination mac-address ..."
   data = {
      'ptp' : PtpCli.ptpMatcherForConfig,
      'profile' : "PTP profile",
      'g8275.1' : "G8275.1 profile",
      'destination': "Set destination Mac address",
      'mac-address': "Set destination Mac address",
      'forwardable': CliMatcher.KeywordMatcher(
         'forwardable',
         helpdesc='set multicast address to 01:1B:19:00:00:00' ),
      'non-forwardable': CliMatcher.KeywordMatcher(
         'non-forwardable',
         helpdesc='set multicast address to 01:80:C2:00:00:0E' )
   }

   @staticmethod
   def handler( mode, args ):
      EthAddr = Tac.Type( "Arnet::EthAddr" )
      if not PtpCli.ptpG82751Active():
         mode.addWarning( "Destination multicast address is only used "
                          "in PTP profile g8275.1" )
      if 'forwardable' in args:
         PtpCli.setIntfAttr( mode, 'pktTxMacAddress', EthAddr.ethAddrPtp )
      elif 'non-forwardable' in args:
         PtpCli.setIntfAttr( mode, 'pktTxMacAddress', EthAddr.ethAddrPtpPeerDelay )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      EthAddr = Tac.Type( "Arnet::EthAddr" )
      if not PtpCli.ptpG82751Active():
         mode.addWarning( "Destination multicast address is only used "
                          "in PTP profile g8275.1" )
      PtpCli.setIntfAttr( mode, 'pktTxMacAddress', EthAddr.ethAddrPtpPeerDelay )

VlanCli.SwitchportModelet.addCommandClass( MacAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp vlan ( all | VLANSET )
#--------------------------------------------------------------------------------
class PtpVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp vlan ( all | VLAN_SET )'
   noOrDefaultSyntax = 'ptp vlan [ all | VLAN_SET ]'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'vlan': 'Configure PTP to run on VLANs in trunk mode',
      'all': 'All allowed vlans on trunk',
      'VLAN_SET': VlanCli.vlanSetMatcher
   }
   handler = PtpCli.setPtpVlan
   noOrDefaultHandler = PtpCli.setPtpVlan

VlanCli.SwitchportModelet.addCommandClass( PtpVlanCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp mpass
#--------------------------------------------------------------------------------
class PtpMpassCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp mpass'
   noOrDefaultSyntax = 'ptp mpass'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'mpass': 'Enable MPASS'
   }
   handler = PtpCli.setPtpMpass
   noOrDefaultHandler = PtpCli.setPtpMpass

LagIntfConfigModelet.addCommandClass( PtpMpassCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp diag role isolated-follower
#--------------------------------------------------------------------------------
class PtpEnableIsolatedFollowerCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp diag role isolated-follower'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'diag': matcherDiag,
      'role': 'Role in boundary clock mode',
      'isolated-follower': matcherIsolatedFollower
   }
   handler = lambda mode, args: PtpCli.enableIsolatedFollower( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.enableIsolatedFollower( mode,
                                                                          no=True )

if togglePtpIsolatedSlaveEnabled():
   VlanCli.SwitchportModelet.addCommandClass( PtpEnableIsolatedFollowerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp isolated-follower vlan VLAN_SET
#--------------------------------------------------------------------------------
class PtpEnableIsolatedFollowerVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp diag isolated-follower vlan VLAN_SET'
   noOrDefaultSyntax = 'ptp diag isolated-follower vlan [ VLAN_SET ]'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'diag': matcherDiag,
      'isolated-follower': matcherIsolatedFollower,
      'vlan': 'Configure PTP to run on VLANs in trunk mode',
      'VLAN_SET': VlanCli.vlanSetMatcher
   }
   handler = PtpCli.enableIsolatedFollowerVlan
   noOrDefaultHandler = PtpCli.enableIsolatedFollowerVlan

if togglePtpIsolatedSlaveEnabled():
   VlanCli.SwitchportModelet.addCommandClass( PtpEnableIsolatedFollowerVlanCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp debug [message {(announce|sync|follow-up|delay-req|delay-resp)}]
#              tx-interface INTF
#-------------------------------------------------------------------------------
class PtpTxIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp debug [ message MESSAGES ] tx-interface INTF'
   noOrDefaultSyntax = 'ptp debug [ message MESSAGES ] tx-interface ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'debug': matcherDebug,
      'message': matcherTxMessage,
      'MESSAGES': CliCommand.SetEnumMatcher( {
         'announce': 'announce message',
         'sync': 'sync message',
         'follow-up': 'followUp message',
         'delay-req': 'delayReq message',
         'delay-resp': 'delayResp message',
      } ),
      'tx-interface': matcherTxIntf,
      'INTF': intfMatcher,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      messages = args.get( 'MESSAGES' )
      if messages:
         messages = ( lagTxMessageType[ m ] for m in messages )
      else:
         messages = list( lagTxMessageStr )
      args[ 'MESSAGES' ] = messages

   handler = PtpCli.setPtpTxIntf
   noOrDefaultHandler = PtpCli.unsetPtpTxIntf

LagIntfConfigModelet.addCommandClass( PtpTxIntfCmd )

#-------------------------------------------------------------------------------
# In "interface" mode
#
# [no|default] ptp local-interface <INTERFACE>
#-------------------------------------------------------------------------------

# Same as the LoopbackIntf matcher, but with a custom helpdesc as decided upon
# in the cli-review.
ptpLocalIntfMatcher = VirtualIntfMatcher(
   'Loopback', minLoopbackIntfNum, maxLoopbackIntfNum,
   helpdesc='Interface to use for receiving and sending PTP messages',
   value=lambda mode, intf: LoopbackIntfCli.LoopbackIntf( intf, mode ) )

class PtpSrcIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp local-interface INTERFACE'
   noOrDefaultSyntax = 'ptp local-interface ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'local-interface': 'PTP local interface',
      'INTERFACE': ptpLocalIntfMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      # no has to be set to None, not False. setIntfAttr has strange logic
      # where no=False means use default intfAttr value.
      # BUG729012
      PtpCli.setPtpSrcIntf( mode, args[ 'INTERFACE' ], no=None )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      PtpCli.setPtpSrcIntf( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpSrcIntfCmd )

# -------------------------------------------------------------------------------
# In "interface" mode
#
# [no|default] ptp region [ vlan ( all | VLAN_SET ) ] domain-number ( ( learn
# lock TIME seconds ) | DOMAIN_NUMBER )
# -------------------------------------------------------------------------------
class PtpRegionDomainIntfVlanCmd( CliCommand.CliCommandClass ):
   _baseSyntax = 'ptp region [ vlan ( all | VLAN_SET ) ] domain-number'
   _domainModeArgs = ' DOMAIN_NUMBER'

   if togglePtpLearnIntfVlanDomainEnabled():
      _domainModeArgs = ' ( ( dynamic locked TIME seconds ) | DOMAIN_NUMBER )'

   syntax = _baseSyntax + _domainModeArgs
   noOrDefaultSyntax = _baseSyntax + ' ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'region': 'Configure attributes for the region connected to this interface',
      'vlan': 'Configure attributes per PTP VLAN',
      'all': 'All PTP VLANs',
      'VLAN_SET': VlanCli.vlanSetMatcher,
      'domain-number': 'PTP domain number',
      'DOMAIN_NUMBER': PtpCli.ptpMatcherForDomainNumber,
   }

   if togglePtpLearnIntfVlanDomainEnabled():
      data.update( {
          'dynamic': 'Allow PTP domain number learning',
          'locked': 'Learned PTP domain number locking configuration',
          'TIME': CliMatcher.FloatMatcher( 0, 1e11,
                     helpdesc=( 'Idle time before a learned PTP domain number is '
                                'replaceable' ),
                     # used only to get <0-1e+11> in help string
                     # any float value is accepted as is (no rounding happens)
                     precisionString='%.1g' ),
          'seconds': 'Seconds'
      } )

   @staticmethod
   def _commonhandler( mode, args, no ):
      defaultPtpVlanIds = { PtpCli.CliConstants.defaultPortDSVlanId }
      vlans = PtpCli.getPtpCommandVlanSet(
         args, mode, allowedOnly=False, default=defaultPtpVlanIds )

      domainMode = None
      if 'dynamic' in args:
         domainMode = Tac.Type( 'Ptp::PtpPortDomainMode' ).learnedPortDomainNumber
      elif 'DOMAIN_NUMBER' in args:
         domainMode = Tac.Type( 'Ptp::PtpPortDomainMode' ).staticPortDomainNumber
      else:
         # Must provide a domain number or learn
         # token unless 'no' is specified
         assert no

      # Ordering below is not important, the agent will handle
      # any intermediate states. For example the mode may be set
      # to learn while lock time is null. The global domain number
      # will be sent out until lock time has a value. This time
      # is assumed to be very short in practice

      # When learn is specified for the domain number
      # we are wiping the static domain number
      PtpCli.setRegionIntfVlanAttr( mode,
                                    vlans=vlans,
                                    attr='networkSideDomainNumber',
                                    value=args.get( 'DOMAIN_NUMBER' ),
                                    no=no )
      # if domainMode is None we set the mode
      # to the default - globalDomainNumber
      PtpCli.setRegionIntfVlanAttr( mode,
                                    vlans=vlans,
                                    attr='domainNumberMode',
                                    value=domainMode,
                                    no=no )
      # When learn is not specified for the domain number
      # we are wiping the lock time regardless of whether
      # 'no' is specified
      PtpCli.setRegionIntfVlanAttr( mode,
                                    vlans=vlans,
                                    attr='lockTime',
                                    value=args.get( 'TIME' ),
                                    no=no )

   @staticmethod
   def handler( mode, args ):
      PtpRegionDomainIntfVlanCmd._commonhandler( mode, args, no=False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      PtpRegionDomainIntfVlanCmd._commonhandler( mode, args, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpRegionDomainIntfVlanCmd )
