# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements PtpTimeSync Cli.  In particular, it provides:
# 
# Global Configuration
# --------------------
# [no|default] ptp hardware-sync interval <interval>
#
# EXEC Mode Commands
# ------------------
# reset ptp hardware-sync
# reset platform arad NAME ptp hardware-sync [ in SandFap ]
#
# Show Commands
# -------------
# show ptp hardware-sync
# show platform arad NAME ptp hardware-sync [ in SandFap ]
#
# Hidden Commands
# ---------------
# reset platform ptp hardware-sync slave NAME
# show platform ptp hardware-sync master NAME
# show platform ptp hardware-sync slave NAME
#
#-------------------------------------------------------------------------------

import BasicCli
import BasicCliModes
import Cell
import ConfigMount
import CliCommand
import CliMatcher
import CliParser
from CliPlugin import PtpCli
import CliPlugin.TechSupportCli
from CliPlugin import PtpTimeSyncModel
from CliPlugin.PtpTimeSyncCliLib import TimestampFeatureConfigAggregatorSm
from CliPlugin.TapAggIntfCli import timestampHeaderSupportedGuard
import CliToken.Mac
import CliToken.Platform
import CliToken.Reset
import LazyMount
import ShowCommand
import Tac

fruConfig = None
cliConfig = None
stateConfig = None
resetConfig = None
status = None
ptpHwCapability = None
cellStatuses = None
hwStatus = None
timestampFeatureConfigDir = None

def ptpTimeSyncSupported():
   if fruConfig.supported:
      return True
   return False

def ptpHwSyncIntervalConfigNoOpsSupported():
   for sliceDir in ptpHwCapability:
      if ptpHwCapability[ sliceDir ].ptpHwSyncIntervalConfigNoOpsSupported:
         return True
   return False

def ptpHwSyncIntervalConfigGuardSupported():
   for sliceDir in ptpHwCapability:
      if ptpHwCapability[ sliceDir ].ptpHwSyncIntervalConfigGuardSupported:
         return True
   return False

def ptpTimeSyncSupportedGuard( mode, token ):
   if ptpTimeSyncSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpTimeSyncAndSyncIntervalConfigSupportedGuard( mode, token ):
   if ptpTimeSyncSupported() and not ptpHwSyncIntervalConfigGuardSupported():
      return None
   return CliParser.guardNotThisPlatform

def broadsyncSupportedGuard( mode, token ):
   broadsync = Tac.Type( "Inventory::PtpTimeSync::InternalSyncMode" ).broadsync
   if ptpTimeSyncSupported() and fruConfig.internalSyncMode == broadsync:
      return None
   return CliParser.guardNotThisPlatform

def updateScdsAndSwitches():
   scds = sorted( fruConfig.scd.keys() )
   switchChips = sorted( fruConfig.switchAsic.keys() )

   for switchChip in resetConfig.slaveResetRequests:
      if switchChip not in switchChips:
         del resetConfig.slaveResetRequests[ switchChip ]

   for switchChip in switchChips:
      if switchChip not in resetConfig.slaveResetRequests:
         resetConfig.slaveResetRequests[ switchChip ] = 0

   return ( scds, switchChips )

def switchChipNames():
   _, switchChips = updateScdsAndSwitches()

   return switchChips

def scdNames():
   scds, _ = updateScdsAndSwitches()

   return scds

slaveNameMatcher = CliMatcher.DynamicNameMatcher( lambda mode: switchChipNames(),
      'Slave Name', pattern='[A-Za-z0-9\\/]+' )

ptpMatcherForTimeSyncShow = PtpCli.guardedPtpShowPartial(
                            guard=ptpTimeSyncSupportedGuard )
ptpMatcherForTimeSyncConfig = PtpCli.guardedPtpConfigPartial(
                              guard=ptpTimeSyncAndSyncIntervalConfigSupportedGuard )

hwTimeSyncNodeForConfig = CliCommand.guardedKeyword(
                           'hardware-sync',
                           helpdesc='PTP Hardware Time Synchronization ' \
                                    'configuration',
                           guard=ptpTimeSyncAndSyncIntervalConfigSupportedGuard )

hwTimeSyncNodeForShow = CliCommand.guardedKeyword(
                           'hardware-sync',
                           helpdesc='Show PTP Hardware Time Synchronization ' \
                                    'information',
                           guard=ptpTimeSyncSupportedGuard )
hwTimeSyncNodeForExec = CliCommand.guardedKeyword(
                           'hardware-sync',
                           helpdesc='PTP Hardware Time Synchronization',
                           guard=ptpTimeSyncSupportedGuard )

def resetPtpTimeSync( mode, args ):
   if stateConfig.state == 'on':
      resetConfig.globalResetRequests += 1 # pylint: disable-msg=E1103
   else:
      print( "PTP is disabled. Enable it first." )

def resetPtpTimeSyncOnSwitch( mode, switchName ):
   updateScdsAndSwitches()
   
   if switchName not in fruConfig.switchAsic:
      return
   switchChip = None
   for sliceDir in hwStatus:
      if switchName in hwStatus[ sliceDir ].switchAsic:
         switchChip = hwStatus[ sliceDir ].switchAsic[ switchName ]
   if not switchChip or switchChip.genId != fruConfig.switchAsic[ switchName ].genId:
      return

   if stateConfig.state == 'on':
      if switchName in resetConfig.slaveResetRequests:
         resetConfig.slaveResetRequests[ switchName ] += 1
   else:
      print( "PTP is disabled. Enable it first." )

#--------------------------------------------------------------------------------
# reset ptp hardware-sync
#--------------------------------------------------------------------------------
class ResetPtpHardwareSyncCmd( CliCommand.CliCommandClass ):
   syntax = 'reset ptp hardware-sync'
   data = {
      'reset': CliToken.Reset.resetMatcher,
      'ptp': PtpCli.ptpMatcherForReset,
      'hardware-sync': hwTimeSyncNodeForExec,
   }
   handler = resetPtpTimeSync

BasicCliModes.EnableMode.addCommandClass( ResetPtpHardwareSyncCmd )

def setPtpTimeSyncHbInterval( mode, args ):
   # if no-ops for the sync interval config command is enabled, ignore the interval
   # argument passed in and set interval to default
   if ptpHwSyncIntervalConfigNoOpsSupported():
      cliConfig.heartbeatInterval = cliConfig.defaultHeartbeatInterval
   else:
      # We support only intervals that are multiples of 100.
      intervalIn100Ms = args[ 'INTERVAL' ] // 100
      cliConfig.heartbeatInterval = intervalIn100Ms * 100

def resetPtpTimeSyncHbInterval( mode, args ):
   cliConfig.heartbeatInterval = cliConfig.defaultHeartbeatInterval

#--------------------------------------------------------------------------------
# ptp hardware-sync interval INTERVAL
#--------------------------------------------------------------------------------
class PtpHardwareSyncIntervalIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp hardware-sync interval INTERVAL'
   noOrDefaultSyntax = 'ptp hardware-sync interval ...'
   data = {
      'ptp': ptpMatcherForTimeSyncConfig,
      'hardware-sync': hwTimeSyncNodeForConfig,
      'interval': CliCommand.guardedKeyword( 'interval',
         helpdesc='Configure PTP Hardware Time Synchronization heartbeat interval',
         guard=broadsyncSupportedGuard ),
      'INTERVAL': CliMatcher.IntegerMatcher( 500, 1500,
         helpdesc=( 'Heartbeat Interval for PTP Hardware Time Synchronization '
                    'in miliseconds ( multiples of 100 only ).' ) ),
   }
   handler = setPtpTimeSyncHbInterval
   noOrDefaultHandler = resetPtpTimeSyncHbInterval

BasicCliModes.GlobalConfigMode.addCommandClass( PtpHardwareSyncIntervalIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac timestamp scale SCALE
#--------------------------------------------------------------------------------
def setTimeScale( mode, args ):
   cliConfig.timeScale = args[ 'SCALE' ]

def setTimeScaleDefault( mode, args ):
   cliConfig.timeScale = 'tai'

class MacTimestampScaleCmd( CliCommand.CliCommandClass ):
   syntax = 'mac _timestamp scale SCALE'
   noOrDefaultSyntax = 'mac _timestamp scale ...'
   data = {
      'mac': CliToken.Mac.macMatcherForConfig,
      '_timestamp': CliCommand.guardedKeyword( 'timestamp',
         helpdesc='Configure global timestamp properties',
         guard=timestampHeaderSupportedGuard ),
      'scale': CliCommand.guardedKeyword( 'scale',
         helpdesc='Time scale for timestamp header',
         guard=ptpTimeSyncSupportedGuard ),
      'SCALE': CliMatcher.EnumMatcher( {
         'utc': 'UTC time scale',
         'tai': 'TAI time scale',
      } )
   }
   handler = setTimeScale
   noOrDefaultHandler = setTimeScaleDefault

BasicCliModes.GlobalConfigMode.addCommandClass( MacTimestampScaleCmd )

#===============================================================================
#   Show Commands
#===============================================================================

def showSlaveInfo( mode, switchName ):
   switchChips = switchChipNames()

   asm = PtpTimeSyncModel.SlavesModel()
   if switchName not in switchChips or switchName not in fruConfig.switchAsic:
      return asm
   switchChip = None
   for sliceDir in hwStatus:
      if switchName in hwStatus[ sliceDir ].switchAsic:
         switchChip = hwStatus[ sliceDir ].switchAsic[ switchName ]
   if not switchChip or switchChip.genId != fruConfig.switchAsic[ switchName ].genId:
      return asm

   sm = PtpTimeSyncModel.SlaveModel()
   sm.phaseDifferences = None
   sm.state = PtpTimeSyncModel.switchStateToEnum[ switchChip.state ]
   if switchChip.state in ( PtpTimeSyncModel.switchStates.switchSynchronizing,
                            PtpTimeSyncModel.switchStates.switchTimeSyncStable ):
      if switchChip.latestInfo:
         sm.localCounter = switchChip.latestInfo.counter
         sm.masterTod = switchChip.latestInfo.tod
         if switchChip.timeToStabilityInSecs:
            sm.timeToStability = switchChip.timeToStabilityInSecs
      phases = [ int( x ) for x in switchChip.phaseDifference.values() ]
      sm.phaseDifferences = phases
      sm.updates = switchChip.numUpdates
      sm.crcErrors = switchChip.crcErrors
      sm.slaveFreqErrors = switchChip.slaveFreqErrors
      sm.invalidTimestampErrors = switchChip.invalidTimestampErrors
      sm.timeoutErrors = switchChip.timeoutErrors
      sm.fifoRetryLimitExceeded = switchChip.fifoRetryLimitExceeded
      sm.retriesLeftSelfTest = switchChip.retriesLeftSelfTest
   asm.slaves[ switchName ] = sm
   return asm

def showAllSlavesInfo( mode ):
   switchChips = switchChipNames()
   m = PtpTimeSyncModel.SlavesModel()
   for switchName in switchChips:
      m.slaves.update( showSlaveInfo( mode, switchName ).slaves )
   return m

def showMasterInfo( mode, scdName ):
   scds = scdNames()
   amm = PtpTimeSyncModel.MastersModel()
   if scdName not in scds:
      return amm
   cellStatus = None
   for cell in cellStatuses:
      if scdName in cellStatuses[ cell ].scd:
         cellStatus = cellStatuses[ cell ]
         break
   if cellStatus is None:
      return amm
   scd = cellStatus.scd[ scdName ]
   mm = PtpTimeSyncModel.MasterModel()
   mm.phaseDifferences = None
   mm.state = PtpTimeSyncModel.scdStateToEnum[ scd.state ]
   mm.pidState = PtpTimeSyncModel.pidStateToEnum[ scd.pidState ]
   mm.skew = status.bsyncMasterClkCycleInfo.clkCycle / 4.0
   if scd.state == PtpTimeSyncModel.scdStates.scdTimeSyncStandbySynchronizing or \
      scd.state == PtpTimeSyncModel.scdStates.scdTimeSyncStandbyStable or \
      scd.pidState != PtpTimeSyncModel.pidStates.servoDisabled :
      if scd.latestInfo:
         mm.updates = scd.latestInfo.updateId
         mm.masterTod = scd.latestInfo.remoteTod
         mm.localTod = scd.latestInfo.localTod
      phases = [ int( x ) for x in scd.phaseDifference.values() ]
      mm.phaseDifferences = phases
      if scd.timeToStabilityInSecs:
         mm.timeToStability = scd.timeToStabilityInSecs

   amm.masters[ scdName ] = mm
   return amm

def showAllMastersInfo( mode ):
   scds = scdNames()
   m = PtpTimeSyncModel.MastersModel()
   for scd in scds:
      m.masters.update( showMasterInfo( mode, scd ).masters )
   return m

def showAllInfo( mode, args ):
   updateScdsAndSwitches()
   m = PtpTimeSyncModel.TimeSyncModel()
   m.configuredState = PtpTimeSyncModel.configStateToEnum[ stateConfig.state ]
   m.operationalState = PtpTimeSyncModel.operationalStateToEnum[ status.state ]
   m.syncInterval = float( stateConfig.operationalIntervalInNs / ( 1000 * 1000 ) )
   m.masters = showAllMastersInfo( mode ).masters
   m.slaves = showAllSlavesInfo( mode ).slaves
   return m

#--------------------------------------------------------------------------------
# show platform ptp hardware-sync master SCD_NAME
#--------------------------------------------------------------------------------
class PlatformPtpHardwareSyncMasterScdnameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform ptp hardware-sync master SCD_NAME'
   data = {
      'platform': CliToken.Platform.platformMatcherForShow,
      'ptp': PtpCli.ptpMatcherForPlatformShow,
      'hardware-sync': hwTimeSyncNodeForShow,
      'master': 'Specific master',
      'SCD_NAME': CliMatcher.DynamicNameMatcher( lambda mode: scdNames(),
         'Master Name' ),
   }
   handler = lambda mode, args: showMasterInfo( mode, args[ 'SCD_NAME' ] )
   cliModel = PtpTimeSyncModel.MastersModel
   hidden = True

BasicCli.addShowCommandClass( PlatformPtpHardwareSyncMasterScdnameCmd )

#--------------------------------------------------------------------------------
# show platform ptp hardware-sync slave SWITCH_NAME
#--------------------------------------------------------------------------------
class PlatformPtpHardwareSyncSlaveSwitchnameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform ptp hardware-sync slave SWITCH_NAME'
   data = {
      'platform': CliToken.Platform.platformMatcherForShow,
      'ptp': PtpCli.ptpMatcherForPlatformShow,
      'hardware-sync': hwTimeSyncNodeForShow,
      'slave': 'Specific slave',
      'SWITCH_NAME': slaveNameMatcher
   }
   handler = lambda mode, args: showSlaveInfo( mode, args[ 'SWITCH_NAME' ] )
   cliModel = PtpTimeSyncModel.SlavesModel
   hidden = True

BasicCli.addShowCommandClass( PlatformPtpHardwareSyncSlaveSwitchnameCmd )

#--------------------------------------------------------------------------------
# show platform ptp hardware-sync
#--------------------------------------------------------------------------------
class PlatformPtpHardwareSyncCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform ptp hardware-sync'
   data = {
      'platform': CliToken.Platform.platformMatcherForShow,
      'ptp': PtpCli.ptpMatcherForPlatformShow,
      'hardware-sync': hwTimeSyncNodeForShow,
   }
   handler = showAllInfo
   cliModel = PtpTimeSyncModel.TimeSyncModel
   hidden = True

BasicCli.addShowCommandClass( PlatformPtpHardwareSyncCmd )

#--------------------------------------------------------------------------------
# show ptp hardware-sync
#--------------------------------------------------------------------------------
class PtpHardwareSyncCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp hardware-sync'
   data = {
      'ptp': ptpMatcherForTimeSyncShow,
      'hardware-sync': hwTimeSyncNodeForShow,
   }
   handler = showAllInfo
   cliModel = PtpTimeSyncModel.TimeSyncModel

BasicCli.addShowCommandClass( PtpHardwareSyncCmd )

#--------------------------------------------------------------------------------
# reset platform ptp hardware-sync slave SWITCH_NAME
#--------------------------------------------------------------------------------
class ResetPtpTimeSyncOnSwitchCmd( CliCommand.CliCommandClass ):
   syntax = 'reset platform ptp hardware-sync slave SWITCH_NAME'
   data = {
      'reset': CliToken.Reset.resetMatcher,
      'platform': CliToken.Platform.platformMatcherForReset,
      'ptp': PtpCli.ptpMatcherForReset,
      'hardware-sync': hwTimeSyncNodeForExec,
      'slave': 'Specific slave',
      'SWITCH_NAME': slaveNameMatcher
   }
   hidden = True
   handler = lambda mode, args: resetPtpTimeSyncOnSwitch( mode,
                                                          args[ 'SWITCH_NAME' ] )

BasicCliModes.EnableMode.addCommandClass( ResetPtpTimeSyncOnSwitchCmd )

# Add 'show ptp hardware-sync' to show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2013-11-20 00:00:00',
   cmds=[ 'show ptp hardware-sync' ],
   cmdsGuard=ptpTimeSyncSupported )

def Plugin( entityManager ):
   global cliConfig, stateConfig, resetConfig, fruConfig
   global status, cellStatuses, hwStatus, ptpHwCapability
   global timestampFeatureConfigDir

   cliConfig = ConfigMount.mount( entityManager, "ptpTimeSync/cliConfig",
                                  "PtpTimeSync::CliConfig", "w" )
   stateConfig = LazyMount.mount( entityManager, "ptpTimeSync/stateConfig",
                                  "PtpTimeSync::StateConfig", "r" )
   resetConfig = LazyMount.mount( entityManager, "ptpTimeSync/resetConfig",
                                  "PtpTimeSync::ResetConfig", "w" )
   fruConfig = LazyMount.mount( entityManager, "ptpTimeSync/config",
                                "PtpTimeSync::Config", "r" )
   status = LazyMount.mount( entityManager, "ptpTimeSync/status/global",
                             "PtpTimeSync::Status", "r" )
   cellStatuses = LazyMount.mount( entityManager, "ptpTimeSync/status/cell",
                                   "Tac::Dir", "ri" )
   hwStatus = LazyMount.mount( entityManager, "ptpTimeSync/hwStatus",
                               "Tac::Dir", "ri" )
   ptpHwCapability = LazyMount.mount( entityManager, "ptp/hwCapability",
                               "Tac::Dir", "ri" )

   mg = entityManager.mountGroup()
   redundancyStatus = mg.mount( Cell.path( "redundancy/status" ),
                                "Redundancy::RedundancyStatus", "r" )
   timestampFeatureConfigDir = mg.mount( "ptpTimeSync/featureConfig",
                                         "Tac::Dir",
                                         "wi" )
   timestampFeatureStatus = mg.mount( "ptpTimeSync/featureStatus",
                                      "PtpTimeSync::TimestampFeatureStatus", "w" )

   def _createTimestampFeatureConfigAggregatorSm():
      entityManager.timestampFeatureConfigAggregatorSm = \
         TimestampFeatureConfigAggregatorSm(
            timestampFeatureConfigDir, timestampFeatureStatus, redundancyStatus )

   mg.close( _createTimestampFeatureConfigAggregatorSm )
