#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from GenericReactor import GenericReactor
from TypeFuture import TacLazyType
from Tracing import (
   defaultTraceHandle,
   Handle
)
from Toggles.PtpTimeSyncLibToggleLib import (
   togglePtpTimeSyncLaunchConfigAggregatorEnabled
)
import Tac

# Trace levels:
# 0 - errors
# 1 - possible errors or transient states preventing normal operation
# 5 - details of operation, only useful for debugging

__defaultTraceHandle__ = Handle( "PtpTimeSync::TimestampFeature" )
th = defaultTraceHandle()
t0 = th.trace0
t5 = th.trace5

RedundancyMode = TacLazyType( 'Redundancy::RedundancyMode' )

EnabledStr = {
   True: 'enabled',
   False: 'disabled'
}

class TimestampFeatureConfigSm:

   cfg = None
   parent = None
   reactorEnabled = None
   reactorRequiresPtpClockSync = None

   def __init__( self, cfg, callback_func ):
      '''
      cfg : in PtpTimeSync::TimestampFeatureConfig::PtrConst
      callback_func: state change notification callback
      '''
      self.cfg = cfg
      self.callback_func = callback_func
      self.reactorEnabled = \
         GenericReactor( cfg, [ 'enabled' ],
                         self.handleEnabled )
      self.reactorRequiresPtpClockSync = \
         GenericReactor( cfg, [ 'requiresPtpClockSync' ],
                         self.handleRequiresPtpClockSync )

   def evaluateConfig( self ):
      t5( 'feature:', self.cfg.name,
          'timestamping:', EnabledStr[ self.cfg.enabled ],
          'PTP clock sync:', EnabledStr[ self.cfg.requiresPtpClockSync ] )
      self.callback_func()

   def handleEnabled( self, reactor ):
      self.evaluateConfig()

   def handleRequiresPtpClockSync( self, reactor ):
      self.evaluateConfig()

class TimestampFeatureConfigAggregatorSm:

   featureConfigDir = None
   featureStatus = None
   redundancyStatus = None

   reactorFeatureConfigDir = None
   reactorFeatureConfig = {}
   reactorRedundancyMode = None

   def __init__( self, featureConfigDir, featureStatus, redundancyStatus ):
      '''
      featureConfigDir : in Tac::Dir::PtrConst
      featureStatus : PtpTimeSync::TimestampFeatureStatus::Ptr
      redundancyStatus : Redundancy::RedundancyStatus::PtrConst
      '''
      self.featureConfigDir = featureConfigDir
      self.featureStatus = featureStatus
      self.redundancyStatus = redundancyStatus
      self.reactorFeatureConfigDir = \
         GenericReactor( self.featureConfigDir,
                         [ 'entityPtr' ],
                         self.handleFeatureConfig )
      self.reactorRedundancyMode = \
         GenericReactor( self.redundancyStatus,
                         [ 'mode' ],
                         self.handleRedundancyMode )

      self.evaluateFeatureConfig()

   def evaluateFeatureConfig( self ):
      enabled = False
      ptpClockSyncEnabled = False
      if not togglePtpTimeSyncLaunchConfigAggregatorEnabled():
         t0( 'PtpTimeSync launch config aggregator disabled"' )
      elif self.redundancyStatus.mode != RedundancyMode.active:
         t5( 'redundancy mode not', RedundancyMode.active )
      else:
         for cfg in self.featureConfigDir.values():
            enabled |= cfg.enabled
            ptpClockSyncEnabled |= cfg.enabled and cfg.requiresPtpClockSync

         self.featureStatus.enabled = enabled
         self.featureStatus.ptpClockSyncEnabled = ptpClockSyncEnabled

      t5( 'timestamping:', EnabledStr[ enabled ],
          'PTP clock sync:', EnabledStr[ ptpClockSyncEnabled ] )

   def handleFeatureConfig( self, reactor, name ):
      enabled = name in self.featureConfigDir

      if enabled:
         cfg = self.featureConfigDir[ name ]
         t5( 'feature:', name, 'timestamping:', EnabledStr[ cfg.enabled ],
             'PTP clock sync:', EnabledStr[ cfg.requiresPtpClockSync ] )
         self.reactorFeatureConfig[ name ] = TimestampFeatureConfigSm(
            cfg,
            self.evaluateFeatureConfig )

      elif name in self.reactorFeatureConfig:
         t5( 'feature:', name, 'timestamping:', EnabledStr[ enabled ] )
         del self.reactorFeatureConfig[ name ]

      self.evaluateFeatureConfig()

   def handleRedundancyMode( self, reactor ):
      t0( 'redundancy mode:', self.redundancyStatus.mode )
      if self.redundancyStatus.mode == RedundancyMode.active:
         self.evaluateFeatureConfig()
