#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AclCliLib
import AclLib
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import MultiRangeRule
import Plugins
import ShowCommand
from BasicCliModes import GlobalConfigMode
from CliMode.Qos import ( CosToTcProfileModeBase, DscpRewriteModeBase,
                          DscpToTcModeBase, ExpToTcProfileModeBase,
                          TcToCosModeBase, TcToExpModeBase )
from CliPlugin import ( QosCli, IntfCli )
from CliPlugin.IntfCli import IntfConfigMode
from CliPlugin.QosCliIntfTypes import ( ethOrLagIntfTypes, ethOrLagOrVlanIntfTypes,
                                  ethOrLagIntfPrefixes, ethOrLagOrVlanIntfPrefixes )
from CliPlugin.QosCli import ( guardAnyTcToCosPerInterface,
                               nodeQosForConfig,
                               matcherMap, tokenTcToCosMapTo,
                               guardExpToTcPerInterface,
                               tcToCosMapNameRule,
                               guardTcToCosPerInterface,
                               guardCpuTcToCosPerInterface )
from CliPlugin.QosCliCommon import ( guardPMapQosAction, matcherExpToTcProfileName,
                                     guardDscpToTcPerInterface, matcherMapExp,
                                     guardDscpToTcSupported, matcherDscpToTcMapName,
                                     matcherTrafficClass, TrafficClassExpression,
                                     matcherTrafficClassValue, matcherMapDscp,
                                     QosProfileMode, matcherSet,
                                     nodeDscpToTcMapTo )
from CliPlugin.QosCliServicePolicy import ( dropPrecedenceValueMatcher,
                                            PolicyMapClassModeQos,
                                            guardTxQueue,
                                            guardUcMcTxQueue,
                                            txQueueRangeFn,
                                            ucTxQueueRangeFn,
                                            mcTxQueueRangeFn )
from CliPlugin.QosCliModel import ( CosToTcProfileAllModel, DscpRewriteMapAllModel,
                                    DscpToTcMapAllModel,
                                    IntfDscpPreserveCollectionModel,
                                    IntfTrustModeCollectionModel,
                                    TcToCosMapAllModel, ExpToTcProfileAllModel,
                                    TcDpToExpMapAllModel )
from Intf.IntfRange import IntfRangeMatcher
from QosTypes import ( tacCos, tacDscp, tacExp, tacTrafficClass, tacDei,
                      tacDropPrecedence )

# -----------------------------------------------------------------------------------
# Guards begin
# -----------------------------------------------------------------------------------
def guardCosMatch( mode, token ):
   if qosAclHwStatus.cosMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardInnerCosMatch( mode, token ):
   if qosAclHwStatus.innerCosMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCosToTcPerPseudowire( mode, token ):
   if not qosHwStatus.pwDecapCosQosMapSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardCosToTcSupported( mode, token ):
   if guardCosToTcPerInterface( mode, token ) and \
      guardCosToTcPerPseudowire( mode, token ):
      return CliParser.guardNotThisPlatform
   return None

def guardDei( mode, token ):
   if qosHwStatus.deiPreserveSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardTcToPriorityGroupMap( mode, token ):
   if qosHwStatus.tcToPriorityGroupMapSupported:
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# For Bali and Trident
# The "[no | default] [mls] qos cos [<0-7>]" command, in "interface config" mode.
# -----------------------------------------------------------------------------------
def guardDefaultCos( mode, token ):
   if qosHwStatus.defaultCosSupported:
      if isinstance( mode, IntfConfigMode ) and mode.intf.isSubIntf():
         if qosHwStatus.trustModePerSubInterfaceSupported:
            return None
         return CliParser.guardNotThisPlatform
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# The "qos map traffic-class <tc> to cos <0-7>" command, in "global config" mode.
# -----------------------------------------------------------------------------------
def guardSetTctoCos( mode, token ):
   if qosHwStatus.cosRewriteSupported:
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# Rewrite enable/disable commands in "global config" mode:
# "[ mls ] qos rewrite [ cos | dscp ]"
# "[ no|default ] [ mls ] qos rewrite"
# -----------------------------------------------------------------------------------
def guardCosRewrite( mode, token ):
   if qosHwStatus.cosRewriteSupported and qosHwStatus.cosRewriteDisableSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpYellowAction( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported and \
      qosAclHwStatus.dscpYellowActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpPreserveMplsEncap( mode, token ):
   if not qosHwStatus.dscpPreserveIpMplsEncapSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardEgressDscpRewrite( mode, token ):
   if not qosHwStatus.dscpRewritePerInterfaceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardDscpRewriteDropPrecedence( mode, token ):
   if not qosHwStatus.dscpRewriteDropPrecedenceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardDscpRewrite( mode, token ):
   if qosHwStatus.dscpRewriteSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardTrustMode( mode, token ):
   if isinstance( mode, IntfConfigMode ) and mode.intf.isSubIntf():
      if qosHwStatus.trustModePerSubInterfaceSupported:
         return None
      return CliParser.guardNotThisPlatform
   return None

def guardInnerVlanMatch( mode, token ):
   if qosAclHwStatus.innerVlanMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardVlanMatch( mode, token ):
   if qosAclHwStatus.vlanMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpRewritePerInterface( mode, token ):
   if not qosHwStatus.dscpRewritePerInterfaceSupported:
      return CliParser.guardNotThisPlatform

   if mode.intf.isSubIntf() and \
         not qosHwStatus.dscpRewritePerSubInterfaceSupported:
      return CliParser.guardNotThisPlatform

   return None

def guardNamedTcDpToExpQosMapSupported( mode, token ):
   if not qosHwStatus.expRewriteSupported:
      return CliParser.guardNotThisPlatform
   return None

# -----------------------------------------------------------------------------------
# For Bali and Trident
# The "[no | default] [mls] qos dscp [<0-63>]" command, in "interface config" mode.
# -----------------------------------------------------------------------------------
def guardDefaultDscp( mode, token ):
   if qosHwStatus.defaultDscpSupported:
      if isinstance( mode, IntfConfigMode ) and mode.intf.isSubIntf():
         if qosHwStatus.trustModePerSubInterfaceSupported:
            return None
         return CliParser.guardNotThisPlatform
      return None
   return CliParser.guardNotThisPlatform

def guardDscpToDropPrecedenceMap( mode, token ):
   if qosHwStatus.dscpToDpMapSupported:
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# The "[ default|no ] [ mls ] qos map exp <0-7> to traffic-class <tc>" command,
# in "global config" mode.
# -----------------------------------------------------------------------------------
def guardExpToAndFromTcMap( mode, token ):
   if qosHwStatus.dynamicExpMappingSupported:
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# The "qos map traffic-class <tc> to dscp <0-63>" command, in "global config" mode.
# -----------------------------------------------------------------------------------
def guardTcToDscpMap( mode, token ):
   if qosHwStatus.tcToDscpMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCosToTcPerInterface( mode, token ):
   if isinstance( mode, IntfConfigMode ):
      if mode.intf.isSubIntf():
         if not qosHwStatus.cosToTcPerSubInterfaceSupported:
            return CliParser.guardNotThisPlatform
         return None
      else:
         if not qosHwStatus.cosToTcPerInterfaceSupported:
            return CliParser.guardNotThisPlatform
         return None

   if not qosHwStatus.cosToTcPerInterfaceSupported and \
      not qosHwStatus.cosToTcPerSubInterfaceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardDeiMatch( mode, token ):
   if qosAclHwStatus.deiMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

# -----------------------------------------------------------------------------------
# Guards end
# -----------------------------------------------------------------------------------


# -----------------------------------------------------------------------------------
# Sysdb mount paths
# -----------------------------------------------------------------------------------
pfcStatus = None
qosAclHwStatus = None
qosConfig = None
qosHwStatus = None
qosInputConfig = None
qosSliceHwStatus = None
qosStatus = None

# -----------------------------------------------------------------------------------
# Nodes and matchers
# -----------------------------------------------------------------------------------
nodeDropPrecedence = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc="Map drop-precedence value",
      guard=guardDscpToDropPrecedenceMap )
matcherInterfaces = CliMatcher.KeywordMatcher( 'interfaces',
      helpdesc='Show QoS status for a specific interface' )
matcherMls = CliMatcher.KeywordMatcher( 'mls',
      helpdesc='MLS global commands' )
matcherDscp = CliMatcher.KeywordMatcher( 'dscp',
      helpdesc='Specify DSCP match' )
matcherIntf = IntfRangeMatcher( explicitIntfTypes=ethOrLagIntfTypes )
matcherDscpPreserveIntf = IntfRangeMatcher(
                           explicitIntfTypes=ethOrLagOrVlanIntfTypes )
nodeMls = CliCommand.Node( matcher=matcherMls, hidden=True )
nodeCosToTcProfileTo = CliCommand.guardedKeyword( 'to',
      helpdesc="Configure a named Cos To Tc Map",
      guard=guardCosToTcSupported )
matcherMlsGlobal = CliMatcher.KeywordMatcher( 'mls',
      helpdesc='MLS global commands' )
nodeMlsGlobal = CliCommand.Node( matcher=matcherMlsGlobal, hidden=True )
nodeMapExp = CliCommand.guardedKeyword( 'exp', helpdesc="Map MPLS Exp value",
      guard=guardExpToAndFromTcMap )
matcherMultipleCosvalues = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( tacCos.min, tacCos.max ), noSingletons=False,
      helpdesc='Class of Service (COS) value(s) or range(s) of COS values' )
matcherMultipleExpValue = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( tacExp.min, tacExp.max ), noSingletons=False,
      helpdesc='MPLS Exp value(s) or range(s) of Exp values' )
matcherMapTrafficClass = CliMatcher.KeywordMatcher( 'traffic-class',
      helpdesc='Mapping traffic-class values' )
nodeToCos = CliCommand.guardedKeyword( 'cos',
      helpdesc="Set Class of Service (CoS) value", guard=guardSetTctoCos )
nodeToExp = CliCommand.guardedKeyword( 'exp', helpdesc="Set MPLS Exp value",
      guard=guardExpToAndFromTcMap )
nodeExpToTcProfileTo = CliCommand.guardedKeyword( 'to',
      helpdesc="Configure a named EXP To Tc Map",
      guard=QosCli.guardExpToTcSupported )
nodeRewriteCos = CliCommand.guardedKeyword( 'cos',
      helpdesc="Set CoS rewrite", guard=guardCosRewrite )
nodeCos = CliCommand.guardedKeyword( 'cos',
      helpdesc='Set default CoS value for the interface', guard=guardDefaultCos )
matcherMapCos = CliMatcher.KeywordMatcher( 'cos',
      helpdesc='Map Class of Service (CoS) values' )
matcherCosValue = CliMatcher.IntegerMatcher( tacCos.min, tacCos.max,
      helpdesc='Class of Service (CoS) value',
      priority=CliParser.PRIO_HIGH )
nodeDei = CliCommand.guardedKeyword( 'dei', helpdesc="Map DEI value",
                                      guard=guardDei )
matcherDeiValue = CliMatcher.IntegerMatcher( tacDei.min, tacDei.max,
      helpdesc='DEI value', priority=CliParser.PRIO_HIGH )
matcherMultipleDeiValues = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( tacDei.min, tacDei.max ), noSingletons=False,
      helpdesc='DEI value(s) or range(s) of DEI values' )
matcherDropPrecedence = CliMatcher.KeywordMatcher( 'drop-precedence',
      helpdesc="Set drop-precedence value" )
matcherDropPrecedenceValue = CliMatcher.IntegerMatcher(
      tacDropPrecedence.min, tacDropPrecedence.max - 1,
      helpdesc='Drop-Precedence value', priority=CliParser.PRIO_HIGH )

nodeToTxQueue = CliCommand.guardedKeyword( 'tx-queue', helpdesc="Set TX queue",
      guard=guardTxQueue )
nodeToUcTxQueue = CliCommand.guardedKeyword( 'uc-tx-queue',
      helpdesc="Set unicast TX queue", guard=guardUcMcTxQueue )
nodeMcTxQueue = CliCommand.guardedKeyword( 'mc-tx-queue',
      helpdesc="Set multicast TX queue", guard=guardUcMcTxQueue )
nodePriorityGroup = CliCommand.guardedKeyword( 'priority-group',
      helpdesc="Set priority group", guard=guardTcToPriorityGroupMap )
nodeTo = CliCommand.guardedKeyword( 'to', helpdesc="Configure a named Map",
      guard=guardEgressDscpRewrite )
nodeRewriteDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc="Set DSCP rewrite", guard=guardDscpRewrite )
nodeRewriteExp = CliCommand.guardedKeyword( 'exp', 
      helpdesc="Set EXP rewrite", guard=guardNamedTcDpToExpQosMapSupported )
cosSetMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 0, 7 ),
   noSingletons=False,
   helpdesc='CoS value(s) or range(s) of CoS values',
   value=lambda mode, grList: list( grList.values() ),
   priority=CliParser.PRIO_HIGH
)
innerCosSetMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 0, 7 ),
   noSingletons=False,
   helpdesc='Inner CoS value(s) or range(s) of inner CoS values',
   value=lambda mode, grList: list( grList.values() ),
   priority=CliParser.PRIO_HIGH
)
cosInnerKeywordMatcher = CliMatcher.KeywordMatcher( "cos",
                                                helpdesc="Specify COS match" )
nodeTrust = CliCommand.guardedKeyword( 'trust',
      helpdesc='Set the trust mode for the interface', guard=guardTrustMode )
nodeActionDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc="DSCP values", guard=guardPMapQosAction )
matcherDscpValue = CliMatcher.IntegerMatcher( tacDscp.min, tacDscp.max,
      helpdesc='DSCP value', priority=CliParser.PRIO_HIGH )
nodeActionCos = CliCommand.guardedKeyword( 'cos',
      helpdesc="Set COS", guard=guardPMapQosAction )
nodeDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc='Set default DSCP value for the interface', guard=guardDefaultDscp )
nodeToDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc="Set Packet DSCP value",
      guard=guardTcToDscpMap )
cosKeywordMatcher = CliMatcher.KeywordMatcher( "cos", helpdesc="Specify COS match" )
deiKeywordMatcher = CliMatcher.KeywordMatcher( "dei", helpdesc="Specify DEI match" )
innerCosKeywordMatcher = CliMatcher.KeywordMatcher( "inner",
                                                helpdesc="Specify inner CoS match" )
vlanKeywordMatcher = CliMatcher.KeywordMatcher( "vlan",
                                                helpdesc="Specify VLAN match" )
vlanInnerKeywordMatcher = CliMatcher.KeywordMatcher( "vlan",
                                                helpdesc="Specify VLAN match" )
innerVlanKeywordMatcher = CliMatcher.KeywordMatcher( "inner",
                                                helpdesc="Specify inner VLAN match" )
vlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 4094 ),
   noSingletons=True,
   helpdesc='VLAN value(s) or range(s) of VLAN values',
   value=lambda mode, grList: list( grList.values() ),
   priority=CliParser.PRIO_HIGH
)
innerVlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 4094 ),
   noSingletons=True,
   helpdesc="Inner VLAN value(s) or range(s) of inner VLAN values",
   value=lambda mode, grList: list( grList.values() ),
   priority=CliParser.PRIO_HIGH
)
matcherExpValue = CliMatcher.IntegerMatcher( tacExp.min, tacExp.max,
      helpdesc='EXP value',
      priority=CliParser.PRIO_HIGH )

tokenTrafficClassHelpdesc = 'Mapping traffic-class values'
MapToCosHelpdesc = 'Map to Cos'
matcherCosToTcProfileName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.cosToTcProfile, helpdesc='Map Name',
      pattern=r'(?!global-profile$)[A-Za-z0-9_:{}\[\]-]+' )
matcherMultipleDscpValues = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( tacDscp.min, tacDscp.max ), noSingletons=False,
      helpdesc='Differentiated Services Code Point (DSCP) value(s) or '
      'range(s) of DSCP values' )
matcherDscpRewriteMapName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.dscpRewriteMap,
      'Map name', pattern='(?!default-map$)[A-Za-z0-9_:{}\\[\\]-]+' )
matcherRewrite = CliMatcher.KeywordMatcher( 'rewrite',
      helpdesc='Set QoS rewrite setting' )
matcherTrafficClassConfig = CliMatcher.KeywordMatcher( 'traffic-class',
      helpdesc='Configure traffic class parameters' )
matcherTrafficClassName = CliMatcher.StringMatcher(
      helpdesc="traffic class name" )
matcherTcToExpNamedMapName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.tcDpToExpNamedMap,
      'Map name', pattern='(?!default-map$)[A-Za-z0-9_:{}\\[\\]-]+' )

def getCurrentGlobalExpToTcMapping( exp ):
   # While creating a profile we copy from the current global mapping
   # if global mapping does not exist for the exp then we fallback on the
   # platform default.
   # We also do this when user had configured a mapping say exp 5 to tc 0 and
   # then goes and does a no exp 5 to tc 0 in the profile.
   tc = qosStatus.expToTcMap[ exp ]
   if tc == tacTrafficClass.invalid:
      tc = qosHwStatus.defaultExpToTcMap[ exp ]
   return tc

def getCurrentGlobalDscpToTcMapping( dscp ):
   # While creating a map we copy from the current global mapping
   # if global mapping does not exist for the dscp then we fallback on the
   # platform default.
   # We also do this when user had configured a mapping say dscp 5 to tc 0 and
   # then goes and does a no dscp 5 to tc 0 in the map.
   tc = qosStatus.dscpToTcMap[ dscp ]
   if tc == tacTrafficClass.invalid:
      tc = qosHwStatus.defaultDscpToTcMap[ dscp ]
   return tc

def getCurrentGlobalCosToTcMapping( cos ):
   # While creating a profile we copy from the current global mapping
   # if global mapping does not exist for the cos then we fallback on the
   # platform default.
   # We also do this when user had configured a mapping say cos 5 to tc 0 and
   # then goes and does a no cos 5 to tc 0 in the profile.
   tc = qosStatus.cosToTcMap[ cos ]
   if tc == tacTrafficClass.invalid:
      tc = qosHwStatus.defaultCosToTcMap[ cos ]
   return tc

# -----------------------------------------------------------------------------------
# Cmd expressions
# -----------------------------------------------------------------------------------
class DscpRewriteMode( DscpRewriteModeBase, BasicCli.ConfigModeBase ):
   name = "Dscp Rewrite Configuration"

   def __init__( self, parent, session, param ):
      DscpRewriteModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DscpToTcMode( DscpToTcModeBase, BasicCli.ConfigModeBase ):
   name = "Dscp To Traffic-Class Mapping Configuration"

   def __init__( self, parent, session, param ):
      DscpToTcModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DeiMatchExpression( CliCommand.CliExpression ):
   expression = "dei DEI_VALUE"
   data = {
      "dei": CliCommand.Node( matcher=deiKeywordMatcher,
                              maxMatches=1, guard=guardDeiMatch ),
      "DEI_VALUE": CliMatcher.IntegerMatcher( 0, 1, helpdesc='DEI value' )
   }

class InnerCosMatchExpression( CliCommand.CliExpression ):
   expression = "cos_ inner_ INNER_COS_RANGE"
   data = {
         "cos_": CliCommand.Node( matcher=cosInnerKeywordMatcher,
                                 maxMatches=1, guard=guardInnerCosMatch ),
         "inner_": CliCommand.Node( matcher=innerCosKeywordMatcher,
                                 maxMatches=1, guard=guardInnerCosMatch ),
         "INNER_COS_RANGE": CliCommand.Node( matcher=innerCosSetMatcher,
                                             maxMatches=1 )
   }

class CosMatchExpression( CliCommand.CliExpression ):
   expression = "cos COS_RANGE"
   data = {
      "cos": CliCommand.Node( matcher=cosKeywordMatcher,
                              maxMatches=1, guard=guardCosMatch ),
      "COS_RANGE": CliCommand.Node( matcher=cosSetMatcher, maxMatches=1 )
   }

class CosToTcProfileMode( CosToTcProfileModeBase, BasicCli.ConfigModeBase ):
   name = "Cos To Traffic-Class Profile Configuration"

   def __init__( self, parent, session, param ):
      CosToTcProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ExpToTcProfileMode( ExpToTcProfileModeBase, BasicCli.ConfigModeBase ):
   name = "EXP To Traffic-Class Profile Configuration"

   def __init__( self, parent, session, param ):
      ExpToTcProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TcToCosMode( TcToCosModeBase, BasicCli.ConfigModeBase ):
   name = "Traffic-Class to Cos Mapping Configuration"

   def __init__( self, parent, session, param ):
      TcToCosModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TcToExpMode( TcToExpModeBase, BasicCli.ConfigModeBase ):
   name = "Traffic-Class to Exp Mapping Configuration"

   def __init__( self, parent, session, param ):
      TcToExpModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

# --------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF ] trust
# --------------------------------------------------------------------------------
class QosInterfacesTrustCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF ] trust'
   data = {
      'mls': nodeMls,
      'qos': QosCli.nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'trust': 'Show trust mode for the interface',
   }

   handler = 'QosCliBasicQosHandler.showInterfacesTrust'
   cliModel = IntfTrustModeCollectionModel


BasicCli.addShowCommandClass( QosInterfacesTrustCmd )

class QosTrustModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( QosTrustModelet )

# --------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos trust MODE
# --------------------------------------------------------------------------------
class QosTrustCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos trust MODE'
   noOrDefaultSyntax = '[ mls ] qos trust [ MODE ]'
   data = {
      'mls': nodeMls,
      'qos': QosCli.nodeQosForConfig,
      'trust': nodeTrust,
      'MODE': CliMatcher.EnumMatcher( {
        'cos': 'Set trust mode to CoS',
        'dscp': 'Set trust mode to DSCP',
      } )
   }

   handler = 'QosCliBasicQosHandler.setQosTrustMode'
   noOrDefaultHandler = handler


QosTrustModelet.addCommandClass( QosTrustCmd )
QosProfileMode.addCommandClass( QosTrustCmd )

# --------------------------------------------------------------------------------
# [ no | default ] qos map cos to traffic-class name PROFILENAME
# --------------------------------------------------------------------------------
class QosMapCosToTrafficClassProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map cos to traffic-class name PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'cos': matcherMapCos,
      'to': nodeCosToTcProfileTo,
      'traffic-class': matcherTrafficClass,
      'name': 'Name of the Cos-to-Tc Map',
      'PROFILENAME': matcherCosToTcProfileName,
   }

   handler = 'QosCliBasicQosHandler.configureCosToTcProfile'
   noOrDefaultHandler = 'QosCliBasicQosHandler.noCosToTcProfile'


GlobalConfigMode.addCommandClass( QosMapCosToTrafficClassProfilenameCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos map ( ( cos ( { COS_VALUE } | COS_VALUES ) ) |
#                   ( exp ( { EXP_VALUE } | EXP_VALUES ) ) )
#                        to traffic-class TC_VALUE
# ( no | default ) [ mls ] qos map (
#                   ( cos ( { COS_VALUE } | COS_VALUES ) ) |
#                   ( exp ( { EXP_VALUE } | EXP_VALUES ) ) ) ...
# --------------------------------------------------------------------------------
class QosMapToTrafficClassToCosOrExpCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map ( '
              '( cos ( { COS_VALUE } | COS_VALUES ) ) | '
              '( exp ( { EXP_VALUE } | EXP_VALUES ) ) ) '
              'to traffic-class TC_VALUE' )
   noOrDefaultSyntax = ( '[ mls ] qos map ( '
                         '( cos ( { COS_VALUE } | COS_VALUES ) ) | '
                         '( exp ( { EXP_VALUE } | EXP_VALUES ) ) ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'cos': matcherMapCos,
      'exp': nodeMapExp,
      'COS_VALUE': matcherCosValue,
      'COS_VALUES': matcherMultipleCosvalues,
      'EXP_VALUE': matcherExpValue,
      'EXP_VALUES': matcherMultipleExpValue,
      'to': 'QoS parameter to map CoS to',
      'traffic-class': matcherTrafficClass,
      'TC_VALUE': matcherTrafficClassValue,
   }

   handler = 'QosCliBasicQosHandler.qosMapToTrafficClassToCosOrExpCmdHandler'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapToTrafficClassToCosOrExpCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC to
#                 ( ( cos COS) | ( dscp DSCP ) | ( exp EXP ) )
# ( no | default ) [ mls ] qos map traffic-class
#                 ( { TC_LIST } | TC_VALUE ) to ( cos | dscp | exp ) ...
# --------------------------------------------------------------------------------
class QosMapTrafficClassToCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC to '
              '( ( cos COS ) | ( dscp DSCP ) | ( exp EXP ) )' )
   noOrDefaultSyntax = ( '[mls] qos map traffic-class TC '
                         'to ( cos | dscp | exp ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': matcherMapTrafficClass,
      'TC': TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'cos': nodeToCos,
      'COS': matcherCosValue,
      'dscp': nodeToDscp,
      'DSCP': matcherDscpValue,
      'exp': nodeToExp,
      'EXP': matcherExpValue,
   }

   handler = 'QosCliBasicQosHandler.qosMapTrafficClassToCmdHandler'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapTrafficClassToCmd )

# --------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos rewrite cos
# --------------------------------------------------------------------------------
class QosRewriteCosCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos rewrite cos'
   noOrDefaultSyntax = syntax
   data = {
      'mls': matcherMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'rewrite': matcherRewrite,
      'cos': nodeRewriteCos,
   }

   handler = 'QosCliBasicQosHandler.setCosRewrite'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosRewriteCosCmd )

# --------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos cos COS
# --------------------------------------------------------------------------------
class QosCosCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos cos COS'
   noOrDefaultSyntax = '[ mls ] qos cos ...'
   data = {
      'mls': nodeMls,
      'qos': QosCli.nodeQosForConfig,
      'cos': nodeCos,
      'COS': CliMatcher.IntegerMatcher( tacCos.min, tacCos.max,
                helpdesc='Class of Service (CoS) value' ),
   }

   handler = 'QosCliBasicQosHandler.setQosCosDefault'
   noOrDefaultHandler = handler


QosTrustModelet.addCommandClass( QosCosCmd )
QosProfileMode.addCommandClass( QosCosCmd )

# ------------------------------------------------------------------------
# show qos map cos to traffic-class <name>
# ------------------------------------------------------------------------
class ShowCosToTcProfile( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos map cos to traffic-class [ NAME ]"

   data = {
      'qos': QosCli.nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'cos': CliCommand.guardedKeyword( 'cos',
         helpdesc='Show CoS To Traffic-Class map configuration',
         guard=guardCosToTcSupported ),
      'to': 'Map to Traffic-Class',
      'traffic-class': 'Map to Traffic-Class',
      'NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: qosInputConfig.cosToTcProfile, 'Map Name',
         pattern=r'(?!global-profile$)[A-Za-z0-9_:{}\[\]-]+' )
   }
   handler = 'QosCliBasicQosHandler.showCosToTcProfile'
   cliModel = CosToTcProfileAllModel


BasicCli.addShowCommandClass( ShowCosToTcProfile )

# -----------------------------------------------------------------------------------
# show qos map traffic-class to cos <name>
# -----------------------------------------------------------------------------------
class ShowTcToCosProfile( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos map traffic-class to cos [ NAME ]"

   data = {
      'qos': QosCli.nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class',
         helpdesc='Show Traffic-Class Map Configuration',
         guard=guardAnyTcToCosPerInterface ),
      'to': 'Map to Traffic-Class',
      'cos': 'cos',
      'NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: qosInputConfig.tcToCosNamedMap, 'Map Name',
         pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )
   }
   handler = 'QosCliBasicQosHandler.showTcToCosProfile'
   cliModel = TcToCosMapAllModel


BasicCli.addShowCommandClass( ShowTcToCosProfile )

# -----------------------------------------------------------------------------------
# "traffic-class <tc> to cos <cos> in TcToCos mode
# -----------------------------------------------------------------------------------
class TcToCosMapCmd( CliCommand.CliCommandClass ):
   syntax = "traffic-class TC [drop-precedence DP] to cos COS "
   noOrDefaultSyntax = "traffic-class TC [drop-precedence DP] to cos ..."
   data = {
      'traffic-class': tokenTrafficClassHelpdesc,
      'TC': CliMatcher.IntegerMatcher( 0, 7, helpdesc='Value of traffic-class' ),
      'drop-precedence': 'Map drop-precedence values',
      'DP': CliMatcher.IntegerMatcher( 0, 2, helpdesc='Value of drop-precedence' ),
      'to': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'to', helpdesc=MapToCosHelpdesc ), ),
      'cos': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'cos', helpdesc=MapToCosHelpdesc ) ),
      'COS': CliMatcher.IntegerMatcher( 0, 7, helpdesc='Value of CoS' ),
   }
   handler = 'QosCliBasicQosHandler.setTcToCosMap'
   noOrDefaultHandler = 'QosCliBasicQosHandler.delTcToCosMap'


TcToCosMode.addCommandClass( TcToCosMapCmd )

# --------------------------------------------------------------------------------
# cos ( COS | { COSVALUES } ) to traffic-class TC_VALUE
# ( no | default ) cos ( COS | { COSVALUES } ) to traffic-class ...
# --------------------------------------------------------------------------------
class CosToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'cos ( COS | { COSVALUES } ) to traffic-class TC_VALUE'
   noOrDefaultSyntax = 'cos ( COS | { COSVALUES } ) to traffic-class ...'
   data = {
      'cos': matcherMapCos,
      'COSVALUES': matcherCosValue,
      'COS': MultiRangeRule.MultiRangeMatcher( lambda: ( tacCos.min, tacCos.max ),
         False, 'Class of Service (COS) value(s) or range(s) of COS values' ),
      'to': 'QoS parameter to map CoS to',
      'traffic-class': matcherTrafficClass,
      'TC_VALUE': matcherTrafficClassValue,
   }

   handler = 'QosCliBasicQosHandler.setCosToTcProfileMap'
   noOrDefaultHandler = 'QosCliBasicQosHandler.delCosToTcProfileMap'


CosToTcProfileMode.addCommandClass( CosToTrafficClassCmd )

# --------------------------------------------------------------------------------
# exp ( EXP | { EXPVALUES } ) to traffic-class TC_VALUE
# ( no | default ) exp ( EXP | { EXPVALUES } ) to traffic-class ...
# --------------------------------------------------------------------------------
class ExpToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'exp ( EXP | { EXPVALUES } ) to traffic-class TC_VALUE'
   noOrDefaultSyntax = 'exp ( EXP | { EXPVALUES } ) to traffic-class ...'
   data = {
      'exp': matcherMapExp,
      'EXPVALUES': matcherExpValue,
      'EXP': MultiRangeRule.MultiRangeMatcher( lambda: ( tacExp.min, tacExp.max ),
         False, 'EXP value(s) or range(s) of EXP values' ),
      'to': 'QoS parameter to map EXP to',
      'traffic-class': matcherTrafficClass,
      'TC_VALUE': matcherTrafficClassValue,
   }

   handler = "QosCliBasicQosHandler.setExpToTcProfileMap"
   noOrDefaultHandler = "QosCliBasicQosHandler.delExpToTcProfileMap"


ExpToTcProfileMode.addCommandClass( ExpToTrafficClassCmd )

# --------------------------------------------------------------------------------
# [ no | default ] qos map exp to traffic-class name PROFILENAME
# --------------------------------------------------------------------------------
class QosMapExpToTrafficClassProfileNameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map exp to traffic-class name PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'exp': matcherMapExp,
      'to': nodeExpToTcProfileTo,
      'traffic-class': matcherTrafficClass,
      'name': 'Name of the EXP-to-Tc Map',
      'PROFILENAME': matcherExpToTcProfileName,
   }

   handler = 'QosCliBasicQosHandler.configureExpToTcProfile'
   noOrDefaultHandler = 'QosCliBasicQosHandler.noExpToTcProfile'


GlobalConfigMode.addCommandClass( QosMapExpToTrafficClassProfileNameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] qos map dscp to traffic-class name MAP_NAME
# --------------------------------------------------------------------------------
class QosMapDscpToTrafficClassMapNameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map dscp to traffic-class name MAP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'dscp': matcherMapDscp,
      'to': nodeDscpToTcMapTo,
      'traffic-class': matcherTrafficClass,
      'name': 'Name of the Map',
      'MAP_NAME': matcherDscpToTcMapName,
   }

   handler = 'QosCliBasicQosHandler.createDscpToTcMap'
   noOrDefaultHandler = 'QosCliBasicQosHandler.removeDscpToTcMap'


GlobalConfigMode.addCommandClass( QosMapDscpToTrafficClassMapNameCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos map dei ( { DEI_VALUE } | DEI_VALUES )
#              to drop-precedence DROP_PRECEDENCE
# ( no | default ) [ mls ] qos map dei
#              ( { DEI_VALUE } | DEI_VALUES ) ...
# --------------------------------------------------------------------------------
class QosMapDeiToDropPrecedenceCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map dei ( { DEI_VALUE } | DEI_VALUES ) to '
              'drop-precedence DROP_PRECEDENCE ' )
   noOrDefaultSyntax = ( '[ mls ] qos map dei '
                         '( { DEI_VALUE } | DEI_VALUES ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'dei': nodeDei,
      'DEI_VALUE': matcherDeiValue,
      'DEI_VALUES': matcherMultipleDeiValues,
      'to': 'QoS parameter to map DEI to',
      'drop-precedence': matcherDropPrecedence,
      'DROP_PRECEDENCE': matcherDropPrecedenceValue,
   }

   handler = 'QosCliBasicQosHandler.setDeiToDp'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapDeiToDropPrecedenceCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC to
#                 ( ( tx-queue TXQ_VALUE ) | ( uc-tx-queue UCTXQ_VALUE ) )
# ( no | default ) [ mls ] qos map traffic-class TC
#                  to ( tx-queue | uc-tx-queue ) ...
# --------------------------------------------------------------------------------
class QosMapTrafficClassToTxQueueCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC to '
              '( ( tx-queue TXQ_VALUE ) | ( uc-tx-queue UCTXQ_VALUE ) )' )
   noOrDefaultSyntax = ( '[ mls ] qos map traffic-class TC '
                         'to ( tx-queue | uc-tx-queue ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': matcherTrafficClass,
      'TC': TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'tx-queue': nodeToTxQueue,
      'TXQ_VALUE': CliMatcher.DynamicIntegerMatcher(
         txQueueRangeFn, helpdesc='Transmit queue id' ),
      'uc-tx-queue': nodeToUcTxQueue,
      'UCTXQ_VALUE': CliMatcher.DynamicIntegerMatcher(
         ucTxQueueRangeFn,
         helpdesc='Unicast transmit queue ID' ),
   }

   handler = 'QosCliBasicQosHandler.setTrafficClassToTxQueueMap'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapTrafficClassToTxQueueCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC
#         to mc-tx-queue MCTXQ_VALUE
# ( no | default ) [ mls ] qos map traffic-class TC
#                          to mc-tx-queue ...
# --------------------------------------------------------------------------------
class QosMapTrafficClassToMcTxQueueCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC '
              'to mc-tx-queue MCTXQ_VALUE' )
   noOrDefaultSyntax = ( '[ mls ] qos map traffic-class TC '
                         'to mc-tx-queue ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': matcherTrafficClass,
      'TC': TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'mc-tx-queue': nodeMcTxQueue,
      'MCTXQ_VALUE': CliMatcher.DynamicIntegerMatcher(
         mcTxQueueRangeFn,
         helpdesc='Multicast transmit queue ID' ),
   }

   handler = 'QosCliBasicQosHandler.setTrafficClassToMcTxQueueMap'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapTrafficClassToMcTxQueueCmd )

def priorityGroupVal( mode=None ):
   pgsSupported = {}
   pgList = []
   if qosHwStatus.hwInitialized:
      for pg in list( qosHwStatus.supportedPgs ):
         pgList.append( pg )
   else:
      for pg in list( range( 0, qosHwStatus.controlPg ) ) + \
          [ qosHwStatus.controlPg + 1 ]:
         pgList.append( pg )
   for pg in pgList:
      pgsSupported.update( { str( pg ): f'Priority Group id {str( pg )}' } )
   return pgsSupported

# --------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC to
#                  priority-group PRIORITYGROUPVALUE
# ( no | default ) [ mls ] qos map traffic-class TC to
#                  priority-group ...
# --------------------------------------------------------------------------------
class QosMapTrafficClassToPriorityGroupCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC to '
              'priority-group PRIORITYGROUPVALUE' )
   noOrDefaultSyntax = ( '[ mls ] qos map traffic-class TC '
                         'to priority-group ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': matcherTrafficClass,
      'TC': TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'priority-group': nodePriorityGroup,
      'PRIORITYGROUPVALUE': CliMatcher.DynamicKeywordMatcher( priorityGroupVal ),
   }

   handler = 'QosCliBasicQosHandler.setTrafficClassToPriorityGroupMap'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapTrafficClassToPriorityGroupCmd )

# --------------------------------------------------------------------------------
# qos traffic-class TC_VALUE name TC_NAME
# ( no | default ) qos traffic-class TC_VALUE name ...
# --------------------------------------------------------------------------------
class QosTrafficClassNameCmd( CliCommand.CliCommandClass ):
   syntax = ( 'qos traffic-class TC_VALUE name TC_NAME ' )
   noOrDefaultSyntax = ( 'qos traffic-class TC_VALUE name ...' )

   data = {
      'qos': QosCli.nodeQosForConfig,
      'traffic-class': matcherTrafficClassConfig,
      'TC_VALUE': matcherTrafficClassValue,
      'name': 'Name of the traffic-class',
      'TC_NAME': matcherTrafficClassName,
   }

   handler = 'QosCliBasicQosHandler.handleTrafficClassName'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosTrafficClassNameCmd )

class VlanMatchExpression( CliCommand.CliExpression ):
   expression = "vlan ( VLAN_SET | ( VLAN_VALUE [ VLAN_MASK ] ) )"
   data = {
      "vlan": CliCommand.Node( matcher=vlanKeywordMatcher, maxMatches=1,
                                 guard=guardVlanMatch ),
      "VLAN_SET": CliCommand.Node( matcher=vlanSetMatcher,
                                   maxMatches=1 ),
      "VLAN_VALUE": CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 4094, helpdesc="VLAN value" ),
         maxMatches=1 ),
      "VLAN_MASK": CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0x0, 0xfff, helpdesc="VLAN mask" ),
         maxMatches=1 )
   }

class InnerVlanMatchExpression( CliCommand.CliExpression ):
   expression = ( "vlan_ inner ( INNER_VLAN_SET | ( INNER_VLAN_VALUE "
                  "[ INNER_VLAN_MASK ] ) )" )
   data = {
      "vlan_": CliCommand.Node( matcher=vlanInnerKeywordMatcher, maxMatches=1,
                                 guard=guardInnerVlanMatch ),
      "inner": CliCommand.Node( matcher=innerVlanKeywordMatcher, maxMatches=1,
                                 guard=guardInnerVlanMatch ),
      "INNER_VLAN_SET": CliCommand.Node( matcher=innerVlanSetMatcher,
                                          maxMatches=1 ),
      "INNER_VLAN_VALUE": CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 4094, helpdesc="Inner VLAN value" ),
         maxMatches=1 ),
      "INNER_VLAN_MASK": CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0x0, 0xfff, helpdesc="Inner VLAN mask" ),
         maxMatches=1 )
   }

# ------------------------------------------------------------------------
# set cos COS
# ( no | default ) set cos ...
# ------------------------------------------------------------------------
class PolicyMapClassModeSetCosCmd( CliCommand.CliCommandClass ):
   syntax = 'set cos COS'
   noOrDefaultSyntax = 'set cos ...'
   data = {
      'set': matcherSet,
      'cos': nodeActionCos,
      'COS': CliMatcher.IntegerMatcher( tacCos.min, tacCos.max,
         helpdesc="Class of Service (CoS) value" ),
   }

   handler = PolicyMapClassModeQos.configureSetCos
   noOrDefaultHandler = handler


PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeSetCosCmd )

# --------------------------------------------------------------------------------
# [ mls ] qos map dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class
#               TC_VALUE [ drop-precedence DROP_PRECEDENCE ]
# ( no | default ) [ mls ] qos map dscp
#               ( { DSCP_VALUE } | DSCP_VALUES ) ...
# --------------------------------------------------------------------------------
class QosMapDscpToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map dscp ( { DSCP_VALUE } | DSCP_VALUES ) to '
              'traffic-class TC_VALUE [ drop-precedence DROP_PRECEDENCE ] ' )
   noOrDefaultSyntax = ( '[ mls ] qos map dscp '
                         '( { DSCP_VALUE } | DSCP_VALUES ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'dscp': matcherMapDscp,
      'DSCP_VALUE': matcherDscpValue,
      'DSCP_VALUES': matcherMultipleDscpValues,
      'to': 'QoS parameter to map DSCP to',
      'traffic-class': matcherTrafficClass,
      'TC_VALUE': matcherTrafficClassValue,
      'drop-precedence': nodeDropPrecedence,
      'DROP_PRECEDENCE': dropPrecedenceValueMatcher,
   }

   handler = 'QosCliBasicQosHandler.setDscpToTcDp'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapDscpToTrafficClassCmd )

# --------------------------------------------------------------------------------
# [ no | default ] qos map traffic-class to dscp MAPNAME
# --------------------------------------------------------------------------------
class QosMapTrafficClassToDscpMapnameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map traffic-class to dscp MAPNAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': matcherMapTrafficClass,
      'to': nodeTo,
      'dscp': nodeToDscp,
      'MAPNAME': matcherDscpRewriteMapName,
   }

   handler = 'QosCliBasicQosHandler.configureDscpRewriteMap'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosMapTrafficClassToDscpMapnameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos rewrite dscp
# --------------------------------------------------------------------------------
class QosRewriteDscpCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos rewrite dscp'
   noOrDefaultSyntax = syntax
   data = {
      'mls': matcherMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'rewrite': matcherRewrite,
      'dscp': nodeRewriteDscp,
   }

   handler = 'QosCliBasicQosHandler.setDscpRewrite'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosRewriteDscpCmd )

# --------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos dscp DSCP
# --------------------------------------------------------------------------------
class QosDscpCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos dscp DSCP'
   noOrDefaultSyntax = '[ mls ] qos dscp ...'
   data = {
      'mls': nodeMls,
      'qos': QosCli.nodeQosForConfig,
      'dscp': nodeDscp,
      'DSCP': CliMatcher.IntegerMatcher( tacDscp.min, tacDscp.max,
         helpdesc='DSCP value' ),
   }

   handler = 'QosCliBasicQosHandler.setQosDscpDefault'
   noOrDefaultHandler = handler


QosTrustModelet.addCommandClass( QosDscpCmd )
QosProfileMode.addCommandClass( QosDscpCmd )

# -----------------------------------------------------------------------------------
# show qos map dscp to traffic-class <name>
# -----------------------------------------------------------------------------------
class ShowDscpToTcProfile( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos map dscp to traffic-class [ NAME ]"

   data = {
      'qos': QosCli.nodeQosForShow,
      'map': "Mapping Configuration of different QoS parameters",
      'dscp': CliCommand.guardedKeyword( 'dscp',
         helpdesc='Configure a DSCP to Tc Map',
         guard=guardDscpToTcSupported ),
      'to': 'Map to Traffic-Class',
      'traffic-class': 'Map to Traffic-Class',
      'NAME': matcherDscpToTcMapName
      }
   handler = 'QosCliBasicQosHandler.showDscpToTcNamedMap'
   cliModel = DscpToTcMapAllModel


BasicCli.addShowCommandClass( ShowDscpToTcProfile )

# ------------------------------------------------------------------------
# show qos map exp to traffic-class <name>
# ------------------------------------------------------------------------
class ShowExpToTcProfile( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos map exp to traffic-class [ NAME ]"

   data = {
      'qos': QosCli.nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'exp': CliCommand.guardedKeyword( 'exp',
         helpdesc='Show EXP To Traffic-Class map configuration',
         guard=QosCli.guardExpToTcSupported ),
      'to': 'Map to Traffic-Class',
      'traffic-class': 'Map to Traffic-Class',
      'NAME': CliMatcher.DynamicNameMatcher(
         lambda mode: qosInputConfig.expToTcProfile, 'Map Name',
         pattern=r'(?!global-profile$)[A-Za-z0-9_:{}\[\]-]+' )
   }
   handler = "QosCliBasicQosHandler.showExpToTcProfile"
   cliModel = ExpToTcProfileAllModel


BasicCli.addShowCommandClass( ShowExpToTcProfile )

# -----------------------------------------------------------------------------------
# "qos map traffic-class to cos name <mapname>" to enter into
# TcToCosMapMode
# -----------------------------------------------------------------------------------
class TcToCosMapNameCmd( CliCommand.CliCommandClass ):
   syntax = "qos map traffic-class to cos name MAP_NAME"
   noOrDefaultSyntax = "qos map traffic-class to cos name MAP_NAME"
   data = {
      'qos': nodeQosForConfig,
      'map': matcherMap,
      'traffic-class': tokenTrafficClassHelpdesc,
      'to': tokenTcToCosMapTo,
      'cos': CliCommand.guardedKeyword( 'cos',
         helpdesc=MapToCosHelpdesc, guard=guardAnyTcToCosPerInterface ),
      'name': 'Name of the map',
      'MAP_NAME': tcToCosMapNameRule
   }
   handler = "QosCliBasicQosHandler.createTcToCosMap"
   noOrDefaultHandler = "QosCliBasicQosHandler.noOrDefaultCreateTcToCosMap"


BasicCli.GlobalConfigMode.addCommandClass( TcToCosMapNameCmd )

#------------------------------------------------------------------------------------
# "qos map traffic-class to exp name <mapname>" to enter into
# TcToExpMapMode
#------------------------------------------------------------------------------------
class TcToExpMapNameCmd( CliCommand.CliCommandClass ):
   syntax = "qos map traffic-class to exp name MAP_NAME"
   noOrDefaultSyntax = syntax
   data = {
      'qos': nodeQosForConfig,
      'map': matcherMap,
      'traffic-class': matcherMapTrafficClass,
      'to': CliCommand.guardedKeyword( 'to', helpdesc='Configure a named Map',\
         guard=guardNamedTcDpToExpQosMapSupported ),
      'exp': CliCommand.guardedKeyword( 'exp',
         helpdesc='Set Packet EXP value', guard=guardNamedTcDpToExpQosMapSupported ),
      'name': 'Name of the map',
      'MAP_NAME': matcherTcToExpNamedMapName
   }
   handler = "QosCliBasicQosHandler.createTcToExpNamedMap"
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( TcToExpMapNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-class TC [ drop-precedence DP ]  to exp EXP
#--------------------------------------------------------------------------------
class TrafficClassToExpCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-class TC [ drop-precedence DP ] to exp EXP'
   noOrDefaultSyntax = 'traffic-class TC [ drop-precedence DP ] to exp [ EXP ]'
   data = {
      'traffic-class': 'Map traffic-class values',
      'TC': TrafficClassExpression,
      'drop-precedence': CliCommand.guardedKeyword( 'drop-precedence', helpdesc=\
         'Map Drop-Precedence values', guard=guardNamedTcDpToExpQosMapSupported ),
      'DP': CliMatcher.IntegerMatcher( 0, 2, helpdesc='Value of drop-precedence' ),
      'to': 'QoS parameter to map traffic-class to',
      'exp': CliCommand.guardedKeyword( 'exp',
         helpdesc='Map EXP values', guard=guardNamedTcDpToExpQosMapSupported ),
      'EXP': matcherExpValue,
   }

   handler = 'QosCliBasicQosHandler.setTcToExpMap'
   noOrDefaultHandler = 'QosCliBasicQosHandler.delTcToExpMap'

TcToExpMode.addCommandClass( TrafficClassToExpCmd )

class TcDpToExpModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )

IntfCli.IntfConfigMode.addModelet( TcDpToExpModelet )

#--------------------------------------------------------------------------------
# qos rewrite traffic-class to exp MAP
# ( no | default ) qos rewrite traffic-class to exp [ MAP ]
#--------------------------------------------------------------------------------
class QosTrafficClassToExpCmd( CliCommand.CliCommandClass ):
   syntax = 'qos rewrite traffic-class to exp MAP'
   noOrDefaultSyntax = 'qos rewrite traffic-class to exp [ MAP ]'
   data = {
      'qos': nodeQosForConfig,
      'rewrite': matcherRewrite,
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class',
                                       helpdesc='Configure a Rewrite Map',
                                       guard=guardNamedTcDpToExpQosMapSupported ),
      'to': 'QoS parameter to map traffic-class to',
      'exp': CliCommand.guardedKeyword( 'exp',
         helpdesc='Map EXP values', guard=guardNamedTcDpToExpQosMapSupported ),
      'MAP': matcherTcToExpNamedMapName,
   }

   handler = "QosCliBasicQosHandler.applyTcDpToExpMap"
   noOrDefaultHandler = "QosCliBasicQosHandler.deleteTcDpToExpMap"

TcDpToExpModelet.addCommandClass( QosTrafficClassToExpCmd )

#--------------------------------------------------------------------------------
# show qos map traffic-class to exp [ MAP ]
#--------------------------------------------------------------------------------
class QosMapTrafficClassToExpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos map traffic-class to exp [ MAP ]'
   data = {
      'qos': QosCli.nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class', 
         "Show Traffic-Class Map Configuration", 
         guard=guardNamedTcDpToExpQosMapSupported ),
      'to': 'Configure a named map', 
      'exp': CliCommand.guardedKeyword( 'exp',
         helpdesc='Map EXP values', guard=guardNamedTcDpToExpQosMapSupported ),
      'MAP': CliMatcher.DynamicNameMatcher( 
         lambda mode : qosInputConfig.tcDpToExpNamedMap, 'Map name',
         pattern='(?!default-map$)[A-Za-z0-9_:{}\\[\\]-]+' ),
   }

   handler = 'QosCliBasicQosHandler.showTcDpToExpNamedMaps'
   cliModel = TcDpToExpMapAllModel

BasicCli.addShowCommandClass( QosMapTrafficClassToExpCmd )

# --------------------------------------------------------------------------------
# [ no | default ] traffic-class TC [ drop-precedence DP ]  to dscp DSCP
# --------------------------------------------------------------------------------
class TrafficClassToDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-class TC [ drop-precedence DP ] to dscp DSCP'
   noOrDefaultSyntax = 'traffic-class TC [ drop-precedence DP ] to dscp [ DSCP ]'
   data = {
      'traffic-class': 'Map traffic-class values',
      'TC': TrafficClassExpression,
      'drop-precedence': CliCommand.guardedKeyword(
         'drop-precedence',
         helpdesc='Map Drop-Precedence values',
         guard=guardDscpRewriteDropPrecedence ),
      'DP': CliMatcher.IntegerMatcher( 0, 2, helpdesc='Value of drop-precedence' ),
      'to': 'QoS parameter to map traffic-class to',
      'dscp': nodeToDscp,
      'DSCP': matcherDscpValue,
   }

   handler = 'QosCliBasicQosHandler.setTcToDscpMap'
   noOrDefaultHandler = 'QosCliBasicQosHandler.delTcToDscpMap'


DscpRewriteMode.addCommandClass( TrafficClassToDscpCmd )

# -----------------------------------------------------------------------------------
# "[ no | default ] dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class TC_VALUE
# -----------------------------------------------------------------------------------
class DscpToTcMapCmd( CliCommand.CliCommandClass ):
   syntax = "dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class TC_VALUE"
   noOrDefaultSyntax = "dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class ..."
   data = {
      'dscp': matcherMapDscp,
      'DSCP_VALUE': matcherDscpValue,
      'DSCP_VALUES': matcherMultipleDscpValues,
      'to': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'to', helpdesc='Map to Traffic-Class' ), ),
      'traffic-class': matcherTrafficClass,
      'TC_VALUE': matcherTrafficClassValue
   }
   handler = 'QosCliBasicQosHandler.setDscpToTcMap'
   noOrDefaultHandler = 'QosCliBasicQosHandler.deleteDscpToTcMap'


DscpToTcMode.addCommandClass( DscpToTcMapCmd )

class DscpPreserveModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagOrVlanIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( DscpPreserveModelet )

# --------------------------------------------------------------------------------
# ( no | default ) qos rewrite dscp mpls encapsulation disabled
# --------------------------------------------------------------------------------
class QosDscpPreserveCmd( CliCommand.CliCommandClass ):
   syntax = 'qos rewrite dscp mpls encapsulation disabled'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'rewrite': matcherRewrite,
      'dscp': nodeRewriteDscp,
      'mpls': CliCommand.guardedKeyword( 'mpls',
         helpdesc='DSCP rewrite setting for MPLS packets',
         guard=guardDscpPreserveMplsEncap ),
      'encapsulation': 'Configure MPLS encapsulation setting',
      'disabled': 'Disable DSCP rewrite',
   }

   handler = 'QosCliBasicQosHandler.applyDscpPreserveMplsEncap'
   noOrDefaultHandler = 'QosCliBasicQosHandler.deleteDscpPreserveMplsEncap'


DscpPreserveModelet.addCommandClass( QosDscpPreserveCmd )
QosProfileMode.addCommandClass( QosDscpPreserveCmd )

# ------------------------------------------------------------------------
# set dscp ( DSCP_NAME | DSCP )
# ( no | default ) set dscp ...
# ------------------------------------------------------------------------
class PolicyMapClassModeSetDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'set dscp ( DSCP_NAME | DSCP )'
   noOrDefaultSyntax = 'set dscp ...'
   data = {
      'set': matcherSet,
      'dscp': nodeActionDscp,
      'DSCP': CliMatcher.IntegerMatcher( 0, AclLib.MAX_DSCP,
         helpdesc="DSCP Value" ),
      'DSCP_NAME': CliMatcher.EnumMatcher(
         { k: v[ 1 ] for k, v in AclCliLib.dscpAclNames.items() } ),
   }

   handler = PolicyMapClassModeQos.configureSetDscp
   noOrDefaultHandler = handler


PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeSetDscpCmd )

# --------------------------------------------------------------------------------
# show qos map traffic-class to dscp [ MAP ]
# --------------------------------------------------------------------------------
class QosMapTrafficClassToDscpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos map traffic-class to dscp [ MAP ]'
   data = {
      'qos': QosCli.nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class',
         "Show Traffic-Class Map Configuration",
         guard=guardEgressDscpRewrite ),
      'to': 'Configure a named map',
      'dscp': CliCommand.guardedKeyword( 'dscp',
         "Set Differentiated Services Code Point (DSCP) value",
         guard=guardTcToDscpMap ),
      'MAP': CliMatcher.DynamicNameMatcher(
         lambda mode: qosInputConfig.dscpRewriteMap, 'Map name',
         pattern='(?!default-map$)[A-Za-z0-9_:{}\\[\\]-]+' ),
   }

   handler = 'QosCliBasicQosHandler.showDscpRewriteMaps'
   cliModel = DscpRewriteMapAllModel


BasicCli.addShowCommandClass( QosMapTrafficClassToDscpCmd )

# --------------------------------------------------------------------------------
# show qos interfaces [ INTFS ] dscp mpls encapsulation disabled
# --------------------------------------------------------------------------------
class QosInterfacesDscpPreserveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show qos interfaces [ INTFS ] dscp mpls encapsulation disabled' )
   data = {
      'qos': QosCli.nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTFS': matcherDscpPreserveIntf,
      'dscp': CliCommand.guardedKeyword( 'dscp',
         helpdesc='Show DSCP Preserve Config for the interface',
         guard=guardDscpPreserveMplsEncap ),
      'mpls': 'DSCP rewrite setting for MPLS packets',
      'encapsulation': 'encapsulation',
      'disabled': 'disabled',
   }

   handler = 'QosCliBasicQosHandler.showInterfacesDscpPreserve'
   cliModel = IntfDscpPreserveCollectionModel


BasicCli.addShowCommandClass( QosInterfacesDscpPreserveCmd )

class DscpRewriteModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):

      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( DscpRewriteModelet )

# --------------------------------------------------------------------------------
# qos rewrite traffic-class to dscp MAP
# ( no | default ) qos rewrite traffic-class to dscp [ MAP ]
# --------------------------------------------------------------------------------
class QosTrafficClassToDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'qos rewrite traffic-class to dscp MAP'
   noOrDefaultSyntax = 'qos rewrite traffic-class to dscp [ MAP ]'
   data = {
      'qos': nodeQosForConfig,
      'rewrite': matcherRewrite,
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class',
         helpdesc='Configure a Rewrite Map', guard=guardDscpRewritePerInterface ),
      'to': 'QoS parameter to map traffic-class to',
      'dscp': nodeToDscp,
      'MAP': matcherDscpRewriteMapName,
   }

   handler = "QosCliBasicQosHandler.applyRewriteMap"
   noOrDefaultHandler = "QosCliBasicQosHandler.deleteRewriteMap"


DscpRewriteModelet.addCommandClass( QosTrafficClassToDscpCmd )

class CosToTcProfileModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( CosToTcProfileModelet )

# --------------------------------------------------------------------------------
# qos map cos to traffic-class MAP
# ( no | default ) qos map cos to traffic-class [ MAP ]
# --------------------------------------------------------------------------------
class QosCosToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map cos to traffic-class MAP'
   noOrDefaultSyntax = 'qos map cos to traffic-class [ MAP ]'
   data = {
      'qos': nodeQosForConfig,
      'map': matcherMap,
      'cos': CliCommand.guardedKeyword( 'cos',
         helpdesc='Configure a Cos To Tc Map', guard=guardCosToTcPerInterface ),
      'to': 'QoS parameter to map CoS to',
      'traffic-class': matcherTrafficClass,
      'MAP': matcherCosToTcProfileName,
   }

   handler = "QosCliBasicQosHandler.applyCosToTcProfile"
   noOrDefaultHandler = "QosCliBasicQosHandler.removeCosToTcProfile"


CosToTcProfileModelet.addCommandClass( QosCosToTrafficClassCmd )

class ExpToTcProfileModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( ExpToTcProfileModelet )

# --------------------------------------------------------------------------------
# qos map exp to traffic-class MAP
# ( no | default ) qos map exp to traffic-class [ MAP ]
# --------------------------------------------------------------------------------
class QosExpToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map exp to traffic-class MAP'
   noOrDefaultSyntax = 'qos map exp to traffic-class [ MAP ]'
   data = {
      'qos': nodeQosForConfig,
      'map': matcherMap,
      'exp': CliCommand.guardedKeyword( 'exp',
         helpdesc='Configure a EXP To Tc Map', guard=guardExpToTcPerInterface ),
      'to': 'QoS parameter to map EXP to',
      'traffic-class': matcherTrafficClass,
      'MAP': matcherExpToTcProfileName,
   }

   handler = "QosCliBasicQosHandler.applyExpToTcProfile"
   noOrDefaultHandler = "QosCliBasicQosHandler.removeExpToTcProfile"


ExpToTcProfileModelet.addCommandClass( QosExpToTrafficClassCmd )

class DscpToTcModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( DscpToTcModelet )

# --------------------------------------------------------------------------------
# qos map dscp to traffic-class MAP_NAME
# ( no | default ) qos map dscp to traffic-class [ MAP_NAME ]
# --------------------------------------------------------------------------------
class QosDscpToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map dscp to traffic-class MAP_NAME'
   noOrDefaultSyntax = 'qos map dscp to traffic-class [ MAP_NAME ]'
   data = {
      'qos': nodeQosForConfig,
      'map': matcherMap,
      'dscp': CliCommand.guardedKeyword( 'dscp',
         helpdesc='Configure a DSCP to Tc Map', guard=guardDscpToTcPerInterface ),
      'to': 'QoS parameter to map DSCP to',
      'traffic-class': matcherTrafficClass,
      'MAP_NAME': matcherDscpToTcMapName,
   }

   handler = "QosCliBasicQosHandler.applyDscpToTcMap"
   noOrDefaultHandler = "QosCliBasicQosHandler.removeDscpToTcMapAttachment"


DscpToTcModelet.addCommandClass( QosDscpToTrafficClassCmd )

class TcToCosModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( TcToCosModelet )

class IntfTcToCosMapCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map traffic-class to cos MAP_NAME'
   noOrDefaultSyntax = 'qos map traffic-class to cos'
   data = {
         'qos': nodeQosForConfig,
         'map': matcherMap,
         'traffic-class': tokenTrafficClassHelpdesc,
         'to': tokenTcToCosMapTo,
         'cos': CliCommand.guardedKeyword( 'cos', helpdesc=MapToCosHelpdesc,
            guard=guardTcToCosPerInterface ),
         'MAP_NAME': tcToCosMapNameRule,
   }
   handler = "QosCliBasicQosHandler.intfTcToCosMap"
   noOrDefaultHandler = "QosCliBasicQosHandler.intfRemoveTcToCosMap"


TcToCosModelet.addCommandClass( IntfTcToCosMapCmd )

class CpuTcToCosModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( CpuTcToCosModelet )

class IntfCpuTcToCosMapCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map cpu traffic-class to cos MAP_NAME'
   noOrDefaultSyntax = 'qos map cpu traffic-class to cos'
   data = {
         'qos': nodeQosForConfig,
         'map': matcherMap,
         'cpu': CliCommand.guardedKeyword( 'cpu', helpdesc="CPU generated traffic",
            guard=guardCpuTcToCosPerInterface ),
         'traffic-class': tokenTrafficClassHelpdesc,
         'to': tokenTcToCosMapTo,
         'cos': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
            'cos', helpdesc=MapToCosHelpdesc ) ),
         'MAP_NAME': tcToCosMapNameRule,
   }
   handler = "QosCliBasicQosHandler.intfCpuTcToCosMap"
   noOrDefaultHandler = "QosCliBasicQosHandler.intfCpuRemoveTcToCosMap"


CpuTcToCosModelet.addCommandClass( IntfCpuTcToCosMapCmd )

# --------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# --------------------------------------------------------------------------------
@Plugins.plugin( provides=( "QosCli", ) )
def Plugin( entityManager ):
   global qosInputConfig, qosConfig, \
          qosStatus, qosHwStatus, \
          pfcStatus, qosSliceHwStatus, \
          qosAclHwStatus

   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = \
         "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   pfcStatus = LazyMount.mount( entityManager, "dcb/pfc/status", "Pfc::Status", "r" )
