# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import Plugins
import QosLib
import ShowCommand
import Tac

from CliPlugin import ( QosCliServicePolicy, AclCli, VlanIntfCli )
# pylint: disable=redefined-builtin
from CliPlugin.QosCliIntfTypes import ethOrLagIntfTypes
from CliPlugin.QosCliCommon import ( QosProfileMode, QosModelet )
from CliPlugin.QosCliServicePolicy import ( guardClassMap, getClassNameRule,
                                            setServicePolicy )
from CliPlugin.QosCliModel import ( ClassMapAllModel, PolicyMapAllModel,
                                    ClassMapWrapper, InterfaceIdModel )
from CliToken.Clear import clearKwNode
from Intf.IntfRange import IntfRangeMatcher
from QosLib import ( coppMapType, isLagPort, CliError, classMapCpType )
from QosTypes import ( tacActionType, tacClassMapCpType,
                       tacActionRateType, tacPMapNm )

# -----------------------------------------------------------------------------------
# Variables for Qos Copp associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosHwStatus = None
qosAclHwStatus = None
cliQosAclConfig = None

# -----------------------------------------------------------------------------------
# Guards begin
# -----------------------------------------------------------------------------------
def guardPMapCopp( mode, token ):
   if qosHwStatus.coppSupported or qosHwStatus.intfCoppSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCMapCopp( mode, token ):
   if qosAclHwStatus.coppDynamicClassSupported or qosHwStatus.intfCoppSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCMapDynamicCopp( mode, token ):
   if qosAclHwStatus.coppDynamicClassSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCoppSystemPolicy( mode, token ):
   if not qosHwStatus.intfCoppSupported and \
      qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfCoppSupported( mode, token ):
   if qosHwStatus.intfCoppSupported and \
      qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfModePolicyCopp( mode, token ):
   if qosHwStatus.intfCoppSupported:
      if isinstance( mode, QosProfileMode ):
         return None
      elif mode.intf.isSubIntf():
         if qosAclHwStatus.subIntfPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VlanIntfCli.VlanIntf ):
         if qosAclHwStatus.sviPolicyQosSupported:
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform

def guardServicePolicyCopp( mode, token ):
   if qosHwStatus.coppSupported:
      return None
   return CliParser.guardNotThisPlatform

def getPMapNameRuleCopp( mode ):
   return QosCliServicePolicy.getPMapNameRule( mode, coppMapType )

def getClassNameRuleCopp( mode ):
   return getClassNameRule( mode, coppMapType )

def getCMapNameRuleCopp( mode ):
   return QosCliServicePolicy.getCMapNameRule( mode, coppMapType )
# -----------------------------------------------------------------------------------
# Guards end
# -----------------------------------------------------------------------------------


# -----------------------------------------------------------------------------------
# Matchers begin
# -----------------------------------------------------------------------------------
matcherType = CliMatcher.KeywordMatcher( 'type',
      helpdesc='Specify type' )
matcherClassName = CliMatcher.DynamicNameMatcher( getClassNameRuleCopp,
      "Class Map Name" )
matcherCmapName = CliMatcher.DynamicNameMatcher( getCMapNameRuleCopp,
      "Class Map Name" )
matcherClass = CliMatcher.KeywordMatcher( 'class',
      helpdesc='Policy criteria' )
matcherCoppSystemPolicy = CliMatcher.KeywordMatcher( tacPMapNm.coppName,
      helpdesc='Policy-map name' )
matcherMatchAny = CliMatcher.KeywordMatcher( 'match-any',
      helpdesc='Logical-OR all match statements under this class-map' )
matcherPmapName = CliMatcher.DynamicNameMatcher( getPMapNameRuleCopp,
      'Policy Map Name',
      pattern='(?!counters$|interface$|summary$)[A-Za-z0-9_:{}\\[\\]-]+' )
# -----------------------------------------------------------------------------------
# Matchers end
# -----------------------------------------------------------------------------------
nodePolicyMap = CliCommand.guardedKeyword( 'policy-map',
      "Configure Policy Map", guard=QosCliServicePolicy.guardPolicyMap )
nodePolicyMapShow = CliCommand.guardedKeyword( 'policy-map',
      "Show Policy Map", guard=QosCliServicePolicy.guardPolicyMap )
nodeCopp = CliCommand.guardedKeyword( 'copp',
      "Control-plane type", guard=guardPMapCopp )
nodeCoppDeprecated = CliCommand.guardedKeyword( 'control-plane',
      "Control-plane type", guard=guardPMapCopp,
      deprecatedByCmd='[no] policy-map type copp <pmap-name>' )
nodeCoppIntf = CliCommand.guardedKeyword( 'copp',
      "Control-plane type", guard=guardIntfCoppSupported )
nodeControlPlaneDeprecated = CliCommand.guardedKeyword( 'control-plane',
      "Control-plane type", guard=guardPMapCopp,
      deprecatedByCmd='show policy-map type copp [ Policy Map Name ]' )
nodeCoppSystemPolicy = CliCommand.guardedKeyword( tacPMapNm.coppName,
      "Copp Policy-map name", guard=guardCoppSystemPolicy )

# -------------------------------------------------------------------------------
# policy-map class mode
# -------------------------------------------------------------------------------
class PolicyMapClassContextCopp( QosCliServicePolicy.PolicyMapClassContext ):
   def newEditEntry( self, cmapName,  # pylint: disable-msg=arguments-differ
                     bypassCheck=False ):
      newClassAction = Tac.newInstance( 'Qos::ClassAction', cmapName )
      if bypassCheck or qosHwStatus.coppActionShaperSupported:
         newClassAction.policyAction.newMember( tacActionType.actionSetShape )
         newClassAction.policyAction[ tacActionType.actionSetShape ].rate = ()
         newClassAction.policyAction[ tacActionType.actionSetShape ].rate.val = \
             tacActionRateType.noValue
         newClassAction.policyAction.newMember( tacActionType.actionSetBandwidth )
         newClassAction.policyAction[ tacActionType.actionSetBandwidth ].rate = ()
         newClassAction.policyAction[ tacActionType.actionSetBandwidth ].rate.val = \
             tacActionRateType.noValue
      self.currentEntry_ = newClassAction

   def commit( self ):
      assert self.mapType_ == coppMapType
      cmapName = self.cmapName_

      def commitStaticORDefaultClass( cmapName ):
         clAction = None
         cpType = classMapCpType( cliQosAclConfig, cmapName )
         if cpType == tacClassMapCpType.cmapCpStatic:
            # since statiClass is being configured, no reordering.
            # Update classAction and return.
            if QosLib.isDefaultClass( self.mapType_, cmapName ):
               clAction = self.map_.classActionDefault
            else:
               if cmapName in self.map_.classAction:
                  clAction = self.map_.classAction[ cmapName ]
               else:
                  QosLib.coppPMapStaticClassIs( self.map_, cmapName, coppMapType )
                  clAction = self.map_.classAction[ cmapName ]

         if clAction is not None:
            QosCliServicePolicy.copyClassAction( self.currentEntry_, clAction )
            return True

         return False

      # For copp static / default class no re-ordering
      if not commitStaticORDefaultClass( cmapName ):
         self._commit()


# -----------------------------------------------------------------
# The show command in 'config-cmap' mode
#
#              show active|pending|diff
# -----------------------------------------------------------------
class CmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new class-map configuration to be applied',
      'active': 'Display the class-map configuration in the current running-config',
      'current': ( 'Display the class-map configuration '
         'in the current running-config' ),
      'diff': ( 'Display the diff between class-map configuration '
          'current running-config and to be applied' ),
   }

   handler = 'QosCliCoppHandler.doCmapModeShowCmd'

# --------------------------------------------------------------------------------
# class-map type ( copp | control-plane ) match-any CMAP
# --------------------------------------------------------------------------------
class QosCoppClassMapModeCmd( CliCommand.CliCommandClass ):
   syntax = 'class-map type ( copp | control-plane ) match-any CMAP'
   data = {
      'class-map': CliCommand.guardedKeyword( 'class-map',
         "Configure Class Map", guard=guardClassMap ),
      'type': matcherType,
      'copp': CliCommand.guardedKeyword( 'copp',
         "Control-plane type", guard=guardCMapDynamicCopp ),
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         "Control-plane type", guard=guardCMapDynamicCopp,
         deprecatedByCmd='class-map type copp match-any <class-name>' ),
      'match-any': matcherMatchAny,
      'CMAP': matcherCmapName,
   }

   handler = 'QosCliCoppHandler.gotoCoppClassMapMode'


BasicCliModes.GlobalConfigMode.addCommandClass( QosCoppClassMapModeCmd )

# --------------------------------------------------------------------------------
# ( no | default ) class-map type ( copp | control-plane ) [ match-any ] CMAP
# --------------------------------------------------------------------------------
class QosCoppClassMapModeNoOrDefaultCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = ( 'class-map type ( copp | control-plane ) '
                         '[ match-any ] CMAP' )
   data = {
      'class-map': CliCommand.guardedKeyword( 'class-map',
         "Configure Class Map", guard=guardClassMap ),
      'type': matcherType,
      'copp': nodeCopp,
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         "Control-plane type", guard=guardPMapCopp,
         deprecatedByCmd='no class-map type copp [ match-any ] <class-name>' ),
      'match-any': matcherMatchAny,
      'CMAP': matcherCmapName,
   }

   noOrDefaultHandler = 'QosCliCoppHandler.deleteCoppClassMap'


BasicCliModes.GlobalConfigMode.addCommandClass( QosCoppClassMapModeNoOrDefaultCmd )

# --------------------------------------------------------------------------------
# show class-map type control-plane [ CMAP ]
# --------------------------------------------------------------------------------
class ClassMapTypeControlPlaneCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show class-map type control-plane [ CMAP ]'
   data = {
      'class-map': CliCommand.guardedKeyword( 'class-map',
         "Show Class Map", guard=guardClassMap ),
      'type': matcherType,
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         "Control-plane type", guard=guardPMapCopp ),
      'CMAP': CliMatcher.DynamicNameMatcher( getCMapNameRuleCopp,
         'Class Map Name' ),
   }

   handler = 'QosCliCoppHandler.ClassMapModeCopp.showClassMap'
   cliModel = ClassMapAllModel


BasicCli.addShowCommandClass( ClassMapTypeControlPlaneCmd )

# ------------------------------------------------------------------------
# Register top-level CLI commands ( policy-map )
# ------------------------------------------------------------------------
# --------------------------------------------------------------------------------
# [ no | default ] policy-map type ( copp | control-plane )
#       ( copp-system-policy | PMAP )
# --------------------------------------------------------------------------------
class PolicyMapTypeCoppPmapCmd( CliCommand.CliCommandClass ):
   syntax = ( 'policy-map type ( copp | control-plane ) '
              '( copp-system-policy | PMAP )' )
   noOrDefaultSyntax = syntax
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'copp': nodeCopp,
      'control-plane': nodeCoppDeprecated,
      'copp-system-policy': nodeCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   handler = 'QosCliCoppHandler.gotoCoppPolicyMapMode'
   noOrDefaultHandler = 'QosCliCoppHandler.deleteCoppPolicyMap'


BasicCliModes.GlobalConfigMode.addCommandClass( PolicyMapTypeCoppPmapCmd )

# ------------------------------------------------------------------------
# show policy-map type copp [ ( copp-system-policy | PMAP ) ]
#
# legacy:
#     show policy-map type control-plane [ ( copp-system-policy | PMAP ) ]
# ------------------------------------------------------------------------
class PolicyMapTypeCoppCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map type ( copp | control-plane )'
               ' [ ( copp-system-policy | PMAP ) ]' )
   data = {
      'policy-map': nodePolicyMapShow,
      'type': matcherType,
      'copp': nodeCopp,
      'control-plane': nodeControlPlaneDeprecated,
      'copp-system-policy': matcherCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   handler = 'QosCliCoppHandler.PolicyMapModeCopp.showPolicyMap'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapTypeCoppCmd )

# ------------------------------------------------------------------------
# show policy-map type copp ( copp-system-policy | PMAP ) class CMAP
#
# legacy:
#     show policy-map type control-plane
#       ( copp-system-policy | PMAP ) class CMAP
# ------------------------------------------------------------------------
class PolicyMapTypeCoppClassCmapnameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map type ( copp | control-plane )'
              ' ( copp-system-policy | PMAP ) class CMAP' )
   data = {
      'policy-map': nodePolicyMapShow,
      'type': matcherType,
      'copp': nodeCopp,
      'control-plane': nodeControlPlaneDeprecated,
      'copp-system-policy': matcherCoppSystemPolicy,
      'PMAP': matcherPmapName,
      'class': matcherClass,
      'CMAP': matcherClassName,
   }

   handler = 'QosCliCoppHandler.PolicyMapClassModeCopp.showPolicyMapClass'
   cliModel = ClassMapWrapper


BasicCli.addShowCommandClass( PolicyMapTypeCoppClassCmapnameCmd )

# -----------------------------------------------------------------
# The show command in 'config-pmap' mode
#
#              show active|pending|diff
# -----------------------------------------------------------------
class PmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new policy-map configuration to be applied',
      'active': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'current': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'diff': ( 'Display the diff between policy-map configuration'
          'current running-config and to be applied' ),
   }

   handler = 'QosCliCoppHandler.doPmapModeShowCmd'

# ------------------------------------------------------------------------
# Match and set value binding rules ( policy-map )
# ------------------------------------------------------------------------

class AbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Abandon all changes',
   }

   handler = 'QosCliCoppHandler.doAbortCmd'


QosCliServicePolicy.PolicyMapClassModeQos.addCommandClass( AbortCmd )
QosProfileMode.addCommandClass( AbortCmd )
QosCliServicePolicy.PolicyMapModeQos.addCommandClass( AbortCmd )

# --------------------------------------------------------------------------------
# show policy-map ( copp | ( interface control-plane ) ) copp-system-policy
# --------------------------------------------------------------------------------
class PolicyMapCoppCoppSystemPolicyShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map ( copp | ( interface control-plane ) ) '
              'copp-system-policy' )
   data = {
      'policy-map': nodePolicyMapShow,
      'copp': nodeCopp,
      'interface': 'Service Policy on interface',
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         helpdesc='Control-plane type', guard=guardPMapCopp,
         deprecatedByCmd="show policy-map copp [ Policy Map Name ]" ),
      'copp-system-policy': nodeCoppSystemPolicy,
   }

   handler = 'QosCliCoppHandler.showPMapInterfaceControlPlane'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapCoppCoppSystemPolicyShowCmd )

# --------------------------------------------------------------------------------
# show policy-map ( copp | ( interface control-plane ) ) PMAP
# --------------------------------------------------------------------------------
class PolicyMapCoppShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map ( copp | ( interface control-plane ) ) PMAP' )
   data = {
      'policy-map': nodePolicyMapShow,
      'copp': nodeCoppIntf,
      'interface': 'Service Policy on interface',
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         helpdesc='Control-plane type', guard=guardIntfCoppSupported,
         deprecatedByCmd='show policy-map copp [ Policy Map Name ]' ),
      'copp-system-policy': nodeCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   handler = 'QosCliCoppHandler.showPMapInterfaceIntfCopp'
   cliModel = InterfaceIdModel


BasicCli.addShowCommandClass( PolicyMapCoppShowCmd )

# --------------------------------------------------------------------------------
# show policy-map interface INTF type control-plane
# --------------------------------------------------------------------------------
class PolicyMapInterfaceIntfTypeControlPlaneCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map interface INTF type control-plane'
   data = {
      'policy-map': nodePolicyMapShow,
      'interface': 'Service Policy on interface',
      'INTF': IntfRangeMatcher( explicitIntfTypes=ethOrLagIntfTypes ),
      'type': matcherType,
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         helpdesc='Control-plane type', guard=guardIntfCoppSupported ),
   }

   handler = 'QosCliServicePolicyHandler.showPMapInterface'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapInterfaceIntfTypeControlPlaneCmd )

# --------------------------------------------------------------------------------
# show policy-map type ( copp | control-plane )
#       [ ( copp-system-policy | PMAP ) ] counters
# --------------------------------------------------------------------------------
class PolicyMapTypeCoppCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map type ( copp | control-plane )'
              ' [ ( copp-system-policy | PMAP ) ] counters' )
   data = {
      'policy-map': nodePolicyMapShow,
      'type': matcherType,
      'copp': nodeCoppIntf,
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         "Control-plane type", guard=guardIntfCoppSupported,
         deprecatedByCmd='show policy-map type copp [ Policy Map Name ]' ),
      'copp-system-policy': matcherCoppSystemPolicy,
      'PMAP': matcherPmapName,
      'counters': 'Policy Map counters',
   }

   handler = 'QosCliServicePolicyHandler.showPolicyMapCounters'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapTypeCoppCountersCmd )

class ServicePolicyInputCoppSystemPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy input copp-system-policy'
   noOrDefaultSyntax = syntax
   data = {
      'service-policy': CliCommand.guardedKeyword( 'service-policy',
         "Configure QoS Control-plane Service Policy",
         guard=guardServicePolicyCopp ),
      'input': 'Apply the policy map to ingress packets',
      'copp-system-policy': nodeCoppSystemPolicy,
   }

   handler = 'QosCliCoppHandler._configureCpServicePolicy'
   noHandler = 'QosCliCoppHandler._configureNoCpServicePolicy'
   defaultHandler = 'QosCliCoppHandler._configureDefaultCpServicePolicy'


AclCli.CpConfigMode.addCommandClass( ServicePolicyInputCoppSystemPolicyCmd )

def setCoppServicePolicy( mode, noOrDefaultKw=None, pmapName=None ):
   if noOrDefaultKw is None:
      intfName = mode.intf.name
      if isLagPort( intfName ):
         mode.addError( CliError[ 'lagPolicingNotSupported' ] )
         return
   setServicePolicy( mode, noOrDefaultKw=noOrDefaultKw, mapType='control-plane',
         pmapName=pmapName, direction='input' )

# --------------------------------------------------------------------------------
# [ no | default ] service-policy type control-plane ( copp-system-policy | PMAP )
# --------------------------------------------------------------------------------
class ServicePolicyTypeControlPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy type control-plane ( copp-system-policy | PMAP )'
   noOrDefaultSyntax = ( 'service-policy type control-plane '
                         '[ copp-system-policy | PMAP ]' )
   data = {
      'service-policy': CliCommand.guardedKeyword( 'service-policy',
         "Service policy configuration",
         guard=QosCliServicePolicy.guardIntfModePolicy ),
      'type': matcherType,
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         "Control-plane type", guard=guardIntfCoppSupported ),
      'copp-system-policy': nodeCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   handler = 'QosCliCoppHandler.doServicePolicyTypeControlPlaneCmd'
   noOrDefaultHandler = 'QosCliCoppHandler.noServicePolicyTypeControlPlaneCmd'


QosModelet.addCommandClass( ServicePolicyTypeControlPlaneCmd )

matcherPolicyMap = CliMatcher.KeywordMatcher( 'policy-map',
   helpdesc='Clear Policy Map' )
nodePolicyMapNew = CliCommand.Node( matcher=matcherPolicyMap,
   guard=QosCliServicePolicy.guardPolicyMap )
# --------------------------------------------------------------------------------
# clear policy-map interface control-plane counters copp-system-policy
# --------------------------------------------------------------------------------

class ClearPMapIntfCtrlPlaneCountersCoppSysPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'clear policy-map interface control-plane counters copp-system-policy'
   data = {
      'clear': clearKwNode,
      'policy-map': nodePolicyMapNew,
      'interface': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'interface',
         helpdesc='Service Policy on interface' ), guard=guardPMapCopp ),
      'control-plane': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'control-plane', helpdesc='Control-plane type' ),
         guard=guardPMapCopp ),
      'counters': 'Policy Map counters',
      'copp-system-policy': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'copp-system-policy', helpdesc='Copp Policy-map name' ),
         guard=guardCoppSystemPolicy ),
   }
   handler = 'QosCliCoppHandler.clearPMapIntfCounters'


BasicCliModes.EnableMode.addCommandClass(
   ClearPMapIntfCtrlPlaneCountersCoppSysPolicyCmd )

# --------------------------------------------------------------------------------
# clear policy-map type qos [ PMAPNAME ] counters
# --------------------------------------------------------------------------------
class ClearPolicyMapTypeQosCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear policy-map type qos [ PMAPNAME ] counters'
   data = {
      'clear': clearKwNode,
      'policy-map': nodePolicyMapNew,
      'type': 'Specify type',
      'qos': CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'qos',
         helpdesc='Qos type' ), guard=QosCliServicePolicy.guardPMapQos ),
      'PMAPNAME': CliMatcher.DynamicNameMatcher( QosCliServicePolicy.getPMapNameQos,
         'Policy Map Name',
         pattern='(?!counters$|interface$|summary$)[A-Za-z0-9_:{}\\[\\]-]+' ),
      'counters': 'Policy Map counters',
   }
   handler = 'QosCliCoppHandler.clearPMapIntfCounters'


BasicCliModes.EnableMode.addCommandClass( ClearPolicyMapTypeQosCountersCmd )

# ------------------------------------------------------------------------------
# configure convert copp-system-policy
# -----------------------------------------------------------------------------
# This will read copp-system-policy from a non-per-port-copp config
# It then creates a new policy map with the same class maps and attaches
# the new policy map to all ports.

def assignClassActionCoppPolicer( classAction, className, cir, cirUnit, bc, bcUnit ):
   # pir and be are 0 because they are not used for copp policers
   classAction.policer = ( className, cir, bc, 0, 0 )
   classAction.policer.cirUnit = cirUnit
   classAction.policer.bcUnit = bcUnit
   action = tacActionType.actionSetDrop
   classAction.policer.redActions.newMember( action )
   classAction.policer.redActions[ action ].value = 1


# Setup hooks in QosCli.py
# pylint: disable-msg=protected-access
QosCliServicePolicy._guardPolicyMapCallback[ coppMapType ] = guardPMapCopp
QosCliServicePolicy._guardClassMapCallback[ coppMapType ] = guardCMapCopp
QosCliServicePolicy._guardIntfModePolicyCallback[ coppMapType ] = \
   guardIntfModePolicyCopp
QosCliServicePolicy._policyMapClassContextType[ coppMapType ] = \
   PolicyMapClassContextCopp
# pylint: enable-msg=protected-access

@Plugins.plugin( provides=( "QosCliCopp", ) )
def Plugin( entityManager ):
   global qosAclHwStatus, cliQosAclConfig, qosHwStatus
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   cliQosAclConfig = ConfigMount.mount( entityManager, "qos/acl/input/cli",
                                        "Qos::Input::AclConfig", "w" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
