#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
import Plugins
import LazyMount
import QosLib
from CliPlugin import SubIntfCli, IntfCli
from CliPlugin.IntfCli import IntfConfigMode
from CliPlugin.QosCliCommon import ( isSubIntfConfigMode, QosProfileMode,
                                     IntfTxQueueRangeConfigMode,
                                     IntfTxQueueConfigMode )
from CliPlugin.QosCliScheduling import QosSchedulingPolicyMode
from CliPlugin.QosCliServicePolicy import guardTxQPrioritySupport
from CliPlugin.QosCliIntfTypes import ethOrLagIntfPrefixes
from QosTypes import ( tacPercent, tacGuaranteedBwVal, tacGuaranteedBwUnit )
from QosLib import isTxQueueNotConfigurable

# Guaranteed Bandwidth Modelet
class QosModeletGuaranteedBw( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )

IntfCli.IntfConfigMode.addModelet( QosModeletGuaranteedBw )

# -----------------------------------------------------------------------------------
# Variables for Qos Wred associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosHwStatus = None
subIntfHwStatus = None

class QosGuaranteedBwModuleBase( QosLib.QosModuleBase ):
   def isDefaultIntfConfig( self, intfConfig ):
      cfgGuaranteedBw = intfConfig.guaranteedBw
      return ( ( cfgGuaranteedBw.bw == tacGuaranteedBwVal.invalid ) and
               ( cfgGuaranteedBw.unit == tacGuaranteedBwUnit.guaranteedBwKbps ) )


QosLib.registerQosModule( 'guaranteedBw', QosGuaranteedBwModuleBase( qosHwStatus ) )

# -----------------------------------------------------------------------------------
# Guards
# -----------------------------------------------------------------------------------
def guardGuaranteedBwPps( mode, token ):
   if not qosHwStatus.guaranteedBwUnitPpsSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardBwWeightSupport( mode, token ):
   if not qosHwStatus.bandwidthWeightSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardGuaranteedBw( mode, token ):
   if ( isinstance( mode, ( IntfTxQueueConfigMode, IntfTxQueueRangeConfigMode ) ) and
        not qosHwStatus.guaranteedBwSupported ):
      return CliParser.guardNotThisPlatform
   if ( isinstance( mode, IntfConfigMode ) and
        mode.intf.isSubIntf() and
        not subIntfHwStatus.subIntfBandwidthSupported ):
      return CliParser.guardNotThisPlatform
   if ( isinstance( mode, IntfConfigMode ) and
        not mode.intf.isSubIntf() and
        not qosHwStatus.parentGuaranteedBwSupported ):
      return CliParser.guardNotThisPlatform
   if ( isinstance( mode, QosProfileMode ) and
        not subIntfHwStatus.subIntfBandwidthSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardTxQPriorityOrGuaranteedBwSupport( mode, token ):
   if isinstance( mode, IntfTxQueueConfigMode ):
      if isTxQueueNotConfigurable( qosHwStatus, mode.txQueue.id ):
         return CliParser.guardNotThisPlatform
   if guardTxQPrioritySupport( mode, token ) and \
      guardGuaranteedBw( mode, token ):
      return CliParser.guardNotThisPlatform
   if ( isSubIntfConfigMode( mode ) and
        not subIntfHwStatus.subIntfTxQueueSchSupported ):
      return CliParser.guardNotThisPlatform
   return None


# -----------------------------------------------------------------------------------
# Tokens
# -----------------------------------------------------------------------------------
nodeGuaranteed = CliCommand.guardedKeyword( 'guaranteed',
      helpdesc='Set guaranteed bandwidth', guard=guardGuaranteedBw )
nodeBandwidth = CliCommand.guardedKeyword( 'bandwidth',
      helpdesc='Configure guaranteed/round-robin minimum bw for this queue',
      guard=guardTxQPriorityOrGuaranteedBwSupport )

# -----------------------------------------------------------------------------------
# Matchers
# -----------------------------------------------------------------------------------
matcherPercent = CliMatcher.IntegerMatcher( tacPercent.min, tacPercent.max,
      helpdesc='Percent value' )

# --------------------------------------------------------------------------------
# bandwidth guaranteed ( ( percent PERCENT ) | ( PPS pps ) | ( KBPS [ kbps ] ) )
# ( no | default ) bandwidth guaranteed ...
# --------------------------------------------------------------------------------
def guaranteedBwPpsRangeFn( mode, context=None ):
   return ( qosHwStatus.minGuaranteedBwPps, qosHwStatus.maxGuaranteedBwPps )

def guaranteedBwKbpsRangeFn( mode, context=None ):
   return ( qosHwStatus.minGuaranteedBwKbps, qosHwStatus.maxGuaranteedBwKbps )

class BandwidthGuaranteedCmd( CliCommand.CliCommandClass ):
   syntax = ( 'bandwidth guaranteed ( ( percent PERCENT ) | '
              '( PPS pps ) | ( KBPS [ kbps ] ) )' )
   noOrDefaultSyntax = 'bandwidth guaranteed ...'
   data = {
      'bandwidth': nodeBandwidth,
      'guaranteed': nodeGuaranteed,
      'percent': 'Bandwidth Guaranteed in Percent',
      'PERCENT': matcherPercent,
      'PPS': CliMatcher.DynamicIntegerMatcher(
         rangeFn=guaranteedBwPpsRangeFn, helpdesc='Bandwidth Guaranteed in Pps' ),
      'pps': CliCommand.guardedKeyword( 'pps',
         helpdesc='Bandwidth Guaranteed in Pps', guard=guardGuaranteedBwPps ),
      'KBPS': CliMatcher.DynamicIntegerMatcher(
         rangeFn=guaranteedBwKbpsRangeFn, helpdesc='Bandwidth Guaranteed in Kbps' ),
      'kbps': 'Bandwidth Guaranteed in Kbps (default unit)',
   }

   handler = "QosCliGuaranteedBwHandler.setGuaranteedBw"
   noOrDefaultHandler = "QosCliGuaranteedBwHandler.noGuaranteedBw"


IntfTxQueueConfigMode.addCommandClass( BandwidthGuaranteedCmd )
SubIntfCli.SubIntfModelet.addCommandClass( BandwidthGuaranteedCmd )
QosProfileMode.addCommandClass( BandwidthGuaranteedCmd )
QosSchedulingPolicyMode.addCommandClass( BandwidthGuaranteedCmd )
QosModeletGuaranteedBw.addCommandClass( BandwidthGuaranteedCmd )

# --------------------------------------------------------------------------------
# bandwidth ( percent PERCENT | weight WEIGHT )
# ( no | default ) bandwidth ...
# --------------------------------------------------------------------------------
def bwWeightRangeFn( mode=None, context=None ):
   return ( 1, qosHwStatus.maxBwWeightValue )

class BandwidthCmd( CliCommand.CliCommandClass ):
   syntax = 'bandwidth ( ( percent PERCENT ) | ( weight WEIGHT ) )'
   noOrDefaultSyntax = 'bandwidth ...'
   data = {
      'bandwidth': nodeBandwidth,
      'percent': CliCommand.guardedKeyword( 'percent',
         helpdesc=( 'Set proportion of the total bandwidth '
            'when scheduling is round-robin' ), guard=guardTxQPrioritySupport ),
      'PERCENT': matcherPercent,
      'weight': CliCommand.guardedKeyword( 'weight',
         helpdesc=( 'Queue weight when the scheduling is round robin' ),
            guard=guardBwWeightSupport ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher( bwWeightRangeFn,
         helpdesc='Weight value' )
   }

   handler = "QosCliGuaranteedBwHandler.setBandwidth"
   noOrDefaultHandler = "QosCliGuaranteedBwHandler.noBandwidth"


IntfTxQueueConfigMode.addCommandClass( BandwidthCmd )

@Plugins.plugin( provides=( "QosCliGuaranteedBw", ) )
def Plugin( entityManager ):
   global qosHwStatus, subIntfHwStatus
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )
