#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliToken
import ConfigMount
import LazyMount
import Plugins
import QosLib
import SharedMem
import ShowCommand
import Smash
import Tac
from PfcTypes import ( tacPfcStatus, tacPfcWatchdogAction )
from CliPlugin import ( FabricConfigCli, IntfCli, TechSupportCli )
from CliPlugin.QosCliIntfTypes import ethIntfPrefixes
from CliPlugin.QosCliCommon import ( counterSupportedEthPhyIntfs, QosProfileMode )
from CliPlugin.QosCliModel import ( numPriority,
                                   PfcFabricIntfStatusModel, PfcFrameCountersModel,
                                   PfcIntfRangeCountersModel,
                                   PfcIntfRangeDetailedCountersModel,
                                   PfcIntfRangeHistoryCountersModel,
                                   PfcIntfRangeModel, PfcIntfRangeStatusModel,
                                   PfcIntfStatusModel, PfcStatusModel,
                                   PfcWatchdogIntfRangeCountersModel,
                                   PfcWatchdogIntfRangeDropCountersModel,
                                   PfcWatchdogNDPrioritiesModel,
                                   PfcWatchdogStatusModel, PfcWatchdogTimerModel )
PfcHistoryCounterKey = Tac.Type( "Qos::PfcHistoryCounterKey" )

# xxxxx
#
# Converts an 8-bit unsigned integer encoding of priorities into a list.
#
# Priority p becomes a member of the list if bit p is set in the 8-bit
# unsigned integer.
#
def prioritiesToList( priorities ):
   return [ p for p in range( numPriority ) if ( 1 << p ) & priorities ]

#
# Converts a boolean array into a list containing array indices
# as members for which the value (array[i]) is True,
# otherwise returns empty list if priority array is empty
#
def priArrayToList( priorities ):
   return [ idx for idx in priorities if priorities[ idx ] ]

#
# Converts a watchdog action into an enum-string for use in a watchdog model.
#
def watchdogActionToEnum( action ):

   if action == tacPfcWatchdogAction.notifyOnly:
      return 'notify-only'

   if action == tacPfcWatchdogAction.errdisable:
      return 'errdisable'

   if action == tacPfcWatchdogAction.drop:
      return 'drop'

   if action == tacPfcWatchdogAction.ignorePfc:
      return 'forward'

   return None

#
# Recursively traverses an attribute hierarchy and obtains the value of the
# leaf attribute.
#
def getAttrRecursive( value, attrs, index=None, allowNone=False ):
   for attr in attrs:
      try:
         value = getattr( value, attr, None )
      except AttributeError:
         return None if allowNone else 0

   if value and index is not None:
      value = value[ index ]

   return 0 if ( not allowNone and value is None ) else value

#
# Obtains the value of an interface counter present in the ethStatistics
# object hierarchy.
#
def statistics( counter, checkpoint, attrs, index=None, allowNone=False ):
   attrs.insert( 0, 'ethStatistics' )

   counterValue = getAttrRecursive( counter, attrs, index, allowNone=allowNone )
   checkpointValue = getAttrRecursive( checkpoint, attrs, index )

   if counterValue is None:
      return None
   else:
      return counterValue - checkpointValue

#
# Obtains the PFC port configuration for an interface.
#
def portPfcConfig( intfName ):
   intfConfig = qosConfig.intfConfig.get( intfName )

   if not intfConfig:
      return None

   portConfig = intfConfig.pfcPortConfig

   return portConfig

#
# Conditionally prepends the DCBX configuration state to the informational
# note for an interface.
#
def portDcbxNote( intfName, pfcEnabled, note ):

   if not pfcEnabled:
      return note

   dcbxMode = dcbxConfig.portEnabled.get( intfName )

   if not ( ( dcbxMode is None ) or ( dcbxMode == 'modeDisabled' ) ):
      return note

   dcbxNote = "DCBX disabled"

   if note is None:
      return dcbxNote

   return dcbxNote + "; " + note

#
# Obtains the status of PFC activation on the interface hardware.
#
def portHwActivationStatus( intfName ):
   portStatus = None

   for sliceHwStatus in qosSliceHwStatus.values():
      portStatus = sliceHwStatus.portStatus.get( intfName )

      if portStatus:
         break

   if portStatus:
      active = portStatus.active
      note = portStatus.note if portStatus.note else None
   else:
      active = False
      note = None

   if not active:
      # If interface PFC activeness is published by a non-slice agent, it will be
      # present in pfcSliceHwStatus instead of qosSliceHwStatus. Need to check both
      for sliceHwStatus in pfcSliceHwStatus.values():
         if intfName in sliceHwStatus.activeIntfs:
            active = True
            break

   return ( active, note )

#
# Creates and populates an instance of PfcStatusModel.
#
def pfcStatusModelGet():
   """Return the model for PFC global status."""

   model = PfcStatusModel()

   model.enabled = qosConfig.pfcGlobalEnabled

   if not pfcStatus.prioritiesSupported:
      model.supported = False
      model.priorities = None
      return model

   model.supported = True
   model.priorities = prioritiesToList( pfcStatus.prioritiesSupported )

   model.rxPfcPrioritiesHonored = prioritiesToList(
         qosConfig.rxPfcPrioritiesHonored )

   return model

#
# Creates and populates an instance of PfcWatchdogTimerModel.
#
def pfcWatchdogTimerModelGet():
   """Return the model for PFC Watchdog global timer configuration."""

   model = PfcWatchdogTimerModel()

   model.timeout = qosConfig.watchdogTimeout
   model.pollInterval = qosConfig.watchdogPollingInterval
   model.operPollInterval = pfcConfig.watchdogOperPollingInterval
   model.recoveryTime = qosConfig.watchdogRecoveryCfg.recoveryTime
   model.forcedRecovery = qosConfig.watchdogRecoveryCfg.forcedRecovery

   return model

#
# Creates and populates an instance of PfcWatchdogTimerModel for an interface.
#
def pfcWatchdogIntfTimerModelGet( intfName, timerConfig ):
   """Return the model for PFC Watchdog interface timer configuration."""

   #
   # Nothing to do if interface-level timer values are not in use.
   #
   if not timerConfig.usePerPortTimerValues:
      return None

   model = PfcWatchdogTimerModel()

   model.timeout = timerConfig.portWatchdogTimeout

   model.pollInterval = timerConfig.portWatchdogPollingInterval

   model.recoveryTime = timerConfig.portWatchdogRecoveryCfg.recoveryTime
   model.forcedRecovery = timerConfig.portWatchdogRecoveryCfg.forcedRecovery

   if intfName not in pfcConfig.portConfig:
      model.operPollInterval = None
      return model

   pfcPortConfig = pfcConfig.portConfig[ intfName ]

   model.operPollInterval = \
      pfcPortConfig.portTimerConfig.portWatchdogOperPollingInterval

   return model

#
# Creates and populates an instance of PfcWatchdogNDPrioritiesModel.
#
def pfcWatchdogNDPrioritiesModelGet():
   """Return the model for PFC Watchdog non-disruptive priorities global status."""

   model = PfcWatchdogNDPrioritiesModel()

   if not pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported:
      model.supported = False
      model.priorities = None
      return model

   if pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported == \
      PFC_WATCHDOG_ND_ALL_PRI:
      model.supported = True
      model.priorities = list( range( PFC_WATCHDOG_ND_ALL_PRI ) )
      return model

   model.supported = True
   model.priorities = prioritiesToList(
      qosConfig.watchdogNonDisruptivePriorities )

   return model

#
# Creates and populates an instance of PfcWatchdogNDActionModel.
#
def pfcWatchdogNDActionModelGet():
   """Return the model for PFC Watchdog non-disruptive action global status."""

   nDAction = watchdogActionToEnum( qosConfig.watchdogNonDisruptiveAction )

   return nDAction if nDAction is not None else 'invalid'

#
# Creates and populates an instance of PfcWatchdogNDPrioritiesModel.
#
def pfcWatchdogForcedNDModelGet():
   """Return the model for PFC Watchdog non-disruptive priorities global status."""

   if not pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported:
      return None

   return qosConfig.forcedNonDisruptiveBehaviour

#
# Creates and populates an instance of PfcWatchdogOverrideActionModel.
#
def pfcWatchdogOverrideActionModelGet():
   """Return the model for PFC Watchdog override action drop status."""

   return qosConfig.overrideAction

#
# Creates and populates an instance of PfcWatchdogStatusModel.
#
def pfcWatchdogStatusModelGet():
   """Return the model for PFC Watchdog global status."""

   #
   # Nothing to do if PFC itself is not supported.
   #
   if not pfcStatus.prioritiesSupported:
      return None

   model = PfcWatchdogStatusModel()

   if not pfcStatus.watchdogSupported:
      model.supported = False
      model.action = None
      model.timer = None
      model.nDPriorities = None
      model.nDAction = None
      model.forcedNd = None
      model.overrideAction = None
      model.hwMonitoredPri = None
      return model

   model.supported = True
   model.action = watchdogActionToEnum( qosConfig.watchdogAction )
   model.timer = pfcWatchdogTimerModelGet()
   model.nDPriorities = pfcWatchdogNDPrioritiesModelGet()
   model.nDAction = pfcWatchdogNDActionModelGet()
   model.forcedNd = pfcWatchdogForcedNDModelGet()
   model.overrideAction = pfcWatchdogOverrideActionModelGet()
   if not pfcStatus.hardwareMonitorSupported:
      model.hwMonitoredPri = None
   else:
      model.hwMonitoredPri = priArrayToList( qosConfig.hwMonitoredPri )
   return model

#
# Creates and populates an instance of PfcIntfStatusModel.
#
def pfcIntfStatusModelGet( intf ):
   """Return the model for PFC status on a given interface."""

   model = PfcIntfStatusModel()

   ( model.active, model.note ) = portHwActivationStatus( intf.name )

   #
   # Configuration specific to the interface.
   #
   portConfig = portPfcConfig( intf.name )

   if not portConfig:
      model.enabled = False
      model.priorities = []
      model.watchdogEnabled = False
      model.watchdogActive = False
      model.watchdogAction = None
      model.watchdogTimer = None
      return model

   model.enabled = portConfig.enabled
   model.priorities = prioritiesToList( portConfig.priorities )

   model.watchdogEnabled = \
      portConfig.watchdogEnabled if pfcStatus.watchdogSupported else False

   model.watchdogAction = watchdogActionToEnum( portConfig.watchdogPortAction )

   model.watchdogTimer = \
      pfcWatchdogIntfTimerModelGet( intf.name, portConfig.portTimerConfig )

   portDcbConfig = pfcConfig.portConfig.get( intf.name )
   model.watchdogActive = \
         model.active and \
         ( portDcbConfig is not None ) and \
         portDcbConfig.watchdogActive

   model.note = portDcbxNote( intf.name, model.enabled, model.note )

   return model
#
# Creates and populates an instance of PfcFrameCountersModel for an interface.
#

def pfcIntfCountersModelGet( intf ):
   """Return the model for PFC counters on a given interface."""

   model = PfcFrameCountersModel()

   counter = intf.counter()
   checkpoint = intf.getLatestCounterCheckpoint()

   model.rxFrames = statistics( counter, checkpoint, [ 'inPfcFrames' ] )
   model.txFrames = statistics( counter, checkpoint, [ 'outPfcFrames' ] )

   return model

#
# Creates and populates an instance of PfcIntfRangeModel.
#
def pfcIntfRangeModelGet( intfs ):
   """Return the model for PFC on a given range of interfaces."""

   model = PfcIntfRangeModel()

   model.status = pfcStatusModelGet()
   model.watchdogStatus = pfcWatchdogStatusModelGet()

   model.interfaceStatuses = {}

   for intf in intfs:
      model.interfaceStatuses[ intf.status().intfId ] = pfcIntfStatusModelGet( intf )

   model.interfaceCounters = {}

   for intf in intfs:
      model.interfaceCounters[ intf.status().intfId ] = \
                                                    pfcIntfCountersModelGet( intf )

   return model

#
# Value function for the PFC interface show command.
#
def showPfcInterface( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control
      - show priority-flow-control [ interfaces <range> ]

      Model := PfcIntfRangeModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeModelGet( intfs )
# xxxxx

# --------------------------------------------------------------------------------
# The Pfc globals
# --------------------------------------------------------------------------------
qosInputConfig = None
qosProfileConfigDir = None
pfcStatus = None
qosHwStatus = None
qosSliceHwStatus = None
pfcSliceHwStatus = None
pfcWdSliceHwStatus = None
qosConfig = None
dcbxConfig = None
pfcConfig = None
pfcHistoryCounterStatus = None
pfcHistoryStatus = None
pfcPacketBufferProfileStatus = None

# --------------------------------------------------------------------------------
# Some CONSTANTS
# --------------------------------------------------------------------------------
PFC_WATCHDOG_INTERMEDIATE_BOUNDARY = 0.2
PFC_WATCHDOG_ND_ALL_PRI = 8

def isDefaultPfcPortTimerConfig( portTimerConfig ):
   return not ( portTimerConfig and portTimerConfig.usePerPortTimerValues )

def isDefaultPfcPortConfig( pfcPortConfig ):
   if pfcPortConfig is None:
      return True
   return ( not pfcPortConfig.enabled
            and pfcPortConfig.watchdogEnabled
            and ( pfcPortConfig.priorities == 0 )
            and ( pfcPortConfig.watchdogPortAction == tacPfcWatchdogAction.invalid )
            and ( isDefaultPfcPortTimerConfig( pfcPortConfig.portTimerConfig ) ) )
# --------------------------------------------------------------------------------
# Pfc module to be registered with Qos
# --------------------------------------------------------------------------------

class QosPfcModuleBase( QosLib.QosModuleBase ):
   def isDefaultIntfConfig( self, intfConfig ):
      cfgPfcPortConfig = intfConfig.pfcPortConfig
      return isDefaultPfcPortConfig( cfgPfcPortConfig )


QosLib.registerQosModule( 'pfc', QosPfcModuleBase( qosHwStatus ) )

# --------------------------------------------------------------------------------
# Pfc
# --------------------------------------------------------------------------------
class pfcModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # interfaces than to look at their name.
      return ( mode.intf.name.startswith( ethIntfPrefixes ) and
               not mode.intf.isSubIntf() )

class PfcIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intf = self.intf_.name
      intfConfig = qosInputConfig.intfConfig.get( intf )
      if intfConfig and intfConfig.pfcPortConfig:
         intfConfig.pfcPortConfig = None


IntfCli.IntfConfigMode.addModelet( pfcModelet )

# priority-flow-control pause watchdog default timeout < 0.01 - 60 > s
# priority-flow-control pause watchdog default recovery-time < 0.01 -60 > s
# no priority-flow-control pause watchdog

def pfcWatchdogGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   return None

def pfcWatchdogPortConfigGuard( mode, token ):
   if not pfcStatus.watchdogPortConfigSupported:
      return CliParser.guardNotThisPlatform
   return None

def pfcWatchdogErrdisableGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   else:
      if pfcStatus.watchdogErrdisableSupported:
         return None
      else:
         for sliceHwStatus in qosSliceHwStatus.values():
            if sliceHwStatus.watchdogErrdisableSupported:
               return None
            else:
               return CliParser.guardNotThisPlatform

def pfcWatchdogNotifyOnlyGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   else:
      if pfcStatus.watchdogNotifyOnlySupported:
         return None
      else:
         for sliceHwStatus in qosSliceHwStatus.values():
            if sliceHwStatus.watchdogNotifyOnlySupported:
               return None
            else:
               return CliParser.guardNotThisPlatform

def pfcWatchdogDropGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   else:
      if pfcStatus.watchdogDropSupported:
         return None
      else:
         for sliceHwStatus in qosSliceHwStatus.values():
            if sliceHwStatus.watchdogDropSupported:
               return None
            else:
               return CliParser.guardNotThisPlatform

def pfcWatchdogForcedNDGuard( mode, token ):
   if pfcStatus.watchdogSupported and \
      pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported and \
      pfcStatus.watchdogForcedNDSupported:
      return None
   return CliParser.guardNotThisPlatform

def pfcWatchdogNonDisruptivePriorityGuard( mode, token ):
   if pfcStatus.watchdogSupported and \
         pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported and \
         pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported != \
      PFC_WATCHDOG_ND_ALL_PRI:
      # In case watchdogMaxNonDisruptivePrioritiesSupported is 8, all pri are
      # ND by default, therefore we guard the enable non disruptive cli
      return None
   return CliParser.guardNotThisPlatform

def pfcWatchdogNonDisruptiveFwdActionGuard( mode, token ):
   if pfcStatus.watchdogSupported and \
      pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported:
      return None
   return CliParser.guardNotThisPlatform

def maxPfcWdNonDisruptPriorities():
   if pfcStatus.watchdogSupported:
      return pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported
   return 0

def pfcTxConfigGuard( mode, token ):
   if pfcStatus.pfcTxSupported:
      return None
   return CliParser.guardNotThisPlatform

def pfcSupported():
   return pfcStatus.pfcTxSupported or pfcStatus.pfcRxSupported

def pfcKeywordGuard( mode, token ):
   """CLI Guard for priority-flow-control keyword"""
   if pfcSupported():
      return None
   return CliParser.guardNotThisPlatform

def pfcPacketBufferGuard( mode, token ):
   """CLI Guard for packet-buffer keyword"""
   if pfcStatus.pfcDynamicPacketBufferSupported:
      return None
   return CliParser.guardNotThisPlatform

def pfcSelectivePrioritiesGuard( mode, token ):
   if pfcStatus.pfcRxSelectivePriorities:
      return None
   return CliParser.guardNotThisPlatform

def pfcWdPollingIntervalRangeFn( mode, ctx ):
   # This code can be called when the startup-config is being parsed
   # and before Sysdb is populated. The range is made the minimum and
   # maximum possible in this case.
   if pfcStatus is None:
      return( tacPfcStatus.minPollingInterval, tacPfcStatus.maxPollingInterval )
   else:
      return(
         round( pfcStatus.minPollingInterval, 3 ),
         round( pfcStatus.maxPollingInterval, 3 ) )

def pfcWdTimeoutRangeFn( mode, ctx ):
   # This code can be called when the startup-config is being parsed
   # and before Sysdb is populated. The range is made the minimum and
   # maximum possible in this case.
   if pfcStatus is None:
      return( tacPfcStatus.minTimeout, tacPfcStatus.maxTimeout )
   else:
      return( round( pfcStatus.minTimeout, 3 ),
              round( pfcStatus.maxTimeout, 3 ) )

def pfcWdRecoverTimeRangeFn( mode, ctx ):
   # This code can be called when the startup-config is being parsed
   # and before Sysdb is populated. The range is made the minimum and
   # maximum possible in this case.
   if pfcStatus is None:
      return( tacPfcStatus.minRecoveryTime, tacPfcStatus.maxRecoveryTime )
   else:
      return( round( pfcStatus.minRecoveryTime, 3 ),
              round( pfcStatus.maxRecoveryTime, 3 ) )

def hwMonitorGuard( mode, token ):
   if mode.name in ( 'Configure', 'config' ):
      if not pfcStatus.hardwareMonitorSupported:
         return CliParser.guardNotThisPlatform
   return None

def guardDlb( mode, token ):
   if qosHwStatus.dlbSupported:
      return None
   return CliParser.guardNotThisPlatform


matcherPfc = CliMatcher.KeywordMatcher( 'priority-flow-control',
      helpdesc='Configure PFC' )
nodePfc = CliCommand.guardedKeyword( 'priority-flow-control', 'Configure PFC',
      guard=pfcKeywordGuard )
matcherMode = CliMatcher.KeywordMatcher( 'mode',
      helpdesc='Configure PFC mode' )
nodePacketBuffer = CliCommand.guardedKeyword( 'packet-buffer', 'Packet buffer',
      guard=pfcPacketBufferGuard )
matcherIntegerPercent = CliMatcher.IntegerMatcher( 0, 100,
      helpdesc="Integer percent in 0..100" )
nodePause = CliCommand.guardedKeyword( 'pause', "Pause configuration",
      guard=pfcWatchdogGuard )
nodePortConfigPause = CliCommand.guardedKeyword( 'pause', "Pause configuration",
      guard=pfcWatchdogPortConfigGuard )
matcherWatchdog = CliMatcher.KeywordMatcher( 'watchdog',
      helpdesc='Enable watchdog on transmit queues' )
nodePfcPriority = CliCommand.guardedKeyword( 'priority', "Configure PFC priority",
      guard=pfcTxConfigGuard )
matcherPriority = CliMatcher.IntegerMatcher( 0, 7, helpdesc='Priority number' )
matcherDefault = CliMatcher.KeywordMatcher( 'default',
      helpdesc='Change default values' )
matcherTimeout = CliMatcher.KeywordMatcher( 'timeout',
      helpdesc='Configure timeout after which port should be errdisabled '
      'or should start dropping on congested priorities' )
matcherTimeoutValue = CliMatcher.DynamicFloatMatcher(
      rangeFn=pfcWdTimeoutRangeFn, helpdesc='Timeout value in seconds',
      precisionString='%.5g' )
matcherRecoveryTime = CliMatcher.KeywordMatcher( 'recovery-time',
      helpdesc='Configure recovery-time after which stuck queue should '
      'recover and start forwarding' )
matcherRecoveryTimeValue = CliMatcher.DynamicFloatMatcher(
      rangeFn=pfcWdRecoverTimeRangeFn, helpdesc='Recovery time in seconds',
      precisionString='%.5g' )
matcherForced = CliMatcher.KeywordMatcher( 'forced',
      helpdesc='Force recover any stuck queue(s) after the duration, '
      'irrespective of whether PFC frames are being received or not' )
matcherZeroValue = CliMatcher.KeywordMatcher( '0',
      helpdesc='No recovery' )
matcherPollingInterval = CliMatcher.KeywordMatcher( 'polling-interval',
      helpdesc='Configure the interval at which the watchdog should poll '
      'the queues' )
matcherPollingIntervalValue = CliMatcher.DynamicFloatMatcher(
      rangeFn=pfcWdPollingIntervalRangeFn, helpdesc='Polling interval in seconds',
      precisionString='%.5g' )
nodeErrdisable = CliCommand.guardedKeyword( 'errdisable',
      "Watchdog action for stuck transmit queues",
      guard=pfcWatchdogErrdisableGuard )
nodeNotifyOnly = CliCommand.guardedKeyword( 'notify-only',
      "No action on the stuck queue",
      guard=pfcWatchdogNotifyOnlyGuard )
nodeDrop = CliCommand.guardedKeyword( 'drop', "Drop traffic on the stuck queue",
      guard=pfcWatchdogDropGuard )
nodeNonDisruptive = CliCommand.guardedKeyword( 'non-disruptive',
      "Watchdog non-disruptive configuration",
      guard=pfcWatchdogNonDisruptivePriorityGuard )
nodeWatchdogPriorityList = CliCommand.guardedKeyword( 'priority',
      "Configure watchdog non-disruptive priorities",
      guard=pfcWatchdogNonDisruptivePriorityGuard )
nodeHardware = CliCommand.guardedKeyword( 'hardware',
      "Configure PFC priority through hardware",
      guard=hwMonitorGuard )
nodeMonitor = CliCommand.guardedKeyword( 'monitor',
      "Configure PFC priority for hardware monitoring",
      guard=hwMonitorGuard )
nodeWatchdogHardware = CliCommand.guardedKeyword( 'hardware',
      "Configure PFC priority through hardware",
      guard=pfcWatchdogNonDisruptivePriorityGuard )
nodeWatchdogNDPort = CliCommand.guardedKeyword( 'port',
      "Select behaviour for all ports", guard=pfcWatchdogForcedNDGuard )
nodeWatchogNDOnly = CliCommand.guardedKeyword( 'non-disruptive-only',
      "Set strict non-disruptive behaviour", guard=pfcWatchdogForcedNDGuard )
nodePfcPriorityList = CliCommand.guardedKeyword( 'priority',
      "Configure PFC priorities to be honored", guard=pfcSelectivePrioritiesGuard )
matcherWatchdogNDAction = CliMatcher.KeywordMatcher( 'action',
      helpdesc='Configure watchdog action for stuck non-disruptive transmit queues' )
nodeDlb = CliCommand.guardedKeyword( 'dlb', "Configure dynamic load balancing",
      guard=guardDlb )
nodeHardwareFwdAction = CliCommand.guardedKeyword( 'hardware',
      "Configure PFC priority through hardware",
      guard=pfcWatchdogNonDisruptiveFwdActionGuard )
matcherNonDisruptiveFwdAction = CliMatcher.KeywordMatcher( 'non-disruptive',
      helpdesc='Watchdog non-disruptive configuration' )

# --------------------------------------------------------------------------------
# priority-flow-control on
# legacy:
#     priority-flow-control mode on
# --------------------------------------------------------------------------------
class PriorityFlowControlOnCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control [ modeDeprecated ] on'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': nodePfc,
      'modeDeprecated': CliCommand.Node( matcher=matcherMode,
         deprecatedByCmd='priority-flow-control on' ),
      'on': 'Enable PFC',
   }

   handler = 'QosCliPfcHandler.pfcOn'
   noOrDefaultHandler = 'QosCliPfcHandler.pfcOff'


pfcModelet.addCommandClass( PriorityFlowControlOnCmd )

# --------------------------------------------------------------------------------
# ( no | default ) priority-flow-control
# --------------------------------------------------------------------------------
class PriorityFlowControlCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'priority-flow-control'
   data = {
      'priority-flow-control': nodePfc,
   }

   noOrDefaultHandler = 'QosCliPfcHandler.noPfc'


pfcModelet.addCommandClass( PriorityFlowControlCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog
# --------------------------------------------------------------------------------
class PriorityFlowControlPauseWatchdogCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
   }

   handler = 'QosCliPfcHandler.pfcWatchdogOnOff'
   noOrDefaultHandler = handler


pfcModelet.addCommandClass( PriorityFlowControlPauseWatchdogCmd )

# --------------------------------------------------------------------------------
# # [no|default] priority-flow-control all off
# legacy:
#     [no|default] priority-flow-control all mode off
# --------------------------------------------------------------------------------
class PriorityFlowControlAllOffCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control all [ modeDeprecated ] off'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': nodePfc,
      'all': 'Configure PFC for all interfaces',
      'modeDeprecated': CliCommand.Node(
         matcher=matcherMode,
         deprecatedByCmd='priority-flow-control all off' ),
      'off': 'Disable PFC',
   }

   handler = 'QosCliPfcHandler.pfcGlobalDisabled'
   noOrDefaultHandler = 'QosCliPfcHandler.pfcGlobalEnabled'


BasicCliModes.GlobalConfigMode.addCommandClass( PriorityFlowControlAllOffCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control priority PRIORITY ( drop | no-drop )
# --------------------------------------------------------------------------------
class PriorityFlowControlPriorityDropCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control priority PRIORITY ( drop | no-drop )'
   noOrDefaultSyntax = 'priority-flow-control priority PRIORITY ...'
   data = {
      'priority-flow-control': nodePfc,
      'priority': nodePfcPriority,
      'PRIORITY': matcherPriority,
      'drop': 'Disable PFC for a priority',
      'no-drop': 'Enable PFC for a priority',
   }

   handler = 'QosCliPfcHandler.pfcPriority'
   noOrDefaultHandler = 'QosCliPfcHandler.noPfcPriority'


pfcModelet.addCommandClass( PriorityFlowControlPriorityDropCmd )

# --------------------------------------------------------------------------------
# priority-flow-control pause watchdog port timer timeout TIMEOUT polling-interval
# ( auto | INTERVAL ) recovery-time ( 0 | RECOVERYTIME ) [ forced ]
# [no | default ] priority-flow-control pause watchdog port timer
# --------------------------------------------------------------------------------
class PriorityFlowControlWatchdogPollingCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port timer timeout TIMEOUT '
              'polling-interval ( auto | INTERVAL ) recovery-time '
              '( 0 | RECOVERYTIME ) [ forced ]' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port timer ...'
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'timer': 'Per port timer values',
      'timeout': matcherTimeout,
      'TIMEOUT': matcherTimeoutValue,
      'polling-interval': matcherPollingInterval,
      'auto': 'Auto polling interval value based on recovery-time and timeout',
      'INTERVAL': matcherPollingIntervalValue,
      'recovery-time': matcherRecoveryTime,
      '0': matcherZeroValue,
      'RECOVERYTIME': matcherRecoveryTimeValue,
      'forced': matcherForced,
   }

   handler = 'QosCliPfcHandler.configWatchdogPerPortTimers'
   noOrDefaultHandler = 'QosCliPfcHandler.noConfigWatchdogPerPortTimers'


pfcModelet.addCommandClass( PriorityFlowControlWatchdogPollingCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware monitor
# priority { PRIORITIES }
# --------------------------------------------------------------------------------
class PfCPauseWatchdogHwMonitorPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware monitor '
              'priority { PRIORITIES }' )
   noOrDefaultSyntax = ( 'priority-flow-control pause watchdog hardware monitor '
                         'priority ...' )
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'hardware': nodeHardware,
      'monitor': nodeMonitor,
      'priority': 'Configure PFC priority',
      'PRIORITIES': matcherPriority,
   }

   handler = 'QosCliPfcHandler.hwMonitor'
   noOrDefaultHandler = 'QosCliPfcHandler.noHwMonitor'


BasicCliModes.GlobalConfigMode.addCommandClass(
      PfCPauseWatchdogHwMonitorPriorityCmd )

# --------------------------------------------------------------------------------
# priority-flow-control pause watchdog default timeout TIMEOUT
# ( no | default ) priority-flow-control pause watchdog default timeout ...
# --------------------------------------------------------------------------------
class PfCPauseWatchdogDefaultTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog default timeout TIMEOUT'
   noOrDefaultSyntax = 'priority-flow-control pause watchdog default timeout ...'
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'default': matcherDefault,
      'timeout': matcherTimeout,
      'TIMEOUT': matcherTimeoutValue,
   }

   handler = 'QosCliPfcHandler.configWatchdogTimeout'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass( PfCPauseWatchdogDefaultTimeoutCmd )

# --------------------------------------------------------------------------------
# priority-flow-control pause watchdog default recovery-time
# RECOVERYTIME [ forced ]
# ( no | default ) priority-flow-control pause watchdog default recovery-time ...
# --------------------------------------------------------------------------------
class PfCPauseWatchdogDefaultRecoveryTimeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog default recovery-time '
              'RECOVERYTIME [ forced ]' )
   noOrDefaultSyntax = ( 'priority-flow-control pause watchdog default '
                         'recovery-time ...' )
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'default': matcherDefault,
      'recovery-time': matcherRecoveryTime,
      'RECOVERYTIME': matcherRecoveryTimeValue,
      'forced': matcherForced,
   }

   handler = 'QosCliPfcHandler.configWatchdogRecovery'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass(
      PfCPauseWatchdogDefaultRecoveryTimeCmd )

# --------------------------------------------------------------------------------
# priority-flow-control pause watchdog default polling-interval INTERVAL
# ( no | default ) priority-flow-control pause watchdog
# default polling-interval ...'
# --------------------------------------------------------------------------------
class PfCPauseWatchdogDefaultPollingIntervalCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog default '
              'polling-interval INTERVAL' )
   noOrDefaultSyntax = ( 'priority-flow-control pause watchdog '
                         'default polling-interval ...' )
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'default': matcherDefault,
      'polling-interval': matcherPollingInterval,
      'INTERVAL': matcherPollingIntervalValue,
   }

   handler = 'QosCliPfcHandler.configWatchdogPollingInterval'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass(
      PfCPauseWatchdogDefaultPollingIntervalCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog action
# ( errdisable | drop | notify-only )
# --------------------------------------------------------------------------------
class PfCPauseWatchdogActionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog action '
              ' ( errdisable | drop | notify-only ) ' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog action ...'
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'action': 'Watchdog action for stuck transmit queues',
      'errdisable': nodeErrdisable,
      'notify-only': nodeNotifyOnly,
      'drop': nodeDrop,
   }

   handler = 'QosCliPfcHandler.configWatchdogAction'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass( PfCPauseWatchdogActionCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog port action
# ( errdisable | drop | notify-only )
# --------------------------------------------------------------------------------
class PfCPauseWatchdogPortActionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port action '
              '( errdisable | drop | notify-only )' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port action ...'
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'action': 'Watchdog action for stuck transmit queues',
      'errdisable': nodeErrdisable,
      'drop': nodeDrop,
      'notify-only': nodeNotifyOnly,
   }

   handler = 'QosCliPfcHandler.configWatchdogPerPortAction'
   noOrDefaultHandler = 'QosCliPfcHandler.disableConfigWatchdogPerPortAction'


pfcModelet.addCommandClass( PfCPauseWatchdogPortActionCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware
# non-disruptive priority { PRIORITY }
# --------------------------------------------------------------------------------
class PfcPauseWatchdogHwNDPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware '
              'non-disruptive priority { PRIORITY }' )
   noOrDefaultSyntax = ( 'priority-flow-control pause watchdog hardware '
                         'non-disruptive priority ...' )
   data = {
      'priority-flow-control': nodePfc,
      'priority': nodeWatchdogPriorityList,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'hardware': nodeWatchdogHardware,
      'non-disruptive': nodeNonDisruptive,
      'PRIORITY': matcherPriority,
   }

   handler = 'QosCliPfcHandler.configWatchdogNonDisruptivePriority'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass( PfcPauseWatchdogHwNDPriorityCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware
# non-disruptive port non-disruptive-only
# --------------------------------------------------------------------------------
class PfCPauseWatchdogHwNonDisruptivePortOnlyCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware '
              'non-disruptive port non-disruptive-only' )
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'hardware': nodeWatchdogHardware,
      'non-disruptive': nodeNonDisruptive,
      'port': nodeWatchdogNDPort,
      'non-disruptive-only': nodeWatchogNDOnly,
   }

   handler = 'QosCliPfcHandler.enableWatchdogNonDisruptiveForced'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass(
      PfCPauseWatchdogHwNonDisruptivePortOnlyCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control priority default { PRIORITIES }
# --------------------------------------------------------------------------------
class PfCPriorityDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control priority default { PRIORITIES }'
   noOrDefaultSyntax = 'priority-flow-control priority default ...'
   data = {
      'priority-flow-control': nodePfc,
      'priority': nodePfcPriorityList,
      'default': matcherDefault,
      'PRIORITIES': CliCommand.Node( matcher=matcherPriority, maxMatches=8 ),
   }

   handler = 'QosCliPfcHandler.configPfcPriorityList'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass( PfCPriorityDefaultCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware
# non-disruptive action forward
# --------------------------------------------------------------------------------
class PfCPauseWdHwNonDisruptiveActionForwardCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware non-disruptive '
              'action forward' )
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'hardware': nodeHardwareFwdAction,
      'non-disruptive': matcherNonDisruptiveFwdAction,
      'action': matcherWatchdogNDAction,
      'forward': 'Forward traffic on the stuck queue',
   }

   handler = 'QosCliPfcHandler.configWatchdogNonDisruptiveAction'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass(
      PfCPauseWdHwNonDisruptiveActionForwardCmd )

# --------------------------------------------------------------------------------
# Priority-flow-control packet-buffer commands
# --------------------------------------------------------------------------------

# --------------------------------------------------------------------------------
# priority-flow-control packet-buffer allocation lossless LOSSLESS percent
#    headroom HEADROOM percent
# ( no | default ) priority-flow-control packet-buffer allocation
# in Config Mode
# --------------------------------------------------------------------------------

class PfCPacketBufferAllocationCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control packet-buffer allocation '
              'lossless LOSSLESS percent headroom HEADROOM percent' )
   noOrDefaultSyntax = 'priority-flow-control packet-buffer allocation ...'
   data = {
      'priority-flow-control': nodePfc,
      'packet-buffer': nodePacketBuffer,
      'allocation': 'Configure global packet buffer pools allocation',
      'lossless': 'Configure lossless pool allocation',
      'LOSSLESS': matcherIntegerPercent,
      'percent': CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher( 'percent', 'Percent' ),
            noResult=True ),
      'headroom': 'Configure headroom pool allocation',
      'HEADROOM': matcherIntegerPercent,
   }

   handler = 'QosCliPfc.configPacketBufferAllocation'
   noOrDefaultHandler = 'QosCliPfc.noOrDefaultPacketBufferAllocation'

BasicCliModes.GlobalConfigMode.addCommandClass( PfCPacketBufferAllocationCmd )

# --------------------------------------------------------------------------------
# QosProfile Mode
# --------------------------------------------------------------------------------

# --------------------------------------------------------------------------------
# PFC commands in QosProfile mode
# --------------------------------------------------------------------------------
# priority-flow-control on
# legacy:
#     priority-flow-control mode on
# --------------------------------------------------------------------------------

class PfCOnProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control [ modeDeprecated ] on'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': nodePfc,
      'modeDeprecated': CliCommand.Node(
         matcher=matcherMode,
         deprecatedByCmd='priority-flow-control on' ),
      'on': 'Enable PFC',
   }

   handler = 'QosCliPfcHandler.qosProfilePfcOn'
   noOrDefaultHandler = 'QosCliPfcHandler.qosProfilePfcOff'


QosProfileMode.addCommandClass( PfCOnProfileCmd )

# --------------------------------------------------------------------------------
# ( no | default ) priority-flow-control in QosProfile Mode
# --------------------------------------------------------------------------------
class PfCProfileCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'priority-flow-control'
   data = {
      'priority-flow-control': nodePfc,
      }

   noOrDefaultHandler = 'QosCliPfcHandler.qosProfileNoPfc'


QosProfileMode.addCommandClass( PfCProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog in QosProfile Mode
# --------------------------------------------------------------------------------
class PfCPauseWatchdogProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog'
   noOrDefaultSyntax = syntax
   data = {
         'priority-flow-control': nodePfc,
         'pause': nodePause,
         'watchdog': matcherWatchdog,
      }

   handler = 'QosCliPfcHandler.qosProfilePfcWatchdogOnOff'
   noOrDefaultHandler = handler


QosProfileMode.addCommandClass( PfCPauseWatchdogProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control priority PRIORITY
# ( drop | no-drop [ dlb ] ) in QosProfile Mode
# --------------------------------------------------------------------------------
class PfCPriorityDropProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control priority PRIORITY ( drop | no-drop [ dlb ] )'
   noOrDefaultSyntax = 'priority-flow-control priority PRIORITY ...'
   data = {
      'priority-flow-control': nodePfc,
      'priority': nodePfcPriority,
      'PRIORITY': matcherPriority,
      'drop': 'Disable PFC for a priority',
      'no-drop': 'Enable PFC for a priority',
      'dlb': nodeDlb,
   }

   handler = 'QosCliPfcHandler.qosProfilePfcPriority'
   noOrDefaultHandler = 'QosCliPfcHandler.qosProfileNoPfcPriority'


QosProfileMode.addCommandClass( PfCPriorityDropProfileCmd )

# --------------------------------------------------------------------------------
# priority-flow-control pause watchdog port timer timeout TIMEOUT polling-interval
# ( auto | INTERVAL ) recovery-time ( 0 | RECOVERYTIME ) [ forced ]
# [no | default ] priority-flow-control pause watchdog port timer - QosProfile Mode
# --------------------------------------------------------------------------------
class PfCWatchdogPollingProfileCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port timer timeout TIMEOUT '
              'polling-interval ( auto | INTERVAL ) recovery-time '
              '( 0 | RECOVERYTIME ) [ forced ]' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port timer ...'
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'timer': 'Per port timer values',
      'timeout': matcherTimeout,
      'TIMEOUT': matcherTimeoutValue,
      'polling-interval': matcherPollingInterval,
      'auto': 'Auto polling interval value based on recovery-time and timeout',
      'INTERVAL': matcherPollingIntervalValue,
      'recovery-time': matcherRecoveryTime,
      '0': matcherZeroValue,
      'RECOVERYTIME': matcherRecoveryTimeValue,
      'forced': matcherForced,
   }

   handler = 'QosCliPfcHandler.qosProfilePortWatchdogTimer'
   noOrDefaultHandler = 'QosCliPfcHandler.qosProfileNoPortWatchdogTimer'


QosProfileMode.addCommandClass( PfCWatchdogPollingProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog port action
# ( errdisable | drop | notify-only ) in QosProfile Mode
# --------------------------------------------------------------------------------
class PfCPauseWdPortActionProfileCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port action '
              '( errdisable | drop | notify-only )' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port action ...'
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'action': 'Watchdog action for stuck transmit queues',
      'errdisable': nodeErrdisable,
      'drop': nodeDrop,
      'notify-only': nodeNotifyOnly,
   }

   handler = 'QosCliPfcHandler.qosProfilePortWatchdogAction'
   noOrDefaultHandler = 'QosCliPfcHandler.qosProfileNoPortWatchdogAction'


QosProfileMode.addCommandClass( PfCPauseWdPortActionProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog override action drop
# in QosProfile Mode
# --------------------------------------------------------------------------------
class PfCPauseWdOverrideActionDropCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog override action drop'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': nodePfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'override': 'Override configured watchdog actions',
      'action': 'Watchdog action for stuck transmit queues',
      'drop': nodeDrop,
   }

   handler = 'QosCliPfcHandler.enableWatchdogOverrideAction'
   noOrDefaultHandler = handler


BasicCliModes.GlobalConfigMode.addCommandClass( PfCPauseWdOverrideActionDropCmd )

#
# --------------------------------------------------------------------------------
# SECTION: Priority Flow Control (PFC) Show Commands
# --------------------------------------------------------------------------------
# This section contains registrations of the priority flow control show
# commands with the CLI infrastructure.
#
# These commands come in synonymous pairs. Commands within a pair share
# the same CLI value function and model, thus producing the same output.
# The pairs have been delineated accordingly in the list below.
#
# Commands:
#
# 1.  show interfaces Fabric PFC
# 2.  show PFC interfaces Fabric status
#
# 3.  show interfaces [ <range> ] PFC status
# 4.  show PFC [ interfaces <range> ] status
#
# 5.  show interfaces [ <range> ] PFC counters
# 6.  show PFC [ interfaces <range> ] counters
#
# 7.  show interfaces [ <range> ] PFC counters detail
# 8.  show PFC [ interfaces <range> ] counters detail
#
# 9.  show interfaces [ <range> ] PFC counters watchdog
# 10. show PFC [ interfaces <range> ] counters watchdog
#
# 11. show interfaces [ <range> ] PFC
# 12. show PFC [ interfaces <range> ]
#
# where PFC = 'priority-flow-control'
#
# NOTE: Registration of any priority flow control show commands with the
#       CLI Tech-Support infrastructure is captured at the end of this
#       section.
# --------------------------------------------------------------------------------
#

#
# CLI guards.
#
def pfcBufferGuard( mode, token ):
   """Guard function for the CLI token: 'buffer'"""

   if pfcStatus.pfcBuffersCounterSupported:
      return None

   return CliParser.guardNotThisPlatform

def pfcShowDetailGuard( mode, token ):
   """Guard function for the CLI token: 'detail'"""

   if pfcStatus.pfcClassFramesCounterSupported:
      return None

   return CliParser.guardNotThisPlatform

def pfcShowHistoryGuard( mode, token ):
   """Guard function for the CLI token: 'history'"""

   if pfcStatus.pfcHistoryCounterSupported:
      return None

   return CliParser.guardNotThisPlatform

def pfcShowWatchdogGuard( mode, token ):
   """Guard function for the CLI token: 'watchdog'"""

   if pfcStatus.watchdogSupported:
      return None

   return CliParser.guardNotThisPlatform

# See BUG412221
def pfcWatchdogDropCountersGuard( mode, token ):
   """Guard function for the CLI token: 'drop'"""
   if pfcStatus.watchdogSupported and pfcStatus.watchdogDropCountersSupported:
      return None

   return CliParser.guardNotThisPlatform

# Aliases to extant CLI guards.
pfcShowPfcGuard = pfcKeywordGuard
pfcShowPfcPacketBufferGuard = pfcPacketBufferGuard

#
# CLI tokens used in the priority flow control show commands.
#

pfcShowKw = CliCommand.guardedKeyword( 'priority-flow-control',
                                       helpdesc="Show interface PFC information",
                                       guard=pfcShowPfcGuard )

pfcClearKw = CliCommand.guardedKeyword( 'priority-flow-control',
                                        helpdesc="Clear PFC information",
                                        guard=pfcKeywordGuard )

statusPfcKw = CliMatcher.KeywordMatcher( 'status',
                                         helpdesc="Show interface PFC status" )

countersPfcKw = CliMatcher.KeywordMatcher( 'counters',
                                           helpdesc="Show interface PFC counters" )

clearCountersPfcKw = CliMatcher.KeywordMatcher( 'counters',
                                                helpdesc="Clear PFC counters" )

detailPfcKw = CliCommand.guardedKeyword( 'detail',
                                         "Detailed view",
                                         pfcShowDetailGuard )

historyPfcKw = CliCommand.guardedKeyword( 'history',
                                          "PFC history counters",
                                          pfcShowHistoryGuard )

watchdogCountersKw = CliCommand.guardedKeyword( 'watchdog',
                                                "PFC watchdog counters",
                                                pfcShowWatchdogGuard )

watchdogDropCountersKw = CliCommand.guardedKeyword( 'drop',
                                                    "PFC watchdog drop counters",
                                                    pfcWatchdogDropCountersGuard )

interfacePfcKw = CliMatcher.KeywordMatcher( 'interfaces',
                                        helpdesc="Show PFC information for a "
                                        "specific interface" )

bufferPfcKw = CliCommand.guardedKeyword( 'buffer',
                                         'PFC buffer counters',
                                         pfcBufferGuard )

def registerPfcIntfCmd( extraSyntax, extraData, cmdHandler, pfcCliModel,
                        deprecateIntfStyle=False ):
   # This registers the following two commands:
   # 1. show interface [ INTF ] priority-flow-control XXX
   # 2. show priority-flow-control [ interface INTF ] XXX
   pfcData = { 'priority-flow-control': pfcShowKw }
   pfcData.update( extraData )

   # register the 'show interface' command
   class ShowIntfPfcCmd( IntfCli.ShowIntfCommand ):
      syntax = 'show interfaces priority-flow-control ' + extraSyntax
      data = pfcData
      handler = cmdHandler
      moduleAtEnd = True
      cliModel = pfcCliModel

   if not deprecateIntfStyle:
      BasicCli.addShowCommandClass( ShowIntfPfcCmd )

   # register the 'show priority-flow-control' command
   class ShowPfcCmd( ShowCommand.ShowCliCommandClass ):
      syntax = "show priority-flow-control [ interfaces INTF ] " + extraSyntax
      data = {
         "priority-flow-control": pfcShowKw,
         "interfaces": interfacePfcKw,
         "INTF": IntfCli.Intf.rangeMatcher,
      }
      data.update( extraData )
      cliModel = pfcCliModel
      handler = cmdHandler

   BasicCli.addShowCommandClass( ShowPfcCmd )

#
# Register command #1:
# show interfaces Fabric priority-flow-control
#
class ShowFabricPfc( ShowCommand.ShowCliCommandClass ):
   syntax = "show interfaces fabric priority-flow-control"
   data = {
      'interfaces': IntfCli.interfacesShowKw,
      'fabric': FabricConfigCli.ifConfigFabricKw,
      'priority-flow-control': pfcShowKw
      }
   cliModel = PfcFabricIntfStatusModel
   handler = 'QosCliPfcHandler.showPfcFabricStatus'


BasicCli.addShowCommandClass( ShowFabricPfc )

#
# Register command #2:
# show priority-flow-control interfaces Fabric status
#
class ShowPfcFabricStatus( ShowCommand.ShowCliCommandClass ):
   syntax = "show priority-flow-control interfaces fabric status"
   data = {
      'priority-flow-control': pfcShowKw,
      'interfaces': interfacePfcKw,
      'fabric': FabricConfigCli.ifConfigFabricKw,
      'status': statusPfcKw
      }
   cliModel = PfcFabricIntfStatusModel
   handler = 'QosCliPfcHandler.showPfcFabricStatus'


BasicCli.addShowCommandClass( ShowPfcFabricStatus )

#
# Register command #3:
# show interfaces [ <range> ] priority-flow-control status
# show priority-flow-control [ interfaces <range> ] status
registerPfcIntfCmd( "status",
                    dict( status=statusPfcKw ),
                    'QosCliPfcHandler.showPfcInterfaceStatus',
                    PfcIntfRangeStatusModel )

#
# Register command #4:
# show interfaces [ <range> ] priority-flow-control counters
# show priority-flow-control [ interfaces <range> ] counters
#
registerPfcIntfCmd( 'counters',
                    dict( counters=countersPfcKw ),
                    'QosCliPfcHandler.showPfcInterfaceCounters',
                    PfcIntfRangeCountersModel )

#
# Register command #5:
# show interfaces [ <range> ] priority-flow-control counters detail
# show priority-flow-control [ interfaces <range> ] counters detail
#
registerPfcIntfCmd( 'counters detail',
                    dict( counters=countersPfcKw,
                          detail=detailPfcKw ),
                    'QosCliPfcHandler.showPfcInterfaceDetailedCounters',
                    PfcIntfRangeDetailedCountersModel )

#
# Register command #6:
# show priority-flow-control [ interfaces <range> ] counters history
#
registerPfcIntfCmd( 'counters history',
                    dict( counters=countersPfcKw,
                          history=historyPfcKw ),
                    'QosCliPfcHandler.showPfcInterfaceHistoryCounters',
                    PfcIntfRangeHistoryCountersModel,
                    deprecateIntfStyle=True )

#
# Register command #7:
# show interfaces [ <range> ] priority-flow-control counters watchdog
# show priority-flow-control [ interfaces <range> ] counters watchdog
#
registerPfcIntfCmd( 'counters watchdog',
                    dict( counters=IntfCli.countersKw,
                          watchdog=watchdogCountersKw ),
                    'QosCliPfcHandler.showPfcWatchdogInterfaceCounters',
                    PfcWatchdogIntfRangeCountersModel )

#
# Register command #8:
# show interfaces [ <range> ] priority-flow-control
# show priority-flow-control [ interfaces <range> ]
#
registerPfcIntfCmd( '', {}, showPfcInterface, PfcIntfRangeModel )

#
# Register command #9:
# show interfaces [ <range> ] priority-flow-control counters watchdog drop
# show priority-flow-control [ interfaces <range> ] counters watchdog drop
#
registerPfcIntfCmd( 'counters watchdog drop',
                    dict( counters=countersPfcKw,
                          watchdog=watchdogCountersKw,
                          drop=watchdogDropCountersKw ),
                    'QosCliPfcHandler.showPfcWatchdogInterfaceDropCounters',
                    PfcWatchdogIntfRangeDropCountersModel )

#
# Register command #10:
# clear priority-flow-control counters watchdog [ drop ]
#
class ClearPfcCountersWatchdog( CliCommand.CliCommandClass ):
   syntax = "clear priority-flow-control counters watchdog [ drop ]"
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'priority-flow-control': pfcClearKw,
      'counters': clearCountersPfcKw,
      'watchdog': watchdogCountersKw,
      'drop': watchdogDropCountersKw
   }
   handler = 'QosCliPfcHandler.clearPfcCountersWatchdogHandler'


BasicCli.EnableMode.addCommandClass( ClearPfcCountersWatchdog )

#
# Register command #11:
# clear priority-flow-control counters history
#
class ClearPfcCountersHistory( CliCommand.CliCommandClass ):
   syntax = "clear priority-flow-control counters history"
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'priority-flow-control': pfcClearKw,
      'counters': clearCountersPfcKw,
      'history': historyPfcKw,
   }
   handler = 'QosCliPfcHandler.clearPfcCountersHistoryHandler'


BasicCli.EnableMode.addCommandClass( ClearPfcCountersHistory )

#
# Register command #12:
# show interfaces [ <range> ] priority-flow-control buffer counters
# show priority-flow-control [ interfaces <range> ] buffer counters
#
registerPfcIntfCmd( 'buffer counters',
                    dict( buffer=bufferPfcKw,
                          counters='Show PFC buffer counters' ),
                    'QosCliPfc.showPfcInterfaceBufferCounters',
                    'QosCliModel.PfcIntfRangeBufferCountersModel' )

#
# Register command #13:
# clear priority-flow-control buffer counters
#
class ClearPfcBufferCounters( CliCommand.CliCommandClass ):
   syntax = "clear priority-flow-control buffer counters"
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'priority-flow-control': pfcClearKw,
      'buffer': bufferPfcKw,
      'counters': 'Clear PFC buffer counters'
   }
   handler = 'QosCliPfc.clearPfcBufferCounters'


BasicCli.EnableMode.addCommandClass( ClearPfcBufferCounters )

#
# Utility function to obtain a set of priority-flow-control packet buffer profile
# names for command completion.
#
def pfcPacketBufferProfileNames( mode ): # pylint: disable=unused-variable
   """Returns packet buffer profile names."""
   profileStatus = pfcPacketBufferProfileStatus.fadtProfileStatus
   # Return a copy to avoid returning temporary keys_view.
   return list( profileStatus.keys() )

#
# Register command:
# show priority-flow-control packet-buffer profile [ { PROFILES } ]
#
class ShowPfcPacketBufferProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show priority-flow-control packet-buffer profile [ { PROFILES } ]'
   data = {
      'priority-flow-control': pfcShowKw,
      'packet-buffer': nodePacketBuffer,
      'profile': 'Global and profile buffer allocations',
      'PROFILES': CliMatcher.DynamicNameMatcher(
            pfcPacketBufferProfileNames,
            helpdesc='Profile names to list or blank for all profiles' ), }
   cliModel = 'QosCliModel.PfcPacketBufferProfileRangeModel'
   handler = 'QosCliPfc.showPfcPacketBufferProfileRange'

BasicCli.addShowCommandClass( ShowPfcPacketBufferProfile )

#
# Registrations under Tech-Support.
#
TechSupportCli.registerShowTechSupportCmd(
   '2010-01-01 00:07:00',
   cmds=[ 'show priority-flow-control' ],
   cmdsGuard=pfcSupported )

# --------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# --------------------------------------------------------------------------------
@Plugins.plugin()
def Plugin( entityManager ):
   global qosInputConfig, pfcStatus, qosHwStatus, qosConfig, dcbxConfig, \
      pfcConfig, qosSliceHwStatus, qosProfileConfigDir, pfcSliceHwStatus, \
      pfcWdSliceHwStatus, pfcHistoryStatus, pfcHistoryCounterStatus, \
      pfcPacketBufferProfileStatus

   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosProfileConfigDir = ConfigMount.mount( entityManager, "qos/profile",
                                            "Qos::QosProfileConfigDir", "w" )
   pfcStatus = LazyMount.mount( entityManager, "dcb/pfc/status", "Pfc::Status", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = Cell.path( 'qos/hardware/status/slice' )
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   pfcSliceHwStatusDirPath = Cell.path( 'qos/hardware/status/pfc/slice' )
   pfcSliceHwStatus = LazyMount.mount( entityManager, pfcSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   pfcWdSliceHwStatusDirPath = Cell.path( 'qos/hardware/status/pfcwd/slice' )
   pfcWdSliceHwStatus = LazyMount.mount( entityManager, pfcWdSliceHwStatusDirPath,
                                         "Tac::Dir", "ri" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   dcbxConfig = LazyMount.mount(
      entityManager, "dcb/dcbx/config", "Dcbx::Config", "r" )
   pfcConfig = LazyMount.mount( entityManager, "dcb/pfc/config", "Pfc::Config", "r" )
   pfcHistoryStatus = LazyMount.mount( entityManager, "dcb/pfc/historyStatus",
                                       "Pfc::HistoryStatus", "r" )
   pfcPacketBufferProfileStatus = LazyMount.mount(
         entityManager,
         "dcb/pfc/packetBufferProfile/status",
         "Pfc::PacketBufferProfileStatus",
         "r" )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   mountPath = "interface/status/counter/pfcHistory"
   pfcHistoryCounterStatus = shmemEm.doMount( mountPath,
                                              "Qos::PfcHistoryStatus",
                                              readerInfo )

   IntfCli.Intf.registerDependentClass( PfcIntf )
