#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import Plugins
import ShowCommand
import Tac
import Tracing
from BasicCliModes import GlobalConfigMode
from CliMode.Qos import ( QosSchedulingModeBase, QosSchedulingIntfModeBase,
                          QosSchedulingIntfGroupModeBase,
                          QosSchedulingPolicyModeBase )
from CliPlugin import ( EthIntfCli, LagIntfCli, SubIntfCli, SwitchIntfCli, QosCli,
                        IntfCli, QosCliServicePolicy )
from CliPlugin.QosCliIntfTypes import ethOrLagIntfPrefixes
from CliPlugin.QosCli import nodeQosForShow
from CliPlugin.QosCliCommon import ( matcherSize, matcherAdjustment, matcherPlus,
                                     matcherMinus, matcherBytes, matcherIntf,
                                     QosProfileMode,
                                     IntfTxQueueConfigMode,
                                     matcherTxQueueIdForName,
                                     matcherTxQueueConfig,
                                     matcherSchProfileToken,
                                     nodeSchedulerToken,
                                     nodeSchProfileResponsive )
from CliPlugin.QosCliModel import ( QosAllSchedulingGroupModel,
                                    QosAllSchedulingHierarchyModel )
from CliPlugin.QosCliShape import SetShapeRateCmd
from CliPlugin.VirtualIntfRule import IntfMatcher

__defaultTraceHandle__ = Tracing.Handle( 'QosCliScheduling' )
t0 = Tracing.trace0

# -----------------------------------------------------------------------------------
# Variables for Qos Scheduling associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
lagInputConfig = None
qosConfig = None
qosStatus = None
qosHwStatus = None
qosSchedulerConfig = None
subIntfHwStatus = None

# -----------------------------------------------------------------------------------
# Guards
# -----------------------------------------------------------------------------------
def guardQosScheduling( mode, token ):
   if not subIntfHwStatus.subIntfSchedulingGroupSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardPerPortCompensation( mode, token ):
   if qosHwStatus.hwInitialized and \
      qosHwStatus.perPortSchedulerCompensationSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNumTxqsPerSubIntf( mode, token ):
   if not subIntfHwStatus.subIntfNumTxqsConfigSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardSubintfScheduling( mode, token ):
   if qosHwStatus.subIntfFairSchedulingSupported:
      return None
   return CliParser.guardNotThisPlatform


# -----------------------------------------------------------------------------------
# Matchers
# -----------------------------------------------------------------------------------
matcherSchedulingGroupName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosSchedulerConfig.qosSchedulerIntfConfig[
         mode.intf ].name,
      helpdesc='Scheduling group name',
      pattern=r'(?!scheduling-group$)[A-Za-z0-9_:{}\[\]-]+' )
matcherSchedulingPolicyName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosSchedulerConfig.policy, helpdesc='Scheduling policy name',
      pattern=r'(?!scheduling-policy$)[A-Za-z0-9_]+' )
matcherTxScheduler = CliMatcher.KeywordMatcher( 'tx-scheduler',
      helpdesc='Configure scheduler parameters' )
matcherPacket = CliMatcher.KeywordMatcher( 'packet',
      helpdesc='Configure packet parameters' )
matcherBytesAdjustmentMatcher = CliMatcher.IntegerMatcher( 1, 100,
      helpdesc='Adjustment value( in bytes )' )
matcherTxQueueName = CliMatcher.StringMatcher(
      helpdesc="transmit queue name" )

# -----------------------------------------------------------------------------------
# Tokens
# -----------------------------------------------------------------------------------
nodeScheduling = CliCommand.guardedKeyword( 'scheduling',
      helpdesc='Configure qos scheduling parameters',
      guard=guardQosScheduling )
nodeShowScheduling = CliCommand.guardedKeyword( 'scheduling',
      helpdesc='Show QoS scheduling parameters',
      guard=guardQosScheduling )
nodeTxScheduler = CliCommand.Node( matcher=matcherTxScheduler,
                                   guard=guardPerPortCompensation )

# -----------------------------------------------------------------------------------
# qos scheduling mode
# -----------------------------------------------------------------------------------
class QosSchedulingMode( QosSchedulingModeBase, BasicCli.ConfigModeBase ):
   name = 'QoS scheduling'

   def __init__( self, parent, session ):
      QosSchedulingModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def clear( self, args ): # pylint: disable=arguments-renamed
      qosSchedulerConfig.qosSchedulerIntfConfig.clear()
      qosSchedulerConfig.policy.clear()

# -----------------------------------------------------------------------------------
# qos scheduling interface mode
# -----------------------------------------------------------------------------------
class QosSchedulingIntfMode( QosSchedulingIntfModeBase, BasicCli.ConfigModeBase ):
   name = 'QoS interface scheduling'

   def __init__( self, parent, session, intfId ):
      self.intfId_ = intfId
      self.intfName_ = intfId.stringValue
      param = self.intfName_
      QosSchedulingIntfModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def abort( self ):
      self.session_.gotoParentMode()

   def intfId( self ):
      return self.intfId_


ethOrLagMatcher = IntfMatcher()
ethOrLagMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
ethOrLagMatcher |= SwitchIntfCli.SwitchIntf.matcher
ethOrLagMatcher |= LagIntfCli.EthLagIntf.matcher

class QosSchedulingIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTFNAME'
   noOrDefaultSyntax = syntax
   data = {
      'interface': 'Configure qos scheduling parameters for interface',
      'INTFNAME': ethOrLagMatcher,
   }
   handler = "QosCliSchedulingHandler.goToQosSchedulingIntfMode"
   noOrDefaultHandler = "QosCliSchedulingHandler.deleteQosSchedulingIntf"


QosSchedulingMode.addCommandClass( QosSchedulingIntfConfigCmd )

# -----------------------------------------------------------------------------------
# qos scheduling group mode
# -----------------------------------------------------------------------------------
class QosSchedulingIntfGroupMode( QosSchedulingIntfGroupModeBase,
                                  BasicCli.ConfigModeBase ):
   name = 'QoS scheduling group'

   def __init__( self, parent, session, context ):
      self.qosSchedulingIntfGroupModeContext = context
      self.intfId_ = context.intfId()
      self.intfName_ = context.intfName()
      self.groupName_ = context.groupName()
      param = ( self.intfName_, self.groupName_ )
      QosSchedulingIntfGroupModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.currentEntry_ = None
      self.qosSchedulingIntfGroupModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.qosSchedulingIntfGroupModeContext is None:
         t0( 'QosSchedulingIntfGroupMode.commitContext has no context' )
         return

      context = self.qosSchedulingIntfGroupModeContext
      self.qosSchedulingIntfGroupModeContext = None
      context.commit()

class QosSchedulingIntfGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'scheduling group GROUPNAME'
   noOrDefaultSyntax = syntax
   data = {
      'scheduling': nodeScheduling,
      'group': 'Configure qos scheduling group for interface',
      'GROUPNAME': matcherSchedulingGroupName,
   }

   handler = "QosCliSchedulingHandler.qosSchedulingIntfGroupConfigCmdHandler"
   noOrDefaultHandler = \
      "QosCliSchedulingHandler.qosSchedulingIntfGroupConfigCmdNoOrDefaultHandler"


QosSchedulingIntfMode.addCommandClass( QosSchedulingIntfGroupConfigCmd )

class QosSchedulingPolicyNameCmd( CliCommand.CliCommandClass ):
   syntax = 'policy POLICYNAME'
   noOrDefaultSyntax = syntax
   data = {
      'policy': 'Configure qos scheduling policy',
      'POLICYNAME': matcherSchedulingPolicyName,
   }

   handler = "QosCliSchedulingHandler.qosSchedulingPolicyNameCmdHandler"
   noOrDefaultHandler = \
      "QosCliSchedulingHandler.qosSchedulingPolicyNameCmdNoOrDefaultHandler"


QosSchedulingIntfGroupMode.addCommandClass( QosSchedulingPolicyNameCmd )

class SubIntfWrapperMatcher( CliMatcher.WrapperMatcher ):
   def __init__( self, matcher ):
      CliMatcher.WrapperMatcher.__init__( self, matcher )

   def valueFunction( self, context ):
      f = self.matcher_.valueFunction( context )

      def g( mode, name ):
         intf = f( mode, name )
         if not intf.isSubIntf():
            t0( 'not subinterface' )
            raise CliParser.InvalidInputError()
         parentStatus = intf.parentStatus()
         if parentStatus:
            parentIntfId = intf.parentStatus().getRawAttribute( 'intfId' )
         else:
            parentIntfId = Tac.Value( 'Arnet::IntfId', intf.name.split( '.' )[ 0 ] )
         if parentIntfId != mode.intfId_:
            t0( 'parent name not match', parentIntfId, '!=', mode.intfId_ )
            raise CliParser.InvalidInputError()
         return intf
      return g


ethOrLagSubIntfMatcher = IntfMatcher()
ethOrLagSubIntfMatcher |= SubIntfCli.subMatcher
ethOrLagSubIntfMatcher |= LagIntfCli.subMatcher
matcherSubIntfName = SubIntfWrapperMatcher( ethOrLagSubIntfMatcher )

class QosSchedulingGroupMemberCmd( CliCommand.CliCommandClass ):
   syntax = 'members { INTFNAME }'
   noOrDefaultSyntax = 'members [ { INTFNAME } ]'
   data = {
      'members': 'Configure qos scheduling group members',
      'INTFNAME': matcherSubIntfName,
   }

   handler = "QosCliSchedulingHandler.qosSchedulingGroupMemberCmdHandler"
   noOrDefaultHandler = \
      "QosCliSchedulingHandler.qosSchedulingGroupMemberCmdNoOrDefaultHandler"


QosSchedulingIntfGroupMode.addCommandClass( QosSchedulingGroupMemberCmd )

# -----------------------------------------------------------------------------------
# qos scheduling policy mode
# -----------------------------------------------------------------------------------
class QosSchedulingPolicyMode( QosSchedulingPolicyModeBase,
                               BasicCli.ConfigModeBase ):
   name = 'QoS scheduling policy'

   def __init__( self, parent, session, context ):
      self.qosSchedulingPolicyModeContext = context
      self.policyName_ = context.policyName()
      param = ( self.policyName_ )
      QosSchedulingPolicyModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.currentEntry_ = None
      self.qosSchedulingPolicyModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.qosSchedulingPolicyModeContext is None:
         t0( 'QosSchedulingPolicyMode.commitContext has no context' )
         return

      context = self.qosSchedulingPolicyModeContext
      self.qosSchedulingPolicyModeContext = None
      context.commit()

class QosSchedulingPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'scheduling policy POLICYNAME'
   noOrDefaultSyntax = syntax
   data = {
      'scheduling': nodeScheduling,
      'policy': 'Configure qos scheduling policy',
      'POLICYNAME': matcherSchedulingPolicyName,
   }
   handler = "QosCliSchedulingHandler.gotoQosSchedulingPolicyMode"
   noOrDefaultHandler = "QosCliSchedulingHandler.deleteQosSchedulingPolicy"


QosSchedulingMode.addCommandClass( QosSchedulingPolicyConfigCmd )
QosSchedulingPolicyMode.addCommandClass( SetShapeRateCmd )

# --------------------------------------------------------------------------------
# qos scheduling
# --------------------------------------------------------------------------------
class QosSchedulingConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'qos scheduling'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'scheduling': nodeScheduling,
   }
   handler = "QosCliSchedulingHandler.goToQosSchedulingMode"
   noOrDefaultHandler = QosSchedulingMode.clear


GlobalConfigMode.addCommandClass( QosSchedulingConfigCmd )

def getAllSchedulingGroups( mode ):
   groups = []
   for intfConfig in qosStatus.intfStatus.values():
      for group in intfConfig.schedulerGroupStatus:
         if group not in groups:
            groups.append( group )
   return groups

# --------------------------------------------------------------------------------
# show qos scheduling group [ GROUP ] [ INTFS ]
# --------------------------------------------------------------------------------
class QosScheduleingGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos scheduling group [ GROUP ] [ INTFS ]'
   data = {
      'qos': nodeQosForShow,
      'scheduling': nodeShowScheduling,
      'group': 'Show QoS scheduling group parameters',
      'GROUP': CliMatcher.DynamicNameMatcher(
         getAllSchedulingGroups,
         pattern=r'(?!scheduling-group$)[A-Za-z0-9_:{}\[\]-]+',
         helpname='group-name',
         helpdesc='Scheduling group name',
         priority=CliParser.PRIO_LOW ),
      'INTFS': matcherIntf,
   }

   handler = "QosCliSchedulingHandler.showQosSchedulingGroup"
   cliModel = QosAllSchedulingGroupModel


BasicCli.addShowCommandClass( QosScheduleingGroupCmd )

# --------------------------------------------------------------------------------
# show qos scheduling hierarchy [ INTFS ]
# --------------------------------------------------------------------------------
class QosSchedulingHierarchyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos scheduling hierarchy [ INTFS ]'
   data = {
      'qos': nodeQosForShow,
      'scheduling': nodeShowScheduling,
      'hierarchy': 'Show QoS scheduling hierarchy parameters',
      'INTFS': matcherIntf,
   }
   handler = "QosCliSchedulingHandler.showQosSchedulingHierarchy"
   cliModel = QosAllSchedulingHierarchyModel


BasicCli.addShowCommandClass( QosSchedulingHierarchyCmd )

# --------------------------------------------------
# Scheduler Compensation Modelet
# --------------------------------------------------
class QosModeletSchedulerCompensation( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( QosModeletSchedulerCompensation )

# ------------------------------------------------------------------------------
# [ no|default ] tx-scheduler packet size adjustment ( plus|minus ) BYTES bytes
# ------------------------------------------------------------------------------
class SetSchedulerCompensationCmd( CliCommand.CliCommandClass ):
   syntax = 'tx-scheduler packet size adjustment ( plus | minus ) BYTES bytes'
   noOrDefaultSyntax = 'tx-scheduler packet size adjustment ...'
   data = {
         'tx-scheduler': nodeTxScheduler,
         'packet': matcherPacket,
         'size': matcherSize,
         'adjustment': matcherAdjustment,
         'plus': matcherPlus,
         'minus': matcherMinus,
         'BYTES': matcherBytesAdjustmentMatcher,
         'bytes': matcherBytes,
   }

   handler = "QosCliSchedulingHandler.setSchedulerCompensation"
   noOrDefaultHandler = "QosCliSchedulingHandler.noSchedulerCompensation"


QosModeletSchedulerCompensation.addCommandClass( SetSchedulerCompensationCmd )
QosProfileMode.addCommandClass( SetSchedulerCompensationCmd )

# --------------------------------------------------------------------------------
# [ no | default ] qos subinterface tx-queue count <4|8>
# --------------------------------------------------------------------------------
class QosNumTxqsPerSubIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'qos subinterface tx-queue count NUMTXQS'
   noOrDefaultSyntax = 'qos subinterface tx-queue count ...'
   data = {
      'qos': QosCli.nodeQosForConfig,
      'subinterface': 'Configure subinterface parameters',
      'tx-queue': QosCliServicePolicy.nodeTxQueue,
      'count': CliCommand.guardedKeyword( 'count',
                   helpdesc='Configure number of transmit queues per subinterface',
                   guard=guardNumTxqsPerSubIntf ),
      'NUMTXQS': CliMatcher.EnumMatcher( {
         '4': 'Transmit queues',
         '8': 'Transmit queues',
       } ),
   }
   handler = 'QosCliSchedulingHandler.handleNumTxqsPerSubIntf'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosNumTxqsPerSubIntfCmd )

# --------------------------------------------------------------------------------
# [ no | default ] qos subinterface scheduling parent round-robin
# --------------------------------------------------------------------------------
class QosParentFairSchedCmd( CliCommand.CliCommandClass ):
   syntax = 'qos subinterface scheduling parent round-robin'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'subinterface': 'Configure subinterface parameters',
      'scheduling': CliCommand.guardedKeyword( 'scheduling',
                       helpdesc='Configure subinterface scheduling parameters',
                       guard=guardSubintfScheduling ),
      'parent': 'Configure scheduling between subinterfaces and their parent '
      'interface',
      'round-robin': 'Configure Round Robin scheduling between subinterfaces and '
                      'their parent interface',
   }
   handler = 'QosGlobalConfigModeHandler.enableParentFairSched'
   noOrDefaultHandler = 'QosGlobalConfigModeHandler.disableParentFairSched'


GlobalConfigMode.addCommandClass( QosParentFairSchedCmd )

# --------------------------------------------------------------------------------
# qos tx-queue TXQ_VALUE nameTXQ_NAME
# ( no | default ) qos tx-queue TXQ_VALUE name ...
# --------------------------------------------------------------------------------
class QosTxQueueNameCmd( CliCommand.CliCommandClass ):
   syntax = ( 'qos tx-queue TXQ_VALUE name TXQ_NAME ' )
   noOrDefaultSyntax = ( 'qos tx-queue TXQ_VALUE name ...' )

   data = {
      'qos': QosCli.nodeQosForConfig,
      'tx-queue': matcherTxQueueConfig,
      'TXQ_VALUE': matcherTxQueueIdForName,
      'name': 'Name of the tx-queue',
      'TXQ_NAME': matcherTxQueueName,
   }

   handler = 'QosGlobalConfigModeHandler.handleTxQueueName'
   noOrDefaultHandler = handler


GlobalConfigMode.addCommandClass( QosTxQueueNameCmd )

#--------------------------------------------------------------------------------
# qos tx-queue TXQ_VALUE scheduler profile responsive
#--------------------------------------------------------------------------------
class QosTxQueueSchedulerProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'qos tx-queue TXQ_VALUE scheduler profile responsive'
   noOrDefaultSyntax = 'qos tx-queue TXQ_VALUE scheduler profile ...'

   data = {
      'qos': QosCli.nodeQosForConfig,
      'tx-queue' : matcherTxQueueConfig,
      'TXQ_VALUE' : matcherTxQueueIdForName,
      'scheduler' : nodeSchedulerToken,
      'profile' : matcherSchProfileToken,
      'responsive' : nodeSchProfileResponsive,
   }

   handler = 'QosGlobalConfigModeHandler.setGlobalTxQueueSchedulerProfile'
   noOrDefaultHandler = \
            'QosGlobalConfigModeHandler.noGlobalTxQueueSchedulerProfile'

GlobalConfigMode.addCommandClass( QosTxQueueSchedulerProfileCmd )

# --------------------------------------------------------------------------------
# [ no | default ] scheduler profile responsive
# --------------------------------------------------------------------------------
class QosTxQueuePerPortSchedulerProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'scheduler profile responsive'
   noOrDefaultSyntax = 'scheduler profile ...'

   data = {
      'scheduler': nodeSchedulerToken,
      'profile': matcherSchProfileToken,
      'responsive': nodeSchProfileResponsive,
   }

   handler = 'QosGlobalConfigModeHandler.setPerPortTxQueueSchedulerProfile'
   noOrDefaultHandler = \
            'QosGlobalConfigModeHandler.noPerPortTxQueueSchedulerProfile'

IntfTxQueueConfigMode.addCommandClass( QosTxQueuePerPortSchedulerProfileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos tx-queue shape rate percent adaptive
# --------------------------------------------------------------------------------
class TxQueueShapeRateAdaptiveCmd( CliCommand.CliCommandClass ):
   syntax = 'qos tx-queue shape rate percent adaptive'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'tx-queue': matcherTxQueueConfig,
      'shape': 'Configure shaping parameters',
      'rate': 'Set rate',
      'percent': 'Rate unit percent',
      'adaptive': "Use parent interface's available bandwidth",
   }
   handler = 'QosGlobalConfigModeHandler.enableTxQueueShapeRateAdaptive'
   noOrDefaultHandler = 'QosGlobalConfigModeHandler.disableTxQueueShapeRateAdaptive'


GlobalConfigMode.addCommandClass( TxQueueShapeRateAdaptiveCmd )

@Plugins.plugin( provides=( "QosCliScheduling", ) )
def Plugin( entityManager ):
   global subIntfHwStatus, qosSchedulerConfig, qosStatus, lagInputConfig, \
      qosConfig, qosHwStatus
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )
   qosSchedulerConfig = ConfigMount.mount( entityManager, "qos/scheduler/config",
                                           "Qos::QosSchedulerConfig", "w" )
   qosConfig = LazyMount.mount( entityManager, "qos/config", "Qos::Config", "r" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   lagInputConfig = LazyMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "r" )
