#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import io
import os
import sys

import difflib

import AclCliLib
import AclLib
import BasicCli
import BasicCliModes
import Cell
import CliCommand
import CliGlobal
import CliMatcher
import CliSession
import CliParser
import ConfigMount
import LazyMount
import MultiRangeRule
import Plugins
import QosLib
import ShowCommand
import Tac
import Tracing
from BasicCliModes import GlobalConfigMode
from CliMode.Fabric import FabricMode
from CliMode.Qos import ( PolicyMapClassModeBase, PolicyMapModeBase )
from IntfRangePlugin.VxlanIntf import VxlanAutoIntfType
from Intf.IntfRange import IntfRangeMatcher
from QosLib import ( coppMapType, tacCMapNm, classMapCpType, pmapQosActionTypes,
                     getCurrentTimeForConfigUpdate, actionFromEnum, qosMapType,
                     rateUnitFromEnum, burstUnitFromEnum, classMapCpStaticType,
                     coppStaticClassFromHwStatus, tacActionRateType, mapTypeToEnum,
                     mapTypeFromEnum, defaultClassName, isCoppStaticClassSupported,
                     builtInClassMapNames, pdpMapType, identicalPolicyMap, CliError,
                     matchOptionFromEnum, programmedPMapHwStatus, matchOptionToEnum,
                     deleteClassPrio,
                     isLagInServicePolicy, isCpuQueueAllocated, isSviInServicePolicy,
                     isVxlanIntfInServicePolicy, bsToBytes, isPmapSupportedForVxlan,
                     isL2ParamsMatchInPmap, isSvi, isPolicerInPmap, isLagPort,
                     directionToEnum, directionFromEnum, fabricIntfName,
                     invalidShapeRate, invalidGuaranteedBw, irToKbps,
                     isSviInServicePolicyOutDirection, isTxQueueNotConfigurable,
                     isTxQueue7Configurable, isLatencyThresholdValid )
from QosCliLib import ( PMapModelContainer, deleteInterfacePolicingConfig )
from QosTypes import ( tacPercent, tacRateUnit, tacBurstUnit, tacLatencyThreshold,
                       tacClassMapCpType, tacActionType, tacMatchOption, tacPMapNm,
                       tacFeatureName, tacPMapHwPrgmStatus, tacDirection, tacCos,
                       tacDropPrecedence, tacShapeRateVal, tacDscp, tacTrustMode,
                       tacGuaranteedBwVal, tacTxQueuePriority, tacQueueType,
                       tacSchedulerCompensation, tacPolicyMapType )
from CliPlugin import ( QosCli, PolicingCli, IntfCli, VlanIntfCli, VxlanCli,
                        FileCli, AclCli, EthIntfCli, LagIntfCli, SubIntfCli )
from CliPlugin.IntfRangeCli import IntfRangeConfigMode
from CliPlugin.VirtualIntfRule import IntfMatcher
from CliPlugin.QosCliIntfTypes import ( ethOrLagIntfPrefixes, ethOrLagIntfTypes,
                                  ethIntfPrefixes )
from CliPlugin.QosCliModel import ( PolicyMapAllModel, ClassMapAllModel,
                                    ClassMapWrapper, QosProfileAllModel,
                                    QosProfileSummaryModel,
                                    IntfDropThresholdsCollectionModel,
                                    IntfLatencyThresholdCollectionModel )
from CliPlugin.QosCli import nodeQosForShow
from CliPlugin.QosCliCommon import ( matcherIntf, guardPMapQosAction,
                                     guardDscpEcnMatch, setIntfConfig,
                                     isSubIntfConfigMode, trafficClassRangeFn,
                                     getConfiguredLagMembers, IntfTxQueueConfigMode,
                                     IntfUcTxQueueModelet, tacSubIntfId,
                                     cliBlockingFail, CLI_TIMEOUT,
                                     cliBlockingToApplyConfigChange,
                                     handleSessionCommit, QosProfileMode,
                                     hwConfigAclVerificationSupported,
                                     defaultWeight, matcherSet,
                                     QosSubIntfModelet, QosModelet )

tacRangeOperatorType = Tac.Type( 'Acl::RangeOperator' )

__defaultTraceHandle__ = Tracing.Handle( 'QosCliServicePolicy' )
t0 = Tracing.trace0

DYNAMIC_PMAP_EDIT_WARN = 'Editing dynamically created ' \
                              '(by agents and not CLI) policy map'

# -----------------------------------------------------------------------------------
# Variables for Qos Service Policy associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosAclHwStatus = None
qosHwStatus = None
profileConfigDir = None
cliQosAclConfig = None
subIntfHwStatus = None
qosSliceHwStatus = None
qosAclSliceHwStatus = None
cliQosAclConfigReadOnly = None
qosAclConfigDir = None
qosConfig = None
qosStatus = None
hwEpochStatus = None
qosInputConfig = None
qosInputProfileConfig = None
defaultPdpPmapCfgReadOnly = None
cliCounterConfig = None
fabricIntfConfigDir = None
qosHwConfig = None
lagInputConfig = None
qosGlobalConfig = None
aclConfig = None

gv = CliGlobal.CliGlobal(
   dict(
      qosAclConfig=None,
      qosAclConfigMergeSm=None,
   )
)

# Using BITMAPs to state which feature configuration has changed
MODIFY_DEF_COS = 0x1
MODIFY_DEF_DSCP = 0x2
MODIFY_TRUST_MODE = 0x4
MODIFY_SHAPE_RATE = 0x8
MODIFY_PFC = 0x10
MODIFY_TXQCONFIG = 0x20
MODIFY_ECN_TXQCONFIG = 0x40
MODIFY_ECN_DELAY_TXQCONFIG = 0x80
MODIFY_WRED_TXQCONFIG = 0x100
MODIFY_DPWRED_TXQCONFIG = 0x200
MODIFY_NONECT_TXQCONFIG = 0x400
MODIFY_DROP_THRESHOLDS = 0x800
MODIFY_LATENCY_THRESHOLD = 0x1000
MODIFY_GUARANTEED_BW = 0x2000
MODIFY_SCHEDULER_COMPENSATION = 0x4000
MODIFY_DSCP_PRESERVE_CONFIG = 0x8000
MODIFY_ECN_TXQ_COUNT_CONFIG = 0X10000
MODIFY_ECNDELAY_TXQCONFIG = 0x20000
MODIFY_DECN_TXQCONFIG = 0x40000
MODIFY_SCHEDULER_PROFILE_TXQCONFIG = 0x80000
MODIFY_ALL = 0x1FFFFF

ROLLBACK_PROFILE = 1

ecnDontCare = Tac.Type( "Acl::Ecn" ).dontCare
invalidDscp = Tac.Value( "Qos::DscpVal" ).invalid

# -----------------------------------------------------------------------------------
# Guards
# -----------------------------------------------------------------------------------
def guardIntfModePolicyQosIn( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      if isinstance( mode, QosProfileMode ):
         return None
      elif mode.intf.isSubIntf():
         if qosAclHwStatus.ingressSubintfPolicyQosSupported:
            return None
         if qosAclHwStatus.subIntfPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VlanIntfCli.VlanIntf ):
         if qosAclHwStatus.sviPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VxlanCli.VxlanIntf ):
         if qosAclHwStatus.vxlanPolicyQosSupported:
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform

def guardIntfModePolicyQosOut( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      if isinstance( mode, QosProfileMode ):
         return CliParser.guardNotThisPlatform
      if isinstance( mode.intf, VxlanCli.VxlanIntf ):
         if qosAclHwStatus.vxlanPolicyQosSupported:
            return None
      elif ( mode.intf.name.startswith( ethIntfPrefixes ) and
             mode.intf.isSubIntf() ):
         if qosAclHwStatus.egressSubintfRateLimitSupported:
            return None
      elif isinstance( mode.intf, VlanIntfCli.VlanIntf ):
         if qosAclHwStatus.egressPolicyQosSviSupported:
            return None
   return CliParser.guardNotThisPlatform

def guardPerInterfaceCounter( mode, token ):
   if qosAclHwStatus.policyQosAsyncTcam and \
      qosAclHwStatus.policyMapPerIntfCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapQosActionDropPrecedence( mode, token ):
   if qosAclHwStatus.dropPrecedenceSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardTxQDropPrecedenceThreshold( mode, token ):
   if isinstance( mode, IntfTxQueueConfigMode ):
      if isTxQueueNotConfigurable( qosHwStatus, mode.txQueue.id ):
         return CliParser.guardNotThisPlatform
   if isSubIntfConfigMode( mode ):
      return CliParser.guardNotThisPlatform
   if qosHwStatus.dropPrecedenceThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPoliceAction( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported:
      if qosAclHwStatus.dropPrecedenceYellowActionSupported or \
         qosAclHwStatus.dscpYellowActionSupported:
         return None
   return CliParser.guardNotThisPlatform

def guardDropPrecedenceYellowAction( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported and \
      qosAclHwStatus.dropPrecedenceYellowActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIngressTrTcmPolicing( mode, token ):   
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressPolicingSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpYellowAction( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported and \
      qosAclHwStatus.dscpYellowActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNamedSharedPolicerSupported( mode, token ):
   if qosAclHwStatus.namedSharedPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

def qosProfiles( mode ):
   qosProfileConfig = profileConfigDir.config
   return qosProfileConfig

def guardHwConfigVerification( mode, token ):
   if qosHwStatus.hwStatusReportingSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDropPrecedenceThreshold( mode, token ):
   if qosHwStatus.dropPrecedenceThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform


_guardClassMapCallback = {}

def guardClassMap( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      return None
   for cb in _guardClassMapCallback.values():
      if cb( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

# Functions for class dynamic name rules
def getClassNameRule( mode, mapType ):
   suggestedCompletions = list( cliQosAclConfig.cmapType[ mapType ].cmap )
   if mapType == coppMapType:
      suggestedCompletions.remove( tacCMapNm.coppDrop )
      suggestedCompletions.append( tacCMapNm.coppDefault )
   else:
      suggestedCompletions.append( tacCMapNm.classDefault )
   return suggestedCompletions

def getClassNameRuleQos( mode ):
   return getClassNameRule( mode, QosLib.qosMapType )

def txQueueRangeFn( mode=None, context=None ):
   if isTxQueue7Configurable( qosHwStatus ):
      return ( 0, qosHwStatus.numTxQueueSupported - 1 )
   return ( 0, qosHwStatus.numTxQueueSupported - 2 )

def ucTxQueueRangeFn( mode=None, context=None ):
   return ( 0, qosHwStatus.numUnicastQueueSupported - 1 )

def mcTxQueueRangeFn( mode=None, context=None ):
   return ( 0, qosHwStatus.numMulticastQueueSupported - 1 )

def guardMplsTrafficClassMatch( mode, token ):
   if qosAclHwStatus.mplsTrafficClassMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDropAction( mode, token ):
   if qosAclHwStatus.dropActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardTxQueue( mode, token ):
   if not qosHwStatus.hwInitialized or 0 == qosHwStatus.numTxQueueSupported:
      return CliParser.guardNotThisPlatform
   if qosHwStatus.numMulticastQueueSupported != 0:
      return CliParser.guardNotThisPlatform
   return None

def guardUcMcTxQueue( mode, token ):
   if not qosHwStatus.hwInitialized or 0 == qosHwStatus.numTxQueueSupported:
      return CliParser.guardNotThisPlatform
   if 0 == qosHwStatus.numMulticastQueueSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardTxQPrioritySupport( mode, token ):
   # By the time this function is called, hwStatus should have been initialized.
   if qosHwStatus.wrrSupported is False:
      return CliParser.guardNotThisPlatform
   if isinstance( mode, IntfTxQueueConfigMode ):
      if isTxQueueNotConfigurable( qosHwStatus, mode.txQueue.id ):
         return CliParser.guardNotThisPlatform

   # check if the txqueue is in configurable scheduling group if we are @
   # txqueue config mode
   if hasattr( mode, 'txQueue' ) and qosHwStatus.isSchGroupConfigurable[
         qosHwStatus.cliTxQueueToHwTxQueue[ mode.txQueue ].schGroupId ] is False:
      return CliParser.guardNotThisPlatform
   if ( isSubIntfConfigMode( mode ) and
        not subIntfHwStatus.subIntfTxQueueSchSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardLatencyThreshold( mode, token ):
   if ( ( not qosHwStatus.queueLatencyThresholdOnSubIntfGen3Supported ) and
         isSubIntfConfigMode( mode ) ):
      return CliParser.guardNotThisPlatform
   if qosHwStatus.latencyThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform


_guardPolicyMapCallback = {}

def guardPolicyMap( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      return None
   for cb in _guardPolicyMapCallback.values():
      if cb( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

def guardPMapQos( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      return None
   return CliParser.guardNotThisPlatform

# Functions for policy map dynamic name rules
def getPMapNameRule( mode, mapType ):
   suggestedCompletions = cliQosAclConfig.pmapType[ mapType ].pmap.members()
   return sorted( suggestedCompletions )

def getPMapNameQos( mode ):
   return getPMapNameRule( mode, QosLib.qosMapType )

# Functions for class map  dynamic name rules
def getCMapNameRule( mode, mapType ):
   suggestedCompletions = cliQosAclConfig.cmapType[ mapType ].cmap.members()
   if mapType == coppMapType:
      # static class maps cannot be edited, remove them.
      suggestedCompletions = [ x for x in
                               suggestedCompletions if classMapCpType(
                                  cliQosAclConfig, x ) !=
                               tacClassMapCpType.cmapCpStatic ]
   return sorted( suggestedCompletions )

def getCMapNameQos( mode ):
   return getCMapNameRule( mode, QosLib.qosMapType )

def getPolicerNameRule( mode ):
   suggestedCompletions = cliQosAclConfig.namedPolicer.members()
   return sorted( suggestedCompletions )

def guardCmapDropCounter( mode, token ):
   if qosAclHwStatus.redCounterSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardSviPolicyQosSupported( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
          qosAclHwStatus.sviPolicyQosSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanPolicyQosSupported( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
          qosAclHwStatus.vxlanPolicyQosSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfModePolicyQos( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      if isinstance( mode, QosProfileMode ):
         return None
      elif mode.intf.isSubIntf():
         if ( qosAclHwStatus.egressSubintfRateLimitSupported
           or qosAclHwStatus.subIntfPolicyQosSupported
           or qosAclHwStatus.ingressSubintfPolicyQosSupported ):
            return None
      elif isinstance( mode.intf, VlanIntfCli.VlanIntf ):
         if qosAclHwStatus.sviPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VxlanCli.VxlanIntf ):
         if qosAclHwStatus.vxlanPolicyQosSupported:
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform


_guardIntfModePolicyCallback = {}
_guardIntfModePolicyCallback[ QosLib.qosMapType ] = guardIntfModePolicyQos

def guardIntfModePolicy( mode, token ):
   for cb in _guardIntfModePolicyCallback.values():
      if cb( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

def guardIngressPolicing( mode, token ):
   if qosAclHwStatus.policyQosSupported and qosAclHwStatus.ingressPolicingSupported:
      return None
   return CliParser.guardNotThisPlatform

def validDropPrecedenceRangeFn( mode=None, context=None ):
   if isinstance( mode, ( PolicyMapClassMode, GlobalConfigMode ) ):
      validDp = QosLib.validDpConfigValues()
   else:
      # During startup config qosHwStatus will be inaccessible so returning
      # all possible values to avoid config failure.
      if mode.session_.startupConfig():
         return ( tacDropPrecedence.min, tacDropPrecedence.max )
      else:
         validDp = QosLib.validDpConfigValues( qosHwStatus )

   return( validDp[ 0 ], validDp[ -1 ] )

def latencyMicrosecondsFn( mode, context ):
   minLatency = tacLatencyThreshold().min
   if isLatencyThresholdValid( qosHwStatus ):
      maxLatency = qosHwStatus.maximumLatencyThreshold / 1000
   else:
      maxLatency = tacLatencyThreshold().maxMicroseconds()
   return ( minLatency, maxLatency )

def latencyMillisecondsFn( mode, context ):
   minLatency = tacLatencyThreshold().min
   if isLatencyThresholdValid( qosHwStatus ):
      maxLatency = qosHwStatus.maximumLatencyThreshold / 1000000
   else:
      maxLatency = tacLatencyThreshold().maxMilliseconds()
   return ( minLatency, maxLatency )

# -----------------------------------------------------------------------------------
# Matchers
# -----------------------------------------------------------------------------------
matcherDirectionType = CliMatcher.EnumMatcher( {
   'input': 'Policy-map at input of an interface',
   'output': 'Policy-map at output of an interface',
} )
matcherCopy = CliMatcher.KeywordMatcher( 'copy',
      helpdesc="Copy configuration from an existing policy map" )
matcherMatchAny = CliMatcher.KeywordMatcher( 'match-any',
      helpdesc='Logical-OR all match statements under this class-map' )
matcherPmapName = CliMatcher.DynamicNameMatcher( getPMapNameQos,
      'Policy Map Name',
      pattern='(?!copp|counters$|interface$|summary$)[A-Za-z0-9_:{}\\[\\]-]+' )
matcherPoliceCirValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify lower rate' )
matcherPoliceCommittedBurstValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify burst size' )
matcherPolicerName = CliMatcher.DynamicNameMatcher( getPolicerNameRule,
      helpdesc="Policer name" )
matchPoliceRateUnit = CliMatcher.EnumMatcher( {
   'bps': 'Rate in bps (default)',
   'kbps': 'The rate expressed in units of kilobits per second',
   'mbps': 'Rate in Mbps',
   'pps': 'Rate in pps',
} )
matcherSrcPMapTypeQos = CliMatcher.DynamicNameMatcher( getPMapNameQos,
      helpdesc="Source policy map name" )
matcherServiceProfile = CliMatcher.KeywordMatcher( 'service-profile',
      helpdesc='Configure QoS profile' )
matcherQosProfileName = CliMatcher.DynamicNameMatcher( qosProfiles,
      helpdesc='QoS profile name', pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )
matcherProfile = CliMatcher.KeywordMatcher( 'profile', helpdesc='Show QoS profile' )
profileNameMatcher = CliMatcher.DynamicNameMatcher(
      qosProfiles, "QoS profile name",
      pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )
matcherTxQueueRange = MultiRangeRule.MultiRangeMatcher(
      rangeFn=txQueueRangeFn, noSingletons=False,
      helpdesc="Tx-Queue ID or range(s) of Tx-Queue IDs" )
matcherUcTxQueueRange = MultiRangeRule.MultiRangeMatcher(
      rangeFn=ucTxQueueRangeFn, noSingletons=False,
      helpdesc="Unicast Tx-Queue ID or range(s) of Unicast Tx-Queue ID" )
matcherMcTxQueueRange = MultiRangeRule.MultiRangeMatcher(
      rangeFn=mcTxQueueRangeFn, noSingletons=False,
      helpdesc="Multicast Tx-Queue ID or range(s) of Multicast Tx-Queue ID" )
matcherRateUnit = CliMatcher.EnumMatcher( {
   'bps': 'Rate in bps (default)',
   'kbps': 'The rate expressed in units of kilobits per second',
   'mbps': 'Rate in Mbps',
} )
matcherBurstUnit = CliMatcher.EnumMatcher( {
   'bytes': 'Burst size in bytes (default unit)',
   'kbytes': 'Burst size in kbytes',
   'mbytes': 'Burst size in mbytes',
} )
matcherPolicePirValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify higher rate' )
matcherPoliceExcessBurstValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify burst size' )
matcherPoliceBurstValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify burst value' )
matcherType = CliMatcher.KeywordMatcher( 'type',
      helpdesc='Specify type' )
matcherPMapNameTypeQos = CliMatcher.DynamicNameMatcher( getPMapNameQos,
      "Policy Map Name",
      pattern=r'(?!counters$|interface$|summary$)[A-Za-z0-9_:{}\[\]-]+' )
matcherEct = CliMatcher.KeywordMatcher( 'ect',
      helpdesc='ECN codepoint ECT(0) or ECT(1)' )
matcherEctCe = CliMatcher.KeywordMatcher( 'ect-ce',
      helpdesc='ECN codepoint ECT(0), ECT(1) or CE' )
matcherNonEct = CliMatcher.KeywordMatcher( 'non-ect',
      helpdesc='ECN codepoint Not-ECT' )
matcherCe = CliMatcher.KeywordMatcher( 'ce',
      helpdesc='ECN codepoint CE' )
dropPrecedenceValueMatcher = CliMatcher.DynamicIntegerMatcher(
        validDropPrecedenceRangeFn,
        helpdesc='Drop precedence value' )

# -----------------------------------------------------------------------------------
# Tokens
# -----------------------------------------------------------------------------------
nodeCounter = CliMatcher.KeywordMatcher( 'counter', helpdesc='Count packets' )
nodePerInterfaceCounter = CliCommand.guardedKeyword( 'counter',
                                          helpdesc='Change counter settings',
                                          guard=guardPerInterfaceCounter )
nodeClassMapConfig = CliCommand.guardedKeyword( 'class-map',
      helpdesc='Configure Class Map', guard=guardClassMap )
nodeDrop = CliCommand.guardedKeyword( 'drop', helpdesc="Drop packets",
                                      guard=guardCmapDropCounter )
nodeDropPrecedence = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc='Set drop-precedence value', guard=guardTxQDropPrecedenceThreshold )
nodePerInterface = CliCommand.guardedKeyword( 'per-interface',
      helpdesc='Use per-interface counters', guard=guardPerInterfaceCounter )
nodePMapQos = CliCommand.guardedKeyword( 'qos', helpdesc="Qos type",
      guard=guardPMapQos )
nodePMapQosDropPrecedence = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc="Set drop precedence", guard=guardPMapQosActionDropPrecedence )
nodePolicer = CliCommand.guardedKeyword( 'policer',
      helpdesc="Configure shared QoS policer",
      guard=guardNamedSharedPolicerSupported )
nodePoliceAction = CliCommand.guardedKeyword( 'action',
      helpdesc='Specify action for lower rate', guard=guardPoliceAction )
nodePoliceSetAction = CliCommand.guardedKeyword( 'set',
      helpdesc='Set action for policed traffic', guard=guardPoliceAction )
nodePoliceActionDP = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc='Mark yellow packets with drop-precedence',
      guard=guardDropPrecedenceYellowAction )
nodePoliceHRate = CliCommand.guardedKeyword( 'rate',
      helpdesc='Set higher rate', guard=guardIngressTrTcmPolicing )
nodePoliceHBs = CliCommand.guardedKeyword( 'burst-size',
      helpdesc='Set burst-size for higher rate', guard=guardIngressTrTcmPolicing )
nodePoliceActionDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc='Mark yellow packets with DSCP value',
      guard=guardDscpYellowAction )
nodePoliceBc = CliCommand.guardedKeyword( 'bc',
      helpdesc='Set committed burst rate', guard=guardIngressPolicing )
nodePoliceCir = CliCommand.guardedKeyword( 'cir',
      helpdesc='Set committed information rate', guard=guardIngressPolicing )
nodePoliceLBs = CliCommand.guardedKeyword( 'burst-size',
      helpdesc="Set burst-size for lower rate", guard=guardIngressPolicing )
nodePoliceLRate = CliCommand.guardedKeyword( 'rate', helpdesc="Set lower Rate",
      guard=guardIngressPolicing )
nodePolicyMapConfig = CliCommand.guardedKeyword( 'policy-map',
      helpdesc="Configure Policy Map", guard=guardPolicyMap )
nodePoliceShared = CliCommand.guardedKeyword( 'shared',
      helpdesc="Use a shared policer", guard=guardNamedSharedPolicerSupported )
nodeQosType = CliCommand.guardedKeyword( 'qos', "QoS type",
      guard=guardPMapQos )
nodeQualityOfService = CliCommand.guardedKeyword( 'quality-of-service',
      helpdesc="Qos type", guard=guardPMapQos )
nodeServicePolicy = CliCommand.guardedKeyword( 'service-policy',
      helpdesc='Service policy configuration', guard=guardIntfModePolicy )
nodeMapTypeQos = CliCommand.guardedKeyword( 'qos',
      helpdesc='Qos type', guard=guardPMapQos )
nodeMapTypeQosDeprecated = CliCommand.guardedKeyword( 'qos',
      helpdesc='Qos type', guard=guardPMapQos,
      deprecatedByCmd='[no] policy-map type quality-of-service <pmap-name>' )
nodeInput = CliCommand.guardedKeyword( 'input',
      helpdesc='Apply the policy map to ingress packets',
      guard=guardIntfModePolicyQosIn )
nodeOutput = CliCommand.guardedKeyword( 'output',
      helpdesc='Apply the policy map to egress packets',
      guard=guardIntfModePolicyQosOut )
nodeMpls = CliCommand.guardedKeyword( 'mpls',
      helpdesc='Specify MPLS match', guard=guardMplsTrafficClassMatch )
nodeTrafficClass = CliCommand.guardedKeyword( 'traffic-class',
      helpdesc='Specify MPLS Traffic Class match', guard=guardMplsTrafficClassMatch )
nodeActionTc = CliCommand.guardedKeyword( 'traffic-class',
      helpdesc="Set Traffic class", guard=guardPMapQosAction )
nodeActionDrop = CliCommand.guardedKeyword( 'drop',
      helpdesc="Drop Packets", guard=guardDropAction )
nodeTxQueue = CliCommand.guardedKeyword( 'tx-queue',
      helpdesc='Configure transmit queue parameters', guard=guardTxQueue )
nodeUcTxQueue = CliCommand.guardedKeyword( 'uc-tx-queue',
      helpdesc='Configure unicast transmit queue parameters',
      guard=guardUcMcTxQueue )
nodeMcTxQueue = CliCommand.guardedKeyword( 'mc-tx-queue',
      helpdesc='Configure multicast transmit queue parameters',
      guard=guardUcMcTxQueue )
nodePriority = CliCommand.guardedKeyword( 'priority',
      helpdesc='Configure priority of this queue', guard=guardTxQPrioritySupport )
nodeLatency = CliCommand.guardedKeyword( 'latency',
      helpdesc='Set latency parameters', guard=guardLatencyThreshold )
nodePolice = CliCommand.guardedKeyword( 'police',
      helpdesc="Configure policer parameters", guard=guardIngressPolicing )
nodePolicyMap = CliCommand.guardedKeyword( 'policy-map', "Show Policy Map",
                                           guard=guardPolicyMap )
nodeDscpMatch = CliCommand.guardedKeyword( 'dscp',
      helpdesc='Specify DSCP match', guard=guardDscpEcnMatch )

# Copy contents of one map entry to another
def copyClassMap( dstEntry, srcEntry ):
   def copyClassMapL2Params( srcMatch, dstMatch ):
      dstMatch.vlanValue = None
      dstMatch.cosValue = None
      dstMatch.deiValue = None
      dstMatch.innerCosValue = None
      dstMatch.innerVlanValue = None

      if srcMatch.vlanValue:
         dstMatch.vlanValue = ( srcMatch.vlanValue.vlan,
                                srcMatch.vlanValue.vlanMask )
         dstMatch.vlanValue.vlanColl.clear()
         dstMatch.vlanValue.maskValid = srcMatch.vlanValue.maskValid
         for key, value in srcMatch.vlanValue.vlanColl.items():
            dstMatch.vlanValue.vlanColl[ key ] = value
      if srcMatch.cosValue:
         dstMatch.cosValue = ( '', )
         dstMatch.cosValue.cosColl.clear()
         for key, value in srcMatch.cosValue.cosColl.items():
            dstMatch.cosValue.cosColl[ key ] = value

      if srcMatch.deiValue:
         dstMatch.deiValue = ( srcMatch.deiValue.deiVal, )

      if srcMatch.innerCosValue:
         dstMatch.innerCosValue = ( '', )
         dstMatch.innerCosValue.innerCosColl.clear()
         for key, value in srcMatch.innerCosValue.innerCosColl.items():
            dstMatch.innerCosValue.innerCosColl[ key ] = value

      if srcMatch.innerVlanValue:
         dstMatch.innerVlanValue = ( srcMatch.innerVlanValue.innerVlan,
                                     srcMatch.innerVlanValue.innerVlanMask )
         dstMatch.innerVlanValue.innerVlanColl.clear()
         dstMatch.innerVlanValue.maskValid = srcMatch.innerVlanValue.maskValid
         for key, value in srcMatch.innerVlanValue.innerVlanColl.items():
            dstMatch.innerVlanValue.innerVlanColl[ key ] = value

   def copyClassMapDscpEcn( srcMatch, dstMatch ):
      dstMatch.dscpEcnValue = ( srcMatch.dscpEcnValue.dscp,
                                srcMatch.dscpEcnValue.ecn )
      dstMatch.dscpEcnValue.dscpColl.clear()
      dstMatch.dscpEcnValue.dscpNameValid = srcMatch.dscpEcnValue.dscpNameValid
      for key, value in srcMatch.dscpEcnValue.dscpColl.items():
         dstMatch.dscpEcnValue.dscpColl[ key ] = value

   def copyClassMapMplsTrafficClass( srcMatch, dstMatch ):
      dstMatch.mplsTrafficClassVal = ( '', )
      dstMatch.mplsTrafficClassVal.mplsTrafficClassColl.clear()
      dstMatch.mplsTrafficClassVal.isValid = srcMatch.mplsTrafficClassVal.isValid
      for key, value in \
          srcMatch.mplsTrafficClassVal.mplsTrafficClassColl.items():
         dstMatch.mplsTrafficClassVal.mplsTrafficClassColl[ key ] = value

   dstEntry.cpType = srcEntry.cpType
   dstEntry.dynamic = srcEntry.dynamic
   # Copy Match Rules
   for op in srcEntry.match:
      srcMatch = srcEntry.match[ op ]
      if op in dstEntry.match:
         dstMatch = dstEntry.match[ op ]
      else:
         # deleting the existing member before attaching new
         del dstEntry.match[ tacMatchOption.matchIpAccessGroup ]
         del dstEntry.match[ tacMatchOption.matchIpv6AccessGroup ]
         del dstEntry.match[ tacMatchOption.matchL2Params ]
         del dstEntry.match[ tacMatchOption.matchDscpEcn ]
         del dstEntry.match[ tacMatchOption.matchMplsTrafficClass ]
         del dstEntry.match[ tacMatchOption.matchMacAccessGroup ]
         dstMatch = dstEntry.match.newMember( op )
      if op in [ tacMatchOption.matchIpAccessGroup,
                 tacMatchOption.matchIpv6AccessGroup,
                 tacMatchOption.matchMacAccessGroup ]:
         dstMatch.strValue = srcMatch.strValue
      elif op in [ tacMatchOption.matchL2Params ]:
         dstMatch.strValue = srcMatch.strValue
         copyClassMapL2Params( srcMatch, dstMatch )
      elif op in [ tacMatchOption.matchDscpEcn ]:
         dstMatch.strValue = srcMatch.strValue
         copyClassMapDscpEcn( srcMatch, dstMatch )
      elif op in [ tacMatchOption.matchMplsTrafficClass ]:
         dstMatch.strValue = srcMatch.strValue
         copyClassMapMplsTrafficClass( srcMatch, dstMatch )

   for op in dstEntry.match:
      dstMatch = dstEntry.match[ op ]
      if op in [ tacMatchOption.matchIpAccessGroup,
                 tacMatchOption.matchIpv6AccessGroup,
                 tacMatchOption.matchL2Params,
                 tacMatchOption.matchDscpEcn,
                 tacMatchOption.matchMplsTrafficClass,
                 tacMatchOption.matchMacAccessGroup ] \
            and op not in srcEntry.match:
         del dstEntry.match[ op ]

def getL2ParamsMatchCombo( match ):
   matchCombo = []
   if match.vlanValue:
      matchCombo.append( 'vlan' )
   if match.cosValue:
      matchCombo.append( 'cos' )
   if match.deiValue:
      matchCombo.append( 'dei' )
   if match.innerVlanValue:
      matchCombo.append( 'vlan inner' )
   if match.innerCosValue:
      matchCombo.append( 'cos inner' )
   return matchCombo

def identicalClassMap( src, dst ):

   def identicalClassMapL2Params( srcMatch, dstMatch ):
      # Returns true only if all the values match
      srcMatchCombo = getL2ParamsMatchCombo( srcMatch )
      dstMatchCombo = getL2ParamsMatchCombo( dstMatch )
      if set( srcMatchCombo ) != set( dstMatchCombo ):
         return False
      if srcMatch.vlanValue and dstMatch.vlanValue:
         if srcMatch.vlanValue.maskValid != dstMatch.vlanValue.maskValid:
            return False

         if srcMatch.vlanValue.maskValid:
            if ( srcMatch.vlanValue.vlan != dstMatch.vlanValue.vlan or
                 srcMatch.vlanValue.vlanMask != dstMatch.vlanValue.vlanMask ):
               return False
         else:
            if ( set( srcMatch.vlanValue.vlanColl.items() ) !=
                 set( dstMatch.vlanValue.vlanColl.items() ) ):
               return False

      if srcMatch.cosValue and dstMatch.cosValue:
         if ( set( srcMatch.cosValue.cosColl.items() ) !=
              set( dstMatch.cosValue.cosColl.items() ) ):
            return False

      if srcMatch.deiValue and dstMatch.deiValue:
         if srcMatch.deiValue.deiVal != dstMatch.deiValue.deiVal:
            return False

      if srcMatch.innerCosValue and dstMatch.innerCosValue:
         if ( set( srcMatch.innerCosValue.innerCosColl.items() ) !=
              set( dstMatch.innerCosValue.innerCosColl.items() ) ):
            return False

      if srcMatch.innerVlanValue and dstMatch.innerVlanValue:
         if srcMatch.innerVlanValue.maskValid != dstMatch.innerVlanValue.maskValid:
            return False

         if srcMatch.innerVlanValue.maskValid:
            if ( srcMatch.innerVlanValue.innerVlan !=
                 dstMatch.innerVlanValue.innerVlan or
                 srcMatch.innerVlanValue.innerVlanMask !=
                 dstMatch.innerVlanValue.innerVlanMask ):
               return False
         else:
            if ( set( srcMatch.innerVlanValue.innerVlanColl.items() ) !=
                 set( dstMatch.innerVlanValue.innerVlanColl.items() ) ):
               return False
      return True

   def identicalClassMapDscpEcn( srcMatch, dstMatch ):
      if srcMatch.dscpEcnValue.dscp != dstMatch.dscpEcnValue.dscp:
         return False
      if srcMatch.dscpEcnValue.dscpNameValid != \
         dstMatch.dscpEcnValue.dscpNameValid:
         return False
      if srcMatch.dscpEcnValue.ecn != dstMatch.dscpEcnValue.ecn:
         return False
      return set( srcMatch.dscpEcnValue.dscpColl.items() ) == \
         set( dstMatch.dscpEcnValue.dscpColl.items() )

   def identicalClassMapMplsTrafficClass( srcMatch, dstMatch ):
      if srcMatch.mplsTrafficClassVal.isValid != \
         dstMatch.mplsTrafficClassVal.isValid:
         return False
      return set( srcMatch.mplsTrafficClassVal.mplsTrafficClassColl.items() ) == \
         set( dstMatch.mplsTrafficClassVal.mplsTrafficClassColl.items() )

   # Basic checks to verify that cmaps are identical
   if src.cpType != dst.cpType:
      return False
   srcOp = set( src.match )
   dstOp = set( dst.match )

   if srcOp != dstOp:
      return False

   for key in srcOp:
      srcMatch = src.match[ key ]
      dstMatch = dst.match[ key ]
      if srcMatch.strValue != dstMatch.strValue:
         return False
      if ( key == 'matchL2Params' and
           not identicalClassMapL2Params( srcMatch, dstMatch ) ):
         return False
      if key == 'matchDscpEcn' and not \
           identicalClassMapDscpEcn( srcMatch, dstMatch ):
         return False
      if key == 'matchMplsTrafficClass' and not \
           identicalClassMapMplsTrafficClass( srcMatch, dstMatch ):
         return False
   return True

# -------------------------------------------------------------------------------
# class-map mode
# -------------------------------------------------------------------------------
# ------------------------------------------------------------------------
# Editing context ( applicable for all modes )
#
# A context is created for each child mode. The context either copies
# an existing sequence entry or creates new entry.
#
# The editedEntries_ hold all sequences that are either new or edited.
# If a "no" command was issued, the "match" or "set" is removed from
# the entry. At commit the old entry is completely replaced with the
# entry in context.
#
# Note that a "no class-map" to remove a sequence is outside of the
# class-map config mode and handled in deleteClassMapMode.
# ------------------------------------------------------------------------
class ClassMapContext():
   def __init__( self, mode, mapType, cmapName, matchType ):
      self.mode = mode
      self.map_ = None
      self.mapType_ = mapType
      self.cmapName_ = cmapName
      self.matchType_ = matchType
      self.editedEntries_ = []
      self.currentEntry_ = None
      self.previousEntry_ = None  # for config rollback

      cmaps = cliQosAclConfig.cmapType[ mapType ].cmap
      if cmapName in cmaps:
         self.map_ = cmaps[ cmapName ]

         # Previous instance for config rollback
         prevEntry = Tac.newInstance( 'Qos::ClassMapConfig', self.cmapName_,
                                      self.mapType_ )
         copyClassMap( prevEntry, self.map_ )
         self.previousEntry_ = prevEntry

   def copyEditEntry( self, entry ):
      # Create a new cmap of type 0; i.e ControlPlane Type
      newEntry = Tac.newInstance( 'Qos::ClassMapConfig', self.cmapName_,
                                  self.mapType_ )
      copyClassMap( newEntry, entry )
      self.editedEntries_.append( newEntry )
      self.currentEntry_ = newEntry
      return newEntry

   def newEditEntry( self ):
      # Add an edit by creating a new entry
      # Create a new cmap of type 0 i.e ControlPlane Type
      entry = Tac.newInstance( 'Qos::ClassMapConfig', self.cmapName_,
                               coppMapType )
      self.editedEntries_.append( entry )
      self.currentEntry_ = entry
      if self.mapType_ == coppMapType:
         entry.cpType = 'cmapCpDynamic'
      else:
         entry.cpType = 'cmapCpNone'
      return entry

   def cmapName( self ):
      return self.cmapName_

   def mapType( self ):
      return self.mapType_

   def matchType( self ):
      return self.matchType_

   def cmap( self ):
      return self.map_  # may be None

   def currentEntry( self ):
      return self.currentEntry_

   def checkTtlFilter( self, aclSubConfig, aclName ):
      for ipRuleConfig in aclSubConfig.ipRuleById.values():
         ttl = ipRuleConfig.filter.ttl
         if ttl and ttl.oper != tacRangeOperatorType.eq:
            self.mode.addWarning( AclCliLib.unsuppRulesWarning(
               aclName, "ip", "ttl option not supported" ) )

      for ip6RuleConfig in aclSubConfig.ip6RuleById.values():
         ttl = ip6RuleConfig.filter.ttl
         if ttl and ttl.oper != tacRangeOperatorType.eq:
            self.mode.addWarning( AclCliLib.unsuppRulesWarning(
               aclName, "ipv6", "ttl option not supported" ) )

   def checkAndAddTtlAclWarning( self ):
      entry = self.map_
      if qosAclHwStatus.qosAclTtlRangeNotSupported \
         and self.mapType_ == tacPolicyMapType.mapQos:
         aclName = None
         for ( option, classMapMatch ) in entry.match.items():
            if option in ( tacMatchOption.matchIpAccessGroup,
                          tacMatchOption.matchIpv6AccessGroup ):
               aclName = classMapMatch.strValue

         for aclTypeConfig in aclConfig.config.values():
            if aclName in aclTypeConfig.acl.keys():
               aclConfig_ = aclTypeConfig.acl[ aclName ]
               for aclSubConfig in aclConfig_.subConfig.values():
                  self.checkTtlFilter( aclSubConfig, aclName )

   def commit( self ):

      # Commit current map. Create a new map if not exist
      cmType = cliQosAclConfig.cmapType[ self.mapType_ ]
      cmaps = cmType.cmap
      newCmap = False
      if self.map_ is None:
         if self.cmapName_ in cmaps:
            self.map_ = cmaps[ self.cmapName_ ]
         else:
            self.map_ = cmaps.newMember( self.cmapName_, cmType.type )
            newCmap = True
      entry = self.map_

      # We bump the version of cmap only if
      # we have actually changed it.
      bumpVersion = False
      for en in self.editedEntries_:
         bumpVersion = not identicalClassMap( entry, en )
         copyClassMap( entry, en )

      # Updating version field
      if bumpVersion:
         self.checkAndAddTtlAclWarning()
         # Update all the service policy config
         # that will get affected because of this change
         currTime = getCurrentTimeForConfigUpdate()

         self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
         self.map_.version += 1

         rc, errMsg = spWaitForHwStatusForCmapChange( self.mode, self.cmapName_,
                                                      self.mapType_, currTime )
         if not rc:
            if errMsg == CLI_TIMEOUT:
               self.mode.addWarning( QosLib.qosTimeoutWarning() )
            else:
               # rollback
               self.mode.addError( f"Error: Cannot commit class-map "
                                   f"{self.cmapName_}, {self.mapType_} ({errMsg})" )

               if newCmap:
                  t0( f"Deleting new class-map {self.cmapName_}" )
                  del cmaps[ self.cmapName_ ]
               else:
                  t0( f"Reverting class-map update for {self.cmapName_}" )
                  copyClassMap( entry, self.previousEntry_ )
                  self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
                  self.map_.version += 1

# -------------------------------------------------------------------------------
# Tokens for Policy maps and class maps.
# -------------------------------------------------------------------------------
# ------------------------------------------------------------------------
# qos policer <name> rate <value> {bps, kbps, mbps} burst-size <value> \
#       {bytes, kbytes, mbytes}
# ------------------------------------------------------------------------
def policerOutOfRange( mode, cir=None, cirUnit=None, bc=None, bcUnit=None,
                       pir=None, pirUnit=None, be=None, beUnit=None,
                       policerMode='committed' ):
   outOfRange = False
   if cirUnit == tacRateUnit.rateUnitPps:
      if bcUnit not in ( None, tacBurstUnit.burstUnitPackets ):
         mode.addError( 'Invalid burst size unit' )
         mode.addError( ' Burst size unit not in packets when rate unit is'
                       ' in PPS not supported' )
         outOfRange = True
      if cir < qosAclHwStatus.policePacketModeMinRatePps or \
         cir > qosAclHwStatus.policePacketModeMaxRatePps:
         mode.addError( 'Invalid lower rate value' )
         mode.addError( f'Lower rate range <'
                        f'{qosAclHwStatus.policePacketModeMinRatePps}-'
                        f'{qosAclHwStatus.policePacketModeMaxRatePps}> pps' )
         outOfRange = True
   else:
      tcirKbps = irToKbps( cir, cirUnit )
      if tcirKbps < qosAclHwStatus.policeByteModeMinRateKbps or \
         tcirKbps > qosAclHwStatus.policeByteModeMaxRateKbps:
         mode.addError( 'Invalid lower rate value' )
         mode.addError( f'Lower rate range <'
                        f'{qosAclHwStatus.policeByteModeMinRateKbps}-'
                        f'{qosAclHwStatus.policeByteModeMaxRateKbps}> kbps' )
         outOfRange = True
   if bcUnit == tacBurstUnit.burstUnitPackets:
      if cirUnit not in ( None, tacRateUnit.rateUnitPps ):
         mode.addError( 'Invalid burst size unit' )
         mode.addError( ' Burst size unit in packets when rate unit is'
                       ' not in PPS not supported' )
         outOfRange = True
      if bc < qosAclHwStatus.policePacketModeMinBurstPackets or \
         bc > qosAclHwStatus.policePacketModeMaxBurstPackets:
         mode.addError( 'Invalid burst size for lower rate' )
         mode.addError( f'Burst range <'
                     f'{qosAclHwStatus.policePacketModeMinBurstPackets}-'
                     f'{qosAclHwStatus.policePacketModeMaxBurstPackets}> packets' )
         outOfRange = True
   else:
      tbcBytes = bsToBytes( bc, bcUnit )
      if tbcBytes < qosAclHwStatus.policeByteModeMinBurstBytes or \
         tbcBytes > qosAclHwStatus.policeByteModeMaxBurstBytes:
         mode.addError( 'Invalid burst size for lower rate' )
         mode.addError( f'Burst range <{qosAclHwStatus.policeByteModeMinBurstBytes}-'
                        f'{qosAclHwStatus.policeByteModeMaxBurstBytes}> bytes' )
         outOfRange = True

   if policerMode == 'trTCM':
      # If cirUnit is in pps, make sure pirUnit is in pps and beUnit is in packets
      if cirUnit == tacRateUnit.rateUnitPps:
         if pirUnit not in ( None, tacRateUnit.rateUnitPps ):
            mode.addError( 'Invalid higher rate unit' )
            mode.addError( ' Higher rate unit not in PPS when lower rate unit is in'
                            ' PPS not supported' )
            outOfRange = True
         if beUnit not in ( None, tacBurstUnit.burstUnitPackets ):
            mode.addError( 'Invalid burst size unit for higher rate' )
            mode.addError( ' Burst size unit not in packets when rate unit'
                           '(lower or higher) is in PPS not supported' )
            outOfRange = True
      else:
         if pirUnit == tacRateUnit.rateUnitPps:
            mode.addError( 'Invalid higher rate unit' )
            mode.addError( ' Higher rate unit in PPS when lower rate unit is '
                           'not in PPS not supported' )
            outOfRange = True
         if beUnit == tacBurstUnit.burstUnitPackets:
            mode.addError( 'Invalid burst size unit for higher rate' )
            mode.addError( ' Burst size unit in packets when rate(lower or higher)'
                           ' unit is not in PPS not supported' )
            outOfRange = True

      # Range check for Pir (pps)
      if cirUnit == tacRateUnit.rateUnitPps and \
         pirUnit in ( tacRateUnit.rateUnitPps, None ):
         teirPps = pir - cir
         if teirPps < qosAclHwStatus.policePacketModeMinRatePps or \
            teirPps > qosAclHwStatus.policePacketModeMaxRatePps:
            mode.addError( 'Invalid higher rate value' )
            mode.addError( f'Higher rate range for specified lower rate is '
                        f'<{qosAclHwStatus.policePacketModeMinRatePps + cir}-'
                        f'%{qosAclHwStatus.policePacketModeMaxRatePps + cir}> pps' )
            outOfRange = True

      # Range check for be (packets)
      if cirUnit == tacRateUnit.rateUnitPps and \
         ( beUnit in ( tacBurstUnit.burstUnitPackets, None ) ):
         if be < qosAclHwStatus.policePacketModeMinBurstPackets or \
            be > qosAclHwStatus.policePacketModeMaxBurstPackets:
            mode.addError( 'Invalid burst size for higher rate' )
            mode.addError( f'Burst range <'
                     f'{qosAclHwStatus.policePacketModeMinBurstPackets}-'
                     f'{qosAclHwStatus.policePacketModeMaxBurstPackets}> packets ' )
            outOfRange = True

      # Range check for cir (bps, kbps, mbps)
      if tacRateUnit.rateUnitPps not in ( cirUnit, pirUnit ):
         tpirKbps = irToKbps( pir, pirUnit )
         teirKbps = tpirKbps - tcirKbps
         if teirKbps < qosAclHwStatus.policeByteModeMinRateKbps or \
            teirKbps > qosAclHwStatus.policeByteModeMaxRateKbps:
            mode.addError( 'Invalid higher rate' )
            mode.addError( f'Higher rate range for specified lower rate is '
                     f'<{qosAclHwStatus.policeByteModeMinRateKbps + tcirKbps}-'
                     f'{qosAclHwStatus.policeByteModeMaxRateKbps + tcirKbps}> kbps' )
            outOfRange = True

      # Range check for be (bytes, kbytes, mbytes)
      if cirUnit != tacRateUnit.rateUnitPps and \
         beUnit != tacBurstUnit.burstUnitPackets:
         tbeBytes = bsToBytes( be, beUnit )
         if tbeBytes < qosAclHwStatus.policeByteModeMinBurstBytes or \
            tbeBytes > qosAclHwStatus.policeByteModeMaxBurstBytes:
            mode.addError( 'Invalid burst size for higher rate' )
            mode.addError( f'Burst range <'
                           f'{qosAclHwStatus.policeByteModeMinBurstBytes}-'
                           f'{qosAclHwStatus.policeByteModeMaxBurstBytes}> bytes' )
            outOfRange = True
   return outOfRange

def pdpPolicerYellowActionNeeded( defPdpPolicy ):
   # Return True if default-pdp-policy contains classActions with policers
   # having yellow actions (e.g Strata)
   if not defPdpPolicy:
      return False
   for classAction in defPdpPolicy.classAction.values():
      if classAction.policer and classAction.policer.yellowActions:
         return True
   return False

def copyClassAction( src, dst ):
   # This function is invoked
   #
   #  Entering policy-map mode: copy from sysdb -> policy map editing copy
   #  Entering policy-class mode: copy from policy map editing copy ->
   #    policy-class mode editing copy
   #
   #  Committing from policy-class mode: copy from policy-class mode editing copy ->
   #    policy map editing copy
   #  Committing from policy-map mode: copy from policy map editing copy -> sysdb
   #
   #  Entering a new user-defined pdp policy map mode: copy from default-pdp-policy
   #  to policy map editing copy
   #  Reverting PDP class action to the action present for the class in
   #  default-pdp-policy

   # If we have removed the action from the source, then it means
   # 'no' command was issued from the action, therefore remove from
   # destination
   for action in dst.policyAction:
      if action not in src.policyAction:
         del dst.policyAction[ action ]

   for action in src.policyAction:
      dstPolicyAction = dst.policyAction.newMember( action )
      srcPolicyAction = src.policyAction[ action ]
      if action in [ tacActionType.actionSetShape,
                     tacActionType.actionSetBandwidth ]:
         dstPolicyAction.rate = ()
         dstPolicyAction.rate.val = srcPolicyAction.rate.val
         dstPolicyAction.rate.rateUnit = srcPolicyAction.rate.rateUnit
      elif action in [ tacActionType.actionSetDscp,
                       tacActionType.actionSetCos,
                       tacActionType.actionSetDrop,
                       tacActionType.actionSetDropPrecedence,
                       tacActionType.actionSetTc ]:
         dstPolicyAction.value = srcPolicyAction.value

   dst.dscpConfiguredAsName = src.dscpConfiguredAsName
   dst.dscpConfiguredAsNameYellow = src.dscpConfiguredAsNameYellow

   # If we have removed policer from source, then it means 'no' command was
   # issued for the policer, remove from destination
   if src.policer:
      dst.policer = ( src.policer.name, src.policer.cir, src.policer.bc,
                      src.policer.pir, src.policer.be )
      dst.policer.named = src.policer.named
      dst.policer.cir = src.policer.cir
      dst.policer.cirUnit = src.policer.cirUnit
      dst.policer.bc = src.policer.bc
      dst.policer.bcUnit = src.policer.bcUnit
      dst.policer.pir = src.policer.pir
      dst.policer.pirUnit = src.policer.pirUnit
      dst.policer.be = src.policer.be
      dst.policer.beUnit = src.policer.beUnit
      dst.policer.cmdVersion = src.policer.cmdVersion

      # Deleting the yellow-actions that were already present in the policer
      # useful when a trtcm-mode policer is changed to a committed one
      for action in dst.policer.yellowActions:
         if action not in src.policer.yellowActions:
            del dst.policer.yellowActions[ action ]

      # Though not now but when hardware allows for separate actions on
      # policed traffic like 'set tc', 'set dscp' etc, then the existing action for
      # yellow-packets will have to be modified. Since drop-precedence doesn't have a
      # value, no value is updated.
      for action in src.policer.yellowActions:
         dstPolicerYellowAction = dst.policer.yellowActions.newMember( action )
         srcPolicerYellowAction = src.policer.yellowActions[ action ]
         if action in [ tacActionType.actionSetDscp,
                        tacActionType.actionSetCos,
                        tacActionType.actionSetDropPrecedence,
                        tacActionType.actionSetTc ]:
            dstPolicerYellowAction.value = srcPolicerYellowAction.value

      action = tacActionType.actionSetDrop
      dst.policer.redActions.newMember( action )
      dst.policer.redActions[ action ].value = 5
   else:
      dst.policer = None
   # Copy 'count' action - used only for PDP classes.
   dst.count = src.count

def ipClassCounts( pmap, cmapExclusionList=None ):
   if cmapExclusionList is None:
      cmapExclusionList = []
   cmaps = cliQosAclConfig.cmapType[ QosLib.qosMapType ].cmap
   ipv4ClassMapCount = ipv6ClassMapCount = 0
   for classPriority in pmap.classPrio:
      cmapName = pmap.classPrio[ classPriority ].cmapName
      if cmapName in cmapExclusionList or cmapName not in cmaps:
         continue
      if 'matchIpAccessGroup' in cmaps[ cmapName ].match:
         ipv4ClassMapCount += 1
      elif 'matchIpv6AccessGroup' in cmaps[ cmapName ].match:
         ipv6ClassMapCount += 1
   return ipv4ClassMapCount, ipv6ClassMapCount

def getDefaultPdpPmapCfg():
   defaultPdpPmapCfg = None
   if defaultPdpPmapCfgReadOnly.pmapType.get( pdpMapType, None ):
      defaultPdpPmapCfg = defaultPdpPmapCfgReadOnly.pmapType[
         pdpMapType ].pmap.get( tacPMapNm.defaultPdpPmapName, None )
   return defaultPdpPmapCfg

# -------- Blocking CLI helpers for Policy Qos ----------------------
def getPMapHwStatus( hwStatus, pmapName, mapType, direction ):
   pmapHwStatusKey = Tac.newInstance( "Qos::PolicyMapHwStatusKey",
                                      direction, pmapName )
   pmapHwStatusTypePtr = hwStatus.pmapType.get( mapType )
   if pmapHwStatusTypePtr is None:
      return None
   return pmapHwStatusTypePtr.pmap.get( pmapHwStatusKey )

def waitForPMapHwPrgmStatus( mode, pmapName, mapType, direction,
                             updateRequestTime, updateStr=None,
                             profile=None, intf=None ):
   if not hwConfigAclVerificationSupported():
      return True, ""

   if mode and mode.session.inConfigSession():
      CliSession.registerSessionOnCommitHandler(
         mode.session_.entityManager, "QOS",
         lambda m, onSessionCommit: handleSessionCommit( m ) )
      qosHwConfig.clearHwStatusErrorRequestTime = Tac.now()
      return True, ""

   spConfigKey = Tac.newInstance( "Qos::ServicePolicyKey", mapType,
                                  direction, pmapName )

   if profile is None:
      if spConfigKey not in qosConfig.servicePolicyConfig:
         return True, ""
      spConfig = qosConfig.servicePolicyConfig[ spConfigKey ]
   else:
      if profile:
         inputConfig = qosInputProfileConfig
      else:
         inputConfig = qosInputConfig
      spConfig = inputConfig.servicePolicyConfig.get( spConfigKey )
      if not spConfig:
         return True, ""

   def getHwPrgmStatus( pmapHwStatusPtr ):
      errMsg = ""
      if pmapHwStatusPtr and \
            pmapHwStatusPtr.prgmStatus == tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
         t0( f"updateRequestTime : {updateRequestTime} for : ", spConfigKey )
         result = True
      else:
         result = False
         if pmapHwStatusPtr:
            try:
               envTimeout = os.getenv( 'QOS_TEST_CLI_TIMEOUT' )
               timeout = int( envTimeout ) if envTimeout else 30
               Tac.waitFor( lambda: ( pmapHwStatusPtr.prgmStatus
                                    != tacPMapHwPrgmStatus.hwPrgmStatusInProgress ),
                           description="waiting for program status success",
                           sleep=True, timeout=timeout, maxDelay=0.1 )
               if pmapHwStatusPtr.prgmStatus \
                     == tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
                  result = True
               else:
                  errMsg = pmapHwStatusPtr.prgmStatusErrMessage
            except Tac.Timeout:
               errMsg = pmapHwStatusPtr.prgmStatusErrMessage
         else:
            # May have to fill up appropriately, can happen when the
            # port-channel flaps or any other non-CLI trigger during
            # config update
            errMsg = "Unknown error"
      return result, errMsg

   if mapType != coppMapType:
      # For Qos Service-Policy, there should be atleast one
      # valid interface for an update to be sent to hw
      # By valid interface here it means it should be either be a:
      # -> physical interfase not part of port-channel
      # -> port-channel which has non zero member interfaces
      sendsHwUpdateForSp = False
      for intfName in spConfig.intfIds:
         if isLagPort( intfName ):
            # for port-channels additionally need to check if
            # it has any members.
            if getConfiguredLagMembers( intfName ):
               sendsHwUpdateForSp = True
         else:
            # if interface is member of a port-channel,
            # then don't send hw update for that case. In all other
            # cases send hw update for that service-policy
            sendsHwUpdateForSp = True
            if intfName in lagInputConfig.phyIntf and \
                   lagInputConfig.phyIntf[ intfName ].lag:
               sendsHwUpdateForSp = False

         if sendsHwUpdateForSp:
            # Found atleast one valid interface
            break

      if not sendsHwUpdateForSp:
         t0( "Does not send hw update for ServicePolicy : ", spConfigKey )
         return True, ""

   t0( "Sends hw update for ServicePolicy : ", spConfigKey )

   # wait for hw status and verify if it was successful
   result = False
   errMsg = ""
   description = updateStr + " to be applied"

   result, errMsg = cliBlockingToApplyConfigChange( mode, updateRequestTime,
                                                    tacFeatureName.others,
                                                    description,
                                                    policyMap=True )
   bestKey = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
   if not result:
      return result, errMsg
   if not qosConfig.servicePolicyConfig.get( bestKey ):
      # If key is not the best policy chosen by QosInputSelectorSm
      # it would not exist in pmapHwStatusTypePtr
      result = True
      bestKey = None
   elif intf:
      if intf not in qosConfig.servicePolicyConfig.get( bestKey ).intfIds:
         result = True
         bestKey = None
         for key in qosConfig.servicePolicyConfig:
            if intf in qosConfig.servicePolicyConfig[ key ].intfIds:
               bestKey = key
               break
   if bestKey:
      if mapType == 'mapQos' or mapType == 'mapPdp' or \
         qosAclHwStatus.coppDynamicClassSupported or \
         ( qosHwStatus.intfCoppSupported and
            qosHwStatus.coppActionPolicerSupported ):
         pmapHwStatus = None
         for aclSliceHwStatus in qosAclSliceHwStatus.values():
            if aclSliceHwStatus.isSlice and qosHwStatus.qosFeatureAgentSupported \
               and qosHwStatus.coppSupported:
               continue
            pmapHwStatus = getPMapHwStatus( aclSliceHwStatus, bestKey.pmapName,
                                             bestKey.type, bestKey.direction )
            result, errMsg = getHwPrgmStatus( pmapHwStatus )
            if not result:
               return result, errMsg
      else:
         # On Sand platforms we have a multiple writer issue currently ( SandAcl,
         # SandFap ) which write to prgmStatus, until that is reserved we are
         # adding a stopgap flag qosAclHwStatus.coppHwStatusReportingSupported
         # which is False only on Sand. Note that we still check the
         # configVersionUpdateTime in cliBlockingToApplyConfigChange, which is
         # guarded by qosAclHwStatus.hwStatusReportingSupported. We cannot set
         # hwStatusReportingSupported to False, since that is also used for
         # verifying Acl programming on Sand chips ( which can possibly fail ).
         # XXX_Copp XXX_seerpini
         for sliceHwStatus in qosSliceHwStatus.values():
            if qosAclHwStatus.coppHwStatusReportingSupported:
               pmapHwStatus = getPMapHwStatus( sliceHwStatus, bestKey.pmapName,
                                                bestKey.type, bestKey.direction )
               result, errMsg = getHwPrgmStatus( pmapHwStatus )
               if not result:
                  return result, errMsg
   return result, errMsg

def delServicePolicyNoLock( mapType, pmapName, direction, intfName,
                            profile=False ):
   if profile:
      inputConfig = qosInputProfileConfig
   else:
      inputConfig = qosInputConfig

   if pmapName is None:
      # if pmapName is not specified, traverse and find attached pmap
      for key, spCfg in inputConfig.servicePolicyConfig.items():
         if intfName in spCfg.intfIds and \
               key.direction == direction and  \
               key.type == mapType:
            pmapName = key.pmapName
            break
      if pmapName is None:
         # do not return any warning/error, otherwise tests need to
         # maintain states for cleanup
         return None
   else:
      # if pmapName specified, match attached pmap with given pmapName
      key = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
      if key not in inputConfig.servicePolicyConfig:
         errMsg = pmapName + " not attached to interface " + intfName
         return errMsg
      else:
         spCfg = inputConfig.servicePolicyConfig[ key ]
         if intfName not in spCfg.intfIds:
            errMsg = pmapName + " not attached to interface " + intfName
            return errMsg
   # all validations done, now delete
   if len( spCfg.intfIds ) > 1:
      del spCfg.intfIds[ intfName ]
   else:
      del spCfg.intfIds[ intfName ]
      del inputConfig.servicePolicyConfig[ key ]
   return None

def delServicePolicy( mode, mapType, pmapName, direction, intfName, profile=False ):
   errMsg = delServicePolicyNoLock( mapType, pmapName, direction, intfName,
                                    profile=profile )
   if errMsg is not None and mode is not None:
      mode.addError( errMsg )

def addServicePolicy( mode, mapType, pmapName, direction, intfName, profile=False ):
   # Check if this interface belongs to any service-policy, if so we need
   # delete the interface from the old service policy, before we add to the
   # new service-policy
   oldKey = None
   waitForHwStatus = False
   if profile:
      inputConfig = qosInputProfileConfig
   else:
      inputConfig = qosInputConfig
   for key, spCfg in inputConfig.servicePolicyConfig.items():
      if intfName not in spCfg.intfIds:
         continue
      if key.direction != direction or key.type != mapType:
         continue
      oldKey = key
      # Do not delete the service policy if its the same AND
      # hwPrgmStatus == Success
      if key.pmapName == pmapName:
         for aclSliceHwStatus in qosAclSliceHwStatus.values():
            pmapHwStatus = getPMapHwStatus( aclSliceHwStatus, pmapName,
                                            mapType, direction )
            if pmapHwStatus and \
               pmapHwStatus.prgmStatus == \
               tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
               finalSpCfg = qosConfig.servicePolicyConfig.get( key )
               if finalSpCfg and intfName in finalSpCfg.intfIds:
                  return None

      if key in inputConfig.servicePolicyConfig:
         delServicePolicy( mode, key.type, key.pmapName, key.direction,
                           intfName, profile=profile )
      waitForHwStatus = True
      break  # interface can have only one service policy

   # Add this interface to new service policy
   key = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
   if key in inputConfig.servicePolicyConfig:
      spCfg = inputConfig.servicePolicyConfig[ key ]
   else:
      spCfg = inputConfig.servicePolicyConfig.newMember( key )
   if intfName not in spCfg.intfIds:
      currTime = getCurrentTimeForConfigUpdate()
      spCfg.intfIds[ intfName ] = True

      # check if pmap exists in config
      pmapTypeConfigPtr = cliQosAclConfig.pmapType.get( key.type )
      if ( ( pmapTypeConfigPtr and pmapTypeConfigPtr.pmap.get( key.pmapName ) ) or
           ( mapType == pdpMapType and pmapName == tacPMapNm.defaultPdpPmapName and
             getDefaultPdpPmapCfg() ) ):
         # wait for all slices to return success status
         rc, errMsg = waitForPMapHwPrgmStatus( mode, key.pmapName, key.type,
                                               key.direction, currTime,
                                               updateStr="service-policy",
                                               profile=profile, intf=intfName )
         waitForHwStatus = False
         if not rc:
            if errMsg == CLI_TIMEOUT:
               if mode:
                  mode.addWarning( QosLib.qosTimeoutWarning() )
            else:
               # rollback
               t0( "Reverting service-policy" )
               delServicePolicyNoLock( key.type, key.pmapName, key.direction,
                                       intfName, profile=profile )
               if oldKey:
                  t0( "Applying back old service-policy to interface" )
               if oldKey and oldKey in inputConfig.servicePolicyConfig:
                  oldSpCfg = inputConfig.servicePolicyConfig[ oldKey ]
               elif oldKey:
                  oldSpCfg = inputConfig.servicePolicyConfig.newMember( oldKey )

                  oldSpCfg.intfIds[ intfName ] = True
               if mode:
                  mode.addError( f"Error: Cannot apply service-policy to "
                                 f"{intfName} ({errMsg})" )
               if profile:
                  return ROLLBACK_PROFILE

   if waitForHwStatus and not mode.session.inConfigSession():
      currTime = getCurrentTimeForConfigUpdate()
      description = "existing service-policy to be detached"
      cliBlockingToApplyConfigChange( mode, currTime, tacFeatureName.others,
                                      description, policyMap=True )
   return None

def identicalDpWredConfig( config1, config2 ):
   if config1 is None and config2 is None:
      return True
   if ( config1 is None and config2 ) or \
         ( config2 is None and config1 ):
      return False
   if set( config1.dpWredConfig ) != set( config2.dpWredConfig ):
      return False
   dpWredConfig1 = config1.dpWredConfig
   dpWredConfig2 = config2.dpWredConfig
   for dp in dpWredConfig1:
      if ( dpWredConfig1[ dp ].minThd != dpWredConfig2[ dp ].minThd or
           dpWredConfig1[ dp ].maxThd != dpWredConfig2[ dp ].maxThd or
           dpWredConfig1[ dp ].unit != dpWredConfig2[ dp ].unit or
           dpWredConfig1[ dp ].maxDroprate != dpWredConfig2[ dp ].maxDroprate or
           dpWredConfig1[ dp ].weight != dpWredConfig2[ dp ].weight ):
         return False
   return True

# ------------------------------------------------------------------------
# Qos Profile Infrastructure
# ------------------------------------------------------------------------
def identicalEcnOrWredConfig( config1, config2 ):
   if ( not config1 and config2 ) or \
          ( not config2 and config1 ):
      return False
   if config1 and ( config1.minThd != config2.minThd or
                    config1.maxThd != config2.maxThd or
                    config1.unit != config2.unit or
                    config1.maxDroprate != config2.maxDroprate or
                    config1.weight != config2.weight ):
      return False
   return True

def identicalDecnConfig( config1, config2 ):
   if ( not config1 and config2 ) or ( not config2 and config1 ):
      return False
   if config1 and ( config1.offset != config2.offset or
                    config1.floor != config2.floor or
                    config1.unit != config2.unit ):
      return False
   return True

def identicalEcnDelayConfigTxQ( config1, config2 ):
   if ( not config1 and config2 ) or \
          ( not config2 and config1 ):
      return False
   if config1 and ( config1.maxThd != config2.maxThd or
                    config1.unit != config2.unit ):
      return False
   return True

def identicalDropThresholds( config1, config2 ):
   if list( config1 ) != list( config2 ):
      return False
   for dp in config1:
      if config1[ dp ] != config2[ dp ]:
         return False
   return True

def identicalLatencyConfig( config1, config2 ):
   if config1 is None and config2 is None:
      return True
   if ( config1 is None and config2 ) or \
         ( config2 is None and config1 ):
      return False
   latencyThreshold1 = config1
   latencyThreshold2 = config2
   if ( latencyThreshold1.configUnit == latencyThreshold2.configUnit and
        latencyThreshold1.threshold == latencyThreshold2.threshold ):
      return True
   return False

def identicalQosProfile( profile, currentEntry ):
   if not profile or not currentEntry:
      return False
   if profile.defaultCos != currentEntry.defaultCos or \
          profile.defaultDscp != currentEntry.defaultDscp or \
          profile.trustMode != currentEntry.trustMode or \
          profile.shapeRate != currentEntry.shapeRate or \
          profile.guaranteedBw != currentEntry.guaranteedBw or \
          profile.schedulerCompensation != currentEntry.schedulerCompensation or \
          profile.dscpPreserveIpMplsEncapMode != \
            currentEntry.dscpPreserveIpMplsEncapMode:
      return False
   if sorted( currentEntry.txQueueConfig ) != \
      sorted( profile.txQueueConfig ):
      return False
   for txQueue in currentEntry.txQueueConfig:
      txQConfig = currentEntry.txQueueConfig[ txQueue ]
      txQueueConfig = profile.txQueueConfig.get( txQueue )
      ecnConfig = txQConfig.ecnConfig
      wredConfig = txQConfig.wredConfig
      ecnConfig2 = txQueueConfig.ecnConfig
      decnConfig1 = txQConfig.bufferBasedDecnConfig
      decnConfig2 = txQueueConfig.bufferBasedDecnConfig
      ecnDelayConfig = txQConfig.ecnDelayConfig
      ecnDelayConfig2 = txQueueConfig.ecnDelayConfig
      wredConfig2 = txQueueConfig.wredConfig
      nonEctConfig = txQConfig.nonEctConfig
      nonEctConfig2 = txQueueConfig.nonEctConfig
      dropThresholds = txQConfig.dropThresholds
      dropThresholds2 = txQueueConfig.dropThresholds
      latencyThreshold = txQConfig.latencyThreshold
      latencyThreshold2 = txQueueConfig.latencyThreshold
      dpConfig = None
      dpConfig2 = None
      if txQConfig.dpConfig and txQueueConfig.dpConfig:
         dpConfig = txQConfig.dpConfig
         dpConfig2 = txQueueConfig.dpConfig
      if txQueueConfig.guaranteedBw != txQConfig.guaranteedBw or \
             txQueueConfig.shapeRate != txQConfig.shapeRate or \
             txQueueConfig.bandwidth != txQConfig.bandwidth or \
             txQueueConfig.priority != txQConfig.priority or \
             txQueueConfig.weightConfig != txQConfig.weightConfig or \
             not identicalDropThresholds( dropThresholds, dropThresholds2 ) or \
             not identicalEcnOrWredConfig( ecnConfig, ecnConfig2 ) or \
             not identicalDecnConfig( decnConfig1, decnConfig2 ) or \
             not identicalEcnDelayConfigTxQ( ecnDelayConfig, ecnDelayConfig2 ) or \
             not identicalEcnOrWredConfig( wredConfig, wredConfig2 ) or \
             txQueueConfig.schedulerProfile != txQConfig.schedulerProfile or \
             not identicalDpWredConfig( dpConfig, dpConfig2 ) or \
             not identicalEcnOrWredConfig( nonEctConfig, nonEctConfig2 ) or \
             not identicalLatencyConfig( latencyThreshold, latencyThreshold2 ):
         return False

   currentEntryQueueLen = len( currentEntry.ecnTxQueueCounterConfig )
   if currentEntryQueueLen != len( profile.ecnTxQueueCounterConfig ):
      return False
   for txQueueId in currentEntry.ecnTxQueueCounterConfig:
      ecnTxQConfig = currentEntry.ecnTxQueueCounterConfig[ txQueueId ]
      ecnTxQueueConfig = profile.ecnTxQueueCounterConfig.get( txQueueId )

      if not ecnTxQConfig or not ecnTxQueueConfig:
         return False
      if ecnTxQConfig.counterEnable != ecnTxQueueConfig.counterEnable:
         return False

   if currentEntry.pfcPortConfig and not profile.pfcPortConfig:
      return False
   if not currentEntry.pfcPortConfig and profile.pfcPortConfig:
      return False
   if currentEntry.pfcPortConfig and \
          ( currentEntry.pfcPortConfig.enabled != profile.pfcPortConfig.enabled or
            currentEntry.pfcPortConfig.priorities !=
            profile.pfcPortConfig.priorities or
            currentEntry.pfcPortConfig.watchdogEnabled !=
            profile.pfcPortConfig.watchdogEnabled or
            currentEntry.pfcPortConfig.portTimerConfig !=
            profile.pfcPortConfig.portTimerConfig or
            currentEntry.pfcPortConfig.watchdogPortAction !=
            profile.pfcPortConfig.watchdogPortAction ):
      return False
   if currentEntry.fabricPfcDlb != profile.fabricPfcDlb:
      return False
   if currentEntry.servicePolicyConfig and not profile.servicePolicyConfig:
      return False
   if not currentEntry.servicePolicyConfig and profile.servicePolicyConfig:
      return False
   if currentEntry.servicePolicyConfig:
      key1 = currentEntry.servicePolicyConfig.key
      key2 = profile.servicePolicyConfig.key
      if key1.pmapName != key2.pmapName or key1.direction != key2.direction or \
             key1.type != key2.type:
         return False
   return True

def copyQosProfile( newProfile, oldProfile ):
   for txQueue in oldProfile.txQueueConfig:
      oldTxQConfig = oldProfile.txQueueConfig[ txQueue ]
      newTxQConfig = newProfile.txQueueConfig.newMember(
            txQueue, oldTxQConfig.priority, oldTxQConfig.bandwidth,
            oldTxQConfig.shapeRate, oldTxQConfig.guaranteedBw )
      newTxQConfig.dpConfig = ( "", )
      ecnConfig = oldTxQConfig.ecnConfig
      if ecnConfig:
         newTxQConfig.ecnConfig = ( ecnConfig.minThd, ecnConfig.maxThd,
                                    ecnConfig.unit, ecnConfig.maxDroprate,
                                    ecnConfig.weight )

      decnConfig = oldTxQConfig.bufferBasedDecnConfig
      if decnConfig:
         newTxQConfig.bufferBasedDecnConfig = (
            decnConfig.offset, decnConfig.floor, decnConfig.unit )

      ecnDelayConfig = oldTxQConfig.ecnDelayConfig
      if ecnDelayConfig:
         newTxQConfig.ecnDelayConfig = ( 0, ecnDelayConfig.maxThd,
                                         ecnDelayConfig.unit, tacPercent.max, 0 )

      newTxQConfig.delayEcnEnabled = oldTxQConfig.delayEcnEnabled
      newTxQConfig.ecnDelayThreshold = oldTxQConfig.ecnDelayThreshold
      newTxQConfig.schedulerProfile = oldTxQConfig.schedulerProfile
      wredConfig = oldTxQConfig.wredConfig
      if wredConfig:
         newTxQConfig.wredConfig = ( wredConfig.minThd, wredConfig.maxThd,
                                     wredConfig.unit, wredConfig.maxDroprate,
                                     wredConfig.weight )
      weightConfig = oldTxQConfig.weightConfig
      if weightConfig != defaultWeight:
         newTxQConfig.weightConfig = weightConfig
      dpConfig = oldTxQConfig.dpConfig
      if dpConfig:
         for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
            if dp in dpConfig.dpWredConfig:
               dpWredParam = dpConfig.dpWredConfig[ dp ]
               newTxQConfig.dpConfig.dpWredConfig.newMember(
                     dpWredParam.minThd,
                     dpWredParam.maxThd,
                     dpWredParam.unit,
                     dpWredParam.maxDroprate,
                     dpWredParam.weight, dp )
      nonEctConfig = oldTxQConfig.nonEctConfig
      if nonEctConfig:
         newTxQConfig.nonEctConfig = ( nonEctConfig.minThd, nonEctConfig.maxThd,
                                       nonEctConfig.unit, nonEctConfig.maxDroprate,
                                       nonEctConfig.weight )
      dropThresholds = oldTxQConfig.dropThresholds
      for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
         if dp in dropThresholds:
            newTxQConfig.dropThresholds[ dp ] = dropThresholds[ dp ]
         elif dp in newTxQConfig.dropThresholds:
            del newTxQConfig.dropThresholds[ dp ]
      newTxQConfig.latencyThreshold = oldTxQConfig.latencyThreshold

   for txQueueId in oldProfile.ecnTxQueueCounterConfig:
      oldTxQCounterConfig = oldProfile.ecnTxQueueCounterConfig[ txQueueId ]
      newTxQCounterConfig = newProfile.ecnTxQueueCounterConfig.newMember( txQueueId )
      newTxQCounterConfig.counterEnable = oldTxQCounterConfig.counterEnable

   if oldProfile.pfcPortConfig:
      newProfile.pfcPortConfig = ( "QosProfile",
                                    oldProfile.pfcPortConfig.enabled, )
      newProfile.pfcPortConfig.priorities = oldProfile.pfcPortConfig.priorities
      newProfile.pfcPortConfig.watchdogEnabled = oldProfile.pfcPortConfig.\
          watchdogEnabled
      newProfile.pfcPortConfig.portTimerConfig = oldProfile.pfcPortConfig.\
                                                 portTimerConfig
      newProfile.pfcPortConfig.watchdogPortAction = oldProfile.pfcPortConfig.\
         watchdogPortAction
      newProfile.fabricPfcDlb = oldProfile.fabricPfcDlb
   newProfile.defaultCos = oldProfile.defaultCos
   newProfile.defaultDscp = oldProfile.defaultDscp
   newProfile.trustMode = oldProfile.trustMode
   newProfile.shapeRate = oldProfile.shapeRate
   newProfile.guaranteedBw = oldProfile.guaranteedBw
   newProfile.schedulerCompensation = oldProfile.schedulerCompensation
   newProfile.dscpPreserveIpMplsEncapMode = oldProfile.dscpPreserveIpMplsEncapMode
   if oldProfile.servicePolicyConfig:
      key = oldProfile.servicePolicyConfig.key
      newProfile.servicePolicyConfig = ( key, )

class QosProfileModeContext():
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profile_ = None
      self.profileName_ = profileName
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.previousEntry_ = None  # used for config rollback
      qosProfileConfig = profileConfigDir.config
      if profileName in qosProfileConfig:
         self.profile_ = qosProfileConfig[ profileName ].qosProfile
         self.profileName_ = profileName
         prevQosProfile = Tac.newInstance( 'Qos::QosProfile',
                                             self.profileName_ )
         copyQosProfile( prevQosProfile, self.profile_ )
         self.previousEntry_ = prevQosProfile

   def copyEditEntry( self ):
      newQosProfile = Tac.newInstance( 'Qos::QosProfile', self.profileName_ )
      copyQosProfile( newQosProfile, self.profile_ )
      self.currentEntry_ = newQosProfile
      return newQosProfile

   def newEditEntry( self ):
      newQosProfile = Tac.newInstance( 'Qos::QosProfile', self.profileName_ )
      self.currentEntry_ = newQosProfile

   def profileName( self ):
      return self.profileName_

   def currentEntry( self ):
      return self.currentEntry_

   def qosProfile( self ):
      return self.profile_

   def commit( self ):
      # Commit current profile
      qosProfileConfig = profileConfigDir.config
      intfToProfileMap = profileConfigDir.intfToProfileMap
      # ModifyBitString contains the information about which parts of the qos/config
      # the qos profile modifies. This is depicted by the different bits being set.
      modifyBitString = 0
      if identicalQosProfile( self.profile_, self.currentEntry_ ):
         return
      if self.profile_ is None:
         profileConfig = qosProfileConfig.newMember( self.profileName_ )
         self.profile_ = profileConfig.qosProfile
      # Incase of rollback to None, delete the stale entries in qos/profile
      if self.currentEntry_ is None:
         self.profile_.txQueueConfig.clear()
         return
      profileConfig = qosProfileConfig.get( self.profileName_ )
      profileConfig.qosProfile = ( self.profileName_, )
      self.profile_ = profileConfig.qosProfile
      for txQueue in self.currentEntry_.txQueueConfig:
         txQConfig = self.currentEntry_.txQueueConfig[ txQueue ]
         txQueueConfig = self.profile_.txQueueConfig.get( txQueue )
         ecnConfig = txQConfig.ecnConfig
         decnConfig = txQConfig.bufferBasedDecnConfig
         ecnDelayConfig = txQConfig.ecnDelayConfig
         wredConfig = txQConfig.wredConfig
         nonEctConfig = txQConfig.nonEctConfig
         dropThresholds = txQConfig.dropThresholds
         dpConfig = txQConfig.dpConfig
         if not txQueueConfig:
            txQueueConfig = self.profile_.txQueueConfig.newMember(
               txQueue, txQConfig.priority, txQConfig.bandwidth,
               txQConfig.shapeRate, txQConfig.guaranteedBw )
            txQueueConfig.dpConfig = ( "", )
            modifyBitString |= MODIFY_TXQCONFIG
         else:
            if txQueueConfig.guaranteedBw != txQConfig.guaranteedBw or \
                   txQueueConfig.shapeRate != txQConfig.shapeRate or \
                   txQueueConfig.bandwidth != txQConfig.bandwidth or \
                   txQueueConfig.weightConfig != txQConfig.weightConfig or \
                   txQueueConfig.priority != txQConfig.priority:
               modifyBitString |= MODIFY_TXQCONFIG
            txQueueConfig.guaranteedBw = txQConfig.guaranteedBw
            txQueueConfig.shapeRate = txQConfig.shapeRate
            txQueueConfig.bandwidth = txQConfig.bandwidth
            txQueueConfig.weightConfig = txQConfig.weightConfig
            txQueueConfig.priority = txQConfig.priority

         if not identicalEcnOrWredConfig( txQueueConfig.ecnConfig, ecnConfig ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_ECN_TXQCONFIG
         if ecnConfig:
            txQueueConfig.ecnConfig = ( ecnConfig.minThd, ecnConfig.maxThd,
                                        ecnConfig.unit, ecnConfig.maxDroprate,
                                        ecnConfig.weight )
         else:
            txQueueConfig.ecnConfig = None

         if not identicalDecnConfig( txQueueConfig.bufferBasedDecnConfig,
                                     decnConfig ):
            modifyBitString |= MODIFY_DECN_TXQCONFIG
         if decnConfig:
            txQueueConfig.bufferBasedDecnConfig = (
               decnConfig.offset, decnConfig.floor, decnConfig.unit )
         else:
            txQueueConfig.bufferBasedDecnConfig = None

         if not identicalEcnDelayConfigTxQ( txQueueConfig.ecnDelayConfig,
                                            ecnDelayConfig ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_ECNDELAY_TXQCONFIG
         if ecnDelayConfig:
            txQueueConfig.ecnDelayConfig = ( 0, ecnDelayConfig.maxThd,
                                             ecnDelayConfig.unit, tacPercent.max, 0 )
         else:
            txQueueConfig.ecnDelayConfig = None

         if txQueueConfig.delayEcnEnabled != txQConfig.delayEcnEnabled or \
            txQueueConfig.ecnDelayThreshold != txQConfig.ecnDelayThreshold:
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_ECN_DELAY_TXQCONFIG
            txQueueConfig.delayEcnEnabled = txQConfig.delayEcnEnabled
            txQueueConfig.ecnDelayThreshold = txQConfig.ecnDelayThreshold

         if txQueueConfig.schedulerProfile != txQConfig.schedulerProfile:
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_SCHEDULER_PROFILE_TXQCONFIG
            txQueueConfig.schedulerProfile = txQConfig.schedulerProfile

         if not identicalEcnOrWredConfig( txQueueConfig.wredConfig, wredConfig ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_WRED_TXQCONFIG
         if wredConfig:
            txQueueConfig.wredConfig = ( wredConfig.minThd, wredConfig.maxThd,
                                         wredConfig.unit, wredConfig.maxDroprate,
                                         wredConfig.weight )
         else:
            txQueueConfig.wredConfig = None

         if txQConfig.weightConfig != defaultWeight:
            txQueueConfig.weightConfig = txQConfig.weightConfig

         if not identicalDpWredConfig( txQueueConfig.dpConfig, dpConfig ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_DPWRED_TXQCONFIG
            if dpConfig:
               for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
                  if dp in dpConfig.dpWredConfig:
                     dpWredParam = dpConfig.dpWredConfig[ dp ]
                     del txQueueConfig.dpConfig.dpWredConfig[ dp ]
                     txQueueConfig.dpConfig.dpWredConfig.newMember(
                           dpWredParam.minThd,
                           dpWredParam.maxThd,
                           dpWredParam.unit,
                           dpWredParam.maxDroprate,
                           dpWredParam.weight, dp )
                  else:
                     del txQueueConfig.dpConfig.dpWredConfig[ dp ]

         if not identicalEcnOrWredConfig( txQueueConfig.nonEctConfig, nonEctConfig ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_NONECT_TXQCONFIG
            if nonEctConfig:
               txQueueConfig.nonEctConfig = ( nonEctConfig.minThd,
                                              nonEctConfig.maxThd,
                                              nonEctConfig.unit,
                                              nonEctConfig.maxDroprate,
                                              nonEctConfig.weight )
            else:
               txQueueConfig.nonEctConfig = None

         if not identicalDropThresholds( txQueueConfig.dropThresholds,
                                         dropThresholds ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_DROP_THRESHOLDS
         for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
            if dp in dropThresholds:
               txQueueConfig.dropThresholds[ dp ] = dropThresholds[ dp ]
            elif dp in txQueueConfig.dropThresholds:
               del txQueueConfig.dropThresholds[ dp ]

         if txQueueConfig.latencyThreshold != txQConfig.latencyThreshold:
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_LATENCY_THRESHOLD
            txQueueConfig.latencyThreshold = txQConfig.latencyThreshold

      for txQueue in self.profile_.txQueueConfig:
         if txQueue not in self.currentEntry_.txQueueConfig:
            del self.profile_.txQueueConfig[ txQueue ]
            modifyBitString |= MODIFY_TXQCONFIG

      for txQueueId in self.currentEntry_.ecnTxQueueCounterConfig:
         ecnTxQCounterCfg = self.currentEntry_.ecnTxQueueCounterConfig[ txQueueId ]
         ecnTxQueueCounterCfg = self.profile_.ecnTxQueueCounterConfig.newMember(
            txQueueId )
         ecnTxQueueCounterCfg.counterEnable = ecnTxQCounterCfg.counterEnable
         modifyBitString |= MODIFY_ECN_TXQ_COUNT_CONFIG

      for txQueueId in self.profile_.ecnTxQueueCounterConfig:
         if txQueueId not in self.currentEntry_.ecnTxQueueCounterConfig:
            del self.profile_.ecnTxQueueCounterConfig[ txQueueId ]
            modifyBitString |= MODIFY_ECN_TXQ_COUNT_CONFIG

      if self.currentEntry_.pfcPortConfig:
         if not self.profile_.pfcPortConfig:
            self.profile_.pfcPortConfig = ( "QosProfile",
               self.currentEntry_.pfcPortConfig.enabled, )
            modifyBitString |= MODIFY_PFC
         if self.profile_.pfcPortConfig.enabled != \
                self.currentEntry_.pfcPortConfig.enabled or \
                self.profile_.pfcPortConfig.priorities != \
                self.currentEntry_.pfcPortConfig.priorities or \
                self.profile_.pfcPortConfig.watchdogEnabled != \
                self.currentEntry_.pfcPortConfig.watchdogEnabled or \
                self.profile_.pfcPortConfig.portTimerConfig != \
                self.currentEntry_.pfcPortConfig.portTimerConfig or \
                self.profile_.pfcPortConfig.watchdogPortAction != \
                self.currentEntry_.pfcPortConfig.watchdogPortAction:
            modifyBitString |= MODIFY_PFC
         self.profile_.pfcPortConfig.enabled = \
               self.currentEntry_.pfcPortConfig.enabled
         self.profile_.pfcPortConfig.priorities = \
               self.currentEntry_.pfcPortConfig.priorities
         self.profile_.pfcPortConfig.watchdogEnabled = \
               self.currentEntry_.pfcPortConfig.watchdogEnabled
         self.profile_.pfcPortConfig.portTimerConfig = \
               self.currentEntry_.pfcPortConfig.portTimerConfig
         self.profile_.pfcPortConfig.watchdogPortAction = \
               self.currentEntry_.pfcPortConfig.watchdogPortAction
         self.profile_.fabricPfcDlb = \
               self.currentEntry_.fabricPfcDlb
      elif self.profile_.pfcPortConfig and not self.currentEntry_.pfcPortConfig:
         self.profile_.pfcPortConfig = None
         self.profile_.fabricPfcDlb = 0
         modifyBitString |= MODIFY_PFC
      if self.profile_.defaultCos != self.currentEntry_.defaultCos:
         modifyBitString |= MODIFY_DEF_COS
         self.profile_.defaultCos = self.currentEntry_.defaultCos
      if self.profile_.defaultDscp != self.currentEntry_.defaultDscp:
         modifyBitString |= MODIFY_DEF_DSCP
         self.profile_.defaultDscp = self.currentEntry_.defaultDscp
      if self.profile_.trustMode != self.currentEntry_.trustMode:
         modifyBitString |= MODIFY_TRUST_MODE
         self.profile_.trustMode = self.currentEntry_.trustMode
      if self.profile_.shapeRate != self.currentEntry_.shapeRate:
         modifyBitString |= MODIFY_SHAPE_RATE
         self.profile_.shapeRate = self.currentEntry_.shapeRate
      if self.profile_.guaranteedBw != self.currentEntry_.guaranteedBw:
         modifyBitString |= MODIFY_GUARANTEED_BW
         self.profile_.guaranteedBw = self.currentEntry_.guaranteedBw
      if self.profile_.schedulerCompensation != \
         self.currentEntry_.schedulerCompensation:
         modifyBitString |= MODIFY_SCHEDULER_COMPENSATION
         self.profile_.schedulerCompensation = \
                              self.currentEntry_.schedulerCompensation
      if self.profile_.dscpPreserveIpMplsEncapMode != \
         self.currentEntry_.dscpPreserveIpMplsEncapMode:
         modifyBitString |= MODIFY_DSCP_PRESERVE_CONFIG
         self.profile_.dscpPreserveIpMplsEncapMode = \
            self.currentEntry_.dscpPreserveIpMplsEncapMode
      if self.currentEntry_.servicePolicyConfig:
         key = self.currentEntry_.servicePolicyConfig.key
         self.profile_.servicePolicyConfig = ( key, )
      elif self.profile_.servicePolicyConfig:
         self.profile_.servicePolicyConfig = None
      if os.environ.get( "TEST_QOS_PROFILE_VERSION" ):
         if os.environ.get( "QOS_PROFILE_VERSION" ):
            os.environ[ "QOS_PROFILE_VERSION" ] = \
                str( int( os.environ[ "QOS_PROFILE_VERSION" ] ) + 1 )
         else:
            os.environ[ "QOS_PROFILE_VERSION" ] = '1'
      for intf in intfToProfileMap:
         if intfToProfileMap[ intf ] == self.profileName_:
            k = applyProfile( mode=self.mode, profileName=self.profileName_,
                              interfaceName=intf,
                              checkCliBlocking=True,
                              modifyBitString=modifyBitString )
            if k == ROLLBACK_PROFILE:
               self.currentEntry_ = self.previousEntry_
               self.commit()

def applyPrevProfileOrRollBack( profileName, prevProfile, intfName ):
   if prevProfile != profileName:
      applyProfile( profileName=prevProfile,
                    interfaceName=intfName,
                    checkCliBlocking=False )
   else:
      return ROLLBACK_PROFILE
   return None

def getIntfIdentity( intf ):
   if hasattr( intf, 'intfId' ):
      intfId = intf.intfId
   elif hasattr( intf, 'name' ):
      intfId = intf.name
   else:
      intfId = intf
   return intfId

def removeProfile( mode, profileName, intfName ):
   intfToProfileMap = profileConfigDir.intfToProfileMap
   # return when non-existent profile is removed
   if intfName not in intfToProfileMap:
      return
   if mode and profileName and intfToProfileMap[ intfName ] != profileName:
      mode.addWarning( f'{profileName} not attached to interface {intfName}' )
      return
   if not profileName:
      profileName = intfToProfileMap[ intfName ]
   if intfName in qosInputProfileConfig.intfConfig:
      intfConfig = qosInputProfileConfig.intfConfig.get( intfName )
      if intfConfig.txQueueConfig:
         intfConfig.txQueueConfig.clear()
      intfConfig.pfcPortConfig = None
      intfConfig.defaultDscp = tacDscp.invalid
      intfConfig.defaultCos = tacCos.invalid
      intfConfig.trustMode = tacTrustMode.invalid
      intfConfig.shapeRate = Tac.Value( 'Qos::ShapeRate' )
      del qosInputProfileConfig.intfConfig[ intfName ]
   if intfName in qosInputProfileConfig.ecnIntfCounterConfig:
      ecnIntfCounterCfg = qosInputProfileConfig.ecnIntfCounterConfig.get( intfName )
      if ecnIntfCounterCfg.ecnTxQueueCounterConfig:
         ecnIntfCounterCfg.ecnTxQueueCounterConfig.clear()
      del qosInputProfileConfig.ecnIntfCounterConfig[ intfName ]
   if qosInputProfileConfig.servicePolicyConfig:
      for key in qosInputProfileConfig.servicePolicyConfig:
         spConfig = qosInputProfileConfig.servicePolicyConfig[ key ]
         if intfName in spConfig.intfIds:
            delServicePolicy( mode, key.type, key.pmapName,
                              key.direction, intfName, profile=True )
   del intfToProfileMap[ intfName ]

def applyProfileToIntf( intf, mode, profileName, checkCliBlocking,
                        modifyBitString, prevProfile, qosProfileConfig,
                        intfToProfileMap, applicableModes ):
   intfName = getIntfIdentity( intf )
   if mode and type( mode ) in applicableModes and \
           intfName in intfToProfileMap and \
           intfToProfileMap[ intfName ] == profileName:
      mode.addWarning( "{profileName} already applied on {intfName}" )
      return None
   for qosProfileName in qosProfileConfig:
      if intfName in intfToProfileMap and \
            intfToProfileMap[ intfName ] == qosProfileName:
         prevProfile = qosProfileName
   if prevProfile != profileName:
      removeProfile( mode, None, intfName )
   intfConfig = qosInputProfileConfig.intfConfig.get( intfName )
   shapeWarning = True
   timestamp = Tac.now()
   profile = qosProfileConfig[ profileName ].qosProfile
   if profile.defaultCos != tacCos.invalid or \
          profile.defaultDscp != tacDscp.invalid or \
          profile.trustMode != tacTrustMode.invalid or \
          profile.shapeRate.rate != tacShapeRateVal.invalid or \
          profile.guaranteedBw.bw != tacGuaranteedBwVal.invalid or \
          profile.schedulerCompensation != \
            tacSchedulerCompensation.invalid or \
          profile.dscpPreserveIpMplsEncapMode:
      if not intfConfig:
         intfConfig = qosInputProfileConfig.intfConfig.newMember( intfName )
         modifyBitString = MODIFY_ALL
      if modifyBitString & MODIFY_DEF_COS:
         intfConfig.defaultCos = profile.defaultCos
         if mode and checkCliBlocking and intfName != fabricIntfName and \
                cliBlockingFail( mode, timestamp, tacFeatureName.defaultCos,
                                 "Default Cos", intfName ):
            if intfName in qosInputProfileConfig.intfConfig:
               del qosInputProfileConfig.intfConfig[ intfName ]
            if prevProfile:
               return applyPrevProfileOrRollBack( profileName, prevProfile,
                                                  intfName )
            return None
      if modifyBitString & MODIFY_DEF_DSCP:
         intfConfig.defaultDscp = profile.defaultDscp
         if mode and checkCliBlocking and intfName != fabricIntfName and \
                cliBlockingFail( mode, timestamp, tacFeatureName.defaultDscp,
                                 "Default Dscp", intfName ):
            if intfName in qosInputProfileConfig.intfConfig:
               del qosInputProfileConfig.intfConfig[ intfName ]
            if prevProfile:
               return applyPrevProfileOrRollBack( profileName, prevProfile,
                                                  intfName )
            return None
      if modifyBitString & MODIFY_TRUST_MODE:
         intfConfig.trustMode = profile.trustMode
         if mode and checkCliBlocking and intfName != fabricIntfName and \
                cliBlockingFail( mode, timestamp, tacFeatureName.trustMode,
                                 "Trust Mode", intfName ):
            if intfName in qosInputProfileConfig.intfConfig:
               del qosInputProfileConfig.intfConfig[ intfName ]
            if prevProfile:
               return applyPrevProfileOrRollBack( profileName, prevProfile,
                                                  intfName )
            return None
      if modifyBitString & MODIFY_SHAPE_RATE:
         intfConfig.shapeRate = profile.shapeRate
      if modifyBitString & MODIFY_GUARANTEED_BW:
         intfConfig.guaranteedBw = profile.guaranteedBw
      if modifyBitString & MODIFY_SCHEDULER_COMPENSATION:
         if intfName == fabricIntfName:
            mode.addWarning( "Scheduler packet size adjustment is not " +
                             "supported on Fabric" )
         else:
            intfConfig.schedulerCompensation = profile.schedulerCompensation
      if modifyBitString & MODIFY_DSCP_PRESERVE_CONFIG:
         if intfName == fabricIntfName:
            mode.addWarning( "DscpPreserveMplsIpEncap is not " +
                             "supported on Fabric" )
         else:
            intfConfig.dscpPreserveIpMplsEncapMode = \
               profile.dscpPreserveIpMplsEncapMode
   elif intfConfig:
      if profile.defaultCos == tacCos.invalid and \
             modifyBitString & MODIFY_DEF_COS:
         intfConfig.defaultCos = profile.defaultCos
      if profile.defaultDscp == tacDscp.invalid and \
             modifyBitString & MODIFY_DEF_DSCP:
         intfConfig.defaultDscp = profile.defaultDscp
      if profile.trustMode == tacTrustMode.invalid and \
             modifyBitString & MODIFY_TRUST_MODE:
         intfConfig.trustMode = profile.trustMode
      if profile.shapeRate.rate == tacShapeRateVal.invalid and \
             modifyBitString & MODIFY_SHAPE_RATE:
         intfConfig.shapeRate = profile.shapeRate
      if subIntfHwStatus.subIntfBandwidthSupported and \
             profile.guaranteedBw.bw == tacGuaranteedBwVal.invalid and \
             modifyBitString & MODIFY_GUARANTEED_BW:
         intfConfig.guaranteedBw = profile.guaranteedBw
      if profile.schedulerCompensation == tacSchedulerCompensation.invalid and\
             modifyBitString & MODIFY_SCHEDULER_COMPENSATION:
         if intfName == fabricIntfName:
            mode.addWarning( "Scheduler packet size adjustment is not " +
                             "supported on Fabric" )
         else:
            intfConfig.schedulerCompensation = profile.schedulerCompensation
      if not profile.dscpPreserveIpMplsEncapMode and \
             modifyBitString & MODIFY_DSCP_PRESERVE_CONFIG:
         if intfName == fabricIntfName:
            mode.addWarning( "DscpPreserveMplsIpEncap is not " +
                             "supported on Fabric" )
         else:
            intfConfig.dscpPreserveIpMplsEncapMode = \
               profile.dscpPreserveIpMplsEncapMode

   ecnIntfCounterCfg = qosInputProfileConfig.ecnIntfCounterConfig.get(
         intfName )
   if modifyBitString & MODIFY_ECN_TXQ_COUNT_CONFIG:
      if ecnIntfCounterCfg:
         for txQueueId in ecnIntfCounterCfg.ecnTxQueueCounterConfig:
            if txQueueId not in profile.ecnTxQueueCounterConfig:
               del ecnIntfCounterCfg.ecnTxQueueCounterConfig[ txQueueId ]
      for txQueueId in profile.ecnTxQueueCounterConfig:
         timestamp = Tac.now
         profileEcnTxQCounterCfg = profile.ecnTxQueueCounterConfig[ txQueueId ]

         if not ecnIntfCounterCfg:
            ecnIntfCounterCfg = qosInputProfileConfig.ecnIntfCounterConfig. \
               newMember( intfName )

         intfEcnTxQCounterCfg = ecnIntfCounterCfg.ecnTxQueueCounterConfig. \
            newMember( txQueueId )
         intfEcnTxQCounterCfg.counterEnable = \
            profileEcnTxQCounterCfg.counterEnable

   # pylint: disable=too-many-nested-blocks
   if modifyBitString & MODIFY_TXQCONFIG:
      if intfConfig:
         for txQueue in intfConfig.txQueueConfig:
            if txQueue not in profile.txQueueConfig:
               del intfConfig.txQueueConfig[ txQueue ]

      for txQueue in profile.txQueueConfig:
         timestamp = Tac.now()
         profileTxQConfig = profile.txQueueConfig[ txQueue ]
         if intfName == fabricIntfName:
            if mode and type( mode ) in applicableModes and \
                   profileTxQConfig.shapeRate != invalidShapeRate and \
                   shapeWarning:
               mode.addWarning( "Tx-Queue shape " +
                                "not supported on fabric interface" )
               shapeWarning = False
            if profileTxQConfig.guaranteedBw == invalidGuaranteedBw and \
                   not profileTxQConfig.ecnConfig and \
                   not profileTxQConfig.bufferBasedDecnConfig and \
                   not profileTxQConfig.ecnDelayConfig and \
                   not profileTxQConfig.delayEcnEnabled and \
                   not profileTxQConfig.wredConfig and \
                   not profileTxQConfig.nonEctConfig and \
                   len( profileTxQConfig.dropThresholds ) == 0 and \
                   ( not profileTxQConfig.dpConfig or
                     len( profileTxQConfig.dpConfig.dpWredConfig ) == 0 ) and \
                   profileTxQConfig.priority == \
                   tacTxQueuePriority.priorityInvalid and \
                   profileTxQConfig.weightConfig == defaultWeight and \
                   profileTxQConfig.bandwidth == tacPercent.invalid:
               if intfConfig and txQueue in intfConfig.txQueueConfig:
                  del intfConfig.txQueueConfig[ txQueue ]
         if not intfConfig:
            intfConfig = qosInputProfileConfig.intfConfig.newMember( intfName )

         # prepare for setting qos/config for the following
         # - tx-q priority / bw percent
         # - tx-q guaranteed bandwidth
         # - tx-q ecn configuration
         # - tx-q wred configuration
         # - tx-q nonect configuration

         # priority / bw percent
         # Explicit strict priority information is required only for
         # subinterfaces TxQs. If interface is FPP, don't configure strict
         # priority.
         if intfName != fabricIntfName and \
            not tacSubIntfId.isSubIntfId( intfName ) and \
            profileTxQConfig.priority == tacTxQueuePriority.priorityStrict:
            priority = tacTxQueuePriority.priorityInvalid
         else:
            priority = profileTxQConfig.priority

         bwPercent = profileTxQConfig.bandwidth

         # guaranteed bandwidth
         guaranteedBw = profileTxQConfig.guaranteedBw
         if intfName != fabricIntfName:
            shapeRate = profileTxQConfig.shapeRate
         else:
            shapeRate = invalidShapeRate

         intfTxQConfig = intfConfig.txQueueConfig.get( txQueue )
         if not intfTxQConfig:
            intfTxQConfig = intfConfig.txQueueConfig.newMember(
               txQueue, priority, bwPercent,
               shapeRate, guaranteedBw )
            intfTxQConfig.dpConfig = ( "", )
         else:
            intfTxQConfig.priority = priority
            intfTxQConfig.bandwidth = bwPercent
            intfTxQConfig.shapeRate = shapeRate
            intfTxQConfig.guaranteedBw = guaranteedBw

         # ecn configuration
         ecnConfig = profileTxQConfig.ecnConfig
         if modifyBitString & MODIFY_ECN_TXQCONFIG and ecnConfig:
            intfTxQConfig.ecnConfig = ( ecnConfig.minThd, ecnConfig.maxThd,
                                        ecnConfig.unit,
                                        ecnConfig.maxDroprate,
                                        ecnConfig.weight )
            if mode and checkCliBlocking and cliBlockingFail( mode,
                            timestamp, tacFeatureName.ecn, "Ecn" ):
               if intfName in qosInputProfileConfig.intfConfig:
                  del qosInputProfileConfig.intfConfig[ intfName ]
               if prevProfile:
                  return applyPrevProfileOrRollBack( profileName,
                                                     prevProfile, intfName )
               return None
         elif modifyBitString & MODIFY_ECN_TXQCONFIG and \
              intfTxQConfig and intfTxQConfig.ecnConfig:
            intfTxQConfig.ecnConfig = None

         decnConfig = profileTxQConfig.bufferBasedDecnConfig
         if modifyBitString & MODIFY_DECN_TXQCONFIG:
            if decnConfig:
               intfTxQConfig.bufferBasedDecnConfig = (
                  decnConfig.offset, decnConfig.floor, decnConfig.unit )
               if mode and checkCliBlocking and cliBlockingFail(
                     mode, timestamp, tacFeatureName.bufferBasedDecn, "DECN" ):
                  if intfName in qosInputProfileConfig.intfConfig:
                     del qosInputProfileConfig.intfConfig[ intfName ]
                  if prevProfile:
                     return applyPrevProfileOrRollBack( profileName,
                                                        prevProfile, intfName )
                  return None
            elif intfTxQConfig:
               intfTxQConfig.bufferBasedDecnConfig = None

         # ecn delay configuration
         ecnDelayConfig = profileTxQConfig.ecnDelayConfig

         if modifyBitString & MODIFY_ECNDELAY_TXQCONFIG and ecnDelayConfig:
            intfTxQConfig.ecnDelayConfig = ( 0, ecnDelayConfig.maxThd,
                                             ecnDelayConfig.unit, tacPercent.max, 0 )
         elif modifyBitString & MODIFY_ECNDELAY_TXQCONFIG and \
              intfTxQConfig and intfTxQConfig.ecnDelayConfig:
            intfTxQConfig.ecnDelayConfig = None

         if modifyBitString & MODIFY_ECN_DELAY_TXQCONFIG:
            intfTxQConfig.delayEcnEnabled = profileTxQConfig.delayEcnEnabled
            intfTxQConfig.ecnDelayThreshold = \
                  profileTxQConfig.ecnDelayThreshold

         if modifyBitString & MODIFY_SCHEDULER_PROFILE_TXQCONFIG:
            intfTxQConfig.schedulerProfile = profileTxQConfig.schedulerProfile

         # Queue weight configuration
         weightConfig = profileTxQConfig.weightConfig
         if weightConfig != defaultWeight:
            intfTxQConfig.weightConfig = weightConfig

         # wred configuration
         wredConfig = profileTxQConfig.wredConfig
         if modifyBitString & MODIFY_WRED_TXQCONFIG and wredConfig:
            intfTxQConfig.wredConfig = ( wredConfig.minThd,
                                         wredConfig.maxThd,
                                         wredConfig.unit,
                                         wredConfig.maxDroprate,
                                         wredConfig.weight )
            if mode and checkCliBlocking and cliBlockingFail( mode,
                           timestamp, tacFeatureName.wred, "Wred" ):
               if intfName in qosInputProfileConfig.intfConfig:
                  del qosInputProfileConfig.intfConfig[ intfName ]
               if prevProfile:
                  return applyPrevProfileOrRollBack( profileName,
                                                     prevProfile, intfName )
               return None
         elif modifyBitString & MODIFY_WRED_TXQCONFIG and \
              intfTxQConfig and intfTxQConfig.wredConfig:
            intfTxQConfig.wredConfig = None

         # dpWred configuration
         dpConfig = profileTxQConfig.dpConfig
         if modifyBitString & MODIFY_DPWRED_TXQCONFIG and \
            dpConfig and dpConfig.dpWredConfig:
            for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
               if dp in dpConfig.dpWredConfig:
                  dpWredParam = dpConfig.dpWredConfig[ dp ]
                  del intfTxQConfig.dpConfig.dpWredConfig[ dp ]
                  intfTxQConfig.dpConfig.dpWredConfig.newMember(
                        dpWredParam.minThd,
                        dpWredParam.maxThd,
                        dpWredParam.unit,
                        dpWredParam.maxDroprate,
                        dpWredParam.weight, dp )
               else:
                  del intfTxQConfig.dpConfig.dpWredConfig[ dp ]
            if mode and checkCliBlocking and cliBlockingFail( mode,
                  timestamp, tacFeatureName.wred, "DpWred" ):
               if intfName in qosInputProfileConfig.intfConfig:
                  del qosInputProfileConfig.intfConfig[ intfName ]
               if prevProfile:
                  return applyPrevProfileOrRollBack( profileName,
                                                     prevProfile, intfName )
               return None
         elif modifyBitString & MODIFY_DPWRED_TXQCONFIG and \
              intfTxQConfig and intfTxQConfig.dpConfig:
            intfTxQConfig.dpConfig.dpWredConfig.clear()

         # nonect configuration
         nonEctConfig = profileTxQConfig.nonEctConfig
         if modifyBitString & MODIFY_NONECT_TXQCONFIG and nonEctConfig:
            intfTxQConfig.nonEctConfig = ( nonEctConfig.minThd,
                                           nonEctConfig.maxThd,
                                           nonEctConfig.unit,
                                           nonEctConfig.maxDroprate,
                                           nonEctConfig.weight )
            if mode and checkCliBlocking and cliBlockingFail( mode,
                           timestamp, tacFeatureName.ecn, "Non-Ect" ):
               if intfName in qosInputProfileConfig.intfConfig:
                  del qosInputProfileConfig.intfConfig[ intfName ]
               if prevProfile:
                  return applyPrevProfileOrRollBack( profileName,
                                                     prevProfile, intfName )
               return None
         elif modifyBitString & MODIFY_NONECT_TXQCONFIG and \
              intfTxQConfig and intfTxQConfig.nonEctConfig:
            intfTxQConfig.nonEctConfig = None

         # dropThreshold configuration
         dropThresholds = profileTxQConfig.dropThresholds
         if modifyBitString & MODIFY_DROP_THRESHOLDS and len( dropThresholds ) != 0:
            for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
               if dp in dropThresholds:
                  intfTxQConfig.dropThresholds[ dp ] = dropThresholds[ dp ]
               elif dp in intfTxQConfig.dropThresholds:
                  del intfTxQConfig.dropThresholds[ dp ]
            if mode and checkCliBlocking and cliBlockingFail(
                  mode, timestamp, tacFeatureName.dropThresholds,
                  "Drop-Thresholds" ):
               if intfName in qosInputProfileConfig.intfConfig:
                  del qosInputProfileConfig.intfConfig[ intfName ]
               if prevProfile:
                  return applyPrevProfileOrRollBack( profileName,
                                                     prevProfile, intfName )
               return None
         elif modifyBitString & MODIFY_DROP_THRESHOLDS and intfTxQConfig:
            intfTxQConfig.dropThresholds.clear()

         # Latency threshold configuration
         if modifyBitString & MODIFY_LATENCY_THRESHOLD:
            latencyThreshold = profileTxQConfig.latencyThreshold
            intfTxQConfig.latencyThreshold = latencyThreshold
         if modifyBitString & MODIFY_LATENCY_THRESHOLD and \
            latencyThreshold != tacLatencyThreshold():
            failed = False
            if mode and checkCliBlocking:
               failed = cliBlockingFail( mode,
                                         timestamp,
                                         tacFeatureName.latencyThreshold,
                                         "Latency Threshold" )
            if failed:
               del qosInputProfileConfig.intfConfig[ intfName ]
               if prevProfile:
                  return applyPrevProfileOrRollBack( profileName,
                                                     prevProfile,
                                                     intfName )
               return None

   if modifyBitString & MODIFY_PFC:
      if profile.pfcPortConfig:
         if isLagPort( intfName ):
            mode.addWarning( "PFC configuration is not supported on "
                             "Port-Channel" )
         else:
            if not intfConfig:
               intfConfig = qosInputProfileConfig.intfConfig.newMember(
                  intfName )
            else:
               intfConfig = qosInputProfileConfig.intfConfig[ intfName ]
            if not intfConfig.pfcPortConfig:
               intfConfig.pfcPortConfig = \
                   ( intfName, profile.pfcPortConfig.enabled, )
            pfcPortConfig = intfConfig.pfcPortConfig
            pfcPortConfig.enabled = profile.pfcPortConfig.enabled
            pfcPortConfig.priorities = profile.pfcPortConfig.priorities
            pfcPortConfig.watchdogEnabled = \
               profile.pfcPortConfig.watchdogEnabled
            pfcPortConfig.portTimerConfig = \
               profile.pfcPortConfig.portTimerConfig
            pfcPortConfig.watchdogPortAction = \
               profile.pfcPortConfig.watchdogPortAction
            intfConfig.fabricPfcDlb = profile.fabricPfcDlb
      elif intfConfig and intfConfig.pfcPortConfig:
         intfConfig.pfcPortConfig = None
         intfConfig.fabricPfcDlb = 0
   if profile.servicePolicyConfig:
      if intfName == fabricIntfName:
         mode.addWarning( "Service Policy is not supported on Fabric" )
      else:
         key = profile.servicePolicyConfig.key
         mapType = mapTypeFromEnum( key.type )
         direction = directionFromEnum( key.direction )
         pmapName = key.pmapName
         rollBackProfile = setServicePolicy( mode, None, mapType, pmapName,
                                          direction, intfName, profile=True )
         if mode and checkCliBlocking and rollBackProfile == ROLLBACK_PROFILE:
            if intfName in qosInputProfileConfig.intfConfig:
               del qosInputProfileConfig.intfConfig[ intfName ]
            if key in qosInputProfileConfig.servicePolicyConfig and \
                      intfName in qosInputProfileConfig.servicePolicyConfig[
                  key ].intfIds:
               del qosInputProfileConfig.servicePolicyConfig[ key ].\
                   intfIds[ intfName ]
               if len( qosInputProfileConfig.servicePolicyConfig[ key ].
                          intfIds ) == 0:
                  del qosInputProfileConfig.servicePolicyConfig[ key ]
            if prevProfile:
               return applyPrevProfileOrRollBack( profileName,
                                                  prevProfile, intfName )
            return None
   else:
      for spKey in qosInputProfileConfig.servicePolicyConfig:
         spConfig = qosInputProfileConfig.servicePolicyConfig[ spKey ]
         if intfName in spConfig.intfIds:
            delServicePolicy( None, spKey.type, spKey.pmapName,
                              spKey.direction, intfName, profile=True )
   intfToProfileMap[ intfName ] = profileName
   setIntfConfig( intfName, profile=True )
   return "reachedEndOfFunction"

def applyProfile( mode=None, profileName=None, noOrDefaultKw=None,
                  interfaceName=None, checkCliBlocking=True,
                  modifyBitString=MODIFY_ALL ):
   intfList = []
   intfName = None
   prevProfile = None
   qosProfileConfig = profileConfigDir.config
   intfToProfileMap = profileConfigDir.intfToProfileMap
   if os.environ.get( 'SKIP_CLI_BLOCK_CHECK' ):
      checkCliBlocking = False
   # This function is called through the cli in IntfConfigMode
   # and Fabric Mode. It can alos be called from the commit()
   # function in QosProfileModeContext. In IntfConfigMode, we
   # can get the applied intfs as mode.intf, while in Fabric mode,
   # we iterate over fabricIntfConfigDir to get the applied intfs.
   # When called from commit(), the intfName is passed as an argument
   applicableModes = [ IntfCli.IntfConfigMode, FabricMode ]
   if mode:
      if isinstance( mode, IntfCli.IntfConfigMode ):
         intfList = [ mode.intf ]
      elif isinstance( mode, FabricMode ):
         for intf in fabricIntfConfigDir.intfConfig:
            intfList.append( fabricIntfConfigDir.intfConfig[ intf ] )
      elif interfaceName:
         intfList = [ interfaceName ]
   else:
      intfList = [ interfaceName ]
   if noOrDefaultKw:
      for intf in intfList:
         intfName = getIntfIdentity( intf )
         removeProfile( mode, profileName, intfName )
      return None
   if profileName in qosProfileConfig:
      for intf in intfList:
         k = applyProfileToIntf( intf, mode, profileName, checkCliBlocking,
                             modifyBitString, prevProfile, qosProfileConfig,
                             intfToProfileMap, applicableModes )
         if k != "reachedEndOfFunction":
            return k
   else:
      for intf in intfList:
         intfName = getIntfIdentity( intf )
         removeProfile( None, None, intfName )
         intfToProfileMap[ intfName ] = profileName
   return None

def pmapHasDropPrecedenceAction( pmapName ):
   pmap = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap.get( pmapName )
   action = tacActionType.actionSetDropPrecedence
   yellowAction = QosLib.validDpConfigValues()[ 0 ]
   if pmap:
      policyActionList = []
      for cmapName in pmap.classAction:
         policyActionList.append( pmap.classAction[ cmapName ].policyAction )
      policyActionList.append( pmap.classActionDefault.policyAction )
      for policyAction in policyActionList:
         if ( action in policyAction ) and (
               policyAction[ action ].value == yellowAction ):
            # Yellow action is not supported along with ecnAllowNonect
            return True
   return False

def appliedUnsupportedOutSPAction( pmapName ):
   actionsAndFlag = [
      [ tacActionType.actionSetCos, qosHwStatus.cosActionInOutSpSupported,
        "set cos" ],
      [ tacActionType.actionSetTc, qosHwStatus.tcActionInOutSpSupported,
        "set traffic-class" ],
      [ tacActionType.actionSetDropPrecedence, qosHwStatus.dpActionInOutSpSupported,
        "set drop-precedence" ],
   ]
   unsupportedAction = []
   pmap = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap.get( pmapName )
   if pmap:
      policyActionList = []
      for cmapName in pmap.classAction:
         policyActionList += pmap.classAction[ cmapName ].policyAction
      policyActionList += pmap.classActionDefault.policyAction
      for action, flag, name in actionsAndFlag:
         if not flag and ( action in policyActionList ):
            unsupportedAction.append( name )
   return unsupportedAction

def isDscpActionInPmap( pmapName ):
   pmap = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap.get( pmapName )
   if pmap:
      policyActionList = []
      for cmapName in pmap.classAction:
         policyActionList += pmap.classAction[ cmapName ].policyAction
      policyActionList += pmap.classActionDefault.policyAction
      if tacActionType.actionSetDscp in policyActionList:
         return True
   return False

def setServicePolicy( mode, noOrDefaultKw=None, mapType='qos', pmapName=None,
                      direction=None, intfName=None, profile=False ):

   if not intfName:
      intfName = mode.intf.name
   mapType = mapTypeToEnum( mapType )
   direction = directionToEnum( direction )
   if noOrDefaultKw:
      delServicePolicy( mode, mapType, pmapName, direction, intfName,
                        profile=profile )
      if not mode.session.inConfigSession():
         currTime = getCurrentTimeForConfigUpdate()
         description = "service-policy to be detached"
         cliBlockingToApplyConfigChange( mode, currTime, tacFeatureName.others,
                                         description,
                                         policyMap=True )
   else:
      if direction == tacDirection.output:
         unSupportedActions = appliedUnsupportedOutSPAction( pmapName )
         error = "Following action(s) are not supported on output service policies:"
         if unSupportedActions:
            for actions in unSupportedActions:
               error += "\n" + actions
            mode.addError( error )
            if profile:
               return ROLLBACK_PROFILE
            return None

      if direction == tacDirection.output and \
            isinstance( mode.intf, VlanIntfCli.VlanIntf ) and \
            isPolicerInPmap( cliQosAclConfig, pmapName ) and \
            not qosAclHwStatus.egressPolicerSupported:
         mode.addError( "Egress Policing not supported on this platform" )
         if profile:
            return ROLLBACK_PROFILE
         return None

      if direction == tacDirection.output and \
            isinstance( mode.intf, VlanIntfCli.VlanIntf ) and \
            isDscpActionInPmap( pmapName ) and \
            not qosAclHwStatus.egressDscpActionSupported:
         mode.addError( "Dscp action on egress not supported on this platform" )
         if profile:
            return ROLLBACK_PROFILE
         return None

      if isLagPort( intfName ) and isPolicerInPmap( cliQosAclConfig, pmapName ) and \
            not qosAclHwStatus.policerOnLagSupported:
         mode.addError( "Policing on Lag not supported on this platform" )
         if profile:
            return ROLLBACK_PROFILE
         return None
      elif qosAclHwStatus.vlanMatchInClassMapSupported and \
            isSvi( intfName ) and isL2ParamsMatchInPmap( cliQosAclConfig, pmapName,
                                                         checkFor=[ 'vlan' ] ):
         errorMsg = "VLAN match in class-maps is not supported in QoS policy-maps"
         errorMsg += " applied to VLAN interfaces."
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return None
      elif isinstance( mode, IntfCli.IntfConfigMode ) and \
           isinstance( mode.intf, VxlanCli.VxlanIntf ) and \
           qosAclHwStatus.vxlanPolicyQosSupported and \
           not isPmapSupportedForVxlan( cliQosAclConfig, pmapName ):
         errorMsg = "Only VLAN match is supported in QoS policy-map(s) applied to"
         errorMsg += " a VxLAN interface"
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return None
      elif qosInputConfig.ecnAllowNonEct and pmapHasDropPrecedenceAction(
            pmapName ) and not qosStatus.ecnAllowNonEctWithDropPrecedence:
         errorMsg = "Policy-map action drop-precedence and allow "
         errorMsg += "non-ect cannot be configured together."
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return None
      elif tacSubIntfId.isSubIntfId( intfName ) and direction == tacDirection.input\
            and not qosAclHwStatus.ingressSubintfPolicyQosSupported:
         errorMsg = "Policy-map on ingress sub-interface is not supported on this"
         errorMsg += " platform"
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return None
      elif tacSubIntfId.isSubIntfId( intfName ) and direction == tacDirection.output\
            and not qosAclHwStatus.egressSubintfRateLimitSupported:
         errorMsg = "Policy-map on egress sub interface is not supported on this"
         errorMsg += " platform"
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return None

      rollBackProfile = addServicePolicy( mode, mapType, pmapName, direction,
                                          intfName, profile=profile )
      if profile:
         return rollBackProfile
   return None

class PolicyMapClassMode( PolicyMapClassModeBase, BasicCli.ConfigModeBase ):
   name = "Policy Map Class Configuration"

   # Each mode object has a session object. We associate the policymap
   # context with the mode object.
   def __init__( self, parent, session, context ):
      self.pmapClassContext = context
      self.pmapName = context.pmapName()
      self.cmapName = context.policyClassName()
      self.mapType_ = context.mapType()
      param = ( self.mapType_, self.pmapName, self.cmapName )
      PolicyMapClassModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'PolicyMapClassMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.subConfig = None
      self.pmapClassContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.pmapClassContext is None:
         t0( 'commitContext has no context' )
         return
      context = self.pmapClassContext
      self.pmapClassContext = None

      if self.mapType_ == pdpMapType and \
         self.pmapName == tacPMapNm.defaultPdpPmapName:
         # Do not commit any configuration of default-pdp-policy classes.
         # This should happen only when replaying a config saved from the output
         # of 'show running-config all', because configuring default-pdp-policy
         # is guarded.
         return
      context.commit()

   # print one or all policy maps
   @staticmethod
   def showPolicyMapClass( mode, args ):
      pmapName = args.get( 'PMAP', 'copp-system-policy' )
      cmapName = args[ 'CMAP' ]
      mapType = args.get( 'copp' ) or args.get( 'control-plane' ) or \
                args.get( 'pdp', 'qos' )
      mapType = 'control-plane' if mapType == 'copp' else mapType
      emapType = mapTypeToEnum( mapType )
      policyMapAllModel = PolicyMapAllModel()
      classMapWrapper = ClassMapWrapper()
      policyMapsContainer = PMapModelContainer( qosConfig, gv.qosAclConfig,
                                                qosStatus, qosHwStatus,
                                                qosSliceHwStatus, qosAclHwStatus,
                                                qosAclSliceHwStatus, emapType, None,
                                                hwEpochStatus, None,
                                                policyMapAllModel )
      if pmapName is None:
         pmapName = mode.pmapName
      if cmapName is None:
         cmapName = mode.cmapName
      policyMapsContainer.populatePolicyMapClassMap( pmapName, cmapName )
      if pmapName in policyMapAllModel.policyMaps:
         pMap = policyMapAllModel.policyMaps[ pmapName ]
         if cmapName in pMap.classMaps:
            classMapWrapper.classMap = pMap.classMaps[ cmapName ]
            return classMapWrapper
         else:
            return classMapWrapper
      else:
         return classMapWrapper

   def configurePolicer( self, mode, no, policerMode='committed', name=None,
                         cir=None, cirUnit=None, bc=None, bcUnit=None,
                         policerYellowAction=None, policerYellowValue=None, pir=None,
                         pirUnit=None, be=None, beUnit=None, policerRedAction=None,
                         cmdVersion=1, default=False ):
      context = self.pmapClassContext
      classAction = context.currentEntry()
      if not no and not default:
         if QosLib.isDefaultClass( self.mapType_, classAction.name ):
            if not qosAclHwStatus.policerInDefaultClassV4V6Supported:
               ipv4ClassCount, ipv6ClassCount = ipClassCounts( context.map_ )
               if ipv4ClassCount > 0 and ipv6ClassCount > 0:
                  self.addError( 'Policer in default-class is not supported when '
                                 'both ipv4 class-maps and ipv6 class-maps are '
                                 'present in policy map. To configure policer in '
                                 'default-class, remove either all ipv4 class-maps'
                                 ' or all ipv6 class-maps from the policy-map' )
                  return
         if tacActionType.actionSetDrop in classAction.policyAction:
            self.addError( 'Drop action not supported when policer enabled' )
            return
         if cirUnit is None:
            if bcUnit == tacBurstUnit.burstUnitPackets:
               cirUnit = tacRateUnit.rateUnitPps
            else:
               cirUnit = tacRateUnit.rateUnitbps
         if bcUnit is None:
            if cirUnit == tacRateUnit.rateUnitPps:
               bcUnit = tacBurstUnit.burstUnitPackets
            else:
               bcUnit = tacBurstUnit.burstUnitBytes
         if pirUnit is None:
            if cirUnit != tacRateUnit.rateUnitPps:
               pirUnit = tacRateUnit.rateUnitbps
            else:
               pirUnit = tacRateUnit.rateUnitPps
         if beUnit is None:
            if pirUnit != tacRateUnit.rateUnitPps:
               beUnit = tacBurstUnit.burstUnitBytes
            else:
               beUnit = tacBurstUnit.burstUnitPackets

         if policerMode == 'committed':
            pir = be = 0
         if( not name and not mode.session_.startupConfig() and
             not( self.mapType_ == pdpMapType and
                  self.pmapName == tacPMapNm.defaultPdpPmapName ) and
             policerOutOfRange( self, cir, cirUnit, bc, bcUnit,
                                pir, pirUnit, be, beUnit, policerMode ) ):
            # When in startup-config, or for default-pdp-policy,
            # ignore checks for policer rates and burst-sizes.
            return
         if isLagInServicePolicy( qosInputConfig, qosInputProfileConfig,
                                  self.pmapName ):
            if not( qosAclHwStatus.policerOnLagSupported or not
                  ( qosAclHwStatus.hwInitialized or
                  mode.session_.guardsEnabled() ) ):
               mode.addError( CliError[ 'lagPolicingNotSupported' ] )
               return
            # Per port CoPP not supported on Port-Channel
            if qosHwStatus.intfCoppSupported and self.mapType == coppMapType:
               mode.addError( CliError[ 'lagPolicingNotSupported' ] )
               return

         if isSviInServicePolicyOutDirection( qosInputConfig, qosInputProfileConfig,
                                              self.pmapName ) and \
               not qosAclHwStatus.egressPolicerSupported:
            mode.addError( "Egress Policing not supported on this platform" )
            return

         policerName = name if name else classAction.name
         if name:
            cir = bc = 0
         classAction.policer = None
         classAction.policer = ( policerName, cir, bc, pir, be )
         classAction.policer.named = bool( name )
         classAction.policer.cir = cir
         classAction.policer.cirUnit = cirUnit
         classAction.policer.bc = bc
         classAction.policer.bcUnit = bcUnit
         classAction.policer.pir = pir
         classAction.policer.pirUnit = pirUnit
         classAction.policer.be = be
         classAction.policer.beUnit = beUnit
         classAction.policer.cmdVersion = cmdVersion
         if policerYellowAction:
            if policerYellowAction == "drop-precedence":
               action = tacActionType.actionSetDropPrecedence
               classAction.policer.yellowActions.newMember( action )
            if policerYellowAction == "dscp":
               action = tacActionType.actionSetDscp
               classAction.policer.yellowActions.newMember( action )
               if isinstance( policerYellowValue, str ):
                  classAction.dscpConfiguredAsNameYellow = True
                  policerYellowValue, isValid = AclCliLib.dscpValueFromCli( self,
                                                               policerYellowValue )
                  assert isValid
               classAction.policer.yellowActions[ action ].value = \
                                                                   policerYellowValue

         if self.mapType_ == pdpMapType and \
            pdpPolicerYellowActionNeeded( getDefaultPdpPmapCfg() ):
            # If policer in PDP policy class should have yellow action (drop),
            # do not forget to add it when configuring the police action.
            # Currently, adding action drop in yellowActions is enough for
            # the required platforms (e.g Strata).
            action = tacActionType.actionSetDrop
            if action not in classAction.policer.yellowActions:
               classAction.policer.yellowActions.newMember( action )

         # set default red actions to Drop
         action = tacActionType.actionSetDrop
         if action not in classAction.policer.redActions:
            classAction.policer.redActions.newMember( action )
            classAction.policer.redActions[ action ].value = 1
         classAction.count = False
      else:
         if self.mapType_ == pdpMapType:
            # 'no/default police' in a PDP policy class will revert to the action for
            # the class specified in the default-pdp-policy.
            src = None
            _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
            if QosLib.isDefaultClass( self.mapType_, classAction.name ):
               src = _defaultPdpPmapCfg.classActionDefault
            else:
               src = _defaultPdpPmapCfg.classAction[ classAction.name ]
            copyClassAction( src, classAction )
         else:
            # no police
            if classAction.policer:
               classAction.policer = None

class PolicyMapClassModeQos( PolicyMapClassMode ):
   name = "Policy Map Qos Class Configuration"

   def outputActionUnsupportedString( self, actionSet ):
      return f"Action set {actionSet} is not supported in output service policies"

   def isPmapApplied( self ):
      pmapName = self.pmapClassContext.pmapName_
      spConfigs = [ qosInputConfig.servicePolicyConfig,
                    qosInputProfileConfig.servicePolicyConfig ]
      for spConfig in spConfigs:
         for key in spConfig:
            if key.pmapName == pmapName:
               return True
         return False

   def isPmapAppliedInOutDirection( self ):
      pmapName = self.pmapClassContext.pmapName_
      spConfigs = [ qosInputConfig.servicePolicyConfig,
                    qosInputProfileConfig.servicePolicyConfig ]
      for spConfig in spConfigs:
         for key in spConfig:
            if key.pmapName == pmapName and key.direction == tacDirection.output:
               return True
         return False

   def configureSetU32( self, no, value, action ):
      context = self.pmapClassContext
      classAction = context.currentEntry()
      # If a class-map inside a policy-map has 'police' action and the match
      # type is 'mac' return True, False otherwise.
      cmaps = gv.qosAclConfig.cmapType[ QosLib.qosMapType ].cmap
      if qosAclHwStatus.matchMacInClassMapSupported and \
         not qosAclHwStatus.matchMacInClassMapAllActionSupported and \
         context.cmapName_ in cmaps and \
         'matchMacAccessGroup' in cmaps[ context.cmapName_ ].match:
         errorMsg = ( f'Policy-map: {context.pmapName_} configured has '
                      f'invalid action: {action}, '
                      f'reconfigure the policy-map to use police action' )
         self.addError( errorMsg )
         return False

      if not no:
         if action not in classAction.policyAction:
            classAction.policyAction.newMember( action )
         classAction.policyAction[ action ].value = value
      else:
         if action in classAction.policyAction:
            del classAction.policyAction[ action ]
      return True

   def configureSetDscp( self, args ):
      if isSviInServicePolicyOutDirection( qosInputConfig, qosInputProfileConfig,
                                          self.pmapName ) and \
            not qosAclHwStatus.egressDscpActionSupported:
         self.addError( "Dscp action on egress not supported on this platform" )
         return False

      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'DSCP_NAME' ) or args.get( 'DSCP' )

      classAction = self.pmapClassContext.currentEntry()
      if no:
         classAction.dscpConfiguredAsName = False
         return self.configureSetU32( no, value, tacActionType.actionSetDscp )

      if isinstance( value, str ):
         # DSCP value provided as a name-string.
         classAction.dscpConfiguredAsName = True
         value, isValid = AclCliLib.dscpValueFromCli( self, value )
         assert isValid
         return self.configureSetU32( no, value, tacActionType.actionSetDscp )
      else:
         # DSCP value provided as a number.
         classAction.dscpConfiguredAsName = False
         return self.configureSetU32( no, value, tacActionType.actionSetDscp )

   def configureSetCos( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'COS' )
      if ( not qosHwStatus.cosActionInOutSpSupported ) and \
         self.isPmapAppliedInOutDirection():
         self.addError( self.outputActionUnsupportedString( "cos" ) )
         return True
      return self.configureSetU32( no, value, tacActionType.actionSetCos )

   def configureSetTc( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'TRAFFIC_CLASS' )
      if ( not qosHwStatus.tcActionInOutSpSupported ) and \
         self.isPmapAppliedInOutDirection():
         self.addError( self.outputActionUnsupportedString( "traffic-class" ) )
         return True
      return self.configureSetU32( no, value, tacActionType.actionSetTc )

   def configureSetDrop( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      context = self.pmapClassContext
      classAction = context.currentEntry()
      if classAction.policer is not None:
         self.addError( 'Drop action not supported when policer enabled' )
         return True
      return self.configureSetU32( no, 1, tacActionType.actionSetDrop )

   def configureSetDropPrecedence( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'DROP_PRECEDENCE' )
      if ( not qosHwStatus.dpActionInOutSpSupported ) and \
         self.isPmapAppliedInOutDirection():
         self.addError( self.outputActionUnsupportedString( "drop-precedence" ) )
         return True

      yellowAction = QosLib.validDpConfigValues()[ 0 ]

      if not qosStatus.ecnAllowNonEctWithDropPrecedence:
         if qosInputConfig.ecnAllowNonEct and value == yellowAction and \
            self.isPmapApplied():
            self.addError( "Policy-map action drop-precedence and allow"
                           " non-ect cannot be configured together." )
            return True
      return self.configureSetU32( no, value, tacActionType.actionSetDropPrecedence )

# ------------------------------------------------------------------------
# show policy-map interface <interface> type qos
# ------------------------------------------------------------------------
def waitForCountersUpdate( mode, mapType ):
   cliCounterConfig.pmapCounterUpdateRequestTime = Tac.now()  # monotonic time
   if mapType == 'control-plane' and \
      not qosAclHwStatus.coppDynamicClassSupported and \
      not qosAclHwStatus.intfCoppSupported:
      hwStatus = qosSliceHwStatus
   else:
      hwStatus = qosAclSliceHwStatus

   def countersUpdated():
      status = True
      if hwStatus == qosSliceHwStatus:
         for sliceHwStatus in qosSliceHwStatus.values():
            if sliceHwStatus.pmapCounterUpdateTime < \
                  cliCounterConfig.pmapCounterUpdateRequestTime:
               status = False
               break
      else:
         for aclSliceHwStatus in qosAclSliceHwStatus.values():
            if aclSliceHwStatus.pmapCounterUpdateTime < \
                  cliCounterConfig.pmapCounterUpdateRequestTime:
               status = False
               break
      return status

   # Check if we have any policy maps of type mapType configured. Prevents
   # "show policy-map counters" from timing out when there are no qos policy-maps
   # configured.
   emapType = mapTypeToEnum( mapType )
   mapTypePolicyPresent = False
   if hwStatus == qosAclSliceHwStatus:
      for aclSliceHwStatus in hwStatus.values():
         pmapType = aclSliceHwStatus.pmapType.get( emapType, None )
         if pmapType is not None and len( pmapType.pmap ) > 0:
            mapTypePolicyPresent = True
            break
   else:
      for sliceHwStatus in hwStatus.values():
         pmapType = sliceHwStatus.pmapType.get( emapType, None )
         if pmapType is not None and len( pmapType.pmap ) > 0:
            # Static CoPP is not divergent so can break the loop
            # if pmapType is present in any of the slices.
            mapTypePolicyPresent = True
            break
   if not mapTypePolicyPresent:
      # change mapType from qos to QoS for better display
      if mapType == 'qos':
         mapType = 'QoS'
      mode.addWarning( f"No {mapType} policy map applied on any interface" )
      return

   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Warning: displaying stale counters" )

def macClassCounts( pmap, cmapExclusionList=None ):
   if cmapExclusionList is None:
      cmapExclusionList = []
   cmaps = cliQosAclConfig.cmapType[ QosLib.qosMapType ].cmap
   macClassMapCount = 0
   for classPriority in pmap.classPrio:
      cmapName = pmap.classPrio[ classPriority ].cmapName
      if cmapName in cmapExclusionList or cmapName not in cmaps:
         continue
      if 'matchMacAccessGroup' in cmaps[ cmapName ].match:
         macClassMapCount += 1
   return macClassMapCount

def isOtherPolicyActionSet( pmaps, pmapName, cmapName ):
   for action in pmaps[ pmapName ].classAction[ cmapName ].policyAction:
      if pmaps[ pmapName ].classAction[ cmapName ].policyAction[ action ]:
         return True
   return False

def cmapPresentInPolicyMap( pmapName, cmap ):
   pmaps = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap
   if not pmapName in pmaps:
      return False

   for classPrio in pmaps[ pmapName ].classPrio:
      if cmap.name == pmaps[ pmapName ].classPrio[ classPrio ].cmapName:
         return True
   return False

def policerInDefaultClass( pmap ):
   classActionDefault = pmap.classActionDefault
   if classActionDefault and classActionDefault.policer:
      return True
   return False

def allowMatchOptionInClassMap( cmap, matchType, vlan=False, cos=False,
                                innerVlan=False, innerCos=False, dei=False ):
   pmaps = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap
   allow = True
   pmapNames = []
   macAclClassCounts = []
   ipv6ClassCounts = []
   vxlanIntfPresent = False
   for pmapName in pmaps:
      cmapPresentInPmap = cmapPresentInPolicyMap( pmapName, cmap )
      if cmapPresentInPmap and \
            not qosAclHwStatus.policerInDefaultClassV4V6Supported \
            and policerInDefaultClass( pmaps[ pmapName ] ):
          # If the class-map is present in a policy map that has a policer in
          # class-default and the match type being configured in the class-map
          # is different from the match type of the existing classe-maps in the
          # policy-map, return False. True Otherwise.
         ipv4ClassCount, ipv6ClassCount = ipClassCounts( pmaps[ pmapName ],
                                              cmapExclusionList=[ cmap.name ] )
         if matchType == 'ip' and ipv6ClassCount or  \
                matchType == 'ipv6' and ipv4ClassCount:
            allow = False
            pmapNames.append( pmapName )
      if cmapPresentInPmap and qosAclHwStatus.vlanMatchInClassMapSupported \
            and isSviInServicePolicy( qosInputConfig, qosInputProfileConfig,
                                      pmapName ):
         # If the class-map is present in the policy-map which is configured on
         # an SVI and the match type configured in the class-map is vlan/vlan+cos
         # return False. True otherwise.
         if matchType == 'l2Params' and vlan:
            allow = False
            pmapNames.append( pmapName )

      if cmapPresentInPmap and qosAclHwStatus.vxlanPolicyQosSupported \
         and isVxlanIntfInServicePolicy( qosInputConfig, qosInputProfileConfig,
                                         pmapName ):
         # If the class-map is present in the policy-map which is configured on
         # a VTI and the match type configured in the class-map is other than vlan
         # return False. True otherwise.
         vxlanIntfPresent = True
         if ( matchType != 'l2Params' or
              ( matchType == 'l2Params' and ( not vlan or cos or dei or innerCos or
                innerVlan ) ) ):
            allow = False
            pmapNames.append( pmapName )

      if cmapPresentInPmap and qosAclHwStatus.matchMacInClassMapSupported:
         if( ( matchType in ( 'mac', 'ipv6' ) ) and not
               qosAclHwStatus.matchMacInClassMapV6Supported ):
            # If mac acl on v6 group is not supported then we can't program
            # class map with ipv6 or mac if other class map in the same policy
            # group has vice versa config
            macClassCount = macClassCounts( pmaps[ pmapName ],
                                                   cmapExclusionList=[ cmap.name ] )
            ipv4ClassCount, ipv6ClassCount = ipClassCounts( pmaps[ pmapName ],
                                                   cmapExclusionList=[ cmap.name ] )
            if matchType == 'mac' and ipv6ClassCount or \
               matchType == 'ipv6' and macClassCount:
               allow = False
               pmapNames.append( pmapName )
               macAclClassCounts.append( macClassCount )
               ipv6ClassCounts.append( ipv6ClassCount )

         if matchType == 'mac' and \
            not qosAclHwStatus.matchMacInClassMapAllActionSupported and \
            isOtherPolicyActionSet( pmaps, pmapName, cmap.name ):
            # If a class-map inside a policy-map has only 'police' action and the
            # match type is 'mac' return True, False otherwise
            allow = False
            pmapNames.append( pmapName )

   return allow, pmapNames, macAclClassCounts, ipv6ClassCounts, vxlanIntfPresent

def allowClassInPolicyMap( self, pmap, cmapName ):
   allow = True
   cmaps = cliQosAclConfig.cmapType[ QosLib.qosMapType ].cmap
   if cmapName in cmaps:
      if not qosAclHwStatus.policerInDefaultClassV4V6Supported and \
            policerInDefaultClass( pmap ):
         ipv4ClassCount, ipv6ClassCount = ipClassCounts( pmap )
         cmapIpType = None
         # If the class-map is already configured with a match type and its
         # match type is different from the match type of the existing
         # class-maps in the policy-map, return False. True otherwise.
         if 'matchIpAccessGroup' in cmaps[ cmapName ].match:
            cmapIpType = 'ipv4'
         if 'matchIpv6AccessGroup' in cmaps[ cmapName ].match:
            cmapIpType = 'ipv6'
         if cmapIpType == 'ipv4' and ipv6ClassCount or \
            cmapIpType == 'ipv6' and ipv4ClassCount:
            allow = False
            errorMsg = 'When policer is in class-default, both ipv4 '
            errorMsg += 'class-maps and ipv6 class-maps are not supported '
            errorMsg += 'in the policy-map'
            self.addError( errorMsg )

      if ( qosAclHwStatus.vlanMatchInClassMapSupported and
           isSviInServicePolicy( qosInputConfig, qosInputProfileConfig, pmap.name )
           and 'matchL2Params' in cmaps[ cmapName ].match ):
         option = 'l2Params'
         vlanMatchPresent = False
         l2ParamsMatch = cmaps[ cmapName ].match[ matchOptionToEnum( 'l2Params' ) ]

         matchString = 'VLAN'
         # Check for presence of vlan match in the class-map
         if l2ParamsMatch.vlanValue:
            vlanMatchPresent = True

         if not vlanMatchPresent:
            # It can be allowed on SVI
            allow = True
         elif not cmapPresentInPolicyMap( pmap.name, cmaps[ cmapName ] ):
            allow = False
         else:
            # If the class-map is already configured with match type vlan
            # and if a policy-map which is configured on an SVI tries to
            # instantiate the class-map, return False. True otherwise.
            allow = allowMatchOptionInClassMap( cmaps[ cmapName ],
                                                option, vlanMatchPresent )
            allow = allow[ 0 ]
         if not allow:
            errorMsg = f"{matchString} match in class-maps is not supported in "
            errorMsg += "QoS policy-maps applied to VLAN interfaces."
            self.addError( errorMsg )

      if qosAclHwStatus.vxlanPolicyQosSupported and \
            isVxlanIntfInServicePolicy( qosInputConfig, qosInputProfileConfig,
                                        pmap.name ):
         cmap = cmaps[ cmapName ]
         vlanMatchPresent = False
         cosMatchPresent = False
         deiMatchPresent = False
         innerCosMatchPresent = False
         innerVlanMatchPresent = False

         option = cmap.match
         l2ParamsMatch = \
            cmap.match.get( matchOptionToEnum( 'l2Params' ), None )
         if l2ParamsMatch:
            if l2ParamsMatch.vlanValue:
               vlanMatchPresent = True
            if l2ParamsMatch.cosValue:
               cosMatchPresent = True
            if l2ParamsMatch.deiValue:
               deiMatchPresent = True
            if l2ParamsMatch.innerCosValue:
               innerCosMatchPresent = True
            if l2ParamsMatch.innerVlanValue:
               innerVlanMatchPresent = True

         if not cmapPresentInPolicyMap( pmap.name, cmap ):
            for matchType in cmap.match:
               if ( matchType != 'matchL2Params' or
                    ( matchType == 'matchL2Params' and
                      ( not vlanMatchPresent or cosMatchPresent or deiMatchPresent or
                        innerVlanMatchPresent or innerCosMatchPresent ) ) ):
                  allow = False
         else:
            # If the class-map is already configured with match type other than vlan
            # and if a policy-map which is configured on a VTI tries to
            # instantiate the class-map, return False. True otherwise.
            allow = allowMatchOptionInClassMap( cmap, option,
                                                vlanMatchPresent,
                                                cosMatchPresent,
                                                innerCos=innerCosMatchPresent,
                                                dei=deiMatchPresent )
            allow = allow[ 0 ]
         if not allow:
            errorMsg = "Only VLAN match is supported in QoS policy-map(s) applied"
            errorMsg += " to a VxLAN interface"
            self.addError( errorMsg )
   return allow

# -------------------------------------------------------------------------------
# policy-map class mode
# -------------------------------------------------------------------------------
class PolicyMapClassContext():
   def __init__( self, session, pmap, cmapName, insertBeforeClass, mapType ):
      self.session_ = session
      self.map_ = pmap
      self.pmapName_ = pmap.name
      self.cmapName_ = cmapName
      self.mapType_ = mapType
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.insertBeforeClass = insertBeforeClass

   def copyEditEntry( self, cmapName ):
      newClassAction = Tac.newInstance( 'Qos::ClassAction', cmapName )

      oldClassAction = None
      if QosLib.isDefaultClass( self.mapType_, cmapName ) is True:
         oldClassAction = self.map_.classActionDefault
      else:
         oldClassAction = self.map_.classAction[ cmapName ]
      assert oldClassAction is not None

      copyClassAction( oldClassAction, newClassAction )
      self.currentEntry_ = newClassAction

   def newEditEntry( self, cmapName ):
      newClassAction = Tac.newInstance( 'Qos::ClassAction', cmapName )
      self.currentEntry_ = newClassAction

   def pmapName( self ):
      return self.pmapName_

   def policyClassName( self ):
      return self.cmapName_

   def mapType( self ):
      return self.mapType_

   def pmap( self ):
      return self.map_  # may be None

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      cmapName = self.cmapName_

      def commitStaticORDefaultClass( cmapName ):
         clAction = None
         if QosLib.isDefaultClass( self.mapType_, cmapName ):
            clAction = self.map_.classActionDefault

         if clAction is not None:
            copyClassAction( self.currentEntry_, clAction )
            return True

         return False

      # For copp static / default class no re-ordering
      if not commitStaticORDefaultClass( cmapName ):
         self._commit()

   def _commit( self ):
      cmapName = self.cmapName_

      # Find the location of the class
      parentClassPrio = self.map_.classPrio
      numberOfClasses = len( parentClassPrio )
      if cmapName in self.map_.classAction:
         parentClassAction = self.map_.classAction[ cmapName ]
         for classPrio in parentClassPrio.values():
            if cmapName == classPrio.cmapName:
               currentPrio = classPrio.index
               break
         # Unless the class is being reordered, we want it to be in the same place
         expectedPrio = currentPrio
      else:
         parentClassAction = self.map_.classAction.newMember( cmapName )
         i = numberOfClasses + 1
         parentClassPrio.newMember( i )
         parentClassPrio[ i ].cmapName = cmapName
         currentPrio = i
         expectedPrio = i

      if self.insertBeforeClass and self.insertBeforeClass in self.map_.classAction:
         for classPrio in parentClassPrio.values():
            if self.insertBeforeClass == classPrio.cmapName:
               expectedPrio = classPrio.index
               break
      if expectedPrio > currentPrio:
         # We dont want the place occupied by insert before class,
         # we want a place just above it.
         expectedPrio -= 1

      # This loops shuffles the priority of classes.
      # Shifts all classes in classPrio  down or up till required class
      # has expected Priority.
      while currentPrio != expectedPrio:
         if currentPrio > expectedPrio:
            parentClassPrio[ currentPrio ].cmapName = \
                parentClassPrio[ currentPrio - 1 ].cmapName
            currentPrio -= 1

         if expectedPrio > currentPrio:
            parentClassPrio[ currentPrio ].cmapName = \
                parentClassPrio[ currentPrio + 1 ].cmapName
            currentPrio += 1

      parentClassPrio[ expectedPrio ].cmapName = cmapName
      copyClassAction( self.currentEntry_, parentClassAction )


_policyMapClassContextType = {}
# -------------------------------------------------------------------------------
# policy-map mode
# -------------------------------------------------------------------------------

class PolicyMapMode( PolicyMapModeBase, BasicCli.ConfigModeBase ):
   name = "Policy Map Configuration"
   # We already have a previous implementation for 'show active'.
   showActiveCmdRegistered_ = True

   # Each mode object has a session object. We associate the policymap
   # context with the mode object.
   def __init__( self, parent, session, context ):
      self.pmapContext = context
      self.pmapClassContext = None
      self.pmapName = context.pmapName()
      self.mapType_ = context.mapType_
      self.shared = context.shared
      param = ( self.mapType_, self.pmapName, self.shared )
      PolicyMapModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'PolicyMapMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.subConfig = None
      self.pmapContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.pmapContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.pmapContext
      self.pmapContext = None

      if self.mapType_ == pdpMapType and \
         self.pmapName == tacPMapNm.defaultPdpPmapName:
         # Do not commit any configuration of default-pdp-policy.
         # This should happen only when replaying a config saved from the output
         # of 'show running-config all', because configuring default-pdp-policy
         # is guarded.
         return
      context.commit()

   def getChildMode( self, mapType, context ):
      assert 0, 'Derived class should implement it'
      # assertion error will happen and the return statement will not be executed
      # without this return statement pylint will give error that we are assigning
      # return value from this function in checkNewPMapClass() function
      return False

   def gotoPolicyMapClassMode( self, pmapName, cmapName, insertBeforeClass ):
      def maxNumClassSupported():
         if self.mapType_ == coppMapType:
            return qosHwStatus.coppNumCMapSupported
         else:
            return qosAclHwStatus.policyQosNumCMapSupported

      def checkPMapClassCopp( cmapName, insertBeforeClass ):
         # Skip this check at startup
         if not self.session_.startupConfig():
            cpType = classMapCpType( cliQosAclConfig, cmapName )
            if cpType == tacClassMapCpType.cmapCpStatic:
               if not isCoppStaticClassSupported( cliQosAclConfig, qosSliceHwStatus,
                                                  cmapName ):
                  self.addError( CliError[ 'cMapNotSupported' ] % cmapName )
                  return False
               if cmapName == tacCMapNm.coppDrop:
                  self.addError( CliError[ 'cannotConfigureDropClass' ] % cmapName )
                  return False
            if insertBeforeClass:
               cpTypeInsertBefore = classMapCpType( cliQosAclConfig,
                                                    insertBeforeClass )
               if tacClassMapCpType.cmapCpStatic in ( cpType, cpTypeInsertBefore ):
                  self.addError( CliError[ 'cannotReorder' ] )
                  return False
         return True

      context = None
      t0( f'gotoPolicyMapClassMode {pmapName}, {cmapName}' )

      # Checks for Copp
      if self.mapType_ == coppMapType:
         if checkPMapClassCopp( cmapName, insertBeforeClass ) is False:
            return

      # Create the context
      if context is None:
         policyMapClassContextType = \
               _policyMapClassContextType.get( self.mapType_,
                                               PolicyMapClassContext )
         context = policyMapClassContextType( self.session_,
                                              self.pmapContext.currentEntry(),
                                              cmapName, insertBeforeClass,
                                              self.mapType_ )
      currentEntry = self.pmapContext.currentEntry()

      def pmapClassPresent( cmapName ):
         if cmapName in currentEntry.classAction:
            return True
         elif QosLib.isDefaultClass( self.mapType_, cmapName ) is True:
            return True
         return False

      def numOfClasses( thisCmapName ):
         curCmapNames = set()
         if qosHwStatus.intfCoppSupported and self.mapType_ == coppMapType:
            # Find out all different supported static classes and
            # configured dynamic classes that does not match this pmap
            pmapCol = cliQosAclConfig.pmapType[ self.mapType_ ].pmap
            for pmapName, pmap in pmapCol.items():
               if pmapName == self.pmapName:
                  continue
               for cmapName in pmap.classAction:
                  cpType = classMapCpType( cliQosAclConfig, cmapName )
                  if cpType == tacClassMapCpType.cmapCpStatic and \
                     not isCoppStaticClassSupported( cliQosAclConfig,
                                                     qosSliceHwStatus, cmapName ):
                     # Not all platforms supports every static classes
                     continue
                  curCmapNames.add( cmapName )

         # List of all (dynamic and static) classmaps that exist
         cmaps = cliQosAclConfig.cmapType[ self.mapType_ ].cmap
         for cmapName in currentEntry.classAction:
            if self.mapType_ == coppMapType:
               cpType = classMapCpType( cliQosAclConfig, cmapName )
               if ( cpType == tacClassMapCpType.cmapCpStatic and
                  not isCoppStaticClassSupported( cliQosAclConfig, qosSliceHwStatus,
                                                  cmapName ) ) or \
                  not isCpuQueueAllocated( qosSliceHwStatus, cmapName,
                                           self.pmapName ):
                  # Not all platforms supports every static classes
                  # Or on demand feature might not be allocated a cpu queue
                  continue
               # Check whether the cmap exists or not for coppMapType
               if cmapName in cmaps:
                  curCmapNames.add( cmapName )
               continue
            curCmapNames.add( cmapName )
         # Remove thisCmapName from curCmapNames as the caller expect
         # the num excluding thisCmapName
         curCmapNames.discard( thisCmapName )
         # +1 to account for default class
         return len( curCmapNames ) + 1

      def allowedClass( cmapName ):
         if self.mapType_ == pdpMapType:
            _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
            if _defaultPdpPmapCfg is None:
               return False
            if cmapName not in _defaultPdpPmapCfg.classAction:
               return False
         return True

      def checkNewPMapClass( cmapName ):
         # At startup, we skip this check.
         if not self.session_.startupConfig():
            # numClasses is  static classes + dynamic classes + default class
            numClasses = numOfClasses( cmapName )
            if self.mapType_ == coppMapType:
               cmapCpType = classMapCpType( cliQosAclConfig, cmapName )
               if tacClassMapCpType.cmapCpStatic != cmapCpType and \
                     not qosAclHwStatus.coppDynamicClassSupported:
                  self.addError( CliError[ 'dynamicClassError' ] )
                  return False

            maxClasses = maxNumClassSupported()
            # ignore the limit if it's 0 (uninitialized) and guards are disabled
            # this will allow the command during btests
            if ( ( numClasses >= maxClasses and
                 ( maxClasses or self.session.guardsEnabled() ) ) and
                  not ( self.mapType_ == coppMapType and
                        self.session.skipConfigCheck() ) ):
               self.addError( CliError[ 'maxClassError' ] % maxClasses )
               return False

            # At startup, platform may not be up. So, the defaultPdpPmapCfg
            # might not be populated. So we need to skip this check at startup time.
            if not allowedClass( cmapName ):
               self.addError( CliError[ 'cannotAddBuiltInCmap' ] )
               return False

         if self.mapType_ == qosMapType and not allowClassInPolicyMap( self,
                                                                       currentEntry,
                                                                       cmapName ):
            return False
         return True

      if pmapClassPresent( cmapName ) is True:
         context.copyEditEntry( cmapName )
      else:
         if checkNewPMapClass( cmapName ) is False:
            return
         if self.mapType_ == coppMapType:
            # See BUG 775770 for details on the CoPP issue this fixes.
            context.newEditEntry(
               cmapName, bypassCheck=self.session_.startupConfig() )
         else:
            context.newEditEntry( cmapName )

      self.pmapClassContext = context
      childMode = self.getChildMode( self.mapType_, context )
      self.session_.gotoChildMode( childMode )

   def noClassPrio( self, cmapName ):
      if cmapName not in self.pmapContext.currentEntry_.classAction:
         # cmap not a part of pmap. Return quietly.
         return
      # we are in PolicyMapMode. Update currentEntry of its Context.
      t0( "Deleting class ", cmapName )
      del self.pmapContext.currentEntry_.classAction[ cmapName ]
      deleteClassPrio( self.pmapContext.currentEntry_.classPrio, cmapName )

   def noCoppStaticClassPrio( self, cmapName ):
      if cmapName not in self.pmapContext.currentEntry_.classAction:
         # cmap not a part of pmap. Return quietly.
         return
      # we are in PolicyMapMode. Update currentEntry of its Context.
      t0( "Deleting class ", cmapName )
      del self.pmapContext.currentEntry_.classAction[ cmapName ]
      deleteClassPrio( self.pmapContext.currentEntry_.coppStaticClassPrio, cmapName )

   def noClassCopp( self, cmapName ):

      if classMapCpType( cliQosAclConfig, cmapName ) == \
            tacClassMapCpType.cmapCpDynamic:
         self.noClassPrio( cmapName )
      else:
         # class must be Static
         # restore the shaping parameters to default.
         if cmapName == tacCMapNm.coppDrop:
            self.addError( CliError[ 'cannotConfigureDropClass' ] % cmapName )
            return
         if QosLib.isDefaultClass( self.mapType_, cmapName ) is True:
            clAction = self.pmapContext.currentEntry_.classActionDefault
         else:
            clAction = self.pmapContext.currentEntry_.classAction.get( cmapName )

         # class-map is not present in policy-map configuration, nothing to do
         if not clAction:
            return

         if qosHwStatus.coppActionShaperSupported:
            shape = clAction.policyAction[ tacActionType.actionSetShape ].rate
            shape.val = tacActionRateType.invalid
            shape.rateUnit = tacRateUnit.rateUnitInvalid
            bw = clAction.policyAction[ tacActionType.actionSetBandwidth ].rate
            bw.val = tacActionRateType.invalid
            bw.rateUnit = tacRateUnit.rateUnitInvalid
         elif qosHwStatus.coppActionPolicerSupported:
            if clAction.policer:
               clAction.policer = None

         if self.pmapName != tacPMapNm.coppName and \
               qosHwStatus.intfCoppSupported:
            self.noCoppStaticClassPrio( cmapName )

   def noClassPdp( self, cmapName ):
      if cmapName == tacCMapNm.classDefault:
         self.addError( CliError[ 'cannotDeleteClassDefault' ] )
      else:
         self.addError( CliError[ 'cannotDeleteBuiltInCmap' ] )

   def noClass( self, cmapName ):
      lastMode = self.session_.modeOfLastPrompt()
      if isinstance( lastMode, PolicyMapClassMode ):
         # we are in PolicyMapClassMode. Set it's context to None.
         if cmapName == lastMode.cmapName:
            lastMode.pmapClassContext = None
      if self.mapType_ == coppMapType:
         self.noClassCopp( cmapName )
      elif self.mapType_ == pdpMapType:
         self.noClassPdp( cmapName )
      else:
         self.noClassPrio( cmapName )

   def setClass( self, no, cmapName, insertBeforeClass=None, default=False ):
      if no:
         self.noClass( cmapName )
      else:
         if self.mapType_ == pdpMapType and default:
            # 'default class <PDP policy class>' will revert to the action for
            # the class specified in the default-pdp-policy.
            src, dst = None, None
            _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
            if QosLib.isDefaultClass( self.mapType_, cmapName ):
               src = _defaultPdpPmapCfg.classActionDefault
               dst = self.pmapContext.currentEntry_.classActionDefault
            else:
               src = _defaultPdpPmapCfg.classAction[ cmapName ]
               self.pmapContext.currentEntry_.classAction.newMember( cmapName )
               dst = self.pmapContext.currentEntry_.classAction[ cmapName ]
            copyClassAction( src, dst )
         else:
            pmapName = self.pmapContext.pmapName()
            self.gotoPolicyMapClassMode( pmapName, cmapName, insertBeforeClass )

   # print one or all policy maps
   @staticmethod
   def showPolicyMap( mode, args ):
      pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
      mapType = args.get( 'copp' ) or args.get( 'control-plane' ) or \
                args.get( 'pdp', 'qos' )
      summary = 'summary' in args
      direction = args.get( 'DIRECTION' )
      policyMapAllModel = PolicyMapAllModel()
      mapType = 'control-plane' if mapType == 'copp' else mapType
      if mapType == 'control-plane' and not direction:
         direction = tacDirection.input
      emapType = mapTypeToEnum( mapType )
      policyMapsContainer = PMapModelContainer( qosConfig, gv.qosAclConfig,
                                                qosStatus, qosHwStatus,
                                                qosSliceHwStatus, qosAclHwStatus,
                                                qosAclSliceHwStatus, emapType, None,
                                                hwEpochStatus, None,
                                                policyMapAllModel )

      if 'counters' in args:
         policyMapsContainer.counterDetail = True
         policyMapsContainer.direction_ = direction
         waitForCountersUpdate( mode, mapType )
         if pmapName is None:
            pmapList = sorted( gv.qosAclConfig.pmapType[ emapType ].pmap )
            for policyName in pmapList:
               policyMapsContainer.populatePolicyMapCounters( policyName )
         else:
            policyMapsContainer.populatePolicyMapCounters( pmapName )
         return policyMapAllModel

      if pmapName is None:
         policyMapsContainer.populateAll( summary )
      else:
         policyMapsContainer.populatePolicyMap( pmapName, summary )
      return policyMapAllModel

class PolicyMapModeQos( PolicyMapMode ):
   name = "Qos Policy Map Configuration"

   def getChildMode( self, mapType, context ):
      childMode = self.childMode( PolicyMapClassModeQos,
                                  context=context )
      return childMode

def deleteComments( mode, mapType, pmapName=None, cmapName=None ):
   if not cliQosAclConfig or not mode or not mapType:
      return
   emapType_ = mapTypeToEnum( mapType )
   mapTypeString = mapType
   if mapType == 'control-plane':
      mapTypeString = 'copp'
   commentKeys_ = []
   if pmapName:
      if mapType == 'qos':
         mapTypeString = 'quality-of-service'
      if emapType_ in cliQosAclConfig.pmapType:
         if pmapName not in cliQosAclConfig.pmapType[ emapType_ ].pmap:
            return
         cmapNames_ = []
         pmap_ = cliQosAclConfig.pmapType[ emapType_ ].pmap[ pmapName ]
         if cmapName:
            if isinstance( cmapName, str ):
               cmapNames_ = [ cmapName ]
         else:
            cmapNames_ = [ c.name for c in pmap_.classAction.values() ]
            cmapNames_.append( pmap_.classActionDefault.name )
         for cmapName_ in cmapNames_:
            commentKeys_.append( f'pmap-c-{mapType}-{pmapName}-{cmapName_}' )
            if mapTypeString != mapType:
               commentKeys_.append(
                  f'pmap-c-{mapTypeString}-{pmapName}-{cmapName_}' )
         if not cmapName:
            commentKeys_.append( f'pmap-{mapType}-{pmapName}' )
            if mapTypeString != mapType:
               commentKeys_.append( f'pmap-{mapTypeString}-{pmapName}' )
   elif cmapName:
      if emapType_ in cliQosAclConfig.cmapType:
         if cmapName not in cliQosAclConfig.cmapType[ emapType_ ].cmap:
            return
         commentKeys_.append( f'cmap-{mapType}-{cmapName}' )
         if mapTypeString != mapType:
            commentKeys_.append( f'cmap-{mapTypeString}-{cmapName}' )
   for commentKey_ in commentKeys_:
      BasicCliModes.removeCommentWithKey( commentKey_ )

def deletePolicyMap( mode, pmapName, mapType='qos', shared=None ):
   emapType = mapTypeToEnum( mapType )
   if mapType == 'pdp' and pmapName == tacPMapNm.defaultPdpPmapName:
      # Cannot delete default-pdp-policy
      mode.addError( f"{pmapName} cannot be deleted" )
      return False

   if pmapName in gv.qosAclConfig.pmapType[ emapType ].pmap:
      pmapShared = gv.qosAclConfig.pmapType[ emapType ].pmap[ pmapName ].shared
      if not pmapShared and shared:
         mode.addError( f"{pmapName} is unshared and cannot be deleted as shared"
                        " policy-map" )
         return False

   lastMode = mode.session_.modeOfLastPrompt()

   if isinstance( lastMode, PolicyMapClassMode ):
      lastMode.pmapClassContext = None
      lastMode = lastMode.parent_

   if isinstance( lastMode, PolicyMapMode ) and lastMode.pmapName == pmapName:
      lastMode.pmapContext = None

   if pmapName not in cliQosAclConfig.pmapType[ emapType ].pmap:
      return False

   deleteComments( mode, mapType, pmapName=pmapName )

   if mapType == 'control-plane' and pmapName == tacPMapNm.coppName:
      # Revert back the pmap to default state
      coppPmap = cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ]
      allClassActions = list( coppPmap.classAction.values() ) + \
                        [ coppPmap.classActionDefault ]
      for clAction in allClassActions:
         clAction.policer = None
         cmapCpType = classMapCpType( cliQosAclConfig, clAction.name )
         if cmapCpType == tacClassMapCpType.cmapCpStatic:
            shape = clAction.policyAction[ tacActionType.actionSetShape ].rate
            bw = clAction.policyAction[ tacActionType.actionSetBandwidth ].rate
            shape.val = tacActionRateType.invalid
            shape.rateUnit = tacRateUnit.rateUnitInvalid
            bw.val = tacActionRateType.invalid
            bw.rateUnit = tacRateUnit.rateUnitInvalid
         else:
            del coppPmap.classAction[ clAction.name ]

      coppPmap.classPrio.clear()

      # Update the config version
      coppPmap.uniqueId = Tac.Value( 'Qos::UniqueId' )
      coppPmap.version += 1
   else:
      del cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ]
      if not mode.session.inConfigSession():
         currTime = getCurrentTimeForConfigUpdate()
         description = "policy-map to be deleted"
         cliBlockingToApplyConfigChange( mode, currTime, tacFeatureName.others,
                                         description,
                                         policyMap=True )
   return True

def copyPolicyMap( src, dst, mapType, mode ):
   # This function is invoked
   #
   # Entering policy-map mode: copy from sysdb -> policy map editing copy
   # Committing from policy-map mode: copy from policy map editing copy -> sysdb

   # If we have removed the class action from the source, then it means
   # 'no' command was issued from the class action, therefore remove from
   # destination

   dst.dynamic = src.dynamic
   dst.shared = src.shared

   for cmap in dst.classAction:
      if cmap not in src.classAction:
         deleteComments( mode, mapTypeFromEnum( dst.type ), pmapName=dst.name,
               cmapName=cmap )
         del dst.classAction[ cmap ]

   allClasses = list( src.classAction )
   if mapType == coppMapType:
      allClasses += [ tacCMapNm.coppDefault ]
   else:
      allClasses += [ tacCMapNm.classDefault ]

   # Make a editing copy for the classes
   for cmap in allClasses:
      if QosLib.isDefaultClass( mapType, cmap ) is True:
         if dst.classDefault is None:
            dst.classDefault = ( cmap, mapType )
            # Do not set match criteria for per port copp
            # we wont be creating any acl in platform
            if not ( mapType == coppMapType and src.name != tacPMapNm.coppName ):
               # Revisit! default is ipv4 vs ipv6
               match = dst.classDefault.match.newMember( 'matchIpAccessGroup' )
               match.strValue = 'default'
         dst.classDefault.cpType = src.classDefault.cpType
         dst.classDefault.cpStaticType = src.classDefault.cpStaticType

         if dst.classActionDefault is None:
            dst.classActionDefault = ( cmap, )
         dstAction = dst.classActionDefault
         srcAction = src.classActionDefault
      else:
         srcAction = src.classAction[ cmap ]
         dstAction = dst.classAction.newMember( cmap )
      copyClassAction( srcAction, dstAction )

   # This is to avoid duplicate valued( by cmapName ) priority in src.classPrio.
   # Refer to Escalation97805 where a duplicated valued cmapPrio
   # resulted in Ale crash.
   cmapPrios = {}
   cmapNames = set()
   for cmapPrio, cmap in src.classPrio.items():
      if cmap.cmapName not in cmapNames:
         cmapPrios[ cmapPrio ] = cmap
         cmapNames.add( cmap.cmapName )
   for cmapPrio in src.classPrio:
      # Delete duplicate key
      if cmapPrio not in cmapPrios:
         del src.classPrio[ cmapPrio ]

   srcNumClasses = len( src.classPrio )
   dstNumClasses = len( dst.classPrio )

   # Shrink or expand classPrio to make room for new classes.
   while srcNumClasses != dstNumClasses:
      if srcNumClasses > dstNumClasses:
         dst.classPrio.newMember( srcNumClasses )
         dst.classPrio[ srcNumClasses ].cmapName = \
             dst.classPrio[ srcNumClasses ].cmapName
         srcNumClasses -= 1
      else:
         del dst.classPrio[ dstNumClasses ]
         dstNumClasses -= 1
   # Copy classPrio
   for index, classPrio in src.classPrio.items():
      dst.classPrio[ index ].cmapName = classPrio.cmapName

   # Shrink or expand coppStaticClassPrio to make room for new classes.
   srcStaticNumClasses = len( src.coppStaticClassPrio )
   dstStaticNumClasses = len( dst.coppStaticClassPrio )
   while srcStaticNumClasses != dstStaticNumClasses:
      if srcStaticNumClasses > dstStaticNumClasses:
         dst.coppStaticClassPrio.newMember( srcStaticNumClasses )
         dst.coppStaticClassPrio[ srcStaticNumClasses ].cmapName = \
             dst.coppStaticClassPrio[ srcStaticNumClasses ].cmapName
         srcStaticNumClasses -= 1
      else:
         del dst.coppStaticClassPrio[ dstStaticNumClasses ]
         dstStaticNumClasses -= 1

   # Copy coppStaticClassPrio
   for index, classPrio in src.coppStaticClassPrio.items():
      if index not in dst.coppStaticClassPrio:
         dst.coppStaticClassPrio.newMember( index )
      dst.coppStaticClassPrio[ index ].cmapName = classPrio.cmapName

def isQosPmapHwStatusOk( mode, aclName, aclType, aclUpdateType ):
   err = {}
   err[ 'error' ] = ''
   if not hwConfigAclVerificationSupported():
      t0( 'Not waiting for hardware. All good.' )
      return True, err

   # Bump up the version of the affected pmaps and wait
   # for the hardware status
   if aclUpdateType in ( AclCli.aclUpdateTypeCreate, AclCli.aclUpdateTypeModify ):
      currTime = getCurrentTimeForConfigUpdate()
      rc, errMsg = spWaitForHwStatusForAclChange( mode, aclName, aclType, currTime )
      if not rc:
         err[ 'error' ] = errMsg
         return rc, err

   return True, err


AclCli.registerAclConfigValidCallback( isQosPmapHwStatusOk )

def spWaitForHwStatusForAclChange( mode, aclName, aclType, updateTime,
                                   updateStr="ACL change" ):
   if aclType == 'ip':
      matchOption = tacMatchOption.matchIpAccessGroup
   elif aclType == 'ipv6':
      matchOption = tacMatchOption.matchIpv6AccessGroup
   elif aclType == 'l2Params':
      matchOption = tacMatchOption.matchL2Params
   elif aclType in ( 'dscpEcn', 'dscp', 'ecn' ):
      matchOption = tacMatchOption.matchDscpEcn
   elif aclType == 'mpls-traffic-class':
      matchOption = tacMatchOption.matchMplsTrafficClass
   elif aclType == 'mac':
      matchOption = tacMatchOption.matchMacAccessGroup
   else:
      return True, ""

   for cmapType in cliQosAclConfig.cmapType.values():
      for cmap in cmapType.cmap.values():
         cmapMatch = cmap.match.get( matchOption )
         if cmapMatch and cmapMatch.strValue == aclName:
            rc, errMsg = spWaitForHwStatusForCmapChange( mode, cmap.name, cmap.type,
                                                         updateTime, updateStr )
            if not rc:
               return rc, errMsg

   return True, ""

def spWaitForHwStatusForCmapChange( mode, cmapName, cmapType, updateTime,
                                    updateStr="class-map change" ):
   if not hwConfigAclVerificationSupported():
      return True, ""

   pmapType = cliQosAclConfig.pmapType.get( cmapType )
   if not pmapType:
      return True, ""
   if cmapType == coppMapType and not \
      qosAclHwStatus.coppDynamicClassSupported:
      return True, ""

   for pmap in pmapType.pmap.values():
      pmapAffected = False
      for classPrio in pmap.classPrio.values():
         if classPrio.cmapName == cmapName:
            pmapAffected = True
            break

      if pmapAffected:
         rc, errMsg = spWaitForHwStatusForPmapChange( mode, pmap.name, pmap.type,
                                                      updateTime, updateStr )
         if not rc:
            return rc, errMsg

   return True, ""

def spWaitForHwStatusForPmapChange( mode, pmapName, pmapType, updateTime,
                                    updateStr="policy-map change" ):
   if mode.session_.startupConfig():
      # no agent running, don't wait
      return True, ""

   if not hwConfigAclVerificationSupported():
      return True, ""

   if pmapType == coppMapType and not \
      qosAclHwStatus.coppDynamicClassSupported:
      return True, ""

   for direction in [ tacDirection.input, tacDirection.output ]:

      # wait for all of the slices to return program status
      rc, errMsg = waitForPMapHwPrgmStatus( mode, pmapName, pmapType,
                                            direction, updateTime,
                                            updateStr )
      if not rc:
         return rc, errMsg

   return True, ""

# -------------------------------------------------------------------------------
# policy-map mode
# -------------------------------------------------------------------------------
class PolicyMapContext():
   def __init__( self, mode, pmapName, pmapType, shared=False ):
      self.mode = mode
      self.map_ = None
      self.pmapName_ = pmapName
      self.mapType_ = pmapType
      self.shared = shared
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.previousEntry_ = None  # used for config rollback

      if pmapName in cliQosAclConfig.pmapType[ self.mapType_ ].pmap:
         self.map_ = cliQosAclConfig.pmapType[ self.mapType_ ].pmap[ pmapName ]

         # this instance will be used when config needs to be rolled back
         prevPolicyMap = Tac.newInstance( 'Qos::PolicyMapConfig', self.pmapName_,
                                           self.mapType_ )
         copyPolicyMap( self.map_, prevPolicyMap, self.mapType_, self.mode )
         self.previousEntry_ = prevPolicyMap

   def copyEditEntry( self, cmapName ):
      newPolicyMap = Tac.newInstance( 'Qos::PolicyMapConfig', self.pmapName_,
                                      self.mapType_ )
      copyPolicyMap( self.map_, newPolicyMap, self.mapType_, self.mode )
      self.currentEntry_ = newPolicyMap

   def newEditEntry( self, cmapName ):
      newPolicyMap = Tac.newInstance( 'Qos::PolicyMapConfig', self.pmapName_,
                                      self.mapType_ )
      newPolicyMap.shared = self.shared
      newPolicyMap.classDefault = ( cmapName, self.mapType_ )
      # Revisit! alway create default class-map for ipv4 type vs ipv6
      match = newPolicyMap.classDefault.match.newMember( 'matchIpAccessGroup' )
      match.strValue = 'default'
      newPolicyMap.classActionDefault = ( cmapName, )
      _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
      if self.mapType_ == pdpMapType and _defaultPdpPmapCfg:
         # Case when mapType is pdpMapType and platforms have populated
         # defaultPdpPmapCfg. If defaultPdpPmapCfg has not been populated yet,
         # the new user-defined policy would be empty( no classes or actions ).

         # Copy classAction
         for cmap in _defaultPdpPmapCfg.classAction:
            src = _defaultPdpPmapCfg.classAction[ cmap ]
            newPolicyMap.classAction.newMember( cmap )
            dst = newPolicyMap.classAction[ cmap ]
            copyClassAction( src, dst )
            if dst.policer:
               # Set cmdVersion of PDP policers to 2.
               dst.policer.cmdVersion = 2
         # Copy classPrio
         for index in _defaultPdpPmapCfg.classPrio:
            newPolicyMap.classPrio.newMember( index )
            newPolicyMap.classPrio[ index ].cmapName = \
               _defaultPdpPmapCfg.classPrio[ index ].cmapName
         src = _defaultPdpPmapCfg.classActionDefault
         dst = newPolicyMap.classActionDefault
         if src:
            # Copy class-default action from default-pdp-policy.
            copyClassAction( src, dst )
            if dst.policer:
               # Currently, all platforms have class-default action as 'count' for
               # PDP policy-maps in default-pdp-policy.
               # If any platform publishes class-default action as a policer action,
               # set cmdVersion to 2 as done for other built-in classes.
               dst.policer.cmdVersion = 2

      self.currentEntry_ = newPolicyMap

   def pmapName( self ):
      return self.pmapName_

   def pmap( self ):
      return self.map_  # may be None

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      # Commit current map. Create a new map if not exist
      # self.map_ is none when user creates pmap for the first time.
      # otherwise, if pmap already exists, self.map_ is assigned
      # appropriate config.
      pmapTypePtr = cliQosAclConfig.pmapType[ self.mapType_ ]
      if self.map_ is None:
         self.map_ = pmapTypePtr.pmap.newMember(
            self.pmapName_, pmapTypePtr.type )

      # we need to bump up the version if we change the pmap or
      # we failed to program it properly or not present in hwstatus
      bumpVersion = True
      prgmdPMapHwStatus = programmedPMapHwStatus( qosHwStatus, qosAclHwStatus,
                                                  qosSliceHwStatus,
                                                  qosAclSliceHwStatus,
                                                  self.mapType_,
                                                  self.pmapName_ )
      if prgmdPMapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
         if identicalPolicyMap( self.currentEntry_, self.map_ ):
            bumpVersion = False
      if bumpVersion:
         copyPolicyMap( self.currentEntry_, self.map_, self.mapType_, self.mode )
         currTime = getCurrentTimeForConfigUpdate()

         self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
         self.map_.version += 1

         rc, errMsg = spWaitForHwStatusForPmapChange( self.mode, self.pmapName_,
                                                      self.mapType_, currTime )
         if not rc:
            if errMsg == CLI_TIMEOUT:
               self.mode.addWarning( QosLib.qosTimeoutWarning() )
            else:
               # rollback
               self.mode.addError( f"Error: Cannot commit policy-map "
                                   f"{self.pmapName_}, {self.mapType_} ({errMsg})" )

               if self.previousEntry_:
                  t0( f"Reverting policy-map update for {self.pmapName_} " )
                  copyPolicyMap( self.previousEntry_, self.map_, self.mapType_,
                                 self.mode )
                  self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
                  self.map_.version += 1
               else:
                  t0( f"Deleting new policy-map: {self.pmapName_}" )
                  del pmapTypePtr.pmap[ self.pmapName_ ]

# PolicyMapCopyContext behaves like the PolicyMapContext except for the
# newEditEntry routine. In case of the copy context, the new edit entry
# should not be empty, instead, it should contain the configuration
# of the source policy it is to be copied from
class PolicyMapCopyContext( PolicyMapContext ):
   def __init__( self, mode, pmapName, srcPmap ):
      assert srcPmap
      super().__init__( mode, pmapName,
            srcPmap.type, srcPmap.shared )
      self.srcPmap_ = srcPmap

   def newEditEntry( self, cmapName ):
      t0( f'PolicyMapCopyContext newEditEntry called for {self.pmapName_}' )
      # We will first call the parent method to create a new policy-map
      super().newEditEntry( cmapName )
      # Once the policy-map is created, we copy the srcPmap contents into the
      # empty policy-map
      copyPolicyMap( self.srcPmap_, self.currentEntry_, self.mapType_, self.mode )

# -----------------------------------------------------------------
# The show command in 'config-pmap' mode
#
#              show active|pending|diff
# -----------------------------------------------------------------
def _showPMapList( pmapType, pmapName, pmap, output=None ):
   if pmap is None:
      return
   if output is None:
      output = sys.stdout

   def _printClassActionQos( classAction ):
      actions = classAction.policyAction
      for actionType in pmapQosActionTypes:
         if actionType in actions:
            if actionType in [ tacActionType.actionSetDscp,
                               tacActionType.actionSetCos,
                               tacActionType.actionSetDropPrecedence,
                               tacActionType.actionSetTc ]:
               actStr = actionFromEnum( actionType )
               action = actions[ actionType ]

               # For DSCP, the name-string should be printed, if configured.
               if ( actionType == tacActionType.actionSetDscp ) and \
                  classAction.dscpConfiguredAsName:
                  name = AclCliLib.dscpNameFromValue( action.value )
                  output.write( f"    set {actStr} {name}\n" )
               else:
                  output.write( f"    set {actStr} {action.value}\n" )

            elif actionType in [ tacActionType.actionSetDrop ]:
               output.write( "    drop\n" )

      policer = classAction.policer
      if policer:
         # BUG197856: Currently a named policer can only be a shared policer.
         # When we support policer profiles for unshared policers, we need to
         # update this check to look at shared and named attributes.
         if policer.named:
            output.write( f"    police shared {policer.name}" )
         else:
            output.write( f"    police rate {policer.cir} "
                          f"{rateUnitFromEnum( policer.cirUnit )} burst-size"
                          f" {policer.bc} {burstUnitFromEnum( policer.bcUnit )}" )
         if policer.pir:
            yellowActions = policer.yellowActions
            for actionType in yellowActions:
               actStr = actionFromEnum( actionType )
               action = yellowActions[ actionType ]
               if actionType in [ tacActionType.actionSetDscp,
                                  tacActionType.actionSetCos,
                                  tacActionType.actionSetDropPrecedence,
                                  tacActionType.actionSetTc ]:
                  output.write( f" action set {actStr} {action.value}" )
               else:
                  output.write( f" action set {actStr}" )
            output.write( f" rate {policer.pir} "
                          f"{rateUnitFromEnum( policer.pirUnit )} burst-size "
                          f"{policer.be} {burstUnitFromEnum( policer.beUnit )}\n" )
         else:
            output.write( "\n" )
      output.write( "\n" )

   def _printClassActionCopp( classAction ):
      actions = classAction.policyAction

      def _getStaticClass( classAction ):
         cpStaticType = classMapCpStaticType( cliQosAclConfig, classAction.name )
         hwCoppStaticClass = coppStaticClassFromHwStatus( qosSliceHwStatus )
         return hwCoppStaticClass[ cpStaticType ]

      if tacActionType.actionSetShape in actions:
         shape = actions[ tacActionType.actionSetShape ].rate
         if shape.val == tacActionRateType.noValue:
            output.write( "    no shape\n" )
         else:
            if shape.val != tacActionRateType.invalid:
               output.write( f"    shape {rateUnitFromEnum( shape.rateUnit )} "
                             f"{shape.val}\n" )
            else:
               staticClass = _getStaticClass( classAction )
               output.write( f"    shape "
                             f"{rateUnitFromEnum( staticClass.defaultRateUnit )} "
                             f"{staticClass.defaultMax}\n" )

      if tacActionType.actionSetBandwidth in actions:
         bandwidth = actions[ tacActionType.actionSetBandwidth ].rate
         if bandwidth.val == tacActionRateType.noValue:
            output.write( "    no bandwidth\n" )
         else:
            if bandwidth.val != tacActionRateType.invalid:
               output.write( f"    bandwidth "
                             f"{rateUnitFromEnum( bandwidth.rateUnit )} "
                             f"{bandwidth.val}\n" )
            else:
               staticClass = _getStaticClass( classAction )
               output.write( f"    bandwidth "
                             f"{rateUnitFromEnum( staticClass.defaultRateUnit )} "
                             f"{staticClass.defaultMin}\n" )

   def _printClassAction( classAction ):
      if classAction is None:
         return
      if classAction.policyAction is None:
         return
      if pmapType == coppMapType:
         _printClassActionCopp( classAction )
      else:
         _printClassActionQos( classAction )

   mapTypeString = mapTypeFromEnum( pmapType )
   if pmapType == coppMapType:
      mapTypeString = 'copp'
   elif pmapType == qosMapType:
      mapTypeString = 'quality-of-service'
   output.write( f"policy-map type {mapTypeString} "
                 f"{'shared ' if pmap.shared else ''}{pmapName}\n" )

   for _, classPrio in sorted( pmap.classPrio.items() ):
      cmapName = classPrio.cmapName
      outputStr = 'built-in ' if pmapType == pdpMapType and \
         cmapName in builtInClassMapNames else ''
      output.write( f"  class {outputStr}{cmapName}\n" )
      classAction = pmap.classAction[ cmapName ]
      _printClassAction( classAction )

   for _, coppStaticClassPrio in sorted(
         pmap.coppStaticClassPrio.items() ):
      cmapName = coppStaticClassPrio.cmapName
      if isCoppStaticClassSupported( cliQosAclConfig, qosSliceHwStatus, cmapName ):
         if cmapName != tacCMapNm.coppDrop:
            output.write( f"  class {cmapName}\n" )
            classAction = pmap.classAction[ cmapName ]
            _printClassAction( classAction )

   cmapName = defaultClassName( pmapType )
   output.write( f"  class {cmapName}\n" )
   classAction = pmap.classActionDefault
   _printClassAction( classAction )

def showPMapCurrent( mode ):
   _showPMapList( mode.mapType_, mode.pmapName, mode.pmapContext.currentEntry_ )

def showPMapActive( mode ):
   mode.showActive()

def showPMapDiff( mode ):
   # Generate diff between active and pending
   activeOutput = io.StringIO()
   _showPMapList( mode.mapType_, mode.pmapName, mode.pmapContext.map_,
                  output=activeOutput )
   pendingOutput = io.StringIO()
   _showPMapList( mode.mapType_, mode.pmapName, mode.pmapContext.currentEntry_,
                  output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue( ).splitlines( ),
                                pendingOutput.getvalue( ).splitlines( ),
                                lineterm='' )
   print( '\n'.join( list( diff ) ) )

def gotoPolicyMapModeCommon( mode, pmapName, mapType, cmapName, shared=None ):
   emapType = mapTypeToEnum( mapType )
   shared = bool( shared )
   context = PolicyMapContext( mode, pmapName, emapType, shared=shared )

   if pmapName in cliQosAclConfig.pmapType[ emapType ].pmap:
      pmapShared = cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ].shared
      sharedErrStr = ""
      currentSharedStr = "shared" if pmapShared else "unshared"
      if pmapShared and not shared:
         sharedErrStr = "unshared"
      elif not pmapShared and shared:
         sharedErrStr = "shared"
      if sharedErrStr != "":
         mode.addError( f"{pmapName} is {currentSharedStr} and cannot be configured"
                        f" as {sharedErrStr} policy-map" )
         return None
      context.copyEditEntry( cmapName )
      if cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ].dynamic:
         mode.addWarning( DYNAMIC_PMAP_EDIT_WARN )
   else:
      context.newEditEntry( cmapName )
   return context

# -----------------------------------------------------------------
# The show command in 'config-cmap' mode
#
#              show active|pending|diff
# -----------------------------------------------------------------

def _showCMapList( clMapType, clMapName, clMap, output=None ):
   if clMap is None:
      return
   if output is None:
      output = sys.stdout

   assert clMap.matchCondition == 'matchConditionAny'
   matchCondition = 'match-any'
   mapTypeString = mapTypeFromEnum( clMapType )
   if clMapType == coppMapType:
      mapTypeString = 'copp'
   output.write( f"class-map type {mapTypeString} {matchCondition} {clMap.name}\n" )
   for matchRule in clMap.match:
      ipStr = matchOptionFromEnum( matchRule )
      clMapMatch = clMap.match[ matchRule ]
      if matchRule == tacMatchOption.matchL2Params:
         outputString = "   match"
         vlanValue = clMapMatch.vlanValue
         cosValue = clMapMatch.cosValue
         innerVlanValue = clMapMatch.innerVlanValue

         if vlanValue is not None:
            if vlanValue.maskValid:
               outputString += f" vlan {clMapMatch.vlanValue.vlan} " \
                  f"0x{clMapMatch.vlanValue.vlanMask:03x}"
            else:
               ids = set()
               for key in vlanValue.vlanColl:
                  ids.update( list( range( key.min, key.max + 1 ) ) )
               vlanStr = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
               outputString += f" vlan {vlanStr}"

         if innerVlanValue is not None:
            if innerVlanValue.maskValid:
               outputString += f" vlan inner {clMapMatch.innerVlanValue.innerVlan}" \
                  f" 0x{clMapMatch.innerVlanValue.innerVlanMask:03x}"
            else:
               ids = set()
               for key in innerVlanValue.innerVlanColl:
                  ids.update( list( range( key.min, key.max + 1 ) ) )
               innerVlanStr = MultiRangeRule.multiRangeToCanonicalString(
                  list( ids ) )
               outputString += f" vlan inner {innerVlanStr}"

         if cosValue is not None:
            ids = set()
            for key in cosValue.cosColl:
               ids.update( list( range( key.min, key.max + 1 ) ) )
            cosStr = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
            outputString += f" cos {cosStr}"

         outputString += "\n"
         output.write( outputString )

      elif matchRule in [ tacMatchOption.matchDscpEcn ]:
         dscpValue = clMapMatch.dscpEcnValue
         ecnStr = ""
         if dscpValue.ecn != ecnDontCare:
            # if ecn is configured with/without dscp
            ecnName = AclCliLib.ecnNameFromValue( dscpValue.ecn )
            ecnStr = f" {ipStr[ 1 ]} {ecnName}"
         if len( dscpValue.dscpColl ) == 0:
            if not dscpValue.dscpNameValid:
               # 'match dscp <dscpId>'
               if dscpValue.dscp != Tac.Type( "Qos::DscpVal" ).invalid:
                  output.write( f"   match {ipStr[ 0 ]} {dscpValue.dscp}{ecnStr}\n" )
               else:
                  # 'match ecn <ecnName>'
                  output.write( f"   match{ecnStr}\n" )
            else:
               # 'match dscp <dscpName>'
               val = dscpValue.dscp
               name = AclCliLib.dscpNameFromValue( val )
               output.write( f"   match {ipStr[ 0 ]} {name}{ecnStr}\n" )
         else:
            # 'match dscp <comma-separated range>'
            ids = set()
            for key in dscpValue.dscpColl:
               ids.update( list( range( key.min, key.max + 1 ) ) )
            dscpStr = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
            output.write( f"   match {ipStr[ 0 ]} {dscpStr}{ecnStr}\n" )
      elif matchRule in [ tacMatchOption.matchMplsTrafficClass ]:
         ipStr = 'traffic-class'
         mplsTrafficClassVal = clMapMatch.mplsTrafficClassVal
         ids = set()
         if len( mplsTrafficClassVal.mplsTrafficClassColl ) == 1:
            for key in mplsTrafficClassVal.mplsTrafficClassColl:
               if key.min == key.max:
                  output.write( f"   match mpls {ipStr} {key.min}\n" )
               else:
                  ids.update( list( range( key.min, key.max + 1 ) ) )
                  mplsTrafficClassStr = \
                     MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                  output.write( f"   match mpls {ipStr} {mplsTrafficClassStr}\n" )
         else:
            for key in mplsTrafficClassVal.mplsTrafficClassColl:
               ids.update( list( range( key.min, key.max + 1 ) ) )
            mplsTrafficClassStr = \
                  MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
            output.write( f"   match mpls {ipStr} {mplsTrafficClassStr}\n" )
      else:
         output.write( f"   match {ipStr} access-group {clMapMatch.strValue}\n" )
   output.write( '\n' )

def showCMapDiff( mode, output=None ):
   if output is None:
      output = sys.stdout
   # generate diff between active and pending
   activeOutput = io.StringIO()
   _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.map_,
                  output=activeOutput )
   pendingOutput = io.StringIO()
   _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.currentEntry_,
                  output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue( ).splitlines( ),
                                pendingOutput.getvalue( ).splitlines( ),
                                lineterm='' )
   output.write( '\n'.join( list( diff ) ) )
   output.write( '\n' )

def showCMapPending( mode ):
   changeOutput = io.StringIO()
   showCMapDiff( mode, changeOutput )
   if changeOutput.getvalue() != '\n':
      # show pending only if there is a change
      _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.currentEntry_ )

def showCMapActive( mode ):
   _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.map_ )

def deleteClassMap( mode, cmapName, mapType='qos' ):
   emapType = mapTypeToEnum( mapType )
   if not cmapName in cliQosAclConfig.cmapType[ emapType ].cmap:
      return
   deleteComments( mode, mapType, cmapName=cmapName )
   # delete the entire map
   del cliQosAclConfig.cmapType[ emapType ].cmap[ cmapName ]
   t0( f'Deleted map {cmapName}' )
   if not mode.session.inConfigSession():
      currTime = getCurrentTimeForConfigUpdate()
      description = "class-map to be deleted"
      cliBlockingToApplyConfigChange( mode, currTime, tacFeatureName.others,
                                      description,
                                      policyMap=True )

class QosModeletServicePolicy( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( ethOrLagIntfPrefixes ) or
            isinstance( mode.intf, ( VlanIntfCli.VlanIntf, VxlanCli.VxlanIntf ) ) )


IntfCli.IntfConfigMode.addModelet( QosModeletServicePolicy )

# ------------------------------------------------------------------------
# [no] service-policy [ type qos ] { input | output } [<policy-name>]
# ------------------------------------------------------------------------
class ModeletServicePolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy [ type qos ] ( input | output ) POLICY'
   noOrDefaultSyntax = 'service-policy [ type qos ] ( input | output ) [ POLICY ]'
   data = {
      'service-policy': nodeServicePolicy,
      'type': matcherType,
      'qos': nodeMapTypeQos,
      'input': nodeInput,
      'output': nodeOutput,
      'POLICY': matcherPMapNameTypeQos,
   }

   handler = "QosCliServicePolicyHandler.modeletServicePolicyCmdHandler"
   noOrDefaultHandler = \
      "QosCliServicePolicyHandler.modeletServicePolicyCmdNoOrDefaultHandler"


QosModeletServicePolicy.addCommandClass( ModeletServicePolicyCmd )

# ------------------------------------------------------------------------
# Service Policy commands in QosProfile mode
# ------------------------------------------------------------------------
class QosProfileServicePolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy [ type qos ] ( input | output ) POLICY'
   noOrDefaultSyntax = 'service-policy [ type qos ] ( input | output ) [ POLICY ]'
   data = {
      'service-policy': nodeServicePolicy,
      'type': matcherType,
      'qos': nodeMapTypeQos,
      'input': nodeInput,
      'output': nodeOutput,
      'POLICY': matcherPMapNameTypeQos,
   }

   handler = "QosCliServicePolicyHandler.qosProfileSetServicePolicy"
   noOrDefaultHandler = handler


QosProfileMode.addCommandClass( QosProfileServicePolicyCmd )

# -----------------------------------------------------------------
# The show command in 'config-pmap' mode
#
#              show active|pending|diff
# -----------------------------------------------------------------
class PmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new policy-map configuration to be applied',
      'active': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'current': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'diff': ( 'Display the diff between policy-map configuration'
          'current running-config and to be applied' ),
   }

   handler = 'QosCliServicePolicyHandler.pmapModeShowCmdHandler'


PolicyMapModeQos.addShowCommandClass( PmapModeShowCmd )

# --------------------------------------------------------------------------------
# show policy-map [ type qos ] [ PMAP ] [ summary ]
# --------------------------------------------------------------------------------
class PolicyMapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map [ type qos ] [ PMAP ] [ summary ]'
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'PMAP': matcherPmapName,
      'summary': 'Policy Map summary',
   }

   handler = 'QosCliServicePolicyHandler.policyMapCmdHandler'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapCmd )

# --------------------------------------------------------------------------------
# show policy-map [ type qos ] PMAP class CMAP
# --------------------------------------------------------------------------------
class PolicyMapPmapClassCmapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map [ type qos ] PMAP class CMAP'
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'PMAP': matcherPmapName,
      'class': 'Policy criteria',
      'CMAP': CliMatcher.DynamicNameMatcher( getClassNameRuleQos,
         'Class Map Name' ),
   }

   handler = 'QosCliServicePolicyHandler.policyMapPmapClassCmapCmdHandler'
   cliModel = ClassMapWrapper


BasicCli.addShowCommandClass( PolicyMapPmapClassCmapCmd )

# --------------------------------------------------------------------------------
# show policy-map [ type qos ] PMAP [ input | output ] counters [ detail ]
# --------------------------------------------------------------------------------
class PolicyMapPmapCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map [ type qos ] PMAP [ DIRECTION ] '
              'counters [ detail ]' )
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'PMAP': matcherPmapName,
      'DIRECTION': matcherDirectionType,
      'counters': 'Policy Map counters',
      'detail': 'More comprehensive output',
   }

   handler = 'QosCliServicePolicyHandler.showPolicyMapCounters'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapPmapCountersCmd )

# --------------------------------------------------------------------------------
# show policy-map [ type qos ] [ interface ] counters
# --------------------------------------------------------------------------------
class PolicyMapPmapInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map [ type qos ] [ interface ] counters'
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'interface': 'Service Policy on interface',
      'counters': 'Policy Map counters',
   }

   handler = 'QosCliServicePolicyHandler.showPolicyMapCounters'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapPmapInterfaceCmd )

# --------------------------------------------------------------------------------
# show policy-map interface ( INTF | SVI_INTF | VTI_INTF )
#       [ type qos ] [ input | output ] [ counters ]
# --------------------------------------------------------------------------------
class PolicyMapInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map interface ( INTF | SVI_INTF | VTI_INTF ) '
              '[ type qos ] [ DIRECTION ] [ counters ]' )
   data = {
      'policy-map': nodePolicyMap,
      'interface': 'Service Policy on interface',
      'INTF': matcherIntf,
      'SVI_INTF': CliCommand.Node( matcher=IntfRangeMatcher( explicitIntfTypes=(
         VlanIntfCli.VlanAutoIntfType, ) ),
         guard=guardSviPolicyQosSupported ),
      'VTI_INTF': CliCommand.Node( matcher=IntfRangeMatcher( explicitIntfTypes=[
         VxlanAutoIntfType ] ), guard=vxlanPolicyQosSupported ),
      'type': matcherType,
      'qos': nodeQosType,
      'DIRECTION': matcherDirectionType,
      'counters': 'Policy Map counters',
   }

   handler = 'QosCliServicePolicyHandler.showPMapInterface'
   cliModel = PolicyMapAllModel


BasicCli.addShowCommandClass( PolicyMapInterfaceCmd )

# --------------------------------------------------------------------------------
# [ no | default ] policy-map type quality-of-service PMAP
# --------------------------------------------------------------------------------
class PolicyMapTypeQosPmapnameCmd( CliCommand.CliCommandClass ):
   syntax = 'policy-map type ( quality-of-service | qos ) PMAP'
   noOrDefaultSyntax = syntax
   data = {
      'policy-map': nodePolicyMapConfig,
      'type': matcherType,
      'quality-of-service': nodeQualityOfService,
      'qos': nodeMapTypeQosDeprecated,
      'PMAP': matcherPMapNameTypeQos,
   }

   handler = 'QosCliServicePolicyHandler.gotoQosPolicyMapMode'
   noOrDefaultHandler = 'QosCliServicePolicyHandler.deleteQosPolicyMap'


GlobalConfigMode.addCommandClass( PolicyMapTypeQosPmapnameCmd )

# --------------------------------------------------------------------------------
# policy-map [ type quality-of-service ] DST_PMAP copy SRC_PMAP
# --------------------------------------------------------------------------------
class CopyQosPolicyMapCmd( CliCommand.CliCommandClass ):
   syntax = ( 'policy-map [ type ( quality-of-service | qos ) ] '
            'DST_PMAP copy SRC_PMAP' )
   data = {
      'policy-map': nodePolicyMapConfig,
      'type': matcherType,
      'quality-of-service': nodeQualityOfService,
      'qos': nodeMapTypeQosDeprecated,
      'DST_PMAP': matcherPMapNameTypeQos,
      'copy': matcherCopy,
      'SRC_PMAP': matcherSrcPMapTypeQos,
   }

   handler = 'QosCliServicePolicyHandler.copyQosPolicyMap'


GlobalConfigMode.addCommandClass( CopyQosPolicyMapCmd )

# --------------------------------------------------------------------------------
# [ no | default ] policy-map type quality-of-service policer drop counter
# --------------------------------------------------------------------------------
class ClassMapDropCounterCmd( CliCommand.CliCommandClass ):
   syntax = 'policy-map type quality-of-service policer drop counter'
   noOrDefaultSyntax = syntax
   data = {
      'policy-map': nodePolicyMapConfig,
      'type': matcherType,
      'quality-of-service': nodeQualityOfService,
      'policer': CliMatcher.KeywordMatcher( 'policer',
                                             helpdesc="policy-map policer" ),
      'drop': nodeDrop,
      'counter': nodeCounter,
   }

   handler = 'QosCliServicePolicyHandler.enableQosAclCmapDropCounter'
   noOrDefaultHandler = 'QosCliServicePolicyHandler.disableQosAclCmapDropCounter'


GlobalConfigMode.addCommandClass( ClassMapDropCounterCmd )

#--------------------------------------------------------------------------------
# [ no | default ] policy-map type quality-of-service counter per-interface
#--------------------------------------------------------------------------------
class TogglePerInterfaceCounter( CliCommand.CliCommandClass ):
   syntax = 'policy-map type quality-of-service counter per-interface'
   noOrDefaultSyntax = syntax
   data = {
      'policy-map': nodePolicyMapConfig,
      'type': matcherType,
      'quality-of-service': nodeQualityOfService,
      'counter': nodePerInterfaceCounter,
      'per-interface': nodePerInterface,
   }

   handler = 'QosCliServicePolicyHandler.enableCntIntf'
   noOrDefaultHandler = 'QosCliServicePolicyHandler.disableCntIntf'

GlobalConfigMode.addCommandClass( TogglePerInterfaceCounter )

# --------------------------------------------------------------------------------
# [ no | default ] class-map [ type qos ] [ match-any ] CMAP
# --------------------------------------------------------------------------------
class ClassMapMatchAnyCmapnameCmd( CliCommand.CliCommandClass ):
   syntax = 'class-map [ type qos ] match-any CMAP'
   noOrDefaultSyntax = 'class-map [ type qos ] [ match-any ] CMAP'
   data = {
      'class-map': nodeClassMapConfig,
      'type': matcherType,
      'qos': nodePMapQos,
      'match-any': matcherMatchAny,
      'CMAP': CliMatcher.DynamicNameMatcher( getCMapNameQos,
                                       helpdesc='Class Map Name' ),
   }

   handler = 'QosCliServicePolicyHandler.gotoQosClassMapMode'
   noOrDefaultHandler = \
      'QosCliServicePolicyHandler.classMapMatchAnyCmapnameCmdNoOrDefaultHandler'


GlobalConfigMode.addCommandClass( ClassMapMatchAnyCmapnameCmd )

# --------------------------------------------------------------------------------
# show class-map [ type qos ] [ CMAP ]
# --------------------------------------------------------------------------------
class ClassMapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show class-map [ type qos ] [ CMAP ]'
   data = {
      'class-map': CliCommand.guardedKeyword( 'class-map',
         "Show Class Map", guard=guardClassMap ),
      'type': matcherType,
      'qos': nodeQosType,
      'CMAP': CliMatcher.DynamicNameMatcher( getCMapNameQos,
         'Class Map Name' ),
   }

   handler = 'QosCliServicePolicyHandler.classMapCmdHandler'
   cliModel = ClassMapAllModel


BasicCli.addShowCommandClass( ClassMapCmd )

# -----------------------------------------------------------------------------------
# "drop-precedence DP drop-threshold percent PERCENT" command under tx-queue
# -----------------------------------------------------------------------------------
class IntfUcTxQueueDpCmd( CliCommand.CliCommandClass ):
   syntax = 'drop-precedence DP drop-threshold percent PERCENT'
   noOrDefaultSyntax = 'drop-precedence DP drop-threshold percent ...'
   data = {
      'drop-precedence': nodeDropPrecedence,
      'DP': dropPrecedenceValueMatcher,
      'drop-threshold': 'Set tail drop-threshold value',
      'percent': 'Set drop-threshold value in percent',
      'PERCENT': CliMatcher.IntegerMatcher( 0, tacPercent.max,
         helpdesc='Percent value between 0 and 100' )
   }

   handler = "QosCliServicePolicyHandler.setDropThresholds"
   noOrDefaultHandler = handler


IntfUcTxQueueModelet.addCommandClass( IntfUcTxQueueDpCmd )

# ------------------------------------------------------------------------
# set drop-precedence DROP_PRECEDENCE
# ( no | default ) set drop-precedence ...
# ------------------------------------------------------------------------
class SetDropPrecedenceCmd( CliCommand.CliCommandClass ):
   syntax = 'set drop-precedence DROP_PRECEDENCE'
   noOrDefaultSyntax = 'set drop-precedence ...'
   data = {
      'set': matcherSet,
      'drop-precedence': nodePMapQosDropPrecedence,
      'DROP_PRECEDENCE': dropPrecedenceValueMatcher,
   }

   handler = PolicyMapClassModeQos.configureSetDropPrecedence
   noOrDefaultHandler = handler


PolicyMapClassModeQos.addCommandClass( SetDropPrecedenceCmd )

# --------------------------------------------------------------------------------
# qos policer POLICER_NAME rate CIR_VALUE [ RATE_UNIT ]
#                    burst-size BC_VALUE [ BURST_UNIT ]
# ( no | default ) qos policer POLICER_NAME
# --------------------------------------------------------------------------------
def rateBurstUnitAdapter( mode, args, argsList ):
   rateUnitsToTypes = {
      'bps': tacRateUnit.rateUnitbps,
      'kbps': tacRateUnit.rateUnitKbps,
      'mbps': tacRateUnit.rateUnitMbps,
      'pps': tacRateUnit.rateUnitPps,
   }

   burstUnitsToTypes = {
      'bytes': tacBurstUnit.burstUnitBytes,
      'kbytes': tacBurstUnit.burstUnitKBytes,
      'mbytes': tacBurstUnit.burstUnitMBytes,
      'packets': tacBurstUnit.burstUnitPackets,
   }

   args[ 'RATE_CIR_UNIT' ] = rateUnitsToTypes[ args.get( 'RATE_UNIT', 'bps' ) ]
   args[ 'BURST_BC_UNIT' ] = burstUnitsToTypes[ args.get( 'BURST_UNIT', 'bytes' ) ]

class QosPolicerNameCmd( CliCommand.CliCommandClass ):
   syntax = ( 'qos policer POLICER_NAME rate CIR_VALUE [ RATE_UNIT ] '
              'burst-size BC_VALUE [ BURST_UNIT ] ' )
   noOrDefaultSyntax = 'qos policer POLICER_NAME ...'
   data = {
      'qos': QosCli.nodeQosForConfig,
      'policer': nodePolicer,
      'POLICER_NAME': CliMatcher.DynamicNameMatcher( getPolicerNameRule,
         helpdesc="Policer name" ),
      'rate': nodePoliceLRate,
      'CIR_VALUE': matcherPoliceCirValue,
      'RATE_UNIT': matchPoliceRateUnit,
      'burst-size': nodePoliceLBs,
      'BC_VALUE': matcherPoliceCommittedBurstValue,
      'BURST_UNIT': matcherBurstUnit,
   }

   adapter = rateBurstUnitAdapter

   handler = 'QosCliServicePolicyHandler.qosPolicerNameCmdHandler'
   noOrDefaultHandler = \
      'QosCliServicePolicyHandler.qosPolicerNameCmdNoOrDefaultHandler'


GlobalConfigMode.addCommandClass( QosPolicerNameCmd )

# --------------------------------------------------------------------------------
# [ no | default ] policing
# --------------------------------------------------------------------------------
class PolicingModeCmd( CliCommand.CliCommandClass ):
   syntax = 'policing'
   noOrDefaultSyntax = syntax
   data = {
      'policing': PolicingCli.nodePolicingMode,
   }

   handler = 'PolicingCli.gotoPolicingMode'
   noOrDefaultHandler = 'PolicingCli.PolicingMode.clear'


GlobalConfigMode.addCommandClass( PolicingModeCmd )

# ------------------------------------------------------------------------
# police rate <value> {bps, kbps, mbps, gbps} burst-size <value>
# {bytes, kbytes, mbytes} [ [ action set {drop-precedence, dscp <value>} ]
# rate <value> {bps, kbps, mbps, gbps} burst-size <value> {bytes, kbytes, mbytes} ]
# ------------------------------------------------------------------------
def guardPoliceRateInPps( mode, token ):
   if qosAclHwStatus.policePacketModeSupported:
      return None
   elif qosAclHwStatus.policePacketModeQosPolicySupported and \
        isinstance( mode, PolicyMapClassModeQos ):
      return None
   return CliParser.guardNotThisPlatform

class CirExpression( CliCommand.CliExpression ):
   expression = 'bps | kbps | mbps | pps'
   data = {
      'bps': 'rate in bps (default unit)',
      'kbps': 'The rate expressed in units of kilobits per second',
      'mbps': 'rate in Mbps',
      'pps': CliCommand.guardedKeyword( 'pps', helpdesc='rate in pps',
         guard=guardPoliceRateInPps ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      rateUnitsToTypes = {
            'bps': tacRateUnit.rateUnitbps,
            'kbps': tacRateUnit.rateUnitKbps,
            'mbps': tacRateUnit.rateUnitMbps,
            'pps': tacRateUnit.rateUnitPps,
            }

      for rate in ( 'bps', 'kbps', 'mbps', 'pps' ):
         if rate in args:
            args[ 'RATE_CIR_UNIT' ] = rateUnitsToTypes[ args[ rate ] ]
            break

class PolicerBurstExpression( CliCommand.CliExpression ):
   expression = 'bytes | kbytes | mbytes | packets'
   data = {
      'bytes': 'burst size in bytes (default unit)',
      'kbytes': 'burst size in kbytes',
      'mbytes': 'burst size in mbytes',
      'packets': CliCommand.guardedKeyword( 'packets',
         helpdesc='burst size in packets', guard=guardPoliceRateInPps ),
      }

   @staticmethod
   def adapter( mode, args, argsList ):
      burstUnitsToTypes = {
            'bytes': tacBurstUnit.burstUnitBytes,
            'kbytes': tacBurstUnit.burstUnitKBytes,
            'mbytes': tacBurstUnit.burstUnitMBytes,
            'packets': tacBurstUnit.burstUnitPackets,
            }

      for rate in ( 'bytes', 'kbytes', 'mbytes', 'packets' ):
         if rate in args:
            args[ 'BURST_BC_UNIT' ] = burstUnitsToTypes[ args[ rate ] ]
            break

class PoliceRateBurstVerTwoCmd( CliCommand.CliCommandClass ):
   syntax = 'police rate CIR [ RATE_UNIT ] burst-size BURST [ BURST_UNIT ]'
   data = {
      'police': nodePolice,
      'rate': nodePoliceLRate,
      'CIR': matcherPoliceCirValue,
      'RATE_UNIT': CirExpression,
      'burst-size': nodePoliceLBs,
      'BURST': matcherPoliceBurstValue,
      'BURST_UNIT': PolicerBurstExpression,
   }

   handler = "QosCliServicePolicyHandler.policeRateBurstVerTwoCmdHandler"


PolicyMapClassModeQos.addCommandClass( PoliceRateBurstVerTwoCmd )

class CirPirUnit( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      nodeCirPirUnit = CliCommand.Node( matcher=matcherRateUnit, alias=name )
      nodepps = CliCommand.guardedKeyword( 'pps', helpdesc='Rate in pps',
            guard=guardPoliceRateInPps, alias=name )

      class CirPirExpression( CliCommand.CliExpression ):
         expression = f'{name}_PIR_CIR_UNIT | {name}_pps'
         data = {
            f'{name}_PIR_CIR_UNIT': nodeCirPirUnit,
            f'{name}_pps': nodepps,
            }

         @staticmethod
         def adapter( mode, args, argsList ):
            rateUnitsToTypes = {
                  'bps': tacRateUnit.rateUnitbps,
                  'kbps': tacRateUnit.rateUnitKbps,
                  'mbps': tacRateUnit.rateUnitMbps,
                  'pps': tacRateUnit.rateUnitPps,
               }

            args[ 'CIR_SPEED_UNIT' ] = \
                  rateUnitsToTypes[ args.get( 'CIRUNIT', 'kbps' ) ]
            args[ 'PIR_SPEED_UNIT' ] = \
                  rateUnitsToTypes[ args.get( 'PIRUNIT', 'kbps' ) ]
      return CirPirExpression

class BurstHigherUnit( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      nodeHBurstUnit = CliCommand.Node( matcher=matcherBurstUnit, alias=name )
      nodePackets = CliCommand.guardedKeyword( 'packets',
            helpdesc='Burst size in packets',
            guard=guardPoliceRateInPps, alias=name )

      class BurstHigherRateExpression( CliCommand.CliExpression ):
         expression = f'{name}_H_BURST_UNIT | {name}_packets'
         data = {
               f'{name}_H_BURST_UNIT': nodeHBurstUnit,
               f'{name}_packets': nodePackets,
            }

         @staticmethod
         def adapter( mode, args, argsList ):
            burstUnitsToTypes = {
                  'bytes': tacBurstUnit.burstUnitBytes,
                  'kbytes': tacBurstUnit.burstUnitKBytes,
                  'mbytes': tacBurstUnit.burstUnitMBytes,
                  'packets': tacBurstUnit.burstUnitPackets,
               }

            args[ 'BURST_RATE_UNIT' ] = \
                  burstUnitsToTypes[ args.get( 'BURSTUNIT', 'bytes' ) ]
            args[ 'HBURST_RATE_UNIT' ] = \
                  burstUnitsToTypes[ args.get( 'HBUNIT', 'bytes' ) ]
      return BurstHigherRateExpression

# ------------------------------------------------------------------------
# police rate CIR [ CIRUNIT ] burst-size BURSTSIZE [ BUNIT ]
#    [ action set ( drop-precedence | (  dscp ( DSCP | DSCP_NAME ) ) ) ]
#      highrate PIRVALUE [ PIRUNIT ] highbsize HBURSTSIZE [ HBUNIT ]
# ------------------------------------------------------------------------
class PoliceRateHigherRateBurstVerTwoCmd( CliCommand.CliCommandClass ):
   syntax = ( 'police rate CIR [ CIRUNIT ] burst-size BURSTSIZE '
              '[ BURSTUNIT ] [ action set ( drop-precedence | ( dscp '
              '( DSCP | DSCP_NAME ) ) ) ] highrate PIRVALUE [ PIRUNIT ] '
              'highbsize HBURSTSIZE [ HBUNIT ]' )
   data = {
      'police': nodePolice,
      'rate': nodePoliceLRate,
      'CIR': matcherPoliceCirValue,
      'CIRUNIT': CirPirUnit(),
      'burst-size': nodePoliceLBs,
      'BURSTSIZE': matcherPoliceBurstValue,
      'BURSTUNIT': BurstHigherUnit(),
      'action': nodePoliceAction,
      'set': nodePoliceSetAction,
      'drop-precedence': nodePoliceActionDP,
      'dscp': nodePoliceActionDscp,
      'DSCP': CliMatcher.IntegerMatcher( 0, AclLib.MAX_DSCP,
         helpdesc="DSCP Value" ),
      'DSCP_NAME': CliMatcher.DynamicKeywordMatcher(
         lambda mode: { key: val[ 1 ] for key, val in
            AclCliLib.dscpAclNames.items() } ),
      'highrate': nodePoliceHRate,
      'PIRVALUE': matcherPolicePirValue,
      'PIRUNIT': CirPirUnit(),
      'highbsize': nodePoliceHBs,
      'HBURSTSIZE': matcherPoliceExcessBurstValue,
      'HBUNIT': BurstHigherUnit(),
      }

   handler = "QosCliServicePolicyHandler.policeRateHigherRateBurstVerTwoCmdHandler"


PolicyMapClassModeQos.addCommandClass( PoliceRateHigherRateBurstVerTwoCmd )

# ------------------------------------------------------------------------
# police cir <value> {bps, kbps, mbps} bc <value> {bytes, kbytes, mbytes}
# ( no | default ) police ...
# ------------------------------------------------------------------------
class PoliceRateBurstVerOneCmd( CliCommand.CliCommandClass ):
   syntax = 'police cir CIR [ RATE_UNIT ] bc BC [ BURST_UNIT ]'
   noOrDefaultSyntax = 'police ...'
   data = {
      'police': nodePolice,
      'cir': nodePoliceCir,
      'CIR': matcherPoliceCirValue,
      'RATE_UNIT': matcherRateUnit,
      'bc': nodePoliceBc,
      'BC': matcherPoliceBurstValue,
      'BURST_UNIT': matcherBurstUnit,
   }

   adapter = rateBurstUnitAdapter

   handler = "QosCliServicePolicyHandler.policeRateBurstVerOneCmdHandler"
   noOrDefaultHandler = \
      "QosCliServicePolicyHandler.policeRateBurstVerOneCmdNoOrDefaultHandler"


PolicyMapClassModeQos.addCommandClass( PoliceRateBurstVerOneCmd )

# ----------------------------
# police shared <policerName>
# ----------------------------
class PolicyMapClassSharedPolicerCmd( CliCommand.CliCommandClass ):
   syntax = 'police shared POLICER_NAME'
   data = {
      'police': nodePolice,
      'shared': nodePoliceShared,
      'POLICER_NAME': matcherPolicerName,
   }

   handler = "QosCliServicePolicyHandler.policyMapClassSharedPolicerCmdHandler"


PolicyMapClassModeQos.addCommandClass( PolicyMapClassSharedPolicerCmd )

# ------------------------------------------------------------------------
# service-profile PROFILE
# ------------------------------------------------------------------------
class ServiceProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'service-profile PROFILE'
   data = {
      'service-profile': matcherServiceProfile,
      'PROFILE': matcherQosProfileName
   }

   handler = "QosCliServicePolicyHandler.serviceProfileCmdHandler"


FabricMode.addCommandClass( ServiceProfileCmd )
QosModelet.addCommandClass( ServiceProfileCmd )
QosSubIntfModelet.addCommandClass( ServiceProfileCmd )

# ------------------------------------------------------------------------
# ( no | default ) service-profile [ PROFILE ]
# ------------------------------------------------------------------------
class ServiceProfileNoDefCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'service-profile [ PROFILE ]'
   data = {
      'service-profile': matcherServiceProfile,
      'PROFILE': matcherQosProfileName
   }

   noOrDefaultHandler = \
      "QosCliServicePolicyHandler.serviceProfileNoDefCmdNoOrDefaultHandler"


FabricMode.addCommandClass( ServiceProfileNoDefCmd )
IntfCli.IntfConfigMode.addCommandClass( ServiceProfileNoDefCmd )

# --------------------------------------------------------------------------------
# show qos profile [ PROFILENAME ]
# --------------------------------------------------------------------------------
class QosProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos profile [ PROFILENAME ]'
   data = {
      'qos': nodeQosForShow,
      'profile': matcherProfile,
      'PROFILENAME': profileNameMatcher,
      }

   handler = QosProfileMode.showQosProfile
   cliModel = QosProfileAllModel


BasicCli.addShowCommandClass( QosProfileCmd )

# --------------------------------------------------------------------------------
# show qos profile [ PROFILENAME ] summary
# --------------------------------------------------------------------------------
class QosProfileSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos profile [ PROFILENAME ] summary'
   data = {
      'qos': nodeQosForShow,
      'profile': matcherProfile,
      'PROFILENAME': profileNameMatcher,
      'summary': 'Qos profile summary',
   }

   handler = "QosCliServicePolicyHandler.showQosProfileSummary"
   cliModel = QosProfileSummaryModel


BasicCli.addShowCommandClass( QosProfileSummaryCmd )

# -----------------------------------------------------------------------------------
# The "[ default|no ] service config verification qos" command,
# in "global config" mode.
# -----------------------------------------------------------------------------------
class ServiceVerifyQos( CliCommand.CliCommandClass ):
   syntax = "service configuration verification qos"
   noSyntax = syntax
   defaultSyntax = syntax
   data = {
      "service": FileCli.serviceKw,
      "configuration": FileCli.configKwAfterService,
      "verification": FileCli.verificationKwAfterService,
      "qos": CliCommand.guardedKeyword( "qos", 'Verify QoS config',
                                         guardHwConfigVerification )
      }

   handler = "QosCliServicePolicyHandler.serviceVerifyQosHandler"
   noHandler = "QosCliServicePolicyHandler.serviceVerifyQosNoHandler"
   defaultHandler = handler


BasicCli.GlobalConfigMode.addCommandClass( ServiceVerifyQos )

class ShowInterfacesDropThresholds( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos interfaces [ INTF ] drop-thresholds"

   data = {
      'qos': nodeQosForShow,
      'interfaces': 'Show QoS status for a specific interface',
      'INTF': IntfRangeMatcher( explicitIntfTypes=ethOrLagIntfTypes ),
      'drop-thresholds': CliCommand.guardedKeyword( 'drop-thresholds',
         helpdesc='Show tail drop-threshold values',
         guard=guardDropPrecedenceThreshold )
   }
   handler = "QosCliServicePolicyHandler.showInterfacesDropThresholds"
   cliModel = IntfDropThresholdsCollectionModel


BasicCli.addShowCommandClass( ShowInterfacesDropThresholds )

# ------------------------------------------------------------------------
# Match and set value binding rules ( policy-map )
# ------------------------------------------------------------------------
# --------------------------------------------------------------------------------
# [ no | default ] class CMAP [ insert-before CLASS ]
# --------------------------------------------------------------------------------
class InsertBeforeClassPmapModeCmd( CliCommand.CliCommandClass ):
   syntax = 'class CMAP [ insert-before CLASS ]'
   noOrDefaultSyntax = syntax
   data = {
      'insert-before': CliCommand.guardedKeyword( 'insert-before',
         helpdesc='insert the class with a higher priority than a given class',
         guard=guardClassMap ),
      'class': 'Policy criteria',
      'CMAP': CliMatcher.DynamicNameMatcher( getClassNameRuleQos,
         helpdesc='Class Map Name' ),
      'CLASS': CliMatcher.DynamicNameMatcher( getClassNameRuleQos,
         helpdesc='Class Map Name' ),
   }

   handler = "QosCliServicePolicyHandler.insertBeforeClassPmapModeCmdHandler"
   noHandler = handler
   defaultHandler = handler


PolicyMapModeQos.addCommandClass( InsertBeforeClassPmapModeCmd )

# ------------------------------------------------------------------------
# set traffic-class TRAFFIC_CLASS
# ( no | default ) set traffic-class ...
# ------------------------------------------------------------------------
class PolicyMapClassModeSetTcCmd( CliCommand.CliCommandClass ):
   syntax = 'set traffic-class TRAFFIC_CLASS'
   noOrDefaultSyntax = 'set traffic-class ...'
   data = {
      'set': matcherSet,
      'traffic-class': nodeActionTc,
      'TRAFFIC_CLASS': CliMatcher.DynamicIntegerMatcher( trafficClassRangeFn,
         helpdesc="Traffic class value" ),
   }

   handler = PolicyMapClassModeQos.configureSetTc
   noOrDefaultHandler = handler


PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeSetTcCmd )

# ------------------------------------------------------------------------
# drop
# ( no | default ) drop ...
# ------------------------------------------------------------------------
class PolicyMapClassModeDropCmd( CliCommand.CliCommandClass ):
   syntax = 'drop'
   noOrDefaultSyntax = 'drop ...'
   data = {
      'drop': nodeActionDrop,
   }

   handler = PolicyMapClassModeQos.configureSetDrop
   noOrDefaultHandler = handler


PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeDropCmd )

# --------------------------------------------------------------------------------
# ( ( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET ) | ( mc-tx-queue MCTXQSET ) )
# --------------------------------------------------------------------------------
class QueueSetIntfRangeConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET )
             | ( mc-tx-queue MCTXQSET )'''
   data = {
      'tx-queue': nodeTxQueue,
      'uc-tx-queue': nodeUcTxQueue,
      'mc-tx-queue': nodeMcTxQueue,
      'TXQSET': matcherTxQueueRange,
      'UCTXQSET': matcherUcTxQueueRange,
      'MCTXQSET': matcherMcTxQueueRange,
   }

   handler = "QosCliServicePolicyHandler.queueSetIntfRangeConfigCmdHandler"


IntfRangeConfigMode.addCommandClass( QueueSetIntfRangeConfigCmd )

# --------------------------------------------------------------------------------
# priority strict
# ( no | default ) priority ...
# --------------------------------------------------------------------------------
class PriorityStrictCmd( CliCommand.CliCommandClass ):
   syntax = 'priority strict'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority': nodePriority,
      'strict': 'Set priority to be strict',
   }

   handler = "QosCliServicePolicyHandler.setPriority"
   noOrDefaultHandler = handler


IntfTxQueueConfigMode.addCommandClass( PriorityStrictCmd )

# --------------------------------------------------------------------------------
# [ no | default ] latency maximum LATENCY microseconds | milliseconds
# --------------------------------------------------------------------------------
class TxQueueLatencyThresholdMicrosecondsCmd( CliCommand.CliCommandClass ):
   syntax = 'latency maximum LATENCY microseconds'
   noOrDefaultSyntax = 'latency maximum ...'
   data = {
      'latency': nodeLatency,
      'maximum': 'Set maximum latency',
      'LATENCY': CliMatcher.DynamicIntegerMatcher(
                     latencyMicrosecondsFn,
                     helpdesc='Latency value in microseconds' ),
      'microseconds': 'Latency value in microseconds',
   }
   handler = "QosCliServicePolicyHandler.handleTxQueueLatencyThreshold"
   noOrDefaultHandler = "QosCliServicePolicyHandler.handleTxQueueLatencyThreshold"


IntfUcTxQueueModelet.addCommandClass( TxQueueLatencyThresholdMicrosecondsCmd )

class TxQueueLatencyThresholdMillisecondsCmd( CliCommand.CliCommandClass ):
   syntax = 'latency maximum LATENCY milliseconds'
   data = {
      'latency': nodeLatency,
      'maximum': 'Set maximum latency',
      'LATENCY': CliMatcher.DynamicIntegerMatcher(
                     latencyMillisecondsFn,
                     helpdesc='Latency value in milliseconds' ),
      'milliseconds': 'Latency value in milliseconds',
   }
   handler = "QosCliServicePolicyHandler.handleTxQueueLatencyThreshold"


IntfUcTxQueueModelet.addCommandClass( TxQueueLatencyThresholdMillisecondsCmd )

# --------------------------------------------------------------------------------
# show qos interfaces INTERFACE latency maximum
# --------------------------------------------------------------------------------
intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher
intfMatcher |= SubIntfCli.subMatcher
intfMatcher |= LagIntfCli.subMatcher

class TxQueueLatencyThresholdShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos interfaces [ INTERFACE ] latency maximum"

   data = {
      'qos': nodeQosForShow,
      'interfaces': 'Show QoS status for a specific interface',
      'INTERFACE': intfMatcher,
      'latency': CliCommand.guardedKeyword( 'latency',
         helpdesc='Show latency parameters', guard=guardLatencyThreshold ),
      'maximum': 'Maximum latency',
   }
   handler = "QosCliServicePolicyHandler.txQueueLatencyThresholdShowHandler"
   cliModel = IntfLatencyThresholdCollectionModel


BasicCli.addShowCommandClass( TxQueueLatencyThresholdShowCmd )

# --------------------------------------------------------------------------------
# [ no | default ] qos profile PROFILENAME
# --------------------------------------------------------------------------------
class QosProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos profile PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'profile': 'QoS profile',
      'PROFILENAME': matcherQosProfileName,
   }

   handler = 'QosCliServicePolicyHandler.gotoQosProfileMode'
   noOrDefaultHandler = 'QosCliServicePolicyHandler.deleteQosProfile'


GlobalConfigMode.addCommandClass( QosProfileProfilenameCmd )

# -------------------------------------------------------------------------------
# 1) To delete Qos Lag intfConfig object when the corresponding interface gets
#    deleted.
# 2) Support for default interface command.
# -------------------------------------------------------------------------------
class QosIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "QosIntfJanitor: interface", self.intf_.name, "going away" )
      intfConfig = qosInputConfig.intfConfig.get( self.intf_.name )
      profileIntfConfig = qosInputProfileConfig.intfConfig.get( self.intf_.name )
      if intfConfig is not None:
         del qosInputConfig.intfConfig[ self.intf_.name ]
      intfShortName = Tac.Value( "Arnet::IntfId", self.intf_.name ).shortName
      numTxQueues = qosHwStatus.numTxQueueSupported
      isSubIntf = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( self.intf_.name )
      if isSubIntf:
         numTxQueues = qosGlobalConfig.numTxqsPerSubIntf
      for txqid in range( numTxQueues ):
         for queueType in tacQueueType.attributes:
            if queueType == tacQueueType.unknown:
               queueType = 'txq'
            elif queueType == tacQueueType.ucq:
               queueType = 'uc-txq'
            elif queueType == tacQueueType.mcq:
               queueType = 'mc-txq'
            else:
               raise NotImplementedError
            commentKey = f"if-{intfShortName}-{queueType}-{txqid}"
            BasicCliModes.removeCommentWithKey( commentKey )
      if profileIntfConfig is not None:
         removeProfile( None, None, self.intf_.name )
         del qosInputProfileConfig.intfConfig[ self.intf_.name ]
      for spKey in qosInputConfig.servicePolicyConfig:
         spConfig = qosInputConfig.servicePolicyConfig[ spKey ]
         if self.intf_.name in spConfig.intfIds:
            delServicePolicy( None, spKey.type, spKey.pmapName, spKey.direction,
                              self.intf_.name )
      for spKey in qosInputProfileConfig.servicePolicyConfig:
         spConfig = qosInputProfileConfig.servicePolicyConfig[ spKey ]
         if self.intf_.name in spConfig.intfIds:
            delServicePolicy( None, spKey.type, spKey.pmapName, spKey.direction,
                              self.intf_.name, profile=True )
      if self.intf_.name in profileConfigDir.intfToProfileMap:
         del profileConfigDir.intfToProfileMap[ self.intf_.name ]
      if self.intf_.name in qosInputConfig.ecnIntfCounterConfig:
         del qosInputConfig.ecnIntfCounterConfig[ self.intf_.name ]
      deleteInterfacePolicingConfig( qosInputConfig, self.intf_.name )

def startQosAclConfigMergeSm():
   gv.qosAclConfig = Tac.newInstance( "Qos::QosAclConfig" )
   gv.qosAclConfigMergeSm = Tac.newInstance( "Qos::Agent::QosAclConfigMergeSm",
                                          qosAclConfigDir, cliQosAclConfigReadOnly,
                                          gv.qosAclConfig )

@Plugins.plugin( provides=( "QosCliServicePolicy", ) )
def Plugin( entityManager ):
   global qosAclHwStatus, qosHwStatus, profileConfigDir, cliQosAclConfig, \
      subIntfHwStatus, qosSliceHwStatus, qosAclSliceHwStatus, qosAclConfigDir, \
      cliQosAclConfigReadOnly, qosConfig, qosStatus, hwEpochStatus, \
      qosInputConfig, qosInputProfileConfig, defaultPdpPmapCfgReadOnly, \
      cliCounterConfig, fabricIntfConfigDir, qosHwConfig, lagInputConfig, \
      qosGlobalConfig, aclConfig
   qosAclHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosInputProfileConfig = ConfigMount.mount(
         entityManager, "qos/input/config/qosProfile",
         "Qos::Input::Config", "w" )
   qosAclSliceHwStatus = LazyMount.mount( entityManager,
         "qos/hardware/acl/status/slice", "Tac::Dir", "ri" )
   qosSliceHwStatusDirPath = \
      "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   hwEpochStatus = LazyMount.mount( entityManager, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   lagInputConfig = LazyMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "r" )
   qosHwConfig = LazyMount.mount( entityManager, "qos/hardware/config",
                                  "Qos::HwConfig", "w" )
   profileConfigDir = ConfigMount.mount( entityManager, "qos/profile",
                                         "Qos::QosProfileConfigDir", "w" )
   cliQosAclConfig = ConfigMount.mount( entityManager, "qos/acl/input/cli",
                                        "Qos::Input::AclConfig", "w" )
   aclConfig = LazyMount.mount( entityManager, "acl/config/cli",
                                        "Acl::Input::Config", "r" )
   cliCounterConfig = LazyMount.mount( entityManager, "qos/cliCounterConfig",
                                       "Qos::CounterConfig", "w" )
   fabricIntfConfigDir = ConfigMount.mount( entityManager,
                                            "qos/fabric/config",
                                            "Interface::FabricIntfConfigDir",
                                            "w" )
   IntfCli.Intf.registerDependentClass( QosIntfJanitor )
   qosGlobalConfig = ConfigMount.mount( entityManager, "qos/global/config",
                                        "Qos::GlobalConfig", "w" )
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )

   # Set up a single instance of QosAclConfigMergeSm. It needs to mount the Cli
   # config and the input dirs for config from EosSdk. The aggregated view of
   # Qos::QosAclConfig is instantiated in startQosAclConfigMergeSm().
   # Note: cliQosAclConfigReadOnly is used to directly access the cli config entity
   # for use by QosAclConfigMergeSm. This can only be used as an input to that merge
   # SM. cliQosAclConfig, on the other hand is a ConfigMount Proxy that the plugin
   # can write to.
   mg = entityManager.mountGroup()

   cliQosAclConfigReadOnly = mg.mount( "qos/acl/input/cli",
                                       "Qos::Input::AclConfig", "wi" )
   defaultPdpPmapCfgReadOnly = mg.mount( "qos/acl/input/defaultPdpPmapCfg",
                                         "Qos::Input::AclConfig", "ri" )
   qosAclConfigDir = mg.mount( "qos/acl/input/eosSdkConfig", "Tac::Dir", "ri" )
   mg.close( startQosAclConfigMergeSm )
