#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# -----------------------------------------------------------------------------------
# SHAPING CLI PLUGIN
# -----------------------------------------------------------------------------------
import Tac
import CliParser
import CliCommand
import CliMatcher
from CliPlugin import IntfCli
from CliPlugin import LagIntfCli
from CliPlugin import ( QosCli, QosCliCommon )
from CliPlugin.QosCliIntfTypes import ethOrLagIntfPrefixes
from CliPlugin.QosCliCommon import isSubIntfConfigMode
import ConfigMount
import LazyMount
import Plugins

import QosLib
from QosTypes import ( tacShapeRateVal,
                       tacShapeRateUnit )

# ------------------------------------------------
# Mount path Holders
# -----------------------------------------------------------------------------------
qosHwStatus = None
qosInputConfig = None
subIntfHwStatus = None

# ------------------------------------------------
# Shaping Modelet
# ------------------------------------------------
class QosModeletShapeRate( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( ethOrLagIntfPrefixes )


IntfCli.IntfConfigMode.addModelet( QosModeletShapeRate )

# -----------------------------------------------------------------------------------
# Shaping Feature class which must be registered with QosCli
# -----------------------------------------------------------------------------------
class QosShapeRateModuleBase( QosLib.QosModuleBase ):
   def isDefaultIntfConfig( self, intfConfig ):
      cfgShapeRate = intfConfig.shapeRate
      return ( ( cfgShapeRate.rate == tacShapeRateVal.invalid ) and
               ( cfgShapeRate.unit == tacShapeRateUnit.shapeRateKbps ) and
               not cfgShapeRate.shared )


QosLib.registerQosModule( 'shapeRate', QosShapeRateModuleBase( QosCli.qosHwStatus ) )

# ------------------------------------------------
# Guards
# ------------------------------------------------
def guardPortShaping( mode, token ):
   if QosCli.qosHwStatus.portShapingSupported and \
      ( QosCli.qosHwStatus.hwInitialized or QosCli.qosHwStatus.forceQosConfig ):
      if isinstance( mode, IntfCli.IntfConfigMode ) and mode.intf.isSubIntf() and \
         not subIntfHwStatus.subIntfShapeRateSupported:
         return CliParser.guardNotThisPlatform
      else:
         return None
   return CliParser.guardNotThisPlatform

def guardPortShapeRatePps( mode, token ):
   if not QosCli.qosHwStatus.portShapeRateUnitPpsSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardSharedShaping( mode, token ):
   if not QosCli.qosHwStatus.sharedLagShapingSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardTxQShapeRatePps( mode, token ):
   if not QosCli.qosHwStatus.txQueueShapeRateUnitPpsSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardShapeRateBurstSize( mode, token ):
   if not QosCli.qosHwStatus.shapeRateBurstSizeConfigSupported:
      return CliParser.guardNotThisPlatform
   return None

# ------------------------------------------------
# Utility functions
# ------------------------------------------------
def txQShapeRateKbpsRangeFn( mode, context=None ):
   if isSubIntfConfigMode( mode ):
      minTxQShapeRate = QosCli.qosHwStatus.minSubIntfTxQueueShapeRate
   else:
      minTxQShapeRate = QosCli.qosHwStatus.minTxQueueShapeRate
   return ( minTxQShapeRate, QosCli.qosHwStatus.maxShapeRateKbps )

def txQShapeRatePpsRangeFn( mode, context=None ):
   return ( QosCli.qosHwStatus.minTxQueueShapeRate,
            QosCli.qosHwStatus.maxShapeRatePps )

def portShapeRateKbpsRangeFn( mode, context=None ):
   return ( QosCli.qosHwStatus.minPortShapeRate,
            QosCli.qosHwStatus.maxShapeRateKbps )

def portShapeRatePpsRangeFn( mode, context=None ):
   return ( QosCli.qosHwStatus.minPortShapeRate,
            QosCli.qosHwStatus.maxShapeRatePps )

def shapeRatePercentRangeFn( mode, context=None ):
   percent = Tac.Value( 'Qos::Percent' )
   return ( percent.min, percent.max )

class BurstValueExpressionFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      class ThdExpr( CliCommand.CliExpression ):
         expression = '( BURST_VAL_BYTES bytes ) | ( BURST_VAL_KBYTES kbytes ) | ' \
         ' ( BURST_VAL_MBYTES mbytes )'
         data = {
            'BURST_VAL_BYTES': burstValBytesMatcher,
            'bytes': "burst size in bytes",
            'BURST_VAL_KBYTES': burstValKBytesMatcher,
            'kbytes': "burst size in kbytes",
            'BURST_VAL_MBYTES': burstValMBytesMatcher,
            'mbytes': "burst size in mbytes",
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            burstUnitsToTypes = {
               'bytes': QosCli.tacBurstUnit.burstUnitBytes,
               'kbytes': QosCli.tacBurstUnit.burstUnitKBytes,
               'mbytes': QosCli.tacBurstUnit.burstUnitMBytes,
            }
            for unit in burstUnitsToTypes:
               unit = args.pop( unit, None )
               if unit:
                  # pylint: disable-next=consider-using-f-string
                  value = args.pop( 'BURST_VAL_%s' % unit.upper() )
                  args[ name ] = ( burstUnitsToTypes[ unit ], value )
                  break

      return ThdExpr

# ------------------------------------------------
# Tokens
# ------------------------------------------------


matcherShape = CliMatcher.KeywordMatcher( 'shape',
      helpdesc='Configure shaping parameters' )
nodeShape = CliCommand.Node( matcher=matcherShape, guard=guardPortShaping )
matcherRate = CliMatcher.KeywordMatcher( 'rate', helpdesc='Set rate' )
matcherKbps = CliMatcher.KeywordMatcher( 'kbps',
      helpdesc='Rate Unit Kbps (default)' )
matcherPortShapeRateValuePps = CliMatcher.DynamicIntegerMatcher(
      rangeFn=portShapeRatePpsRangeFn, helpdesc='Shape rate PPS' )
nodePortShapeRateValuePps = CliCommand.Node( matcher=matcherPortShapeRateValuePps,
      guard=guardPortShapeRatePps )
matcherTxQPortShapeRateUnitPps = CliMatcher.DynamicIntegerMatcher(
      rangeFn=txQShapeRatePpsRangeFn, helpdesc='Shape rate PPS' )
nodeTxQPortShapeRateUnitPps = CliCommand.Node(
      matcher=matcherTxQPortShapeRateUnitPps, guard=guardTxQShapeRatePps )
matcherPortShapeRateUnitPps = CliMatcher.KeywordMatcher( 'pps',
      helpdesc='Rate unit PPS' )
nodePortShapeRateUnitPps = CliCommand.Node( matcher=matcherPortShapeRateUnitPps,
      guard=guardPortShapeRatePps )
nodeTxQShapeRateUnitPps = CliCommand.Node( matcher=matcherPortShapeRateUnitPps,
      guard=guardTxQShapeRatePps )
matcherPortShapeRateValuePercent = CliMatcher.DynamicIntegerMatcher(
      rangeFn=shapeRatePercentRangeFn, helpdesc='Shape rate percent' )
nodePortShapeRateValuePercent = CliCommand.Node(
      matcher=matcherPortShapeRateValuePercent, guard=guardPortShaping )
matcherPortShapeRatePercent = CliMatcher.KeywordMatcher( 'percent',
      helpdesc='Rate unit percent' )
nodePortShapeRatePercent = CliCommand.Node(
      matcher=matcherPortShapeRatePercent, guard=guardPortShaping )
matcherShared = CliMatcher.KeywordMatcher( 'shared',
      helpdesc='Divide the shape rate equally across the port-channel members' )
nodeShared = CliCommand.Node( matcher=matcherShared, guard=guardSharedShaping )
matcherBurstSize = CliMatcher.KeywordMatcher( 'burst-size',
      helpdesc='Configure burst size' )
nodeBurstSize = CliCommand.Node( matcher=matcherBurstSize,
                                 guard=guardShapeRateBurstSize )
burstValBytesMatcher = CliMatcher.IntegerMatcher(
   1024, 511 * 1000 * 1000, helpdesc='burst-size (in bytes)' )
burstValKBytesMatcher = CliMatcher.IntegerMatcher(
   1, 511 * 1000, helpdesc='burst-size (in kbytes)' )
burstValMBytesMatcher = CliMatcher.IntegerMatcher(
   1, 511, helpdesc='burst-size (in mbytes)' )
burstValueMatcher = BurstValueExpressionFactory()

# ------------------------------------------------
# Cli Actions
# ------------------------------------------------
def shapeRateAdapter( mode, args, argsList ):
   if 'VALUE_KBPS' in args:
      args[ 'RATE' ] = args[ 'VALUE_KBPS' ]
      args[ 'UNIT' ] = args.get( 'kbps' )
   elif 'VALUE_PPS' in args:
      args[ 'RATE' ] = args[ 'VALUE_PPS' ]
      args[ 'UNIT' ] = args[ 'pps' ]
   elif 'VALUE_PERCENT' in args:
      args[ 'RATE' ] = args[ 'VALUE_PERCENT' ]
      args[ 'UNIT' ] = args[ 'percent' ]

# ------------------------------------------------
# Cli Command Registration
# ------------------------------------------------

# --------------------------------------------------------------------------------
# shape rate ( ( VALUE_KBPS [ kbps ] ) | ( VALUE_PPS pps ) |
#            ( VALUE_PERCENT percent ) [ burst-size BURST_VALUE BURST_UNIT ] )
# --------------------------------------------------------------------------------
class SetPortRateCmd( CliCommand.CliCommandClass ):
   syntax = ( 'shape rate ( VALUE_KBPS [ kbps ] ) | ( VALUE_PPS pps ) | '
               '( VALUE_PERCENT percent ) [ burst-size BURST ]' )
   noOrDefaultSyntax = 'shape rate ...'
   data = {
         'shape': nodeShape,
         'rate': matcherRate,
         'VALUE_KBPS': CliMatcher.DynamicIntegerMatcher(
            rangeFn=portShapeRateKbpsRangeFn, helpdesc='Shape rate Kbps' ),
         'kbps': matcherKbps,
         'VALUE_PPS': nodePortShapeRateValuePps,
         'pps': nodePortShapeRateUnitPps,
         'VALUE_PERCENT': nodePortShapeRateValuePercent,
         'percent': nodePortShapeRatePercent,
         'burst-size': nodeBurstSize,
         'BURST': burstValueMatcher,
   }

   adapter = shapeRateAdapter

   handler = 'QosCliShapeHandler.setPortRateCmdHandler'
   noOrDefaultHandler = handler


QosCliCommon.QosProfileMode.addCommandClass( SetPortRateCmd )
QosModeletShapeRate.addCommandClass( SetPortRateCmd )

# --------------------------------------------------------------------------------
# shape rate ( VALUE_KBPS [ kbps ] ) | ( VALUE_PPS pps ) |
#            ( VALUE_PERCENT percent ) [ burst-size BURST_VALUE BURST_UNIT ]
#            in tx-queue mode
# --------------------------------------------------------------------------------
class SetShapeRateCmd( CliCommand.CliCommandClass ):
   syntax = ( 'shape rate ( VALUE_KBPS [ kbps ] ) | ( VALUE_PPS pps ) | '
              '( VALUE_PERCENT percent ) [ burst-size BURST ]' )
   noOrDefaultSyntax = 'shape rate ...'

   data = {
         'shape': nodeShape,
         'rate': matcherRate,
         'VALUE_KBPS': CliMatcher.DynamicIntegerMatcher(
            rangeFn=txQShapeRateKbpsRangeFn, helpdesc='Shape rate Kbps' ),
         'kbps': matcherKbps,
         'VALUE_PPS': nodeTxQPortShapeRateUnitPps,
         'pps': nodeTxQShapeRateUnitPps,
         'VALUE_PERCENT': nodePortShapeRateValuePercent,
         'percent': nodePortShapeRatePercent,
         'burst-size': nodeBurstSize,
         'BURST': burstValueMatcher,
   }

   adapter = shapeRateAdapter

   handler = 'QosCliShapeHandler.setShapeRateCmdHandler'
   noOrDefaultHandler = handler


QosCliCommon.IntfTxQueueConfigMode.addCommandClass( SetShapeRateCmd )

# --------------------------------------------------------------------------------
# shape rate ( VALUE_KBPS [ kbps ] ) | ( VALUE_PPS pps ) |
#            ( VALUE_PERCENT percent ) shared
#            [ burst-size BURST_VALUE BURST_UNIT ]
# --------------------------------------------------------------------------------
class SetPortRateSharedCmd( CliCommand.CliCommandClass ):
   syntax = ( 'shape rate ( VALUE_KBPS [ kbps ] ) | ( VALUE_PPS pps ) | '
              '( VALUE_PERCENT percent ) [ burst-size BURST ] shared' )
   data = {
         'shape': nodeShape,
         'rate': matcherRate,
         'VALUE_KBPS': CliMatcher.DynamicIntegerMatcher(
            rangeFn=portShapeRateKbpsRangeFn, helpdesc='Shape rate Kbps' ),
         'kbps': matcherKbps,
         'VALUE_PPS': nodePortShapeRateValuePps,
         'pps': nodePortShapeRateUnitPps,
         'VALUE_PERCENT': nodePortShapeRateValuePercent,
         'percent': nodePortShapeRatePercent,
         'burst-size': nodeBurstSize,
         'BURST': burstValueMatcher,
         'shared': nodeShared,
   }

   adapter = shapeRateAdapter

   handler = 'QosCliShapeHandler.setPortRateSharedCmdHandler'


QosCliCommon.QosProfileMode.addCommandClass( SetPortRateSharedCmd )
LagIntfCli.LagIntfConfigModelet.addCommandClass( SetPortRateSharedCmd )

# --------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# --------------------------------------------------------------------------------
@Plugins.plugin( provides=( "QosCliShape", ) )
def Plugin( entityManager ):
   global qosInputConfig, qosHwStatus, subIntfHwStatus

   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )
