#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import MultiRangeRule
import Plugins
from CliPlugin import ( QosCliWred, QosCliServicePolicy, QosCliCommon )
from CliPlugin.QosCliWred import ( RandomDetectCmdsThdWithLatencyFactory,
                                   RandomDetectCmdsThdFactory,
                                   nodeTxQueueRandomDetect )
from CliPlugin.QosCliCommon import IntfTxQueueConfigMode
from QosLib import isTxQueueNotConfigurable

# -----------------------------------------------------------------------------------
# Variables for Qos Modelets associated mount paths from Sysdb
# -----------------------------------------------------------------------------------
qosHwStatus = None
subIntfHwStatus = None
qosGlobalConfig = None

# -----------------------------------------------------------------------------------
# Guards
# -----------------------------------------------------------------------------------
def guardQueueWeight( mode, token ):
   if isinstance( mode, IntfTxQueueConfigMode ):
      if isTxQueueNotConfigurable( qosHwStatus, mode.txQueue.id ):
         return CliParser.guardNotThisPlatform
   if qosHwStatus.queueWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNonEctParams( mode, token ):
   if qosHwStatus.nonEctThdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNonEctWeight( mode, token ):
   if qosHwStatus.nonEctWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardSubIntfTxQueue( mode, token ):
   if not qosHwStatus.hwInitialized or 0 == qosHwStatus.numTxQueueSupported:
      return CliParser.guardNotThisPlatform
   if qosHwStatus.numMulticastQueueSupported != 0:
      return CliParser.guardNotThisPlatform
   if ( not subIntfHwStatus.subIntfTxQueueShapeRateSupported and
        not subIntfHwStatus.subIntfTxQueueSchSupported ):
      return CliParser.guardNotThisPlatform
   return None

def subIntfTxQueueRangeFn( mode=None ):
   if qosGlobalConfig.numTxqsPerSubIntf == 8 and \
      qosHwStatus.tc7ToAnyTxQueueRestricted:
      if qosHwStatus.isTxQueue7Configurable:
         return ( 0, qosHwStatus.numTxQueueSupported - 1 )
      else:
         return ( 0, qosHwStatus.numTxQueueSupported - 2 )
   return ( 0, qosGlobalConfig.numTxqsPerSubIntf - 1 )


# -----------------------------------------------------------------------------------
# Matchers
# -----------------------------------------------------------------------------------
matcherDroprate = CliMatcher.IntegerMatcher( 1, 100,
      helpdesc='Specify maximum drop rate' )
matcherMinimumThresholdEcn = CliMatcher.KeywordMatcher( 'minimum-threshold',
      helpdesc='Set minimum threshold for ECN' )
thresholdMatcher = RandomDetectCmdsThdFactory()
thresholdLatencyMatcher = RandomDetectCmdsThdWithLatencyFactory()
matcherSubIntfTxQueueRange = MultiRangeRule.MultiRangeMatcher(
      rangeFn=subIntfTxQueueRangeFn, noSingletons=False,
      helpdesc="Tx-Queue ID or range(s) of Tx-Queue IDs" )

nodeSubIntfTxQueue = CliCommand.guardedKeyword( 'tx-queue',
      helpdesc='Configure transmit queue parameters',
      guard=guardSubIntfTxQueue )
nodeTxQueue = CliCommand.guardedKeyword( 'tx-queue',
      helpdesc='Configure transmit queue parameters',
      guard=QosCliServicePolicy.guardTxQueue )
nodeUcTxQueue = CliCommand.guardedKeyword( 'uc-tx-queue',
      helpdesc='Configure unicast transmit queue parameters',
      guard=QosCliServicePolicy.guardUcMcTxQueue )
nodeMcTxQueue = CliCommand.guardedKeyword( 'mc-tx-queue',
      helpdesc='Configure multicast transmit queue parameters',
      guard=QosCliServicePolicy.guardUcMcTxQueue )

# --------------------------------------------------------------------------------
# [ no | default ] queue length weight <WEIGHT>
# --------------------------------------------------------------------------------
class AvrgWeightCmd( CliCommand.CliCommandClass ):
   syntax = 'queue length weight WEIGHT'
   noOrDefaultSyntax = 'queue length weight ...'
   data = {
      'queue': CliCommand.guardedKeyword( 'queue',
                                          helpdesc='Set queue parameters',
                                          guard=guardQueueWeight ),
      'length': 'Monitor queue lengths',
      'weight': 'weight',
      'WEIGHT': CliMatcher.DynamicIntegerMatcher(
                                           QosCliWred.weightRangeFn,
                                           helpdesc='Value of weight' ),
      }
   handler = 'QosModeletsHandler.setWeight'
   noOrDefaultHandler = 'QosModeletsHandler.setNoOrDefaultWeight'


QosCliCommon.IntfUcTxQueueModelet.addCommandClass( AvrgWeightCmd )

# --------------------------------------------------------------------------------
# [ no | default ] random-detect non-ect minimum-threshold <MIN_THD> <unit>
# maximum-threshold <MAX_THD> <unit> [ weight WEIGHT ]
# unit can be either bytes, kbytes, mbytes, or segments
# --------------------------------------------------------------------------------
class RandomDetectNonEctCmd( CliCommand.CliCommandClass ):
   syntax = 'random-detect non-ect minimum-threshold THD_EXPR [ weight WEIGHT ]'
   noOrDefaultSyntax = 'random-detect non-ect ...'
   data = {
      'random-detect': nodeTxQueueRandomDetect,
      'non-ect': CliCommand.guardedKeyword( 'non-ect',
                                            helpdesc='Set NON-ECT parameters',
                                            guard=guardNonEctParams ),
      'minimum-threshold': matcherMinimumThresholdEcn,
      'THD_EXPR': thresholdMatcher,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set NON-ECT weight',
                                           guard=guardNonEctWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher(
                                           QosCliWred.weightRangeFn,
                                           helpdesc='Value of NON-ECT weight' ),
   }

   adapter = QosCliWred.randomDetectCmdsAdapter
   handler = 'QosModeletsHandler.setNonEctParams'
   noOrDefaultHandler = 'QosModeletsHandler.setNoOrDefaultNonEctParams'


QosCliCommon.IntfUcTxQueueModelet.addCommandClass( RandomDetectNonEctCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ( ( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET ) |
#                                          ( mc-tx-queue MCTXQSET ) )
# --------------------------------------------------------------------------------
class QueueSetIntfRangeModeletCmd( CliCommand.CliCommandClass ):
   syntax = ( '( ( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET ) | '
              '( mc-tx-queue MCTXQSET ) )' )
   noOrDefaultSyntax = syntax
   data = {
      'tx-queue': nodeTxQueue,
      'uc-tx-queue': nodeUcTxQueue,
      'mc-tx-queue': nodeMcTxQueue,
      'TXQSET': QosCliServicePolicy.matcherTxQueueRange,
      'UCTXQSET': QosCliServicePolicy.matcherUcTxQueueRange,
      'MCTXQSET': QosCliServicePolicy.matcherMcTxQueueRange,
   }

   handler = 'QosModeletsHandler.queueSetIntfRangeModeletCmdHandler'
   noOrDefaultHandler =\
      'QosModeletsHandler.queueSetIntfRangeModeletCmdNoOrDefaultHandler'


QosCliCommon.QosModelet.addCommandClass( QueueSetIntfRangeModeletCmd )
QosCliCommon.QosProfileMode.addCommandClass( QueueSetIntfRangeModeletCmd )

# --------------------------------------------------------------------------------
# [ no | default ] ( tx-queue TXQSET )
# --------------------------------------------------------------------------------
class QueueSetSubIntfRangeModeletCmd( CliCommand.CliCommandClass ):
   syntax = ( 'tx-queue TXQSET' )
   noOrDefaultSyntax = syntax
   data = {
      'tx-queue': nodeSubIntfTxQueue,
      'TXQSET': matcherSubIntfTxQueueRange,
   }

   handler = 'QosModeletsHandler.queueSetSubIntfRangeModeletCmdHandler'
   noOrDefaultHandler =\
      'QosModeletsHandler.queueSetSubIntfRangeModeletCmdNoOrDefaultHandler'


QosCliCommon.QosSubIntfModelet.addCommandClass( QueueSetSubIntfRangeModeletCmd )

@Plugins.plugin( provides=( "QosModelets", ) )
def Plugin( entityManager ):
   global qosHwStatus, subIntfHwStatus, qosGlobalConfig
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )
   qosGlobalConfig = ConfigMount.mount( entityManager, "qos/global/config",
                                        "Qos::GlobalConfig", "w" )
