# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import datetime
import Arnet
import TableOutput
import Tac
import TacSigint
from Ark import (
   timestampToStr,
)
from ArnetModel import (
   MacAddress,
)
from CliModel import (
      Bool,
      Dict,
      Enum,
      Float,
      Int,
      List,
      Model,
      Str,
      Submodel,
)
from Ethernet import (
   convertMacAddrToDisplay,
)
from GeneratorCliUtils import (
   generatorDirectionShowStr,
   packetRateShowStr,
)
from IntfModels import (
   Interface,
)
from RFC2544InitiatorCliUtils import (
   overallTestStateShowStr,
   individualTestStateShowStr,
   TestFailureReasonEnum,
   TestStateEnum,
   testFailureReasonShowStr,
   benchmarkTestSupportedKwStr,
)
from collections import (
   namedtuple,
)

UniqueId = namedtuple( 'UniqueId', [ 'timestamp', 'pid', 'count' ] )

def macAddrStrOrNone( macAddr ):
   if macAddr is None:
      return 'none'
   else:
      return convertMacAddrToDisplay( macAddr.stringValue )

class PacketRateAndUnitModel( Model ):
   rate = Float( help="Packet rate" )
   unit = Enum( values=packetRateShowStr,
                help="Packet rate unit" )

class TestTrialBaseModel( Model ):
   state = Enum( values=overallTestStateShowStr,
                 help="State of test trial" )
   duration = Float( help="Trial duration in seconds", optional=True )
   packetsSent = Int( help="Number of packets sent by the generator",
                      optional=True )
   packetsReceived = Int( help="Number of packets received by the generator",
                          optional=True )
   # Ideally packet rate should always exist.
   # The only reason packetRate is optional is to handle the case
   # where packetRateUnit is not set properly in agent code and is
   # "rateUnitInvalid", in which case there is no point of showing
   # any packet rate value.
   packetRate = Submodel( valueType=PacketRateAndUnitModel,
                          help="Trial packet rate",
                          optional=True )

class TestResultBaseModel( Model ):
   state = Enum( values=individualTestStateShowStr,
                 help="Individual state of test" )
   stateReason = Enum( values=TestFailureReasonEnum.attributes,
                       help="Reason for test state", optional=True )

class TestReportBaseModel( Model ):
   state = Enum( values=overallTestStateShowStr,
                 help="Overall state of test" )
   stateReason = Enum( values=TestFailureReasonEnum.attributes,
                       help="Reason for test state",
                       optional=True )
   startTime = Float( help="Timestamp when test was started", optional=True )
   endTime = Float( help="Timestamp when test was ended", optional=True )

# Even though currently there is nothing extra in ThroughputTestTrialModel
# as compared to TestTrialBaseModel, it is better to keep it separate
# so that in future it will be easier to add throughput specific details.
# If we directly use TestTrialBaseModel as "trials" in
# ThroughputTestResultDetailModel, and in future we need to add throughput
# specific details, we will need to deprecate the old model, add new model
# which might not be the right thing to do.
class ThroughputTestTrialModel( TestTrialBaseModel ):
   pass 

class ThroughputTestResultDetailModel( Model ):
   startTime = Float( help="Timestamp when test was started", optional=True )
   endTime = Float( help="Timestamp when test was ended", optional=True )
   trials = Dict( keyType=int, valueType=ThroughputTestTrialModel,
                  help="A mapping of test trials to results" )

# Throughput test result per packet size
class ThroughputTestResultModel( TestResultBaseModel ):
   packetsSent = Int( help="Number of packets sent by the generator for this test",
                      optional=True )
   rateAndUnit = Submodel( valueType=PacketRateAndUnitModel,
                           help="Final throughput rate",
                           optional=True )
   detail = Submodel( valueType=ThroughputTestResultDetailModel,
                      help="Detailed throughput test trial results",
                      optional=True )

class ThroughputTestReportModel( TestReportBaseModel ):
   results = Dict(
      keyType=int, valueType=ThroughputTestResultModel,
      help="A mapping of test packet sizes to throughput test results"
      )

class FrameLossRateTestTrialModel( TestTrialBaseModel ):
   stateReason = Enum( values=TestFailureReasonEnum.attributes,
                       help="Reason for test state",
                       optional=True )

# Frame loss rate test result per packet size
class FrameLossRateTestResultModel( TestResultBaseModel ):
   startTime = Float( help="Timestamp when test was started", optional=True )
   endTime = Float( help="Timestamp when test was ended", optional=True )
   # Frame loss rate test result has no "summary result" as such.
   # It consists of multiple trials starting with max packet rate
   # and then subsequent trials at lower packet rates until we see two trials
   # with no packet loss. We need to display each trial and the corresponding
   # packet loss as part of Frame Loss Rate test. So, both summary and detailed
   # version of the show command contain the same information.
   trials = Dict( keyType=int, valueType=FrameLossRateTestTrialModel,
                  help="A mapping of test trials to results" )

class FrameLossRateTestReportModel( TestReportBaseModel ):
   rateAndUnit = Submodel( valueType=PacketRateAndUnitModel,
                           help="Maximum packet rate",
                           optional=True )
   results = Dict(
      keyType=int, valueType=FrameLossRateTestResultModel,
      help="A mapping of test packet sizes to frame loss rate test results"
      )

class ExecIdModel( Model ):
   timestamp = Int( help="UTC timestamp" )
   pid = Int( help="PID" )
   count = Int( help="Count" )

# Following class object represents all the tests run as part of one
# exec command.
class TestReportModel( Model ):
   profileName = Str( help="RFC2544 inititator profile name",
                      optional=True )
   direction = Enum( values=generatorDirectionShowStr,
                     help="Direction of packets injected by generator",
                     optional=True )
   sourceMac = MacAddress( help="Source MAC address in test packets",
                           optional=True )
   destinationMac = MacAddress( help="Destination MAC address in test packets",
                                optional=True )
   # The test reports per interface are saved in a collection indexed on
   # "execId : Ark::UniqueId;".
   execId = Submodel( valueType=ExecIdModel,
                      help="Unique ID for this test report" )
   # Keeping each benchmark test as a separate SubModel is needed because
   # each test will have different things in the results.
   throughputTest = Submodel( valueType=ThroughputTestReportModel,
                              help="Throughput test report",
                              optional=True )
   frameLossRateTest = Submodel( valueType=FrameLossRateTestReportModel,
                                 help="Frame loss rate test report",
                                 optional=True )

class IntfTestReportModel( Model ):
   # Initially we will have only the latest test result saved. Having it
   # as a List will make it scalable in future when we support "last N results"
   # filter for the show command
   testReports = List( valueType=TestReportModel,
                       help="Historic test reports" )

class Rfc2544TestReportModel( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=IntfTestReportModel,
                      help="A mapping of interfaces to its test reports" )
   enabled = Bool( help="RFC2544 generator is enabled and running" )

   def renderDetailedThroughputResult( self, packetSize,
                                       throughputTestResultDetail ):
      startTimeStr = "-"
      endTimeStr = "-"
      durationStr = "-"
      testStartTime = throughputTestResultDetail.startTime
      testEndTime = throughputTestResultDetail.endTime
      if testStartTime is not None:
         startTimeStr = \
            timestampToStr( testStartTime, relative=False ) + \
            " (" + timestampToStr( testStartTime ) + ")"
      if testEndTime is not None:
         endTimeStr = \
            timestampToStr( testEndTime, relative=False ) + \
            " (" + timestampToStr( testEndTime ) + ")"
      if testStartTime is not None and testEndTime is not None:
         durationStr = str( datetime.timedelta(
                         seconds=( testEndTime - testStartTime ) ) )
      print( "Packet size:", str( packetSize ) + " bytes" )
      print( "Start time:", startTimeStr )
      print( "End time:", endTimeStr )
      print( "Test duration:", durationStr )
      trialResults = throughputTestResultDetail.trials
      if not trialResults:
         print( "No test trial results available" )
         print( "" )
         return

      headings = ( "Trial", "State", "Duration", "Packets Sent",
                   "Packets Received", "Loss %", "Traffic Rate" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      detailedTable = TableOutput.createTable( headings )
      detailedTable.formatColumns(
         formatRight, formatLeft, formatRight, formatRight,
         formatRight, formatRight, formatRight )

      addWarning = False
      for trialId in sorted( trialResults ):
         trialResult = trialResults[ trialId ]
         stateStr = overallTestStateShowStr[ trialResult.state ]
         durationStr = "-"
         packetsSent = "-"
         packetsReceived = "-"
         loss = "-"
         packetRate = "-"
         if trialResult.duration is not None:
            durationStr = str( datetime.timedelta(
                            seconds=( trialResult.duration ) ) )
         testPacketsSent_ = trialResult.packetsSent
         testPacketsReceived_ = trialResult.packetsReceived
         if testPacketsSent_ is not None:
            packetsSent = testPacketsSent_
         if testPacketsReceived_ is not None:
            packetsReceived = testPacketsReceived_
         if testPacketsSent_ and testPacketsReceived_ and \
            testPacketsSent_ >= testPacketsReceived_:
            lossValue = ( ( testPacketsSent_ - testPacketsReceived_ ) /
                          testPacketsSent_ ) * 100
            loss = float( '%.3f' % ( lossValue ) )
         if testPacketsReceived_ > testPacketsSent_ and \
            trialResult.state == TestStateEnum.flowStateRunning:
            addWarning = True
         if trialResult.packetRate:
            packetRate = str(
               float( '%.3f' % trialResult.packetRate.rate ) ) + \
               " " + packetRateShowStr[ trialResult.packetRate.unit ]

         detailedTable.newRow( trialId, stateStr, durationStr, packetsSent,
                               packetsReceived, loss, packetRate )
      print( detailedTable.output() )
      if addWarning:
         print( "! The number of packets received may not match the "
                "number of packets sent while a test is running" )
         print( "" )

   def renderThroughputTestReport( self, throughputTest ):
      print( "Test: throughput" )
      stateStr = "State: " + overallTestStateShowStr[
                 throughputTest.state ]
      if throughputTest.stateReason is not None:
         reasonStr = testFailureReasonShowStr.get( throughputTest.stateReason )
         if reasonStr is not None:
            stateStr += ", Reason: " + reasonStr
      print( stateStr )
      # CliReview approved syntax for showing start time and end time is:
      # '2022-01-24 13:37:21 (0:03:23 ago)'
      # Since startTime and endTime is saved using Ark::UniqueId, we don't need
      # to provide "now" argument to timestampToStr() function call.
      startTime = "-"
      endTime = "-"
      duration = "-"
      if throughputTest.startTime is not None:
         startTime = \
            timestampToStr( throughputTest.startTime, relative=False ) + \
            " (" + timestampToStr( throughputTest.startTime ) + ")"
      if throughputTest.endTime is not None:
         endTime = \
            timestampToStr( throughputTest.endTime, relative=False ) + \
            " (" + timestampToStr( throughputTest.endTime ) + ")"
      if throughputTest.state not in [ None,
         TestStateEnum.flowStateRunning, TestStateEnum.flowStateNone ]:
         duration = str( datetime.timedelta(
                         seconds=( throughputTest.endTime -
                                   throughputTest.startTime ) ) )
      print( "Start time:", startTime )
      print( "End time:", endTime )
      print( "Test duration:", duration )

      results_ = throughputTest.results
      if not results_:
         print( "No throughput result available" )
         print( "" )
         return

      headings = ( "Packet Size (Bytes)", "Packets Sent", "Throughput",
                   "State" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatRight, formatRight, formatRight, formatLeft )

      detail = False
      for packetSize in sorted( results_ ):
         throughputTestResult = results_[ packetSize ]
         if throughputTestResult.detail is not None:
            detail = True
         throughputTestStateStr = individualTestStateShowStr[
            throughputTestResult.state ]
         if throughputTestResult.stateReason:
            throughputReasonStr = testFailureReasonShowStr.get(
               throughputTestResult.stateReason )
            if throughputReasonStr is not None:
               throughputTestStateStr += ", Reason: " + throughputReasonStr
         throughputRateAndUnit = "-"
         packetsSent = "-"
         if throughputTestResult.rateAndUnit is not None and \
            throughputTestResult.rateAndUnit.rate:
            rateAndUnit = throughputTestResult.rateAndUnit
            throughputRateAndUnit = str(
               float( '%.3f' % rateAndUnit.rate ) ) + \
               " " + packetRateShowStr[ rateAndUnit.unit ]
         if throughputTestResult.packetsSent:
            packetsSent = throughputTestResult.packetsSent
         table.newRow( packetSize, packetsSent,
                       throughputRateAndUnit, throughputTestStateStr )
      if detail:
         print( "Final result:" )
      print( table.output() )

      if detail:
         for packetSize in sorted( results_ ):
            throughputTestResult = results_[ packetSize ]
            self.renderDetailedThroughputResult(
               packetSize, throughputTestResult.detail )

   def renderFrameLossRateTestResult( self, packetSize,
                                      frameLossRateTestResult,
                                      firstPacketSize ):
      if firstPacketSize:
         print( "" )
      print( "Packet size:", str( packetSize ) + " bytes" )
      # We want to say "success" if the frame loss rate test completed
      # with two subsequent trials without any packet loss. Hence use,
      # individualTestStateShowStr
      stateStr = "State: " + individualTestStateShowStr[
                 frameLossRateTestResult.state ]
      if frameLossRateTestResult.stateReason is not None:
         reasonStr = testFailureReasonShowStr.get(
            frameLossRateTestResult.stateReason )
         if reasonStr is not None:
            stateStr += ", Reason: " + reasonStr
      print( stateStr )
      # CliReview approved syntax for showing start time and end time is:
      # '2022-01-24 13:37:21 (0:03:23 ago)'
      # Since startTime and endTime is saved using Ark::UniqueId, we don't need
      # to provide "now" argument to timestampToStr() function call.
      startTime = "-"
      endTime = "-"
      duration = "-"
      if frameLossRateTestResult.startTime is not None:
         startTime = \
            timestampToStr( frameLossRateTestResult.startTime, relative=False ) + \
            " (" + timestampToStr( frameLossRateTestResult.startTime ) + ")"
      if frameLossRateTestResult.endTime is not None:
         endTime = \
            timestampToStr( frameLossRateTestResult.endTime, relative=False ) + \
            " (" + timestampToStr( frameLossRateTestResult.endTime ) + ")"
      if frameLossRateTestResult.state not in [ None,
         TestStateEnum.flowStateRunning, TestStateEnum.flowStateNone ]:
         duration = str( datetime.timedelta(
                         seconds=( frameLossRateTestResult.endTime -
                                   frameLossRateTestResult.startTime ) ) )
      print( "Start time:", startTime )
      print( "End time:", endTime )
      print( "Test duration:", duration )

      trialResults = frameLossRateTestResult.trials
      if not trialResults:
         print( "No test trial results available" )
         print( "" )
         return

      headings = ( "Traffic Rate", "Duration", "Packets Sent",
                   "Packets Received", "Packets Lost", "State" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      detailedTable = TableOutput.createTable( headings )
      detailedTable.formatColumns(
         formatRight, formatRight, formatRight, formatRight,
         formatRight, formatLeft )

      addWarning = False
      for trialId in sorted( trialResults ):
         trialResult = trialResults[ trialId ]
         # Whether the trial experienced some packet loss or not, the
         # sysdb state is "success" because we were able to "complete"
         # the trial. So here, in case of "success" we actually mean
         # "completed", hence using overallTestStateShowStr.
         trialStateStr = overallTestStateShowStr[ trialResult.state ]
         if trialResult.stateReason is not None:
            trialReasonStr = testFailureReasonShowStr.get(
               trialResult.stateReason )
            if trialReasonStr is not None:
               trialStateStr += ", Reason: " + trialReasonStr
         packetRate = "-"
         durationStr = "-"
         packetsSent = "-"
         packetsReceived = "-"
         packetsLost = "-"
         # Packet Rate should ideally always exist. This is only being extra
         # cautious here in case the packet rate unit is invalid or not set
         # correctly in sysdb by agent code.
         if trialResult.packetRate:
            packetRate = str(
               float( '%.3f' % trialResult.packetRate.rate ) ) + \
               " " + packetRateShowStr[ trialResult.packetRate.unit ]
         if trialResult.duration is not None:
            durationStr = str( datetime.timedelta(
                               seconds=( trialResult.duration ) ) )
         testPacketsSent_ = trialResult.packetsSent
         testPacketsReceived_ = trialResult.packetsReceived
         if testPacketsSent_ is not None:
            packetsSent = testPacketsSent_
         if testPacketsReceived_ is not None:
            packetsReceived = testPacketsReceived_
         if testPacketsSent_ and testPacketsReceived_ and \
            testPacketsSent_ >= testPacketsReceived_:
            lossValue = testPacketsSent_ - testPacketsReceived_
            loss = float( '%.3f' % ( lossValue / testPacketsSent_ * 100 ) )
            packetsLost = str( lossValue ) + " (" + str( loss ) + " %)" 
         if testPacketsReceived_ > testPacketsSent_ and \
            trialResult.state == TestStateEnum.flowStateRunning:
            addWarning = True
         detailedTable.newRow( packetRate, durationStr, packetsSent,
                               packetsReceived, packetsLost, trialStateStr )
      print( detailedTable.output() )
      if addWarning:
         print( "! The number of packets received may not match the "
                "number of packets sent while a test is running" )
         print( "" )

   def renderFrameLossRateTestReport( self, frameLossRateTest ):
      print( "Test: frame loss rate" )
      rateAndUnitStr = "-"
      if frameLossRateTest.rateAndUnit is not None:
         rateAndUnit = frameLossRateTest.rateAndUnit
         rateAndUnitStr = str(
            float( '%.3f' % rateAndUnit.rate ) ) + \
            " " + packetRateShowStr[ rateAndUnit.unit ]
      print( "Traffic rate:", rateAndUnitStr )
      stateStr = "State: " + overallTestStateShowStr[
                 frameLossRateTest.state ]
      if frameLossRateTest.stateReason is not None:
         reasonStr = testFailureReasonShowStr.get( frameLossRateTest.stateReason )
         if reasonStr is not None:
            stateStr += ", Reason: " + reasonStr
      print( stateStr )
      # CliReview approved syntax for showing start time and end time is:
      # '2022-01-24 13:37:21 (0:03:23 ago)'
      # Since startTime and endTime is saved using Ark::UniqueId, we don't need
      # to provide "now" argument to timestampToStr() function call.
      startTime = "-"
      endTime = "-"
      duration = "-"
      if frameLossRateTest.startTime is not None:
         startTime = \
            timestampToStr( frameLossRateTest.startTime, relative=False ) + \
            " (" + timestampToStr( frameLossRateTest.startTime ) + ")"
      if frameLossRateTest.endTime is not None:
         endTime = \
            timestampToStr( frameLossRateTest.endTime, relative=False ) + \
            " (" + timestampToStr( frameLossRateTest.endTime ) + ")"
      if frameLossRateTest.state not in [ None,
         TestStateEnum.flowStateRunning, TestStateEnum.flowStateNone ]:
         duration = str( datetime.timedelta(
                         seconds=( frameLossRateTest.endTime -
                                   frameLossRateTest.startTime ) ) )
      print( "Start time:", startTime )
      print( "End time:", endTime )
      print( "Test duration:", duration )

      # Per packet size results
      results_ = frameLossRateTest.results
      if not results_:
         print( "No frame loss rate result available" )
         print( "" )
         return

      # firstPacketSize is needed to explicitly print a blank line before the
      # first packet size in the show output. For other packetSizes,
      # the blank line is added by default by the table.
      firstPacketSize = True
      for packetSize in sorted( frameLossRateTest.results ):
         frameLossRateTestResult = frameLossRateTest.results[ packetSize ]
         self.renderFrameLossRateTestResult(
            packetSize, frameLossRateTestResult, firstPacketSize )
         firstPacketSize = False

   def renderTestReport( self, testReport ):
      # If following things are not available, then it will
      # display 'none' for those entries.
      profileName = 'none'
      if testReport.profileName:
         profileName = testReport.profileName
      print( "Profile:", profileName )
      if testReport.direction is None:
         directionStr = "none"
      else:
         directionStr = generatorDirectionShowStr[ testReport.direction ]
      print( "Direction:", directionStr )
      print( "Source MAC:", macAddrStrOrNone( testReport.sourceMac ) )
      print( "Destination MAC:", macAddrStrOrNone( testReport.destinationMac ) )
      # If any test report is present, set reportPresent to True.
      # If there is no test report present display "No test report available".
      reportPresent = False
      if testReport.throughputTest is not None:
         self.renderThroughputTestReport( testReport.throughputTest )
         reportPresent = True
      if testReport.frameLossRateTest is not None:
         self.renderFrameLossRateTestReport( testReport.frameLossRateTest )
         reportPresent = True
      if not reportPresent:
         print ( "No test result available" )
         print ( "" )

   def renderIntfTestReport( self, intf, intfTestReport ):
      print( "Interface:", intf )
      if not intfTestReport.testReports:
         # In case user entered some interface using INTF filter in the show
         # command and there is no test report available for that interface.
         print ( "No test report available" )
         print ( "" )
         return
      # We cannot use "Ark::UniqueId" direrctly as a key for python
      # dictionary. Hence, using the namedtuple "UniqueId".
      execIdToTestReportMap = {}
      for testReport in intfTestReport.testReports:
         execId = testReport.execId
         execIdToTestReportMap[
            UniqueId( execId.timestamp, execId.pid, execId.count ) ] = testReport

      # Display the most recent report first
      for execId in sorted( execIdToTestReportMap, reverse=True ):
         self.renderTestReport( execIdToTestReportMap[ execId ] )

   def render( self ):
      if not self.enabled:
         print( "RFC2544 generator is not enabled" )
         return
      if not self.interfaces:
         print ( "No test report available" )
         return
      for intf in Arnet.sortIntf( self.interfaces ):
         self.renderIntfTestReport( intf, self.interfaces[ intf ] )

class BenchmarkTestModel( Model ):
   test = Enum( values=benchmarkTestSupportedKwStr,
                help="RFC2544 benchmark test" )

class Rfc2544ProfileModel( Model ):
   benchmarkTests = List( valueType=BenchmarkTestModel,
                          help="List of RFC2544 benchmark tests" )
   direction = Enum( values=generatorDirectionShowStr,
                     help="Direction of packets injected by generator" )
   packetSizes = List( valueType=int,
                       help="List of packet sizes" )
   duration = Float( help="Test duration in seconds" )
   packetRate = Submodel( valueType=PacketRateAndUnitModel,
                          help="Packet rate with unit",
                          optional=True )
   sourceMac = MacAddress( help="Source MAC address in test packets" )
   destinationMac = MacAddress( help="Destination MAC address in test packets" )

class Rfc2544StatusModel( Model ):
   # Depends on generator/featureConfig
   enabled = Bool( help="RFC2544 generator is enabled" )
   # Depends on generator/featureStatus and generator/hwFeatureStatus
   running = Bool( help="RFC2544 generator is running" )
   profileNames = Dict( keyType=str, valueType=Rfc2544ProfileModel,
                        help="A mapping of profile names to test profiles" )

   def renderProfiles( self, profiles ):
      if not profiles:
         print( "No profiles configured" )
         return

      for profileName in sorted( profiles ):
         profile = profiles[ profileName ]
         print( "Profile:", profileName )
         benchmarkTestStr = "none"
         if profile.benchmarkTests:
            benchmarkTestStr = ""
            # profileConfig.benchmarkTest is an ordered collection, indexed on
            # the enum Rfc2544Initiator::BenchmarkTest.
            # Since Rfc2544ProfileModel.benchmarkTests is a list, the order will be
            # preserved and we should not sort it here again.
            for benchmarkTest in profile.benchmarkTests:
               if benchmarkTestStr:
                  benchmarkTestStr += ", "
               benchmarkTestStr += benchmarkTestSupportedKwStr[ benchmarkTest.test ]
         print( "Test:", benchmarkTestStr )
         print( "Direction:", generatorDirectionShowStr[ profile.direction ] )
         packetSizeStr = "none"
         if profile.packetSizes:
            packetSizeStr = ""
            for packetSize in sorted( profile.packetSizes ):
               if packetSizeStr:
                  packetSizeStr += ", "
               packetSizeStr += str( packetSize )
            packetSizeStr += " bytes"
         print( "Packet size:", packetSizeStr )
         print( 'Test duration:', str( profile.duration ) + ' seconds' )
         rateStr = "none"
         if profile.packetRate is not None:
            rateAndUnit = profile.packetRate
            rateStr = str( float( '%.3f' % rateAndUnit.rate ) ) + \
                      " " + packetRateShowStr[ rateAndUnit.unit ]
         print( "Traffic rate:", rateStr )
         print( "Source MAC:", macAddrStrOrNone( profile.sourceMac ) )
         print( "Destination MAC:", macAddrStrOrNone( profile.destinationMac ) )
         print( "" )

   def render( self ):
      reasonStr = ""
      stateStr = ""
      if self.running:
         stateStr = "active"
      elif self.enabled:
         stateStr = "inactive"
         reasonStr = "RFC2544 generator is not programmed in hardware"
      elif not self.enabled:
         stateStr = "inactive"
         reasonStr = "RFC2544 is not enabled"

      if reasonStr:
         stateStr +=  ", Reason: " + reasonStr
      print( "Status:", stateStr )

      self.renderProfiles( self.profileNames )
