# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import decimal
import BasicCli
import CliCommand
import LazyMount
import ShowCommand
import Tac
import Tracing
from CliPlugin.GeneratorCliLib import (
   generatorsNode,
   interfaceNode,
   rfc2544ConfigKwMatcher,
)
from CliPlugin.RFC2544InitiatorCli import (
   intfMatcher,
   rfc2544ProfileMatcher,
)
from CliPlugin.RFC2544InitiatorCliModel import (
   ThroughputTestResultModel,
   ThroughputTestReportModel,
   ExecIdModel,
   TestReportModel,
   IntfTestReportModel,
   Rfc2544TestReportModel,
   Rfc2544StatusModel,
   Rfc2544ProfileModel,
   PacketRateAndUnitModel,
   BenchmarkTestModel,
   ThroughputTestTrialModel,
   ThroughputTestResultDetailModel,
   FrameLossRateTestReportModel,
   FrameLossRateTestResultModel,
   FrameLossRateTestTrialModel,
)
from CliToken.Monitor import (
   monitorMatcherForShow,
)
from EoamTypes import (
   FeatureEnabledEnum,
   tacRateUnit,
)
from RFC2544InitiatorCliUtils import (
   Rfc2544BenchmarkTestEnum,
   TestStateEnum,
   TestFailureReasonEnum,
   packetSizeKwStr,
)

__defaultTraceHandle__ = Tracing.Handle( 'RFC2544InitiatorCliShow' )
t0 = Tracing.trace0

# Globals
rfc2544ExecStatusDir = None
featureConfig = None
featureStatus = None
hwFeatureStatus = None
rfc2544ProfileConfigDir = None
testFailedStates = list( TestStateEnum.attributes )
testFailedStates.remove( TestStateEnum.flowStateNone )
testFailedStates.remove( TestStateEnum.flowStateRunning )
testFailedStates.remove( TestStateEnum.flowStateSuccess )

class ShowRfc2544TestReport:
   # Returns the packet rate in the unit in which it is above 1 and below
   # 1000. It preserves, the result to 3 decimal places.
   # Since it is highly likely that the throughput is much lower than the
   # configured rate (the one configured in profileConfig), we can not
   # guarantee that we will be able to display result in the rate unit that
   # was configured. So instead, keeping the result output consistently
   # depend only on the final throughput rate.
   #
   # Example1:
   # Config rate: 2 Gbps
   # Throughput rate: 234 Mbps
   # Result will be shown as 234.0 Mbps
   #
   # Example2:
   # Config rate: 2 Gbps
   # Throughput rate: 1234 Mbps
   # Result will be shown as 1.234 Gbps
   def getPacketRateAndUnit( self, resultRate, resultUnit ):
      if resultUnit == tacRateUnit.rateUnitInvalid:
         return ( resultRate, resultUnit )

      # Ideally, the result should be in bps only since it is supposed to be
      # normalized, but still just being extra cautious and trying to keep
      # CliPlugin code idempotent.
      throughputRateAndUnit = Tac.Value( "FlowGenerator::PacketRateAndUnit",
                                         resultRate, resultUnit )
      rateAndUnitBps = throughputRateAndUnit.convertPacketRateUnit(
                       tacRateUnit.rateUnitbps )
      rate = rateAndUnitBps.packetRateValue
      if rate < 1000:
         return ( rate, rateAndUnitBps.packetRateUnit )

      # If rate in bps is greater than 1000, convert it to a value between
      # 1 and 1000.
      validRateUnits = [ tacRateUnit.rateUnitbps, tacRateUnit.rateUnitKbps,
                         tacRateUnit.rateUnitMbps, tacRateUnit.rateUnitGbps ]
      i = 0
      while ( decimal.Decimal( rate ).quantize( decimal.Decimal( '0.001' ),
              rounding=decimal.ROUND_HALF_DOWN ) >= 1000 and \
            i < len( validRateUnits ) - 1 ):
         rate = rate / 1000.0
         i += 1

      return ( rate, validRateUnits[ i ] )

   def addThroughputTrialResult( self, trialId, trialResult, 
                                 throughputTestResultDetailModel ):
      throughputTestTrialModel = ThroughputTestTrialModel()
      throughputTestResultDetailModel.trials[
         trialId ] = throughputTestTrialModel
      throughputTestTrialModel.state = trialResult.flowState
      if trialResult.flowState != TestStateEnum.flowStateNone:
         throughputTestTrialModel.packetsSent = trialResult.txPktCount
         throughputTestTrialModel.packetsReceived = trialResult.rxPktCount
         rate, unit = self.getPacketRateAndUnit(
            trialResult.packetRate.packetRateValue,
            trialResult.packetRate.packetRateUnit )
         if unit != tacRateUnit.rateUnitInvalid:
            packetRateAndUnitModel = PacketRateAndUnitModel()
            # pylint: disable-next=consider-using-f-string
            packetRateAndUnitModel.rate = float( '%.3f' % rate )
            packetRateAndUnitModel.unit = unit
            throughputTestTrialModel.packetRate = packetRateAndUnitModel
         if trialResult.flowState != TestStateEnum.flowStateRunning:
            throughputTestTrialModel.duration = trialResult.testDurationInSeconds

   def addDetailedThroughputResult( self, throughputResult,
                                    throughputTestResultDetailModel ):
      # When Rfc2544Initiator::ThroughputResult is created in agent code,
      # startTime and endTime is set to the current time at the time of creation.
      # If ThroughputResult.testState is not set, it means the testing is not started
      # yet, in which case we don't want to set start and end time.
      if throughputResult.testState != TestStateEnum.flowStateNone:
         throughputTestResultDetailModel.startTime = float(
            throughputResult.startTime.timestamp )
         # If test is still running for this packet size, means all trials have not
         # finished, so we cannot show endTime
         if throughputResult.testState != TestStateEnum.flowStateRunning:
            throughputTestResultDetailModel.endTime = float(
               throughputResult.endTime.timestamp )

      for trialId, trialResult in throughputResult.throughputTrialResult.items():
         self.addThroughputTrialResult(
            trialId, trialResult, throughputTestResultDetailModel )
      # Eventually, Rfc2544Initiator::ThroughputResult->throughputResult should
      # always be present in
      # Rfc2544Initiator::ThroughputResult->throughputTrialResult.
      # Keeping the CLI code idempotent to avoid any assumptions.
      # So, if the finalThroughputResult corresponds to a valid trial
      # which is not present in the collection of trials, then we
      # display it.
      finalThroughputResult = throughputResult.throughputResult
      if finalThroughputResult and \
         finalThroughputResult.trialId and \
         finalThroughputResult.trialId not in \
         throughputResult.throughputTrialResult:
         self.addThroughputTrialResult(
            finalThroughputResult.trialId,
            finalThroughputResult,
            throughputTestResultDetailModel )

   def addThroughputTestResult( self, testExecStatus, throughputTestReportModel,
                                detailed ):
      profileConfig = testExecStatus.profileConfig
      # Take a union of packet sizes in profileConfig.packetSizeInBytes and
      # status. Ideally, if a packet size exists in testExecStatus.throughputResult,
      # then it should have existed in profileConfig.packetSizeInBytes too.
      # Just being extra cautious here in case of some mismatch between config
      # and status.
      # If the test has not run for some packet size, we will show
      # that packet size too, and the state will be "not started".
      configPacketSizes = set()
      if profileConfig:
         configPacketSizes = set( profileConfig.packetSizeInBytes )
      execStatusPacketSizes = set( testExecStatus.throughputResult )
      throughputTestPacketSizes = list(
         execStatusPacketSizes.union( configPacketSizes ) )
      for packetSize in throughputTestPacketSizes:
         throughputTestResultModel = ThroughputTestResultModel()
         # Set defaults.
         # It means the test has not started yet. If the results are available
         # for some packet size, it will be updated later in the code below.
         throughputTestResultModel.state = TestStateEnum.flowStateNone
         if detailed:
            throughputTestResultDetailModel = ThroughputTestResultDetailModel()
            throughputTestResultModel.detail = throughputTestResultDetailModel
         throughputTestReportModel.results[
            int( packetSizeKwStr[ packetSize ] ) ] = \
            throughputTestResultModel

      # If throughput result exists, use the values from there
      for packetSize, throughputResult in testExecStatus.throughputResult.items():
         throughputTestResultModel = \
            throughputTestReportModel.results[
            int( packetSizeKwStr[ packetSize ] ) ]

         throughputTestResultModel.state = throughputResult.testState
         if throughputResult.testState in testFailedStates and \
            throughputResult.testFailureReason != \
            TestFailureReasonEnum.noReason:
            throughputTestResultModel.stateReason = \
               throughputResult.testFailureReason
         finalThroughputResult = throughputResult.throughputResult
         throughputTestResultModel.packetsSent = finalThroughputResult.txPktCount
         rate, unit = self.getPacketRateAndUnit(
            finalThroughputResult.packetRate.packetRateValue,
            finalThroughputResult.packetRate.packetRateUnit )
         if unit != tacRateUnit.rateUnitInvalid:
            packetRateAndUnitModel = PacketRateAndUnitModel()
            # pylint: disable-next=consider-using-f-string
            packetRateAndUnitModel.rate = float( '%.3f' % rate )
            packetRateAndUnitModel.unit = unit
            throughputTestResultModel.rateAndUnit = packetRateAndUnitModel
         if detailed and throughputResult.throughputTrialResult:
            self.addDetailedThroughputResult(
               throughputResult, throughputTestResultModel.detail )

   def addThroughputTestReport( self, testExecStatus, testReportModel, detailed ):
      throughputTestReportModel = ThroughputTestReportModel()
      # Default state, which will be over-written by actual state if
      # testExecStatus.testStatus exists for throughput test 
      throughputTestReportModel.state = TestStateEnum.flowStateNone

      # If testExecStatus.testStatus exists, use the values from there
      testStatus = testExecStatus.testStatus.get( 
                     Rfc2544BenchmarkTestEnum.benchmarkThroughput )
      if testStatus is not None:
         throughputTestReportModel.state = testStatus.testState
         if testStatus.testState in testFailedStates:
            if testStatus.testFailureReason != TestFailureReasonEnum.noReason:
               # If agent code assigned a reason, use it.
               throughputTestReportModel.stateReason = \
                  testStatus.testFailureReason
            elif testStatus.testState == TestStateEnum.flowStateAborted:
               # In case of TestStateEnum.flowStateAborted which means "stop" exec
               # command was issued, the agent code may not have set the reason.
               # And here we know what the reason needs to be, so we can set
               # it explicitly
               throughputTestReportModel.stateReason = \
                  TestFailureReasonEnum.execRequestStop
      # When the Nominal Rfc2544Initiator::TestStatus is created in agent code,
      # startTime and endTime is set to the current time at the time of creation.
      # If TestStatus.testState is not set, it means the testing is not started
      # yet, in which case we don't want to set start and end time
      if testStatus and testStatus.testState != TestStateEnum.flowStateNone:
         throughputTestReportModel.startTime = \
            float( testStatus.startTime.timestamp )
         # If the test is still running, we can not show duration and
         # end time
         if testStatus.testState != TestStateEnum.flowStateRunning:
            throughputTestReportModel.endTime = float( testStatus.endTime.timestamp )

      self.addThroughputTestResult(
         testExecStatus, throughputTestReportModel, detailed )
      testReportModel.throughputTest = throughputTestReportModel

   def addFrameLossRateTrialResult( self, trialId, trialResult,
                                    frameLossRateTestResultModel ):
      frameLossRateTestTrialModel = FrameLossRateTestTrialModel()
      frameLossRateTestResultModel.trials[
         trialId ] = frameLossRateTestTrialModel
      frameLossRateTestTrialModel.state = trialResult.flowState
      rate, unit = self.getPacketRateAndUnit(
         trialResult.packetRate.packetRateValue,
         trialResult.packetRate.packetRateUnit )
      if unit != tacRateUnit.rateUnitInvalid:
         packetRateAndUnitModel = PacketRateAndUnitModel()
         # pylint: disable-next=consider-using-f-string
         packetRateAndUnitModel.rate = float( '%.3f' % rate )
         packetRateAndUnitModel.unit = unit
         frameLossRateTestTrialModel.packetRate = packetRateAndUnitModel
      if trialResult.flowState != TestStateEnum.flowStateNone:
         frameLossRateTestTrialModel.packetsSent = trialResult.txPktCount
         frameLossRateTestTrialModel.packetsReceived = trialResult.rxPktCount
         if trialResult.testFailureReason != TestFailureReasonEnum.noReason:
            frameLossRateTestTrialModel.stateReason = \
               trialResult.testFailureReason
         if trialResult.flowState != TestStateEnum.flowStateRunning:
            frameLossRateTestTrialModel.duration = trialResult.testDurationInSeconds

   def addFrameLossRateTestResult( self, testExecStatus,
                                   frameLossRateTestReportModel ):
      profileConfig = testExecStatus.profileConfig
      # Take a union of packet sizes in profileConfig.packetSizeInBytes and
      # status. Ideally, if a packet size exists in
      # testExecStatus.frameLossRateResult, then it should have existed
      # in profileConfig.packetSizeInBytes too.
      # Just being extra cautious here in case of some mismatch between config
      # and status.
      # If the test has not run for some packet size, we will show
      # that packet size too, and the state will be "not started".
      configPacketSizes = set()
      if profileConfig:
         configPacketSizes = set( profileConfig.packetSizeInBytes )
      execStatusPacketSizes = set( testExecStatus.frameLossRateResult )
      frameLossRateTestPacketSizes = list(
         execStatusPacketSizes.union( configPacketSizes ) )
      for packetSize in frameLossRateTestPacketSizes:
         frameLossRateTestResultModel = FrameLossRateTestResultModel()
         # Set defaults.
         # It means the test has not started yet. If the results are available
         # for some packet size, it will be updated later in the code below.
         frameLossRateTestResultModel.state = TestStateEnum.flowStateNone
         frameLossRateTestReportModel.results[
            int( packetSizeKwStr[ packetSize ] ) ] = \
            frameLossRateTestResultModel

      # If frameLossRate result exists, use the values from there
      for packetSize, frameLossRateResult in \
            testExecStatus.frameLossRateResult.items():
         frameLossRateTestResultModel = \
            frameLossRateTestReportModel.results[
            int( packetSizeKwStr[ packetSize ] ) ]

         frameLossRateTestResultModel.state = frameLossRateResult.testState
         if frameLossRateResult.testState in testFailedStates and \
            frameLossRateResult.testFailureReason != \
            TestFailureReasonEnum.noReason:
            frameLossRateTestResultModel.stateReason = \
               frameLossRateResult.testFailureReason
         # If FrameLossRateResult.testState is not set, it means the testing is not
         # started yet, in which case we don't want to set start and end time.
         if frameLossRateResult.testState != TestStateEnum.flowStateNone:
            frameLossRateTestResultModel.startTime = float(
               frameLossRateResult.startTime.timestamp )
            # If test is still running for this packet size, means all trials have
            # not finished, so we cannot show endTime
            if frameLossRateResult.testState != TestStateEnum.flowStateRunning:
               frameLossRateTestResultModel.endTime = float(
                  frameLossRateResult.endTime.timestamp )

         for trialId, trialResult in \
               frameLossRateResult.frameLossRateTrialResult.items():
            self.addFrameLossRateTrialResult(
               trialId, trialResult, frameLossRateTestResultModel )

   def addFrameLossRateTestReport( self, testExecStatus, testReportModel ):
      frameLossRateTestReportModel = FrameLossRateTestReportModel()
      # Default state, which will be over-written by actual state if
      # testExecStatus.testStatus exists for frame loss rate test
      frameLossRateTestReportModel.state = TestStateEnum.flowStateNone

      # If testExecStatus.testStatus exists, use the values from there
      testStatus = testExecStatus.testStatus.get(
                     Rfc2544BenchmarkTestEnum.benchmarkFrameLossRate )
      if testStatus is not None:
         frameLossRateTestReportModel.state = testStatus.testState
         if testStatus.testState in testFailedStates:
            if testStatus.testFailureReason != TestFailureReasonEnum.noReason:
               # If agent code assigned a reason, use it.
               frameLossRateTestReportModel.stateReason = \
                  testStatus.testFailureReason
            elif testStatus.testState == TestStateEnum.flowStateAborted:
               # In case of TestStateEnum.flowStateAborted which means "stop" exec
               # command was issued, the agent code may not have set the reason.
               # And here we know what the reason needs to be, so we can set
               # it explicitly
               frameLossRateTestReportModel.stateReason = \
                  TestFailureReasonEnum.execRequestStop
      # If TestStatus.testState is not set, it means the testing is not started
      # yet, in which case we don't want to set start and end time.
      if testStatus and testStatus.testState != TestStateEnum.flowStateNone:
         frameLossRateTestReportModel.startTime = \
            float( testStatus.startTime.timestamp )
         # If the test is still running, we can not show duration and
         # end time
         if testStatus.testState != TestStateEnum.flowStateRunning:
            frameLossRateTestReportModel.endTime = float(
               testStatus.endTime.timestamp )

      rate, unit = self.getPacketRateAndUnit(
         testExecStatus.profileConfig.packetRateAndUnit.packetRateValue,
         testExecStatus.profileConfig.packetRateAndUnit.packetRateUnit )
      if unit != tacRateUnit.rateUnitInvalid:
         packetRateAndUnitModel = PacketRateAndUnitModel()
         # pylint: disable-next=consider-using-f-string
         packetRateAndUnitModel.rate = float( '%.3f' % rate )
         packetRateAndUnitModel.unit = unit
         frameLossRateTestReportModel.rateAndUnit = packetRateAndUnitModel

      self.addFrameLossRateTestResult(
         testExecStatus, frameLossRateTestReportModel )
      testReportModel.frameLossRateTest = frameLossRateTestReportModel

   def addTestReport( self, execId, testExecStatus, intfTestReportModel, detailed ):
      testReportModel = TestReportModel()
      profileConfig = testExecStatus.profileConfig
      execIdModel = ExecIdModel()
      execIdModel.timestamp = execId.timestamp
      execIdModel.pid = execId.pid
      execIdModel.count = execId.count
      testReportModel.execId = execIdModel
      intfTestReportModel.testReports.append( testReportModel )
      throughputTestConfigured = False
      frameLossRateTestConfigured = False
      if profileConfig is not None:
         testReportModel.profileName = profileConfig.profileName
         testReportModel.direction = profileConfig.direction
         testReportModel.sourceMac = profileConfig.sourceMac
         testReportModel.destinationMac = profileConfig.destinationMac
         throughputTestConfigured = Rfc2544BenchmarkTestEnum.benchmarkThroughput \
                                    in profileConfig.benchmarkTest
         frameLossRateTestConfigured = \
            Rfc2544BenchmarkTestEnum.benchmarkFrameLossRate in \
            profileConfig.benchmarkTest

      # Add throughput test report if it is available.
      # If testing has not started and testExecStatus is not updated with
      # throughput test, addThroughputTestReport() function
      # does not populate the entities in ThroughputTestReportModel.
      if throughputTestConfigured or \
         Rfc2544BenchmarkTestEnum.benchmarkThroughput in \
         testExecStatus.testStatus or \
         testExecStatus.throughputResult:
         self.addThroughputTestReport( testExecStatus, testReportModel, detailed )

      # Add frameLossRate test report if it is available.
      # If testing is not started and results are not available
      # for frameLossRate test, then addFrameLossRateTestReport will
      # take care of it and not populate the optional entities
      # in FrameLossRateTestReportModel.
      if ( frameLossRateTestConfigured or
           Rfc2544BenchmarkTestEnum.benchmarkFrameLossRate in
           testExecStatus.testStatus or
           testExecStatus.frameLossRateResult ):
         self.addFrameLossRateTestReport( testExecStatus, testReportModel )

   def createRfc2544TestReport( self, mode, args, rfc2544TestReportModel ):
      intfFilter = args.get( 'INTFS' )
      detailed = bool( args.get( 'detailed' ) )
      # If user has explicitly given interface filters, then we will
      # create IntfTestReportModel for each of them. If test reports
      # are not available for some of these interfaces, then we will
      # display "No test report available" for those interfaces.
      intfNames = []
      if intfFilter is not None:
         intfNames = [ intf.name for intf in intfFilter ]
         for intfName in intfNames:
            intfTestReportModel = IntfTestReportModel()
            rfc2544TestReportModel.interfaces[ intfName ] = intfTestReportModel

      for intfName, intfExecStatus in rfc2544ExecStatusDir.intfExecStatus.items():
         if intfNames and intfName not in intfNames:
            # This is not one of the filter intfs, so skip
            continue
         if intfName in rfc2544TestReportModel.interfaces:
            intfTestReportModel = rfc2544TestReportModel.interfaces[ intfName ]
         else:
            # If there was no intf filter, we need to create IntfTestReportModel
            intfTestReportModel = IntfTestReportModel()
            rfc2544TestReportModel.interfaces[ intfName ] = intfTestReportModel
         for execId, testExecStatus in intfExecStatus.testExecStatus.items():
            self.addTestReport(
               execId, testExecStatus, intfTestReportModel, detailed )

def showRfc2544TestReport( mode, args ):
   s = ShowRfc2544TestReport()
   rfc2544TestReportModel = Rfc2544TestReportModel()
   # We only check on featureStatus to indicate the enabled state in show
   # command. It is helpful if the show command displays the test reports
   # in a corner case where config is disabled but status indicates it as enabled,
   # then we can easily catch it using these commands.
   if featureStatus.featureEnabled == FeatureEnabledEnum.rfc2544:
      rfc2544TestReportModel.enabled = True
   else:
      rfc2544TestReportModel.enabled = False
      t0( "RFC2544 is not enabled" )
      return rfc2544TestReportModel
   s.createRfc2544TestReport( mode, args, rfc2544TestReportModel )
   return rfc2544TestReportModel

class ShowRfc2544TestReportCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor rfc2544 generators test-report [ interface { INTFS } ]' \
            '[ detailed ]'
   data = { 'monitor' : monitorMatcherForShow,
            'rfc2544' : rfc2544ConfigKwMatcher,
            'generators' : generatorsNode,
            'test-report' : 'RFC2544 test reports',
            'interface' : interfaceNode,
            'INTFS' : CliCommand.Node( intfMatcher ),
            'detailed' : 'Display detailed test results'
          }

   cliModel = Rfc2544TestReportModel
   handler = showRfc2544TestReport

class ShowRfc2544Status:
   def createRfc2544Status( self, mode, args, rfc2544StatusModel ):
      # pylint: disable=simplifiable-if-statement
      if featureConfig.featureEnabled.get( FeatureEnabledEnum.rfc2544 ):
         rfc2544StatusModel.enabled = True
      else:
         rfc2544StatusModel.enabled = False
      if featureStatus.featureEnabled == FeatureEnabledEnum.rfc2544 and \
         hwFeatureStatus.featureEnabled == FeatureEnabledEnum.rfc2544:
         rfc2544StatusModel.running = True
      else:
         rfc2544StatusModel.running = False

      profileFilter = args.get( 'PROFILE' )
      for profileName, profileConfig in \
            rfc2544ProfileConfigDir.profileConfig.items():
         if profileFilter and profileName not in profileFilter:
            continue

         rfc2544ProfileModel = Rfc2544ProfileModel()
         rfc2544StatusModel.profileNames[ profileName ] = rfc2544ProfileModel
         # Benchmark tests
         # profileConfig.benchmarkTest is an ordered collection, indexed on
         # the enum Rfc2544Initiator::BenchmarkTest.
         # Since rfc2544ProfileModel.benchmarkTests is a list, the order will be
         # preserved and we should not sort it here again.
         for benchmarkTest in profileConfig.benchmarkTest:
            benchmarkTestModel = BenchmarkTestModel()
            benchmarkTestModel.test = benchmarkTest
            rfc2544ProfileModel.benchmarkTests.append( benchmarkTestModel )

         # Direction
         rfc2544ProfileModel.direction = profileConfig.direction

         # Packet sizes
         for packetSize in profileConfig.packetSizeInBytes:
            rfc2544ProfileModel.packetSizes.append(
               int( packetSizeKwStr[ packetSize ] ) )

         # Duration
         rfc2544ProfileModel.duration = profileConfig.testDurationInSeconds
         
         # rate and unit
         if profileConfig.packetRateAndUnit.packetRateUnit != \
            tacRateUnit.rateUnitInvalid:
            packetRateAndUnitModel = PacketRateAndUnitModel()
            packetRateAndUnitModel.rate = float(
               profileConfig.packetRateAndUnit.packetRateValue )
            packetRateAndUnitModel.unit = \
               profileConfig.packetRateAndUnit.packetRateUnit
            rfc2544ProfileModel.packetRate = packetRateAndUnitModel

         # Source and desitination MAC
         rfc2544ProfileModel.sourceMac = profileConfig.sourceMac
         rfc2544ProfileModel.destinationMac = profileConfig.destinationMac

def showRfc2544Status( mode, args ):
   s = ShowRfc2544Status()
   rfc2544StatusModel = Rfc2544StatusModel()
   s.createRfc2544Status( mode, args, rfc2544StatusModel )
   return rfc2544StatusModel

class ShowRfc2544StatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor rfc2544 generators profiles [ { PROFILE } ]'
   data = { 'monitor' : monitorMatcherForShow,
            'rfc2544' : rfc2544ConfigKwMatcher,
            'generators' : generatorsNode,
            'profiles' : 'RFC2544 profiles',
            'PROFILE' : rfc2544ProfileMatcher,
          }

   cliModel = Rfc2544StatusModel
   handler = showRfc2544Status

BasicCli.addShowCommandClass( ShowRfc2544TestReportCmd )
BasicCli.addShowCommandClass( ShowRfc2544StatusCmd )
#-----------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-----------------------------------------------------------
def Plugin( entityManager ):
   global rfc2544ExecStatusDir
   global featureConfig
   global featureStatus
   global hwFeatureStatus
   global rfc2544ProfileConfigDir

   rfc2544ExecStatusDir = LazyMount.mount(
         entityManager, 'rfc2544Initiator/execStatus',
         'Rfc2544Initiator::ExecStatusDir', 'r' )
   featureStatus = LazyMount.mount(
         entityManager, 'generator/featureStatus',
         'FlowGenerator::FeatureStatus', 'r' )
   featureConfig = LazyMount.mount(
         entityManager, 'generator/featureConfig',
         'FlowGenerator::FeatureConfig', 'r' )
   hwFeatureStatus = LazyMount.mount(
      entityManager, 'generator/hwFeatureStatus',
      'FlowGenerator::FeatureStatus', 'r' )
   rfc2544ProfileConfigDir = LazyMount.mount(
         entityManager, 'rfc2544Initiator/profileConfigDir',
         'Rfc2544Initiator::ProfileConfigDir', 'r' )
