# Copyright (c) 2008-2011, 2013-2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements RADIUS configuration.
#
# In enable mode:
#
#     show radius
#     clear aaa counters radius
#
# In config mode:
#
#     [no] radius-server key <key-text>
#     [no] radius-server timeout <1-1000>
#     [no] radius-server deadtime <1-1000>
#     [no] radius-server retransmit <0-100>
#     [no] radius-server host <ip-addr-or-hostname> [vrf <vrf-name>]
#           [auth-port <0-65535>] [acct-port <0-65535>] [timeout <1-1000>]
#           [retransmit <0-100>] [key <key-text>]
#     [no] radius-server attribute <32> include-in-access-req format <nas-id-string>
#     [no] radius-server qos dscp <0-63>
#     [no] ip radius source-interace <interface-name>
#     [no] radius-server tls ssl-profile <profileName>
#
# Child mode of config mode:
#
#     aaa group server radius <server-group-name>
#        [no] server <ip-addr-or-hostname> [auth-port <0-65535>] \
#            [acct-port <0-65535>]
#
#-------------------------------------------------------------------------------
import CliPlugin.TechSupportCli
from CliPlugin import AaaCli
from CliPlugin import IntfCli
from CliPlugin.VrfCli import VrfExprFactory
import AaaCliLib
from AaaPluginLib import hostProtocol
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliParser
import CliToken.Clear
import CliToken.Ip
import DscpCliLib
import HostnameCli
import LazyMount
import RadiusGroup
import ReversibleSecretCli
import ShowCommand
import Tac
import Tracing
from Toggles import RadiusToggleLib

t0 = Tracing.trace0

# Need to define this here because I can't access definitions in .tac files
# from Python due to missing support for constAttr.
_timeoutMin = 1
_timeoutMax = 1000

_defaultRetries = Tac.Value( "Radius::Retries" ).value
_retriesMin = 0
_retriesMax = 100

_deadtimeMin = 1
_deadtimeMax = 1000

radiusConfig = None
sslConfig = None
sharedSecretConfig = None

def radiusHost( mode, hostname, vrf, port, acctPort, tlsEnabled, create=False ):
   hosts = radiusConfig.host
   assert vrf != ""

   if tlsEnabled:
      proto = hostProtocol.protoRadsec
   else:
      proto = hostProtocol.protoRadius

   spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                     acctPort=acctPort, vrf=vrf, protocol=proto )

   if spec in hosts:
      host = hosts[ spec ]
   elif create:
      if tlsEnabled:
         t0( "Creating tls host:", hostname, ":", vrf, ":", port )
      else:
         t0( "Creating host:", hostname, ":", vrf, ":", port, ":", acctPort )
      host = hosts.newMember( spec )
      if host is None:
         t0( "Unable to create Host:", hostname, ":", vrf, ":", port, ":",
             acctPort, " with tls flag:", tlsEnabled )
      else:
         host.index = AaaCliLib.getHostIndex( hosts )
   else:
      host = None
   if host is not None:
      assert host.hostname == hostname
      assert host.vrf == vrf
      assert host.acctPort == acctPort
      assert host.protocol == proto
      assert host.port == port
   return host

#-------------------------------------------------------------------------------
# "show radius" in enable mode
#-------------------------------------------------------------------------------

class ShowRadiusCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show radius"
   data = { "radius" : 'Details of RADIUS operation' }
   cliModel = "RadiusModel.ShowRadius"
   handler = "RadiusHandler.showRadius"

BasicCli.addShowCommandClass( ShowRadiusCommand )

#-------------------------------------------------------------------------------
# "clear aaa counters radius" in enable mode
#-------------------------------------------------------------------------------
class ClearRadiusCounterCommand( CliCommand.CliCommandClass ):
   syntax = "clear aaa counters radius"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'aaa' : AaaCli.aaaAfterClearMatcher,
      'counters' : AaaCli.aaaCounterMatcher,
      'radius' : "Clear RADIUS counters"
      }
   handler = "RadiusHandler.clearCounters"

BasicCli.EnableMode.addCommandClass( ClearRadiusCounterCommand )

#-------------------------------------------------------------------------------
# config mode commands
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

radiusServerKwMatcher = CliMatcher.KeywordMatcher(
   'radius-server',
   helpdesc='Modify RADIUS parameters' )

keyExpression = ReversibleSecretCli.defaultReversiblePwdCliExpr

#-------------------------------------------------------------------------------
# "[no] radius-server key <KEY>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerKeyCommand( CliCommand.CliCommandClass ):
   syntax = '''radius-server ( ( key <KEY> ) |
               ( shared-secret profile <SECRET_PROFILE> ) )'''
   noOrDefaultSyntax = 'radius-server ( key | shared-secret ) ...'
   data = {
      'radius-server' : radiusServerKwMatcher,
      'key' : 'Set RADIUS secret key',
      '<KEY>' : keyExpression,
      'shared-secret': 'Use a shared-secret profile for key',
      'profile': 'Use a shared-secret profile for key',
      '<SECRET_PROFILE>': CliMatcher.DynamicNameMatcher(
         lambda mode: sharedSecretConfig.profile,
         'Shared-secret profile name' ),
   }
   handler = "RadiusHandler.setRadiusServerKey"
   noOrDefaultHandler = "RadiusHandler.noRadiusServerKey"

configMode.addCommandClass( RadiusServerKeyCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server timeout <1-1000>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerTimeoutCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server timeout <TIMEOUT>"
   noOrDefaultSyntax = "radius-server timeout ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'timeout' : 'Time to wait for a RADIUS server to respond',
      '<TIMEOUT>' : CliMatcher.IntegerMatcher(
         _timeoutMin, _timeoutMax,
         helpdesc='Number of seconds' )
   }
   @staticmethod
   def handler( mode, args ):
      radiusConfig.timeout = args[ '<TIMEOUT>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      radiusConfig.timeout = radiusConfig.defaultTimeout

configMode.addCommandClass( RadiusServerTimeoutCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server deadtime <1-1000>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerDeadtimeCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server deadtime <DEADTIME>"
   noOrDefaultSyntax = "radius-server deadtime ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'deadtime' : 'Time to skip a nonresponsive server',
      '<DEADTIME>' : CliMatcher.IntegerMatcher( _deadtimeMin, _deadtimeMax,
                                                helpdesc='Number of minutes' )
   }
   @staticmethod
   def handler( mode, args ):
      deadtime = args[ '<DEADTIME>' ]
      radiusConfig.deadtime = deadtime * 60

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      radiusConfig.deadtime = 0

configMode.addCommandClass( RadiusServerDeadtimeCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server retransmit <0-100>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerRetransmitCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server retransmit <RETRIES>"
   noOrDefaultSyntax = "radius-server retransmit ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'retransmit' : 'Specify the number of retries for the active server',
      '<RETRIES>' : CliMatcher.IntegerMatcher(
         _retriesMin, _retriesMax,
         helpdesc=( f'Number of retries (default {_defaultRetries})' ) ),
   }
   @staticmethod
   def handler( mode, args ):
      radiusConfig.retries = args[ '<RETRIES>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      radiusConfig.retries = radiusConfig.defaultRetries

configMode.addCommandClass( RadiusServerRetransmitCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server attribute 32 include-in-access-req format ( ( format NAS )
# | fqdn | hostname | disabled )
# in config mode
#-------------------------------------------------------------------------------
attrKwMatcher = CliMatcher.KeywordMatcher(
   'attribute', helpdesc='Specify the attribute for the active server' )

class RadiusServerNasIdCommand( CliCommand.CliCommandClass ):
   syntax = '''radius-server attribute 32 include-in-access-req
    ( ( format NAS ) | hostname | fqdn )'''
   noOrDefaultSyntax = "radius-server attribute 32 include-in-access-req ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'attribute' : attrKwMatcher,
      '32' : 'NAS-Identifier',
      'include-in-access-req' :
      'Send NAS-Identifier attribute in Access-Request packet',
      'format' : 'Specify the format of the NAS-Idendifier',
      'NAS' : CliMatcher.StringMatcher( helpname='LINE',
         helpdesc='Simple string' ),
      'hostname' : 'Hostname configured for the switch',
      'fqdn' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'fqdn',
         helpdesc='Fully qualified domain name of the switch'), hidden=True ),
   }
   handler = "RadiusHandler.setRadiusServerAttribute"
   noOrDefaultHandler = "RadiusHandler.noRadiusServerAttribute"

configMode.addCommandClass( RadiusServerNasIdCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server qos dscp <0-63>" in config mode
#-------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( configMode,
                                   "RadiusHandler.setDscp",
                                   "RadiusHandler.noDscp",
                                   tokenProto=radiusServerKwMatcher )

#-------------------------------------------------------------------------------
# "[no] radius-server host <ip-addr-or-hostname> [ VRF ]
#  ( ( [auth-port <0-65535>] [acct-port <0-65535>]
#  [timeout <1-1000>] [retransmit <1-100>] 
#  [ key <key-text> | shared-secret profile <profile> ] ) |
#  ( tls [ssl-profile <profile-name>]
#  [port <0-65535>] [timeout <1-1000>] [retransmit <1-100>] ) )"
# in config mode
#-------------------------------------------------------------------------------

sslProfileNameDynMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: ( name for name in sslConfig.profileConfig ),
   helpname="WORD",
   helpdesc="Profile name",
   priority=CliParser.PRIO_LOW )

# timeout and retransmit attributes are defined again at the end of TLS cli
# instead of optimising them to preserve the order of old CLI.
class RadiusTlsServerHostCommand( CliCommand.CliCommandClass ):
   syntax = '''radius-server host <HOSTNAME> [ VRF ]
               [ ( ( [ auth-port <AUTHPORT> ] [ acct-port <ACCTPORT> ]
               [ timeout <TIMEOUT> ] [ retransmit <RETRIES> ]
               [ ( ( key <KEY> ) |
                     ( shared-secret profile <SECRET_PROFILE> ) ) ] ) |
               ( tls [ ssl-profile <PROFILENAME> ]
               [ port <TLSPORT> ] [ timeout <TIMEOUT> ]
               [ retransmit <RETRIES> ] ) ) ]
               '''
   # Same as above, but everything after 'host' becomes optional.
   noOrDefaultSyntax = '''radius-server host [ <HOSTNAME> [ VRF ]
               [ ( ( [ auth-port <AUTHPORT> ] [ acct-port <ACCTPORT> ]
               [ timeout <TIMEOUT> ] [ retransmit <RETRIES> ]
               [ ( ( key <KEY> ) |
                     ( shared-secret profile <SECRET_PROFILE> ) ) ] ) |
               ( tls [ ssl-profile <PROFILENAME> ]
               [ port <TLSPORT> ] [ timeout <TIMEOUT> ]
               [ retransmit <RETRIES> ] ) ) ] ]
               '''
   data = { 'radius-server' : radiusServerKwMatcher,
            'host' : 'RADIUS server configuration',
            '<HOSTNAME>' : HostnameCli.IpAddrOrHostnameMatcher(
               helpname='WORD',
               helpdesc='Hostname or IP address of RADIUS server',
               ipv6=True ),
            'VRF' : VrfExprFactory( helpdesc='VRF for this Radius server' ),
            'auth-port' : ( f'RADIUS server authentication port (default is '
                            f'{RadiusGroup.defaultPort})' ),
            '<AUTHPORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                                             helpdesc="Number of the port to use" ),
            'acct-port' : ( f'RADIUS server accounting port (default is '
                            f'{RadiusGroup.defaultAcctPort})' ),
            '<ACCTPORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                                                      helpdesc="Acct port number" ),
            'key' : 'Encryption key for this RADIUS server (overrides default)',
            '<KEY>' : keyExpression,
            'shared-secret': 'Use a shared-secret profile for key',
            'profile': 'Use a shared-secret profile for key',
            '<SECRET_PROFILE>': CliMatcher.DynamicNameMatcher(
               lambda mode: sharedSecretConfig.profile,
               'Shared-secret profile name' ),
            'tls' : 'Enable TLS for radius server',
            'ssl-profile' : 'Configure SSL profile to use (overrides global)',
            "<PROFILENAME>" : sslProfileNameDynMatcher,
            'port' : ( f'RADIUS server TLS port (default is '
                       f'{RadiusGroup.defaultTlsPort})' ),
            '<TLSPORT>' : CliMatcher.IntegerMatcher(
               0, 65535, helpdesc="Number of the port to use" ),
            'timeout' : 'Time to wait for this RADIUS server to respond ' \
            '(overrides default)',
            '<TIMEOUT>' : CliMatcher.IntegerMatcher(
               _timeoutMin, _timeoutMax,
               helpdesc='Timeout value in seconds to ' \
               'wait for the server\'s response' ),
            'retransmit' :
            'Specify the number of retries for the server (overrides default)',
            '<RETRIES>' : CliMatcher.IntegerMatcher(
               _retriesMin, _retriesMax,
               helpdesc='Retry limit' ),
            }
   handler = "RadiusHandler.setRadiusServerTls"
   noOrDefaultHandler = "RadiusHandler.noRadiusServerTls"

configMode.addCommandClass( RadiusTlsServerHostCommand )


#-------------------------------------------------------------------------------
# "[no] ip radius [ VRF ] source-interface <interface-name>"
# in config mode
#-------------------------------------------------------------------------------
class RadiusSourceIntfCommand( CliCommand.CliCommandClass ):
   syntax = "ip radius [ VRF ] source-interface INTF"
   noOrDefaultSyntax = "ip radius [ VRF ] source-interface ..."
   data = {
      "ip" : CliToken.Ip.ipMatcherForConfig,
      "radius" : 'Commands for RADIUS',
      "VRF" : VrfExprFactory( helpdesc='Specify VRF' ),
      "source-interface" : 'Interface providing the IP source '
                           'address of RADIUS packets',
      "INTF" : IntfCli.Intf.matcherWithIpSupport
      }

   handler = "RadiusHandler.setRadiusSrcIntf"
   noOrDefaultHandler = "RadiusHandler.noRadiusSrcIntf"

configMode.addCommandClass( RadiusSourceIntfCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server dynamic-authorization port  <0-65535>" in config mode
#-------------------------------------------------------------------------------
dynamicAuthorizationKwMatcher = CliMatcher.KeywordMatcher(
   'dynamic-authorization',
   helpdesc='Modify dynamic authorization parameters' )

class RadiusServerDynamicAuthorizationCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server dynamic-authorization port <DYNAUTHPORT>"
   noOrDefaultSyntax = "radius-server dynamic-authorization port ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'dynamic-authorization' : dynamicAuthorizationKwMatcher,
      'port' : f'Listen port for UDP (default is {RadiusGroup.defaultDynAuthPort})',
      '<DYNAUTHPORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                           helpdesc="Number of the port to use" ),
   }

   @staticmethod
   def handler( mode, args ):
      radiusConfig.dynAuthPort = args[ '<DYNAUTHPORT>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      radiusConfig.dynAuthPort = RadiusGroup.defaultDynAuthPort

configMode.addCommandClass( RadiusServerDynamicAuthorizationCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server tls ssl-profile <profileName>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerSslProfileCommand( CliCommand.CliCommandClass ):
   syntax = 'radius-server tls ssl-profile <PROFILENAME>'
   noOrDefaultSyntax = 'radius-server tls ssl-profile ...'
   data = {
      'radius-server' : radiusServerKwMatcher,
      'tls' : 'Enable TLS ssl-profile for radius server',
      'ssl-profile' : 'Configure global SSL profile to use',
      '<PROFILENAME>' : sslProfileNameDynMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      radiusConfig.sslProfile = args[ '<PROFILENAME>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      radiusConfig.sslProfile = ''

configMode.addCommandClass( RadiusServerSslProfileCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server dynamic-authorization tls ssl-profile <profileName>"
# in config mode
#-------------------------------------------------------------------------------
class RadiusDynamicAuthorizationSslProfileCommand( CliCommand.CliCommandClass ):
   syntax = 'radius-server dynamic-authorization tls ssl-profile <PROFILENAME>'
   noOrDefaultSyntax = 'radius-server dynamic-authorization tls ssl-profile ...'
   data = {
      'radius-server' : radiusServerKwMatcher,
      'dynamic-authorization' : dynamicAuthorizationKwMatcher,
      'tls' : 'Enable TLS and listen on TCP port 2083 for incoming connections',
      'ssl-profile' : 'Configure SSL profile to use',
      '<PROFILENAME>' : sslProfileNameDynMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      radiusConfig.dynAuthSslProfile = args[ '<PROFILENAME>' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      radiusConfig.dynAuthSslProfile = ''

if RadiusToggleLib.toggleRadsecCoaToggleEnabled():
   configMode.addCommandClass( RadiusDynamicAuthorizationSslProfileCommand )

#-------------------------------------------------------------------------------
# Register "show tech-support" commands
#-------------------------------------------------------------------------------
def showRadiusGuard():
   # skip if no tacacs-servers are configured
   return radiusConfig.host

CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2010-08-27 04:35:31',
   cmds=[ 'show radius' ],
   cmdsGuard=showRadiusGuard,
   summaryCmds=[ 'show radius' ],
   summaryCmdsGuard=showRadiusGuard )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global radiusConfig, sslConfig, sharedSecretConfig
   radiusConfig = ConfigMount.mount( entityManager, "security/aaa/radius/config",
                                     "Radius::Config", "w" )
   sslConfig = LazyMount.mount( entityManager,
                                'mgmt/security/ssl/config',
                                'Mgmt::Security::Ssl::Config', 'r' )
   sharedSecretConfig = LazyMount.mount(
      entityManager, 'mgmt/security/sh-sec-prof/config',
      'Mgmt::Security::SharedSecretProfile::Config', 'r' )
