#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliToken.Clear
import ConfigMount
import HostnameCli
import ReversibleSecretCli
import ShowCommand
from AaaCliLib import radiusServerGroupNameMatcher
from CliPlugin import IpAddrMatcher, Ip6AddrMatcher, TechSupportCli
from CliPlugin.RadiusProxyMode import RadiusProxyClientGroupConfigMode
from CliPlugin.RadiusProxyMode import RadiusProxyConfigMode
from CliPlugin.VrfCli import VrfExprFactory
from Radius import MAX_KEY_SIZE
from Toggles import RadiusProxyToggleLib
from TypeFuture import TacLazyType

radiusProxyConfig = None
TimeoutConstants = TacLazyType( 'RadiusProxy::TimeoutConstants' )

radiusKwMatcher = CliMatcher.KeywordMatcher(
   'radius',
   helpdesc='Modify radius proxy' )

proxyKwMatcher = CliMatcher.KeywordMatcher(
   'proxy',
   helpdesc='Configure radius proxy mode parameters' )

# --------------------------------------------------------------------------------
# radius proxy config mode
# (config)# radius proxy
# (config-radius-proxy)#
# --------------------------------------------------------------------------------
class RadiusProxyCmd( CliCommand.CliCommandClass ):
   syntax = 'radius proxy'
   noOrDefaultSyntax = syntax

   data = {
      'radius': radiusKwMatcher,
      'proxy': proxyKwMatcher,
   }

   handler = 'RadiusProxyCliHandler.radiusProxyCmdHandler'
   noOrDefaultHandler = 'RadiusProxyCliHandler.noRadiusProxyCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( RadiusProxyCmd )

clientKwMatcher = CliMatcher.KeywordMatcher(
   'client', helpdesc='Modify client parameters' )
keyExpression = ReversibleSecretCli.defaultReversiblePwdCliExpr

# --------------------------------------------------------------------------------
# (config-radius-proxy)# client key
# 0     Indicates that the key string is not encrypted
# 7     Specifies that a HIDDEN key will follow
# 8a    Specifies that a AES-256-GCM encrypted key will follow
# LINE  Unobfuscated key string
# --------------------------------------------------------------------------------
class RadiusProxyClientKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'client key KEY'
   noOrDefaultSyntax = 'client key ...'

   data = {
      'client': clientKwMatcher,
      'key': f'Set client secret key, allowed max size is { MAX_KEY_SIZE }',
      'KEY': keyExpression,
   }

   handler = 'RadiusProxyCliHandler.radiusProxyClientKeyCmdHandler'
   noOrDefaultHandler = 'RadiusProxyCliHandler.noRadiusProxyClientKeyCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   RadiusProxyConfigMode.addCommandClass( RadiusProxyClientKeyCmd )

clientGroupNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: radiusProxyConfig.clientGroup,
   helpdesc='Name of client group',
   helpname='WORD' )

# --------------------------------------------------------------------------------
# (config-radius-proxy)# client group foo
# (config-rp-cg-foo)#
# --------------------------------------------------------------------------------
class RadiusProxyClientGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'client group NAME'
   noOrDefaultSyntax = syntax
   data = {
      'client': clientKwMatcher,
      'group': 'Client group definitions',
      'NAME': clientGroupNameMatcher,
      }

   handler = 'RadiusProxyCliHandler.clientGroupCmdHandler'
   noOrDefaultHandler = 'RadiusProxyCliHandler.noClientGroupCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   RadiusProxyConfigMode.addCommandClass( RadiusProxyClientGroupCmd )

# --------------------------------------------------------------------------------
# (config-rp-cg-foo)# client <ip-addr-or-hostname> vrf <vrfname> [ key <> ]
# --------------------------------------------------------------------------------
prefixMatcher = IpAddrMatcher.IpPrefixMatcher( helpdesc='IPv4 prefix' )
v6PrefixMatcher = Ip6AddrMatcher.Ip6PrefixMatcher( helpdesc='IPv6 prefix' )
ipv4AddrMatcher = IpAddrMatcher.IpAddrMatcher( helpdesc='IPv4 address' )
ipv6AddrMatcher = Ip6AddrMatcher.Ip6AddrMatcher( helpdesc='IPv6 address' )

class ClientConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''client ( ipv4 V4ADDR | PREFIX )
               | ( ipv6 V6ADDR | PREFIXV6 )
               | ( host HOSTNAME )
               VRF [ key KEY ]'''
   noOrDefaultSyntax = '''client ( ipv4 V4ADDR | PREFIX )
                          | ( ipv6 V6ADDR | PREFIXV6 )
                          | ( host HOSTNAME )
                          VRF ...'''
   data = {
      'client': 'Add client to the group',
      'ipv4': 'IPv4 address of the client',
      'V4ADDR': ipv4AddrMatcher,
      'PREFIX': prefixMatcher,
      'ipv6': 'IPv6 address of the client',
      'V6ADDR': ipv6AddrMatcher,
      'PREFIXV6': v6PrefixMatcher,
      'host': 'Client hostname configuration',
      'HOSTNAME': HostnameCli.HostnameMatcher(
         helpname='WORD',
         helpdesc='Hostname of the client' ),
      'VRF': VrfExprFactory( helpdesc='VRF for this client' ),
      'key': 'Key for this client (overrides global)',
      'KEY': keyExpression,
   }

   handler = 'RadiusProxyCliHandler.clientConfigCmdHandler'
   noOrDefaultHandler = 'RadiusProxyCliHandler.noClientConfigCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   RadiusProxyClientGroupConfigMode.addCommandClass( ClientConfigCmd )

groupKwNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
                               'group', helpdesc='Group name' ),
                               noResult=True )

# --------------------------------------------------------------------------------
# (config-rp-cg-foo)# server group <G1> <G2>
# --------------------------------------------------------------------------------
class ClientToServerGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'server group { GROUPS }'
   noOrDefaultSyntax = 'server group ...'

   data = {
      'server': 'Radius server',
      'group': groupKwNode,
      'GROUPS': radiusServerGroupNameMatcher,
   }

   handler = 'RadiusProxyCliHandler.clientToServerGroupCmdHandler'
   noOrDefaultHandler = 'RadiusProxyCliHandler.noClientToServerGroupCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   RadiusProxyClientGroupConfigMode.addCommandClass( ClientToServerGroupCmd )

# --------------------------------------------------------------------------------
# (config-radius-proxy)# dynamic-authorization
# --------------------------------------------------------------------------------
class DynAuthCmd( CliCommand.CliCommandClass ):
   syntax = 'dynamic-authorization'
   noOrDefaultSyntax = 'dynamic-authorization'

   data = {
      'dynamic-authorization': 'Enable dynamic authorization',
   }

   handler = 'RadiusProxyCliHandler.dynAuthCmdHandler'
   noOrDefaultHandler = 'RadiusProxyCliHandler.noDynAuthCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   RadiusProxyConfigMode.addCommandClass( DynAuthCmd )

# --------------------------------------------------------------------------------
# (config-radius-proxy)# client session idle-timeout <1-86400> seconds
# --------------------------------------------------------------------------------
class clientMappingTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'client session idle-timeout TIMEOUT seconds'
   noOrDefaultSyntax = 'client session idle-timeout ...'

   data = {
      'client': clientKwMatcher,
      'session': 'Radius proxy client sessions',
      'idle-timeout': 'Minimum time to wait before clearing the client session',
      'TIMEOUT': CliMatcher.IntegerMatcher( TimeoutConstants.minTimeout,
                                            TimeoutConstants.maxTimeout,
                                            helpdesc='Number of seconds' ),
      'seconds': 'Idle timeout to be specified as seconds'
   }

   handler = 'RadiusProxyCliHandler.clientMappingTimeoutCmdHandler'
   noOrDefaultHandler = 'RadiusProxyCliHandler.noClientMappingTimeoutCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   RadiusProxyConfigMode.addCommandClass( clientMappingTimeoutCmd )

# --------------------------------------------------------------------------------
# switch# show radius proxy client group [ GROUP ]
# --------------------------------------------------------------------------------
class ShowClientGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show radius proxy client group [ GROUP ]'
   data = {
      'radius': 'Details of RADIUS operation',
      'proxy': 'Details of radius proxy',
      'client': 'Details of client side of radius proxy',
      'group': 'Group by group details of clients',
      'GROUP': clientGroupNameMatcher,
   }

   handler = 'RadiusProxyCliHandler.showClientGroupCmdHandler'
   cliModel = 'RadiusProxyModel.showClientGroupCmd'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   BasicCli.addShowCommandClass( ShowClientGroupCmd )

# --------------------------------------------------------------------------------
# switch# show radius proxy server group [ GROUP ]
# --------------------------------------------------------------------------------
class ShowServerGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show radius proxy server group [ GROUP ]'
   data = {
      'radius': 'Details of RADIUS operation',
      'proxy': 'Details of radius proxy',
      'server': 'Details of servers',
      'group': 'Group by group details of servers',
      'GROUP': radiusServerGroupNameMatcher,
   }

   handler = 'RadiusProxyCliHandler.showServerGroupCmdHandler'
   cliModel = 'RadiusProxyModel.showServerGroupCmd'

if RadiusProxyToggleLib.toggleRadiusProxyDeadTimerEnabled():
   BasicCli.addShowCommandClass( ShowServerGroupCmd )

# --------------------------------------------------------------------------------
# switch# clear radius proxy counters client group
# --------------------------------------------------------------------------------
class ClearClientGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'clear radius proxy counters client group [ GROUP ]'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'radius': 'Clear radius proxy details',
      'proxy': 'Clear radius proxy details',
      'counters': 'Clear radius proxy counters',
      'client': 'Clear client side counters',
      'group': 'Clear group by group counters of clients',
      'GROUP': clientGroupNameMatcher,
   }
   handler = 'RadiusProxyCliHandler.clearClientGroupCmdHandler'

if RadiusProxyToggleLib.toggleRadiusProxyForAPEnabled():
   BasicCli.EnableMode.addCommandClass( ClearClientGroupCmd )

# -------------------------------------------------------------------------------
# Register commands to show tech-support
# -------------------------------------------------------------------------------

TechSupportCli.registerShowTechSupportCmd(
   '2023-11-21 21:21:21',
   cmds=[ 'show radius proxy client group' ],
   cmdsGuard=lambda: radiusProxyConfig and radiusProxyConfig.radiusProxyEnabled
)

def Plugin( entityManager ):
   global radiusProxyConfig
   radiusProxyConfig = ConfigMount.mount( entityManager, 'radiusproxy/config',
                                          'RadiusProxy::Config', 'w' )
