#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import CliMatcher
import LazyMount

rcfStatus = None

rcfFunctionStatusHelp = '''Routing control function state.
  active - Routing control function is valid
  missingReference - Routing control function is missing an external reference
  missingSymbol - Routing control function symbol is missing'''

# Used to map CLI render of RCF function status to JSON camelcase form
# Keep this in sync with Rcf::RcfFunctionStatusEnum in RcfClientLib.tac
RCF_FUNCTION_STATUS_DICT = { 'active': 'active',
                             'inactive': 'missingReference',
                             'missing reference': 'missingReference',
                             'prefix_list_v4 not found': 'missingReference',
                             'prefix_list_v6 not found': 'missingReference',
                             'as_path_list not found': 'missingReference',
                             'community_list not found': 'missingReference',
                             'ext_community_list not found': 'missingReference',
                             'large_community_list not found': 'missingReference',
                             'roa_table not found': 'missingReference',
                             'rib_tunnel not found': 'missingReference',
                             'missing symbol': 'missingSymbol' }
RCF_FUNCTION_UPDATE_REASON_DICT = {
   'function definition change': 'functionDefinitionChange',
   'external reference change': 'externalReferenceChange'
}

#----------------------------------------------------------------------------------
#                                K E Y W O R D S
#----------------------------------------------------------------------------------

rcfKw = CliMatcher.KeywordMatcher( 'rcf', helpdesc="Routing control function" )

#----------------------------------------------------------------------------------
#                                M A T C H E R S
#----------------------------------------------------------------------------------

def getExistingFunctionNames( mode ):
   return [ ( func[ 0 ] + "()" ) for func in rcfStatus.functionNames.items() ]

def getExistingFunctionNamesNoParens( mode ):
   return [ ( func[ 0 ] ) for func in rcfStatus.functionNames.items() ]

def stripParens( funcname ):
   if funcname.endswith( '()' ):
      funcname = funcname.removesuffix( '()' )
   elif funcname.endswith( '(' ):
      funcname = funcname.removesuffix( '(' )
   return funcname

rcfUnscopedFunctionPattern = "([A-Za-z_])([A-Za-z0-9_]*)"
rcfFunctionPattern = r"%s(\.%s)*" % ( rcfUnscopedFunctionPattern,
                                      rcfUnscopedFunctionPattern )
rcfFunctionMatcher = CliMatcher.DynamicNameMatcher(
         getExistingFunctionNames,
         helpdesc="Routing control function name",
         pattern=rcfFunctionPattern + r"\(\)",
         helpname='FUNCNAME()' )

rcfFunctionMatcherParensOptional = CliMatcher.DynamicNameMatcher(
         getExistingFunctionNamesNoParens,
         helpdesc="Routing control function name",
         pattern=rcfFunctionPattern + r"(\(\)|\()?",
         helpname="FUNCNAME" )

#-------------------------------------------------------------------------------
#                                P L U G I N
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global rcfStatus

   rcfStatus = LazyMount.mount( entityManager,
                                Cell.path( 'routing/rcf/status' ),
                                'Rcf::Status', 'r' )
