#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.RcfDebugRenderLib import Annotations
from CliPlugin.RcfDebugModels import FragmentTypes
from Assert import (
   assertEqual,
   assertIsNone,
)

class FragmentHandler:
   """
   This class provides a dispatch for the handling of any fragments in an RCF
   functions evaluation. All methods are static as the class holds no state.
   FragmentHandler.handle() is called with a fragment and the function will determine
   which FragmentHandler method is to be called to handle the given fragmentType.
   That handler will then be invoked and it will write the appropriate RCF code and
   annotations to InvocationPrinter scratchpads. Some fragments may require
   additional handling, such as function calls.

   All handle* functions for this class have the following pattern
      Return: None

      Args:
         fragment (RcfDebugModels.RcfDebugFunctionFragment):
            The fragment that needs to be handled based on it's fragmentType.
            See RcfDebugModels.FragmentTypes for the full list.
         codeManager (CodeManager):
            The object responsible for maintaining where the renderer is in the
            RCF functions code. Required to retrieve the code corresponding to a
            given fragment.
         printer (InvocationPrinter):
            The object where the fragments representation is written to.
   """

   @staticmethod
   def handleWithoutAnnotation( fragment, codeManager, printer ):
      assertIsNone( fragment.result )
      codeChunk = codeManager.popCodeForFragment( fragment )
      printer.writeToScratchpads( codeChunk, Annotations.BlankBar )

   @staticmethod
   def handleContinuation( fragment, codeManager, printer ):
      codeChunk = codeManager.popCodeForFragment( fragment )
      printer.writeToScratchpads( codeChunk, Annotations.ContinuationBar )

   @staticmethod
   def handleEval( fragment, codeManager, printer ):
      assertIsNone( fragment.result )
      if fragment.continuation:
         FragmentHandler.handleContinuation( fragment, codeManager, printer )
         return
      codeChunk = codeManager.popCodeForFragment( fragment )
      printer.writeToScratchpads( codeChunk, Annotations.EvalBar )

   @staticmethod
   def handleResult( fragment, codeManager, printer ):
      if fragment.continuation:
         FragmentHandler.handleContinuation( fragment, codeManager, printer )
         return
      codeChunk = codeManager.popCodeForFragment( fragment )
      printer.writeToScratchpads( codeChunk,
                                  Annotations.fromResult( fragment.result ) )

   @staticmethod
   def handleFunctionCall( fragment, codeManager, printer ):
      if not fragment.continuation:
         printer.addFunctionInvocation( fragment.invocation )
      FragmentHandler.handleResult( fragment, codeManager, printer )

   @staticmethod
   def blockOrBracketHelper( fragment, codeManager, printer, character, annotation ):
      assertIsNone( fragment.result )
      assertEqual( fragment.location.length, 1 )
      codeChunk = codeManager.popCode( 1 )
      assertEqual( codeChunk, character )
      printer.writeToScratchpads( codeChunk, annotation )

   @staticmethod
   def handleOpenBlock( fragment, codeManager, printer ):
      FragmentHandler.blockOrBracketHelper( fragment, codeManager, printer,
                                            "{", Annotations.BlankBar )

   @staticmethod
   def handleCloseBlock( fragment, codeManager, printer ):
      FragmentHandler.blockOrBracketHelper( fragment, codeManager, printer,
                                            "}", Annotations.BlankBar )

   @staticmethod
   def handleOpenExplicitBracket( fragment, codeManager, printer ):
      FragmentHandler.blockOrBracketHelper( fragment, codeManager, printer,
                                            "(", Annotations.OpenBracketAnnotation )

   @staticmethod
   def handleCloseExplicitBracket( fragment, codeManager, printer ):
      FragmentHandler.blockOrBracketHelper( fragment, codeManager, printer,
                                            ")", Annotations.CloseBracketAnnotation )

   @staticmethod
   def handleOpenImplicitBracket( fragment, codeManager, printer ):
      assertIsNone( fragment.result )
      # Don't add more white space if it is not needed.
      n = printer.getTrailingWhitespaceInScratchpads()
      if n > 1:
         # The scratchpads have two or more characters of whitespace.
         # Trim one from each so the space is reused for the bracket.
         printer.trimTrailingCharInScratchpads()
      codeChunk = " "
      printer.writeToScratchpads( codeChunk, Annotations.OpenBracketAnnotation )

   @staticmethod
   def handleCloseImplicitBracket( fragment, codeManager, printer ):
      assertIsNone( fragment.result )
      # Don't add more white space if it is not needed.
      codeAhead = codeManager.peekAheadInLine( 2 )
      n = len( codeAhead ) - len( codeAhead.lstrip() )
      if n > 1:
         # There are two or more characters of whitespace coming.
         # Use one of them for this bracket.
         codeChunk = codeManager.popCode( 1 )
      else:
         codeChunk = " "
      printer.writeToScratchpads( codeChunk, Annotations.CloseBracketAnnotation )

   @staticmethod
   def handleOpenExpression( fragment, codeManager, printer ):
      assertIsNone( fragment.result )
      assert printer.expressionInProgress is False
      printer.expressionInProgress = True

   @staticmethod
   def handleCloseExpression( fragment, codeManager, printer ):
      assertIsNone( fragment.result )
      assert printer.expressionInProgress is True
      printer.expressionInProgress = False
      # BlankBar annotation lines are printed above unevaluated lines within in an
      # expression. There may or may not be annotations for the last line in the
      # expression. Ensure the blank line is rendered if there are not other
      # annotations for this line.
      printer.annotationScratchpad.renderEvenIfEmpty = True

   @staticmethod
   def handle( fragment, codeManager, printer ):
      FragmentHandler.dispatch[ fragment.fragmentType ]( fragment, codeManager,
                                                         printer )

FragmentHandler.dispatch = {
   "definition": FragmentHandler.handleWithoutAnnotation,
   "openBlock": FragmentHandler.handleOpenBlock,
   "closeBlock": FragmentHandler.handleCloseBlock,
   "statement": FragmentHandler.handleResult,
   "ifStatement": FragmentHandler.handleResult,
   "elseStatement": FragmentHandler.handleEval,
   "condition": FragmentHandler.handleResult,
   "functionCall": FragmentHandler.handleFunctionCall,
   "assignment": FragmentHandler.handleEval,
   "logicalOperator": FragmentHandler.handleResult,
   "openExplicitBracket": FragmentHandler.handleOpenExplicitBracket,
   "closeExplicitBracket": FragmentHandler.handleCloseExplicitBracket,
   "openImplicitBracket": FragmentHandler.handleOpenImplicitBracket,
   "closeImplicitBracket": FragmentHandler.handleCloseImplicitBracket,
   "openStatementExpression": FragmentHandler.handleOpenExpression,
   "closeStatementExpression": FragmentHandler.handleCloseExpression,
}
# Ensure we are covering all possible fragments
assertEqual( set( FragmentHandler.dispatch ), set( FragmentTypes ) )
