#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Example CAPI format
{
    "nlri": "10.1.2.3/32",
    "application": "inbound",
    "rcfCodeUnits": {
        "example_code_unit": {
            "functions": {
                "foo": {
                    "lines": {
                        "1": "function foo() {",
                        "2": "   if ...",
                    }
                }
            }
        }
        "example_code_unit_2":
        { ... }
    },
    "peers": {
        "1.1.1.1": {
            "rcfEval": {
                "entryPoint": "foo",
                "result": {
                   "value": "true",
                },
                "functionEvaluations": {
                    "foo": {
                        "codeUnitName": "example_code_unit",
                        "invocations": [
                            {
                                "callStackDepth": 0,
                                "result": {
                                   "value": "true",
                                }
                                "fragments":
                                [
                                    {
                                        "fragmentType": "definition",
                                        "location": {
                                            "line": 1,
                                            "column": 0,
                                            "length": 14
                                        }
                                    },
                                    {
                                        "fragmentType": "block",
                                        ...
                                    },
                                    {
                                        "fragmentType": "ifStatement",
                                        ...
                                    },
                                    ...
                                        {
                                            "fragmentType": "functionCall",
                                            ...
                                            "invocation": {
                                                "functionName": "bar",
                                                "invocationIndex": 0
                                            },
                                            "result": {
                                               "value": "true",
                                            }
                                        },
                                    ...
                                ]
                            }
                        ]
                    },
                    "bar": {
                        "codeUnitName": "example_code_unit_2",
                        ...
                    }
                }
            }
        }
    }
}
'''

import Tac
from TypeFuture import TacLazyType

from CliModel import (
   Bool,
   Dict,
   Enum,
   Int,
   List,
   Model,
   Str,
   Submodel,
)

FunctionDomainType = TacLazyType( "Rcf::Metadata::FunctionDomain" )

class RcfDebugFragmentLocation( Model ):
   '''
   Model to represent a fragments location within a code unit.
   '''
   line = Int( help="Line number of the location within a code unit" )
   column = Int( help="Column number of the location on a line" )
   length = Int( optional=True, help="Length of the fragment in bytes" )

class RcfDebugFunctionInvocationReference( Model ):
   '''
   Model to represent a reference to the invocation of another RCF function.

   This model can contain values of function arguments that we passed in future.
   '''
   functionName = Str( help="Name of the RCF function that was invoked" )
   invocationIndex = Int( help="A function can be invoked multiple times during RCF "
                          "evaluation. The evaluation of the called function may "
                          "differ each time. This index is used to indicate the "
                          "particular invocation being referenced" )

FragmentTypes = Tac.Type( "Rcf::Debug::FragmentTypes" ).attributes

class RcfDebugResult( Model ):
   '''
   Model to represent a result produced during an RCF evaluation.
   '''
   termination = Enum( values=( "noTermination",
                                "returnToCaller",
                                "exitToPoa" ),
                       help="Trilean termination flag",
                       optional=True )
   value = Enum( values=( "true", "false", "unknown" ),
                 help="Trilean result value" )
   reason = Str( help="Explanation of the result", optional=True )

class RcfDebugFragment( Model ):
   '''
   Model to represent a fragment of code that was part of an RCF function invocation.

   In future this model can be extended to include
   - LHS and RHS values, before and after the expression.
   - Expression text
   '''
   fragmentType = Enum( values=FragmentTypes, help="Type of code the fragment "
                        "represents" )
   continuation = Bool( help="Whether this fragment is for an expression continuing "
                        "from a previous line" )
   location = Submodel( valueType=RcfDebugFragmentLocation,
                        help="Location the fragment occurred at in the RCF code" )
   result = Submodel( valueType=RcfDebugResult, optional=True,
                      help="Result of the code fragments evaluation" )
   invocation = Submodel( valueType=RcfDebugFunctionInvocationReference,
                          optional=True,
                          help="Reference to the RCF invocation that occurred in "
                          " the code fragment" )

class RcfDebugFunctionInvocation( Model ):
   '''
   Model to represent the evaluation of an RCF function invocation.
   '''
   fragments = List( valueType=RcfDebugFragment,
                     help="Fragments representing pieces of code relevant to the "
                     "evaluation of the invoked function. Sorted first by their "
                     "location (line and column number) and then by the order they "
                     "occurred" )
   callStackDepth = Int( help="How deep in the call stack this invocation was "
                         "evaluated" )
   result = Submodel( valueType=RcfDebugResult,
                      help="Result of the function invocations evaluation" )

class RcfDebugFunction( Model ):
   '''
   Model to represent information relating to an RCF function during the course of
   the RCF evaluation.
   '''
   invocations = List( valueType=RcfDebugFunctionInvocation,
                       help="A function can be invoked multiple times during RCF "
                       "evaluation. The evaluation of the called function may "
                       "differ each time. This is a list of all of the invocations"
                       "that occurred" )
   codeUnitName = Str( help="Code unit containing the function definition",
                       optional=True )
   functionDomain = Enum( values=FunctionDomainType.attributes,
                          help="Source of this RCF function." )

class RcfDebugEvaluation( Model ):
   '''
   Model to represent the debug information from the RCF evaluation.
   '''
   entryPoint = Str( help="The initial RCF expression that was invoked" )
   result = Submodel( valueType=RcfDebugResult,
                      help="The overall result of the RCF evaluation" )
   functionEvaluations = Dict( keyType=str, valueType=RcfDebugFunction,
                               help="Evaluation information for each function "
                               "invoked during the RCF evaluation keyed by "
                               "function name" )

class RcfFunctionText( Model ):
   '''
   Model to represent the RCF text/code in a given function.
   '''
   lines = Dict( keyType=int, valueType=str,
                 help="Lines of RCF text. Keyed by line number" )
   _rcfCodeVersion = Int( help="The version of the RCF text this model was "
                          "constructed from (hidden)" )

   def rcfCodeVersion( self ):
      # Accesser for protected member
      return self._rcfCodeVersion

class RcfCodeUnitText( Model ):
   '''
   Model to represent the RCF functions in a given code unit.
   '''
   functions = Dict( keyType=str, valueType=RcfFunctionText,
                     help="Reference to the text of RCF functions where evaluation "
                     "occurred, keyed by the RCF function name" )
