#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Figure from AID/8667

Demonstrate flow of RCF debug rendering for an given RCF function invocation

      +----------+                                       +----------------------+
      |          |                                       |InvocationCodeManager |
      | +--------+--+                                    |      +--------+      |
      | |           |                                    |      |        |      |
      +-+  +--------+--+                                 |      |RCF text|      |
        |  |           |           +---------------------|      |  ...   |      |
        +--+ Fragments |           |                     |      |  ...   |      |
  +------- |           |           v                     |      |        |      |
  |        +-----------+   +-----------+                 |      +--------+      |
  |                        | Fragment  |                 +----------------------+
  |  +------------------+  |  Handler  | +--------------------------------------+
  |  |InvocationRenderer|  |           | |          InvocationPrinter           |
  |  |                  |  +-----------+ |   =============         +----------+ |
  |  |                  |        |       |  |  Write to   |        |Annotation| |
  |  |                  |        +-------+->| Scratchpads |-----+->|Scratchpad| |
  +->|Fill in code      |        |       |   =============      |  +----------+ |
     |between fragments |        |       |                      v        |      |
     |                  |        |       |                +----------+   |      |
     |                  |        |       |  +----------+  |   Code   |   |      |
     |                  |        |       |  | Pending  |  |Scratchpad|   +---+  |
     |                  |        +-------+->| Function |  +----------+       |  |
     |                  |                |  |  Calls   |-------+  |          |  |
     |If fragment is on |                |  +----------+       v  v          |  |
     |a new line        |                |                   =============   |  |
     |- advance in code |                |                  |    Flush    |  |  |
     |- flush ----------+----------------+----------------->| Scratchpads |<-+  |
     |                  |                |                   =============      |
     |                  |                |                         |            |
     |                  |                |                         v            |
     |                  |                |              +---------------------+ |
     |                  |                |       +------|Output Buffer        | |
     |If no fragments   |                |       v      |1. TextRenderer      | |
     |left              |                |   =========  |2. ScratchpadRenderer| |
     |- flush ----------+----------------+->|  Flush  | |3. NewlineRenderer   | |
     |                  |  +----------+  |   =========  |4. InvocationRenderer| |
     |                  |  |  stdout  |  |       |      +---------------------+ |
     |                  |  |          |<-+-------+                              |
     +------------------+  +----------+  +--------------------------------------+
'''

from CliPlugin.RcfDebugRenderLib import Scratchpad
import CliPlugin.PolicyEvalModels

from CliPlugin.RcfDebugRenderLib import (
   Annotations,
   InvocationCodeManager,
   InvocationContext,
   RenderContext,
)
from CliPlugin.RcfDebugFragmentHandler import FragmentHandler

def formatCallStackPrefix( callStackDepth ):
   return "|" * callStackDepth

def formatLineNumber( lineNumber, width ):
   if lineNumber is not None:
      assert len( str( lineNumber ) ) <= width
      return "{:>{width}}".format( lineNumber, width=width )
   else:
      return " " * width

class TextRenderer:
   """
   Representation of text in the InvocationPrinter to be rendered to the CLI.

   Constructor Arguments:
      text (str):
         The text to be rendered to the CLI.
      callStackDepth (int):
         How deep in the call stack this invocation is.
   """
   def __init__( self, text, callStackDepth=0 ):
      self.text = text
      self.callStackDepth = callStackDepth

   def render( self, maxLineNumberWidth=None ):
      """
      Render the text to the CLI.
      Args:
         maxLineNumberWidth (int):
            Not used in this implementation.
      """
      output = formatCallStackPrefix( self.callStackDepth )
      if self.text:
         output += self.text.rstrip()
      return output + '\n'

class NewlineRenderer( TextRenderer ):
   """
   Representation of a newline in the InvocationPrinter to be rendered the CLI.
   For example used to add vertical spacing between a line of code that calls a
   function and the function explanation.
   Empty lines are typically skipped, only explicit blank lines are rendered.

   Constructor Arguments:
      callStackDepth (int):
         How deep in the call stack the invocation is.
   """

   def __init__( self, callStackDepth=0 ):
      super().__init__( None, callStackDepth=callStackDepth )

class ScratchpadRenderer( TextRenderer ):
   """
   Representation of a line of annotation or code in the InvocationPrinter to
   be rendered to the CLI.

   Constructor Arguments:
      scratchpad (Scratchpad):
         The scratchpad containing the code or annotation to be rendered.
      callStackDepth (int):
         How deep in the call stack this invocation is.
      lineNumber (int):
         An optional line number to render, only used if the line is for user
         provided RCF text.
   """
   def __init__( self, scratchpad, callStackDepth=0, lineNumber=None, ):
      text = None
      if not scratchpad.empty() or scratchpad.renderEvenIfEmpty:
         text = scratchpad.read()
      super().__init__(
         text, callStackDepth=callStackDepth )
      self.lineNumber = lineNumber

   def render( self, maxLineNumberWidth=1 ):
      """
      Render the scratchpad to the CLI.
      Args:
         maxLineNumberWidth (int):
            The width of the largest line number encountered in the function. This is
            required to provide the same column spacing for line numbers for every
            line.
      """
      if self.text is None:
         return ''
      output = formatCallStackPrefix( self.callStackDepth )
      output += formatLineNumber( self.lineNumber, maxLineNumberWidth )
      output += " " if output else ""
      output += self.text
      return output.rstrip() + '\n'

class InvocationPrinter:
   """
   This type is responsible for managing the output buffering on an invocations
   render. As fragments are processed two scratchpads are written to for a given
   line of code. An annotationScratchpad and a codeScratchpad.
   Once the renderer moves on to a new line of RCF code, these scratchpads are
   flushed to the outputBuffer in the order they should be rendered. When the
   function invocation is finished, and the largest line number width is known to
   render the line number spacing consistently, the outputBuffer is flushed.
   The InvocationRenderer is responsible for determining when the various flush
   methods should be called.

   Constructor Arguments:
      context (InvocationContext):
         Context object for this specific invocation of a function.
      annotationsBelowCode (bool):
         Indicates whether annotations should be rendered above code or below
         code. Default is to above code.
   Attributes:
      annotationScratchpad (str):
         Annotations to be rendered for the current line.
      codeScratchpad (str):
         RCF code to be rendered for the current line.
      expressionInProgress (bool):
         An openExpression fragment has been encountered but the closeExpression
         fragment has not. The openExpression could have been on a previous line.
         This is used to ensure an expression explanation is rendered together
         before rendering any called functions within the expression.
      pendingFunctionCalls (List of InvocationRenderer objects):
         Explanations of RCF function calls are not rendered immediately. They are
         deferred until expression explanations have completed and a newline is
         encountered. At this point they are flushed.
      outputBuffer (List of *Renderer objects):
         RCF code lines, annotation lines, newlines, or RCF function invocations to
         be rendered.
   """

   def __init__( self, context, annotationsBelowCode=False ):
      self.context = context
      self.annotationScratchpad = Scratchpad()
      self.codeScratchpad = Scratchpad()
      self.expressionInProgress = False
      self.pendingFunctionCalls = []
      self.outputBuffer = []
      self.annotationsBelowCode = annotationsBelowCode

   def writeToScratchpads( self, code, annotation ):
      """
      Write RCF code and annotations into their respective scratchpads.

      Args:
         code (str):
            The RCF code to be written to a scratchpad.
         annotation (str):
            The annotation to be written to a scratchpad. Whitespace the same length
            as the RCF code is used if None.
      """
      self.annotationScratchpad.write( annotation.getText( len( code ) ) )
      self.codeScratchpad.write( code )

   def getTrailingWhitespaceInScratchpads( self ):
      """
      Count the minimum number of trailing whitespace characters in the scratchpads.
      """
      annotationStr = self.annotationScratchpad.read()
      annotationTrailingWhitespaceCount = ( len( annotationStr ) -
                                            len( annotationStr.rstrip() ) )
      codeStr = self.codeScratchpad.read()
      codeTrailingWhitespaceCount = len( codeStr ) - len( codeStr.rstrip() )
      return min( annotationTrailingWhitespaceCount, codeTrailingWhitespaceCount )

   def trimTrailingCharInScratchpads( self ):
      """
      Trim a single character from the end of the scratchpads.
      """
      self.annotationScratchpad.trimTrailingChar()
      self.codeScratchpad.trimTrailingChar()

   def addFunctionInvocation( self, invocation ):
      """
      Add a function invocation to be rendered. These invocations are not rendered
      immediately but are placed in a pending collection to be drained when
      flushFunctionCallsToBuffer is called.

      Args:
         invocation (RcfDebugModels.RcfDebugFunctionInvocationReference):
            The CLI models containing the details of the invocation that is
            referenced by a fragment.
      """
      if self.context.renderContext.isHiddenFunction( invocation.functionName ):
         return
      functionCallRenderer = InvocationRenderer(
         invocation.functionName,
         self.context.renderContext,
         invocationIndex=invocation.invocationIndex )
      self.pendingFunctionCalls.append( functionCallRenderer )

   def writeToOutput( self, renderer ):
      """
      Insert a *Renderer object into the output buffer directly. Typically
      used to insert a TextRenderer or NewlineRenderer

      Args:
         renderer (TextRenderer|NewlineRenderer):
            The object to be rendered to stdout
      """
      assert self.codeScratchpad.empty() and self.annotationScratchpad.empty()
      assert not isinstance( renderer, list )
      self.outputBuffer.append( renderer )

   def flushScratchpadsToBuffer( self, lineNumber, finalFlush=False ):
      """
      Move the contents of the completed scratchpads to the outputBuffer and reset.

      Args:
         lineNumber (int):
            The line number of the RCF code currently in the scratchpad.
         finalFlush (bool):
            Indicates the invocation is finished and this is the last time
            this method will be called.
      """
      assert len( self.annotationScratchpad ) == len( self.codeScratchpad )
      assert not ( finalFlush and self.expressionInProgress )

      callStackDepth = self.context.callStackDepth
      if self.context.codeUnitKey.isPoaWrapper():
         # Never print line numbers for POA wrappers
         lineNumber = None
      newLines = [ ScratchpadRenderer( self.annotationScratchpad,
                                       callStackDepth=callStackDepth ),
                   ScratchpadRenderer( self.codeScratchpad,
                                       callStackDepth=callStackDepth,
                                       lineNumber=lineNumber ) ]
      if self.annotationsBelowCode:
         newLines = reversed( newLines )
      self.outputBuffer += newLines
      self.annotationScratchpad = Scratchpad()
      self.codeScratchpad = Scratchpad()
      if not self.expressionInProgress:
         # We want to keep all parts of an expression grouped, so we only
         # render function calls when we are at the end of a line and not in the
         # middle of an expression.
         self.flushFunctionCallsToBuffer( finalFlush )

   def flushFunctionCallsToBuffer( self, finalFlush ):
      """
      Move any pending function call explanations to the outputBuffer. Should be
      invoked once a newline is reached and no expression are still being explained.

      Args:
         finalFlush (bool):
            Indicates the invocation is finished and this is the last time
            this method will be called.
      """
      assert self.codeScratchpad.empty() and self.annotationScratchpad.empty()
      assert not self.expressionInProgress
      if self.pendingFunctionCalls:
         nlRenderer = NewlineRenderer()
         for invocationRenderer in self.pendingFunctionCalls:
            self.outputBuffer += [ nlRenderer, invocationRenderer ]
         self.pendingFunctionCalls = []
         if not finalFlush:
            # Spacing is only needed after the function call(s) if there is more
            # output for this invocation.
            self.outputBuffer.append( NewlineRenderer() )

   def flush( self, lastLineNumber ):
      """
      Render the contents of the outputBuffer. This is left until the end of the
      functions invocations so that the largest line number width is known for
      formatting.

      Args:
         lastLineNumber (int):
            The last line number encountered in the RCF function.
      """
      # last minute flush
      self.flushScratchpadsToBuffer( lastLineNumber, finalFlush=True )

      maxLineNumberWidth = len( str( lastLineNumber ) )
      assert self.outputBuffer
      output = ''
      for renderer in self.outputBuffer:
         output += renderer.render( maxLineNumberWidth=maxLineNumberWidth )
      self.outputBuffer = []
      return output

class InvocationRenderer:
   """
   This type is responsible for rendering the debugging information for a given RCF
   function invocation.
   This type will manage
      * passing fragments on to the FragmentHandler.
      * rendering the code between fragments within a line.
      * rendering unevaluated lines within an expression.
      * flushing scratchpads.

   Constructor Arguments:
      functionName (str):
         Name of the function being rendered.
      renderContext (RenderContext):
         The global rendering context, provides access to the CliModel.
      invocationIndex (int):
         Index of the invocation being rendered within the CliModel as a function
         may be invoked multiple times. Defaults to 0.

   Attributes:
      invocation (RcfDebugFunctionInvocation):
         The CLI model (CAPI model) representing the given RCF function invocation.
      context (InvocationContext):
         The context for this particular invocation, contains the global context.
      codeManager (InvocationCodeManager):
         The object responsible for maintaining where the renderer is in the
         RCF functions code.
      printer (InvocationPrinter):
         Object responsible for managing RCF code lines, annotation lines, newlines,
         and RCF function invocations to be rendered.
   """
   fillLinesWithoutCode = False
   fillEmptyAnnotationsLineBetweenCode = True

   def __init__( self, functionName, renderContext, invocationIndex=0 ):
      self.invocation = renderContext.getInvocation( functionName, invocationIndex )
      codeUnitKey = renderContext.getCodeUnitKeyForFunction( functionName )
      self.context = InvocationContext(
         functionName, codeUnitKey, invocationIndex, self.invocation.callStackDepth,
         renderContext )
      self.codeManager = InvocationCodeManager( self.context )
      self.printer = InvocationPrinter( self.context )
      self.fragmentHandler = FragmentHandler

   def fillCodeInLine( self, upToOffset=None ):
      """
      Fill the code between fragments, up to a column number in the RCF code.

      Args:
         upToColumn (int):
            The column to fill the code up until.
      """
      code = self.codeManager.popLine( upToOffset=upToOffset )
      self.printer.writeToScratchpads( code, Annotations.BlankBar )

   def fillCodeBetweenLines( self, fromLine, toLine ):
      """
      Fill in any lines with code within a certain range without any annotations.

      Args:
         fromLine (int):
            The line to start filling RCF code from (exclusive)
         toLine (int):
            The line to end filling RCF code on (exclusive)
      """
      for lineNumber in range( fromLine + 1, toLine ):
         self.codeManager.advanceLineNumber( lineNumber )
         line = self.codeManager.popLine()
         if not self.fillLinesWithoutCode:
            # Skip lines that are only comments, determine by checking if there
            # is only whitespace before the first comment marker in the line.
            if not line.split( "#" )[ 0 ].strip():
               continue
         self.printer.writeToScratchpads( line, Annotations.BlankBar )
         self.printer.codeScratchpad.renderEvenIfEmpty = True
         self.printer.annotationScratchpad.renderEvenIfEmpty \
            = self.fillEmptyAnnotationsLineBetweenCode
         self.printer.flushScratchpadsToBuffer( lineNumber )

   def getFragmentIterator( self ):
      return iter( self.invocation.fragments )

   def iterFragments( self ):
      """
      Yield the next fragment in the RCF functions invocation to be render.
      If the next fragment is on a new line, manage:
        * flushing scratchpads.
        * rendering unevaluated lines within an expression.
        * advancing the code manager to the next line.

      Return: (RcfDebugFunctionFragment) The next fragment to be handled.
      """
      fragments = self.getFragmentIterator()
      fragment = next( fragments ) # pylint: disable=stop-iteration-return
      self.codeManager.advanceLineNumber( fragment.location.line )
      yield fragment

      for fragment in fragments:
         if fragment.location.line != self.codeManager.currentLineNumber:
            # moving onto a new line, we can dump the scratchpads into the
            # outputBuffer and reset
            self.fillCodeInLine()
            self.printer.flushScratchpadsToBuffer(
               self.codeManager.currentLineNumber )

            # An expression could be in progress over multiple lines.
            if self.printer.expressionInProgress:
               # Render the lines with code within the expression that
               # were not evaluated.
               self.fillCodeBetweenLines( self.codeManager.currentLineNumber,
                                          fragment.location.line )

            self.codeManager.advanceLineNumber( fragment.location.line )

         yield fragment

   def fillCodeUnitDetails( self ):
      """
      Render a line of text stating the code unit a function was defined in.
      """
      text = self.context.codeUnitKey.toStrepForDebugCli()
      self.printer.writeToOutput( TextRenderer( text, self.context.callStackDepth ) )

   def render( self, maxLineNumberWidth=None ):
      """
      Render the RCF function invocation to the CLI.

      This is a process of iterating over the fragments in the function invocations
      and rendering the code and annotations for each fragment (via FragmentHander)

      It is also necessary to render the code between fragments.

      Args:
         maxLineNumberWidth (int):
            Not used in this implementation.
      """

      self.fillCodeUnitDetails()
      for fragment in self.iterFragments():
         self.fillCodeInLine( upToOffset=fragment.location.column )
         self.fragmentHandler.handle( fragment, self.codeManager, self.printer )

      self.fillCodeInLine()
      return self.printer.flush( self.codeManager.currentLineNumber )

resultToCliString = {
   "true": "allowed",
   "false": "disallowed",
   "unknown": "disallowed"
}

def renderRcf( rcfEval, rcfCodeUnits ):
   context = RenderContext( rcfCodeUnits, rcfEval )
   mainFunctionName = context.getEntryPointName()
   mainFunctionResult = context.getEntryPointResult()

   noReturnString = ""
   if mainFunctionResult.termination == "noTermination":
      noReturnString = "! End of function without return or exit statement\n"
   print( "The path was {} by the evaluation of {}().\n{}".format(
      resultToCliString[ mainFunctionResult.value ], mainFunctionName,
      noReturnString ) )
   if not rcfEval.functionEvaluations:
      # The RCF function must be undefined or referencing undefined policy.
      print( mainFunctionResult.reason )
      return
   if context.isHiddenFunction( mainFunctionName ):
      # Builtins and OpenConfig functions don't have text and so are not rendered.
      return
   renderer = InvocationRenderer( mainFunctionName, context )
   print( renderer.render(), end='' )

# Making this a function so it is more obvious why tests need to import this file
def installRenderHook():
   CliPlugin.PolicyEvalModels.rcfDebugRenderHook = renderRcf

installRenderHook()
