# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSession
from CliPlugin.RcfCliSessionHelpers import isStartupConfig
from Toggles import RcfLibToggleLib

class RcfCodeUnitUrlInfo:
   """Scratchpad type for code unit URL info, models Rcf::RcfCodeUnitUrlInfo"""
   def __init__( self ):
      # If 'lastPulledUrl' is specified, a 'commit' should commit both
      # 'lastPulledUrl' and 'editSincePull'. If 'lastPulledUrl' is not specified,
      # a 'commit' should only commit the new 'editSincePull' value if a
      # 'RcfCodeUnitUrlInfo' exists in Sysdb for the specified unit name.
      self.lastPulledUrl = None
      self.editSincePull = False

class RcfScratchpad:
   """Temporary scratchpad for holding RCF data before it is published"""
   def __init__( self ):
      # Maps unit name (str) to rcfText (str) for that unit. The single unnamed code
      # unit exists in this collection with an empty string ( "" ) key.
      # A unit in this dict with value None indicates the unit is being deleted.
      self.rcfCodeUnitTexts = {}
      # Maps OpenConfig function name (str) to json blob (str) for that function.
      # A unit in this dict with value None indicates the unit is being deleted.
      self.openConfigFunctions = {}
      # Maps unit name (str) to URL metadata for that unit (RcfCodeUnitUrlInfo).
      # The single unnamed code unit exists in this collection with an empty string
      # ( "" ) key.
      self.rcfCodeUnitUrlInfos = {}
      # Maps code unit names to associated config tags.
      # A unit in this dict with a value of None indicates that the previous config
      # tag association is being deleted.
      self.rcfCodeUnitConfigTags = {}
      self.rcfLintResult = None
      self.openConfigValidationResult = None

   def getCodeUnitUrlInfo( self, unitName ):
      return self.rcfCodeUnitUrlInfos.get( unitName )

   def getOrInitCodeUnitUrlInfo( self, unitName ):
      if unitName not in self.rcfCodeUnitUrlInfos:
         self.rcfCodeUnitUrlInfos[ unitName ] = RcfCodeUnitUrlInfo()
      return self.rcfCodeUnitUrlInfos.get( unitName )

   def setCodeUnit( self, unitName, unitText, editSincePull, url ):
      """Helper function used by CLI command handlers to set scratchpad state
      for a single code unit.

      Args:
      - unitName (str): Name of the code unit
      - unitText (str): Text for the code unit
      - editSincePull (bool): Whether or not this code unit has been edited since it
         was last pulled (if at all)
      - url (str): URL representing location from which this code unit was pulled
      """
      # New RCF code for this unit will overwrite previous code for this unit
      self.rcfCodeUnitTexts[ unitName ] = unitText
      codeUnitUrlInfo = self.getOrInitCodeUnitUrlInfo( unitName )
      if url:
         codeUnitUrlInfo.lastPulledUrl = url
      codeUnitUrlInfo.editSincePull = editSincePull
      # Since we updated the scratchpad, reset the lint result to ensure another
      # lint is needed before the new scratchpad contents can be committed
      self.rcfLintResult = None
      self.openConfigValidationResult = None

   def setOpenConfigFunction( self, functionName, openConfigFunction ):
      """Helper function used by CLI command handlers to set scratchpad state
      for an OpenConfig function.

      Args:
      - functionName (str): Name of the function
      - openConfigFunction (str): Json blob for the function
      """
      self.openConfigFunctions[ functionName ] = openConfigFunction
      # Since we updated the scratchpad, reset the lint result to ensure another
      # lint is needed before the new scratchpad contents can be committed
      self.rcfLintResult = None
      self.openConfigValidationResult = None

   def hasVerifiedCode( self ):
      """ Whether the code held in this scratchpad is valid and ready to be
      committed to config.
      """
      rcfLintResultOk = self.result and self.result.success
      if not RcfLibToggleLib.toggleRcfPolicyDefinitionsEnabled():
         openConfigValidationResultOk = True
      else:
         openConfigValidationResultOk = self.ocResult and self.ocResult.success
      return rcfLintResultOk and openConfigValidationResultOk

   def hasVerifiedCodeVersion( self, rcfCodeVersionSysdb ):
      """ Check if the rcfCodeVersion of rcfLintResult matches the
      rcfCodeVersion in Sysdb.
      """
      rcfLintVersionOk = ( self.result and
                           self.result.rcfLintRequest.rcfCodeVersion ==
                              rcfCodeVersionSysdb )
      if not RcfLibToggleLib.toggleRcfPolicyDefinitionsEnabled():
         openConfigVersionOk = True
      else:
         openConfigVersionOk = ( self.ocResult and
                                 self.ocResult.request.rcfCodeVersion ==
                                    rcfCodeVersionSysdb )
      return rcfLintVersionOk and openConfigVersionOk

   def isReadyToCommit( self, mode, rcfConfig, rcfStatus ):
      hasVerifiedCode = self.hasVerifiedCode()
      hasVerifiedCodeVersion = self.hasVerifiedCodeVersion(
            rcfConfig.rcfCodeVersion )

      if isStartupConfig( mode ):
         # We don't lint during startup
         assert not ( self.result or self.ocResult )
         # The Rcf agent will indicate any errors in the RcfStatus
         return True
      elif hasVerifiedCode and hasVerifiedCodeVersion:
         # The code has been verified, fallthrough and commit it
         return True
      else:
         mode.addWarning(
            'Pending routing control function changes need to be compiled '
            'successfully before committing' )
         return False

   @property
   def result( self ):
      return self.rcfLintResult

   @property
   def ocResult( self ):
      return self.openConfigValidationResult

def getOrInitScratchpad( mode ):
   scratchpad = getScratchpad( mode )
   if scratchpad is None:
      scratchpad = initializeScratchpad( mode )
   return scratchpad

def getSessionAttribute( mode, attr ):
   if mode.session.inConfigSession():
      em = mode.session_.entityManager_
      sessionName = CliSession.currentSession( em )
      csAttribute = mode.session.sessionData( attr )
      return csAttribute.get( sessionName ) if csAttribute else None
   else:
      return mode.session.sessionData( attr )

def initializeSessionAttribute( mode, attr, initValue ):
   if mode.session.inConfigSession():
      em = mode.session_.entityManager_
      sessionName = CliSession.currentSession( em )
      csAttribute = mode.session.sessionData( attr )
      if csAttribute is None:
         mode.session.sessionDataIs( attr, {} )
         csAttribute = mode.session.sessionData( attr )
      if sessionName not in csAttribute:
         csAttribute[ sessionName ] = initValue
      attribute = csAttribute[ sessionName ]
   else:
      attribute = mode.session.sessionData( attr )
      if attribute is None:
         mode.session.sessionDataIs( attr, initValue )
         attribute = mode.session.sessionData( attr )
   return attribute

def removeSessionAttribute( mode, attr ):
   if mode.session.inConfigSession():
      em = mode.session_.entityManager_
      sessionName = CliSession.currentSession( em )
      csAttribute = mode.session.sessionData( attr )
      if csAttribute and sessionName in csAttribute:
         del csAttribute[ sessionName ]
   else:
      mode.session.sessionDataIs( attr, None )

def getScratchpadAttr( mode ):
   if mode.session.inConfigSession():
      return 'rcfConfigSessionScratchpads'
   else:
      return 'rcfScratchpad'

def getScratchpad( mode ):
   return getSessionAttribute( mode, getScratchpadAttr( mode ) )

def initializeScratchpad( mode ):
   return initializeSessionAttribute( mode, getScratchpadAttr( mode ),
                                      RcfScratchpad() )

def removeScratchpad( mode ):
   removeSessionAttribute( mode, getScratchpadAttr( mode ) )
