# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
import time
import Tac
import CliModel
import RedSupLib
import Ark

RedundancyMode = Tac.Type( "Election::RedundancyMode" )
RedundancyProtocol = Tac.Type( "Redundancy::RedundancyProtocol" )
PeerElectionState = Tac.Type( "Election::PeerState" )

class RedundancyStates( CliModel.Model ):
   myMode = CliModel.Enum( values=RedundancyMode.attributes,
                           help="Redundancy mode of the supervisor" )
   peerMode = CliModel.Enum( values=RedundancyMode.attributes,
                             help="Redundancy mode of the peer supervisor" )
   configuredProtocol = CliModel.Enum( values=RedundancyProtocol.attributes,
                                       help="Configured redundancy protocol" )
   operationalProtocol = CliModel.Enum( values=RedundancyProtocol.attributes,
                                        help="Operational redundancy protocol" )
   slotId = CliModel.Int( help="Slot id of the supervisor" )
   unitDesc = CliModel.Enum( values=( "Primary", "Secondary" ),
                             help="Description of supervisor" )
   communicationDesc = CliModel.Enum( values=( "Up", "Down" ),
                                      help="Description of communication link" )
   peerState = CliModel.Enum( values=PeerElectionState.attributes,
                              help="Standby supervisor state" )
   globalStageCompletionTimeout = CliModel.Float( 
                            help="switchover completion timeout in seconds" )
   globalStageCompletionTimeoutDefault = CliModel.Float( 
                      help="switchover completion timeout default in seconds" )
   switchoverReady = CliModel.Bool( help="Is switch ready for switchover" )
   switchoverReadyTimeElapsed = CliModel.Bool(
         help="Has the mandatory wait time after switchover elapsed " )
   allAgentSsoReady = CliModel.Bool( help="Agents not ready for SSO switchover" )
   agentsNotReady = CliModel.List( valueType=str,
                                   help="Agents not ready for SSO switchover",
                                   optional=True )
   pcieSsoReady = CliModel.Bool( help="Is PCIE switch ready for switchover",
                                 optional=True )
   pcieModulesNotReady = CliModel.List( valueType=str,
                         help="PCIE modules not ready for SSO switchover",
                         optional=True )

   lastRedundancyModeChangeTime = CliModel.Float(
                                 help="Time of last redundancy mode change in UTC",
                                 optional=True )
   lastRedundancyModeChangeReason = CliModel.Str(
                                    help="Reason for last redundancy mode change" )

   lastSwitchoverTime = CliModel.Float(
                                 help="Time of last switchover in UTC",
                                 optional=True )
   lastSwitchoverReason = CliModel.Str(
                                 help="Reason for last switchover",
                                 optional=True )
   cardsSsoUnsupported = CliModel.List( valueType=str,
                                        help="Cards SSO unsupported",
                                        optional=True )

   def render( self ):
      print( "  my state = %s" % RedSupLib.redModeToDisplayFormat( self.myMode,
                                   self.operationalProtocol ) )
      print( "peer state = %s" % RedSupLib.redModeToDisplayFormat( self.peerMode,
                                   self.operationalProtocol ) )
      print( "      Unit = %s" % self.unitDesc )
      print( "   Unit ID = %d" % self.slotId )
      print( "" )
      msgStr = "Redundancy Protocol (%s) = %s"
      print( msgStr % ( "Operational",
                    RedSupLib.protocolToDisplayFormat( self.operationalProtocol ) ) )
      print( msgStr % ( "Configured",
                     RedSupLib.protocolToDisplayFormat( self.configuredProtocol ) ) )
      print( "Communications = %s" % self.communicationDesc )
      if self.operationalProtocol == "sso":
         if not self.globalStageCompletionTimeout:
            print( "switchover completion timeout disabled" )
         elif self.globalStageCompletionTimeout == \
            self.globalStageCompletionTimeoutDefault:
            print( "switchover completion timeout = %s seconds (default)" %
                   self.globalStageCompletionTimeout )
         else:
            print( "switchover completion timeout = %s seconds" %
                   self.globalStageCompletionTimeout )
      if self.switchoverReady:
         print( "Ready for switchover" )
      else:
         switchoverNotReadyDesc = ""
         printDummyAgent = False
         printUnsupportedCards = False
         if self.configuredProtocol == self.operationalProtocol and \
            self.operationalProtocol == "simplex" and \
            self.myMode == "active":
            switchoverNotReadyDesc = "Redundancy protocol is simplex"
         elif self.peerState == "notInserted":
            switchoverNotReadyDesc = "Peer supervisor is not present"
         elif self.peerState == "poweredOff":
            switchoverNotReadyDesc = "Peer supervisor is powered off"
         elif self.peerState == "poweringOn":
            switchoverNotReadyDesc = "Peer supervisor is either "\
                                       "booting up or disabled"
         elif self.peerState == "unknownPeerState":
            switchoverNotReadyDesc = "Peer state is unknown"
         elif self.agentsNotReady:
            switchoverNotReadyDesc = "Agents not ready in standby supervisor"
         elif self.pcieModulesNotReady:
            switchoverNotReadyDesc = "PCIE devices not ready "\
                                      "in standby supervisor"
         elif self.cardsSsoUnsupported:
            switchoverNotReadyDesc = "SSO unsupported cards are present"
            printUnsupportedCards = True
         elif not self.allAgentSsoReady:
            switchoverNotReadyDesc = "Agents not ready in standby supervisor"
            printDummyAgent = True
         elif not self.switchoverReadyTimeElapsed:
            switchoverNotReadyDesc = "Agents not ready in standby supervisor"
            printDummyAgent = True
         print( "Not ready for switchover (%s)" % switchoverNotReadyDesc )
         if self.agentsNotReady:
            print( "Agents not ready =" )
            for agent in self.agentsNotReady:
               print( "   %s" % agent )
         elif self.pcieModulesNotReady:
            print( "PCIE modules not ready =" )
            for module in self.pcieModulesNotReady:
               print( "   %s" % module )
         if printUnsupportedCards:
            print( "Cards SSO unsupported =" )
            for card in sorted( self.cardsSsoUnsupported ):
               print( "   %s" % card )
         if printDummyAgent:
            print( "Agents not ready =" )
            print( "   ElectionMgr" )
      print( "" )
      if self.myMode == "active":
         if self.lastRedundancyModeChangeTime:
            modeChangeTime = Ark.timestampToStr( self.lastRedundancyModeChangeTime -
                                                   Tac.utcNow() + Tac.now() )
         else:
            modeChangeTime = Ark.timestampToStr( self.lastRedundancyModeChangeTime )

         print( "Last operational redundancy mode change time = %s" 
                                                               % modeChangeTime )
         print( "Last operational redundancy mode change reason = %s" % 
            self.lastRedundancyModeChangeReason )
         if self.lastSwitchoverTime:
            switchoverTime = Ark.timestampToStr( self.lastSwitchoverTime -
                                                   Tac.utcNow() + Tac.now() )
         else:
            switchoverTime = Ark.timestampToStr( self.lastSwitchoverTime )
         print( "Last peer switchover time = %s" % switchoverTime )
         print( "Last peer switchover cause = %s" % 
                                                 self.lastSwitchoverReason )


class SsoSwitchoverLogs( CliModel.Model ):
   switchoverCount = CliModel.Int( help="Number of switchovers since reload" )
   lastSwitchoverTime = CliModel.Float( help="Time of last switchover",
                                        optional=True )
   lastSwitchoverStatus = CliModel.Enum( values=( "Completed", "In Progress" ),
                                         help="Status of last switchover",
                                         optional=True )
   class SwitchoverLog( CliModel.Model ):
      time = CliModel.Float( help="Time of the log relative to onset of switchover" )
      msg = CliModel.Str( help="Log message" )
   logs = CliModel.List( valueType=SwitchoverLog,
                         help="List of switchover logs",
                         optional=True )

   def render( self ):
      print( "Total number of Stateful Switchovers completed since reload:",
             end=' ' )
      print( self.switchoverCount )

      utcDelta = Tac.utcNow() - Tac.now()
      if self.lastSwitchoverTime:
         lastSwitchoverTime = self.lastSwitchoverTime - utcDelta
         print( "Latest Stateful Switchover occured %s @ %s (%s)" % (
                  Ark.timestampToStr( lastSwitchoverTime ),
                  Ark.timestampToStr( lastSwitchoverTime, relative=False ),
                  self.lastSwitchoverStatus.lower() ) )
      for log in self.logs:
         print( "%10.6f: %s" % ( log.time, log.msg ) )

class RedundancyTimeSync( CliModel.Model ):
   lastMonoTimeUpdate = CliModel.Float( help="Time of last update to Monotonic time "
                                              "sync" )
   currentMonoTimeDelta = CliModel.Float( help="Time difference of current Monotonic"
                                              " time (seconds)" )
   failCount = CliModel.Int( help="Number of times that Monotonic time failed to "
                                  "update" )
   lastErrorUpdating = CliModel.Str( help="Reason for last error while updating "
                                          "Monotonic time",
                                     optional=True )

   def render( self ):
      print( "Last Monotonic Time updated at %s" % time.ctime(
                                                          self.lastMonoTimeUpdate ) )
      if self.failCount == 0:
         print( "Current Monotonic Time Difference = %f"
                % self.currentMonoTimeDelta )
      else:
         print( "Number of times Monotonic Time update failed "
                "since last successful update = %d" % self.failCount )
         print( "Last Error encountered while updating "
                "monotonic time is : %s" % self.lastErrorUpdating )

class PeerSupeCliCmdModel( CliModel.Model ): 
   output = CliModel.Str( optional=True,  
      help='Output of the CLI command ran on the peer supervisor' )

   def render( self ):
      print( self.output )
