#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliPlugin.ReloadCli as ReloadCli # pylint: disable=consider-using-from-import
import CliPlugin.RedSupCli as RedSupCli # pylint: disable=consider-using-from-import
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
import BasicCliUtil

# Ensure that the secondary supervisor is ready to take over before reloading
# If it is not, prompt the user to make sure they still want to reload.
def doElectionCheckBeforeReload( mode, power, now ):
   # If disableElectionCheckBeforeReload is True, 
   # we know the peer supervisor will be disabled and don't need a warning.
   if getattr( mode, 'disableElectionCheckBeforeReload', False ):
      return True

   allowReload = True

   configFile = mode.session_.configFile
   if now or configFile is not None:
      # Don't prompt the user in non-interactive mode, although we don't expect
      # reload to be in a config file. Prompts are also not shown if the user
      # specified 'now' when reloading.
      return True
   
   electionStatus = RedSupCli.electionStatus
   
   if electionStatus.peerState != 'notInserted':
      peerRedMode = electionStatus.peerRedundancyMode
      
      message = None
      if peerRedMode == 'unknown' or electionStatus.peerState == 'unknownPeerState':
         message = 'The status of the peer supervisor is currently unknown.'
      elif peerRedMode == 'disabled':
         message = 'The peer supervisor is currently disabled.'

      if message is not None:
         while True:
            answer = BasicCliUtil.getChoice( mode, 
                  message + ' Proceed with reload? ', 
                  [ 'yes', 'cancel' ], 'yes' )
            if ReloadCli.answerMatches( answer, 'yes' ):
               pass
            elif ReloadCli.answerMatches( answer, 'cancel' ):
               allowReload = False
            else:
               # we don't understand the user's answer so prompt again
               print( "Invalid response" )
               continue

            # if we got here, we don't want to show the prompt again
            break

   return allowReload

# Register doElectionCheckBeforeReload to run before reloading.
# Register with a high priority so it happens near the beginning of the reload.
def registerReloadElectionCli():
   ReloadCli.registerReloadHook( doElectionCheckBeforeReload,
                                 'ReloadElectionCli', 'RUN_FIRST' )

# Only register this plugin on modular systems.
FruCli.registerModularSystemCallback( registerReloadElectionCli )
