#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliUtil
import CliPlugin.FruCli as FruCli # pylint: disable=consider-using-from-import
import CliPlugin.RedSupCli as RedSupCli # pylint: disable=consider-using-from-import
import CliPlugin.ReloadCli as ReloadCli # pylint: disable=consider-using-from-import
import LazyMount
import Tac
import WaitForWarmup
import os

fileStatus = None
em = None

FileReplicationStatus = Tac.Type( "FileReplication::FileStatus::Status" )

def Plugin( entityManager ):
   global fileStatus, em
   em = entityManager
   fileStatus = LazyMount.mount( entityManager, 'redundancy/fileReplication/status',
                                 'FileReplication::Status', 'r' )

# Get failed and unsynced files from Sysdb and return them in a tuple
def getFiles():
   failedFiles = []
   unsyncedFiles = []
   
   for r in fileStatus.requester.values():
      for f in r.file:
         if r.file[ f ].status == FileReplicationStatus.unsynchronized:
            unsyncedFiles.append( f )
         elif r.file[ f ].status == FileReplicationStatus.failed:
            failedFiles.append( f )
         
   return ( failedFiles, unsyncedFiles )

def getFileList( failedFiles, unsyncedFiles ):
   message = ''
   if len( failedFiles ) > 0:
      message += 'The following files failed to synchronize:\n   '
      message += '\n   '.join( failedFiles )
   if len( unsyncedFiles ) > 0:
      if message != '':
         message += '\n'
      message += 'The following files are currently unsynchronized:\n   '
      message += '\n   '.join( unsyncedFiles )
   
   return message
   

def getFileSummary( failedFiles, unsyncedFiles ):
   failedCount = len( failedFiles )
   unsyncedCount = len( unsyncedFiles )

   if failedCount == 0 and unsyncedCount == 0:
      return None
   
   message = ''
   if failedCount > 0:
      message += '%d files failed to synchronize.' % failedCount
   if unsyncedCount > 0:
      if message != '':
         message += ' '
      message += '%d files are currently unsynchronized.' % unsyncedCount
   return message

def filesSynchronized():
   files = getFiles()
   return len( files[ 0 ] ) == 0 and len( files[ 1 ] ) == 0

def doFileSyncBeforeReload( mode, power, now ):
   couldReload = True

   # Don't prompt the user in non-interactive mode or "now" is specified,
   # although we still want to wait for file replication.
   interactive = not now and mode.session_.isInteractive()

   electionStatus = RedSupCli.electionStatus

   # We should only worry about syncing if the second supervisor is valid
   if ( electionStatus.peerState == 'inserted' and
        electionStatus.peerRedundancyMode == 'standby' ):

      timeScale = float( os.environ.get( "RELOAD_FR_TIME_SCALE", "1" ) )

      # Wait for FileReplicator to be ready (max 5 seconds)
      try:
         WaitForWarmup.wait( em, agentList=[ 'FileReplicator' ], sleep=True,
                             warnAfter=False, timeout=5 * timeScale )
      except Tac.Timeout:
         pass

      waitForSync = True

      timeout = 10
      nonInteractiveProceed = False

      while True:
         if waitForSync:
            try:
               # Wait for files to sync
               Tac.waitFor( filesSynchronized, timeout=timeout * timeScale,
                            sleep=True, warnAfter=False,
                            maxDelay=0.5 )
            except Tac.Timeout:
               pass
         else:
            waitForSync = True

         # If all files are synchronized, exit
         files = getFiles()
         if len( files[ 0 ] ) == 0 and len( files[ 1 ] ) == 0:
            break

         # If the peer becomes disabled, exit
         if electionStatus.peerRedundancyMode == 'disabled':
            print( 'The other supervisor became disabled.'
                   ' Aborting file synchronization.' )
            break

         if not interactive:
            if nonInteractiveProceed:
               # proceed to reload
               mode.addWarning( "Not all files are replicated to the standby, "
                                "proceed to reload" )
               break

            # print a nicer message after the first wait
            print( "Waiting for file replication to complete..." )
            nonInteractiveProceed = True
            # the next wait is 20 seconds (total 30 seconds)
            timeout = 20
            continue

         promptMessage = getFileSummary( *files )
         answer = BasicCliUtil.getChoice( mode, 
               promptMessage + " Wait for synchronization to complete? ", 
               [ 'yes', 'no', 'cancel', 'view' ], 'yes' )

         # pylint: disable-next=no-else-continue
         if ReloadCli.answerMatches( answer, "yes" ):
            continue
         elif ReloadCli.answerMatches( answer, "no" ):
            pass
         elif ReloadCli.answerMatches( answer, "cancel" ):
            couldReload = False
         elif ReloadCli.answerMatches( answer, 'view' ):
            print( getFileList( *files ) )
            print( '' )
            # Don't wait for synchronization of files before prompting the user
            # again
            waitForSync = False
            continue
         else:
            # we don't understand the user's answer so prompt again
            print( "Invalid response" )
            continue

         # if we got here, we don't want to show the prompt again
         break
   return couldReload

# Register doFileSyncBeforeReload to be run before a reload takes place.
# Register with low priority so it gets run after any file saving operations.
#
# This should always be called (even with 'reload now').
def registerReloadFileSyncCli():
   ReloadCli.registerReloadHook( doFileSyncBeforeReload, 'ReloadFileSyncCli',
                                 'RUN_LAST', always=True )

# Only register this plugin on modular systems
FruCli.registerModularSystemCallback( registerReloadFileSyncCli )
