# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


import CliCommand
import CliMatcher
import ConfigMount
import Tac

from CliPlugin.RcfCliLib import (
   rcfFunctionMatcher,
   rcfKw,
)
from CliPlugin.RouteMapCli import mapNameMatcher
from CliPlugin.RouterGeneralCli import RouterGeneralVrfMode
from CliPlugin.RouterGeneralCli import routerGeneralVrfCleanupHook
from ConfigConsistencyChecker import UndefinedReferenceChecker
from Toggles import RcfLibToggleLib

config = None
RouteControlConstant = Tac.Type( 'Routing::RouteControl::Constant' )

#-------------------------------------------------------------------------------
# Config handler functions 
#-------------------------------------------------------------------------------
def maybeDeleteVrfConfig( vrfConfig ):
   if vrfConfig.fibIpv4Policy.mapOrRcf != RouteControlConstant.policyDefault:
      return
   elif vrfConfig.fibIpv6Policy.mapOrRcf != RouteControlConstant.policyDefault:
      return
   elif vrfConfig.resIpv4PolicyMap != RouteControlConstant.policyDefault:
      return
   elif vrfConfig.resIpv6PolicyMap != RouteControlConstant.policyDefault:
      return
   else:
      del config.vrfConfig[ vrfConfig.vrfName ]


matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
                                         helpdesc='IPv4 configuration commands' )
matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
                                         helpdesc='IPv6 configuration commands' )
matcherRib = CliMatcher.KeywordMatcher( 'rib',
                                        helpdesc='Routing table' )
#-------------------------------------------------------------------------------
# [ no | default ] rib ( ipv4 | ipv6 ) fib policy ( ROUTE_MAP | ( rcf FUNCTION ) )
# in "config-router-general-vrf" mode
#--------------------------------------------------------------------------------
class RibFibPolicyCmd( CliCommand.CliCommandClass ):
   if RcfLibToggleLib.toggleRcfFibPolicyEnabled():
      syntax = 'rib ( ipv4 | ipv6 ) fib policy ( ROUTE_MAP | ( rcf FUNCTION ) )'
   else:
      syntax = 'rib ( ipv4 | ipv6 ) fib policy ROUTE_MAP'
   noOrDefaultSyntax = 'rib ( ipv4 | ipv6 ) fib policy ...'
   data = {
      'rib': matcherRib,
      'ipv4': matcherIpv4,
      'ipv6': matcherIpv6,
      'fib': 'FIB configuration commands',
      'policy': 'FIB installation policy',
      'ROUTE_MAP': mapNameMatcher,
      'rcf' : rcfKw,
      'FUNCTION' : rcfFunctionMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      assert mode.vrfName
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args

      if RcfLibToggleLib.toggleRcfFibPolicyEnabled() and 'rcf' in args:
         # rcfFunctionMatcher requires function names to end in "()", but we
         # don't store the trailing () internally so strip them off here
         policyMapOrRcf = Tac.Value( "Routing::RouteControl::PolicyMapOrRcf",
                                     args[ 'FUNCTION' ][ : -2 ],
                                     True )
      else:
         if ( RcfLibToggleLib.toggleRcfFibPolicyEnabled() and
              args[ 'ROUTE_MAP' ].lower() == 'rcf' ):
            # pylint: disable-next=consider-using-f-string
            mode.addError( "%s is not a valid route map name" % args[ 'ROUTE_MAP' ] )
            return
         policyMapOrRcf = Tac.Value( "Routing::RouteControl::PolicyMapOrRcf",
                                     args[ 'ROUTE_MAP' ],
                                     False )
      
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         vrfConfig = config.vrfConfig.newMember( mode.vrfName )

      if ipv4:
         vrfConfig.fibIpv4Policy = policyMapOrRcf 
      elif ipv6:
         vrfConfig.fibIpv6Policy = policyMapOrRcf 

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      assert mode.vrfName
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         return

      policyMapOrRcf = Tac.Value( "Routing::RouteControl::PolicyMapOrRcf" )
      if ipv4:
         vrfConfig.fibIpv4Policy = policyMapOrRcf 
      elif ipv6:
         vrfConfig.fibIpv6Policy = policyMapOrRcf
   
      maybeDeleteVrfConfig( vrfConfig )

RouterGeneralVrfMode.addCommandClass( RibFibPolicyCmd )

#-------------------------------------------------------------------------------
# [ no | default ] rib ( ipv4 | ipv6 ) resolution policy POLICY
# in "config-router-general-vrf" mode
#--------------------------------------------------------------------------------
class RibResolutionPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'rib ( ipv4 | ipv6 ) resolution policy POLICY'
   noOrDefaultSyntax = 'rib ( ipv4 | ipv6 ) resolution policy ...'
   data = {
      'rib': matcherRib,
      'ipv4': matcherIpv4,
      'ipv6': matcherIpv6,
      'resolution': 'Recursive resolution configuration commands',
      'policy': 'Recursive resolution policy',
      'POLICY': mapNameMatcher,
   }

   @staticmethod
   def handler( mode, args):
      assert mode.vrfName
      mapName = args[ 'POLICY' ]
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         vrfConfig = config.vrfConfig.newMember( mode.vrfName )
      if ipv4:
         vrfConfig.resIpv4PolicyMap = mapName
      elif ipv6:
         vrfConfig.resIpv6PolicyMap = mapName
   
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      assert mode.vrfName
      ipv4 = 'ipv4' in args
      ipv6 = 'ipv6' in args
      vrfConfig = config.vrfConfig.get( mode.vrfName, None )
      if vrfConfig is None:
         return
      if ipv4:
         vrfConfig.resIpv4PolicyMap = RouteControlConstant.policyDefault
      elif ipv6:
         vrfConfig.resIpv6PolicyMap = RouteControlConstant.policyDefault
      maybeDeleteVrfConfig( vrfConfig )

RouterGeneralVrfMode.addCommandClass( RibResolutionPolicyCmd )


# Register callback handler for cleaning 'router general' vrf sub-modes
def cleanupRouterGeneralVrfMode( vrfName ):
   del config.vrfConfig[ vrfName ]

routerGeneralVrfCleanupHook.addExtension( cleanupRouterGeneralVrfMode )

class RouterGeneralRouteMapReferenceGatherer:
   """
   This class gathers references to route maps in commands under
   the 'router general' mode.
   """
   @staticmethod
   def gather( feature ):
      references = set()
      for vrfName in config.vrfConfig:
         vrfConfig = config.vrfConfig[ vrfName ]
         for attrName in [ 'fibIpv4Policy', 'fibIpv6Policy' ]:
            policy = getattr( vrfConfig, attrName )
            if not policy.rcf and policy.mapOrRcf:
               references.add( policy.mapOrRcf )
         for attrName in [ 'resIpv4PolicyMap', 'resIpv6PolicyMap' ]:
            routeMap = getattr( vrfConfig, attrName )
            if routeMap:
               references.add( routeMap )
      return references

UndefinedReferenceChecker.addReferenceGatherer(
      [ "Route map" ], RouterGeneralRouteMapReferenceGatherer )


def Plugin( entityManager ):
   global config

   config = ConfigMount.mount( entityManager,
                               'routing/general/config/routeControl',
                               'Routing::RouteControl::Config', 'w' )
