# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCli
import CliMatcher
from CliPlugin.RouteCacheModels import ThresholdLevels, Summary, VrfPrefix
from CliPlugin.RouteCacheCliLib import cacheKw, thresholdKw
from CliPlugin.RouteCacheShowCliLib import runAgentCommand
from CliPlugin.VrfCli import VrfExprFactory
from CliPlugin.IraIpRouteCliLib import prefixMatcher, isValidPrefixWithError
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliToken.Ipv4 import ipv4MatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
import ShowCommand
from TypeFuture import TacLazyType

matcherRoute = CliMatcher.KeywordMatcher( 'route', helpdesc='Route information' )
matcherAddrOrPrefix6 = Ip6AddrMatcher.Ip6AddrOrPrefixExprFactory(
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )


class ShowRouteCacheThreshold( ShowCommand.ShowCliCommandClass ):
   syntax = '''show route cache threshold'''
   data = {
         'route': matcherRoute,
         'cache': cacheKw,
         'threshold': thresholdKw,
         }
   cliModel = ThresholdLevels
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cliModel = ThresholdLevels
      cmd = "showRouteCacheThreshold"

      return runAgentCommand( mode, cmd, cliModel )

BasicCli.addShowCommandClass( ShowRouteCacheThreshold )

class ShowRouteCacheSummary( ShowCommand.ShowCliCommandClass ):
   syntax = '''show route cache [ VRF ] summary'''
   data = {
         'route': matcherRoute,
         'cache': cacheKw,
         'VRF': VrfExprFactory( helpdesc='Display VRF state', inclAllVrf=True ),
         'summary': "Display route cache summary ",
         }
   cliModel = Summary
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cliModel = Summary
      cmd = "showRouteCacheSummary"
      if args.get( 'VRF' ):
         cmd += '#V' + args.get( 'VRF' )

      return runAgentCommand( mode, cmd, cliModel )

BasicCli.addShowCommandClass( ShowRouteCacheSummary )

class ShowRouteCacheVrfPrefix( ShowCommand.ShowCliCommandClass ):
   syntax = '''show route cache VRF [ ipv4 ] [ PREFIX ]'''
   data = {
         'route': matcherRoute,
         'cache': cacheKw,
         'VRF': VrfExprFactory( helpdesc='Display VRF state', inclAllVrf=True ),
         'ipv4': ipv4MatcherForShow,
         'PREFIX': prefixMatcher,
         }
   cliModel = VrfPrefix
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cliModel = VrfPrefix
      cmd = "showRouteCacheVrf#V"
      cmd += args[ 'VRF' ]
      if 'PREFIX' in args:
         prefix = args.get( 'PREFIX' )
         if prefix:
            if not isValidPrefixWithError( mode, prefix ):
               return None
         cmd += '#P' + args[ 'PREFIX' ]

      return runAgentCommand( mode, cmd, cliModel )

BasicCli.addShowCommandClass( ShowRouteCacheVrfPrefix )

ArnetIpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
class ShowRouteCacheVrfPrefixV6( ShowCommand.ShowCliCommandClass ):
   syntax = '''show route cache VRF ipv6 [ PREFIX ]'''
   data = {
         'route': matcherRoute,
         'cache': cacheKw,
         'VRF': VrfExprFactory( helpdesc='Display VRF state', inclAllVrf=True ),
         'ipv6': ipv6MatcherForShow,
         'PREFIX': matcherAddrOrPrefix6,
         }
   cliModel = VrfPrefix
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cliModel = VrfPrefix
      cmd = "showRouteCacheVrfV6#V"
      cmd += args[ 'VRF' ]
      if 'PREFIX' in args:
         prefix = args[ 'PREFIX' ].stringValue
         cmd += '#P' + prefix

      return runAgentCommand( mode, cmd, cliModel )

BasicCli.addShowCommandClass( ShowRouteCacheVrfPrefixV6 )
