# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliModel
import TableOutput
import ArnetModel

class RouteCounterEntry( CliModel.Model ):
   prefix = ArnetModel.IpGenericPrefix( help='Route prefix' )
   packetCount = CliModel.Int( help='Packet count' )
   byteCount = CliModel.Int( help='Byte count' )
   backupPacketCount = CliModel.Int( help='Backup path packet count', optional=True )
   backupByteCount = CliModel.Int( help='Backup path byte count', optional=True )
   programmed = CliModel.Bool( help='Counter is programmed in hardware',
                               optional=True )

class RouteCounters( CliModel.Model ):
   counters = CliModel.List( valueType=RouteCounterEntry,
                             help='List of prefix counters' )

   uniqueBackup = CliModel.Bool( help='Dedicated backup enabled in VRF',
                                 optional=True )

class VrfRouteCounters( CliModel.Model ):
   vrfs = CliModel.Dict( keyType=str, valueType=RouteCounters,
                           help='A mapping of VRF name to per prefix counters' )

   def render( self ):
      if not self.vrfs:
         return

      justifyRight = TableOutput.Format( justify='right' )
      justifyLeft = TableOutput.Format( justify='left' )
      defaultFormat = [ justifyLeft ] + [ justifyRight ] * 2
      uniqueBackupFormat = [ justifyLeft ] + [ justifyRight ] * 4
      tableHeader = "VRF: {vrfName}"
      tableLegend = "Codes: * - counter not programmed in hardware"

      # Currently the only entry in the legend is used to note whether or not
      # the counter is programmed in hardware.  This check only
      # occurs on platforms advertising RouteIpv4 support. Because of this we don't
      # want to print the legend for other platforms.  This is a shortcut to avoid
      # repeating this check in each iteration.  Grab the first counter entry
      # and check for this optional attr.  If it exists in one counter it will
      # be present in all counters.
      firstVrf = next( iter( self.vrfs ) )
      firstCounterList = self.vrfs[ firstVrf ].counters
      if firstCounterList and firstCounterList[ 0 ].programmed is not None:
         print( tableLegend )

      for vrfName in self.vrfs:
         uniqueBackup = self.vrfs[ vrfName ].uniqueBackup
         if uniqueBackup:
            table = TableOutput.createTable( ( 'Prefix',
                                             ( 'Primary',
                                               ( 'Packet Count', 'Byte Count' ) ),
                                             ( 'Backup',
                                               ( 'Packet Count', 'Byte Count' ) ) ) )
            table.formatColumns( *uniqueBackupFormat )
         else:
            table = TableOutput.createTable( ( 'Prefix', 'Packet Count',
                                               'Byte Count' ) )
            table.formatColumns( *defaultFormat )

         for routeCounterEntry in self.vrfs[ vrfName ].counters:
            prefix = str( routeCounterEntry.prefix )
            # Platforms that don't support programming validation will not set this
            # attribute in the model.  Because of this we only want to mark the entry
            # as unprogrammed if this attribute is explicitly set to false.
            if routeCounterEntry.programmed is False:
               prefix += "*"

            if uniqueBackup:
               if routeCounterEntry.backupPacketCount is None:
                  backupPacketCount = '-'
                  backupByteCount = '-'
               else:
                  backupPacketCount = routeCounterEntry.backupPacketCount
                  backupByteCount = routeCounterEntry.backupByteCount

               table.newRow( prefix,
                             routeCounterEntry.packetCount,
                             routeCounterEntry.byteCount,
                             backupPacketCount,
                             backupByteCount )
            else:
               table.newRow( prefix,
                             routeCounterEntry.packetCount,
                             routeCounterEntry.byteCount )
         print( tableHeader.format( vrfName=vrfName ) )
         print( table.output() )
