# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import ConfigMount
from IpLibConsts import DEFAULT_VRF
import Tracing
from CliPlugin.IraRouterKernel import (
   RouterKernelConfigMode,
   routerKernelVrfHook
)

kernelProtoConfig = None

traceHandle = Tracing.Handle( 'RouterKernelCli' )
t5 = traceHandle.trace5 # Info

keywordProtocol = CliMatcher.KeywordMatcher( 'protocol',
      helpdesc='per-protocol kernel parameters' )
keywordUdpProtocol = CliMatcher.KeywordMatcher( 'udp', 
      helpdesc='UDP' )
keywordUnknownPort = CliMatcher.KeywordMatcher( 'unknown-port',
      helpdesc='Unknown port handling' )
keywordSendUnreachable = CliMatcher.KeywordMatcher( 'send-unreachable',
      helpdesc='Unreachable port handling' )
keywordAlways = CliMatcher.KeywordMatcher( 'always',
      helpdesc='Always send an ICMP reachable for packets sent to an unknown port' )

# Set / Reset the handling for for udp unknown port handling
def setKernelProtocolBehavior( mode, enable=False ):
   assert mode.vrfName
   if mode.vrfName != DEFAULT_VRF:
      mode.addError( "command only supported in the default VRF" )
      return

   kernelProtoConfig.udpSendUnreachable = enable

def setKernelUdpBehavior( mode, args ):
   t5( "Setting kernel udp behavior send-unreachable" )
   setKernelProtocolBehavior( mode, enable=True )

def clearKernelUdpBehavior( mode, args ):
   t5( "disabling kernel udp behavior to not send unreachable" )
   enable = kernelProtoConfig.udpSendUnreachableDefaultFalse
   setKernelProtocolBehavior( mode, enable=enable )

#-------------------------------------------------------------------------------
#ip1(config-router-kernel)#
# [ no | default ] protocol udp unknown-port send-unreachable always
#-------------------------------------------------------------------------------
class KernelUdpBehaviorCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol udp unknown-port send-unreachable always'
   noOrDefaultSyntax = 'protocol udp unknown-port send-unreachable ...'

   data = {
      'protocol': keywordProtocol,
      'udp': keywordUdpProtocol,
      'unknown-port': keywordUnknownPort,
      'send-unreachable': keywordSendUnreachable,
      'always': keywordAlways,
   }

   handler = setKernelUdpBehavior
   noOrDefaultHandler = clearKernelUdpBehavior

RouterKernelConfigMode.addCommandClass( KernelUdpBehaviorCmd )

def clearKernelUdpBehaviorExtension( vrf, enable ):
   kernelProtoConfig.udpSendUnreachable = False

routerKernelVrfHook.addExtension( clearKernelUdpBehaviorExtension )

def Plugin( entityManager ):
   global kernelProtoConfig

   kernelProtoConfig = ConfigMount.mount( entityManager,
                                          'routing/kernel/protocol',
                                          'AclAgent::KernelProtoConfig', 'w' )
