#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-from-import
# pylint: disable=consider-using-f-string

from __future__ import absolute_import, division, print_function
from ArnetLib import asnStrToNum
from Toggles import (
   BgpCommonToggleLib,
   RoutingLibToggleLib,
   UcmpLibToggleLib,
   RcfLibToggleLib
)
from CliParser import guardNotThisPlatform, guardNonDefaultVrf
from CliCommand import (
      CliCommandClass,
      Node,
      SetEnumMatcher,
      singleNode,
      hiddenKeyword,
)
from CliPlugin.RoutingBgpCli import (
      BgpCmdBaseClass,
      BgpNEDCmdBaseClass,
      configForVrf,
      TpFieldSetIpv4Mode,
      TpFieldSetIpv6Mode,
      VspFieldSetIpv4Mode,
      VspFieldSetIpv6Mode,
      getCurrentBgpAs,
      mpDirOptions,
      mpActionOptions,
      mpIncludeOptions,
      peergroupNameMatcher,
      OrrPositionNameExpression,
      RouterBgpAfIpMcastSharedModelet,
      RouterBgpAfIpUniSharedModelet,
      RouterBgpAfLabelSharedModelet,
      RouterBgpAfSharedModelet,
      RouterBgpAfSharedVrfModelet,
      RouterBgpBaseAfEvpnMode,
      RouterBgpBaseAfIpMulticastMode,
      RouterBgpBaseAfIpv6MulticastMode,
      RouterBgpBaseAfIpUniMode,
      RouterBgpBaseAfIpv6UniMode,
      RouterBgpBaseAfLabelV4Mode,
      RouterBgpBaseAfLinkStateMode,
      RouterBgpBaseAfSrTeMode,
      RouterBgpBaseAfDpsMode,
      RouterBgpBaseMode,
      RouterBgpDefaultVrfMode,
      RouterBgpSharedModelet,
      RouterBgpVrfAfIpMulticastMode,
      RouterBgpVrfAfIpMode,
      RouterBgpVrfAfIp6Mode,
      RouterBgpVrfAfIpv6MulticastMode,
      RouterBgpVrfMode,
      RouterBgpVrfSharedModelet,
      RouterBgpOrrPositionMode,
      routingSupportedGuard,
      TrafficPolicyFieldSetMappingsMode,
      VrfSelectionPolicyFieldSetMappingsMode,
      nhLuOrigLfibBackupIpFwdSupportedGuard,
      RouterBgpRfdPolicyMode,
      RouterBgpRouteDistinguisherMode,
      rfdPolicyNameMatcher,
      epePeerSetNameMatcher,
      hwSupportedGuard,
)
from CliPlugin.RouteMapCli import (
      mapNameMatcher,
      RtSooExtCommCliMatcher,
      prefixListNameMatcher,
)
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.Ip6AddrMatcher import Ip6PrefixMatcher
from CliPlugin.IpGenAddrMatcher import IpGenPrefixMatcher
import CliPlugin.AclCli as AclCli
from CliPlugin.RouteDistinguisher import RdDistinguisherMatcher
import CliPlugin.VrfCli as VrfCli
import CliPlugin.RcfCliLib as RcfCliLib
from CliPlugin.MaintenanceCliLib import maintenanceKwMatcher
from CliPlugin.IpRibLib import ResolutionRibsExprFactory
import CliPlugin.MplsCli as MplsCli
from CliPlugin.UcmpLibCliLib import ucmpSizeRange
from CliPlugin.WanTEShowCli import pathSelectionSupportedGuard
from CliPlugin import TunnelGlobalConfigMode
from CliPlugin import TunnelProfilesMode
import CliToken.RoutingBgp as bgpTokens
import CliToken.Clear as Clear
from CliToken.Router import routerMatcherForConfig as routerMatcher
from CliToken.Community import CommunityConstExpr
from CliToken import IpRibLibCliTokens
from CliToken.UcmpLibCliTokens import ( ucmpLibTokenUcmp, ucmpDevRangeMatcher )
from BgpLib import (
   bgpConfigAttrsAfMap,
   peerConfigAttrsAfMap,
   PeerConfigKey,
   vpnAfTypeMapInv,
)
from IpLibConsts import DEFAULT_VRF
import ConfigMount
import LazyMount
from CliMatcher import (
      DynamicIntegerMatcher,
      DynamicKeywordMatcher,
      DynamicNameMatcher,
      EnumMatcher,
      IntegerMatcher,
      KeywordMatcher,
      PatternMatcher,
)
import CliMatcher
import Tac
import BasicCliModes
import BasicCli
from TypeFuture import TacLazyType
import six
import re

AsnNotation = TacLazyType( 'Routing::AsnNotation' )
RedistInternalStateEnum = TacLazyType( "Routing::Bgp::RedistInternalState" )
ReflectedRouteAttrStateEnum = TacLazyType( "Routing::Bgp::ReflectedRouteAttrState" )
LabelRangeInfo = TacLazyType( 'Mpls::LabelRangeInfo' )
VrfLocalLabelConfig = TacLazyType( "Routing::Bgp::VrfLocalLabelConfig" )

U32_MAX_VALUE = 0xFFFFFFFF

routingHardwareStatus = None
bgpConfig = None
bgpVrfConfigDir = None

def ucmpSupportedGuard( mode, token ):
   rhs = routingHardwareStatus
   if rhs is None:
      rhs = mode.sysdbRoot[ 'routing' ][ 'hardware' ][ 'status' ]
   if rhs.ucmpSupported:
      return None
   else:
      return guardNotThisPlatform

def udpTunnelSupportedGuard( mode, token ):
   if mode.vrfName != DEFAULT_VRF:
      return guardNonDefaultVrf
   return None

bgpNode = Node( matcher=bgpTokens.bgp, guard=routingSupportedGuard )
ucmpNode = Node( matcher=ucmpLibTokenUcmp, guard=ucmpSupportedGuard )
udpTunnelNode = Node( matcher=bgpTokens.udpTunnel, guard=udpTunnelSupportedGuard )

def nhTypeFromAfiSafi( af, args ):
   pnht = None
   pathAfiSafiNhType = Tac.Type( 'Routing::Bgp::PathAfiSafiNexthopType' )
   if af == 'ipv4':
      pnht = pathAfiSafiNhType.ipv4UniNexthopsAll
   elif af == 'ipv6':
      if '6pe' in args:
         pnht = pathAfiSafiNhType.sixPeNexthopsAll
      else:
         pnht = pathAfiSafiNhType.ipv6UniNexthopsAll
   elif af == 'ipv4 labeled-unicast':
      pnht = pathAfiSafiNhType.ipv4MplsNexthopsAll
   elif af == 'ipv6 labeled-unicast':
      pnht = pathAfiSafiNhType.ipv6MplsNexthopsAll
   elif af == 'evpn':
      if 'vxlan' in args:
         pnht = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapVxlan
      elif 'mpls' in args:
         pnht = pathAfiSafiNhType.l2vpnEvpnNexthopsEncapMpls
   elif af == 'vpn-ipv4':
      pnht = pathAfiSafiNhType.ipv4MplsVpnNexthopsEncapMpls
   elif af == 'vpn-ipv6':
      pnht = pathAfiSafiNhType.ipv6MplsVpnNexthopsEncapMpls
   else:
      raise Exception( 'Next-hop resolution ribs not supported in af %s' % af )

   assert pnht
   return pnht

def nhResRibsConfigHelper( af, args=None, no=None, vrfName=DEFAULT_VRF ):
   config = configForVrf( vrfName )
   pnht = nhTypeFromAfiSafi( af, args )
   if no:
      config.nhResolutionRibProfileConfig[ pnht ] = \
         config.nhResolutionRibProfileConfigDefault[ pnht ]
      if pnht in config.nhResolutionRibProfileRouteMapConfig:
         del config.nhResolutionRibProfileRouteMapConfig[ pnht ]
      if pnht in config.nhResolutionRibUdpProfileTunnelConfig:
         del config.nhResolutionRibUdpProfileTunnelConfig[ pnht ]
      return

   if 'ROUTE_MAP' in args:
      config.nhResolutionRibProfileRouteMapConfig[ pnht ] = args[ 'NAME' ]
      config.nhResolutionRibProfileConfig[ pnht ] = \
         config.nhResolutionRibProfileConfigDefault[ pnht ]
      return

   if pnht in config.nhResolutionRibProfileRouteMapConfig:
      del config.nhResolutionRibProfileRouteMapConfig[ pnht ]

   # The method nhResRibsConfigHelper is called from a variety of contexts.
   # Ensure only 1 "next-hop resolution ribs" CLI factory is used in its
   # syntax.
   allExprNames = [ name for name in ResolutionRibsExprFactory.validExprNames
                    if name in args ]
   assert len( allExprNames ) == 1
   name = allExprNames[ 0 ]

   if 'system-default' in args:
      # Only used with 'udp-tunnel' option
      args[ name ].useDefault = True
   config.nhResolutionRibProfileConfig[ pnht ] = args[ name ]

def setNhResRibsUdpTunnelConfig( af, args=None, no=None, vrfName=DEFAULT_VRF ):
   if af not in [ 'ipv4', 'ipv6' ]:
      return
   config = configForVrf( vrfName )
   pnht = nhTypeFromAfiSafi( af, args )
   if 'udp-tunnel' in args:
      config.nhResolutionRibUdpProfileTunnelConfig[ pnht ] = True
   elif pnht in config.nhResolutionRibUdpProfileTunnelConfig:
      del config.nhResolutionRibUdpProfileTunnelConfig[ pnht ]

# This function selects the range of the TCP buffer tx/rx buffer range.
# If the system total memory is below 8GB the range is 4KB to 64MB i
# otherwise the range is 4KB to 128MB. This function looks at the
# /proc/meminfo file system to find out the total system memory.
def bufferRangeFn( mode=None, context=None ):
   if hasattr( bufferRangeFn, "smallSystem" ):
      if bufferRangeFn.smallSystem:
         return ( 4, 64 * 1024 )
      else:
         return ( 4, 128 * 1024 )

   # If the total system memory is below 8GB the buffer range is 4KB-64MB
   # otherwise the range is 4KB-128MB
   smallMemSizeLimit = 8 * 1024 * 1024
   smallSystem = True
   # Look for system memory size
   output = Tac.run( [ 'cat', '/proc/meminfo' ],
                     stdout=Tac.CAPTURE, stderr=Tac.DISCARD, asRoot=True,
                     ignoreReturnCode=True )
   if output is None:
      # error getting the system memory assume small System
      return ( 4, 64 * 1024 )
   # Find the Total memory
   match = re.search( r"\s*MemTotal:\s+(\d+)\s*", output )
   if match is None:
      # error getting the system memory assume small System
      return ( 4, 64 * 1024 )
   g = match.groups()
   memSize = int( g[ 0 ] )
   if memSize > smallMemSizeLimit:
      smallSystem = False
   if smallSystem is False:
      bufferRangeFn.smallSystem = False
      return ( 4, 128 * 1024 )
   else:
      bufferRangeFn.smallSystem = True
      return ( 4, 64 * 1024 )

def removeNexthopSelfReceivedVpnRoutesConfig( config, vpnAf, vrfName=None ):
   ''' Removes 'no neighbor default encapsulation mpls next-hop-self
   received-vpnv[4|6]-routes'
   '''
   if vpnAf in [ 'vpn-ipv4', 'vpn-ipv6' ]:
      afiSafi = vpnAfTypeMapInv[ vpnAf ]
      del config.nexthopSelfLocalLabelAlloc[ afiSafi ]
   else:
      raise ValueError()

def resetBgpAfModeConfig( config, addrFamily, vrfName=None ):
   # This reset logic for af-specific attributes relies on a specific format of said
   # attributes in the config entity which causes them to be populated into
   # bgpConfigAttrsAfMap.
   # Unfortunately nhResolutionRibProfileConfig was not able to follow that
   # format and consequently does not get populated into bgpConfigAttrsAfMap
   # Therefore we'll handle this single attribute here manually
   if addrFamily == 'ipv6':
      nhResRibsConfigHelper( addrFamily, args={}, no=True, vrfName=vrfName )
      nhResRibsConfigHelper( addrFamily, args={ '6pe': '6pe' }, no=True,
         vrfName=vrfName )
   elif addrFamily in ( 'ipv4', 'ipv4 labeled-unicast', 'ipv6 labeled-unicast' ):
      nhResRibsConfigHelper( addrFamily, args=None, no=True, vrfName=vrfName )
   elif addrFamily == 'evpn':
      nhResRibsConfigHelper( addrFamily, args={ 'mpls': 'mpls' }, no=True,
         vrfName=vrfName )
      nhResRibsConfigHelper( addrFamily, args={ 'vxlan': 'vxlan' }, no=True,
         vrfName=vrfName )
   # pylint: disable-next=consider-using-in
   elif ( addrFamily == 'vpn-ipv4' or addrFamily == 'vpn-ipv6' ):
      nhResRibsConfigHelper( addrFamily, args=None, no=True, vrfName=vrfName )
      # nexthopSelfLocalLabelAlloc attribute doesn't follow bgpConfigAttrsAfMap
      # and hence needs to be removed explicitly.
      removeNexthopSelfReceivedVpnRoutesConfig( config, addrFamily, vrfName=vrfName )

   for key, afMap in six.iteritems( bgpConfigAttrsAfMap ):
      attrName = afMap.get( addrFamily )
      if attrName:
         attr = getattr( config, attrName )
         if hasattr( attr, 'clear' ):
            attr.clear()
         elif attr in Tac.Type( 'Ip::TristateBool' ).attributes:
            setattr( config, attrName, 'isInvalid' )
         else:
            if key == 'bgpRedistInternal':
               # The redistribute-internal config is a special case because we
               # do not reset the attribute back to the default/initial value.
               if attr != RedistInternalStateEnum.notConfigured:
                  setattr( config, attrName,
                           RedistInternalStateEnum.enableRedistInt )
            else:
               defVal = getattr( config, attrName + 'Default', None )
               if defVal is not None:
                  setattr( config, attrName, defVal )

   if vrfName is not None and vrfName != DEFAULT_VRF:
      config = configForVrf( vrfName )
   for peerConfig in config.neighborConfig.values():
      for _, afMap in six.iteritems( peerConfigAttrsAfMap ):
         attrName = afMap.get( addrFamily )
         if attrName:
            attr = getattr( peerConfig, attrName )
            if hasattr( attr, 'clear' ):
               attr.clear()
            else:
               if hasattr( peerConfig, attrName + 'Present' ):
                  setattr( peerConfig, attrName + 'Present', False )
               defVal = getattr( peerConfig, attrName + 'Default' )
               setattr( peerConfig, attrName, defVal )

      if addrFamily == "ipv6":
         peerConfig.sixPePresent = False
         peerConfig.sixPe = peerConfig.sixPeDefault

      if addrFamily in [ 'ipv4', 'ipv6' ]:
         l = list( peerConfig.routeTargetExportFilterDisabledAf )
         for nlriType in l:
            if nlriType.ipVersion == int( addrFamily[ -1 ] ):
               del peerConfig.routeTargetExportFilterDisabledAf[ nlriType ]

#-----------------------------------------------------------------------------
# "[no|default] local-as AS_NUM"
#-----------------------------------------------------------------------------
class SetLocalAsCmd( BgpCmdBaseClass ):
   syntax = "local-as AS_NUM"
   noOrDefaultSyntax = "local-as [ AS_NUM ]"
   data = BgpCmdBaseClass._createSyntaxData( {
         'local-as': bgpTokens.localAs,
         'AS_NUM': bgpTokens.AsNumCliExpr,
   } )
   handler = "RoutingBgpInstanceCliHandler.SetLocalAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetLocalAsCmd_noOrDefaultHandler"

RouterBgpVrfMode.addCommandClass( SetLocalAsCmd )

#-------------------------------------------------------------------------------
# "bgp fec skip in-place update event { route-changes | drain-undrain | all }"
# "{ no|default } bgp fec skip in-place update"
# Note: 'drain-undrain' keyword in above command is hidden and deprecated.
#
# "bgp fec in-place update event drain-undrain [ new-route ]"
# "{ no|default } bgp fec in-place update event drain-undrain [ new-route ]"
#
# "bgp fec in-place update event peer-init"
# "{ no|default } bgp fec in-place update event peer-init"
#
# "bgp fec in-place update timeout { <iar-timeout> | disabled }"
# "{ no|default } bgp fec in-place update timeout"
#
# commands, in "router-bgp" mode.
#-------------------------------------------------------------------------------

class SetFecSkipIarCmd( CliCommandClass ):
   syntax = 'bgp fec skip in-place update event IAR_EVENT'
   noOrDefaultSyntax = 'bgp fec skip in-place update ...'
   data = {
      'bgp': bgpNode,
      'fec': bgpTokens.fec,
      'skip': bgpTokens.skip,
      'in-place': bgpTokens.inPlace,
      'update': bgpTokens.iarUpdate,
      'event': bgpTokens.iarEvent,
      'IAR_EVENT': bgpTokens.IarEventTypeExpression,
   }
   handler = "RoutingBgpInstanceCliHandler.SetFecSkipIarCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetFecSkipIarCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.SetFecSkipIarCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetFecSkipIarCmd )

class SetFecDrainUndrainIarCmd( CliCommandClass ):
   syntax = 'bgp fec in-place update event drain-undrain [ new-route ]'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'fec': bgpTokens.fec,
      'in-place': bgpTokens.inPlace,
      'update': bgpTokens.iarUpdate,
      'event': bgpTokens.iarEvent,
      'drain-undrain': bgpTokens.drainUndrain,
      'new-route': bgpTokens.newRoute,
   }
   handler = "RoutingBgpInstanceCliHandler.SetFecDrainUndrainIarCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetFecDrainUndrainIarCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetFecDrainUndrainIarCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetFecDrainUndrainIarCmd )

class SetFecPeerInitIarCmd( CliCommandClass ):
   syntax = 'bgp fec in-place update event peer-init'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'fec': bgpTokens.fec,
      'in-place': bgpTokens.inPlace,
      'update': bgpTokens.iarUpdate,
      'event': bgpTokens.iarEvent,
      'peer-init': bgpTokens.peerInit,
   }
   handler = "RoutingBgpInstanceCliHandler.SetFecPeerInitIarCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetFecPeerInitIarCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetFecPeerInitIarCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetFecPeerInitIarCmd )

class SetIarTimeoutCmd( BgpCmdBaseClass ):
   syntax = 'bgp fec in-place update timeout ( TIMEOUT | disabled )'
   noOrDefaultSyntax = 'bgp fec in-place update timeout ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'bgp': bgpNode,
         'fec': bgpTokens.fec,
         'in-place': bgpTokens.inPlace,
         'update': bgpTokens.iarUpdate,
         'timeout': bgpTokens.iarTimeout,
         'TIMEOUT': bgpTokens.numSeconds3600RangeMatcher,
   } )
   handler = "RoutingBgpInstanceCliHandler.SetIarTimeoutCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetIarTimeoutCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetIarTimeoutCmd )

#-------------------------------------------------------------------------------
# [no|default] route-target export { { evpn { ipv4 | ipv6 } } |
#                                      vpn-ipv4 | vpn-ipv6 } filter disabled
#-------------------------------------------------------------------------------

class RouteTargetExportFilterDisabledCmd( BgpCmdBaseClass ):
   syntax = 'route-target export VPN_NLRI_TYPE filter disabled'
   noOrDefaultSyntax = 'route-target export VPN_NLRI_TYPE filter ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'route-target': bgpTokens.routeTarget,
      'export': bgpTokens.export,
      'VPN_NLRI_TYPE': bgpTokens.VpnNlriTypeExpr,
      'filter': 'Set route-target export filtering behavior',
   } )
   handler = \
      "RoutingBgpInstanceCliHandler.RouteTargetExportFilterDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouteTargetExportFilterDisabledCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RouteTargetExportFilterDisabledCmd )

#-------------------------------------------------------------------------------
# [ (no|default) ] neighbor default send-community
# [ ( [ extended ] [ large ] [ LBW ] [ standard ] )  | disabled ]
#     under 'bgp' mode'
#-------------------------------------------------------------------------------

class SetNeighborDefaultSendCommunity( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor default send-community '
              '[ ( [ standard ] [ extended ] [ large ] ) | disabled ]' )
   noOrDefaultSyntax = ( 'neighbor default send-community '
                         '[ standard ] [ extended ] [ large ] ...' )
   data = BgpNEDCmdBaseClass._createSyntaxData( { 'neighbor': bgpTokens.neighbor,
            'default': bgpTokens.neighborDefaultGlobal,
            'send-community': bgpTokens.sendCommunity,
            'standard': bgpTokens.standardCommunity,
            'extended': bgpTokens.extendedCommunity,
            'large': bgpTokens.largeCommunity,
            } )
   handler = "RoutingBgpInstanceCliHandler.SetNeighborDefaultSendCommunity_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetNeighborDefaultSendCommunity_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborDefaultSendCommunity )

# -------------------------------------------------------------------------------
# [ no|default ] neighbor default index routes peer disabled,
# in "router-bgp" mode.
# -------------------------------------------------------------------------------

class NeighborDefaultIndexRoutesPeerDisabledCmd( CliCommandClass ):
   syntax = 'neighbor default index routes peer disabled'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': bgpTokens.neighbor,
      'default': bgpTokens.neighborDefaultGlobal,
      'index': Node( bgpTokens.index, guard=hwSupportedGuard ),
      'routes': bgpTokens.routes,
      'peer': 'Advertised to and received from peer',
      'disabled': bgpTokens.disabled,
   }
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "NeighborDefaultIndexRoutesPeerDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "NeighborDefaultIndexRoutesPeerDisabledCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( NeighborDefaultIndexRoutesPeerDisabledCmd )

#-------------------------------------------------------------------------------
# "[no|default] router bgp <as>" command, in "config" mode.
#-------------------------------------------------------------------------------

class RouterBgpCmd( CliCommandClass ):
   syntax = 'router bgp ( ASN | CURR_ASN )'
   noOrDefaultSyntax = 'router bgp ...'
   data = {
         'router': routerMatcher,
         'bgp': bgpNode,
         'ASN': bgpTokens.AsNumCliExpr,
         'CURR_ASN': DynamicKeywordMatcher( getCurrentBgpAs ),
   }
   handler = "RoutingBgpInstanceCliHandler.RouterBgpCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      currAsn = args.get( 'CURR_ASN' )
      if currAsn:
         args[ 'CURR_ASN' ] = asnStrToNum( currAsn )

      # The expression used for ASN populates 'AS_NUM'.
      args[ 'AS_NUMBER' ] = args.get( 'AS_NUM' ) or args.get( 'CURR_ASN' )

BasicCli.GlobalConfigMode.addCommandClass( RouterBgpCmd )

#-------------------------------------------------------------------------------
# "[no|default] service routing configuration bgp no-equals-default" command,
# in "config" mode.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] command clear all disabled" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class ClearAllDisabledCmd( CliCommandClass ):
   syntax = 'command clear all disabled'
   noOrDefaultSyntax = syntax
   data = {
      'command': 'Configure command options',
      'clear': 'BGP clear command',
      'all': 'Clear all BGP sessions',
      'disabled': 'Disallow clearing all BGP sessions'
   }
   handler = "RoutingBgpInstanceCliHandler.ClearAllDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ClearAllDisabledCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( ClearAllDisabledCmd )

#-------------------------------------------------------------------------------
# "[no|default] update wait-install" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
def fibAckSupportedGuard( mode, token ):
   if routingHardwareStatus.fibAckSupported:
      return None
   return guardNotThisPlatform

class SetFibSyncBatchSizeCmd( CliCommandClass ):
   syntax = 'update wait-install [ batch-size BATCH_SIZE ]'
   noOrDefaultSyntax = 'update wait-install ...'
   data = { 'update': bgpTokens.update,
            'wait-install': Node( bgpTokens.waitInstall,
                                  guard=fibAckSupportedGuard ),
            'batch-size': bgpTokens.batchSize,
            'BATCH_SIZE':  IntegerMatcher( 0, 10000, helpdesc='batch size' ),
   }
   handler = "RoutingBgpInstanceCliHandler.SetFibSyncBatchSizeCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetFibSyncBatchSizeCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetFibSyncBatchSizeCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp labeled-unicast rib { ip | tunnel }" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpLuRibCmd( CliCommandClass ):
   _ipSyntax = '( ip [ IP_RM IP_RM_NAME ] )'
   _tunnelSyntax = '( tunnel [ TUNNEL_RM TUNNEL_RM_NAME ] )'
   syntax = 'bgp labeled-unicast rib ( {ip} [ {tunnel} ] ) | ' \
            '( {tunnel} [ {ip} ] )'.format( ip=_ipSyntax, tunnel=_tunnelSyntax )
   noOrDefaultSyntax = 'bgp labeled-unicast rib'
   data = {
         'bgp': bgpTokens.bgp,
         'labeled-unicast': bgpTokens.lu,
         'rib': bgpTokens.rib,
         'ip': bgpTokens.ip,
         'tunnel': bgpTokens.tunnel,
         'IP_RM': bgpTokens.ribRouteMap,
         'TUNNEL_RM': bgpTokens.ribRouteMap,
         'IP_RM_NAME': mapNameMatcher,
         'TUNNEL_RM_NAME': mapNameMatcher
         }
   handler = "RoutingBgpInstanceCliHandler.BgpLuRibCmd_handler"
   noOrDefaultHandler = "RoutingBgpInstanceCliHandler.BgpLuRibCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      enumVal = 'tunnel'
      if 'ip' in args and 'tunnel' in args:
         enumVal = 'ipAndTunnel'
      elif 'ip' in args:
         enumVal = 'ip'
      args[ 'rib' ] = getattr( Tac.Type( 'Routing::Bgp::BgpLuRibTypes' ), enumVal )

RouterBgpSharedModelet.addCommandClass( BgpLuRibCmd )

#-------------------------------------------------------------------------------
# "[no|default] shutdown [reason MESSAGE]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class ShutdownCmd( CliCommandClass ):
   syntax = 'shutdown [ reason MESSAGE ]'
   noOrDefaultSyntax = 'shutdown ...'
   data = { 'shutdown': 'Shut down BGP',
            'reason': bgpTokens.reason,
            'MESSAGE': bgpTokens.messageMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.ShutdownCmd_handler"
   noOrDefaultHandler = "RoutingBgpInstanceCliHandler.ShutdownCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( ShutdownCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp log-neighbor-changes" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class LogNeighborChangesCmd( CliCommandClass ):
   syntax = 'bgp log-neighbor-changes'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpTokens.bgp,
      'log-neighbor-changes': bgpTokens.logNeighborChanges,
   }
   handler = "RoutingBgpInstanceCliHandler.LogNeighborChangesCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.LogNeighborChangesCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.LogNeighborChangesCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( LogNeighborChangesCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp client-to-client reflection" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class ClientToClientReflectionCmd( CliCommandClass ):
   syntax = 'bgp client-to-client reflection'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'client-to-client': 'client to client configuration',
      'reflection': 'Enable client to client route reflection',
   }
   handler = "RoutingBgpInstanceCliHandler.ClientToClientReflectionCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.ClientToClientReflectionCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.ClientToClientReflectionCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( ClientToClientReflectionCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp cluster-id <cluster-id> " command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetClusterIdCmd( CliCommandClass ):
   syntax = 'bgp cluster-id CLUSTER_ID'
   noOrDefaultSyntax = 'bgp cluster-id ...'
   data = {
      'bgp': bgpNode,
      'cluster-id': bgpTokens.cluster,
      'CLUSTER_ID': IpAddrMatcher.ipAddrMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.SetClusterIdCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetClusterIdCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetClusterIdCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp aggregate-route community inheritance loose" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpAggregateCommunityInheritanceLooseCmd( CliCommandClass ):
   syntax = 'bgp aggregate-route community inheritance loose'
   noOrDefaultSyntax = 'bgp aggregate-route ...'
   data = {
      'bgp': bgpNode,
      'aggregate-route': 'Aggregate route',
      'community': 'Aggregate route community',
      'inheritance': 'Aggregate route community generation from contributors',
      'loose': 'Ignore ATOMIC_AGGREGATE in contributors for community inheritance',
   }
   handler = \
      "RoutingBgpInstanceCliHandler.BgpAggregateCommunityInheritanceLooseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "BgpAggregateCommunityInheritanceLooseCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( BgpAggregateCommunityInheritanceLooseCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp allowas-in [count]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetAllowAsCmd( CliCommandClass ):
   syntax = 'bgp allowas-in [ COUNT ]'
   noOrDefaultSyntax = 'bgp allowas-in ...'
   data = {
      'bgp': bgpNode,
      'allowas-in': bgpTokens.allowAsIn,
      'COUNT': bgpTokens.allowAsRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.SetAllowAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetAllowAsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetAllowAsCmd )

#---------------------------------------------------------------------------------
# [no|default] maintenance receiver route-map default
#    ( ( local-preference LOCAL_PREF ) | disabled )
#---------------------------------------------------------------------------------
class MaintenanceReceiverRouteMapCmd( CliCommandClass ):
   syntax = ( "maintenance receiver route-map default"
              "( ( local-preference LOCAL_PREF ) | disabled )" )
   noOrDefaultSyntax = "maintenance receiver route-map default  ..."

   data = {
         'maintenance': maintenanceKwMatcher,
         'receiver': 'Configure maintenance receiver route-map attributes',
         'route-map': 'Configure maintenance receiver route-map attributes',
         'default': 'default route-map',
         'local-preference': 'Set local-preference for default maintenance'
                             ' receiver route-map',
         'LOCAL_PREF': IntegerMatcher( 0, U32_MAX_VALUE,
                                       helpdesc='BGP local preference' ),
         'disabled': 'Disable maintenance receiver route-map'
         }
   handler = "RoutingBgpInstanceCliHandler.MaintenanceReceiverRouteMapCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "MaintenanceReceiverRouteMapCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( MaintenanceReceiverRouteMapCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp redistribute-internal " command,
# in "router-bgp", "router-bgp-af" mode
#--------------------------------------------------------------------------------
class BgpRedistInternalCmd( CliCommandClass ):
   syntax = 'bgp redistribute-internal'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'redistribute-internal': bgpTokens.bgpRedistInternal,
   }
   handler = "RoutingBgpInstanceCliHandler.BgpRedistInternalCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.BgpRedistInternalCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.BgpRedistInternalCmd_defaultHandler"

for m in [ RouterBgpSharedModelet, RouterBgpAfIpUniSharedModelet ]:
   m.addCommandClass( BgpRedistInternalCmd )

#-------------------------------------------------------------------------------
# "[no|default] timers bgp <keepalive> <hold>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpSetTimerCmd( CliCommandClass ):
   # pylint: disable=no-method-argument
   # pylint: disable=too-many-function-args
   syntax = '''timers bgp (
      ( KEEPALIVE ( HOLD_TIME | 0 ) [ min-hold-time MIN_HOLD_TIME ]
                           [ send-failure hold-time SEND_FAILURE_HOLD_TIME ] ) |
      ( min-hold-time MIN_HOLD_TIME
                           [ send-failure hold-time SEND_FAILURE_HOLD_TIME ] ) |
      ( send-failure hold-time SEND_FAILURE_HOLD_TIME )
   )'''

   noOrDefaultSyntax = 'timers bgp ...'
   data = {
      'timers': bgpTokens.timers,
      'bgp': bgpTokens.bgpTimers,
      'KEEPALIVE': bgpTokens.keepaliveTimeRangeMatcher,
      'HOLD_TIME': bgpTokens.holdTimeRangeMatcher,
      '0': bgpTokens.holdForever,
      'min-hold-time': bgpTokens.bgpMinHoldTime,
      'MIN_HOLD_TIME': bgpTokens.holdTimeRangeMatcher,
      'send-failure': bgpTokens.bgpSendFailure,
      'hold-time': bgpTokens.bgpSendFailureHoldTime,
      'SEND_FAILURE_HOLD_TIME': bgpTokens.sendFailureHoldTimeRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.BgpSetTimerCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.BgpSetTimerCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.BgpSetTimerCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( BgpSetTimerCmd )

#---------------------------------------------------------------------------------
# "[no|default] graceful-restart [ ( restart-time | stalepath-time ) SECONDS ]
# in "router-bgp" and "router-bgp-af"  mode.  The 'restart-time' and 'stalepath-time'
# keywords are only applicable in "router-bgp" mode.
#---------------------------------------------------------------------------------
class SetGracefulRestartCmd( CliCommandClass ):
   syntax = ( 'graceful-restart' )
   noOrDefaultSyntax = syntax
   data = {
         'graceful-restart': bgpTokens.gracefulRestart,
   }
   handler = "RoutingBgpInstanceCliHandler.SetGracefulRestartCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetGracefulRestartCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetGracefulRestartCmd_defaultHandler"

for m in [ RouterBgpAfSharedModelet, RouterBgpSharedModelet,
           RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetGracefulRestartCmd )
if BgpCommonToggleLib.toggleHandlePfLossEnabled():
   RouterBgpBaseAfDpsMode.addCommandClass( SetGracefulRestartCmd )

class SetGracefulRestartTimesCmd( CliCommandClass ):
   syntax = ( 'graceful-restart ( restart-time | stalepath-time ) SECONDS' )
   noOrDefaultSyntax = 'graceful-restart ( restart-time | stalepath-time ) ...'
   data = {
      'graceful-restart': bgpTokens.gracefulRestart,
      'restart-time': bgpTokens.restartTime,
      'stalepath-time': bgpTokens.stalepathTime,
      'SECONDS': bgpTokens.secondsRangeMatcher,
      }
   handler = "RoutingBgpInstanceCliHandler.SetGracefulRestartTimesCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetGracefulRestartTimesCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetGracefulRestartTimesCmd_defaultHandler"
   # The "no graceful-restart [ restart-time | stalepath-time ]" command operates
   # differently from the "default..." command for the non-default VRF.  The
   # 'default' option is the same as unconfigured, which will cause the non-default
   # vrf to use the value configured in the default vrf.  The 'no' option explicitly
   # sets parameters to the system defaults.

RouterBgpSharedModelet.addCommandClass( SetGracefulRestartTimesCmd )

#-------------------------------------------------------------------------------
# "[no|default] graceful-restart-helper [ restart-time < seconds > ] [ long-lived ]"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetGracefulRestartHelperCmd( CliCommandClass ):
   syntax = '''
      graceful-restart-helper
      [ long-lived
      | ( EXT_RESTART_TIME
          [ long-lived
          | ( stale-route route-map MAP_NAME [ session-failure GR_OPTIONAL ] ) ]
      ) ]'''
   noOrDefaultSyntax = 'graceful-restart-helper ...'
   data = {
      'graceful-restart-helper': bgpTokens.grHelper,
      'EXT_RESTART_TIME': bgpTokens.GrHelperRestartTimeExp,
      'long-lived': bgpTokens.llgrHelper,
      'stale-route': bgpTokens.grHelperStaleRoute,
      'route-map': Node( bgpTokens.routeMap,
                         maxMatches=1 ),
      'MAP_NAME': Node( mapNameMatcher, maxMatches=1 ),
      'session-failure': bgpTokens.sessionFailure,
      'GR_OPTIONAL': bgpTokens.GrHelperRmGrOptionalExp
   }
   handler = "RoutingBgpInstanceCliHandler.SetGracefulRestartHelperCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetGracefulRestartHelperCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetGracefulRestartHelperCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetGracefulRestartHelperCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp restart hitless [ disabled ]" command,
#  in "router-bgp" mode.
# -------------------------------------------------------------------------------
class BgpHitlessRestartDisabledCmd( CliCommandClass ):
   syntax = 'bgp restart hitless [ disabled ]'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'restart': 'BGP process restart',
      'hitless':
         'Attempt a hitless restart of the BGP process, including during ASU or SSO',
      'disabled': 'Disable hitless agent restart for multi agent model'
   }
   handler = "RoutingBgpInstanceCliHandler.BgpHitlessRestartDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.BgpHitlessRestartDisabledCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( BgpHitlessRestartDisabledCmd )

# -------------------------------------------------------------------------------
# "[no|default] bgp enforce-first-as" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpEnforceAsFirstCmd( CliCommandClass ):
   syntax = 'bgp enforce-first-as'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'enforce-first-as': bgpTokens.enforceFirstAs,
   }
   handler = "RoutingBgpInstanceCliHandler.BgpEnforceAsFirstCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.BgpEnforceAsFirstCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.BgpEnforceAsFirstCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( BgpEnforceAsFirstCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath as-path multipath-relax [match <n>]" command
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetAsPathMultiPathRelaxCmd( CliCommandClass ):
   syntax = ( 'bgp bestpath as-path multipath-relax '
              '[ match ASPATH_LEN ]' )
   noOrDefaultSyntax = 'bgp bestpath as-path multipath-relax ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'as-path': bgpTokens.asPath,
         'multipath-relax': bgpTokens.multiPathRelax,
         'match': bgpTokens.asMatch,
         'ASPATH_LEN': bgpTokens.asPathLenMatcher
   }
   handler = "RoutingBgpInstanceCliHandler.SetAsPathMultiPathRelaxCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetAsPathMultiPathRelaxCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetAsPathMultiPathRelaxCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetAsPathMultiPathRelaxCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath as-path ignore" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class IgnoreAsPathCmd( CliCommandClass ):
   syntax = 'bgp bestpath as-path ignore'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'as-path': bgpTokens.asPath,
         'ignore': bgpTokens.ignore,
   }
   handler = "RoutingBgpInstanceCliHandler.IgnoreAsPathCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.IgnoreAsPathCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.IgnoreAsPathCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( IgnoreAsPathCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath d-path" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class BgpBestPathDomainPath( CliCommandClass ):
   syntax = 'bgp bestpath d-path'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'd-path': bgpTokens.domainPath
   }
   handler = "RoutingBgpInstanceCliHandler.BgpBestPathDomainPath_handler"
   noHandler = "RoutingBgpInstanceCliHandler.BgpBestPathDomainPath_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.BgpBestPathDomainPath_defaultHandler"

RouterBgpBaseMode.addCommandClass( BgpBestPathDomainPath )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath skip next-hop igp-cost" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class SetSkipNhIgpCostCmd( CliCommandClass ):
   syntax = 'bgp bestpath skip next-hop igp-cost'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'skip': bgpTokens.skipMatcher,
         'next-hop': bgpTokens.skipNextHop,
         'igp-cost': bgpTokens.skipNhIgpCost,
   }
   handler = "RoutingBgpInstanceCliHandler.SetSkipNhIgpCostCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetSkipNhIgpCostCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.SetSkipNhIgpCostCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetSkipNhIgpCostCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath origin-as validity" command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class BgpBestPathOriginAsValidityCmd( BgpCmdBaseClass ):
   syntax = 'bgp bestpath origin-as validity [ disabled ]'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
      'bgp': bgpNode,
      'bestpath': bgpTokens.bestPath,
      'origin-as': bgpTokens.originAsBestPath,
      'validity': bgpTokens.originAsValidityBestPath
   } )
   handler = "RoutingBgpInstanceCliHandler.BgpBestPathOriginAsValidityCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "BgpBestPathOriginAsValidityCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( BgpBestPathOriginAsValidityCmd )

#--------------------------------------------------------------------------
# "bgp additional-paths install" command, in "router-bgp" mode
#--------------------------------------------------------------------------
class SetAddpathInstallCmd( CliCommandClass ):
   syntax = 'bgp additional-paths install [ ecmp-primary ]'
   noOrDefaultSyntax = 'bgp additional-paths install ...'
   data = { 'bgp': bgpNode,
            'additional-paths': bgpTokens.addPath,
            'install': bgpTokens.addpathInstall,
            'ecmp-primary': bgpTokens.addpathInstallEcmpPrimary,
   }
   handler = "RoutingBgpInstanceCliHandler.SetAddpathInstallCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetAddpathInstallCmd_noOrDefaultHandler"

if BgpCommonToggleLib.toggleBgpLuPicEnabled():
   for m in [ RouterBgpSharedModelet, RouterBgpAfIpUniSharedModelet,
              RouterBgpAfLabelSharedModelet ]:
      m.addCommandClass( SetAddpathInstallCmd )
else:
   for m in [ RouterBgpSharedModelet, RouterBgpAfIpUniSharedModelet ]:
      m.addCommandClass( SetAddpathInstallCmd )


#--------------------------------------------------------------------------
# "bgp additional-paths receive" command, in "router-bgp" mode
#--------------------------------------------------------------------------
class SetAddPathReceiveCmd( CliCommandClass ):
   syntax = 'bgp additional-paths receive'
   noOrDefaultSyntax = syntax
   data = { 'bgp': bgpNode,
            'additional-paths': bgpTokens.addPath,
            'receive': bgpTokens.addpathReceive,
   }
   handler = "RoutingBgpInstanceCliHandler.SetAddPathReceiveCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetAddPathReceiveCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetAddPathReceiveCmd_defaultHandler"

for m in [ RouterBgpSharedModelet,
           RouterBgpAfSharedModelet,
           RouterBgpAfLabelSharedModelet,
           RouterBgpBaseAfDpsMode ]:
   m.addCommandClass( SetAddPathReceiveCmd )

#--------------------------------------------------------------------------
# "bgp additional-paths send any" command in "router-bgp" mode.
#--------------------------------------------------------------------------
class SetAddpathSendCmd( CliCommandClass ):
   syntax = ( 'bgp additional-paths send ( any | backup | '
              '( [ ecmp ] [ limit LIMIT ] ) )' )
   # For backward compatibility we will accept 'any' in no/default
   noOrDefaultSyntax = 'bgp additional-paths send [ ANY_HIDDEN | ... ]'
   data = { 'bgp': bgpNode,
            'additional-paths': bgpTokens.addPath,
            'send': bgpTokens.addpathSend,
            'any': bgpTokens.appAny,
            'backup': bgpTokens.appBackup,
            'ecmp': bgpTokens.appEcmp,
            'limit': bgpTokens.appLimit,
            'LIMIT': bgpTokens.appLimitRangeMatcher,
            'ANY_HIDDEN': hiddenKeyword( 'any' ),
   }
   handler = "RoutingBgpInstanceCliHandler.SetAddpathSendCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetAddpathSendCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.SetAddpathSendCmd_defaultHandler"

for m in [ RouterBgpSharedModelet,
           RouterBgpAfIpUniSharedModelet,
           RouterBgpAfLabelSharedModelet,
           RouterBgpBaseAfEvpnMode,
           RouterBgpBaseAfDpsMode ]:
   m.addCommandClass( SetAddpathSendCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp route install-map <rmap-name>" command
# under the "router-bgp" and under the address-family modes
#-------------------------------------------------------------------------------
class BgpRouteInstallMapCmd( BgpCmdBaseClass ):
   syntax = 'bgp route install-map MAP_NAME'
   noOrDefaultSyntax = 'bgp route install-map ...'
   data = { 'bgp': bgpNode,
            'route': bgpTokens.route,
            'install-map': bgpTokens.installMap,
            'MAP_NAME': mapNameMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.BgpRouteInstallMapCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.BgpRouteInstallMapCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( BgpRouteInstallMapCmd )
RouterBgpAfIpUniSharedModelet.addCommandClass( BgpRouteInstallMapCmd )

#-------------------------------------------------------------------------------
# "[no|default] network ( PREFIX | ( ADDR mask MASK ) )" command, in
# "router-bgp" mode.
#
# "network ADDR route-map MAP " command, in "router-bgp" mode
#
# The no version of this command is handled by no network command
# as it specifies trailing garbage
#
# It's possible to specify network statements both inside and outside
# the address-family block (although when specified inside, they must
# be networks of the corresponding IP version). Supporting networks
# outside the address-family block is done for backward compatibility
# reasons.
#--------------------------------------------------------------------------------
class SetNetworkCmdBase( BgpCmdBaseClass ):
   noOrDefaultSyntax = 'network ( PREFIX_OR_ADDRMASK | PREFIX_OR_ADDRMASK6 ) ...'
   data = {
         'network': bgpTokens.network,
         'PREFIX_OR_ADDRMASK': IpAddrMatcher.ipPrefixExpr(
                                    'Network address',
                                    'Network mask',
                                    'Prefix',
                                    overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                                    maskKeyword=True
                                ),
         'PREFIX_OR_ADDRMASK6': Ip6PrefixMatcher(
                                    'IPv6 address prefix',
                                    overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
                                ),
         'route-map': bgpTokens.routeMap,
         'MAP_NAME': mapNameMatcher,
         'POLICY': bgpTokens.RouteMapOrRcfFunction,
   }
   handler = "RoutingBgpInstanceCliHandler.SetNetworkCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetNetworkCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'prefix' ] = args.get(
            'PREFIX_OR_ADDRMASK',
            args.get( 'PREFIX_OR_ADDRMASK6' ) )

class SetNetworkCmdRouteMapOnly( SetNetworkCmdBase, BgpCmdBaseClass ):
   syntax = ( 'network ( PREFIX_OR_ADDRMASK | PREFIX_OR_ADDRMASK6 ) '
              '[ route-map MAP_NAME ]' )
   data = SetNetworkCmdBase.data.copy()

RouterBgpAfIpMcastSharedModelet.addCommandClass( SetNetworkCmdRouteMapOnly )

class SetNetworkCmdRouteMapAndRcf( SetNetworkCmdBase, BgpCmdBaseClass ):
   syntax = 'network ( PREFIX_OR_ADDRMASK | PREFIX_OR_ADDRMASK6 ) '
   if RcfLibToggleLib.toggleRcfNetworkCommandEnabled():
      syntax += "[ POLICY ]"
   else:
      syntax += "[ route-map MAP_NAME ]"
   data = SetNetworkCmdBase.data.copy()

for m in [ RouterBgpSharedModelet,
           RouterBgpAfIpUniSharedModelet ]:
   m.addCommandClass( SetNetworkCmdRouteMapAndRcf )

networkLUCliHandler = SetNetworkCmdRouteMapOnly
if RcfLibToggleLib.toggleRcfNetworkCommandLUEnabled():
   networkLUCliHandler = SetNetworkCmdRouteMapAndRcf
RouterBgpAfLabelSharedModelet.addCommandClass( networkLUCliHandler )

#-------------------------------------------------------------------------------
# "[no|default] bgp aspath-cmp-include-nexthop" command, in
# "router-bgp" mode. If enabled, next-hop would be included in as-path
# comparison. This knob allows us to go back to old behaviour
#-------------------------------------------------------------------------------
class SetInstanceAspathCmpIncNhCmd( CliCommandClass ):
   syntax = 'bgp aspath-cmp-include-nexthop'
   noOrDefaultSyntax = 'bgp aspath-cmp-include-nexthop ...'
   data = {
         'bgp': bgpNode,
         'aspath-cmp-include-nexthop': bgpTokens.aspathCmpIncludeNh,
   }
   hidden = True
   handler = "RoutingBgpInstanceCliHandler.SetInstanceAspathCmpIncNhCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetInstanceAspathCmpIncNhCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceAspathCmpIncNhCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceAspathCmpIncNhCmd )

#-------------------------------------------------------------------------------
# 'bgp bestpath tie-break ( cluster-list-length | originator-id | router-id | age )'
#-------------------------------------------------------------------------------
class SetBestPathTieBreakCmd( CliCommandClass ):
   syntax = 'bgp bestpath tie-break ( TIE_BREAK | age )'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'tie-break': bgpTokens.tieBreak,
         'TIE_BREAK': EnumMatcher( {
               'cluster-list-length': (
                     'Tie-break BGP paths based on path cluster list length'
               ),
               'originator-id': 'Tie-break BGP paths based on path originator-id',
               'router-id': 'Tie-break BGP paths based on path router-id',
         } ),
         # The 'age' kw cannot be in an enum matcher as it is hidden.
         'age': bgpTokens.age,
   }
   handler = "RoutingBgpInstanceCliHandler.SetBestPathTieBreakCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetBestPathTieBreakCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetBestPathTieBreakCmd_defaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'age' in args:
         args[ 'TIE_BREAK' ] = 'age'

#-------------------------------------------------------------------------------
# "[no|default] bgp tie-break-on-age" command, in
# "router-bgp" mode. If enabled, We add new ecmp group entries to the tail of
# linked list. This is O(1) operation. So it results in better performance. But
# which path becomes ECMP head is not deterministic
# Note: This command is hidden and is replaced by
# "[no|default] bgp bestpath tie-break age" which is also hidden at the
# moment for implementation detail purposes.
#
#
# "[no|default] bgp tie-break-on-router-id" command, in
# "router-bgp" mode. If enabled, We add tie-break ECMP path entries in an ECMP
# group based on router-id alone. This results in slightly inferior performance
# compared to tie-break-on-age but will result in deterministic ordering of paths
# in the ECMP group
# Note: This command is hidden and is replaced by
# "[no|default] bgp bestpath tie-break router-id"
#
#
# "[no|default] bgp tie-break-on-originator-id" command, in
# "router-bgp" mode. If enabled, We add tie-break ECMP path entries in an ECMP
# group based on originator-id alone.
# Note: This command is hidden and will not be replaced. Its functionality is
# included within "[no|default] bgp bestpath tie-break originator-id". Another
# hidden command will be added that follows the new model but won't be
# advertised.
#
#
# "[no|default] bgp tie-break-on-cluster-list-length" command, in
# "router-bgp" mode. If enabled, We add tie-break ECMP path entries in an ECMP
# group based on cluster-list-length alone.
# Note: This command is hidden and is replaced by
# "[no|default] bgp bestpath tie-break cluster-list-length"
#-------------------------------------------------------------------------------
class SetEcmpTbDeprecatedCmd( CliCommandClass ):
   syntax = 'bgp TIE_BREAK'
   noOrDefaultSyntax = syntax + '...'
   data = {
         'bgp': bgpNode,
         'TIE_BREAK': EnumMatcher( {
               'tie-break-on-age': (
                     'Tie-break BGP paths in a ECMP group based on the '
                     'order of arrival'
               ),
               'tie-break-on-router-id': (
                     'Tie-break BGP paths in a ECMP group'
                     ' based on path router-id'
               ),
               'tie-break-on-originator-id': (
                     'Tie-break BGP paths in a ECMP group'
                     ' based on path originator-id'
               ),
               'tie-break-on-cluster-list-length': (
                     'Tie-break BGP paths in a ECMP group'
                     ' based on path cluster list length'
               ),
         } )
   }
   hidden = True
   handler = "RoutingBgpInstanceCliHandler.SetEcmpTbDeprecatedCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetEcmpTbDeprecatedCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetEcmpTbDeprecatedCmd_defaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'TIE_BREAK' ] = args[ 'TIE_BREAK' ].replace( 'tie-break-on-', '' )

RouterBgpSharedModelet.addCommandClass( SetBestPathTieBreakCmd )
RouterBgpSharedModelet.addCommandClass( SetEcmpTbDeprecatedCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath ecmp-fast" command, in "router-bgp" mode. Which is
# enabled by default. We add new ecmp group entries the tail of the
# list. This is our ranking based on "age" which is an O(1) operation. Best
# performance is achieved when enabled, but ECMP head is not deterministic.
# Disabling it reverts the ECMP best-path selection to deterministic.
#-------------------------------------------------------------------------------
class SetBestPathEcmpFastCmd( CliCommandClass ):
   syntax = 'bgp bestpath ecmp-fast'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'ecmp-fast': bgpTokens.ecmpFast,
   }
   handler = "RoutingBgpInstanceCliHandler.SetBestPathEcmpFastCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetBestPathEcmpFastCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetBestPathEcmpFastCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetBestPathEcmpFastCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp advertise-inactive" command, in
# "router-bgp" mode. If enabled, we consider BRIB winners for advertisements in
# BGP policy code (and not active routes in RIB)
#-------------------------------------------------------------------------------
class SetInstanceAdvertiseInactive( CliCommandClass ):
   syntax = 'bgp advertise-inactive'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'advertise-inactive': bgpTokens.advertiseInactive,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceAdvertiseInactive_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetInstanceAdvertiseInactive_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceAdvertiseInactive_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceAdvertiseInactive )

# Aggregate-Address base class
class SetAggregateAddressBaseCmd( BgpCmdBaseClass ):
   data = {
         'aggregate-address': bgpTokens.aggregate,
         'PREFIX': IpAddrMatcher.ipPrefixExpr(
                       'Network address',
                       'Network mask',
                       'Prefix',
                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ),
         'PREFIX6': Ip6PrefixMatcher(
                        'IPv6 address prefix',
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
                    ),
         'OPTIONS': SetEnumMatcher( {
               bgpTokens.asSet.keyword_: bgpTokens.asSet.helpdesc_,
               bgpTokens.summaryOnly.keyword_: bgpTokens.summaryOnly.helpdesc_,
               bgpTokens.advertiseOnly.keyword_: bgpTokens.advertiseOnly.helpdesc_,
         } ),
         'attribute-map': singleNode( bgpTokens.attrMap ),
         'ATTR_MAP': Node( mapNameMatcher, maxMatches=1 ),
         'attribute': singleNode( bgpTokens.attributeRcf ),
         'rcf': singleNode( bgpTokens.rcfAttrSet ),
         'ATTR_FUNCTION': Node( RcfCliLib.rcfFunctionMatcher, maxMatches=1 ),
         'match-map': singleNode( bgpTokens.matchMap ),
         'MATCH_MAP': Node( mapNameMatcher, maxMatches=1 ),
   }
   handler = "RoutingBgpInstanceCliHandler.SetAggregateAddressBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetAggregateAddressBaseCmd_noOrDefaultHandler"
   if RcfLibToggleLib.toggleRcfAggregateAttributeMapEnabled():
      attrMapSyntax = ( "[ ( attribute-map ATTR_MAP ) | "
                      "( attribute rcf ATTR_FUNCTION ) ]" )
   else:
      attrMapSyntax = "( attribute-map ATTR_MAP ) "

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PREFIX' ] = args.get(
            'PREFIX',
            args.get( 'PREFIX6' )
      )
      options = args.get( 'OPTIONS' )
      if options:
         args[ 'summary-only' ] = 'summary-only' in options
         args[ 'advertise-only' ] = 'advertise-only' in options
         args[ 'as-set' ] = 'as-set' in options

#-------------------------------------------------------------------------------
# "[no|default] aggregate-address (<prefix> | <address> <mask>) [as-set]
#      [summary-only] [attribute-map <mapname>] [match-map <mapname>]
#      [advertise-only]" command,
#      in "router-bgp" mode.
#
# "[no|default] aggregate-address <v6prefix> [as-set]
#      [summary-only] [attribute-map <mapname>] [match-map <mapname>]
#      [advertise-only]" command,
#      in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetAggregateAddressCmd( SetAggregateAddressBaseCmd, BgpCmdBaseClass ):
   syntax = (
         'aggregate-address ( PREFIX | PREFIX6 ) [ { OPTIONS | '
         f'{ SetAggregateAddressBaseCmd.attrMapSyntax } '
         '| ( match-map MATCH_MAP ) } ]'
   )
   noOrDefaultSyntax = 'aggregate-address ( PREFIX | PREFIX6 ) ...'
   data = SetAggregateAddressBaseCmd.data.copy()

RouterBgpSharedModelet.addCommandClass( SetAggregateAddressCmd )

#-------------------------------------------------------------------------------
# "[no|default] aggregate-address (<prefix> | <address> <mask>) [as-set]
#      [summary-only] [attribute-map <mapname>] [match-map <mapname>]
#      [advertise-only]" command,
#      in "router-bgp/address-family ipv4 multicast" mode.
#-------------------------------------------------------------------------------
class SetIPv4MulticastAggregateAddressCmd( SetAggregateAddressBaseCmd,
                                           BgpCmdBaseClass ):
   # restrict to IPv4 addresses only
   syntax = (
         'aggregate-address PREFIX [ { OPTIONS | ( '
         f'{ SetAggregateAddressBaseCmd.attrMapSyntax } '
         ') | ( match-map MATCH_MAP ) } ]'
   )
   noOrDefaultSyntax = 'aggregate-address PREFIX ...'
   data = SetAggregateAddressBaseCmd.data.copy()
   data.pop( 'PREFIX6' )

RouterBgpBaseAfIpMulticastMode.addCommandClass(
   SetIPv4MulticastAggregateAddressCmd )
RouterBgpVrfAfIpMulticastMode.addCommandClass(
   SetIPv4MulticastAggregateAddressCmd )

#-------------------------------------------------------------------------------
# "[no|default] aggregate-address <v6prefix> [as-set]
#      [summary-only] [attribute-map <mapname>] [match-map <mapname>]
#      [advertise-only]" command,
#      in "router-bgp/address-family ipv6 multicast" mode.
#-------------------------------------------------------------------------------
class SetIPv6MulticastAggregateAddressCmd( SetAggregateAddressBaseCmd,
                                           BgpCmdBaseClass ):
   # restrict to IPv6 addresses only
   syntax = (
         'aggregate-address PREFIX6 [ { OPTIONS | ( '
         f'{ SetAggregateAddressBaseCmd.attrMapSyntax } '
         ' ) | ( match-map MATCH_MAP ) } ]'
   )
   noOrDefaultSyntax = 'aggregate-address PREFIX6 ...'
   data = SetAggregateAddressBaseCmd.data.copy()
   data.pop( 'PREFIX' )

RouterBgpBaseAfIpv6MulticastMode.addCommandClass(
   SetIPv6MulticastAggregateAddressCmd )
RouterBgpVrfAfIpv6MulticastMode.addCommandClass(
   SetIPv6MulticastAggregateAddressCmd )

#-------------------------------------------------------------------------------
# "[no|default] auto-aggregation-domain (<prefix> | <address> <mask>) [as-set]
#      [eligible-routes <mapname>] [aggregate-attributes <mapname>]
#      [ group-by {as-path|local-preference|med|community|ext-community}+ ]" command,
#      in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceAutoAggDomainCmd( BgpCmdBaseClass ):
   syntax = (
         'auto-aggregation-domain ( PREFIX | PREFIX6 ) '
         '[ { as-set | ( eligible-routes RTS ) | ( aggregate-attributes ATTRS ) } ] '
         '[ group-by { GROUP_ARGS } ]'
   )
   noOrDefaultSyntax = 'auto-aggregation-domain ( PREFIX | PREFIX6 ) ...'
   data = {
         'auto-aggregation-domain': bgpTokens.autoAggDomain,
         'PREFIX': IpAddrMatcher.ipPrefixExpr(
                       'Network address',
                       'Network mask',
                       'Prefix',
                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
         ),
         'PREFIX6': Ip6PrefixMatcher(
                        'IPv6 address prefix',
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
         ),
         'as-set': Node( matcher=bgpTokens.asSet, maxMatches=1 ),
         'eligible-routes': Node( matcher=bgpTokens.eligibleRoutes, maxMatches=1 ),
         'RTS': mapNameMatcher,
         'aggregate-attributes': Node(
               matcher=bgpTokens.aggAttributes,
               maxMatches=1
         ),
         'ATTRS': mapNameMatcher,
         'group-by': bgpTokens.groupBy,
         'GROUP_ARGS': SetEnumMatcher( {
               bgpTokens.groupByAsPath.keyword_: bgpTokens.groupByAsPath.helpdesc_,
               bgpTokens.groupByLocPref.keyword_: bgpTokens.groupByLocPref.helpdesc_,
               bgpTokens.groupByMed.keyword_: bgpTokens.groupByMed.helpdesc_,
               bgpTokens.groupByCommunity.keyword_:
                     bgpTokens.groupByCommunity.helpdesc_,
               bgpTokens.groupByExtCommunity.keyword_:
                     bgpTokens.groupByExtCommunity.helpdesc_,
         } ),
   }

   hidden = True

   _groupArgsMap = {
         'as-path': 'groupByAsPath',
         'local-preference': 'groupByLocPref',
         'med': 'groupByMed',
         'community': 'groupByCommunity',
         'ext-community': 'groupByExtCommunity',
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceAutoAggDomainCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceAutoAggDomainCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PREFIX' ] = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )

      domainArgs = {}
      if 'as-set' in args:
         domainArgs[ 'asset' ] = True
      if 'RTS' in args:
         domainArgs[ 'matchmap' ] = args[ 'RTS' ][ 0 ]
      if 'ATTRS' in args:
         domainArgs[ 'attrmap' ] = args[ 'ATTRS' ][ 0 ]
      args[ 'DOMAIN_ARGS' ] = domainArgs

      if 'GROUP_ARGS' in args:
         args[ 'GROUP_ARGS' ] = {
               SetInstanceAutoAggDomainCmd._groupArgsMap[ arg ]: True for arg in
               args.get( 'GROUP_ARGS' )
         }

RouterBgpSharedModelet.addCommandClass( SetInstanceAutoAggDomainCmd )

#-------------------------------------------------------------------------------
# "[no|default] router-id <ip-address>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceRouterIdCmd( BgpCmdBaseClass ):
   syntax = 'router-id ROUTER_ID'
   # The no rule shouldn't have trailingGarbage, or else 'no router bgp 1'
   # executed in router-bgp mode matches the 'no router-id' rule.
   noOrDefaultSyntax = 'router-id [ ROUTER_ID ]'
   data = {
         'router-id': bgpTokens.routerId,
         'ROUTER_ID': IpAddrMatcher.IpAddrMatcher(
            helpdesc='BGP router-id in IP address format' )
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceRouterIdCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceRouterIdCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceRouterIdCmd )

#-------------------------------------------------------------------------------
# "[no|default] distance bgp <distance>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceBgpDistanceCmd( CliCommandClass ):
   syntax = 'distance bgp EXTERNAL [ INTERNAL LOCAL ]'
   noOrDefaultSyntax = 'distance bgp ...'
   data = {
         'distance': bgpTokens.distance,
         'bgp': bgpTokens.distanceBgp,
         'EXTERNAL': IntegerMatcher(
               1,
               255,
               helpdesc=(
                     'Distance for external routes OR external, internal and '
                     'local routes'
               )
         ),
         'INTERNAL': IntegerMatcher(
               1,
               255,
               helpdesc='BGP internal route admin distance'
         ),
         'LOCAL': IntegerMatcher(
               1,
               255,
               helpdesc='BGP local route admin distance'
         )
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceBgpDistanceCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetInstanceBgpDistanceCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceBgpDistanceCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceBgpDistanceCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp listen [ limit <> | range network/length peer-group
# <peer-group name> ]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
# pylint: disable-next=useless-object-inheritance
class SetInstanceListenLimitRangeBaseClass( object ):
   data = {
         'bgp': bgpNode,
         'listen': bgpTokens.listen,
         'LIMIT_KW_DEPRECATED': bgpTokens.limitDeprecated,
         'LIMIT': bgpTokens.limitMatcher,
         'dynamic': bgpTokens.dynamic,
         'peer': bgpTokens.peerAfterDynamic,
         'max': bgpTokens.maxAfterDynamicPeer,
         'range': bgpTokens.subnetRange,
         'PREFIX': IpAddrMatcher.ipPrefixExpr(
                       'Network address',
                       'Network mask',
                       'Prefix',
                       overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                       maskKeyword=True
                   ),
         'PREFIX6': Ip6PrefixMatcher(
                        'IPv6 address prefix',
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO
                    ),
         'peer-id': 'Additional specification for identifying a peer',
         'include': 'Include following fields as part of peer identifier',
         'router-id': 'Include router ID as part of peer identifier',
         'peer-group': bgpTokens.peerGroup,
         'PGKEY': peergroupNameMatcher,
         'PEER_SPEC': bgpTokens.PeerSpecExpression,
   }

class SetInstanceListenLimitDeprecatedCmd(
      CliCommandClass,
      SetInstanceListenLimitRangeBaseClass
):
   syntax = 'bgp listen LIMIT_KW_DEPRECATED LIMIT'
   noOrDefaultSyntax = 'bgp listen LIMIT_KW_DEPRECATED ...'
   data = SetInstanceListenLimitRangeBaseClass.data.copy()
   handler = \
      "RoutingBgpInstanceCliHandler.SetInstanceListenLimitDeprecatedCmd_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceListenLimitDeprecatedCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetInstanceListenLimitDeprecatedCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceListenLimitDeprecatedCmd )

class SetInstanceDynamicPeerMaxLimit(
      CliCommandClass,
      SetInstanceListenLimitRangeBaseClass
):
   syntax = 'dynamic peer max LIMIT'
   noOrDefaultSyntax = 'dynamic peer max ...'
   data = SetInstanceListenLimitRangeBaseClass.data.copy()
   handler = "RoutingBgpInstanceCliHandler.SetInstanceDynamicPeerMaxLimit_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceDynamicPeerMaxLimit_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceDynamicPeerMaxLimit_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceDynamicPeerMaxLimit )

class SetInstanceListenRangePrefixPgCmd(
      CliCommandClass,
      SetInstanceListenLimitRangeBaseClass
):
   syntax = """bgp listen range ( PREFIX | PREFIX6 )
               [ peer-id include router-id ] peer-group PGKEY PEER_SPEC"""
   noOrDefaultSyntax = syntax.replace( 'PEER_SPEC', '...' )

   data = SetInstanceListenLimitRangeBaseClass.data.copy()
   handler = "RoutingBgpInstanceCliHandler.SetInstanceListenRangePrefixPgCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetInstanceListenRangePrefixPgCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'PREFIX' ] = args.get( 'PREFIX', args.get( 'PREFIX6' ) )
      args[ 'PGKEY' ] = PeerConfigKey( args[ 'PGKEY' ] )

RouterBgpSharedModelet.addCommandClass( SetInstanceListenRangePrefixPgCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp default ipv4-unicast" command, in "router-bgp" mode.
# "[no|default] bgp default ipv4-unicast transport ipv6" command, in "router-bgp"
#    mode.
# "[no|default] bgp default ipv6-unicast" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetIpv4UnicastCmd( CliCommandClass ):
   syntax = 'bgp default ipv4-unicast [ transport ipv6 ]'
   noOrDefaultSyntax = 'bgp default ipv4-unicast [ transport ipv6 ] ...'
   data = {
         'bgp': bgpNode,
         'default': bgpTokens.default,
         'ipv4-unicast': bgpTokens.iPv4Unicast,
         'transport': bgpTokens.transportForIpv4Unicast,
         'ipv6': bgpTokens.iPv6AfterTransport,
   }
   handler = "RoutingBgpInstanceCliHandler.SetIpv4UnicastCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetIpv4UnicastCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.SetIpv4UnicastCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetIpv4UnicastCmd )

class SetIpv6UnicastCmd( CliCommandClass ):
   syntax = 'bgp default ipv6-unicast'
   noOrDefaultSyntax = 'bgp default ipv6-unicast ...'
   data = {
         'bgp': bgpNode,
         'default': bgpTokens.default,
         'ipv6-unicast': bgpTokens.iPv6Unicast,
   }
   handler = "RoutingBgpInstanceCliHandler.SetIpv6UnicastCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetIpv6UnicastCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.SetIpv6UnicastCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetIpv6UnicastCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp missing-policy [address-family all] [include { community-list |
# sub-route-map | prefix-list } ] direction [ in | out ] action" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class MissingPolicyCmd( CliCommandClass ):
   includes_ = mpIncludeOptions()
   syntax = ( 'bgp missing-policy [ address-family all ] [ include { %s } ] '
              'direction DIR action ACT' % ' | '.join( includes_ ) )
   noOrDefaultSyntax = syntax.replace( 'ACT', '...' )

   data = {
      'bgp': bgpNode,
      'missing-policy': bgpTokens.missingPolicy,
      'address-family': bgpTokens.mpAddrFamily,
      'all': bgpTokens.mpAddrFamilyAll,
      'include': bgpTokens.mpInclude,
      'sub-route-map': singleNode( bgpTokens.mpSubRouteMap ),
      'prefix-list': singleNode( bgpTokens.mpPrefixList ),
      'community-list': singleNode( bgpTokens.mpCommList ),
      'direction': bgpTokens.mpDirection,
      'DIR': EnumMatcher( mpDirOptions ),
      'action': bgpTokens.mpAction,
      'ACT': EnumMatcher( mpActionOptions )
   }
   handler = "RoutingBgpInstanceCliHandler.MissingPolicyCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.MissingPolicyCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( MissingPolicyCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp missing-policy [include { community-list | prefix-list |
# sub-route-map }] direction [ in | out ] action" command, in router bgp af mode.
#-------------------------------------------------------------------------------
class MissingPolicyAfSharedCmd( CliCommandClass ):
   includes_ = mpIncludeOptions()
   syntax = 'bgp missing-policy [ include { %s } ] direction DIR action ACT' % (
      ' | '.join( includes_ ) )
   noOrDefaultSyntax = syntax.replace( 'ACT', '...' )

   data = {
      'bgp': bgpNode,
      'missing-policy': bgpTokens.missingPolicy,
      'include': bgpTokens.mpInclude,
      'sub-route-map': singleNode( bgpTokens.mpSubRouteMap ),
      'prefix-list': singleNode( bgpTokens.mpPrefixList ),
      'community-list': singleNode( bgpTokens.mpCommList ),
      'direction': bgpTokens.mpDirection,
      'DIR': EnumMatcher( mpDirOptions ),
      'action': bgpTokens.mpAction,
      'ACT': EnumMatcher( mpActionOptions )
   }
   handler = "RoutingBgpInstanceCliHandler.MissingPolicyAfSharedCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.MissingPolicyAfSharedCmd_noOrDefaultHandler"

RouterBgpAfSharedModelet.addCommandClass( MissingPolicyAfSharedCmd )
RouterBgpAfLabelSharedModelet.addCommandClass( MissingPolicyAfSharedCmd )
RouterBgpBaseAfSrTeMode.addCommandClass( MissingPolicyAfSharedCmd )
RouterBgpBaseAfLinkStateMode.addCommandClass( MissingPolicyAfSharedCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp auto-local-addr" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceAutoLocalAddrCmd( CliCommandClass ):
   syntax = 'bgp auto-local-addr'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'auto-local-addr': bgpTokens.autoLocalAddr,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceAutoLocalAddrCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetInstanceAutoLocalAddrCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceAutoLocalAddrCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceAutoLocalAddrCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp control-plane-filter default-allow"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceControlPlaneFilterDefaultAllowCmd( BgpCmdBaseClass ):
   syntax = 'bgp control-plane-filter default-allow'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'control-plane-filter': bgpTokens.controlPlane,
         'default-allow': bgpTokens.defaultAllow,
   }
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetInstanceControlPlaneFilterDefaultAllowCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetInstanceControlPlaneFilterDefaultAllowCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass(
      SetInstanceControlPlaneFilterDefaultAllowCmd
)

#----------------------------------------------------------------------------------
# " bgp transport ( ipv4 | ipv6 ) mss [ MSS ]" command in "router-bgp" mode
#----------------------------------------------------------------------------------
class SetTransportAfMssCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport ( ( ipv4 mss MSS) | ( ipv6 mss MSS6) )'
   noOrDefaultSyntax = 'bgp transport ( ipv4 | ipv6 ) mss ...'
   data = {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'ipv4': bgpTokens.ipv4,
         'ipv6': bgpTokens.ipv6,
         'mss': bgpTokens.maxSegSize,
         'MSS': bgpTokens.mssV4RangeMatcher,
         'MSS6': bgpTokens.mssV6RangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.SetTransportAfMssCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetTransportAfMssCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'MSS' ] = args.get( 'MSS' ) or args.get( 'MSS6' )

RouterBgpSharedModelet.addCommandClass( SetTransportAfMssCmd )

#----------------------------------------------------------------------------------
# "[no|default] bgp transport listen-port <portnum>" command, in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetTransportListenPortCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport listen-port PORT'
   noOrDefaultSyntax = 'bgp transport listen-port ...'
   data = {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'listen-port': bgpTokens.listenPort,
         'PORT': bgpTokens.portNumRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.SetTransportListenPortCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetTransportListenPortCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetTransportListenPortCmd )

#----------------------------------------------------------------------------------
# "[no|default] bgp transport qos dscp DSCP" command, in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetTransportQosDscpCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport qos dscp DSCP'
   noOrDefaultSyntax = 'bgp transport qos dscp ...'
   data = {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'qos': bgpTokens.qos,
         'dscp': bgpTokens.dscp,
         'DSCP': bgpTokens.dscpRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.SetTransportQosDscpCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetTransportQosDscpCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetTransportQosDscpCmd )

#----------------------------------------------------------------------------------
# "[no|default] bgp transport pmtud [ disabled ]" command in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetTransportPmtudCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport pmtud [ disabled ]'
   noOrDefaultSyntax = 'bgp transport pmtud ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'pmtud': bgpTokens.pathMtuDiscovery, } )
   handler = "RoutingBgpInstanceCliHandler.SetTransportPmtudCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetTransportPmtudCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetTransportPmtudCmd )

# ----------------------------------------------------------------------------------
# "[no|default] bgp transport buffer SIZE kbytes" command in "router-bgp" mode.
# ----------------------------------------------------------------------------------
class SetTransportSockBufferCmd( BgpCmdBaseClass ):
   syntax = 'bgp transport buffer SIZE kbytes'
   noOrDefaultSyntax = 'bgp transport buffer ...'
   data = {
         'bgp': bgpNode,
         'transport': bgpTokens.transport,
         'buffer': bgpTokens.buffer,
         'SIZE': CliMatcher.DynamicIntegerMatcher(
                     bufferRangeFn,
                     helpdesc='Value of send and receive buffer in Kilobytes' ),
         'kbytes': bgpTokens.kbytes,
   }
   handler = "RoutingBgpInstanceCliHandler.SetTransportSockBufferCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetTransportSockBufferCmd_noOrDefaultHandler"

if BgpCommonToggleLib.toggleBgpSockBufferSizeEnabled():
   RouterBgpSharedModelet.addCommandClass( SetTransportSockBufferCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp confederation identifier <asn>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceConfedIdCmd( CliCommandClass ):
   syntax = 'bgp confederation identifier AS_NUM'
   noOrDefaultSyntax = 'bgp confederation identifier ...'
   data = {
         'bgp': bgpNode,
         'confederation': bgpTokens.confed,
         'identifier': bgpTokens.confedId,
         'AS_NUM': bgpTokens.AsNumCliExpr,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceConfedIdCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceConfedIdCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceConfedIdCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp confederation peers <asn>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceConfedPeersCmd( CliCommandClass ):
   syntax = 'bgp confederation peers { ASN_MULTIRANGE }'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'confederation': bgpTokens.confed,
         'peers': bgpTokens.confedPeers,
         'ASN_MULTIRANGE': bgpTokens.AsNumMultiRangeExpr,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceConfedPeersCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceConfedPeersCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceConfedPeersCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp asn notation {asplain|asdot}" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceAsnNotationCmd( BgpCmdBaseClass ):
   syntax = 'bgp asn notation ( asplain | asdot )'
   noOrDefaultSyntax = 'bgp asn notation ...'
   data = {
         'bgp': bgpNode,
         'asn': bgpTokens.asn,
         'notation': bgpTokens.asnNotation,
         'asplain': bgpTokens.asnNotationAsplain,
         'asdot': bgpTokens.asnNotationAsdot,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceAsnNotationCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceAsnNotationCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      if args.pop( 'asplain', None ):
         args[ 'ASN_NOTATION' ] = AsnNotation.asnNotationAsplain
      elif args.pop( 'asdot', None ):
         args[ 'ASN_NOTATION' ] = AsnNotation.asnNotationAsdot

RouterBgpSharedModelet.addCommandClass( SetInstanceAsnNotationCmd )

#-------------------------------------------------------------------------------
# "[no|default] (ip|ipv6) access-group <aclName>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpAfServiceAclMixin( object ): # pylint: disable=useless-object-inheritance
   data = {
         'ip': AclCli.ipKwForServiceAclMatcher,
         'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
         'access-group': AclCli.accessGroupKwMatcher,
         'ACL': AclCli.ipAclNameMatcher,
         'ACL6': AclCli. ip6AclNameMatcher,
         'in': AclCli.inKwMatcher,
   }

class SetInstanceServiceAclCmd( BgpAfServiceAclMixin, BgpCmdBaseClass ):
   syntax = 'ip access-group ACL [ in ]'
   noOrDefaultSyntax = 'ip access-group [ ACL ] [ in ]'
   data = BgpAfServiceAclMixin.data.copy()
   handler = "RoutingBgpInstanceCliHandler.SetInstanceServiceAclCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceServiceAclCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceServiceAclCmd )

class SetInstanceServiceAclV6Cmd( BgpAfServiceAclMixin, BgpCmdBaseClass ):
   syntax = 'ipv6 access-group ACL6 [ in ]'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL6 ] [ in ]'
   data = BgpAfServiceAclMixin.data.copy()
   handler = "RoutingBgpInstanceCliHandler.SetInstanceServiceAclV6Cmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceServiceAclV6Cmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceServiceAclV6Cmd )

#-------------------------------------------------------------------------------
# [no|default] maximum-paths NUM_PATHS [
#    ( source openconfig [ global ] { ipv4-unicast | ipv6-unicast } )
#    | ( ecmp NUM_ECMP_RTS ) ] in "router bgp" mode
#-------------------------------------------------------------------------------
def maxEcmpRangeFn( mode, context ):
   # This code can be called when the startup-config is being parsed,
   # which happens before the FruAgent has discovered all the hardware
   # and populated Sysdb, allowing any value in this case and returning
   # the actual hardware supported (routingHardwareStatus.maxLogicalProtocolEcmp) in
   # the other case
   if routingHardwareStatus and routingHardwareStatus.maxLogicalProtocolEcmp:
      return ( 1, routingHardwareStatus.maxLogicalProtocolEcmp )
   return ( 1, 0xffffffff )

# These matchers must be defined here, to avoid a circular dependency with
# CliToken/RoutingBgp.py
maxPathsRangeMatcher = DynamicIntegerMatcher(
      maxEcmpRangeFn, helpdesc='Value for maximum number of equal cost paths' )
maxPathsEcmpRangeMatcher = DynamicIntegerMatcher(
      maxEcmpRangeFn, helpdesc='Value for maximum number of installed ECMP routes' )

class SetMaxPathsCmd( CliCommandClass ):
   syntax = "maximum-paths NUM_PATHS [ ecmp NUM_ECMP_RTS ]"
   if BgpCommonToggleLib.toggleOCMaximumPathsEnabled():
      syntax = ( "maximum-paths NUM_PATHS [ ( source openconfig [ global ] "
                 "{ ipv4-unicast | ipv6-unicast } ) | ( ecmp NUM_ECMP_RTS ) ]" )
   noOrDefaultSyntax = "maximum-paths ..."
   data = {
      'maximum-paths': bgpTokens.maxNumPaths,
      'NUM_PATHS': maxPathsRangeMatcher,
      'source': hiddenKeyword( 'source' ),
      'openconfig': hiddenKeyword( 'openconfig' ),
      'global': hiddenKeyword( 'global' ),
      'ipv4-unicast': hiddenKeyword( 'ipv4-unicast', maxMatches=1 ),
      'ipv6-unicast': hiddenKeyword( 'ipv6-unicast', maxMatches=1 ),
      'ecmp': bgpTokens.ecmp,
      'NUM_ECMP_RTS': maxPathsEcmpRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.SetMaxPathsCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetMaxPathsCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.SetMaxPathsCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetMaxPathsCmd )

#-------------------------------------------------------------------------------
# [no|default] default-route export VPN [route-map MAP_NAME | rcf FUNC] [always]
#-------------------------------------------------------------------------------
class SetVrfVpnDefaultExportCmd( BgpCmdBaseClass ):
   syntax = 'default-route export VPN '

   if RcfLibToggleLib.toggleRcfDefaultOriginateEnabled():
      syntax += ( '[ [ { always | ( route-map MAP_NAME ) } | '
                  '{ always | ( rcf FUNCTION ) } ] | disabled ]' )
   else:
      syntax += '[ { always | ( route-map MAP_NAME ) } | disabled ]'
   noOrDefaultSyntax = 'default-route export VPN ...'
   data = BgpCmdBaseClass._createSyntaxData(
         { 'default-route': bgpTokens.defaultRoute,
           'export': Node( bgpTokens.export, maxMatches=1 ),
           'VPN': bgpTokens.vpnMatcher,
           'always': Node( bgpTokens.defaultRouteExportAlways, maxMatches=1 ),
           'route-map': Node( bgpTokens.routeMap, maxMatches=1 ),
           'MAP_NAME': Node( mapNameMatcher, maxMatches=1 ),
           'rcf': Node( bgpTokens.rcf, maxMatches=1 ),
           'FUNCTION': Node( RcfCliLib.rcfFunctionMatcher, maxMatches=1 ),
         } )
   handler = "RoutingBgpInstanceCliHandler.SetVrfVpnDefaultExportCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetVrfVpnDefaultExportCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      exportDefOrigin = set()

      exportDefOrigin.add( ( 'always', 'always' in args ) )

      if 'route-map' in args:
         exportDefOrigin.add( ( 'mapName', args.get( 'MAP_NAME' ) ) )
      elif 'rcf' in args:
         exportDefOrigin.add( ( 'rcfName', args.get( 'FUNCTION' ) ) )

      args[ 'EXPORT_DEF_ORIGIN' ] = exportDefOrigin

#-------------------------------------------------------------------------------
# [no|default] default-route export VPN [route-map MAP_NAME | rcf FUNC] [always]
#     in "router-bgp-vrf-af" mode
#-------------------------------------------------------------------------------
RouterBgpVrfSharedModelet.addCommandClass( SetVrfVpnDefaultExportCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( SetVrfVpnDefaultExportCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp always-compare-med" in "router bgp" mode
#---------------------------------------------------------------------------------
class SetInstanceAlwaysCompareMedCmd( CliCommandClass ):
   syntax = 'bgp always-compare-med'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'always-compare-med': bgpTokens.alwaysCompareMed,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceAlwaysCompareMedCmd_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceAlwaysCompareMedCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceAlwaysCompareMedCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceAlwaysCompareMedCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp bestpath med ( missing-as-worst | confed [ missing-as-worst ] )"
# in "router bgp" mode
#---------------------------------------------------------------------------------
class SetBestPathMedCmd( CliCommandClass ):
   syntax = 'bgp bestpath med ( missing-as-worst | ( confed [ missing-as-worst ] ) )'
   noOrDefaultSyntax = syntax + ' ...'
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'med': bgpTokens.med,
         'missing-as-worst': bgpTokens.missingAsWorst,
         'confed': bgpTokens.medConfed,
   }
   handler = "RoutingBgpInstanceCliHandler.SetBestPathMedCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetBestPathMedCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.SetBestPathMedCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetBestPathMedCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp convergence [ slow-peer ] time SECONDS"
# in "router-bgp" mode.
#---------------------------------------------------------------------------------
class SetConvergenceTimeCmd( CliCommandClass ):
   syntax = 'bgp convergence [ slow-peer ] time SECONDS'
   noOrDefaultSyntax = 'bgp convergence [ slow-peer ] time ...'
   data = {
      'bgp': bgpNode,
      'convergence': bgpTokens.convergence,
      'slow-peer': bgpTokens.convergenceSlowPeer,
      'time': bgpTokens.convergenceTime,
      'SECONDS': bgpTokens.secondsRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.SetConvergenceTimeCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetConvergenceTimeCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetConvergenceTimeCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetConvergenceTimeCmd )

#---------------------------------------------------------------------------------
# "[no|default] update wait-for-convergence
# in "router-bgp" and "router-bgp-af" mode.
#---------------------------------------------------------------------------------
class SetUpdateWaitForConvergenceCmd( CliCommandClass ):
   syntax = 'update wait-for-convergence'
   noOrDefaultSyntax = syntax
   data = {
      'update': bgpTokens.update,
      'wait-for-convergence': bgpTokens.waitForConvergence,
   }
   handler = "RoutingBgpInstanceCliHandler.SetUpdateWaitForConvergenceCmd_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler.SetUpdateWaitForConvergenceCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetUpdateWaitForConvergenceCmd_defaultHandler"

for m in [ RouterBgpAfLabelSharedModelet,
           RouterBgpAfSharedModelet,
           RouterBgpSharedModelet, ]:
   m.addCommandClass( SetUpdateWaitForConvergenceCmd )


#---------------------------------------------------------------------------------
# "[no|default] bgp peer-mac-resolution-timeout <peer-mac-resolution-timeout>
# in "router-bgp" mode.
#---------------------------------------------------------------------------------
class SetInstancePeerMacResolutionTimeoutCmd( CliCommandClass ):
   syntax = 'bgp peer-mac-resolution-timeout TIMEOUT'
   noOrDefaultSyntax = 'bgp peer-mac-resolution-timeout ...'
   data = {
         'bgp': bgpNode,
         'peer-mac-resolution-timeout': bgpTokens.peerMacResolutionTimeout,
         'TIMEOUT': IntegerMatcher(
               0,
               600,
               helpdesc='Time to wait for proactive resolution'
         ),
   }
   hidden = True
   handler = \
      "RoutingBgpInstanceCliHandler.SetInstancePeerMacResolutionTimeoutCmd_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler.SetInstancePeerMacResolutionTimeoutCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetInstancePeerMacResolutionTimeoutCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstancePeerMacResolutionTimeoutCmd )

#---------------------------------------------------------------------------------
# [no|default] bgp host-routes fib direct-install under "config-bgp" mode
#
# This command does nothing, and only exists for update-path sustainability.
# See http://cl/2620911.
#---------------------------------------------------------------------------------
class HostRoutesFibDirectInstallCmd( CliCommandClass ):
   syntax = 'bgp host-routes fib direct-install'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'host-routes': bgpTokens.hostRoutes,
         'fib': bgpTokens.fib,
         'direct-install': bgpTokens.directInstall,
   }
   handler = "RoutingBgpInstanceCliHandler.HostRoutesFibDirectInstallCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.HostRoutesFibDirectInstallCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( HostRoutesFibDirectInstallCmd )

#-----------------------------------------------------------------------------
# "[no|default] bgp route-reflector preserve-attributes [always]"
# in "config-bgp" mode.
#-----------------------------------------------------------------------------
class SetInstancePreserveAttrsCmd( CliCommandClass ):
   syntax = 'bgp route-reflector preserve-attributes [ always ]'
   noOrDefaultSyntax = 'bgp route-reflector preserve-attributes ...'
   data = {
         'bgp': bgpNode,
         'route-reflector': bgpTokens.routeReflector,
         'preserve-attributes': bgpTokens.preserveAttrs,
         'always': bgpTokens.always,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstancePreserveAttrsCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetInstancePreserveAttrsCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstancePreserveAttrsCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstancePreserveAttrsCmd )

#-------------------------------------------------------------------------------
# "[no|default] default-metric <metric>" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetInstanceDefaultMetricCmd( BgpCmdBaseClass ):
   syntax = 'default-metric METRIC'
   noOrDefaultSyntax = 'default-metric ...'
   data = {
         'default-metric': bgpTokens.defaultMetric,
         'METRIC': IntegerMatcher( 0, U32_MAX_VALUE, helpdesc='Default metric' ),
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceDefaultMetricCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceDefaultMetricCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetInstanceDefaultMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] ucmp mode <mode#> <maxUcmp> <deviation>" command
#-------------------------------------------------------------------------------
class UcmpModeCmd( CliCommandClass ):
   syntax = 'ucmp mode MODE [ MAX [ DEV ] ]'
   noOrDefaultSyntax = 'ucmp mode ...'

   data = {
         'ucmp': ucmpNode,
         'mode': bgpTokens.ucmpMode,
         'MODE': bgpTokens.ucmpModeRangeMatcher,
         'MAX': ucmpSizeRange,
         'DEV': ucmpDevRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.UcmpModeCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.UcmpModeCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.UcmpModeCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( UcmpModeCmd )

#-------------------------------------------------------------------------------
# "[no|default] ucmp link-bandwidth update-delay <num>" command
# "[no|default] ucmp link-bandwidth encoding-weighted" command
# "[no|default] ucmp link-bandwidth recursive" command
#-------------------------------------------------------------------------------
class UcmpLinkBandwidthCmd( CliCommandClass ):
   # To add support for `recursive ( static weight | bgp )` tokens in the future,
   # add another Node to the data field for each token, each with maxMatch=1 set,
   # and change the syntax to: ( recursive { ( static | user weight ) | bgp } )
   # See AID/11447 for CLI and phased delivery details.
   syntax = (
         'ucmp link-bandwidth ( recursive [ user weight ] ) | encoding-weighted | '
         '( update-delay DELAY )'
   )
   noOrDefaultSyntax = (
         'ucmp link-bandwidth ( recursive | encoding-weighted | update-delay ) ...'
   )
   data = {
         'ucmp': ucmpNode,
         'link-bandwidth': bgpTokens.ucmpLinkbw,
         'recursive': bgpTokens.ucmpRecursive,
         'user': Node(
            'Inherit from resolving SDK routes',
            hidden=not UcmpLibToggleLib.toggleSdkLbwInheritanceEnabled() ),
         'weight': 'Inherit from weight field',
         'encoding-weighted': bgpTokens.ucmpEncodingWeighted,
         'update-delay': bgpTokens.ucmpLinkbwDelay,
         'DELAY': bgpTokens.ucmpDelayRangeMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.UcmpLinkBandwidthCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.UcmpLinkBandwidthCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.UcmpLinkBandwidthCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( UcmpLinkBandwidthCmd )

#-------------------------------------------------------------------------------
# "[no|default] ucmp fec threshold trigger <%> clear <%> warning-only" command
#-------------------------------------------------------------------------------
class UcmpFecThresholdCmd( CliCommandClass ):
   syntax = 'ucmp fec threshold trigger PERCENT clear CLR_PERCENT warning-only'
   noOrDefaultSyntax = 'ucmp fec threshold ...'
   data = {
         'ucmp': ucmpNode,
         'fec': bgpTokens.ucmpFec,
         'threshold': bgpTokens.ucmpThreshold,
         'trigger': bgpTokens.ucmpTrigger,
         'PERCENT': bgpTokens.ucmpFecPercentTrigMatcher,
         'clear': bgpTokens.ucmpClear,
         'CLR_PERCENT': bgpTokens.ucmpFecPercentClearMatcher,
         'warning-only': bgpTokens.ucmpWarnOnly
   }
   handler = "RoutingBgpInstanceCliHandler.UcmpFecThresholdCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.UcmpFecThresholdCmd_noHandler"
   defaultHandler = "RoutingBgpInstanceCliHandler.UcmpFecThresholdCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( UcmpFecThresholdCmd )

#--------------------------------------------------------------------------
# "bgp next-hop-unchanged" command
#--------------------------------------------------------------------------
class SetInstanceNexthopUnchangedCmd( CliCommandClass ):
   syntax = 'bgp next-hop-unchanged'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'next-hop-unchanged': bgpTokens.nexthopUnchanged,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceNexthopUnchangedCmd_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceNexthopUnchangedCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceNexthopUnchangedCmd_defaultHandler"

for m in [
      RouterBgpSharedModelet,
      RouterBgpAfSharedModelet,
      RouterBgpAfLabelSharedModelet,
]:
   m.addCommandClass( SetInstanceNexthopUnchangedCmd )

# --------------------------------------------------------------------------
# "neighbor default next-hop-self" command, in "router-bgp-af" mode
# --------------------------------------------------------------------------
class SetInstanceNexthopSelfCmd( CliCommandClass ):
   syntax = 'neighbor default next-hop-self'
   noOrDefaultSyntax = syntax
   data = {
         'neighbor': bgpTokens.neighbor,
         'default': bgpTokens.neighborDefaultGlobal,
         'next-hop-self': bgpTokens.nextHopSelf,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceNexthopSelfCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetInstanceNexthopSelfCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceNexthopSelfCmd_defaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( SetInstanceNexthopSelfCmd )

RouterBgpAfLabelSharedModelet.addCommandClass( SetInstanceNexthopSelfCmd )

#---------------------------------------------------------------------------------
# "[no|default] bgp next-hop address-family ipv6" command, in "router-bgp-af" mode
#---------------------------------------------------------------------------------
#---------------------------------------------------------------------------------
# "[no|default] bgp next-hop address-family ipv6" command, in "router-bgp-af" mode
#---------------------------------------------------------------------------------
class SetInstanceNextHopAfIpv6Cmd( BgpCmdBaseClass ):
   syntax = 'bgp next-hop address-family ipv6'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'next-hop': bgpTokens.nextHop,
         'address-family': bgpTokens.nextHopAf,
         'ipv6': bgpTokens.ipv6,
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceNextHopAfIpv6Cmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceNextHopAfIpv6Cmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( SetInstanceNextHopAfIpv6Cmd )

#-----------------------------------------------------------------------------
# "[no|default] aigp-session ( ibgp | ebgp | confederation )" command, in
# "config-bgp-af" mode
#-----------------------------------------------------------------------------
class AigpSessionAfCommand( CliCommandClass ):
   syntax = "aigp-session PEER_TYPE"
   noOrDefaultSyntax = syntax
   data = {
      'aigp-session': bgpTokens.aigpSession,
      'PEER_TYPE': bgpTokens.aigpPeerType,
   }
   handler = "RoutingBgpInstanceCliHandler.AigpSessionAfCommand_handler"
   noHandler = "RoutingBgpInstanceCliHandler.AigpSessionAfCommand_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.AigpSessionAfCommand_defaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( AigpSessionAfCommand )
RouterBgpAfLabelSharedModelet.addCommandClass( AigpSessionAfCommand )

#---------------------------------------------------------------------------------
# "[no|default] next-hop resolution disabled" in "address-family vpn-ipv4" and
# "address-family vpn-ipv6" modes
#---------------------------------------------------------------------------------
class NexthopResolutionDisabledCmd( CliCommandClass ):
   syntax = 'next-hop resolution disabled'
   noOrDefaultSyntax = syntax
   data = {
      'next-hop': bgpTokens.nextHopKw,
      'resolution': IpRibLibCliTokens.matcherResolution,
      'disabled': bgpTokens.disabledKwMatcherForNhResolution,
   }
   handler = "RoutingBgpInstanceCliHandler.NexthopResolutionDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.NexthopResolutionDisabledCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( NexthopResolutionDisabledCmd )

# -------------------------------------------------------------------------------
# "[no|default] next-hop originate tunnel type udp policy POLICYNAME" in
# "address-family ipv*" and "address-family vpn-ipv*" modes
# -------------------------------------------------------------------------------
class NextHopOriginateTunnelCmd( CliCommandClass ):
   syntax = 'next-hop originate tunnel type udp policy POLICYNAME'
   noOrDefaultSyntax = 'next-hop originate tunnel type udp ...'
   data = {
      'next-hop': bgpTokens.nextHopKw,
      'originate': bgpTokens.nextHopTunnelOriginate,
      'tunnel': TunnelGlobalConfigMode.matcherTunnel,
      'type': TunnelGlobalConfigMode.matcherType,
      'udp': TunnelGlobalConfigMode.matcherUdp.matcher_,
      'policy': TunnelProfilesMode.matcherPolicy,
      'POLICYNAME': TunnelProfilesMode.matcherPolicyName,
   }
   handler = "RoutingBgpInstanceCliHandler.NextHopOriginateTunnelCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.NextHopOriginateTunnelCmd_noOrDefaultHandler"

if BgpCommonToggleLib.toggleIpOverDynamicGueTunnelsEnabled():
   for m in [ RouterBgpBaseAfIpUniMode, RouterBgpBaseAfIpv6UniMode ]:
      m.addCommandClass( NextHopOriginateTunnelCmd )

#---------------------------------------------------------------------------------
# "[no|default] redistribute <routing protocol> <routemap>" command, in
# "router-bgp" mode and "router-bgp-af" mode.
#---------------------------------------------------------------------------------
class RedistributeBaseCmd( object ): # pylint: disable=useless-object-inheritance
   syntax = 'redistribute PROTOCOL'

   noOrDefaultSyntax = syntax + ' ...'
   noOrDefaultSyntaxWithMatch = syntax + ' [ MATCH_EXPR ] ...'
   noOrDefaultSyntaxWithNssaMatch = syntax + ' match nssa-external ...'

   @staticmethod
   def makeDataDict( data, needRcf=False ):
      # The data dict of the base class contains all keyword and expression matchers
      # for all derived classes, except PROTOCOL and MATCH_EXPR, which must be
      # defined by the appropriate child classes.
      baseData = {
            'redistribute': bgpTokens.redistribute,
            'route-map': bgpTokens.routeMap,
            'MAP_NAME': mapNameMatcher,
            'include': bgpTokens.redistInclude,
            'leaked': bgpTokens.leaked,
            'ISIS_LEVEL': bgpTokens.IsisLevelExpr,
            'match': bgpTokens.asMatch,
      }
      baseData.update( data )
      if needRcf:
         baseData.update( {
            'rcf': bgpTokens.rcf,
            'FUNCTION': RcfCliLib.rcfFunctionMatcher,
         } )
      return baseData

#---------------------------------------------------------------------------------
# [no|default] redistribute ( connected | static | attached-host )
#    [ route-map MAP_NAME ]
#---------------------------------------------------------------------------------
class RedistributeMcastCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.McastRedistExpr,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeMcastCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeMcastCmd_noOrDefaultHandler"

for m in [
      RouterBgpBaseAfIpMulticastMode,
      RouterBgpVrfAfIpMulticastMode, ]:
   m.addCommandClass( RedistributeMcastCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ( connected | static ) [ route-map <name> ]
#-------------------------------------------------------------------------------
class RedistributeMcast6Cmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.McastRedist6Expr,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeMcast6Cmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeMcast6Cmd_noOrDefaultHandler"

for m in [
      RouterBgpBaseAfIpv6MulticastMode,
      RouterBgpVrfAfIpv6MulticastMode, ]:
   m.addCommandClass( RedistributeMcast6Cmd )

class RouteInputBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = "route input address-family ADDR_FAMILY [ rcf FUNCTION ]"
   noOrDefaultSyntax = "route input address-family ADDR_FAMILY ..."
   handler = "RoutingBgpInstanceCliHandler.RouteInputBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouteInputBaseCmd_noOrDefaultHandler"

   @staticmethod
   def makeRouteInputData( data ):
      baseData = {
         "route": "Route configuration",
         "input": "Input from BGP address family",
         "address-family": bgpTokens.addrFamily,
         "rcf": bgpTokens.rcf,
         "FUNCTION": RcfCliLib.rcfFunctionMatcher,
      }
      data.update( baseData )
      return data

# -------------------------------------------------------------------------------
# [no|default] route input address-family ipv4 unicast [ rcf FUNCTION ]
# -------------------------------------------------------------------------------
class RouteInputMcastV4UnicastCmd( RouteInputBaseCmd ):
   # pylint: disable-msg=W0212
   data = BgpCmdBaseClass._createSyntaxData( RouteInputBaseCmd.makeRouteInputData( {
      "ADDR_FAMILY": bgpTokens.Ipv4UnicastAfExpr } ) )

for m in [ RouterBgpBaseAfIpMulticastMode,
           RouterBgpVrfAfIpMulticastMode, ]:
   m.addCommandClass( RouteInputMcastV4UnicastCmd )

# -------------------------------------------------------------------------------
# [no|default] route input address-family ipv6 unicast [ rcf FUNCTION ]
# -------------------------------------------------------------------------------
class RouteInputMcastV6UnicastCmd( RouteInputBaseCmd ):
   # pylint: disable-msg=W0212
   data = BgpCmdBaseClass._createSyntaxData( RouteInputBaseCmd.makeRouteInputData( {
      "ADDR_FAMILY": bgpTokens.Ipv6UnicastAfExpr } ) )

for m in [ RouterBgpBaseAfIpv6MulticastMode,
           RouterBgpVrfAfIpv6MulticastMode, ]:
   m.addCommandClass( RouteInputMcastV6UnicastCmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute ( connected | static ) [ include leaked ]
#    [ route-map MAP_NAME ]
#---------------------------------------------------------------------------------
redistConnStatSyntax = RedistributeBaseCmd.syntax + """\
      [ include leaked ] \
      [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ] \
"""
needRcfForRedist = True

class ConnStatRedistributeProtoBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = redistConnStatSyntax
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.ConnectedStaticExpression,
   }, needRcf=needRcfForRedist ) )
   handler = "RoutingBgpInstanceCliHandler.ConnStatRedistributeProtoBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "ConnStatRedistributeProtoBaseCmd_noOrDefaultHandler"

class ConnStatRedistributeProtoCmd( ConnStatRedistributeProtoBaseCmd ):
   pass
RouterBgpSharedModelet.addCommandClass( ConnStatRedistributeProtoCmd )

# ---------------------------------------------------------------------------------
# [no|default] redistribute ( connected | static ) [ include leaked ]
#    [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ] [ disabled ]
#    in "router bgp" ipv4/v6 unicast mode
# ---------------------------------------------------------------------------------
redistConnStatAfSyntax = redistConnStatSyntax + " [ disabled ]"

class ConnStatRedistributeProtoAfCmd( ConnStatRedistributeProtoBaseCmd ):
   syntax = redistConnStatAfSyntax
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   # pylint: disable-msg=W0212
   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.ConnectedStaticExpression,
   }, needRcf=needRcfForRedist ) )

RouterBgpAfIpUniSharedModelet.addCommandClass(
      ConnStatRedistributeProtoAfCmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute ( rip | aggregate | attached-host )
#    [ route-map MAP_NAME ] in "router bgp" mode
#---------------------------------------------------------------------------------
class RedistributeProtoCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.ProtocolExpression,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeProtoCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeProtoCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( RedistributeProtoCmd )

# ---------------------------------------------------------------------------------
# [no|default] redistribute rip [ route-map MAP_NAME ] [ disabled ]
# in "router bgp" ipv4 unicast mode
# ---------------------------------------------------------------------------------
class RedistributeRipAfCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ] [ disabled ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.RipExpr,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeRipAfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeRipAfCmd_noOrDefaultHandler"

for m in [ RouterBgpVrfAfIpMode, RouterBgpBaseAfIpUniMode ]:
   m.addCommandClass( RedistributeRipAfCmd )

# ---------------------------------------------------------------------------------
# [no|default] redistribute attached-host [ route-map MAP_NAME ] [ disabled ]
#    in "router bgp" ipv4/v6 unicast mode
# ---------------------------------------------------------------------------------
class RedistributeAttachedHostAfCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ] [ disabled ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.AttachedHostExpr,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeAttachedHostAfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeAttachedHostAfCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( RedistributeAttachedHostAfCmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute dynamic [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ]
#    in "router bgp" mode
#---------------------------------------------------------------------------------
class RedistributeDynamicPolicyCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.DynamicPolicyExpr,
   }, needRcf=True ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeDynamicPolicyCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeDynamicPolicyCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( RedistributeDynamicPolicyCmd )

# ---------------------------------------------------------------------------------
# [no|default] redistribute dynamic [ ( route-map MAP_NAME | rcf FUNCTION ) ]
#    [ disabled ] in "router bgp" ipv4/v6 unicast mode
# ---------------------------------------------------------------------------------
class RedistributeDynamicPolicyAfCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ] [ disabled ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.DynamicPolicyExpr,
   }, needRcf=True ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeDynamicPolicyAfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RedistributeDynamicPolicyAfCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( RedistributeDynamicPolicyAfCmd )

class RedistributeBgpBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' leaked [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpNode,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeBgpBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeBgpBaseCmd_noOrDefaultHandler"

# ---------------------------------------------------------------------------------
# [no|default] redistribute bgp leaked [ route-map MAP_NAME ] in "router bgp" mode
# ---------------------------------------------------------------------------------
class RedistributeBgpCmd( RedistributeBgpBaseCmd ):
   pass
RouterBgpSharedModelet.addCommandClass( RedistributeBgpCmd )

# ---------------------------------------------------------------------------------
# [no|default] redistribute bgp leaked [ route-map MAP_NAME ] [ disabled ]
# in "router bgp" ipv4/v6 unicast mode
# ---------------------------------------------------------------------------------
class RedistributeBgpAfCmd( RedistributeBgpBaseCmd ):
   syntax = RedistributeBgpBaseCmd.syntax + ' [ disabled ]'
   data = RedistributeBgpBaseCmd.data.copy()

RouterBgpAfIpUniSharedModelet.addCommandClass(
      RedistributeBgpAfCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute isis [ ISIS_LEVEL ] [ include leaked ]
#    [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ]
#-------------------------------------------------------------------------------
redistIsisSyntax = RedistributeBaseCmd.syntax + """
      [ ISIS_LEVEL ] [ include leaked ]
      [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ]
"""

class RedistributeIsisBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = redistIsisSyntax
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.isis,
   }, needRcf=True ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeIsisBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeIsisBaseCmd_noOrDefaultHandler"

class RedistributeIsisCmd( RedistributeIsisBaseCmd ):
   pass

# -------------------------------------------------------------------------------
# [no|default] redistribute isis [ ISIS_LEVEL ] [ include leaked ]
#    [ ( route-map MAP_NAME ) | ( rcf FUNCTION ) ] [ disabled ]
# -------------------------------------------------------------------------------
redistIsisAfSyntax = redistIsisSyntax + " [ disabled ]"

class RedistributeIsisAfCmd( RedistributeIsisBaseCmd ):
   syntax = redistIsisAfSyntax
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax
   # pylint: disable-msg=W0212
   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.isis,
   }, needRcf=True ) )

for m in [
      RouterBgpSharedModelet,
      RouterBgpAfIpMcastSharedModelet,
]:
   m.addCommandClass( RedistributeIsisCmd )

RouterBgpAfIpUniSharedModelet.addCommandClass(
      RedistributeIsisAfCmd )

#-------------------------------------------------------------------------------
# '[no|default] redistribute dhcp [ route-map MAP_NAME ]'
#-------------------------------------------------------------------------------
class RedistributeDhcpCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.dhcp,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeDhcpCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeDhcpCmd_noOrDefaultHandler"

for m in [ RouterBgpVrfAfIp6Mode, RouterBgpBaseAfIpv6UniMode ]:
   m.addCommandClass( RedistributeDhcpCmd )

class RedistributeOspfMatchIntExtLeakedBaseCmd( RedistributeBaseCmd,
      BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ MATCH_EXPR ] [ include leaked ] [ route-map MAP_NAME ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithMatch
   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'MATCH_EXPR': bgpTokens.OspfMatchIntExtExpr,
   } ) )
   handler = \
      "RoutingBgpInstanceCliHandler.RedistributeOspfMatchIntExtLeakedBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RedistributeOspfMatchIntExtLeakedBaseCmd_noOrDefaultHandler"

#-------------------------------------------------------------------------------
# [no|default] redistribute ( ospf | ospfv3 ) match ( internal | external )
#    [ include leaked ] [ route-map MAP_NAME ] in "router bgp" mode
#-------------------------------------------------------------------------------
class RedistributeOspfOrOspf3MatchIntExtLeakedCmd(
      RedistributeOspfMatchIntExtLeakedBaseCmd ):

   data = RedistributeOspfMatchIntExtLeakedBaseCmd.data.copy()
   data.update( { 'PROTOCOL': bgpTokens.Ospf3OrOspfExpr } )

RouterBgpSharedModelet.addCommandClass( RedistributeOspfOrOspf3MatchIntExtLeakedCmd )

class RedistributeOspfMatchNssaExtLeakedBaseCmd( RedistributeBaseCmd,
      BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + \
               ' MATCH_EXPR [ include leaked ] [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithNssaMatch
   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'MATCH_EXPR': bgpTokens.OspfMatchNssaExtExpr,
   } ) )
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "RedistributeOspfMatchNssaExtLeakedBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RedistributeOspfMatchNssaExtLeakedBaseCmd_noOrDefaultHandler"

# -------------------------------------------------------------------------------
# [no|default] redistribute ( ospf | ospfv3 ) match nssa-external [ 1 | 2 ]
#    [ include leaked ] [ route-map MAP_NAME ] in "router bgp" mode
# -------------------------------------------------------------------------------
class RedistributeOspfOrOspf3MatchNssaExtLeakedCmd(
      RedistributeOspfMatchNssaExtLeakedBaseCmd ):

   data = RedistributeOspfMatchNssaExtLeakedBaseCmd.data.copy()
   data.update( { 'PROTOCOL': bgpTokens.Ospf3OrOspfExpr } )

RouterBgpSharedModelet.addCommandClass(
      RedistributeOspfOrOspf3MatchNssaExtLeakedCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ( ospf | ospf3 ) match ( internal | external )
#    [ route-map MAP_NAME ]
#-------------------------------------------------------------------------------
class RedistributeOspfMatchIntExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = ( RedistributeBaseCmd.syntax +
              ' [ MATCH_EXPR ] [ route-map MAP_NAME ]' )
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3OrOspfExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchIntExtExpr,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeOspfMatchIntExtCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RedistributeOspfMatchIntExtCmd_noOrDefaultHandler"

RouterBgpAfIpMcastSharedModelet.addCommandClass( RedistributeOspfMatchIntExtCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ospf3 match nssa-external [ 1 | 2 ]
# [ route-map MAP_NAME ]
#-------------------------------------------------------------------------------
class RedistributeOspf3MatchNssaExtCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = RedistributeBaseCmd.syntax + ' MATCH_EXPR  [ route-map MAP_NAME ]'
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntaxWithNssaMatch

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.Ospf3OrOspfExpr,
         'MATCH_EXPR': bgpTokens.OspfMatchNssaExtExpr,
   } ) )
   handler = "RoutingBgpInstanceCliHandler.RedistributeOspf3MatchNssaExtCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RedistributeOspf3MatchNssaExtCmd_noOrDefaultHandler"

RouterBgpAfIpMcastSharedModelet.addCommandClass( RedistributeOspf3MatchNssaExtCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ospfv3 match ( internal | external ) [ include leaked ]
#    [ route-map MAP_NAME ] in "router bgp" ipv4/v6 unicast mode
#-------------------------------------------------------------------------------
class RedistributeOspf3MatchIntExtLeakedCmd(
      RedistributeOspfMatchIntExtLeakedBaseCmd ):

   syntax = RedistributeOspfMatchIntExtLeakedBaseCmd.syntax + ' [ disabled ]'

   data = RedistributeOspfMatchIntExtLeakedBaseCmd.data.copy()
   data.update( { 'PROTOCOL': bgpTokens.Ospf3IncludingHiddenExpr } )

RouterBgpAfIpUniSharedModelet.addCommandClass(
      RedistributeOspf3MatchIntExtLeakedCmd )

#-------------------------------------------------------------------------------
# [no|default] redistribute ospfv3 match nssa-external [ 1 | 2 ]
#    [ include leaked ] [ route-map MAP_NAME ] in "router bgp" ipv4/v6 unicast mode
#-------------------------------------------------------------------------------
class RedistributeOspf3MatchNssaExtLeakedCmd(
      RedistributeOspfMatchNssaExtLeakedBaseCmd ):

   syntax = RedistributeOspfMatchNssaExtLeakedBaseCmd.syntax + ' [ disabled ]'

   data = RedistributeOspfMatchNssaExtLeakedBaseCmd.data.copy()
   data.update( { 'PROTOCOL': bgpTokens.Ospf3IncludingHiddenExpr } )

RouterBgpAfIpUniSharedModelet.addCommandClass(
      RedistributeOspf3MatchNssaExtLeakedCmd
)

# -------------------------------------------------------------------------------
# [no|default] redistribute ospf match ( internal | external ) [ include leaked ]
#    [ route-map MAP_NAME ] in "router bgp" ipv4 unicast mode
# -------------------------------------------------------------------------------
class RedistributeOspfMatchIntExtLeakedCmd(
      RedistributeOspfMatchIntExtLeakedBaseCmd ):

   syntax = RedistributeOspfMatchIntExtLeakedBaseCmd.syntax + ' [ disabled ]'

   data = RedistributeOspfMatchIntExtLeakedBaseCmd.data.copy()
   data.update( { 'PROTOCOL': bgpTokens.OspfExpr } )

for m in [ RouterBgpVrfAfIpMode, RouterBgpBaseAfIpUniMode ]:
   m.addCommandClass( RedistributeOspfMatchIntExtLeakedCmd )

# -------------------------------------------------------------------------------
# [no|default] redistribute ospf match nssa-external [ 1 | 2 ]
#    [ include leaked ] [ route-map MAP_NAME ] in "router bgp" ipv4 unicast mode
# -------------------------------------------------------------------------------
class RedistributeOspfMatchNssaExtLeakedCmd(
      RedistributeOspfMatchNssaExtLeakedBaseCmd ):

   syntax = RedistributeOspfMatchNssaExtLeakedBaseCmd.syntax + ' [ disabled ]'

   data = RedistributeOspfMatchNssaExtLeakedBaseCmd.data.copy()
   data.update( { 'PROTOCOL': bgpTokens.OspfExpr } )

for m in [ RouterBgpVrfAfIpMode, RouterBgpBaseAfIpUniMode ]:
   m.addCommandClass( RedistributeOspfMatchNssaExtLeakedCmd )

#---------------------------------------------------------------------------------
# [no|default] redistribute user [ rcf FUNCTION ] in "router bgp" mode
# ---------------------------------------------------------------------------------
redistUserSyntax = RedistributeBaseCmd.syntax + """\
      [ rcf FUNCTION ] \
"""

class UserRedistributeBaseCmd( RedistributeBaseCmd, BgpCmdBaseClass ):
   syntax = redistUserSyntax
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.UserExpression,
   }, needRcf=True ) )
   handler = "RoutingBgpInstanceCliHandler.UserRedistributeBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.UserRedistributeBaseCmd_noOrDefaultHandler"

class UserRedistributeCmd( UserRedistributeBaseCmd ):
   pass

RouterBgpSharedModelet.addCommandClass( UserRedistributeCmd )

# ---------------------------------------------------------------------------------
# [no|default] redistribute user [ rcf FUNCTION ] [ disabled ]
#    in "router bgp" ipv4/v6 unicast mode
# ---------------------------------------------------------------------------------
redistUserAfSyntax = redistUserSyntax + " [ disabled ]"

class UserRedistributeAfCmd( UserRedistributeBaseCmd ):
   syntax = redistUserAfSyntax
   noOrDefaultSyntax = RedistributeBaseCmd.noOrDefaultSyntax

   # pylint: disable-msg=W0212
   data = BgpCmdBaseClass._createSyntaxData( RedistributeBaseCmd.makeDataDict( {
         'PROTOCOL': bgpTokens.UserExpression,
   }, needRcf=True ) )

RouterBgpAfIpUniSharedModelet.addCommandClass(
      UserRedistributeAfCmd )

# ---------------------------------------------------------------------------------
# [no|default] tunnel source-protocol ( ldp | ( isis segment-routing ) )
#       [ rcf FUNCTION ]
# in "address-family ipv4 labeled-unicast"
#
# Enables redistribution of different protocol tunnels into BGP LU. (AID8019)
#---------------------------------------------------------------------------------
class RedistributeTunnelCmd( BgpCmdBaseClass ):
   syntax = 'tunnel source-protocol PROTOCOL [ rcf FUNCTION ]'
   noOrDefaultSyntax = 'tunnel source-protocol PROTOCOL ...'

   data = {
         'tunnel': bgpTokens.tunnelProtocol,
         'source-protocol': bgpTokens.tunnelSrcProtocol,
         'PROTOCOL': bgpTokens.TunnelProtocolExpression,
         'rcf': bgpTokens.rcf,
         'FUNCTION': RcfCliLib.rcfFunctionMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.RedistributeTunnelCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RedistributeTunnelCmd_noOrDefaultHandler"

RouterBgpBaseAfLabelV4Mode.addCommandClass( RedistributeTunnelCmd )

#-------------------------------------------------------------------------------
# "[no|default] rd <rd>" command in "router-bgp-vrf" mode
#-------------------------------------------------------------------------------
class SetInstanceRouteDistinguisher( BgpCmdBaseClass ):
   syntax = 'rd RD'
   noOrDefaultSyntax = 'rd ...'
   data = {
         'rd': bgpTokens.rd,
         'RD': RdDistinguisherMatcher( 'BGP route distinguisher' ),
   }
   handler = "RoutingBgpInstanceCliHandler.SetInstanceRouteDistinguisher_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetInstanceRouteDistinguisher_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( SetInstanceRouteDistinguisher )

# ------------------------------------------------------------------------------
# "[no|default] rd evpn domain (remote|all) <rd>" command in "router-bgp-vrf"
# mode
# ------------------------------------------------------------------------------
class SetInstanceRouteDistinguisherRemoteDomain( BgpCmdBaseClass ):
   syntax = 'rd evpn domain ( remote | all ) RD'
   noOrDefaultSyntax = 'rd evpn domain ( remote | all ) [ RD ]'
   # This CLI is available if
   # (1) EvpnL3DciRdRt toggle is on; OR
   # (2) EvpnL3DciWithDps toggle is on and it is the SFE platform (on which the
   #     pathSelectionSupportedGuard takes effect)
   data = {
         'rd': bgpTokens.rd,
         'evpn': ( bgpTokens.evpn if BgpCommonToggleLib.toggleEvpnL3DciRdRtEnabled()
                   else Node( bgpTokens.evpn, guard=pathSelectionSupportedGuard ) ),
         'domain': bgpTokens.domain,
         'remote': bgpTokens.domainRemote,
         'all': bgpTokens.domainAll,
         'RD': RdDistinguisherMatcher( 'BGP route distinguisher' ),
   }

   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetInstanceRouteDistinguisherRemoteDomain_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetInstanceRouteDistinguisherRemoteDomain_noOrDefaultHandler"

if ( BgpCommonToggleLib.toggleEvpnL3DciRdRtEnabled() or
     BgpCommonToggleLib.toggleEvpnL3DciWithDpsEnabled() ):
   RouterBgpVrfSharedModelet.addCommandClass(
         SetInstanceRouteDistinguisherRemoteDomain )

#-------------------------------------------------------------------------------
# "[no|default] rd auto" command in "router-bgp" mode
#-------------------------------------------------------------------------------

class RdAutoCommand( CliCommandClass ):
   syntax = 'rd auto'
   noOrDefaultSyntax = syntax
   data = {
      'rd': 'BGP route distinguisher',
      'auto': 'Auto Generate Route Distinguisher'
   }
   handler = "RoutingBgpInstanceCliHandler.RdAutoCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RdAutoCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RdAutoCommand )

# -------------------------------------------------------------------------------
# "[no|default] vrf-label (vpn-ipv4|vpn-ipv6) <label>
#  command, in "router-bgp-vrf" mode
# -------------------------------------------------------------------------------

def checkConflictingLabelRangeForVrfLocalLabel( rangeType, base, size,
                                                conflictingConfigList ):
   if rangeType != LabelRangeInfo.rangeTypeStatic:
      return
   minLabel = base
   maxLabel = base + size - 1
   for vrfName, vrfConfig in bgpVrfConfigDir.vrfConfig.items():
      for attrName in ( 'vrfLocalLabelAfVpnV4', 'vrfLocalLabelAfVpnV6' ):
         vrfLocalLabelConfig = getattr( vrfConfig, attrName )
         if vrfLocalLabelConfig.rangeType == rangeType:
            label = vrfLocalLabelConfig.label
            if label < minLabel or label > maxLabel:
               conflictingConfigList.append(
                     "BGP VRF %s local IP lookup MPLS label %s" % \
                           ( vrfName, label ) )

def handleUpdatedLabelRangeForVrfLocalLabel( rangeType, labelRange ):
   if rangeType != LabelRangeInfo.rangeTypeStatic:
      return
   base = labelRange.base
   for vrfConfig in bgpVrfConfigDir.vrfConfig.values():
      for attrName in ( 'vrfLocalLabelAfVpnV4', 'vrfLocalLabelAfVpnV6' ):
         vrfLocalLabelConfig = getattr( vrfConfig, attrName )
         if vrfLocalLabelConfig.rangeType == rangeType:
            label = vrfLocalLabelConfig.label
            labelIndex = label - base
            value = VrfLocalLabelConfig( rangeType, label, labelIndex )
            setattr( vrfConfig, attrName, value )

class MplsVpnVrfLocalLabelCmd( BgpCmdBaseClass ):
   syntax = 'vrf-label ( vpn-ipv4 | vpn-ipv6 ) LABEL'
   noOrDefaultSyntax = 'vrf-label ( vpn-ipv4 | vpn-ipv6 ) ...'
   data = {
         'vrf-label': bgpTokens.vrfLocalLabel,
         'vpn-ipv4': bgpTokens.vpnIpv4,
         'vpn-ipv6': bgpTokens.vpnIpv6,
         'LABEL': MplsCli.topLabelValMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.MplsVpnVrfLocalLabelCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.MplsVpnVrfLocalLabelCmd_noOrDefaultHandler"

RouterBgpVrfMode.addCommandClass( MplsVpnVrfLocalLabelCmd )

#-----------------------------------------------------------------------------------
# "[no|default] route-target (import|export|both)
#  [ evpn [( domain (remote|all) )]|vpn-ipv4|vpn-ipv6 ] <rt> "
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
# Note: Usage of [no|default] route-target ( import|export|both ) <rt> is deprecated.
#
# "[no|default] route-target ( import | export )
#  ( flow-spec-vpn-ipv4 | flow-spec-vpn-ipv6 ) <rt>",
# applicable to "router-bgp-vrf" mode only
#-----------------------------------------------------------------------------------
class RouteTargetImportExportVpnRouteCmd( BgpCmdBaseClass ):
   syntax = 'route-target IMPORT_EXPORT [ VPN_TYPE ] RT'
   noOrDefaultSyntax = 'route-target IMPORT_EXPORT [ VPN_TYPE ] [ RT ]'
   data = {
         'route-target': bgpTokens.routeTarget,
         'IMPORT_EXPORT': EnumMatcher( {
               'import': bgpTokens.routeImport.helpdesc_,
               'export': bgpTokens.export.helpdesc_,
         } ),
         'VPN_TYPE': bgpTokens.RouteTargetVpnTypeExpr,
         'RT': RtSooExtCommCliMatcher( 'Route Target' )
   }
   handler = \
      "RoutingBgpInstanceCliHandler.RouteTargetImportExportVpnRouteCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouteTargetImportExportVpnRouteCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetImportExportVpnRouteCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetImportExportVpnRouteCmd )

class RouteTargetBothRouteCmd( BgpCmdBaseClass ):
   syntax = 'route-target both RT'
   noOrDefaultSyntax = 'route-target both [ RT ]'
   data = {
         'route-target': bgpTokens.routeTarget,
         'both': bgpTokens.bothImportExport,
         'RT': RtSooExtCommCliMatcher( 'Route Target' )
   }
   hidden = True
   handler = "RoutingBgpInstanceCliHandler.RouteTargetBothRouteCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouteTargetBothRouteCmd_noOrDefaultHandler"

RouterBgpVrfMode.addCommandClass( RouteTargetBothRouteCmd )
RouterBgpVrfAfIpMode.addCommandClass( RouteTargetBothRouteCmd )
RouterBgpVrfAfIp6Mode.addCommandClass( RouteTargetBothRouteCmd )

class RouteTargetImportExportFlowspecVpnRouteCmd( BgpCmdBaseClass ):
   syntax = 'route-target IMPORT_EXPORT VPN_TYPE RT'
   noOrDefaultSyntax = 'route-target IMPORT_EXPORT VPN_TYPE [ RT ]'
   data = {
         'route-target': bgpTokens.routeTarget,
         'IMPORT_EXPORT': EnumMatcher( {
               'import': bgpTokens.routeImport.helpdesc_,
               'export': bgpTokens.export.helpdesc_,
         } ),
         'VPN_TYPE': bgpTokens.RouteTargetFlowspecVpnTypeExpr,
         'RT': RtSooExtCommCliMatcher( 'Route Target' )
   }
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouteTargetImportExportVpnRouteCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouteTargetImportExportVpnRouteCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass(
   RouteTargetImportExportFlowspecVpnRouteCmd )

#-----------------------------------------------------------------------------------
# "[no|default] route-target ( import|export ) ( evpn|vpn-ipv4|vpn-ipv6 ) route-map
# <rm> " command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
#-----------------------------------------------------------------------------------
class RouteTargetImportExportRouteMapCmd( BgpCmdBaseClass ):
   syntax = 'route-target IMPORT_EXPORT VPN_TYPE route-map MAP'
   noOrDefaultSyntax = 'route-target IMPORT_EXPORT VPN_TYPE route-map [ MAP ]'
   data = {
         'route-target': bgpTokens.routeTarget,
         'IMPORT_EXPORT': EnumMatcher( {
               'import': bgpTokens.routeImport.helpdesc_,
               'export': bgpTokens.export.helpdesc_,
         } ),
         'VPN_TYPE': bgpTokens.RouteTargetVpnTypeExpr,
         'route-map': bgpTokens.routeMap,
         'MAP': mapNameMatcher,
   }
   handler = \
      "RoutingBgpInstanceCliHandler.RouteTargetImportExportRouteMapCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouteTargetImportExportRouteMapCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetImportExportRouteMapCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetImportExportRouteMapCmd )

#-----------------------------------------------------------------------------------
# "[no|default] route-target export ( vpn-ipv4|vpn-ipv6 ) rcf <rcf>"
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
#-----------------------------------------------------------------------------------
class RouteTargetExportRcfCmd( BgpCmdBaseClass ):
   syntax = ( 'route-target export VPN_TYPE rcf VPN_FUNCTION '
              '[ vrf-route filter-rcf VRF_FUNCTION ]' )
   noOrDefaultSyntax = 'route-target export VPN_TYPE rcf ...'
   data = {
         'route-target': bgpTokens.routeTarget,
         'export': bgpTokens.export.helpdesc_,
         'VPN_TYPE': bgpTokens.RouteTargetRcfMplsVpnTypeExpr,
         'rcf': bgpTokens.rcf,
         'VPN_FUNCTION': RcfCliLib.rcfFunctionMatcher,
         'vrf-route': bgpTokens.rcfVrfRoute,
         'filter-rcf': bgpTokens.rcfFilter,
         'VRF_FUNCTION': RcfCliLib.rcfFunctionMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.RouteTargetExportRcfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouteTargetExportRcfCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetExportRcfCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetExportRcfCmd )

# -----------------------------------------------------------------------------------
# "[no|default] route-target export evpn rcf <rcf>"
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
# -----------------------------------------------------------------------------------
class RouteTargetExportEvpnRcfCmd( BgpCmdBaseClass ):
   syntax = ( 'route-target export evpn rcf VPN_FUNCTION ' )
   noOrDefaultSyntax = 'route-target export evpn rcf ...'
   data = {
         'route-target': bgpTokens.routeTarget,
         'export': bgpTokens.export.helpdesc_,
         'evpn': bgpTokens.evpn,
         'rcf': bgpTokens.rcf,
         'VPN_FUNCTION': RcfCliLib.rcfFunctionMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.RouteTargetExportEvpnRcfCmd_handler"
   noOrDefaultHandler = "RoutingBgpInstanceCliHandler." + \
      "RouteTargetExportEvpnRcfCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetExportEvpnRcfCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetExportEvpnRcfCmd )

# -----------------------------------------------------------------------------------
# "[no|default] route-target import ( vpn-ipv4|vpn-ipv6 ) rcf <rcf>"
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
# -----------------------------------------------------------------------------------
class RouteTargetImportRcfCmd( BgpCmdBaseClass ):
   syntax = ( 'route-target import VPN_TYPE rcf VRF_FUNCTION '
              '[ vpn-route filter-rcf VPN_FUNCTION ]' )
   noOrDefaultSyntax = 'route-target import VPN_TYPE rcf ...'
   data = {
         'route-target': bgpTokens.routeTarget,
         'import': bgpTokens.routeImport.helpdesc_,
         'VPN_TYPE': bgpTokens.RouteTargetRcfMplsVpnTypeExpr,
         'rcf': bgpTokens.rcf,
         'VRF_FUNCTION': RcfCliLib.rcfFunctionMatcher,
         'vpn-route': bgpTokens.rcfVpnRoute,
         'filter-rcf': bgpTokens.rcfFilter,
         'VPN_FUNCTION': RcfCliLib.rcfFunctionMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.RouteTargetImportRcfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouteTargetImportRcfCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetImportRcfCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetImportRcfCmd )

# -----------------------------------------------------------------------------------
# "[no|default] route-target ( import ) ( evpn ) rcf <rcf>"
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
# -----------------------------------------------------------------------------------
class RouteTargetImportEvpnRcfCmd( BgpCmdBaseClass ):
   syntax = ( 'route-target import evpn rcf VRF_FUNCTION ' )
   noOrDefaultSyntax = 'route-target import evpn rcf ...'
   data = {
         'route-target': bgpTokens.routeTarget,
         'import': bgpTokens.routeImport.helpdesc_,
         'evpn': bgpTokens.evpn,
         'rcf': bgpTokens.rcf,
         'VRF_FUNCTION': RcfCliLib.rcfFunctionMatcher,
   }
   handler = "RoutingBgpInstanceCliHandler.RouteTargetImportEvpnRcfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouteTargetImportEvpnRcfCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetImportEvpnRcfCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetImportEvpnRcfCmd )

#-----------------------------------------------------------------------------------
# "[no|default] route-target export ( evpn|vpn-ipv4|vpn-ipv6 ) imported-route"
# command in "router-bgp-vrf" and "router-bgp-vrf-af "mode
#-----------------------------------------------------------------------------------
# pylint: disable-next=useless-object-inheritance
class RouteTargetExportBaseCmd( object ):
   data = {
         'route-target': bgpTokens.routeTarget,
         'export': bgpTokens.export,
         'evpn': bgpTokens.evpn,
         'imported-route': bgpTokens.importedRoute,
         'vpn-ipv4': bgpTokens.vpnIpv4,
         'vpn-ipv6': bgpTokens.vpnIpv6,
   }

class RouteTargetExportEvpnCmd( RouteTargetExportBaseCmd, BgpCmdBaseClass ):
   syntax = 'route-target export evpn imported-route'
   noOrDefaultSyntax = syntax
   data = RouteTargetExportBaseCmd.data.copy()
   handler = "RoutingBgpInstanceCliHandler.RouteTargetExportEvpnCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouteTargetExportEvpnCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetExportEvpnCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetExportEvpnCmd )

class RouteTargetExportAfVpnCmd( RouteTargetExportBaseCmd, BgpCmdBaseClass ):
   syntax = 'route-target export ( vpn-ipv4 | vpn-ipv6 ) imported-route'
   noOrDefaultSyntax = syntax
   data = RouteTargetExportBaseCmd.data.copy()
   handler = "RoutingBgpInstanceCliHandler.RouteTargetExportAfVpnCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouteTargetExportAfVpnCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'VPN_TYPE' ] = args.get( 'vpn-ipv4', args.get( 'vpn-ipv6' ) )

RouterBgpVrfSharedModelet.addCommandClass( RouteTargetExportAfVpnCmd )
RouterBgpAfSharedVrfModelet.addCommandClass( RouteTargetExportAfVpnCmd )

class RouteTargetImportExportEvpnRemoteDomainRouteCmd( BgpCmdBaseClass ):
   syntax = 'route-target IMPORT_EXPORT evpn domain ( remote | all ) RT'
   noOrDefaultSyntax = ( 'route-target IMPORT_EXPORT evpn domain '
                         '[ remote | all ] [ RT ]' )
   # This CLI is available if
   # (1) EvpnL3DciRdRt toggle is on; OR
   # (2) EvpnL3DciWithDps toggle is on and it is the SFE platform (on which the
   #     pathSelectionSupportedGuard takes effect)
   data = {
         'route-target': bgpTokens.routeTarget,
         'IMPORT_EXPORT': EnumMatcher( {
               'import': bgpTokens.routeImport.helpdesc_,
               'export': bgpTokens.export.helpdesc_,
         } ),
         'evpn': ( bgpTokens.evpn if BgpCommonToggleLib.toggleEvpnL3DciRdRtEnabled()
                   else Node( bgpTokens.evpn, guard=pathSelectionSupportedGuard ) ),
         'domain': bgpTokens.domain,
         'remote': bgpTokens.domainRemote,
         'all': bgpTokens.domainAll,
         'RT': RtSooExtCommCliMatcher( 'Route Target' )
   }
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouteTargetImportExportEvpnRemoteDomainRouteCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouteTargetImportExportEvpnRemoteDomainRouteCmd_noOrDefaultHandler"

if ( BgpCommonToggleLib.toggleEvpnL3DciRdRtEnabled() or
     BgpCommonToggleLib.toggleEvpnL3DciWithDpsEnabled() ):
   RouterBgpVrfSharedModelet.addCommandClass(
         RouteTargetImportExportEvpnRemoteDomainRouteCmd )
   RouterBgpAfSharedVrfModelet.addCommandClass(
         RouteTargetImportExportEvpnRemoteDomainRouteCmd )

#---------------------------------------------------------
# The "clear bgp [ipv4|ipv6] access-list counters" command
#---------------------------------------------------------
class ClearBgpAclCounters( CliCommandClass ):
   syntax = 'clear bgp [ ipv4 | ipv6 ] access-list counters'
   data = {
         'clear': Clear.clearKwNode,
         'bgp': 'Bgp',
         'ipv4': AclCli.ipv4KwForServiceAclMatcher,
         'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
         'access-list': AclCli.accessListKwMatcher,
         'counters': bgpTokens.counters,
   }
   handler = "RoutingBgpInstanceCliHandler.ClearBgpAclCounters_handler"

BasicCliModes.EnableMode.addCommandClass( ClearBgpAclCounters )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family ADDR_FAMILY'
#--------------------------------------------------------------------------------
class RouterBgpAddrFamilyCommand( BgpCmdBaseClass ):
   syntax = 'address-family ADDR_FAMILY'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'ADDR_FAMILY': bgpTokens.AddrFamilyExpr,
   }
   handler = "RoutingBgpInstanceCliHandler.RouterBgpAddrFamilyCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpAddrFamilyCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RouterBgpAddrFamilyCommand )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family link-state'
#
# Currently there's no requirement to support BGP Link State in
# non-default VRF. So its supported in default vrf mode only
#--------------------------------------------------------------------------------
class RouterBgpLinkStateCommand( BgpCmdBaseClass ):
   syntax = 'address-family link-state'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'link-state': bgpTokens.linkState,
   }
   handler = "RoutingBgpInstanceCliHandler.RouterBgpLinkStateCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpLinkStateCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RouterBgpLinkStateCommand )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family path-selection'
#
#--------------------------------------------------------------------------------
class RouterBgpDpsCommand( BgpCmdBaseClass ):
   syntax = 'address-family path-selection'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'path-selection': bgpTokens.dps,
   }

   handler = "RoutingBgpInstanceCliHandler.RouterBgpDpsCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpDpsCommand_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RouterBgpDpsCommand )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family ( ipv4 | ipv6 )' in default vrf and vrf modes.
#--------------------------------------------------------------------------------
def afAdapter( mode, args, argsList ):
   args[ 'af' ] = args.get( 'ipv4', args.get( 'ipv6' ) )

class RouterBgpVrfAfCommand( BgpCmdBaseClass ):
   syntax = 'address-family ( ipv4 | ipv6 )'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'ipv4': bgpTokens.ipv4,
         'ipv6': bgpTokens.ipv6,
   }
   handler = "RoutingBgpInstanceCliHandler.RouterBgpVrfAfCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpVrfAfCommand_noOrDefaultHandler"
   adapter = afAdapter

RouterBgpVrfMode.addCommandClass( RouterBgpVrfAfCommand )
RouterBgpDefaultVrfMode.addCommandClass( RouterBgpVrfAfCommand )

#--------------------------------------------------------------------------------
# '[ no | default ] address-family ( ipv4 | ipv6 ) multicast' in default vrf and vrf
# modes.
#--------------------------------------------------------------------------------
class RouterBgpVrfAfMulticastCommand( BgpCmdBaseClass ):
   syntax = 'address-family ( ipv4 | ipv6 ) multicast'
   noOrDefaultSyntax = syntax
   data = {
         'address-family': bgpTokens.addrFamily,
         'ipv4': bgpTokens.ipv4,
         'ipv6': bgpTokens.ipv6,
         'multicast': bgpTokens.multicast,
   }
   handler = \
      "RoutingBgpInstanceCliHandler.RouterBgpVrfAfMulticastCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RouterBgpVrfAfMulticastCommand_noOrDefaultHandler"
   adapter = afAdapter


RouterBgpVrfMode.addCommandClass( RouterBgpVrfAfMulticastCommand )

#--------------------------------------------------------------------------------
# vrf VRFNAME
#--------------------------------------------------------------------------------
class BgpVrfCmd( CliCommandClass ):
   syntax = 'vrf VRFNAME'
   noOrDefaultSyntax = 'vrf VRFNAME ...'
   data = {
         'vrf': bgpTokens.vrf,
         'VRFNAME': DynamicNameMatcher(
               VrfCli.getAllVrfNames,
               helpdesc='VRF name'
         ),
   }
   handler = "RoutingBgpInstanceCliHandler.BgpVrfCmd_handler"
   noOrDefaultHandler = "RoutingBgpInstanceCliHandler.BgpVrfCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( BgpVrfCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath skip peer type ebgp ibgp
#  command, in "router-bgp" mode
#-------------------------------------------------------------------------------
class SetBestpathCompatibleEbgpIbgp( CliCommandClass ):
   syntax = 'bgp bestpath skip peer type ebgp ibgp'
   noOrDefaultSyntax = syntax
   data = {
         'bgp': bgpNode,
         'bestpath': bgpTokens.bestPath,
         'skip': bgpTokens.skipMatcher,
         'peer': bgpTokens.peerBestpathMatcher,
         'type': bgpTokens.typeBestpathMatcher,
         'ebgp': bgpTokens.eBgpMatcher,
         'ibgp': bgpTokens.iBgpMatcher,
   }
   handler = \
      "RoutingBgpInstanceCliHandler.SetBestpathCompatibleEbgpIbgp_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler.SetBestpathCompatibleEbgpIbgp_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetBestpathCompatibleEbgpIbgp_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetBestpathCompatibleEbgpIbgp )

#-------------------------------------------------------------------------------
# "[no|default] bgp bestpath next-hop resolution tunnel nexthop-group ecmp disabled"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd( CliCommandClass ):
   syntax = 'bgp bestpath next-hop resolution tunnel nexthop-group ecmp disabled'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'bestpath': bgpTokens.bestPath,
      'next-hop': bgpTokens.bestPathNexthop,
      'resolution': bgpTokens.bestPathNexthopResolution,
      'tunnel': bgpTokens.bestPathNexthopTunnel,
      'nexthop-group': bgpTokens.bestPathNhg,
      'ecmp': bgpTokens.bestPathEcmp,
      'disabled': bgpTokens.bestPathEcmpDisabled,
   }
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd_handler"
   noHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass(
   BgpBestpathNexthopResolutionNhgTunnelEcmpDisabledCmd )

#---------------------------------------------------------------------------
# "[no|default] bgp bestpath skip aigp-metric" command, in "router-bgp" mode
#---------------------------------------------------------------------------
class SetSkipAigpMetricCmd( CliCommandClass ):
   syntax = 'bgp bestpath skip aigp-metric'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'bestpath': bgpTokens.bestPath,
      'skip': bgpTokens.skipMatcher,
      'aigp-metric': bgpTokens.skipAigp,
   }
   handler = "RoutingBgpInstanceCliHandler.SetSkipAigpMetricCmd_handler"
   noHandler = "RoutingBgpInstanceCliHandler.SetSkipAigpMetricCmd_noHandler"
   defaultHandler = \
      "RoutingBgpInstanceCliHandler.SetSkipAigpMetricCmd_defaultHandler"

RouterBgpSharedModelet.addCommandClass( SetSkipAigpMetricCmd )

#-------------------------------------------------------------------------------
# "[no|default] bgp skip rib-install [ bestpath-selection ] " command, in
# IPv4/v6 unicast AfiSafi modes
#-------------------------------------------------------------------------------
class SkipRibBestpathCommand( CliCommandClass ):
   syntax = 'bgp skip rib-install [ bestpath-selection ]'
   noOrDefaultSyntax = 'bgp skip rib-install ...'
   data = {
      'bgp': bgpNode,
      'skip': 'Skip installing of routes in RIB and (optionally) bestpath selection',
      'rib-install': 'Skip installing BGP routes in RIB',
      'bestpath-selection': 'Skip bestpath selection steps',
   }
   handler = "RoutingBgpInstanceCliHandler.SkipRibBestpathCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SkipRibBestpathCommand_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( SkipRibBestpathCommand )

#--------------------------------------------------------------------------------
# "[no|default] vpn layer-3 export paths ( all | best )" cmd in "router-bgp" mode
#--------------------------------------------------------------------------------
class SetVpnExportPathsMethod( BgpCmdBaseClass ):
   syntax = 'vpn LAYER export paths METHOD'
   # When both layer2 and layer3 are supported change the above line to the
   # following. Also follow the instructions in the definition of LAYER token
   # syntax = 'vpn [ LAYER ] export paths METHOD'
   # noOrDefaultSyntax = 'vpn [ LAYER ] export paths ...'
   noOrDefaultSyntax = 'vpn LAYER export paths ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'vpn': bgpTokens.vpn,
      'LAYER': bgpTokens.layerMatcher,
      'export': bgpTokens.export,
      'paths': bgpTokens.paths,
      'METHOD': bgpTokens.exportMethodMatcher,
   } )
   handler = "RoutingBgpInstanceCliHandler.SetVpnExportPathsMethod_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetVpnExportPathsMethod_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( SetVpnExportPathsMethod )

# ----------------------------------------------------------------------------------
# "[ no | default ] <protocol> instance <instance-name> [ identifier <id> ]
#  [ link connectivity-check disabled ]"
#  under link-state address-family config mode
# ----------------------------------------------------------------------------------
class ConfigLsImportCmd( CliCommandClass ):
   syntax = 'PROTOCOL instance INST [ identifier ID ]'
   noOrDefaultSyntax = 'PROTOCOL [ instance INST ... ]'
   data = {
      'PROTOCOL': EnumMatcher( {
         'isis': 'IS-IS protocol',
      } ),
      'instance': bgpTokens.protocolInstance,
      'INST': bgpTokens.instanceNameRule,
      'identifier': bgpTokens.linkStateIdentifier,
      'ID': bgpTokens.lsIdentifierValueRangeMatcher,
   }
   if BgpCommonToggleLib.toggleBgpLsDisableTwoWayCheckEnabled():
      syntax = 'PROTOCOL instance INST [ identifier ID ] \
                [ link connectivity-check disabled ]'
      data = {
         'PROTOCOL': EnumMatcher( {
            'isis': 'IS-IS protocol',
         } ),
         'instance': bgpTokens.protocolInstance,
         'INST': bgpTokens.instanceNameRule,
         'identifier': bgpTokens.linkStateIdentifier,
         'ID': bgpTokens.lsIdentifierValueRangeMatcher,
         'link': 'nlri link',
         'connectivity-check': 'Check links exist in both directions',
         'disabled': 'Disable two way links check',
      }
   handler = "RoutingBgpInstanceCliHandler.ConfigLsImportCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ConfigLsImportCmd_noOrDefaultHandler"

RouterBgpBaseAfLinkStateMode.addCommandClass( ConfigLsImportCmd )

# ----------------------------------------------------------------------------------
#  " [ no | default  ] <protocol> role propagator "
#      under link-state address-family config mode
# ---------------------------------------------------------------------------------
class ConfigLsRoleCmd( CliCommandClass ):
   syntax = 'PROTOCOL role propagator'
   noOrDefaultSyntax = syntax
   data = {
      'PROTOCOL': EnumMatcher( {
         'isis': 'IS-IS protocol',
      } ),
      "role": "Configure as specific role",
      "propagator": "Configure as propagator",
   }
   handler = "RoutingBgpInstanceCliHandler.ConfigLsRoleCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ConfigLsRoleCmd_noOrDefaultHandler"

RouterBgpBaseAfLinkStateMode.addCommandClass( ConfigLsRoleCmd )

# ----------------------------------------------------------------------------------
# " [ no | default ] bgp peer segments"
#     under link-state address-family config mode (see AID11677)
# ----------------------------------------------------------------------------------
class ConfigLsImportBgpEpeCmd( CliCommandClass ):
   syntax = ( 'bgp peer segments' )
   noOrDefaultSyntax = syntax
   data = {
      'bgp': bgpNode,
      'peer': 'BGP peering segments configuration',
      'segments': 'Enable generation of BGP peering segments'
   }
   handler = "RoutingBgpInstanceCliHandler.ConfigLsImportBgpEpeCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ConfigLsImportBgpEpeCmd_noOrDefaultHandler"

if RoutingLibToggleLib.toggleBgpLsProducerBgpEpeEnabled():
   RouterBgpBaseAfLinkStateMode.addCommandClass( ConfigLsImportBgpEpeCmd )

# ----------------------------------------------------------------------------------
# " [ no | default  ] path-selection " under link-state address-family config mode
# ----------------------------------------------------------------------------------
class ConfigLsImportDpsCmd( CliCommandClass ):
   syntax = ( 'path-selection' )
   noOrDefaultSyntax = syntax
   data = {
      'path-selection': bgpTokens.dps,
   }
   handler = "RoutingBgpInstanceCliHandler.ConfigLsImportDpsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ConfigLsImportDpsCmd_noOrDefaultHandler"

RouterBgpBaseAfLinkStateMode.addCommandClass( ConfigLsImportDpsCmd )

# ----------------------------------------------------------------------------------
# " [ no | default ] path-selection role
#        ( propagator | ( consumer [ propagator ] ) )"
#     under link-state address-family config mode
# ----------------------------------------------------------------------------------
class ConfigLsRoleDpsCmd( CliCommandClass ):
   syntax = ( 'path-selection role '
              '( propagator | ( consumer [ propagator ] ) )' )
   noOrDefaultSyntax = "path-selection role ..."
   data = {
      'path-selection': bgpTokens.dps,
      "role": "Configure as specific role",
      "propagator": "Configure as propagator",
      "consumer": "Configure as consumer",
   }
   handler = "RoutingBgpInstanceCliHandler.ConfigLsRoleDpsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ConfigLsRoleDpsCmd_noOrDefaultHandler"

RouterBgpBaseAfLinkStateMode.addCommandClass( ConfigLsRoleDpsCmd )

class ResolutionRibsCmd( CliCommandClass ):
   syntax = 'next-hop resolution ribs ( RIBS | ( ROUTE_MAP NAME ) )'
   if BgpCommonToggleLib.toggleIpOverDynamicGueTunnelsEnabled():
      syntax = 'next-hop resolution ribs ( [ udp-tunnel ] RIBS |' \
         ' ( ROUTE_MAP NAME ) | udp-tunnel [ system-default ] )'
   noOrDefaultSyntax = 'next-hop resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKw,
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'RIBS': ResolutionRibsExprFactory(),
      'ROUTE_MAP': bgpTokens.resolutionRouteMap,
      'NAME': mapNameMatcher,
      'udp-tunnel': udpTunnelNode,
      'system-default': IpRibLibCliTokens.matcherSystemDefault,
   }
   handler = "RoutingBgpInstanceCliHandler.ResolutionRibsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ResolutionRibsCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( ResolutionRibsCmd )

class ResolutionRibsLuCmd( CliCommandClass ):
   noOrDefaultSyntax = 'next-hop resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKw,
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
   }
   syntax = 'next-hop resolution ribs RIBS_HIDDEN'
   data[ 'RIBS_HIDDEN' ] = ResolutionRibsExprFactory()
   handler = "RoutingBgpInstanceCliHandler.ResolutionRibsLuCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ResolutionRibsLuCmd_noOrDefaultHandler"

RouterBgpAfLabelSharedModelet.addCommandClass( ResolutionRibsLuCmd )

class Resolution6peRibsCmd( CliCommandClass ):
   syntax = 'next-hop 6pe resolution ribs TUNNEL_ONLY_RIBS'
   noOrDefaultSyntax = 'next-hop 6pe resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKw,
      '6pe': '6pe specific next hop resolution policy',
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'TUNNEL_ONLY_RIBS': ResolutionRibsExprFactory(),
   }
   handler = "RoutingBgpInstanceCliHandler.Resolution6peRibsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.Resolution6peRibsCmd_noOrDefaultHandler"

RouterBgpBaseAfIpv6UniMode.addCommandClass( Resolution6peRibsCmd )

#-------------------------------------------------------------------------------
# "[no|default] next-hop ( mpls | vxlan ) resolution ribs RIBS"
# under 'address-family evpn' mode
#-------------------------------------------------------------------------------

class ResolutionMplsRibsCmd( CliCommandClass ):
   noOrDefaultSyntax = 'next-hop mpls resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKw,
      'mpls': 'MPLS specific next hop resolution policy',
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
   }
   syntax = 'next-hop mpls resolution ribs RIBS_HIDDEN'
   data[ 'RIBS_HIDDEN' ] = ResolutionRibsExprFactory()
   handler = "RoutingBgpInstanceCliHandler.ResolutionMplsRibsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ResolutionMplsRibsCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( ResolutionMplsRibsCmd )

class ResolutionVxlanRibsCmd( CliCommandClass ):
   syntax = '''next-hop vxlan resolution ribs IP_ONLY_RIBS'''
   noOrDefaultSyntax = 'next-hop vxlan resolution ribs ...'
   data = {
      'next-hop': bgpTokens.nextHopKw,
      'vxlan': 'Vxlan specific next hop resolution policy',
      'resolution': IpRibLibCliTokens.matcherResolution,
      'ribs': bgpTokens.ribs,
      'IP_ONLY_RIBS': ResolutionRibsExprFactory(),
   }
   handler = \
      "RoutingBgpInstanceCliHandler.ResolutionVxlanRibsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.ResolutionVxlanRibsCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( ResolutionVxlanRibsCmd )

#--------------------------------------------------------------------------------
# "[no|default] next-hop labeled-unicast originate lfib-backup uninstall-delay
#  UNINSTALL_DELAY seconds" cmd in "router-bgp" mode
#--------------------------------------------------------------------------------
lfibBackupNode = Node( matcher=bgpTokens.lfibBackup,
                       guard=nhLuOrigLfibBackupIpFwdSupportedGuard )
class SetNexthopLuOriginateLfibBackupUninstallDelayCmd( BgpCmdBaseClass ):
   syntax = ( 'next-hop labeled-unicast originate lfib-backup uninstall-delay '
              '( disabled | ( UNINSTALL_DELAY seconds ) )' )
   noOrDefaultSyntax = ( 'next-hop labeled-unicast originate lfib-backup '
                         'uninstall-delay ...' )
   data = BgpCmdBaseClass._createSyntaxData( {
      'next-hop': bgpTokens.neighborNextHop,
      'labeled-unicast': bgpTokens.lu,
      'originate': bgpTokens.nextHopLuOriginateKwMatcher,
      'lfib-backup': lfibBackupNode,
      'uninstall-delay': ( "Next hop labeled unicast origination LFIB "
                           "uninstall delay configuration" ),
      'UNINSTALL_DELAY': bgpTokens.lfibDelayRangeMatcher,
      'seconds': bgpTokens.secondsKwMatcher, 
   } )
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetNexthopLuOriginateLfibBackupUninstallDelayCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetNexthopLuOriginateLfibBackupUninstallDelayCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass(
   SetNexthopLuOriginateLfibBackupUninstallDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] next-hop labeled-unicast originate lfib-backup ip-forwarding
#  [ disabled ]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNhLuOrigLfibBackupIpFwdCmd( BgpCmdBaseClass ):
   syntax = ( 'next-hop labeled-unicast originate lfib-backup ip-forwarding '
              '[ disabled ]' )
   noOrDefaultSyntax = ( 'next-hop labeled-unicast originate lfib-backup '
                         'ip-forwarding ...' )
   data = BgpCmdBaseClass._createSyntaxData( {
      'next-hop': bgpTokens.neighborNextHop,
      'labeled-unicast': bgpTokens.lu,
      'originate': bgpTokens.nextHopLuOriginateKwMatcher,
      'lfib-backup': lfibBackupNode,
      'ip-forwarding': bgpTokens.ipForwarding,
   } )
   handler = "RoutingBgpInstanceCliHandler.SetNhLuOrigLfibBackupIpFwdCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetNhLuOrigLfibBackupIpFwdCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNhLuOrigLfibBackupIpFwdCmd )

#-------------------------------------------------------------------------------
# "[no|default] nexthop ( ADDR | ADDR6 ) originate" command, in
# "address-family ipv4|ipv6" mode.
#--------------------------------------------------------------------------------
class SetEpeNexthopOriginateCmd( BgpCmdBaseClass ):
   syntax = 'next-hop ADDR originate'
   noOrDefaultSyntax = 'next-hop ADDR originate ...'
   data = {
         'next-hop': bgpTokens.nextHopKw,
         'ADDR': bgpTokens.ipGenAddrMatcher,
         'originate': bgpTokens.nextHopEpeLuOriginate
   }
   handler = "RoutingBgpInstanceCliHandler.SetEpeNexthopOriginateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetEpeNexthopOriginateCmd_noOrDefaultHandler"

RouterBgpAfLabelSharedModelet.addCommandClass( SetEpeNexthopOriginateCmd )

#-------------------------------------------------------------------------------
# "[no|default] nexthop ( ADDR | ADDR6 ) originate lfib-backup ip-forwarding
# [ disabled ] command, in "address-family ipv4|ipv6" mode.
#--------------------------------------------------------------------------------
class SetEpeNexthopOriginateLfibBackupIpFwdCmd( BgpCmdBaseClass ):
   syntax = 'next-hop ADDR originate lfib-backup ip-forwarding [ disabled ]'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
         'next-hop': bgpTokens.nextHopKw,
         'ADDR': bgpTokens.ipGenAddrMatcher,
         'originate': bgpTokens.nextHopEpeLuOriginate,
         'lfib-backup': bgpTokens.lfibBackup,
         'ip-forwarding': bgpTokens.ipForwarding,
         } )
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetEpeNexthopOriginateLfibBackupIpFwdCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "SetEpeNexthopOriginateLfibBackupIpFwdCmd_noOrDefaultHandler"

RouterBgpAfLabelSharedModelet.addCommandClass(
      SetEpeNexthopOriginateLfibBackupIpFwdCmd )
#-------------------------------------------------------------------------------
# The "optimal-route-reflection position POSITION_NAME" command in
# 'config-router-bgp' mode
#-------------------------------------------------------------------------------
class RouterBgpOrrPositionCmd( BgpNEDCmdBaseClass ):
   syntax = 'optimal-route-reflection position POSITION_NAME'
   noOrDefaultSyntax = 'optimal-route-reflection position POSITION_NAME ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'optimal-route-reflection': bgpTokens.orr,
      'position': bgpTokens.orrPosition,
      'POSITION_NAME': OrrPositionNameExpression,
   } )
   handler = \
      "RoutingBgpInstanceCliHandler.RouterBgpOrrPositionCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpOrrPositionCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RouterBgpOrrPositionCmd )

# The 'igp-prefix IP_PREFIX | IP_PREFIX [ ( before | after ) IP_PREFIX ]'
# command in 'config-router-bgp-orr-position-NAME' mode
class RouterBgpOrrPositionPrefixCmd( BgpNEDCmdBaseClass ):
   syntax = 'igp-prefix PREFIX [ POSITION REF_PREFIX ]'
   noOrDefaultSyntax = 'igp-prefix PREFIX ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'igp-prefix': bgpTokens.orrIgpPrefix,
      'PREFIX': IpGenPrefixMatcher( helpdesc='Prefix to add',
                                    helpdesc4='IPv4 prefix to add',
                                    helpdesc6='IPv6 prefix to add' ),
      'POSITION': EnumMatcher(
        { 'before': 'Add this prefix immediately before the specified prefix',
          'after': 'Add this prefix immediately after the specified prefix' } ),
      'REF_PREFIX': IpGenPrefixMatcher( helpdesc='IPv4 or IPv6 prefix to add new '
                                                 'prefix before or after' ),
   } )
   handler = "RoutingBgpInstanceCliHandler.RouterBgpOrrPositionPrefixCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpOrrPositionPrefixCmd_noOrDefaultHandler"

RouterBgpOrrPositionMode.addCommandClass( RouterBgpOrrPositionPrefixCmd )

# The "[ no | default ] traffic-policy field-set mappings" command in
# "config-router-bgp-vrf" mode.
#-------------------------------------------------------------------------------
class TrafficPolicyFieldSetMappingsCommand( CliCommandClass ):
   syntax = 'traffic-policy field-set mappings'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-policy': bgpTokens.trafficPolicy,
      'field-set': bgpTokens.fieldSet,
      'mappings': bgpTokens.mappings
   }
   handler = \
      "RoutingBgpInstanceCliHandler.TrafficPolicyFieldSetMappingsCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "TrafficPolicyFieldSetMappingsCommand_noOrDefaultHandler"

#-------------------------------------------------------------------------------
# The "[ no | default ] disabled" command in
# "config-traffic-policy-field-set-mappings" mode.
#-------------------------------------------------------------------------------
class TpFieldSetMappingsDisabledCmd( CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': bgpTokens.disabledKwMatcherForTpFieldSetMappings
   }
   handler = "RoutingBgpInstanceCliHandler.TpFieldSetMappingsDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.TpFieldSetMappingsDisabledCmd_noOrDefaultHandler"

#-------------------------------------------------------------------------------
# The "[ no | default ] field-set { ipv4 | ipv6 } FIELD_SET_NAME" command in
# "config-traffic-policy-field-set-mappings" mode.
#-------------------------------------------------------------------------------
def getIpPrefixFieldSetNames( mode, mapCtx='tp', version='4' ):
   assert version in [ '4', '6' ]
   assert mapCtx in [ 'tp', 'vsp' ]
   config = configForVrf( mode.vrfName )
   return getattr( config, mapCtx + 'FieldSetMappingBindingsIpv' + version )

def getTpIpv4PrefixFieldSetNames( mode ):
   return getIpPrefixFieldSetNames( mode, mapCtx='tp', version='4' )

def getTpIpv6PrefixFieldSetNames( mode ):
   return getIpPrefixFieldSetNames( mode, mapCtx='tp', version='6' )

class FieldSetCommand( CliCommandClass ):
   syntax = 'field-set ( ipv4 FSNAME ) | ( ipv6 FSNAME6 )'
   noOrDefaultSyntax = syntax
   data = {
      'field-set': bgpTokens.fieldSet,
      'ipv4': bgpTokens.ipv4,
      'ipv6': bgpTokens.ipv6,
      'FSNAME': PatternMatcher(
         helpname='WORD',
         helpdesc='IP prefix field-set name',
         pattern=r'.+',
      ),
      'FSNAME6': PatternMatcher(
         helpname='WORD',
         helpdesc='IPv6 prefix field-set name',
         pattern=r'.+',
      ),
   }

   handler = "RoutingBgpInstanceCliHandler.FieldSetCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.FieldSetCommand_noOrDefaultHandler"

class TpFieldSetCommand( FieldSetCommand ):
   data = FieldSetCommand.data.copy()
   data.update( {
      'FSNAME': DynamicNameMatcher(
         getTpIpv4PrefixFieldSetNames,
         helpdesc='IP prefix field-set name'
      ),
      'FSNAME6': DynamicNameMatcher(
         getTpIpv6PrefixFieldSetNames,
         helpdesc='IPv6 prefix field-set name'
      ),
   } )

#-------------------------------------------------------------------------------
# The "[ no | default ] community COMM" command in
# "config-traffic-policy-field-set-ipv*" modes or
# "config-vrf-selection-policy-field-set-ipv*" modes
#-------------------------------------------------------------------------------
PEERS_AND_COMMUNITY_ERROR_MSG = ( '\'peers ...\' and \'community ...\' are ' +
                                  'mutually exclusive' )
PEERS_ND_VRF_ERROR_MSG = '\'peers ...\' is not supported in non-default VRF'

class FieldSetCommunityCmd( CliCommandClass ):
   syntax = 'community COMM_VALUE'
   noOrDefaultSyntax = 'community [ COMM_VALUE ]'
   data = {
      'community': bgpTokens.community,
      'COMM_VALUE': CommunityConstExpr,  # communityValue
   }
   handler = "RoutingBgpInstanceCliHandler.FieldSetCommunityCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.FieldSetCommunityCmd_noOrDefaultHandler"

class FieldSetPeersCmd( CliCommandClass ):
   syntax = 'peers ( ibgp | ebgp | ( group { GROUPS } ) )'
   noOrDefaultSyntax = 'peers ...'
   data = { 'peers': 'BGP Peer filters',
            'ibgp': 'Internal BGP peers',
            'ebgp': 'External BGP peers',
            'group': 'Peer groups',
            'GROUPS': peergroupNameMatcher, }
   handler = "RoutingBgpInstanceCliHandler.FieldSetPeersCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.FieldSetPeersCmd_noOrDefaultHandler"

RouterBgpVrfSharedModelet.addCommandClass( TrafficPolicyFieldSetMappingsCommand )
TrafficPolicyFieldSetMappingsMode.addCommandClass(
   TpFieldSetMappingsDisabledCmd )
TrafficPolicyFieldSetMappingsMode.addCommandClass( TpFieldSetCommand )
TpFieldSetIpv4Mode.addCommandClass( FieldSetCommunityCmd )
TpFieldSetIpv6Mode.addCommandClass( FieldSetCommunityCmd )

TpFieldSetIpv4Mode.addCommandClass( FieldSetPeersCmd )
TpFieldSetIpv6Mode.addCommandClass( FieldSetPeersCmd )

# The "[ no | default ] vrf selection policy field-set mappings" command in
# "config-router-bgp-vrf" mode.
# -------------------------------------------------------------------------------
class VrfSelectionPolicyFieldSetMappingsCommand( CliCommandClass ):
   syntax = 'vrf selection policy field-set mappings'
   noOrDefaultSyntax = syntax
   data = {
      'vrf': bgpTokens.vrfVsp,
      'selection': bgpTokens.selectionVsp,
      'policy': bgpTokens.policyVsp,
      'field-set': bgpTokens.fieldSet,
      'mappings': bgpTokens.mappings
   }
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "VrfSelectionPolicyFieldSetMappingsCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "VrfSelectionPolicyFieldSetMappingsCommand_noOrDefaultHandler"

# -------------------------------------------------------------------------------
# The "[ no | default ] disabled" command in
# "config-vrf-selection-policy-field-set-mappings" mode.
# -------------------------------------------------------------------------------
class VspFieldSetMappingsDisabledCmd( CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': bgpTokens.disabledKwMatcherForVspFieldSetMappings
   }
   handler = "RoutingBgpInstanceCliHandler.VspFieldSetMappingsDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "VspFieldSetMappingsDisabledCmd_noOrDefaultHandler"

# -------------------------------------------------------------------------------
# The "[ no | default ] field-set { ipv4 | ipv6 } FIELD_SET_NAME" command in
# "config-vrf-selection-policy-field-set-mappings" mode.
# -------------------------------------------------------------------------------
def getVspIpv4PrefixFieldSetNames( mode ):
   return getIpPrefixFieldSetNames( mode, mapCtx='vsp', version='4' )

def getVspIpv6PrefixFieldSetNames( mode ):
   return getIpPrefixFieldSetNames( mode, mapCtx='vsp', version='6' )

class VspFieldSetCommand( FieldSetCommand ):
   data = FieldSetCommand.data.copy()
   data.update( {
      'FSNAME': DynamicNameMatcher(
         getVspIpv4PrefixFieldSetNames,
         helpdesc='IP prefix field-set name'
      ),
      'FSNAME6': DynamicNameMatcher(
         getVspIpv6PrefixFieldSetNames,
         helpdesc='IPv6 prefix field-set name'
      ),
   } )

RouterBgpDefaultVrfMode.addCommandClass(
   VrfSelectionPolicyFieldSetMappingsCommand )
VrfSelectionPolicyFieldSetMappingsMode.addCommandClass(
   VspFieldSetMappingsDisabledCmd )
VrfSelectionPolicyFieldSetMappingsMode.addCommandClass( VspFieldSetCommand )
VspFieldSetIpv4Mode.addCommandClass( FieldSetCommunityCmd )
VspFieldSetIpv6Mode.addCommandClass( FieldSetCommunityCmd )

# -------------------------------------------------------------------------------
# The "route-flap-damping <RFD POLICY NAME>" command in
# 'config-router-bgp' mode
# -------------------------------------------------------------------------------
class RouterBgpRfdPolicyCmd( BgpCmdBaseClass ):
   syntax = 'route-flap-damping RFD_POLICY_NAME'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
      'route-flap-damping': bgpTokens.rfdKw,
      'RFD_POLICY_NAME': rfdPolicyNameMatcher,
   } )
   handler = "RoutingBgpInstanceCliHandler.RouterBgpRfdPolicyCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RouterBgpRfdPolicyCmd_noOrDefaultHandler"

RouterBgpBaseMode.addCommandClass( RouterBgpRfdPolicyCmd )

# -------------------------------------------------------------------------------
# The "penalty threshold reuse <> suppression <> maximum <>" command in
# 'router-bgp-rfd' mode
# -------------------------------------------------------------------------------
class RfdPenaltyThresholdConfigCmd( BgpCmdBaseClass ):
   syntax = 'penalty threshold reuse REUSE suppression SUPPRESSION maximum CEILING'
   noOrDefaultSyntax = 'penalty threshold'
   data = BgpCmdBaseClass._createSyntaxData( {
      'penalty': bgpTokens.penaltyKw,
      'threshold': bgpTokens.penaltyThresholdKw,
      'reuse': bgpTokens.penaltyThresholdReuseKw,
      'REUSE': bgpTokens.penaltyReuseThresholdMatcher,
      'suppression': bgpTokens.penaltyThresholdSuppressedKw,
      'SUPPRESSION': bgpTokens.penaltySuppressionThresholdMatcher,
      'maximum': bgpTokens.penaltyThresholdMaximumKw,
      'CEILING': bgpTokens.penaltySuppressionThresholdMatcher,
      } )
   handler = "RoutingBgpInstanceCliHandler.RfdPenaltyThresholdConfigCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RfdPenaltyThresholdConfigCmd_noOrDefaultHandler"

RouterBgpRfdPolicyMode.addCommandClass( RfdPenaltyThresholdConfigCmd )

# -------------------------------------------------------------------------------
# The "penalty change <>" command in 'router-bgp-rfd' mode
# -------------------------------------------------------------------------------
class RfdPenaltyChangeConfigCmd( BgpCmdBaseClass ):
   syntax = "penalty change ( PENALTY | 0 )"
   noOrDefaultSyntax = "penalty change"
   data = BgpCmdBaseClass._createSyntaxData( {
      'penalty': bgpTokens.penaltyKw,
      'change': bgpTokens.penaltyChangeKw,
      'PENALTY': bgpTokens.penaltyValueMatcher,
      '0': bgpTokens.penaltyZero,
      } )
   handler = "RoutingBgpInstanceCliHandler.RfdPenaltyChangeConfigCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RfdPenaltyChangeConfigCmd_noOrDefaultHandler"

RouterBgpRfdPolicyMode.addCommandClass( RfdPenaltyChangeConfigCmd )

# -------------------------------------------------------------------------------
# The "penalty withdrawal <>" command in 'router-bgp-rfd' mode
# -------------------------------------------------------------------------------
class RfdPenaltyWithdrawalConfigCmd( BgpCmdBaseClass ):
   syntax = "penalty withdrawal PENALTY"
   noOrDefaultSyntax = "penalty withdrawal"
   data = BgpCmdBaseClass._createSyntaxData( {
      'penalty': bgpTokens.penaltyKw,
      'withdrawal': bgpTokens.penaltyWithdrawalKw,
      'PENALTY': bgpTokens.penaltyValueMatcher,
      } )
   handler = \
      "RoutingBgpInstanceCliHandler.RfdPenaltyWithdrawalConfigCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.RfdPenaltyWithdrawalConfigCmd_noOrDefaultHandler"

RouterBgpRfdPolicyMode.addCommandClass( RfdPenaltyWithdrawalConfigCmd )

# -------------------------------------------------------------------------------
# The "penalty advertisement withdrawal-tracking <>" command in 'router-bgp-rfd'
# mode
# -------------------------------------------------------------------------------
class RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd( BgpCmdBaseClass ):
   syntax = "penalty advertisement withdrawal-tracking ( PENALTY | 0 )"
   noOrDefaultSyntax = "penalty advertisement"
   data = BgpCmdBaseClass._createSyntaxData( {
      'penalty': bgpTokens.penaltyKw,
      'advertisement': bgpTokens.penaltyAdvertisementKw,
      'withdrawal-tracking': bgpTokens.penaltyWithdrawalTrackingKw,
      'PENALTY': bgpTokens.penaltyValueMatcher,
      '0': bgpTokens.penaltyZero,
      } )
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd_noOrDefaultHandler"

RouterBgpRfdPolicyMode.addCommandClass(
      RfdPenaltyAdvertisementWithdrawalTrackingConfigCmd )

# -------------------------------------------------------------------------------
# The "penalty decay half-life <> " command in 'router-bgp-rfd' mode
# -------------------------------------------------------------------------------
class RfdPenaltyDecayHalfLifeConfigCmd( BgpCmdBaseClass ):
   syntax = "penalty decay half-life HALF_LIFE ( seconds | minutes )"
   noOrDefaultSyntax = "penalty decay"
   data = BgpCmdBaseClass._createSyntaxData( {
      'penalty': bgpTokens.penaltyKw,
      'decay': bgpTokens.penaltyDecayKw,
      'half-life': bgpTokens.halfLifeKw,
      'HALF_LIFE': bgpTokens.halfLifeValueMatcher,
      'seconds': bgpTokens.halfLifeSecondsKw,
      'minutes': bgpTokens.halfLifeMinutesKw,
      } )
   handler = \
      "RoutingBgpInstanceCliHandler." + \
      "RfdPenaltyDecayHalfLifeConfigCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "RfdPenaltyDecayHalfLifeConfigCmd_noOrDefaultHandler"

RouterBgpRfdPolicyMode.addCommandClass( RfdPenaltyDecayHalfLifeConfigCmd )

class RfdSuppressionIgnoreConfigCmd( BgpCmdBaseClass ):
   syntax = 'suppression ignored ( { prefix-list ( ipv4 IPV4_PREFIX_LIST )' \
                                               ' | ( ipv6 IPV6_PREFIX_LIST ) } )'
   noOrDefaultSyntax = 'suppression ignored ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'suppression': bgpTokens.suppressionConfigKw,
      'ignored': bgpTokens.suppressionIgnoredKw,
      'prefix-list': Node( bgpTokens.suppressionIgnoredPrefixListKw, maxMatches=2 ),
      'ipv4': singleNode( bgpTokens.ipv4PrefixListKw ),
      'IPV4_PREFIX_LIST': singleNode( prefixListNameMatcher ),
      'ipv6': singleNode( bgpTokens.ipv6PrefixListKw ),
      'IPV6_PREFIX_LIST': singleNode( prefixListNameMatcher ),
      } )

   handler = "RoutingBgpInstanceCliHandler.handlerRfdSuppressionIgnoreConfigCmd"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.noOrDefaultHandlerRfdSuppressionIgnoreConfigCmd"

RouterBgpRfdPolicyMode.addCommandClass( RfdSuppressionIgnoreConfigCmd )

# -------------------------------------------------------------------------------
# The "route-distinguisher" command in 'config-router-bgp' mode
# -------------------------------------------------------------------------------
class RouterBgpRouteDistinguisherCmd( BgpCmdBaseClass ):
   syntax = 'route-distinguisher'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
      'route-distinguisher': KeywordMatcher( 'route-distinguisher',
         helpdesc='Bgp route distinguisher configuration' )
   } )
   handler = 'RoutingBgpInstanceCliHandler.RouterBgpRouteDistinguisherCmd_handler'
   noOrDefaultHandler = 'RoutingBgpInstanceCliHandler.'\
         'RouterBgpRouteDistinguisherCmd_noOrDefaultHandler'

if BgpCommonToggleLib.toggleRdAutoGenEnabled():
   RouterBgpBaseMode.addCommandClass( RouterBgpRouteDistinguisherCmd )

# --------------------------------------------------------------------------------
# [ no | default ] assignment auto range MIN MAX
# --------------------------------------------------------------------------------
def getRdAutoRange( mode, context ):
   return ( context.sharedResult[ 'MIN' ], 65535 )

class RouteDistinguisherAssignmentAutoRangeConfigCmd( BgpCmdBaseClass ):
   syntax = 'assignment auto range MIN MAX'
   noOrDefaultSyntax = 'assignment auto range'
   data = BgpCmdBaseClass._createSyntaxData( {
         'assignment': bgpTokens.rdAssignment,
         'auto': bgpTokens.rdAuto,
         'range': KeywordMatcher( 'range',
            helpdesc='Configure range for assigned value field of rd' ),
         'MIN': Node( IntegerMatcher( 1, 65535,
            helpdesc="Start of range (Default:4096)" ),
            storeSharedResult=True ),
         'MAX': DynamicIntegerMatcher( getRdAutoRange,
            helpdesc="End of range (Default:65535)" ),
   } )
   handler = 'RoutingBgpInstanceCliHandler.'\
      'handlerRouteDistinguisherAssignmentAutoRangeConfigCmd'
   noOrDefaultHandler = 'RoutingBgpInstanceCliHandler.'\
      'noOrDefaultHandlerRouteDistinguisherAssignmentAutoRangeConfigCmd'

# --------------------------------------------------------------------------------
# [ no | default ] assignment auto address-family ( l2evpn | .. )
# --------------------------------------------------------------------------------
class RouteDistinguisherAssignmentAutoAfiConfigCmd( BgpCmdBaseClass ):
   syntax = 'assignment auto AUTO_RD_ADDRESS_FAMILY'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
         'assignment': bgpTokens.rdAssignment,
         'auto': bgpTokens.rdAuto,
         'AUTO_RD_ADDRESS_FAMILY': bgpTokens.AutoRdAdressFamilyExpression,
   } )
   handler = 'RoutingBgpInstanceCliHandler.'\
      'handlerRouteDistinguisherAssignmentAutoAfiConfigCmd'
   noOrDefaultHandler = 'RoutingBgpInstanceCliHandler.'\
      'noOrDefaultHandlerRouteDistinguisherAssignmentAutoAfiConfigCmd'

if BgpCommonToggleLib.toggleRdAutoGenEnabled():
   RouterBgpRouteDistinguisherMode.addCommandClass(
         RouteDistinguisherAssignmentAutoRangeConfigCmd )
   RouterBgpRouteDistinguisherMode.addCommandClass(
         RouteDistinguisherAssignmentAutoAfiConfigCmd )

# --------------------------------------------------------------------------------
# [ no | default ] lfib entry installation skipped
# --------------------------------------------------------------------------------
class LfibEntryInstallationSkippedCmd( BgpCmdBaseClass ):
   syntax = 'lfib entry installation skipped'
   noOrDefaultSyntax = syntax
   data = {
      'lfib': 'LFIB',
      'entry': 'LFIB entry',
      'installation': 'LFIB entry installation',
      'skipped':
      'Skip LFIB entry installation and next hop self route advertisements',
   }
   handler = "RoutingBgpInstanceCliHandler.LfibEntryInstallationSkippedCmd_handler"
   noOrDefaultHandler = "RoutingBgpInstanceCliHandler." + \
      "LfibEntryInstallationSkippedCmd_noOrDefaultHandler"

RouterBgpAfLabelSharedModelet.addCommandClass( LfibEntryInstallationSkippedCmd )

# -------------------------------------------------------------------------------
# Set explicit-null label for locally-terminated routes
# the optional 'implicit-null' token is used for saving config
# or can be used directly for the same effect as 'no' or 'default'
# -------------------------------------------------------------------------------
class LabelLocalTerminationExplicitNullCmd( BgpCmdBaseClass ):
   syntax = 'label local-termination ( explicit-null | implicit-null )'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
      'label': 'Label to be advertised',
      'local-termination': 'Local termination label to be advertised',
      'explicit-null': 'Advertise explicit-null',
      'implicit-null': 'Advertise implicit-null',
   } )
   handler = \
      "RoutingBgpInstanceCliHandler.LabelLocalTerminationExplicitNullCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler." + \
      "LabelLocalTerminationExplicitNullCmd_noOrDefaultHandler"

RouterBgpAfLabelSharedModelet.addCommandClass(
   LabelLocalTerminationExplicitNullCmd )

# -------------------------------------------------------------------------------
# "[no|default] peer-set NAME [ community NUMBER ] command,
#  in "address-family ipv4|ipv6" mode.
# --------------------------------------------------------------------------------
class SetEpePeerSetCmd( BgpCmdBaseClass ):
   syntax = 'peer-set PEER_SET [ community COMM_VALUE ]'
   noOrDefaultSyntax = syntax
   data = BgpCmdBaseClass._createSyntaxData( {
         'peer-set': bgpTokens.neighborPeerSet,
         'PEER_SET': epePeerSetNameMatcher,
         'community': bgpTokens.community,
         'COMM_VALUE': CommunityConstExpr,  # communityValue
         } )
   handler = "RoutingBgpInstanceCliHandler.SetEpePeerSetCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpInstanceCliHandler.SetEpePeerSetCmd_noOrDefaultHandler"

if BgpCommonToggleLib.toggleArBgpNexthopLuPeerSetOriginateEnabled():
   RouterBgpAfIpUniSharedModelet.addCommandClass( SetEpePeerSetCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global bgpVrfConfigDir
   global routingHardwareStatus

   bgpConfig = ConfigMount.mount( entityManager,
                                  'routing/bgp/config',
                                  'Routing::Bgp::Config', 'w' )
   bgpVrfConfigDir = ConfigMount.mount( entityManager, 'routing/bgp/vrf/config',
                                        'Routing::Bgp::VrfConfigDir', 'w' )
   routingHardwareStatus = LazyMount.mount(
         entityManager,
         'routing/hardware/status',
         'Routing::Hardware::Status',
         'r' )

   MplsCli.checkConflictingLabelRangeHook.addExtension(
         checkConflictingLabelRangeForVrfLocalLabel )
   MplsCli.handleUpdatedLabelRangeHook.addExtension(
         handleUpdatedLabelRangeForVrfLocalLabel )
