#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BgpLib
from BgpLib import (
      tcpAoAlgorithmMap,
)
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
import CliMatcher
import CliToken.Bfd as bfdTokens
import CliToken.RoutingBgp as bgpTokens
import CliPlugin.IntfCli as IntfCli # pylint: disable=consider-using-from-import
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
import CliPlugin.RcfCliLib as RcfCliLib # pylint: disable=consider-using-from-import
from CliPlugin import ConfigTagCommon
from CliPlugin.RouteMapCli import (
      LinkBandwidthValueCliMatcher,
      mapNameMatcher,
      prefixListNameMatcher,
)
from CliPlugin.RoutingBgpCli import (
      BgpCmdBaseClass,
      BgpNEDCmdBaseClass,
      bgpNeighborConfig,
      mpDirOptions,
      mpActionOptions,
      mpIncludeOptions,
      NedModeKwMatcher,
      OrrPositionNameExpression,
      PeerCliExpression,
      peergroupNameMatcher,
      epePeerSetNameMatcher,
      ancestorPeerGroupNameMatcher,
      RouterBgpAfIpUniSharedModelet,
      RouterBgpAfIpUniAndMcastSharedModelet,
      RouterBgpAfLabelSharedModelet,
      RouterBgpAfSharedModelet,
      RouterBgpAfUnsupportedPrefixListSharedModelet,
      RouterBgpAfSharedVrfModelet,
      RouterBgpAfVpnModelet,
      RouterBgpBaseAfEvpnMode,
      RouterBgpBaseAfIpUniMode,
      RouterBgpBaseAfIpv6UniMode,
      RouterBgpBaseAfIpMulticastMode,
      RouterBgpBaseAfIpv6MulticastMode,
      RouterBgpBaseAfSrTeMode,
      RouterBgpBaseAfLinkStateMode,
      RouterBgpBaseAfDpsMode,
      RouterBgpBaseMode,
      RouterBgpVrfAfIpMode,
      RouterBgpVrfAfIp6Mode,
      RouterBgpVrfAfIpMulticastMode,
      RouterBgpVrfAfIpv6MulticastMode,
      RouterBgpSharedModelet,
      RouterBgpVrfSharedModelet,
      V4OrPeerGroupCliExpression,
      V4OrV6OrPeerGroupCliExpression,
      V6OrPeerGroupCliExpression,
      V4V6PeerKeyCliExpression,
      nhLuOrigLfibBackupIpFwdSupportedGuard,
      hwSupportedGuard,
      rfdPolicyNameMatcher,
)
from CliPlugin.WanTEShowCli import pathSelectionSupportedGuard
from Intf.IntfRange import intfRangeMatcher
import ConfigMount
import LazyMount
import Tac
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import ReversibleSecretCli
from TypeFuture import TacLazyType
from Toggles import (
   BgpCommonToggleLib,
   RoutingLibToggleLib,
   RcfLibToggleLib,
   ConfigTagToggleLib,
   MvpnLibToggleLib,
)

bgpConfig = None
sharedSecretProfileConfig = None

PeerAfStateEnum = TacLazyType( "Routing::Bgp::PeerAFState" )
MetricOutStateEnum = TacLazyType( "Routing::Bgp::MetricOutState" )
TcpAoAlgorithmEnum = TacLazyType( "Arnet::TcpAoAlgorithm" )
AncestorPgConfigType = TacLazyType( "Routing::Bgp::AncestorPeerGroupConfig" )
RoutingBgpCliHandler = CliDynamicPlugin( "RoutingBgpCliHandler" )

# Common helper methods

def validatePeer( mode, key ):
   SUPPORTED_INTFS = [ 'Ethernet', 'Vlan', 'Vxlan', 'Loopback', 'Internal', 'Fabric',
                       'Management', 'Port-Channel', 'Test', 'Tunnel', 'Switch' ]

   assert key is not None
   if key.type == 'peerGroup':
      if key not in bgpConfig.neighborConfig:
         return
   if key.type == 'peerIpv6' and key.v6Addr.isLinkLocal:
      mode.addErrorAndStop( "Link-local address must use <addr>%<ifname> syntax" )
   if key.type == 'peerIpv6Ll' and \
      not any( tag in key.llIntf for tag in SUPPORTED_INTFS ):
      # pylint: disable-next=consider-using-f-string
      mode.addErrorAndStop( "Link-local interface, '%s', must be one of: %s"
                            % ( key.llIntf,
                                ', '.join( SUPPORTED_INTFS ) ) )
   addr = None
   if key.type == 'peerIpv4':
      addr = Tac.Value( 'Arnet::IpAddr', stringValue=key.v4Addr )
   elif key.type == 'peerIpv6':
      addr = key.v6Addr
   if addr is not None and addr.isZero:
      # pylint: disable-next=consider-using-f-string
      mode.addErrorAndStop( "%s is an invalid peer address" % str( addr ) )

def neighborAfActivateHelper( mode, peer, afStateEnum ):
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   attr = BgpLib.peerConfigAttrsAfMap[ 'af' ].get( mode.addrFamily )
   if attr:
      setattr( config, attr, afStateEnum )
   else:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Invalid address family %s" % peer )
   RoutingBgpCliHandler.delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-accepted-routes NUM_ACCEPTED_ROUTES
#    [warning-limit THRESHOLD]"
#-----------------------------------------------------------------------------
class SetNeighborMaxAcceptedRoutesCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER maximum-accepted-routes NUM_ACCEPTED_ROUTES '
              '[ warning-limit THRESHOLD ]' )
   noOrDefaultSyntax = 'neighbor PEER maximum-accepted-routes ...'
   data = { 'neighbor': bgpTokens.neighbor,
            'PEER': PeerCliExpression,
            'maximum-accepted-routes': bgpTokens.maxAcceptedRoutes,
            'NUM_ACCEPTED_ROUTES': bgpTokens.numMaxRoutesRangeMatcher,
            'warning-limit': bgpTokens.warningLimit,
            'THRESHOLD': bgpTokens.thresholdRangeMatcher
          }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborMaxAcceptedRoutesCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborMaxAcceptedRoutesCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborMaxAcceptedRoutesCmd )
for modelet in [ RouterBgpAfIpUniSharedModelet, RouterBgpBaseAfEvpnMode ]:
   modelet.addCommandClass( SetNeighborMaxAcceptedRoutesCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-advertised-routes NUM_ADV_ROUTES
#    [warning-limit THRESHOLD [percent] ]"
#-----------------------------------------------------------------------------
class SetNeighborMaxAdvRoutesCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor PEER maximum-advertised-routes NUM_ADV_ROUTES '
              '[ warning-limit ( ( THRESHOLD percent ) | ABS_THRESHOLD ) ]' )
   noOrDefaultSyntax = 'neighbor PEER maximum-advertised-routes ...'
   data = { 'neighbor': bgpTokens.neighbor,
            'PEER': PeerCliExpression,
            'maximum-advertised-routes': bgpTokens.maxAdvRoutes,
            'NUM_ADV_ROUTES': bgpTokens.numMaxRoutesRangeMatcher,
            'warning-limit': bgpTokens.warningLimit,
            'THRESHOLD': bgpTokens.thresholdPercentageMatcher,
            'percent': bgpTokens.percent,
            'ABS_THRESHOLD': bgpTokens.thresholdRangeMatcher
          }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborMaxAdvRoutesCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborMaxAdvRoutesCmd_noOrDefaultHandler"
   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ABS_THRESHOLD' in args:
         args[ 'THRESHOLD' ] = args.pop( 'ABS_THRESHOLD' )

RouterBgpSharedModelet.addCommandClass( SetNeighborMaxAdvRoutesCmd )
for modelet in [ RouterBgpAfSharedModelet, RouterBgpAfLabelSharedModelet,
                 RouterBgpBaseAfSrTeMode ]:
   modelet.addCommandClass( SetNeighborMaxAdvRoutesCmd )

#-------------------------------------------------------------------------------
# "[no|default] missing-policy address-family all [include { community-list |
# sub-route-map | prefix-list } ] direction [ in | out ] action" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborMissingPolicyCmd( BgpCmdBaseClass ):
   includes_ = mpIncludeOptions()
   syntax = (
      # pylint: disable-next=consider-using-f-string
      'neighbor PEER missing-policy address-family all [ include { %s } ] '
      'direction DIR action ACT' % ' | '.join( includes_ ) )
   noOrDefaultSyntax = syntax.replace( 'ACT', '...' )

   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'missing-policy': bgpTokens.missingPolicy,
      'address-family': bgpTokens.mpAddrFamily,
      'all': bgpTokens.mpAddrFamilyAll,
      'include': bgpTokens.mpInclude,
      'sub-route-map': CliCommand.singleNode( bgpTokens.mpSubRouteMap ),
      'prefix-list': CliCommand.singleNode( bgpTokens.mpPrefixList ),
      'community-list': CliCommand.singleNode( bgpTokens.mpCommList ),
      'direction': bgpTokens.mpDirection,
      'DIR': CliMatcher.EnumMatcher( mpDirOptions ),
      'action': bgpTokens.mpAction,
      'ACT': CliMatcher.EnumMatcher( mpActionOptions )
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborMissingPolicyCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborMissingPolicyCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborMissingPolicyCmd )

#-------------------------------------------------------------------------------
# "[no|default] missing-policy [include { community-list | prefix-list |
# sub-route-map }] direction [ in | out ] action" command, in router bgp af mode.
#-------------------------------------------------------------------------------
class SetNeighborMissingPolicyAfCmd( BgpCmdBaseClass ):
   includes_ = mpIncludeOptions()
   # pylint: disable-next=consider-using-f-string
   syntax = ( 'neighbor PEER missing-policy [ include { % s } ] direction DIR '
              'action ACT' % ( ' | '.join( includes_ ) ) )
   noOrDefaultSyntax = syntax.replace( 'ACT', '...' )

   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'missing-policy': bgpTokens.missingPolicy,
      'include': bgpTokens.mpInclude,
      'sub-route-map': CliCommand.singleNode( bgpTokens.mpSubRouteMap ),
      'prefix-list': CliCommand.singleNode( bgpTokens.mpPrefixList ),
      'community-list': CliCommand.singleNode( bgpTokens.mpCommList ),
      'direction': bgpTokens.mpDirection,
      'DIR': CliMatcher.EnumMatcher( mpDirOptions ),
      'action': bgpTokens.mpAction,
      'ACT': CliMatcher.EnumMatcher( mpActionOptions )
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborMissingPolicyAfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborMissingPolicyAfCmd_noOrDefaultHandler"

for modelet in [ RouterBgpAfSharedModelet, RouterBgpAfLabelSharedModelet,
                 RouterBgpBaseAfSrTeMode, RouterBgpBaseAfLinkStateMode ]:
   modelet.addCommandClass( SetNeighborMissingPolicyAfCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR local-as AS_NUM no-prepend \
#  replace-as [fallback]"
# command, in "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborLocalAsCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER local-as '
              '( AS_NUM no-prepend replace-as [ fallback ] ) | disabled' )
   noOrDefaultSyntax = 'neighbor PEER local-as ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'local-as': bgpTokens.localAs,
         'AS_NUM': bgpTokens.AsNumCliExpr,
         'no-prepend': bgpTokens.noPrepend,
         'replace-as': bgpTokens.replaceLocalAs,
         'fallback': bgpTokens.fallbackAs,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLocalAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLocalAsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLocalAsCmd )

#----------------------------------------------------------------------------
# "[no|default] neighbor <addr> activate" command
#----------------------------------------------------------------------------
deactivateNedMatcher = NedModeKwMatcher(
      bgpTokens.deactivate.keyword_,
      bgpTokens.deactivate.helpdesc_
)

class RouterBgpAfActivateCommand( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER ( activate | deactivate )'
   noOrDefaultSyntax = syntax

   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'activate': bgpTokens.activate,
      'deactivate': deactivateNedMatcher
   }
   handler = "RoutingBgpNeighborCliHandler.RouterBgpAfActivateCommand_handler"
   noHandler = "RoutingBgpNeighborCliHandler.RouterBgpAfActivateCommand_noHandler"
   defaultHandler = \
      "RoutingBgpNeighborCliHandler.RouterBgpAfActivateCommand_defaultHandler"

modes = [
   RouterBgpBaseAfIpUniMode,
   RouterBgpBaseAfIpMulticastMode,
   RouterBgpBaseAfIpv6MulticastMode,
   RouterBgpAfLabelSharedModelet,
   RouterBgpBaseAfEvpnMode,
   RouterBgpBaseAfSrTeMode,
   RouterBgpBaseAfLinkStateMode,
   RouterBgpBaseAfDpsMode,
   RouterBgpVrfAfIpMode,
   RouterBgpVrfAfIp6Mode,
   RouterBgpVrfAfIpMulticastMode,
   RouterBgpVrfAfIpv6MulticastMode,
]
for m in modes:
   m.addCommandClass( RouterBgpAfActivateCommand )

class NeighborReflectorClientOrrPositionCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER route-reflector-client ' \
            'optimal-route-reflection position ' \
            '( peer-address | POSITION_NAME )'
   noOrDefaultSyntax = 'neighbor PEER route-reflector-client ' \
                       'optimal-route-reflection ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-reflector-client': bgpTokens.routeReflectorClient,
      'optimal-route-reflection': bgpTokens.orr,
      'position': bgpTokens.orrPosition,
      'peer-address': bgpTokens.orrPositionPeer,
      'POSITION_NAME': OrrPositionNameExpression,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborReflectorClientOrrPositionCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborReflectorClientOrrPositionCmd_noOrDefaultHandler"

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-reflector-client \
#     optimal-route-reflection position peer-address | POSITION-NAME"
# command, in "address-family ipv4/6" mode.
#-------------------------------------------------------------------------------
class IpUniNeighborReflectorClientOrrPositionCmd(
      NeighborReflectorClientOrrPositionCmd ):
   data = NeighborReflectorClientOrrPositionCmd.data.copy()

for m in ( RouterBgpBaseAfIpUniMode, RouterBgpBaseAfIpv6UniMode ):
   m.addCommandClass( IpUniNeighborReflectorClientOrrPositionCmd )

# -----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR forwarding failover trigger session" command,
# in "config-bgp-af" mod
# -----------------------------------------------------------------------------
class NeighborFwdFailoverTriggerSessionCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER forwarding failover trigger session [disabled]'
   noOrDefaultSyntax = 'neighbor PEER forwarding failover trigger session ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'forwarding': bgpTokens.forwarding,
      'failover': bgpTokens.fwdFailover,
      'trigger': bgpTokens.fwdFailoverTrigger,
      'session': bgpTokens.fwdFailoverTriggerSession,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.NeighborFwdFailoverTriggerSessionCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborFwdFailoverTriggerSessionCmd_noOrDefaultHandler"

for m in [ RouterBgpAfIpUniSharedModelet, ]:
   m.addCommandClass( NeighborFwdFailoverTriggerSessionCmd )

#-------------------------------------------------------------------------------
# 'neighbor ( ( PEER activate 6pe [ ipv6-unicast receive ] ) )'
# in router bgp, address-family ipv6
#-------------------------------------------------------------------------------
class RouterBgpNeighborActivate6peCommand( CliCommand.CliCommandClass ):
   syntax = '''neighbor ( ( PEER ( activate | deactivate ) )
                        | ( 6PE_PEER ( ( activate 6pe [ ipv6-unicast receive ] )
                                     | deactivate ) ) )'''
   noOrDefaultSyntax = 'neighbor PEER ...'

   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      '6PE_PEER': V4OrPeerGroupCliExpression,
      'activate': bgpTokens.activate,
      'deactivate': deactivateNedMatcher,
      '6pe': bgpTokens.sixPe,
      'ipv6-unicast': 'Activate IPv6 Unicast for the neighbor',
      'receive': 'Activate IPv6 Unicast receive side only',
   }

   handler = \
      "RoutingBgpNeighborCliHandler.RouterBgpNeighborActivate6peCommand_handler"
   noHandler = \
      "RoutingBgpNeighborCliHandler.RouterBgpNeighborActivate6peCommand_noHandler"
   defaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "RouterBgpNeighborActivate6peCommand_defaultHandler"

RouterBgpBaseAfIpv6UniMode.addCommandClass( RouterBgpNeighborActivate6peCommand )

#-------------------------------------------------------------------------------
# '[no|default] neighbor PEER activate ( layer-2 | layer-3 )
#     under 'address-family evpn' mode'
#-------------------------------------------------------------------------------
class RouterBgpNeighborActivateEvpnCommand( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER activate ( layer-2 | layer-3 )'
   noOrDefaultSyntax = syntax

   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'activate': bgpTokens.activate,
      'layer-2': bgpTokens.layer2,
      'layer-3': bgpTokens.layer3,
   }

   hidden = True
   handler = \
      "RoutingBgpNeighborCliHandler.RouterBgpNeighborActivateEvpnCommand_handler"
   noHandler = \
      "RoutingBgpNeighborCliHandler.RouterBgpNeighborActivateEvpnCommand_noHandler"
   defaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "RouterBgpNeighborActivateEvpnCommand_defaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      args[ 'routeType' ] = args.get( 'layer-2', args.get( 'layer-3' ) )

RouterBgpBaseAfEvpnMode.addCommandClass( RouterBgpNeighborActivateEvpnCommand )

#-------------------------------------------------------------------------------
# '[no|default] neighbor PEER domain remote
#     under 'address-family evpn' mode'
#-------------------------------------------------------------------------------
class RouterBgpNeighborDomainRemoteEvpnCommand( BgpCmdBaseClass ):
   syntax = 'neighbor PEER domain remote'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'domain': bgpTokens.domain,
      'remote': bgpTokens.domainRemote
   }
   handler = \
      "RoutingBgpNeighborCliHandler.RouterBgpNeighborDomainRemoteEvpnCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "RouterBgpNeighborDomainRemoteEvpnCommand_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass( RouterBgpNeighborDomainRemoteEvpnCommand )

class RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor PEER next-hop-self received-evpn-routes '
              'route-type ROUTE_TYPES [ inter-domain ]' )
   noOrDefaultSyntax = syntax
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop-self': bgpTokens.nextHopSelf,
         'received-evpn-routes': bgpTokens.receivedEvpnRoutes,
         'route-type': bgpTokens.evpnRouteTypeNhSelf,
         'ROUTE_TYPES': bgpTokens.EvpnRouteTypeExpression,
         'inter-domain': bgpTokens.interDomain
   }
   handler = \
      "RoutingBgpNeighborCliHandler." + \
      "RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd_noOrDefaultHandler"

RouterBgpBaseAfEvpnMode.addCommandClass(
      RouterBgpNeighborRouteTypeNextHopSelfEvpnCmd )

#
#----------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> timers <keepalive> <hold>" command,
# in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetNeighborPeerTimersCmd( BgpNEDCmdBaseClass ):
   syntax = '''neighbor PEER timers (
      ( KEEPALIVE ( HOLD_TIME | 0 ) [ min-hold-time MIN_HOLD_TIME ]
                           [ send-failure hold-time SEND_FAILURE_HOLD_TIME ] ) |
      ( min-hold-time MIN_HOLD_TIME
                           [ send-failure hold-time SEND_FAILURE_HOLD_TIME ] ) |
      ( send-failure hold-time SEND_FAILURE_HOLD_TIME )
   )'''

   noOrDefaultSyntax = 'neighbor PEER timers ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'timers': bgpTokens.timers,
      'bgp': bgpTokens.bgpTimers,
      'KEEPALIVE': bgpTokens.keepaliveTimeRangeMatcher,
      'HOLD_TIME': bgpTokens.holdTimeRangeMatcher,
      '0': bgpTokens.holdForever,
      'min-hold-time': bgpTokens.bgpMinHoldTime,
      'MIN_HOLD_TIME': bgpTokens.holdTimeRangeMatcher,
      'send-failure': bgpTokens.bgpSendFailure,
      'hold-time': bgpTokens.bgpSendFailureHoldTime,
      'SEND_FAILURE_HOLD_TIME': bgpTokens.sendFailureHoldTimeRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborPeerTimersCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborPeerTimersCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborPeerTimersCmd )

#----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR idle-restart-timer IDLE_RESTART_TIMER" command,
# in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetIdleRestartTimerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER idle-restart-timer ( IDLE_RESTART_TIMER | disabled )'
   noOrDefaultSyntax = 'neighbor PEER idle-restart-timer ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'idle-restart-timer': bgpTokens.idleRestartTimer,
      'IDLE_RESTART_TIMER': bgpTokens.idleRestartTimerRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetIdleRestartTimerCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetIdleRestartTimerCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetIdleRestartTimerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR remote-as AS_NUM" command,
# in "config-bgp" mode
#-------------------------------------------------------------------------------
class SetNeighborAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER remote-as ( AS_NUM | disabled )'
   noOrDefaultSyntax = 'neighbor PEER remote-as ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'remote-as': bgpTokens.remoteAs,
      'AS_NUM': bgpTokens.AsNumCliExpr,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborAsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborAsCmd )

#---------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-map <name> [ in|out ]" command, in
# "config-bgp" mode
#--------------------------------------------------------------------------------

rmInNedMatcher = NedModeKwMatcher( bgpTokens.rmIn.keyword_,
                                   bgpTokens.rmIn.helpdesc_ )
rmOutNedMatcher = NedModeKwMatcher( bgpTokens.rmOut.keyword_,
                                    bgpTokens.rmOut.helpdesc_ )
class SetNeighborRouteMapBaseCmd( BgpNEDCmdBaseClass ):
   handler = "RoutingBgpNeighborCliHandler.SetNeighborRouteMapBaseCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborRouteMapBaseCmd_noOrDefaultHandler"
   @staticmethod
   def adapter( mode, args, argsList ):
      if 'in' in args or 'NED_IN' in args:
         args[ 'inout' ] = 'in'
      elif 'out' in args or 'NED_OUT' in args:
         args[ 'inout' ] = 'out'

class SetNeighborRouteMapCmd( SetNeighborRouteMapBaseCmd ):
   syntax = '''neighbor PEER route-map
            ( ( MAP_NAME ( in | out ) )
            | ( ( NED_IN | NED_OUT ) disabled ) )'''
   noOrDefaultSyntax = 'neighbor PEER route-map [ MAP_NAME ] ( in | out )'
   data = SetNeighborRouteMapBaseCmd._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-map': bgpTokens.routeMap,
      'MAP_NAME': mapNameMatcher,
      'in': bgpTokens.rmIn,
      'out': bgpTokens.rmOut,
      'NED_IN': rmInNedMatcher,
      'NED_OUT': rmOutNedMatcher,
   } )

# SR-TE afiSafi does not supports outbound route-map if SrTeDpsPolicy toggle is off,
# so create separate command
class SetNeighborRouteMapSrTeCmd( SetNeighborRouteMapBaseCmd ):
   syntax = 'neighbor PEER route-map ( MAP_NAME in ) | ( NED_IN disabled )'
   noOrDefaultSyntax = 'neighbor PEER route-map [ MAP_NAME ] in'
   data = SetNeighborRouteMapBaseCmd._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-map': bgpTokens.routeMap,
      'MAP_NAME': mapNameMatcher,
      'in': bgpTokens.rmIn,
      'NED_IN': rmInNedMatcher,
   } )

neighborRouteMapModes = [ RouterBgpSharedModelet,
                          RouterBgpAfSharedModelet,
                          RouterBgpAfLabelSharedModelet,
                          RouterBgpBaseAfSrTeMode ]

for m in neighborRouteMapModes:
   m.addCommandClass( SetNeighborRouteMapCmd )

#---------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR prefix-list <name> [ in|out ]" command,
# in "config-bgp" mode and address-family block
#--------------------------------------------------------------------------------
prefixListNedIn = NedModeKwMatcher(
      bgpTokens.prefixListIn.keyword_,
      bgpTokens.prefixListIn.helpdesc_,
)
prefixListNedOut = NedModeKwMatcher(
      bgpTokens.prefixListOut.keyword_,
      bgpTokens.prefixListOut.helpdesc_,
)

class SetNeighborPrefixListCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER prefix-list ( PREFIX_LIST_NAME ( in | out ) ) | '
              '( ( NED_IN | NED_OUT ) disabled )' )
   noOrDefaultSyntax = ( 'neighbor PEER prefix-list [ PREFIX_LIST_NAME ] '
                         '( in | out ) ...' )
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'prefix-list': bgpTokens.prefixList,
         'PREFIX_LIST_NAME': prefixListNameMatcher,
         'in': bgpTokens.prefixListIn,
         'out': bgpTokens.prefixListOut,
         'NED_IN': prefixListNedIn,
         'NED_OUT': prefixListNedOut,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborPrefixListCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborPrefixListCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'in' in args or 'NED_IN' in args:
         args[ 'inout' ] = 'prefixListIn'
      elif 'out' in args or 'NED_OUT' in args:
         args[ 'inout' ] = 'prefixListOut'

RouterBgpAfIpUniAndMcastSharedModelet.addCommandClass( SetNeighborPrefixListCmd )

class SetUnsupportedNeighborPrefixListCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER prefix-list ( PREFIX_LIST_NAME ( in | out ) ) | '
              '( ( NED_IN | NED_OUT ) disabled )' )
   noOrDefaultSyntax = ( 'neighbor PEER prefix-list [ PREFIX_LIST_NAME ] '
                         '( in | out ) ...' )
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'prefix-list': bgpTokens.prefixList,
         'PREFIX_LIST_NAME': prefixListNameMatcher,
         'in': bgpTokens.prefixListIn,
         'out': bgpTokens.prefixListOut,
         'NED_IN': prefixListNedIn,
         'NED_OUT': prefixListNedOut,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.SetUnsupportedNeighborPrefixListCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetUnsupportedNeighborPrefixListCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'in' in args or 'NED_IN' in args:
         args[ 'inout' ] = 'prefixListIn'
      elif 'out' in args or 'NED_OUT' in args:
         args[ 'inout' ] = 'prefixListOut'

RouterBgpAfUnsupportedPrefixListSharedModelet.addCommandClass(
   SetUnsupportedNeighborPrefixListCmd )

#-----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR import-localpref LOCAL_PREF" command,
# in "config-bgp" mode
#-----------------------------------------------------------------------------------
class SetNeighborImportLocalPrefCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER import-localpref LOCAL_PREF'
   noOrDefaultSyntax = 'neighbor PEER import-localpref ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'import-localpref': bgpTokens.importLocalPref,
      'LOCAL_PREF': bgpTokens.localPrefMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborImportLocalPrefCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborImportLocalPrefCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborImportLocalPrefCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR export-localpref LOCAL_PREF" command, in
# "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborExportLocalPrefCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER export-localpref LOCAL_PREF'
   noOrDefaultSyntax = 'neighbor PEER export-localpref ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'export-localpref': bgpTokens.exportLocalPref,
      'LOCAL_PREF': bgpTokens.localPrefMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborExportLocalPrefCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborExportLocalPrefCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborExportLocalPrefCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR graceful-restart" command, in
# "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborGracefulRestartCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER graceful-restart [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER graceful-restart ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'graceful-restart': bgpTokens.gracefulRestart,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborGracefulRestartCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborGracefulRestartCmd_noOrDefaultHandler"

for m in [ RouterBgpSharedModelet, RouterBgpAfSharedModelet,
           RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetNeighborGracefulRestartCmd )

if BgpCommonToggleLib.toggleHandlePfLossEnabled():
   RouterBgpBaseAfDpsMode.addCommandClass( SetNeighborGracefulRestartCmd )

#-----------------------------------------------------------------------------
# With toggle disabled:
#
# "[no|default] neighbor GROUP | ADDR graceful-restart-helper
#  [ restart-time < SECONDS > ] [ long-lived ]" command, in "config-bgp" mode
#
# With toggle enabled:
#
# "[no|default] neighbor GROUP | ADDR graceful-restart-helper
#  [ restart-time < NUMBER > [ minutes | hours | days ] ]
#  [ stale-route route-map | long-lived ]" command, in "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborGrHelperCmd( BgpNEDCmdBaseClass ):
   syntax = '''
      neighbor PEER graceful-restart-helper
      [ long-lived
      | ( EXT_RESTART_TIME
         [ long-lived
         | ( stale-route route-map MAP_NAME [ session-failure GR_OPTIONAL ] ) ]
      ) ] [ disabled ]'''
   noOrDefaultSyntax = 'neighbor PEER graceful-restart-helper [ long-lived | '\
                       'restart-time ] ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'graceful-restart-helper': bgpTokens.grHelper,
         'EXT_RESTART_TIME': bgpTokens.GrHelperRestartTimeExp,
         'long-lived': bgpTokens.llgrHelper,
         'stale-route': bgpTokens.grHelperStaleRoute,
         'route-map': bgpTokens.routeMap,
         'MAP_NAME': mapNameMatcher,
         'session-failure': bgpTokens.sessionFailure,
         'GR_OPTIONAL': bgpTokens.GrHelperRmGrOptionalExp
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborGrHelperCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborGrHelperCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborGrHelperCmd )

# -----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR graceful-restart-helper
#  stale-route route-map" command, in "router-bgp-af" mode
# -----------------------------------------------------------------------------
class SetNeighborGrHelperStaleRmAfCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER graceful-restart-helper stale-route route-map MAP_NAME' \
            ' [ session-failure GR_OPTIONAL ]'
   noOrDefaultSyntax = \
         'neighbor PEER graceful-restart-helper stale-route route-map ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'graceful-restart-helper': bgpTokens.grHelper,
         'stale-route': bgpTokens.grHelperStaleRoute,
         'route-map': bgpTokens.routeMap,
         'MAP_NAME': mapNameMatcher,
         'session-failure': bgpTokens.sessionFailure,
         'GR_OPTIONAL': bgpTokens.GrHelperRmGrOptionalExp } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborGrHelperStaleRmAfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborGrHelperStaleRmAfCmd_noOrDefaultHandler"

for m in [ RouterBgpAfIpUniSharedModelet, RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetNeighborGrHelperStaleRmAfCmd )

if BgpCommonToggleLib.toggleHandlePfLossEnabled():
   RouterBgpBaseAfDpsMode.addCommandClass( SetNeighborGrHelperStaleRmAfCmd )
   RouterBgpBaseAfSrTeMode.addCommandClass( SetNeighborGrHelperStaleRmAfCmd )
   RouterBgpBaseAfEvpnMode.addCommandClass( SetNeighborGrHelperStaleRmAfCmd )
   RouterBgpBaseAfLinkStateMode.addCommandClass( SetNeighborGrHelperStaleRmAfCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR metric-out <metric>"
#  command, in "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborMetricOutCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER metric-out ( METRIC | disabled )'
   noOrDefaultSyntax = 'neighbor PEER metric-out ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'metric-out': bgpTokens.metricOut,
      'METRIC': bgpTokens.metricRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborMetricOutCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborMetricOutCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborMetricOutCmd )

sharedSecretProfileMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: sharedSecretProfileConfig.profile,
   'Shared-secret profile name' )

tcpAoAlgorithms = {
   'hmac-sha1-96': 'HMAC-SHA-1-96 algorithm',
   'aes-128-cmac-96': 'AES-128-CMAC-96 algorithm',
   'hmac-sha-256': 'HMAC-SHA-256-128 algorithm',
}

class BgpTcpSharedSecretProfileExpression( CliCommand.CliExpression ):
   expression = "shared-secret profile PROFILE_NAME algorithm ALGORITHM"
   data = {
      'shared-secret': 'Shared secret profile',
      'profile': 'Shared secret profile',
      'PROFILE_NAME': sharedSecretProfileMatcher,
      'algorithm': 'Cryptographic algorithm to use',
      'ALGORITHM': CliMatcher.EnumMatcher( tcpAoAlgorithms ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      profile = args.get( 'PROFILE_NAME', '' )
      algorithm = tcpAoAlgorithmMap[ args.get( 'ALGORITHM', 'none' ) ]
      if ( algorithm != 'noAlgorithm' and
           getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent ):
         mode.addWarning( 'TCP-AO is only supported in multi-agent mode' )
      defaultSecret = Tac.Value( 'SecretCliLib::Secret' )
      args[ 'AUTH_KEY' ] = Tac.Value( 'Routing::Bgp::BgpTcpAuthKey',
                                      defaultSecret,
                                      profile, algorithm )

#-----------------------------------------------------------------------------
# "[no|default] neighbor PEER password ( SECRET |
#               shared-secret profile PROFILE_NAME algorithm ALGORITHM )" command,
# in "config-bgp" mode
# We do not have a 'disabled' version for the password command because the
# 'disabled' token matches autoPasswdRule (specifically rule1)
#-----------------------------------------------------------------------------
class NeighborPasswordCommand( BgpNEDCmdBaseClass ):
   syntax = ( "neighbor PEER password ( SECRET | PROFILE )" )

   noOrDefaultSyntax = "neighbor PEER password ..."
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'password': 'Password to use in computation of hash',
      'SECRET': ReversibleSecretCli.ReversiblePasswordCliExpression(
                  cleartextMatcher=ReversibleSecretCli.cleartextAuthMatcher,
                  obfuscatedTextMatcher=ReversibleSecretCli.type7AuthMatcher,
                  algorithm='DES', returnEncryptedText=True ),
      'PROFILE': BgpTcpSharedSecretProfileExpression,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborPasswordCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborPasswordCommand_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      peer = args[ 'PEER' ]
      authPasswdToken = args.get( 'SECRET' )
      password = ''
      if authPasswdToken:
         authPasswdText, authPasswdAlgo = args[ 'SECRET' ]
         password = ReversibleSecretCli.tryDecodeToken( authPasswdText,
                                                        peer.stringValue + '_passwd',
                                                        authPasswdAlgo,
                                                        mode=mode )
         if not password or password.isDefaultSecret():
            return
         elif password.clearTextSize() > 80:
            mode.addErrorAndStop( 'Configured password is too long for MD5 hash' )
         args[ 'AUTH_KEY' ] = Tac.Value( 'Routing::Bgp::BgpTcpAuthKey', password,
                                         '', TcpAoAlgorithmEnum.noAlgorithm )

RouterBgpSharedModelet.addCommandClass( NeighborPasswordCommand )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR maximum-routes NUM_ROUTES
#  [ warning-limit THRESHOLD percent ] [ warning-only ]" command
#-----------------------------------------------------------------------------
class SetNeighborMaxRoutesCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER maximum-routes NUM_ROUTES '
              '[ warning-limit ( ( THRESHOLD percent ) | ABS_THRESHOLD ) ] '
              '[ warning-only ]'
   )
   noOrDefaultSyntax = 'neighbor PEER maximum-routes ...'
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'maximum-routes': bgpTokens.maxRoutes,
         'NUM_ROUTES': bgpTokens.numMaxRoutesRangeMatcher,
         'warning-limit': bgpTokens.threshold,
         'THRESHOLD': bgpTokens.thresholdPercentageMatcher,
         'percent': bgpTokens.percent,
         'ABS_THRESHOLD': bgpTokens.thresholdRangeMatcher,
         'warning-only': bgpTokens.warning,
   }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborMaxRoutesCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborMaxRoutesCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ABS_THRESHOLD' in args:
         args[ 'THRESHOLD' ] = args.pop( 'ABS_THRESHOLD' )

for m in [
      RouterBgpAfIpUniSharedModelet,
      RouterBgpBaseAfEvpnMode,
      RouterBgpSharedModelet
]:
   m.addCommandClass( SetNeighborMaxRoutesCmd )


#-------------------------------------------------------------------------------
# "neighbor ADDR rcf in FUNCTION" command,
# in "config-bgp" mode
#-------------------------------------------------------------------------------
directions = {
  'in': bgpTokens.rcfIn.helpdesc_,
  'out': bgpTokens.rcfOut.helpdesc_,
}

class SetNeighborRcfCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER rcf DIRECTION ' \
            '( ( FUNCTION [ disabled ] ) | disabled )'
   noOrDefaultSyntax = 'neighbor PEER rcf DIRECTION ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'rcf': bgpTokens.rcf,
         'DIRECTION': CliMatcher.EnumMatcher( directions ),
         'FUNCTION': RcfCliLib.rcfFunctionMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborRcfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborRcfCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( SetNeighborRcfCmd )
RouterBgpAfLabelSharedModelet.addCommandClass( SetNeighborRcfCmd )
RouterBgpBaseAfEvpnMode.addCommandClass( SetNeighborRcfCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-to-peer" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetRouteToPeerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER route-to-peer [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER route-to-peer ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-to-peer': bgpTokens.routeToPeer,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetRouteToPeerCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetRouteToPeerCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetRouteToPeerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR remove-private-as [all [replace-as] ]"
# command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborRemovePrivateAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER remove-private-as [ [ all [ replace-as ] ] | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER remove-private-as ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'remove-private-as': bgpTokens.removePrivateAs,
      'all': bgpTokens.removePrivateAsAll,
      'replace-as': bgpTokens.replacePrivateAs,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborRemovePrivateAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborRemovePrivateAsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborRemovePrivateAsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR remove-private-as ingress [ replace-as ]"
# command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborRemovePrivateAsIngressCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER remove-private-as ingress [ replace-as | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER remove-private-as ingress ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'remove-private-as': bgpTokens.removePrivateAs,
      'ingress': bgpTokens.removePrivateAsIngress,
      'replace-as': bgpTokens.replacePrivateAsIngress,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborRemovePrivateAsIngressCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborRemovePrivateAsIngressCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborRemovePrivateAsIngressCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR as-path prepend-own disabled" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborPrependOwnAsDisabledCmd( BgpNEDCmdBaseClass ):
   # The 'disabled' token for NED mode is different than the disabled token
   # that is part of this command.  Use an underscore to disambiguate.
   syntax = 'neighbor PEER as-path prepend-own _disabled [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER as-path prepend-own _disabled ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'as-path': bgpTokens.asPath,
      'prepend-own': bgpTokens.prependOwn,
      '_disabled': bgpTokens.prependOwnDisabled,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborPrependOwnAsDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborPrependOwnAsDisabledCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborPrependOwnAsDisabledCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR as-path remote-as replace out"
# command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborReplaceRemoveAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER as-path remote-as replace out [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER as-path remote-as replace out ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'as-path': bgpTokens.asPath,
      'remote-as': bgpTokens.remoteAs,
      'replace': bgpTokens.replaceRemoteAs,
      'out': bgpTokens.out,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborReplaceRemoveAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborReplaceRemoveAsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborReplaceRemoveAsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR next-hop-self" command,
# in "router-bgp-af" and "router-bgp" modes.
#-------------------------------------------------------------------------------
class SetNeighborNextHopSelfCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-self [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop-self ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'next-hop-self': bgpTokens.nextHopSelf,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborNextHopSelfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborNextHopSelfCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborNextHopSelfCmd )
RouterBgpAfIpUniSharedModelet.addCommandClass( SetNeighborNextHopSelfCmd )

RouterBgpAfLabelSharedModelet.addCommandClass( SetNeighborNextHopSelfCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR next-hop-peer" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborNextHopPeerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-peer [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop-peer ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'next-hop-peer': bgpTokens.nextHopPeer,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborNextHopPeerCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborNextHopPeerCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborNextHopPeerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR description <desc>" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborDescriptionCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER description DESC'
   noOrDefaultSyntax = 'neighbor PEER description ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'description': bgpTokens.description,
      'DESC': bgpTokens.descriptionMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborDescriptionCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborDescriptionCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborDescriptionCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR send-community
# [ standard ] [ extended ] [ large ]
# [ link-bandwidth [aggregate LINK_BW | divide ( equal | ratio ) ] ]"
# command, in "router-bgp" mode.
#
# This is broken into several CliCommandClasses to simplify the syntax
# and processing.  In particular, the LBW parameter is allowed but hidden
# in some cases.
#-------------------------------------------------------------------------------
class SendLinkBandwidthCommExpr( CliCommand.CliExpression ):
   expression = '''link-bandwidth
                   ( aggregate [ LINK_BW ] ) | ( divide DIV )'''
   data = { 'link-bandwidth': bgpTokens.sendLinkBandwidthCommunity,
            'aggregate': bgpTokens.sendLbwAggregate,
            'LINK_BW': LinkBandwidthValueCliMatcher(
               'Reference link speed in bits/second' ),
            'divide': bgpTokens.sendLbwDivide,
            'DIV': CliMatcher.EnumMatcher( {
               'equal': ( 'Send link-bandwidth equal divided '
                          'attribute to this neighbor' ),
               'ratio': ( 'Send link-bandwidth ratio divided '
                          'attribute to this neighbor' ),
            } )
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'link-bandwidth' in args:
         if 'aggregate' in args:
            additiveOrDivide = {
               'additive': {
                  'lbwVal': args.get( 'LINK_BW' ),
               }
            }
         elif 'divide' in args:
            additiveOrDivide = {
               'divide': {
                  'equalOrRatio': args[ 'DIV' ]
               }
         }
         args[ 'LBW' ] = { 'additiveOrDivide': additiveOrDivide }

# The class below is used to hide the link-bandwidth keyword
# when link-bandwidth token is used with standard/large community
# Note that we cannot hide the expression directly as of now
# and so we need to create a duplicate class to have link-bandwidth
# hidden (BUG 480785)
class SendLinkBandwidthCommHiddenExpr( SendLinkBandwidthCommExpr ):
   data = SendLinkBandwidthCommExpr.data.copy()
   data[ 'link-bandwidth' ] = CliCommand.Node(
      matcher=bgpTokens.sendLinkBandwidthCommunity,
      hidden=True )

class SetNeighborSendCommunityCmd( BgpNEDCmdBaseClass ):
   '''The base command, includes the no/default form.'''
   syntax = 'neighbor PEER send-community [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER send-community ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborSendCommunityCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborSendCommunityCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityCmd )

class SetNeighborSendCommunityStandardLargeCmd( CliCommand.CliCommandClass ):
   '''The command which accepts combinations of standard and large communities.'''
   syntax = ( 'neighbor PEER send-community '
              '( ( standard [ large ] ) | large ) [ LBW ]' )
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
      'standard': bgpTokens.sendStandardCommunity,
      'large': bgpTokens.sendLargeCommunity,
      'LBW': SendLinkBandwidthCommHiddenExpr
   }
   handler = \
      "RoutingBgpNeighborCliHandler.SetNeighborSendCommunityStandardLargeCmd_handler"

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityStandardLargeCmd )

class SetNeighborSendCommunityExtendedCmd( CliCommand.CliCommandClass ):
   '''The command which accepts extended and LBW communities.'''
   syntax = ( 'neighbor PEER send-community '
              '( [ standard ] extended  [ large ] [ LBW ] ) | LBW' )
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
      'standard': bgpTokens.sendStandardCommunity,
      'large': bgpTokens.sendLargeCommunity,
      'extended': bgpTokens.sendExtCommunity,
      'LBW': SendLinkBandwidthCommExpr
   }
   handler = \
      "RoutingBgpNeighborCliHandler.SetNeighborSendCommunityExtendedCmd_handler"

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityExtendedCmd )

class SetNeighborSendCommunityAddOrRemoveCmd( CliCommand.CliCommandClass ):
   '''The command which accepts combinations of add/remove.'''
   syntax = ( 'neighbor PEER send-community ( add | remove ) '
              '( standard | extended | large | LBW )' )
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'send-community': bgpTokens.sendCommunity,
      'add': bgpTokens.sendCommunityAdd,
      'remove': bgpTokens.sendCommunityRemove,
      'standard': bgpTokens.sendStandardCommunity,
      'extended': bgpTokens.sendExtCommunity,
      'large': bgpTokens.sendLargeCommunity,
      'LBW': SendLinkBandwidthCommExpr
   }
   handler = \
      "RoutingBgpNeighborCliHandler.SetNeighborSendCommunityAddOrRemoveCmd_handler"

RouterBgpSharedModelet.addCommandClass( SetNeighborSendCommunityAddOrRemoveCmd )

# ----------------------------------------------------------------------------------
# "[no|default] neighbor ADDR command-tag <TAG>,
# in "router-bgp" mode.
# This command is not available for peer-groups. Only peer IP addresses are allowed.
# ----------------------------------------------------------------------------------
class NeighborCommandTagCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER CONFIG_TAG_EXPR'
   noOrDefaultSyntax = 'neighbor PEER command-tag ...'

   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': V4V6PeerKeyCliExpression,
         'CONFIG_TAG_EXPR': ConfigTagCommon.ConfigTagExpr,
         'command-tag': ConfigTagCommon.configTagKwNode,
   }
   handler = "RoutingBgpNeighborCliHandler.NeighborCommandTagCmd_handler"
   noOrDefaultHandler = \
         "RoutingBgpNeighborCliHandler.NeighborCommandTagCmd_noOrDefaultHandler"

if ConfigTagToggleLib.toggleFeatureConfigTagBgpEnabled():
   RouterBgpSharedModelet.addCommandClass( NeighborCommandTagCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR shutdown command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborShutdownCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER shutdown [ reason MESSAGE ] [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER shutdown ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'shutdown': 'Disable the neighbor',
         'reason': bgpTokens.reason,
         'MESSAGE': bgpTokens.messageMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborShutdownCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborShutdownCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborShutdownCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR dont-capability-negotiate" command, in
#   "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborDontCapabilityNegotiateCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER dont-capability-negotiate [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER dont-capability-negotiate ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'dont-capability-negotiate': bgpTokens.dontCapabilityNegotiate,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.NeighborDontCapabilityNegotiateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborDontCapabilityNegotiateCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborDontCapabilityNegotiateCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR route-reflector-client
#     [meshed | reflection intra-cluster disabled ]"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborReflectorClientCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER route-reflector-client ' \
      '[ meshed | ( [ reflection intra-cluster ] disabled ) ]'
   noOrDefaultSyntax = 'neighbor PEER route-reflector-client ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-reflector-client': bgpTokens.routeReflectorClient,
      'meshed': CliCommand.Node( bgpTokens.meshed, hidden=True ),
      'reflection': bgpTokens.reflection,
      'intra-cluster': bgpTokens.intraCluster,
      'disabled': BgpCmdBaseClass.data[ 'disabled' ]
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborReflectorClientCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborReflectorClientCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborReflectorClientCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR stall [fail]" hidden command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborStallCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER stall [ fail ]'
   noOrDefaultSyntax = 'neighbor PEER stall ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'stall': bgpTokens.stallHidden,
      'fail': bgpTokens.stallFail
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborStallCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborStallCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborStallCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR sockfault [greedyrecvestabfail]" hidden
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborSockFaultCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER sockfault [ greedyrecvestabfail ]'
   noOrDefaultSyntax = 'neighbor PEER sockfault ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'sockfault': bgpTokens.sockFaultHidden,
      'greedyrecvestabfail': bgpTokens.sockFaultGreedy
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborSockFaultCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborSockFaultCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborSockFaultCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR local-v6-addr IP6_ADDR" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborLocalIp6AddrCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER local-v6-addr ( IP6_ADDR | disabled )'
   noOrDefaultSyntax = 'neighbor PEER local-v6-addr ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4OrPeerGroupCliExpression,
      'local-v6-addr': bgpTokens.localIp6Addr,
      'IP6_ADDR': Ip6AddrMatcher.Ip6AddrMatcher(
         "The local IPv6 address of the neighbor" )
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLocalIp6AddrCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLocalIp6AddrCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLocalIp6AddrCmd )

#-----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR next-hop-v6-addr IP6_ADDR in"
# command in "router-bgp" mode.
#-----------------------------------------------------------------------------------
class SetNeighborIp6NextHopAddrCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-v6-addr ( IP6_ADDR in ) | disabled'
   noOrDefaultSyntax = 'neighbor PEER next-hop-v6-addr ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4OrPeerGroupCliExpression,
      'next-hop-v6-addr': bgpTokens.nextHopV6Addr,
      'IP6_ADDR': Ip6AddrMatcher.Ip6AddrMatcher(
         "IPv6 next-hop address for the neighbor" ),
      'in': bgpTokens.nextHopV6AddrIn,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborIp6NextHopAddrCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborIp6NextHopAddrCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborIp6NextHopAddrCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR local-v4-addr IP4_ADDR" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborLocalIp4AddrCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER local-v4-addr ( IP4_ADDR | disabled )'
   noOrDefaultSyntax = 'neighbor PEER local-v4-addr ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V6OrPeerGroupCliExpression,
      'local-v4-addr': bgpTokens.localIp4Addr,
      'IP4_ADDR': IpAddrMatcher.IpAddrMatcher(
         "The local IPv4 address of the neighbor" )
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLocalIp4AddrCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLocalIp4AddrCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLocalIp4AddrCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR auto-local-addr"
#-------------------------------------------------------------------------------
class SetNeighborAutoLocalAddrPeerCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER auto-local-addr [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER auto-local-addr'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'auto-local-addr': bgpTokens.autoLocalAddr,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborAutoLocalAddrPeerCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborAutoLocalAddrPeerCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborAutoLocalAddrPeerCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR additional-paths receive disabled"
#-------------------------------------------------------------------------------
class SetNeighborAddPathRecvCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER additional-paths receive [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER additional-paths receive'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'additional-paths': bgpTokens.addPath,
         'receive': bgpTokens.addpathReceive,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborAddPathRecvCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborAddPathRecvCmd_noOrDefaultHandler"

modelets = [ RouterBgpSharedModelet,
             RouterBgpAfSharedModelet,
             RouterBgpAfLabelSharedModelet,
             RouterBgpBaseAfDpsMode ]
for modelet in modelets:
   modelet.addCommandClass( SetNeighborAddPathRecvCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> additional-paths send any disabled"
#-------------------------------------------------------------------------------
class SetNeighborAddPathSendCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER additional-paths send '
              '( ( any [ ( prefix-list PREFIX_LIST_NAME ) | disabled ] ) |'
              '  ( ( backup | ( [ ecmp ] [ limit LIMIT ] ) )'
              '    [ prefix-list PREFIX_LIST_NAME ] ) |'
              '  disabled )' )
   # For backward compatibility we will accept 'any' in no/default
   noOrDefaultSyntax = 'neighbor PEER additional-paths send [ ANY_HIDDEN | ... ]'

   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'additional-paths': bgpTokens.addPath,
         'send': bgpTokens.addpathSend,
         'any': bgpTokens.appAny,
         'backup': bgpTokens.appBackup,
         'ecmp': bgpTokens.appEcmp,
         'limit': bgpTokens.appLimit,
         'LIMIT': bgpTokens.appLimitRangeMatcher,
         'prefix-list': bgpTokens.apSendPrefixList,
         'PREFIX_LIST_NAME': prefixListNameMatcher,
         'ANY_HIDDEN': CliCommand.hiddenKeyword( 'any' ),
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborAddPathSendCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborAddPathSendCmd_noOrDefaultHandler"

modelets = [ RouterBgpSharedModelet,
             RouterBgpAfIpUniSharedModelet,
             RouterBgpAfLabelSharedModelet,
             RouterBgpBaseAfEvpnMode,
             RouterBgpBaseAfDpsMode ]
for modelet in modelets:
   modelet.addCommandClass( SetNeighborAddPathSendCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor GROUP | ADDR default-originate
#                                    [route-map MAP_NAME | rcf FUNC] [always]
#     in "router-bgp" or "router-bgp-af" mode
#-------------------------------------------------------------------------------

# pylint: disable-next=useless-object-inheritance
class SetNeighborDefaultOriginateBaseClass( object ):
   baseData = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'default-originate': bgpTokens.defOrigin,
         'always': CliCommand.Node( bgpTokens.defOriginAlways,
                                    maxMatches=1 ),
         'route-map': CliCommand.Node( bgpTokens.routeMap,
                                       maxMatches=1 ),
         'MAP_NAME': CliCommand.Node( mapNameMatcher, maxMatches=1 ),
         'rcf': CliCommand.Node( bgpTokens.rcf, maxMatches=1 ),
         'FUNCTION': CliCommand.Node( RcfCliLib.rcfFunctionMatcher, maxMatches=1 ),
   }

class SetNeighborDefaultOriginateCmd( SetNeighborDefaultOriginateBaseClass,
                                      BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER default-originate '
              '[ { always | ( route-map MAP_NAME ) } | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER default-originate ...'
   data = BgpNEDCmdBaseClass._createSyntaxData(
         SetNeighborDefaultOriginateBaseClass.baseData )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborDefaultOriginateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborDefaultOriginateCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborDefaultOriginateCmd )

class SetNeighborDefaultOriginateAfCmd( SetNeighborDefaultOriginateBaseClass,
                                        BgpCmdBaseClass ):
   syntax = 'neighbor PEER default-originate '

   if RcfLibToggleLib.toggleRcfDefaultOriginateEnabled():
      syntax += ( '[ [ { always | ( route-map MAP_NAME ) } | '
                  '{ always | ( rcf FUNCTION ) } ] | disabled ]' )
   else:
      syntax += '[ { always | ( route-map MAP_NAME ) } | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER default-originate...'
   data = BgpCmdBaseClass._createSyntaxData(
         SetNeighborDefaultOriginateBaseClass.baseData )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborDefaultOriginateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborDefaultOriginateCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( SetNeighborDefaultOriginateAfCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor GROUP | ADDR default-originate
#                                    [route-map MAP_NAME]
#     in "router-bgp-af" vpn mode ( evpn | vpn-ipv4 | vpn-ipv6 )
#
# Same as the previous but without 'always'
#-------------------------------------------------------------------------------

class SetVpnNeighborDefaultOriginateHiddenCmd( SetNeighborDefaultOriginateBaseClass,
                                               BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER default-originate '
              '[ ( route-map MAP_NAME ) | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER default-originate ...'
   data = BgpNEDCmdBaseClass._createSyntaxData(
         SetNeighborDefaultOriginateBaseClass.baseData )
   hidden = True
   handler = \
      "RoutingBgpNeighborCliHandler.SetVpnNeighborDefaultOriginateHiddenCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetVpnNeighborDefaultOriginateHiddenCmd_noOrDefaultHandler"

#-------------------------------------------------------------------------------
# [no|default] neighbor GROUP | ADDR default-route
#                                    [route-map MAP_NAME | rcf FUNC]
#     in "router-bgp-af" vpn mode ( evpn | vpn-ipv4 | vpn-ipv6 )
#
# Same as the previous but without 'always'
#-------------------------------------------------------------------------------
class SetVpnNeighborDefaultOriginateCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER default-route '

   if RcfLibToggleLib.toggleRcfDefaultOriginateEnabled():
      syntax += '[ ( POLICY_TYPE ) | disabled ]'
   else:
      syntax += '[ ( route-map MAP_NAME ) | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER default-route ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'default-route': bgpTokens.defaultRoute,
         'POLICY_TYPE': bgpTokens.RouteMapOrRcfFunction,

   } )

   handler = "RoutingBgpNeighborCliHandler.SetVpnNeighborDefaultOriginateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetVpnNeighborDefaultOriginateCmd_noOrDefaultHandler"

RouterBgpAfVpnModelet.addCommandClass( SetVpnNeighborDefaultOriginateHiddenCmd )
RouterBgpAfVpnModelet.addCommandClass( SetVpnNeighborDefaultOriginateCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR enforce-first-as" command, in
# "config-bgp" mode
#-----------------------------------------------------------------------------
class SetNeighborEnforceFirstAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER enforce-first-as [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER enforce-first-as'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'enforce-first-as': bgpTokens.enforceFirstAs,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborEnforceFirstAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborEnforceFirstAsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborEnforceFirstAsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR update-source UPDATE_SRC_INTF | ADDR"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class NeighborUpdateSrcCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor ( PEER update-source ( UPDATE_SRC_INTF | disabled ) ) |'
              '         ( V4_PEER update-source ( V4_ADDR | disabled ) ) |'
              '         ( V6_PEER update-source ( V6_ADDR | disabled ) )' )
   noOrDefaultSyntax = 'neighbor PEER update-source ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'V4_PEER': V4OrPeerGroupCliExpression,
      'V6_PEER': V6OrPeerGroupCliExpression,
      'update-source': bgpTokens.updateSrc,
      'UPDATE_SRC_INTF': IntfCli.Intf.matcherWithIpSupport,
      'V4_ADDR': IpAddrMatcher.IpAddrMatcher( 'IPv4 source address' ),
      'V6_ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'IPv6 source address' ),
      } )
   handler = "RoutingBgpNeighborCliHandler.NeighborUpdateSrcCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborUpdateSrcCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NeighborUpdateSrcCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR ebgp-multihop [TTL]" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborEbgpMultiHopTtlCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER ebgp-multihop [ TTL | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER ebgp-multihop ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'ebgp-multihop': bgpTokens.ebgpMultiHop,
      'TTL': bgpTokens.ttlRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborEbgpMultiHopTtlCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborEbgpMultiHopTtlCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborEbgpMultiHopTtlCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR ttl maximum-hops NUM_HOPS" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborTtlMaxHopsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER ttl maximum-hops NUM_HOPS'
   noOrDefaultSyntax = 'neighbor PEER ttl maximum-hops ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'ttl': bgpTokens.ttl,
      'maximum-hops': bgpTokens.maxHops,
      'NUM_HOPS': bgpTokens.numHopsRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborTtlMaxHopsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborTtlMaxHopsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborTtlMaxHopsCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR link-bandwidth update-delay DELAY"
# command in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborLinkbwDelayCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth update-delay DELAY'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth update-delay ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'link-bandwidth': bgpTokens.linkBwSend,
      'update-delay': bgpTokens.ucmpLinkbwDelay,
      'DELAY': bgpTokens.ucmpDelayRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLinkbwDelayCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLinkbwDelayCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkbwDelayCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR weight [WEIGHT]" command
#-------------------------------------------------------------------------------
class SetNeighborWeightCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER weight WEIGHT'
   noOrDefaultSyntax = 'neighbor PEER weight ...'
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'weight': bgpTokens.weight,
         'WEIGHT': CliMatcher.IntegerMatcher( 0, 65535, helpdesc='Weight to assign' )
   }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborWeightCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborWeightCmd_noOrDefaultHandler"

for m in [ RouterBgpSharedModelet, RouterBgpAfSharedModelet ]:
   m.addCommandClass( SetNeighborWeightCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR allowas-in [NUM_AS]" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborAllowAsCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER allowas-in [ NUM_AS | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER allowas-in ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'allowas-in': bgpTokens.allowAsIn,
      'NUM_AS': bgpTokens.allowAsRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborAllowAsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborAllowAsCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborAllowAsCmd )

#-------------------------------------------------------------------------------
# Deprecated version of the Command:-
# "[no|default] neighbor GROUP | ADDR transport connection-mode passive"
# command, in "router-bgp" mode.
#
# New version of the command:-
# "[no|default] neighbor GROUP | ADDR passive"
# command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborPassiveCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER ( passive | '
              '( transport connection-mode passive ) ) [ disabled ]' )
   noOrDefaultSyntax = ( 'neighbor PEER ( passive | '
                         '( transport connection-mode passive ) ) ...' )
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'passive': bgpTokens.passive,
      'transport': bgpTokens.transport,
      'connection-mode': bgpTokens.connectionModeDeprecated,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborPassiveCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborPassiveCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborPassiveCmd )

#----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR transport remote-port PORT_NUM" command
# in "router-bgp" mode.
#----------------------------------------------------------------------------------
class SetNeighborRemotePortCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER transport remote-port PORT_NUM'
   noOrDefaultSyntax = 'neighbor PEER transport remote-port ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'transport': bgpTokens.transport,
      'remote-port': bgpTokens.remotePort,
      'PORT_NUM': bgpTokens.remotePortNumRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborRemotePortCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborRemotePortCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborRemotePortCmd )

#----------------------------------------------------------------------------------
# " [no|default] neighbor <addr|peer-group> transport pmtud" command
# "neighbor <addr|peer-group> transport pmtud [disabled]" command
# in "router-bgp" mode. With "no == default" and states to "default inherit".
#
# For this command, the 'disabled' keyword is not dependent on NED mode.
#----------------------------------------------------------------------------------
class SetNeighborTransportPmtudCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER transport pmtud [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER transport pmtud ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'transport': bgpTokens.transport,
         'pmtud': bgpTokens.pathMtuDiscovery, } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborTransportPmtudCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborTransportPmtudCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborTransportPmtudCmd )

#------------------------------------------------------------------------------
# Deprecated version of the command:-
# "[no|default] neighbor ADDR peer-group PEER_GROUP" command,
# in "router-bgp" mode.
#
# New version of the command:-
# "[no|default] neighbor ADDR peer group PEER_GROUP" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborPeerGroupCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER ( ( peer group ) | peer-group ) PEER_GROUP'
   noOrDefaultSyntax = 'neighbor PEER ( ( peer group ) | peer-group ) ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4V6PeerKeyCliExpression,
      'peer': bgpTokens.peer,
      'group': bgpTokens.group,
      'peer-group': bgpTokens.peerGroupDeprecated,
      'PEER_GROUP': peergroupNameMatcher
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborPeerGroupCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborPeerGroupCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborPeerGroupCmd )

#-------------------------------------------------------------------------------
# Deprecated version of the command:
# "[no|default] neighbor PEER_GROUP peer-group" command,
# in "router-bgp" mode.
#
# New version of the command:
# "neighbor PEER_GROUP peer group [ { GROUPS } ]" and
# "no|default neighbor PEER_GROUP peer group ..."
# in "router-bgp" mode.
#-------------------------------------------------------------------------------

class ConfigureNeighborPeerGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor GROUP ( ( peer group ) | peer-group ) [ { GROUPS } ]'
   # Ancestor peer groups are an all or none operation.
   # The no/default form does not have to match the previously given ancestor
   # peer groups - so specify trailing garbage.
   noOrDefaultSyntax = 'neighbor GROUP ( ( peer group ) | peer-group ) ...'
   data = {
      'neighbor': bgpTokens.neighbor,
      'GROUP': peergroupNameMatcher,
      'peer': bgpTokens.peer,
      'group': bgpTokens.group,
      'peer-group': bgpTokens.peerGroupDeprecated,
      'GROUPS': CliCommand.Node(
         matcher=ancestorPeerGroupNameMatcher,
         maxMatches=AncestorPgConfigType.ancestorPeerGroupKeysMax ),
   }
   handler = "RoutingBgpNeighborCliHandler.ConfigureNeighborPeerGroupCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.ConfigureNeighborPeerGroupCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( ConfigureNeighborPeerGroupCmd )

class SetNeighborBfdIntervalCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER bfd interval INTERVAL min-rx MINRX multiplier MULTIPLIER'
   noOrDefaultSyntax = 'neighbor PEER bfd interval ...'
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'bfd': bgpTokens.bfd,
      'interval': 'Set transmit rate in milliseconds for BFD (does not enable BFD)',
      'INTERVAL': bfdTokens.matcherTxRxIntervalMs,
      'min-rx': 'Set expected minimum incoming rate in milliseconds for BFD',
      'MINRX': bfdTokens.matcherTxRxIntervalMs,
      'multiplier': 'Sets the BFD multiplier',
      'MULTIPLIER': bfdTokens.matcherMultiplierVal,
   }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborBfdIntervalCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborBfdIntervalCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborBfdIntervalCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr> bfd [c-bit]" command,
# in "router-bgp" mode.
#
# Deprecated version of the command:
# "[no|default] neighbor <addr> fall-over bfd" command,
# in "router-bgp" mode.
#-------------------------------------------------------------------------------
class SetNeighborBfdWithDampingCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER [ fall-over ] bfd [ ( [ c-bit ]'
              ' [ damping [ seconds TIME [ max-multiplier MULTIPLIER ] ] ] )'
              ' | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER [ fall-over ] bfd ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'fall-over': bgpTokens.fallOverDeprecated,
         'bfd': bgpTokens.bfd,
         'c-bit': 'Enable support for the BFD C-bit',
         'damping': 'Enable support for the BFD Damping',
         'seconds': 'Damping time in seconds',
         'TIME': CliMatcher.IntegerMatcher( 10, 3600, helpdesc='Number of seconds' ),
         'max-multiplier': 'Maximum multiplier for damping time',
         'MULTIPLIER': CliMatcher.IntegerMatcher( 1, 60, helpdesc='Max multiplier' )
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborBfdWithDampingCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborBfdWithDampingCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborBfdWithDampingCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr|pg> link-bandwidth
#  [ auto | default <lbw> | adjust auto [ percent <%> ] ] command, in
#  "router-bgp" mode.
#-------------------------------------------------------------------------------
# pylint: disable-next=useless-object-inheritance
class SetNeighborLinkBwAutoBase( object ):
   syntax = 'neighbor PEER link-bandwidth auto'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth auto'
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
         'auto': bgpTokens.linkBwAuto,
         'percent': bgpTokens.linkBwPercent,
         'PERCENT': bgpTokens.linkBwPercentRangeMatcher,
   }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLinkBwAutoBase_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLinkBwAutoBase_noOrDefaultHandler"

class SetNeighborLinkBwAutoPercentCmd(
      BgpNEDCmdBaseClass,
      SetNeighborLinkBwAutoBase
):
   syntax = SetNeighborLinkBwAutoBase.syntax + ' [ percent PERCENT ] [disabled]'
   noOrDefaultSyntax = SetNeighborLinkBwAutoBase.noOrDefaultSyntax
   data = BgpNEDCmdBaseClass._createSyntaxData(
         SetNeighborLinkBwAutoBase.data.copy()
   )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLinkBwAutoPercentCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborLinkBwAutoPercentCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwAutoPercentCmd )

class SetNeighborLinkBwAdjustCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth adjust auto [ percent PERCENT ] [disabled]'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth adjust auto'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
         'adjust': bgpTokens.linkBwAdjust,
         'auto': bgpTokens.linkBwAutoAfterAdjust,
         'percent': bgpTokens.linkBwPercent,
         'PERCENT': bgpTokens.linkBwPercentRangeMatcher,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLinkBwAdjustCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLinkBwAdjustCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwAdjustCmd )

class SetNeighborLinkBwCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLinkBwCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLinkBwCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwCmd )

class SetNeighborLinkBwDefaultCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER link-bandwidth default COMM [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER link-bandwidth default ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'link-bandwidth': bgpTokens.linkBw,
         'default': bgpTokens.linkBwDefault,
         'COMM': LinkBandwidthValueCliMatcher( 'Link speed in bits/second' )
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborLinkBwDefaultCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborLinkBwDefaultCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborLinkBwDefaultCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr> route-map <name> in [disabled]" command in
# address-family sr-te block
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr> route-map <name> <in|out>" command in address-family
# block
#-------------------------------------------------------------------------------

#---------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> next-hop address-family ipv6 [originate]"
#      command, in "router-bgp-af" mode
#---------------------------------------------------------------------------------
class SetNeighborNextHopAfV6OriginateCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop address-family ipv6 [ originate ] [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop address-family ipv6 ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop': bgpTokens.neighborNextHop,
         'address-family': bgpTokens.nextHopAf,
         'ipv6': bgpTokens.ipv6,
         'originate': bgpTokens.originate
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.SetNeighborNextHopAfV6OriginateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborNextHopAfV6OriginateCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( SetNeighborNextHopAfV6OriginateCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR aigp-session [disabled]" command,
# in "config-bgp-af" mod
#-----------------------------------------------------------------------------
class NeighborAigpSessionCommand( BgpNEDCmdBaseClass ):
   syntax = "neighbor PEER aigp-session [ disabled ]"
   noOrDefaultSyntax = "neighbor PEER aigp-session ..."
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'aigp-session': bgpTokens.aigpSession,
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborAigpSessionCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborAigpSessionCommand_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( NeighborAigpSessionCommand )
RouterBgpAfLabelSharedModelet.addCommandClass( NeighborAigpSessionCommand )

#-------------------------------------------------------------------------------
# "[no|default] neighbor default encapsulation ( mpls | vxlan | path-selection )"
# "[no|default] neighbor default encapsulation segment-routing ipv6 locator LOC"
# "[no|default] neighbor default encapsulation mpls next-hop-self
#   source-interface INTF"
# in 'address-family evpn' mode
#-------------------------------------------------------------------------------
def getSrv6LocatorNames( mode ):
   # Update when locator configuration becomes available
   return []

locatorNameMatcher = CliMatcher.DynamicNameMatcher( getSrv6LocatorNames,
                                                    'SRv6 locator name' )

class SetNeighborDefaultEncapCmd( BgpCmdBaseClass ):
   syntax = '''neighbor default encapsulation
               ( vxlan
               | ( mpls [ next-hop-self source-interface INTF ] ) '''
   if RoutingLibToggleLib.toggleSRv6BgpL3VpnEnabled():
      syntax += '| ( segment-routing ipv6 locator LOCATOR ) '
   if BgpCommonToggleLib.toggleEvpnL3DciWithDpsEnabled():
      syntax += '| path-selection '
   syntax += ')'
   noOrDefaultSyntax = 'neighbor default encapsulation ...'
   data = {
         'neighbor': bgpTokens.neighbor,
         'default': bgpTokens.default,
         'encapsulation': bgpTokens.encap,
         'mpls': bgpTokens.mplsForEvpn,
         'vxlan': bgpTokens.vxlan,
         'next-hop-self': bgpTokens.nextHopSelf,
         'source-interface': bgpTokens.sourceInterface,
         'INTF': IntfCli.Intf.matcherWithIpSupport,
         'segment-routing': bgpTokens.segmentRoutingForEvpn,
         'ipv6': bgpTokens.ipv6Encaps,
         'locator': bgpTokens.srv6Locator,
         'LOCATOR': locatorNameMatcher,
         'path-selection': CliCommand.Node(
            matcher=bgpTokens.dps, guard=pathSelectionSupportedGuard ),
   }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborDefaultEncapCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborDefaultEncapCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argList ):
      if 'vxlan' in args:
         args[ 'vxlan' ] = 'encapVxlan'
      elif 'mpls' in args:
         args[ 'mpls' ] = 'encapMpls'
         if 'INTF' in args:
            intf = args.get( 'INTF' )
            args[ 'INTF' ] = intf.name
      elif 'ipv6' in args:
         args[ 'srv6' ] = 'encapSrv6'
      elif 'path-selection' in args:
         args[ 'dps' ] = 'encapDps'

RouterBgpBaseAfEvpnMode.addCommandClass( SetNeighborDefaultEncapCmd )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <addr|peer-group> encapsulation
#      ( mpls | vxlan | path-selection )"
# "[no|default] neighbor <addr|peer-group> encapsulation mpls next-hop-self
#   source-interface INTF"
# in 'address-family evpn' mode
#-------------------------------------------------------------------------------
class SetNeighborEncapCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor PEER encapsulation '
              '( vxlan | ( mpls [ next-hop-self source-interface INTF ] ) ' )
   if BgpCommonToggleLib.toggleEvpnL3DciWithDpsEnabled():
      syntax += '| path-selection '
   syntax += ')'
   noOrDefaultSyntax = 'neighbor PEER encapsulation ...'
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'encapsulation': bgpTokens.encap,
      'mpls': bgpTokens.mplsForEvpn,
      'vxlan': bgpTokens.vxlan,
      'next-hop-self': bgpTokens.nextHopSelf,
      'source-interface': bgpTokens.sourceInterface,
      'INTF': IntfCli.Intf.matcherWithIpSupport,
      'path-selection': CliCommand.Node(
         matcher=bgpTokens.dps, guard=pathSelectionSupportedGuard ),
   }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborEncapCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborEncapCmd_noOrDefaultHandler"

   @staticmethod
   def adapter( mode, args, argList ):
      if 'vxlan' in args:
         args[ 'vxlan' ] = 'encapVxlan'
      elif 'mpls' in args:
         args[ 'mpls' ] = 'encapMpls'
         if 'INTF' in args:
            intf = args.get( 'INTF' )
            args[ 'INTF' ] = intf.name
      elif 'path-selection' in args:
         args[ 'dps' ] = 'encapDps'

RouterBgpBaseAfEvpnMode.addCommandClass( SetNeighborEncapCmd )
# -------------------------------------------------------------------------------
# "[no|default] neighbor default encapsulation mpls multicast protocol p2mp mldp
# -------------------------------------------------------------------------------

class SetEvpnNeighborDefaultOismMplsP2MpMldp( BgpCmdBaseClass ):
   syntax = ( 'neighbor default encapsulation mpls multicast protocol p2mp mldp' )
   noOrDefaultSyntax = 'neighbor default encapsulation mpls multicast protocol'
   data = {
         'neighbor': bgpTokens.neighbor,
         'default': bgpTokens.default,
         'encapsulation': bgpTokens.encap,
         'mpls': bgpTokens.mpls,
         'multicast': CliMatcher.KeywordMatcher( 'multicast',
            helpdesc='Configure EVPN multicast' ),
         'protocol': CliMatcher.KeywordMatcher( 'protocol',
            helpdesc='Configure Protocol for MPLS' ),
         'p2mp': CliMatcher.KeywordMatcher( 'p2mp', helpdesc='Configure p2mp' ),
         'mldp': CliMatcher.KeywordMatcher( 'mldp', helpdesc='Configure mldp' ),
   }
   handler = "RoutingBgpNeighborCliHandler.SetEvpnNeighborDefaultOismMplsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler."\
   "SetEvpnNeighborDefaultOismMplsCmd_noOrDefaultHandler"

if MvpnLibToggleLib.toggleOismMplsTunnelEnabled():
   RouterBgpBaseAfEvpnMode.addCommandClass( SetEvpnNeighborDefaultOismMplsP2MpMldp )

#-------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR out-delay DELAY
#  [ changes [ withdrawals ] ]" commands
#-------------------------------------------------------------------------------
class PeerOutDelayCmdMixin( object ): # pylint: disable=useless-object-inheritance
   data = {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'out-delay': bgpTokens.outDelayMatcherForConfig,
         'DELAY': CliMatcher.IntegerMatcher(
               0,
               3600,
               helpdesc='Time in seconds by which route updates should be delayed.'
         ),
         'changes': bgpTokens.outDelayChangesMatcherForConfig,
         'withdrawals': bgpTokens.outDelayWithdrawalsMatcherForConfig
   }

class SetNeighborOutDelayCmd( PeerOutDelayCmdMixin, BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER out-delay DELAY [ changes [ withdrawals ] ]'
   noOrDefaultSyntax = 'neighbor PEER out-delay ...'
   data = PeerOutDelayCmdMixin.data.copy()
   handler = "RoutingBgpNeighborCliHandler.SetNeighborOutDelayCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborOutDelayCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborOutDelayCmd )

class SetNeighborOutDelayHiddenCmd(
      PeerOutDelayCmdMixin,
      CliCommand.CliCommandClass
):
   syntax = 'neighbor PEER out-delay DELAY'
   noOrDefaultSyntax = 'neighbor PEER out-delay ...'
   data = PeerOutDelayCmdMixin.data.copy()
   hidden = True
   handler = "RoutingBgpNeighborCliHandler.SetNeighborOutDelayHiddenCmd_handler"
   noHandler = "RoutingBgpNeighborCliHandler.SetNeighborOutDelayHiddenCmd_noHandler"
   defaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborOutDelayHiddenCmd_defaultHandler"

RouterBgpAfSharedModelet.addCommandClass( SetNeighborOutDelayHiddenCmd )

#-------------------------------------------------------------------------------
# '[no|default] neighbor <addr|peer-group> next-hop-self \
#            [v4-mapped-v6] source-interface <interface-name>
#     under 'address-family <addrFamily> labeled-unicast' mode'
#-------------------------------------------------------------------------------
class SetNeighborNextHopSelfAfSrcIntfCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-self [ v4-mapped-v6 ] source-interface ' \
         '( INTF | disabled )'
   noOrDefaultSyntax = 'neighbor PEER next-hop-self [ v4-mapped-v6 ] ' \
         'source-interface ...'

   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop-self': bgpTokens.nextHopSelf,
         'source-interface': bgpTokens.sourceInterface,
         'INTF': IntfCli.Intf.matcherWithIpSupport,
         'v4-mapped-v6': bgpTokens.v4MappedV6,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.SetNeighborNextHopSelfAfSrcIntfCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborNextHopSelfAfSrcIntfCmd_noOrDefaultHandler"

RouterBgpAfLabelSharedModelet.addCommandClass( SetNeighborNextHopSelfAfSrcIntfCmd )

#----------------------------------------------------------------------------------
# "[no|default] neighbor interface <interface-range> peer-group <pg-name>
#  [ ( ( remote-as <ASN> ) | ( peer-filter PEER_FILTER ) ) ]"
#  command, in "router-bgp" mode.
# The no and default versions of this command behave the same which is to remove the
# config whether in normal config mode or in no-equals-default mode. Hence there is
# no need for a 'disabled' version for no-equals-default mode.
#----------------------------------------------------------------------------------
class SetNeighborInterfaceCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor interface INTFS peer-group PG '
              '[ ( ( remote-as AS_NUM ) | ( peer-filter PEER_FILTER ) ) ]' )
   noOrDefaultSyntax = 'neighbor interface INTFS ...'

   data = {
         'neighbor': bgpTokens.neighbor,
         'interface': bgpTokens.interface,
         'INTFS': intfRangeMatcher,
         'peer-group': bgpTokens.peerGroup,
         'PG': peergroupNameMatcher,
         'remote-as': bgpTokens.remoteAs,
         'AS_NUM': bgpTokens.AsNumCliExpr,
         'peer-filter': bgpTokens.peerFilter,
         'PEER_FILTER': bgpTokens.peerFilterNameMatcher,
   }
   handler = "RoutingBgpNeighborCliHandler.SetNeighborInterfaceCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborInterfaceCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborInterfaceCmd )

#-------------------------------------------------------------------------------
# The "[no|default] neighbor <peer|group> route refresh demarcated
# stale-path removal [disabled|(timeout <secs> )]" command, in "router-bgp" mode.
#-------------------------------------------------------------------------------
class RouteRefreshConfigCmd( BgpNEDCmdBaseClass ):
   syntax = ( 'neighbor PEER route refresh demarcated stale-path removal '
              '[ ( timeout TIMEOUT ) | disabled ]' )
   noOrDefaultSyntax = 'neighbor PEER route ...'
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route': 'Route Refresh configuration',
      'refresh': 'Route Refresh configuration',
      'demarcated': 'Demarcated Route Refresh configuration',
      'stale-path': 'Stale paths removal configuration',
      'removal': 'Stale paths removal configuration',
      'timeout': 'Set the time (in seconds) before timing out waiting for ' +
      'End of Route Refresh demarcation',
      'TIMEOUT': CliMatcher.IntegerMatcher(
         0, bgpTokens.U16_MAX_VALUE,
         helpdesc='Time (in seconds) before timing out waiting for ' +
         'End of Route Refresh demarcation' ),
   } )
   handler = "RoutingBgpNeighborCliHandler.RouteRefreshConfigCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.RouteRefreshConfigCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( RouteRefreshConfigCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor <addr|peer-group> next-hop-unchanged [disabled]
#     in "router-bgp-af" and "router-bgp"  modes
#-------------------------------------------------------------------------------
class SetNeighborNextHopUnchangedCmd( BgpNEDCmdBaseClass ):
   syntax = 'neighbor PEER next-hop-unchanged [ disabled ]'
   noOrDefaultSyntax = syntax
   data = BgpNEDCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop-unchanged': bgpTokens.nexthopUnchanged,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNeighborNextHopUnchangedCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborNextHopUnchangedCmd_noOrDefaultHandler"

for m in [ RouterBgpSharedModelet,
           RouterBgpAfSharedModelet,
           RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( SetNeighborNextHopUnchangedCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor PEER route-target export VPN_NLRI_TYPE filter [ disabled ]
#-------------------------------------------------------------------------------
class NeighborRouteTargetExportFilterDisabledCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER route-target export VPN_NLRI_TYPE filter [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER route-target export VPN_NLRI_TYPE filter ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'route-target': bgpTokens.routeTarget,
         'export': bgpTokens.export,
         'VPN_NLRI_TYPE': bgpTokens.VpnNlriTypeExpr,
         'filter': 'Set route-target export filtering behavior',
   } )
   handler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborRouteTargetExportFilterDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborRouteTargetExportFilterDisabledCmd_noOrDefaultHandler"

for m in [ RouterBgpVrfSharedModelet, RouterBgpAfSharedVrfModelet ]:
   m.addCommandClass( NeighborRouteTargetExportFilterDisabledCmd )

#-------------------------------------------------------------------------------
# [no|default] neighbor <peer|peer-group> next-hop labeled-unicast originate
#    [ third-party | disabled ]
# in address-family ipv4|ipv6 mode
#-------------------------------------------------------------------------------
class NeighborNexthopLuOriginateCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER next-hop labeled-unicast originate'
   syntax += ' [ third-party | disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop labeled-unicast originate ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop': bgpTokens.neighborNextHop,
         'labeled-unicast': bgpTokens.lu,
         'originate': bgpTokens.nextHopLuOriginateKwMatcher
   } )
   data.update( {
      'third-party': bgpTokens.thirdParty } )
   handler = "RoutingBgpNeighborCliHandler.NeighborNexthopLuOriginateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborNexthopLuOriginateCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass( NeighborNexthopLuOriginateCmd )

# -------------------------------------------------------------------------------
# [no|default] neighbor <peer|peer-group> next-hop labeled-unicast peer-set PEER_SET
#    originate [ disabled ]
# in address-family ipv4|ipv6 mode
# -------------------------------------------------------------------------------
class NeighborNexthopLuPeerSetOriginateCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER next-hop labeled-unicast peer-set PEER_SET originate '\
            '[ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER next-hop labeled-unicast peer-set PEER_SET '\
                       'originate ...'
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop': bgpTokens.neighborNextHop,
         'labeled-unicast': bgpTokens.lu,
         'peer-set': bgpTokens.neighborPeerSet,
         'PEER_SET': epePeerSetNameMatcher,
         'originate': bgpTokens.nextHopLuOriginateKwMatcher
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.NeighborNexthopLuPeerSetOriginateCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborNexthopLuPeerSetOriginateCmd_noOrDefaultHandler"

if BgpCommonToggleLib.toggleArBgpNexthopLuPeerSetOriginateEnabled():
   RouterBgpAfIpUniSharedModelet.\
      addCommandClass( NeighborNexthopLuPeerSetOriginateCmd )


#-------------------------------------------------------------------------------
# [no|default] neighbor <peer|peer-group> next-hop labeled-unicast originate
#    lfib-backup ip-forwarding [ disabled ]
# in address-family ipv4|ipv6 mode
#-------------------------------------------------------------------------------
class NeighborNhLuOrigLfibBackupIpFwdCmd( BgpCmdBaseClass ):
   syntax = ( 'neighbor PEER next-hop labeled-unicast originate lfib-backup '
              'ip-forwarding [ disabled ]' )
   noOrDefaultSyntax = ( 'neighbor PEER next-hop labeled-unicast originate '
                         'lfib-backup ip-forwarding ...' )
   data = BgpCmdBaseClass._createSyntaxData( {
         'neighbor': bgpTokens.neighbor,
         'PEER': PeerCliExpression,
         'next-hop': bgpTokens.neighborNextHop,
         'labeled-unicast': bgpTokens.lu,
         'originate': bgpTokens.nextHopLuOriginateKwMatcher,
         'lfib-backup': CliCommand.Node( matcher=bgpTokens.lfibBackup,
                           guard=nhLuOrigLfibBackupIpFwdSupportedGuard ),
         'ip-forwarding': bgpTokens.ipForwarding,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.NeighborNhLuOrigLfibBackupIpFwdCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborNhLuOrigLfibBackupIpFwdCmd_noOrDefaultHandler"

RouterBgpAfIpUniSharedModelet.addCommandClass(
      NeighborNhLuOrigLfibBackupIpFwdCmd )

# -------------------------------------------------------------------------------
# [no|default] neighbor <peer|peer-group> path-attribute as-path as-sets
#    treat-as-withdraw [ disabled ]
# in address-family ipv4|ipv6 mode
# -------------------------------------------------------------------------------
# Handling actions for path attributes (e.g: treat-as-withdraw, attribute discard)
pathAction = {
   'treat-as-withdraw': 'Treat as withdrawn route',
}

class NeighborAsSetsCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER path-attribute as-path as-sets ACTION [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER path-attribute as-path as-sets ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'path-attribute': 'Path attribute handling commands',
      'as-path': bgpTokens.asPath,
      'as-sets': 'Specify handling of AS paths with AS_SET or AS_CONFED_SET',
      'ACTION': CliMatcher.EnumMatcher( pathAction ),
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborAsSetsCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborAsSetsCmd_noOrDefaultHandler"

if BgpCommonToggleLib.toggleRejectAsSetConfedSetEnabled():
   RouterBgpAfIpUniSharedModelet.addCommandClass( NeighborAsSetsCmd )

#-----------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR" command,
# in "config-bgp" mode
#-----------------------------------------------------------------------------------
class NoNeighborCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'neighbor PEER'
   data = {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4V6PeerKeyCliExpression,
   }
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NoNeighborCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( NoNeighborCmd )

#-----------------------------------------------------------------------------
# Deprecated version of the command:
# "[no|default] neighbor GROUP | ADDR soft-reconfiguration inbound"
# in "config-bgp" mode
# "neighbor GROUP | ADDR soft-reconfiguration inbound all in "config-bgp"
# mode
#
# New Version of the command:
# "[no|default] neighbor GROUP | ADDR rib-in pre-policy retain"
# in "config-bgp" mode
# "neighbor <addr|peer-group> rib-in pre-policy retain all in "config-bgp"
# mode
#-----------------------------------------------------------------------------
class SetNeighborSoftReconfigInboundCmd( BgpNEDCmdBaseClass ):
   syntax = (
      'neighbor PEER '
      '( ( rib-in pre-policy retain ) | ( soft-reconfiguration inbound ) ) '
      '[ all | disabled ]'
   )
   noOrDefaultSyntax = (
      'neighbor PEER '
      '( ( rib-in pre-policy retain ) | ( soft-reconfiguration inbound ) ) ...'
   )
   data = BgpNEDCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'rib-in': bgpTokens.ribIn,
      'pre-policy': bgpTokens.prePolicy,
      'retain': bgpTokens.retain,
      'soft-reconfiguration': bgpTokens.softReconfigDeprecated,
      'inbound': bgpTokens.inbound,
      'all': bgpTokens.softReconfigAll,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler.SetNeighborSoftReconfigInboundCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "SetNeighborSoftReconfigInboundCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNeighborSoftReconfigInboundCmd )

# -----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR rib-in delay DELAY event peer-init"
# in "router-bgp" mode
# -----------------------------------------------------------------------------
class RibInDelayCmd( BgpCmdBaseClass ):
   syntax = "neighbor PEER rib-in delay DELAY event peer-init"
   noOrDefaultSyntax = "neighbor PEER rib-in delay ..."
   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'rib-in': bgpTokens.ribIn,
      'delay': bgpTokens.ribInDelay,
      'DELAY': bgpTokens.ribInDelayTimeMatcher,
      'event': bgpTokens.ribInDelayEvent,
      'peer-init': bgpTokens.ribInDelayPeerInit,
   } )
   handler = "RoutingBgpNeighborCliHandler.RibInDelayCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.RibInDelayCmd_noOrDefaultHandler"

if BgpCommonToggleLib.toggleBgpInDelayEnabled():
   RouterBgpSharedModelet.addCommandClass( RibInDelayCmd )

#-----------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR \
#     next-hop resolution v4-mapped-v6 translation [disabled]"
#  in "config-bgp-af" mode
#-----------------------------------------------------------------------------
class NeighborV4MappedV6NextHopTranslationCommand( BgpCmdBaseClass ):
   syntax = "neighbor PEER next-hop resolution v4-mapped-v6 translation [ disabled ]"
   noOrDefaultSyntax = \
      "neighbor PEER next-hop resolution v4-mapped-v6 translation ..."
   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'next-hop': bgpTokens.neighborNextHop,
      'resolution': bgpTokens.resolution,
      'v4-mapped-v6': bgpTokens.v4MappedV6,
      'translation': bgpTokens.v4MappedV6Translation,
   } )
   handler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborV4MappedV6NextHopTranslationCommand_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler." + \
      "NeighborV4MappedV6NextHopTranslationCommand_noOrDefaultHandler"

for m in [ RouterBgpBaseAfIpv6UniMode, RouterBgpVrfAfIp6Mode ]:
   m.addCommandClass( NeighborV4MappedV6NextHopTranslationCommand )

RouterBgpAfLabelSharedModelet.addCommandClass(
   NeighborV4MappedV6NextHopTranslationCommand )

# -------------------------------------------------------------------------------
# "[no|default] neighbor PEER route-flap-damping
#                           ( ( RFD_POLICY_NAME <> ) | disabled )"
# command, in "router-bg-af" mode.
# -------------------------------------------------------------------------------

rfdLogOptions = {
      bgpTokens.rfdLogPathEvents.keyword_: bgpTokens.rfdLogPathEvents.helpdesc_,
      bgpTokens.rfdLogConfigEvents.keyword_: bgpTokens.rfdLogConfigEvents.helpdesc_
}
class NeighborRFDCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER route-flap-damping \
         ( ( RFD_POLICY_NAME [monitor-only] [ log RFD_LOG_OPTIONS ]) | \
         disabled )'
   noOrDefaultSyntax = 'neighbor PEER route-flap-damping ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'route-flap-damping': bgpTokens.rfdKw,
      'RFD_POLICY_NAME': rfdPolicyNameMatcher,
      'monitor-only': bgpTokens.rfdMonitorOnlyKw,
      'log': bgpTokens.rfdLog,
      'RFD_LOG_OPTIONS': CliCommand.SetEnumMatcher( rfdLogOptions ),
      'disabled': bgpTokens.rfdDisabledKw,
      } )
   handler = "RoutingBgpNeighborCliHandler.NeighborRFDCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborRFDCmd_noOrDefaultHandler"

modes = [
      RouterBgpBaseAfIpUniMode,
      RouterBgpBaseAfIpv6UniMode,
      RouterBgpVrfAfIpMode,
      RouterBgpVrfAfIp6Mode,
]
for m in modes:
   m.addCommandClass( NeighborRFDCmd )

# -------------------------------------------------------------------------------
# "[no|default] neighbor GROUP | ADDR session tracker <name>"
# command, in "router-bgp" mode.
# -------------------------------------------------------------------------------
class NeighborMultipathDisabledCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER multi-path [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER multi-path ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': PeerCliExpression,
      'multi-path': 'Multi-path for a peer/peer-group'
   } )
   handler = "RoutingBgpNeighborCliHandler.NeighborMultipathDisabledCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.NeighborMultipathDisabledCmd_noOrDefaultHandler"

# Register command in ip(6)unicast and ip(6)labeled-unicast address-families
# for now
for m in [ RouterBgpAfIpUniSharedModelet, RouterBgpAfLabelSharedModelet ]:
   m.addCommandClass( NeighborMultipathDisabledCmd )

# -----------------------------------------------------------------------------------
# [ no | default ] neighbor PEER peer segment node [disabled]
# command, in "router-bgp" mode (see AID11677). Note that this command is *not*
# supported with IPv6 link-local peers.
# -----------------------------------------------------------------------------------
class SetNeighborPeerSegmentNodeCmd( BgpCmdBaseClass ):
   syntax = 'neighbor PEER peer segment node [ disabled ]'
   noOrDefaultSyntax = 'neighbor PEER peer segment node ...'
   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'PEER': V4OrV6OrPeerGroupCliExpression,
      'peer': bgpTokens.peer,
      'segment': 'BGP Link State peer segment configuration',
      'node': 'Generate a BGP Link State peer node segment for this neighbor'
   } )

   handler = "RoutingBgpNeighborCliHandler.SetNeighborPeerSegmentNode_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNeighborPeerSegmentNode_noOrDefaultHandler"

# Note that PeerNode segments can only be configured in the default VRF, so the
# command is added to RouterBgpBaseMode (and not RouterBgpSharedModelet).
if RoutingLibToggleLib.toggleBgpLsProducerBgpEpeEnabled():
   RouterBgpBaseMode.addCommandClass( SetNeighborPeerSegmentNodeCmd )

# ---------------------------------------------------------------------------------
# "[no|default] neighbor ( default | PEER ) received attribute discard <Attributes>"
# command, in "router-bgp" mode.
# ---------------------------------------------------------------------------------
class SetNbrRcvdBgpAttrFilteringCmd( BgpCmdBaseClass ):
   _nei = 'neighbor default '
   if BgpCommonToggleLib.toggleArBgpNbrRcvdBgpAttributeFilteringEnabled():
      _nei = 'neighbor ( default | PEER ) '
   syntax = _nei + 'received attribute discard ( { BGP_ATTRIBUTES } | [ disabled ] )'
   noOrDefaultSyntax = _nei + 'received attribute discard'

   data = BgpCmdBaseClass._createSyntaxData( {
      'neighbor': bgpTokens.neighbor,
      'default': bgpTokens.neighborDefaultGlobal,
      'PEER': PeerCliExpression,
      'received': CliCommand.Node( matcher=bgpTokens.receivedPathAttr,
                                   guard=hwSupportedGuard ),
      'attribute': bgpTokens.attribute,
      'discard': bgpTokens.discardPathAttr,
      'BGP_ATTRIBUTES': bgpTokens.BgpAttributeMultiRangeExpr,
   } )
   handler = "RoutingBgpNeighborCliHandler.SetNbrRcvdBgpAttrFilteringCmd_handler"
   noOrDefaultHandler = \
      "RoutingBgpNeighborCliHandler.SetNbrRcvdBgpAttrFilteringCmd_noOrDefaultHandler"

RouterBgpSharedModelet.addCommandClass( SetNbrRcvdBgpAttrFilteringCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global sharedSecretProfileConfig

   bgpConfig = ConfigMount.mount( entityManager,
                                  'routing/bgp/config',
                                  'Routing::Bgp::Config', 'w' )
   sharedSecretProfileConfig = LazyMount.mount( entityManager,
                            "mgmt/security/sh-sec-prof/config",
                            "Mgmt::Security::SharedSecretProfile::Config", "r" )
