# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for BGP exec mode commands

# pylint: disable=anomalous-backslash-in-string, protected-access
# pylint: disable=consider-using-f-string
from __future__ import absolute_import, division, print_function
import Tac, LazyMount, Tracing, os, sys, Cell
import AgentDirectory
import BasicCli
import CliToken.Clear, CliToken.Ip, CliToken.Ipv6
import CliToken.RoutingBgpShowCliTokens as ShowTokens
import Arnet
from BgpLib import (
      PeerConfigKey,
      bgpConfigAttrsAfMap,
      peerConfigAttrsAfMap,
)
from CliModel import List
from CliPlugin import VrfCli
from CliPlugin import AclCli
from CliPlugin import BgpCliModels
from CliPlugin import BgpNeighborsCliModels
from CliPlugin import Ip6AddrMatcher
from CliPlugin import IpAddrMatcher
from CliPlugin.IpGenAddrMatcher import isIpv4
from CliPlugin.VrfCli import (
      VrfExecCmdDec,
      generateVrfCliModel,
)
from CliPlugin.RoutingBgpCli import ( ipv4PeerMatcher, ipv6PeerMatcher,
                            ipv6LlPeerMatcher, peergroupNameMatcher )
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from ConfigConsistencyChecker import UndefinedReferenceChecker
import re
import SmashLazyMount
import ConfigMount
import CliCommand
import CliGlobal
import CliMatcher
import ShowCommand
import six

traceHandle = Tracing.Handle( 'Bgp' )
# pylint: disable=multiple-statements, no-self-use
# pylint: disable=undefined-loop-variable, redefined-outer-name, unused-variable
t0 = traceHandle.trace0
t1 = traceHandle.trace1
t2 = traceHandle.trace2

gv = CliGlobal.CliGlobal( dict( sysname=None, skipAgentCheck=None ) )
bgpConfig = None
bgpClearReq = None
bgpClearResp = None
aclListConfig = None
bgpVrfClearReqDir = None
bgpVrfClearRespDir = None
bgpVrfConfigDir = None
bgpLuLfib = None 
l3Config = None
l3ProtocolAgentModelStatus = None
defaultVrfProtocolStatus = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
bgpTestControlReq = None

# Global dictionary to overloaded functions for ArBgp support
arBgpShowCmdDict = {}
ribdBgpShowCmdDict = {}

# Address family Identifier
AFI_IP = 1
AFI_IP6 = 2
# Sub Address family Identifier
PA4MP_SAFI_UNI = 1
PA4MP_SAFI_SR_TE = 73

# Following APIs enable us to avoid direct symbolic dependency on Ribd in
# similar way as arBgpShowCmdDict based mechanism works for ArBgp CLI
# implementation
def cliRibdShowCommand():
   cb = ribdBgpShowCmdDict.get( 'cliRibdShowCommand' )
   assert cb is not None
   return cb

def showRibCapiCommand():
   return ribdBgpShowCmdDict.get( 'showRibCapiCommand' )

def showRibDamiCommand():
   return ribdBgpShowCmdDict.get( 'showRibDamiCommand' )

def EmptyResponseException():
   func = ribdBgpShowCmdDict.get( 'getEmptyResponseException' )
   return func() if callable( func ) else None

def AmiResponseException():
   func = ribdBgpShowCmdDict.get( 'getAmiResponseException' )
   return func() if callable( func ) else None

def getBgpPeerListModel():
   func = ribdBgpShowCmdDict.get( 'getBgpPeerListModel' )
   return func() if callable( func ) else None

def getBgpPeerModel():
   func = ribdBgpShowCmdDict.get( 'getBgpPeerModel' )
   return func() if callable( func ) else None

invalidPosixRegexErrMsg = 'Invalid posix regular expression'

# pylint: disable=useless-return
def doShowBgpInactive( mode, *args, **kwargs ):
   mode.addError( 'BGP inactive' )
   return None

def doShowBgpOutputNotSupported( mode, *args, **kwargs ):
   mode.addError( "The command is only supported in the"
                 " multi-agent routing protocol model." )
   return None

def doShowBgpOutputNotSupportedInMultiAgent( mode, *args, **kwargs ):
   mode.addError( "The command is only supported in the"
                  " ribd routing protocol model." )
   return None
# pylint: enable=useless-return

class ArBgpShowOutput:
   '''
   Decorator class that can be used to tag that a certain cli command's handler
   function can be redirected to Arbgp implementation, based on whether ribd is
   running or Arbgp is running.
   Function calls using the new cli parser should avoid using positional arguments
   in place of keyword arguments
   @vrfLookup: For commands which are supported in both ribd and arbgp modes,
   this decorator will lookup and set the vrfName parameter.  For commands which
   are arbgp only this is not done, unless the vrfLookup=True is also set.
   '''
   def __init__( self, arBgpShowCallback, arBgpModeOnly=False,
                 ribdModeOnly=False, vrfLookup=False, requiresAgent=True ):
      self.arBgpShowCallback = arBgpShowCallback
      self.arBgpModeOnly = arBgpModeOnly
      self.ribdModeOnly = ribdModeOnly
      self.vrfLookup = vrfLookup or not arBgpModeOnly
      self.requiresAgent = requiresAgent
      assert not ( arBgpModeOnly and ribdModeOnly )
   def __call__( self, func ):
      def arBgpShowOutput( *args, **kwargs ):
         protocolAgentModel = l3Config.protocolAgentModel
         if protocolAgentModel == ProtoAgentModel.multiAgent:
            if self.ribdModeOnly:
               return doShowBgpOutputNotSupportedInMultiAgent( *args, **kwargs )
            agentMissing = self.requiresAgent and not gv.skipAgentCheck and \
                           not AgentDirectory.agentIsRunning( gv.sysname, 'Bgp' )
            if protocolAgentModel != \
                  l3ProtocolAgentModelStatus.protocolAgentModel or \
                  not defaultVrfProtocolStatus.entityPtr.get( 'Bgp' ) or \
                  agentMissing:
               return doShowBgpInactive( *args, **kwargs )
            elif self.arBgpModeOnly:

               callback = arBgpShowCmdDict.get( self.arBgpShowCallback,
                                                func )
            else:
               callback = arBgpShowCmdDict.get( self.arBgpShowCallback,
                     doShowBgpOutputNotSupportedInMultiAgent )
            if self.vrfLookup:
               # The VrfExecCmdDec sets up the vrfName correctly even when the cmd
               # is being executed under a routing-context and the vrfName is not
               # specified as an argument. In case of ArBgp though, some show cmds 
               # have C++ implementation which do not use the VrfExecCmdDec, due to
               # which these commands give default vrf output even when they are run
               # under some routing-context vrf mode.
               # The fix is to do the vrf lookup here, before invoking the 
               # corresponding show command function in ArBgpCli.py with the correct
               # vrfName.
               # There is a special case for "show ip bgp peer-group" command where
               # vrfName is not part of kwargs but rather it is part of a list 
               # represented by kwargs['pgAndVrfName']
               vrfName = kwargs['pgAndVrfName'][1] if self.arBgpShowCallback == \
                         "doShowIpBgpPeerGroup" else kwargs.get( 'vrfName', None )
               vrfName = VrfCli.vrfMap.lookupCliModeVrf( args[ 0 ], vrfName )
               if self.arBgpShowCallback == "doShowIpBgpPeerGroup":
                  kwargs['pgAndVrfName'] = ( kwargs['pgAndVrfName'][0], vrfName )
               elif 'vrfName' in kwargs:
                  kwargs['vrfName'] = vrfName
            return callback( *args, **kwargs )
         else:
            if not self.arBgpModeOnly:
               return func( *args, **kwargs )
            else:
               return doShowBgpOutputNotSupported( *args, **kwargs )
      return arBgpShowOutput

class BgpVrfRoutingContextDefaultOnly:
   """Decorator class to process show commands according to the VRF in
   routing-context VRF mode.
   These show commands should display output only in case of default VRF.
   For non-default VRF, no output should be displayed.
   """
   def __init__( self, cliModel ):
      """Takes cliModel with which the cmd is registered as constructor arg"""
      self.cliModel = cliModel

   def __call__( self, func ):
      def bgpVrfRoutingContext( *args, **kwargs ):
         """Performs a vrf lookup and returns a default( empty ) instance of
         cliModel if vrfName is not default.
         For non-default VRF, the text output will be empty and CAPI output will
         be an empty model.
         """
         mode = args[ 0 ] # mode is always the first arg
         vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName=None )
         if vrfName != DEFAULT_VRF:
            return self.cliModel()

         return func( *args, **kwargs )

      return bgpVrfRoutingContext

BGP_ROUTE_V6            = 0x0001
BGP_ROUTE_DETAIL        = 0x0002

BGP_NOT_SPECIFIED = 0x00
BGP_RECEIVED_ROUTES = 0x01
BGP_ADVERTISED_ROUTES = 0x02
BGP_RECV_FILTERED_ROUTES = 0x03
BGP_NOT_INSTALLED_ROUTES = 0x4
BGP_INSTALLED_ROUTES = 0x5

cliribdFastPort = "9889"

BGP_SAFI_ALL = 0
BGP_SAFI_UNICAST = 1
BGP_SAFI_LABELS = 2
BGP_SAFI_SR_TE = 3

# pylint: disable=unnecessary-lambda
allVrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='VRF name',
      inclDefaultVrf=True, inclAllVrf=True )

routeSummaryVrfModel = generateVrfCliModel( BgpCliModels.BgpRouteHeader,
                                            'Per VRF BGP route summary',
                                            uncheckedModel=True,
                                            revision=3 )

queuedWithdrawalsVrfModel = generateVrfCliModel(
   BgpCliModels.BgpQueuedWithdrawals,
   'Per VRF BGP queued withdrawals',
   uncheckedModel=True )

COMMUNITY_NO_EXPORT = 0xffffff01 #no export outside confederation
COMMUNITY_NO_ADVERTISE = 0xffffff02 #no advertise to other peers
COMMUNITY_NO_EXPORT_SUB = 0xffffff03 #no export to external peers
COMMUNITY_INTERNET = 0 #community internet
COMMUNITY_GSHUT = 0xffff0000 #community GSHUT

def getCommunityValuesScalarList( commValues ):
   commList = []
   for cv in commValues:
      if isinstance( cv, six.integer_types ):
         commList.append( cv )
      elif cv == 'no-export':
         commList.append( COMMUNITY_NO_EXPORT )
      elif cv == 'no-advertise':
         commList.append( COMMUNITY_NO_ADVERTISE )
      elif cv == 'local-as':
         commList.append( COMMUNITY_NO_EXPORT_SUB )
      elif cv == 'internet':
         commList.append( COMMUNITY_INTERNET )
      elif cv == 'GSHUT':
         commList.append( COMMUNITY_GSHUT )
      elif ':' in cv:
         ( asn, num ) = cv.split( ':' )
         commList.append( int( asn ) << 16 | int( num ) & 0xffff )

   return commList

# vrfName is defaulted to None so as to allow cliRibdShowCommand to
# deduce the correct vrfName based on the routing context the cli is in.
def _doShowIpBgpRoute( mode, ipv6=False, prefix=None, detail=None,
                       commValues=None, commListName=None, exact=None, peer=None,
                       showPeerColumn=False, familyFilterType=None, filterType=None,
                       aspRegex=None, commRegex=None,
                       highPerf=False, flag=0, longerPrefixes=False,
                       internal=None, vrfName=None, labUni=None, srTe=None,
                       largeCommValues=None, largeCommListName=None ):

   cliRibd = False
   cliRibdCmd = ""

   args = {}
   if ipv6:
      flag |= BGP_ROUTE_V6
      args = { 'ipv6' : int( ipv6 ) }
   cmd = 'MIO_DGET_BGP_ROUTE'
   mlen = None
   if srTe is not None:
      args[ 'safi_filter_type' ] = BGP_SAFI_SR_TE
   elif labUni:
      args[ 'safi_filter_type' ] = BGP_SAFI_LABELS
   else:
      args[ 'safi_filter_type' ] = BGP_SAFI_ALL
   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if detail is not None:
      flag |= BGP_ROUTE_DETAIL
      args[ 'show_detail' ] = 1
      if srTe is not None:
         # For SR-TE SAFI, aggregation and policy advertisement
         # are not supported. Hence only check if the tunnel encap
         # has to be displayed
         args[ 'show_sr_te_tunnel_encap' ] = 1
         if prefix is not None:
            args[ 'show_prefix_detail' ] = 1
      else:
         args[ 'show_contributors' ] = 1
         if not longerPrefixes and prefix is not None and '/' in prefix:
            args[ 'show_prefix_detail' ] = 1
   else:
      args[ 'show_detail' ] = 0
   # Use Rib fast port for show ip|ipv6 bgp prefix
   if prefix is not None:
      if '/' in prefix:
         addr, mlen = prefix.split( '/' )
         args[ 'walkdown' ] = int( longerPrefixes )
      else:
         addr = prefix
         args[ 'walkup' ] = 1
      if srTe is not None:
         if isIpv4( addr ):
            args[ 'addrSrTeV4' ] = addr
         else:
            args[ 'addrSrTeV6' ] = addr
      elif isIpv4(addr):
         args[ 'addrv4' ] = addr
         if mlen:
            args[ 'masklenv4' ] = int( mlen )
      else:
         args[ 'addrv6' ] = addr
         if mlen:
            args[ 'masklenv6' ] = int( mlen )
      if not longerPrefixes:
         args[ 'show_detail' ] = 1
   if peer is not None:
      if peer.type == 'peerIpv4':
         key = 'peerv4'
      elif peer.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         key = 'peerv6'
      else:
         assert False, "Cannot handle peer type: %s" % ( peer.type )

      args[ key ] = peer.gatedRepr()

      # Right now, options for the above commands (communities, detail, etc.) 
      # will still go through RibCapi
      peerRouteFilters = { BGP_NOT_SPECIFIED: 'routes',
                           BGP_RECEIVED_ROUTES: 'received-routes',
                           BGP_ADVERTISED_ROUTES: 'advertised-routes',
                           BGP_RECV_FILTERED_ROUTES: 'received-routes filtered' }

      if ( detail or prefix or commValues or aspRegex or commRegex or labUni or srTe
           or highPerf or largeCommValues ):
         cliRibd = False
      elif filterType in peerRouteFilters:
         routeType = peerRouteFilters[ filterType ]
         cliRibd = True
         cliRibdCmd = 'show ip%s bgp' % [ '', 'v6' ][ ipv6 ]
         cliRibdCmd += ' peer %s ' % ( peer.gatedRepr() )
         if familyFilterType:
            cliRibdCmd += 'ipv6 ' if familyFilterType == BGP_V6_ROUTES else 'ipv4 '
         cliRibdCmd += '%s' % routeType
   else:
      #
      # For the following show commands, fall back to cliribd for performance:
      #
      # 'show ip{v6} bgp installed'
      # 'show ip{v6} bgp not-installed'
      #
      routeFilters = { BGP_INSTALLED_ROUTES : 'installed',
                       BGP_NOT_INSTALLED_ROUTES : 'not-installed' }
      if filterType in routeFilters:
         routeType = routeFilters[ filterType ]
         cliRibd = True
         cliRibdCmd = 'show ip%s bgp' % [ '', 'v6' ][ ipv6 ]
         cliRibdCmd += ' %s' % routeType
   if showPeerColumn:
      args[ 'show_peers' ] = 1

   if familyFilterType is not None:
      args[ 'family_filter_type' ] = familyFilterType
   if filterType is not None:
      args[ 'filter_type' ] = filterType
   if commValues is not None:
      args[ 'communities' ] = getCommunityValuesScalarList( commValues )
   if largeCommValues is not None:
      args[ 'largeCommunities' ] = largeCommValues
   if exact is not None:
      args[ 'exact' ] = 1
   if aspRegex is not None:
      args[ 'aspath_regex' ] = aspRegex
   if commRegex is not None:
      args[ 'comm_regex' ] = commRegex
   if commListName is not None:
      if exact is not None:
         commListName = ".exact." + commListName
      args[ 'commlist_filter' ] = commListName
   if largeCommListName is not None:
      if exact is not None:
         largeCommListName = ".exact." + largeCommListName
      args[ 'largeCommList_filter' ] = largeCommListName

   if cliRibd and mode.session_.outputFormat_ == 'text':
      cliRibdShowCommand()( mode, cliRibdCmd, clientName="BGP", vrf=vrfName )
      return None
   elif highPerf and mode.session_.outputFormat_ == "text" and prefix is None:
      args[ 'format' ] = 2 
      return showRibDamiCommand()( sys.stdout.fileno(), cmd, args,
                                   clientName='BGP' )
   else:
      try:
         args = args or None
         return showRibCapiCommand()( mode, BgpCliModels.BgpRouteHeader,
                                      cmd, args, clientName='BGP', 
                                      highPerf=highPerf )
      except ( EmptyResponseException(), AmiResponseException() ):
         return None

#-------------------------------------------------------------------------------
# Explanation of Bgp/ArBgp command decorators
# The ordering of the decorators on these commands matters, but is implementation
# dependent

# ArBgpShowOutput takes a key to the arBgpShowCmdDict (see ArBgp CliPlugins).
# This key determines which function to call

# VrfExecCmdDec accepts a vrf function, which is uses to generate a series of
# vrfs, and a cliModel. This cli model can have the __cliPrinted__ attribute set
# to true, which means that it has already been written to console by the
# cli printer. 

# This is where the ordering matters. If the command referenced by the key in
# ArBgpShowOutput uses the c++ implementation that renders using cli print then
# it is placed first. These commands should return a cliPrinted model.
# The VrfExecCmdDec will then exit without doing any additional work,
# as the output has already been rendered

# If VrfExecCmdDec is placed first then it will iterate over each vrf on the
# test device and call the ArBgp function for each vrf name. With cli printed
# commands this will result in multiple renders and, in json output, multiple
# objects where there should be only one
#------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "show ip bgp [labeled-unicast] [ <ip> | <prefix> | ( installed | not-installed ) ]
#  [ longer-prefixes ] [internal] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgp' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgp( mode, addr=None, longerPrefixes=None, detail=None, internal=None,
                 vrfName=None, labUni=None ):
   flag = 0
   highPerf = True
   longerPrefixes = bool( longerPrefixes )
   if detail:
      flag |= BGP_ROUTE_DETAIL
   routeType = BGP_NOT_SPECIFIED 

   result = _doShowIpBgpRoute( mode, prefix=addr, detail=detail,
                               highPerf=highPerf, flag=flag,
                               longerPrefixes=longerPrefixes,
                               filterType=routeType,
                               vrfName=vrfName, labUni=labUni )
   return result

#-------------------------------------------------------------------------------
# "show ipv6 bgp [labeled-unicast] [ <ip> | <prefix> | [installed] | 
# [ not-installed ] ] [ longer-prefixes ] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6Bgp' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6Bgp( mode, addr=None, longerPrefixes=None, detail=None, vrfName=None,
                   internal=None, labUni=None ):
   highPerf = True
   longerPrefixes = bool( longerPrefixes )
   routeType = BGP_NOT_SPECIFIED
   # pylint: disable=singleton-comparison
   if addr != None:
      prefix = addr.stringValue
   else:
      prefix = None
   # pylint: enable=singleton-comparison
   result = _doShowIpBgpRoute( mode, prefix=prefix, detail=detail,
                               longerPrefixes=longerPrefixes,
                               filterType=routeType,
                               ipv6=True, highPerf=highPerf, vrfName=vrfName,
                               labUni=labUni )
   return result

#-------------------------------------------------------------------------------
# Deprecated version of the command:
# "show ipv6 bgp community [<communities>] [exact] [detail] [vrf <vrfName>]"
#
# New Version of the command:
# "show ipv6 bgp match community [<communities>] [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpRoutesCommunities' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesCommunities( mode, values=None, exact=None, detail=None,
                                    internal=None, vrfName=None,
                                    labUni=None, epeOnly=None ):
   result = _doShowIpBgpRoute( mode, labUni=labUni,
                               detail=detail,
                               commValues=set( values ),
                               exact=exact, ipv6=True,
                               vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpRoutesDetail' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesDetail( mode, detail=None,
                               internal=None, vrfName=None,
                               labUni=None, epeOnly=None ):
   highPerf = True
   result = _doShowIpBgpRoute( mode, detail=detail, ipv6=True,
                               vrfName=vrfName, highPerf=highPerf,
                               labUni=labUni )
   return result

#----------------------------------------------------------------------
# show ip bgp large-community [<large-community name>] [exact] [detail]
#----------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpLargeCommunity', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpLargeCommunity( mode, values=None, detail=None,
                               vrfName=None, exact=None, labUni=None,
                               epeOnly=None ):

   result = _doShowIpBgpRoute( mode, labUni=labUni, detail=detail,
                               vrfName=vrfName, exact=exact,
                               largeCommValues=set( values ) )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpLargeCommunity', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpLargeCommunity( mode, values=None, detail=None,
                               labUni=None, epeOnly=None,
                               vrfName=None, exact=None ):

   result = _doShowIpBgpRoute( mode, detail=detail, ipv6=True,
                               vrfName=vrfName, exact=exact, labUni=labUni,
                               largeCommValues=set( values ) )
   return result

#-------------------------------------------------------------------------------
# "show ipv6 bgp community-list [<community-list name>] [exact] [detail]
# [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpRoutesCommunityList' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesCommunityList( mode, commListName=None, exact=None,
                                      labUni=None, epeOnly=None,
                                      detail=None, vrfName=None ):
   result = _doShowIpBgpRoute( mode, detail=detail,
                               commListName=commListName,
                               exact=exact, ipv6=True,
                               labUni=labUni,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
#"show ip(v6) bgp large-community-list [<large-community-list name>] [exact]
# [detail]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesLargeCommunityList', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesLargeCommunityList( mode, largeCommListName=None, exact=None,
        detail=None, vrfName=None, labUni=None, epeOnly=None ):
   result = _doShowIpBgpRoute( mode, labUni=labUni, detail=detail,
                               largeCommListName=largeCommListName,
                               exact=exact,
                               vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpRoutesLargeCommunityList', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesLargeCommunityList( mode, largeCommListName=None, exact=None,
        detail=None, vrfName=None, labUni=None, epeOnly=None ):
   result = _doShowIpBgpRoute( mode, labUni=labUni, detail=detail,
                               largeCommListName=largeCommListName,
                               exact=exact, ipv6 = True,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip[v6] bgp neighbors [<ip>] [bfd] [vrf <vrfName>]"
#     (hidden options) [internal [verbose]] [task-stats [reset]]
#-------------------------------------------------------------------------------
peerListVrfModel = generateVrfCliModel( BgpNeighborsCliModels.BgpPeerList,
                                        "Per Vrf BGP Peer Information",
                                        revision=2 )

def _doShowIpBgpNeighbors( mode, addr=None, bfd=None, internal=None,
                           vrfName=None, ipv6=False, highPerf=True ):
   cmd = 'MIO_DGET_BGP_PEER'
   args = {}

   if vrfName and vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   if ipv6:
      args[ 'ipv6' ] = True

   if addr is not None:
      if addr.type == 'peerIpv4':
         key = 'addrv4'
      elif addr.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         key = 'addrv6'
      else:
         assert False, "Cannot handle peer type: %s" % ( addr.type )

      args[ key ] = addr.gatedRepr()

   peerModel = getBgpPeerListModel()

   try:
      retval = showRibCapiCommand()( mode, peerModel, 
                                     cmd, args, clientName='BGP', 
                                     highPerf=highPerf,
                                     skipNullResponses=False )
      if retval is not None and bfd is not None:
         retval._showBfd = True
      return retval
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowIpBgpNeighbors' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=peerListVrfModel )
def doShowIpBgpNeighbors( mode, addr=None, bfd=None, internal=None,
                          taskStats=None, iar=None, vrfName=None, routerId=None ):
   return _doShowIpBgpNeighbors( mode, addr, bfd, internal, vrfName )

@ArBgpShowOutput( 'doShowIpv6BgpNeighbors' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=peerListVrfModel )
def doShowIpV6BgpNeighbors( mode, addr=None, bfd=None, internal=None,
                            taskStats=None, iar=None, vrfName=None,
                            routerId=None ):
   return _doShowIpBgpNeighbors( mode, addr, bfd, internal, vrfName,
                                 ipv6=True )

#-------------------------------------------------------------------------------
# "show bgp
#     ( ( ipv4 unicast ) | ( ipv6 unicast ) | ( ipv4 multicast ) |
#       ( ipv6 multicast ) | ( ipv4 labeled-unicast ) | ( ipv6 labeled-unicast ) )
#     [ community [<communities>] [exact] ] [detail]
#     [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNlriAf' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowBgpNlriAf( mode, addressFamilyAndType=None,
                     pfxAddr=None, longerPrefixes=None,
                     communityValuesAndExact=None, aspListName=None,
                     detail=None, vrfName=None, internal=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show bgp neighbors [<ip>] [vrf <vrfName>]"
#     (hidden options) [internal [verbose] [rib-out] [dump-duplicates]]
#                      [task-stats [reset]]
#
# Note: 'verbose' option prints AdjRibin paths
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNeighbors' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=peerListVrfModel )
def doShowBgpNeighbors( mode, addr=None, internal=None, taskStats=None,
                        vrfName=None ):
   doShowBgpOutputNotSupported( mode )


#-------------------------------------------------------------------------------
# "show ip bgp
#     [labeled-unicast]
#     community <communities> [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesCommunities' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesCommunities( mode, values=None, exact=None, detail=None,
                                  labUni=None, epeOnly=None, vrfName=None ):
   result = _doShowIpBgpRoute( mode,
                               labUni=labUni,
                               commValues=set( values ),
                               exact=exact,
                               detail=detail,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp community regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesCommunitiesRegex' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesCommunitiesRegexp( mode, regex=None, vrfName=None ):
   result = showBgpCommunityRegexpHelper( mode, regex, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp [labeled-unicast] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesDetail' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesDetail( mode, detail=None, vrfName=None,
                             labUni=None, epeOnly=None ):
   highPerf = True
   flag = 0
   if detail:
      flag |= BGP_ROUTE_DETAIL
   result = _doShowIpBgpRoute( mode, detail=detail,
                               highPerf=highPerf, flag=flag,
                               vrfName=vrfName, labUni=labUni )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp as-path access-list <name> [vrf <vrfName>] [detail]"
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesAsPath' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesAsPath( mode, detail=None, aspListName=None,
                             labUni=None, vrfName=None ):

   result = _doShowIpBgpRoute( mode, detail=detail,
                               labUni=labUni,
                               vrfName=vrfName )

   return result

# -------------------------------------------------------------------------------
# "show ipv6 bgp as-path access-list <name> [vrf <vrfName>] [detail]"
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpRoutesAsPath' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRoutesAsPath( mode, detail=None, aspListName=None,
                               labUni=None, vrfName=None ):

   result = _doShowIpBgpRoute( mode, detail=detail,
                               labUni=labUni,
                               ipv6=True,
                               vrfName=vrfName )
   return result

# -------------------------------------------------------------------------------
# "show ip bgp [labeled-unicast] community-list [<community-list name>] [exact]
#     [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRoutesCommunityList' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpRoutesCommunityList( mode, commListName=None, exact=None, detail=None,
                                    labUni=None, epeOnly=None, vrfName=None ):
   result = _doShowIpBgpRoute( mode, detail=detail,
                               labUni=labUni,
                               commListName=commListName,
                               exact=exact,
                               vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp peer-group [peergroupName] [vrf <vrfName>]"
#-------------------------------------------------------------------------------

def doParseNeighborIntoDict( mode, family, vrfName ):
   '''
   Sends the correct CLI w.r.t to af to cliribd and
   receives the output back in dict format
   '''
   cmd = 'show %s bgp' % ( 'ip' if family == 'v4' else 'ipv6' )
   cmd += ' neighbor pyformat'
   out = cliRibdShowCommand()( mode, cmd, clientName="BGP", output = False,
                               vrf=vrfName )
   # pylint: disable=eval-used
   return eval( '{' + out.replace( '\n', "" ) + '}' )
   # pylint: enable=eval-used

def doParseDynamicNeighborsIntoDict( mode, vrfName, pgName ):
   '''
   Sends the correct CLI w.r.t to peer-group info to cliribd and
   receives the output back in dict format
   '''
   cmd = 'show ip bgp'
   cmd += ' peer-group %s pyformat' % pgName
   out = cliRibdShowCommand()( mode, cmd, clientName="BGP", output = False,
                               vrf=vrfName )
   # pylint: disable=eval-used
   return eval( '{' + out.replace( '\n', "" ) + '}' )
   # pylint: enable=eval-used

def configForVrf( vrfName ):
   config = bgpConfig
   if vrfName != DEFAULT_VRF:
      if vrfName in bgpVrfConfigDir.vrfConfig:
         config = bgpVrfConfigDir.vrfConfig[ vrfName ]
      else:
         return None
   return config

# We get pyformat dicts from cliribd for "show ip[v6] bgp neighbors".
# We fetch relevant information from these dicts and use it in this show command for
# static peer group members
# For dynamic peer groups, we get pyformat dicts for 'show ip bgp peer-group <name>'
# and fetch the information from these dicts. Peer-group information 
# from different VRFs is then put together

# example:
# peerGroups['pg1' ] = 
#       {'as':100,
#        'static':{'vrf1':{'v4':[],'v6':[]},'vrf2':{'v4':[],'v6':[]},...},
#        'listen':{'vrf1':{'networkAndAs':[],'vrf2':{'networkAndAs':[]},... },
#        'dynamic':{'vrf1':{},'vrf2':{},...}}
def compilePeerGroupInfo( mode, name='', vrfName=None ):
   '''Prepare our list of configured peer Groups and its attributes
      peerGroups members are keyed by peer-group name'''
   # top-level PeerGroup info dict
   peerGroups = {}
   # empty peer-group info dicts that can be cloned
   peerGroupMembersEmpty = { 'v4' : [], 'v6' : [] }
   peerGroupListenRangeEmpty = { 'networkAndAs' : [] }

   def setPeerGroupDefaults( peerGroups, key, vrf ):
      peerGroups.setdefault( key, { 'as' : None, 'static' : {},
                                    'listen' : {}, 'dynamic' : {}, 'intf' : {} } )
      peerGroups[ key ][ 'static' ].setdefault( vrf, { 'v4' : [], 'v6' : [] } )
      peerGroups[ key ][ 'listen' ].setdefault( vrf, { 'networkAndAs' : [] } )
      peerGroups[ key ][ 'dynamic' ].setdefault( vrf, {} )
      peerGroups[ key ][ 'intf' ].setdefault( vrf, False )

   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName == ALL_VRF_NAME:
      vrfList = VrfCli.getAllVrfNames()
   else:
      vrfList = [ vrfName ]

   defaultVrfBc = configForVrf( DEFAULT_VRF )
   for vrf in vrfList:
      bc = configForVrf( vrf )
      if bc is None:
         continue
      # iterate over static peers first
      # pylint: disable=consider-using-in
      for peer in bc.neighborConfig.values():
         if peer.isPeerGroupPeer:
            key = peer.peerGroupKey.group
            if name and key != name:
               continue
            setPeerGroupDefaults( peerGroups, key, vrf ) 
            if peer.key.type == 'peerIpv4':
               peerGroups[ key ][ 'static' ][ vrf ][ 'v4' ].append( peer.key )
            elif peer.key.type == 'peerIpv6' or peer.key.type == 'peerIpv6Ll':
               peerGroups[ key ][ 'static' ][ vrf ][ 'v6' ].append( peer.key )
            peerGroups[ key ][ 'as' ] = \
                  defaultVrfBc.neighborConfig[ peer.peerGroupKey ].asNumber
         elif peer.key.group == 'peerGroup':
            if name and key != name:
               continue
            key = peer.key.group
            setPeerGroupDefaults( peerGroups, key, vrf )
            peerGroups[ key ][ 'as' ] = peer.asNumber
      # pylint: disable=consider-using-in

      # iterate over listen-range config next
      for lr in bc.listenRange.values():
         key = lr.peergroupName
         if name and key != name:
            continue
         setPeerGroupDefaults( peerGroups, key, vrf )
         for network in lr.prefixList:
            peerGroups[ key ][ 'listen' ][ vrf ][ 'networkAndAs' ].append( 
               ( str( network ), lr.prefixList[ network ] ) )
         dynPeerGroup = doParseDynamicNeighborsIntoDict( mode, vrf, key )
         if key in dynPeerGroup:
            peerGroups[ key ][ 'dynamic' ][ vrf ] = dynPeerGroup[ key ]

      # iterate over the "neighbor interface" config
      for peer in bc.neighIntfConfig.values():
         key = peer.peerGroup.group
         # if a peer group filter is specified, consider only if it matches
         # the filter
         if name and key != name:
            continue
         setPeerGroupDefaults( peerGroups, key, vrf )
         peerGroups[ key ][ 'intf' ][ vrf ] = True

   return peerGroups

desc = "Per-VRF BGP peer groups"
peerGroupVrfModel = generateVrfCliModel( BgpCliModels.BgpPeerGroups, desc )

# The following AllBgpPeerGroups model uses the generated peerGroupVrfModel and adds
# a list of empty peer-group names. The emptyPeerGroups is not a per VRF element
# and thus cannot be in BgpCliModels.BgpPeerGroups. This additional output is only
# available in multi-agent mode.
class AllBgpPeerGroups( peerGroupVrfModel ):
   emptyPeerGroups = List( valueType=str,
                           help='List of peer group names with no active members',
                           optional=True )

@ArBgpShowOutput( 'doShowIpBgpPeerGroup' )
def doShowIpBgpPeerGroup( mode, pgAndVrfName=( '', None ) ):
   '''
   This show command prints information pertaining to peer-groups 
   # Peer-groups can have both static and dynamic (listen-range) members
   and they can span multiple VRFs
   '''
   vrfName = pgAndVrfName[ 1 ]
   # do a quick sanity check to see if BGP is active
   out = cliRibdShowCommand()( mode, 'show bgp statistics',
                               clientName="BGP", output=False,
                               vrf=vrfName )
   if 'BGP inactive' in out:
      print( out )
      return peerGroupVrfModel()
   pgName = pgAndVrfName[ 0 ]
   peerGroups = compilePeerGroupInfo( mode, pgName, vrfName )
   indent = ' '

   def numStaticPeers( peerGroup ):
      num = 0
      for vrf in peerGroup[ 'static' ]:
         num += len( peerGroup[ 'static' ][ vrf ][ 'v4' ] )
         num += len( peerGroup[ 'static' ][ vrf ][ 'v6' ] )
      return num

   def numListenRange( peerGroup ):
      num = 0
      for vrf in peerGroup[ 'listen' ]:
         num += len( peerGroup[ 'listen' ][ vrf ][ 'networkAndAs' ] )
      return num

   def numDynamicPeers( peerGroup ):
      num = 0
      for vrf in peerGroup[ 'dynamic' ]:
         num += len( peerGroup[ 'dynamic' ][ vrf ] )
      return num

   # pylint: disable=consider-using-in
   # pylint: disable=consider-using-dict-items
   def hasInterfacePeers( v6PeerInfo, peerGroup ):
      for vrf in v6PeersInfo:
         for peer in v6PeersInfo[ vrf ]:
            if isInterfacePeer( vrf, peer, peerGroup ):
               return True
      return False
   # pylint: enable=consider-using-dict-items
   # pylint: enable=consider-using-in

   def isInterfacePeer( vrf, peer, peerGroup ):
      bc = configForVrf( vrf )
      peerKey = PeerConfigKey( peer )
      return peerKey.type == 'peerIpv6Ll' and \
             peerKey not in bc.neighborConfig and \
             peerKey.llIntf in bc.neighIntfConfig and \
             bc.neighIntfConfig[ peerKey.llIntf ].peerGroup.group == peerGroup

   def displayPeerDetails( peerStr, peerInfo ):
      print( '%s%s, state: %s' % ( indent * 6, peerStr, peerInfo[ 'state' ] ) )
      print( '%sNegotiated MP Capabilities:' % ( indent * 8 ) )
      print( '%sIPv4 Unicast: %s' %
             ( indent * 12, 'Yes' if peerInfo[ 'mpv4u' ] else 'No' ) )
      print( '%sIPv6 Unicast: %s' %
             ( indent * 12, 'Yes' if peerInfo[ 'mpv6u' ] else 'No' ) )
      print( '%sIPv4 SR-TE: %s' %
             ( indent * 12, 'Yes' if peerInfo[ 'mpv4srte' ] else 'No' ) )
      print( '%sIPv6 SR-TE: %s' %
             ( indent * 12, 'Yes' if peerInfo[ 'mpv6srte' ] else 'No' ) )

   # Iterate through our peer Group list
   # pylint: disable=too-many-nested-blocks
   for key in sorted( peerGroups.keys() ):
      peerGroup = peerGroups[ key ]
      asAndTypeStr = ""
      if peerGroup[ 'as' ]:
         asType = 'internal' if peerGroup[ 'as' ] == bgpConfig.asNumber \
               else 'external'
         asAndTypeStr = ', remote AS %d, peer-group %s' % \
               ( peerGroup[ 'as' ], asType )
      print( 'BGP peer-group is %s%s' % ( key, asAndTypeStr ) )
      print( '%sBGP version 4' % ( indent * 2 ) )

      # Holds the information about all the v6 peers. This is used to display
      # information about interface peers (i.e) peers configured using
      # "neighbor interface" command.
      # Logic to fetch the info about interface peers:
      # interface peers = all v6 peers - static v6 peers - dynamic v6 peers
      v6PeersInfo = {}
      for vrf in peerGroup[ 'static' ]:
         if peerGroup[ 'static' ][ vrf ][ 'v6' ] or peerGroup[ 'intf' ][ vrf ]:
            v6PeersInfo[ vrf ] = doParseNeighborIntoDict( mode, 'v6', vrf )

      # static peer-group members info
      if numStaticPeers( peerGroup ):
         print( '%sStatic peer-group members:' % ( indent * 2 ) )
      for vrf in sorted( peerGroup[ 'static' ].keys() ):
         if ( peerGroup[ 'static' ][ vrf ][ 'v4' ] or \
               peerGroup[ 'static' ][ vrf ][ 'v6' ] ):
            print( '%sVRF %s:' % ( indent * 4, vrf ) )
         else:
            continue
         for family in [ 'v4', 'v6' ]:
            if family == 'v6':
               if not peerGroup[ 'static' ][ vrf ][ family ]:
                  continue
               nbrsInfo = v6PeersInfo[ vrf ]
            else:
               nbrsInfo = doParseNeighborIntoDict( mode, family, vrf )
            for peer in peerGroup[ 'static' ][ vrf ][ family ]:
               peerStr = peer.stringValue
               ni = None
               peerAddr = None
               if peerStr in nbrsInfo:
                  ni = nbrsInfo[ peerStr ]
                  peerAddr = peerStr
               elif peer.gatedRepr() in nbrsInfo:
                  # Applicable only to link-local peers:
                  # if the interface is not known to gated the response
                  # from gated will have the interface name in kernal format
                  # as opposed to eos format. (i.e) fe80::1%et1 as opposed to
                  # fe80::1%Et1
                  ni = nbrsInfo[ peer.gatedRepr() ]
                  peerAddr = peer.gatedRepr()
               if ni:
                  displayPeerDetails( peerStr, ni )
               if vrf in v6PeersInfo and peerAddr in v6PeersInfo[ vrf ]:
                  # Remove static v6 peers. We will now have the dynamic peers
                  # and the interface peers for this vrf
                  del v6PeersInfo[ vrf ][ peerAddr ]
      
      # listen range subnets info
      if numListenRange( peerGroup ):
         print( '%sListen-range subnets:' % ( indent * 2 ) )
      for vrf in sorted( peerGroup[ 'listen' ].keys() ):
         if peerGroup[ 'listen' ][ vrf ][ 'networkAndAs' ]:
            print( '%sVRF %s:' % ( indent * 4, vrf ) )
         else:
            continue
         for lr in peerGroup[ 'listen' ][ vrf ][ 'networkAndAs' ]:
            peerSpec = lr[ 1 ]
            if peerSpec.hasAsn:
               spec = 'remote AS %s' % peerSpec.asn
            else:
               spec = 'peer filter %s' % peerSpec.peerFilter
            print( '%s%s, %s' % ( indent * 6, lr[ 0 ], spec ) )

      # dynamic peer-group members info
      if numDynamicPeers( peerGroup ):
         print( '%sDynamic peer-group members:' % ( indent * 2 ) )
      for vrf in sorted( peerGroup[ 'dynamic' ].keys() ):
         if peerGroup[ 'dynamic' ][ vrf ]:
            print( '%sVRF %s:' % ( indent * 4, vrf ) )
         else:
            continue
         for peer in peerGroup[ 'dynamic' ][ vrf ][ 'peer-group-members' ]:
            if vrf in v6PeersInfo and peer[ 'addr' ] in v6PeersInfo[ vrf ]:
               # Remove dynamic v6 peers. We will now have only the
               # interface peers for this vrf
               del v6PeersInfo[ vrf ][ peer[ 'addr' ] ]
            print( '%s%s, state: %s' % ( indent * 6, peer[ 'addr' ],
                                         'Established' ) )
            print( '%sNegotiated MP Capabilities:' % ( indent * 8 ) )
            print( '%sIPv4 Unicast: %s' %
                   ( indent * 12, peer[ 'capabilities' ][ 0 ][ 'ncaps-mpv4u' ] ) )
            print( '%sIPv6 Unicast: %s' %
                   ( indent * 12, peer[ 'capabilities' ][ 1 ][ 'ncaps-mpv6u' ] ) )
            print( '%sIPv4 SR-TE: %s' %
                   ( indent * 12, peer[ 'capabilities' ][ 2 ][ 'ncaps-mpv4srte' ] ) )
            print( '%sIPv6 SR-TE: %s' %
                   ( indent * 12, peer[ 'capabilities' ][ 3 ][ 'ncaps-mpv6srte' ] ) )

      if hasInterfacePeers( v6PeersInfo, key ):
         print( '%sInterface peer-group members:' % ( indent * 2 ) )
         for vrf in sorted( v6PeersInfo.keys() ):
            if v6PeersInfo[ vrf ]:
               bc = configForVrf( vrf )
               print( '%sVRF %s:' % ( indent * 4, vrf ) )
               for peer in v6PeersInfo[ vrf ]:
                  if isInterfacePeer( vrf, peer, key ):
                     ni = v6PeersInfo[ vrf ][ peer ]
                     displayPeerDetails( peer, ni )
   return peerGroupVrfModel()

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> [ ( ipv4 | ipv6 ) unicast ]
# ( received-routes [ filtered ] | routes |
#   advertised-routes [queued-withdrawals] ) [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------

BGP_V4_ROUTES = 0x01
BGP_V6_ROUTES = 0x02

def showBgpRegexHelperValidateParams( mode, regex, nbrAddr, routeType, ipv6 ):
   if len( regex ) > 256:
      mode.addError( 'Regular expression exceeds the maximum allowed'
                     'length of 256 characters' )

   if ipv6:
      ipv6 = 1
   else:
      ipv6 = 0
   if nbrAddr:
      if routeType == 'routes':
         routeType = BGP_NOT_SPECIFIED
      elif routeType == 'advertised-routes':
         routeType = BGP_ADVERTISED_ROUTES
      elif routeType == 'received-routes filtered':
         routeType = BGP_RECV_FILTERED_ROUTES
      else:
         routeType = BGP_RECEIVED_ROUTES

   return ipv6, routeType

def getRegexMode():
   return aclListConfig.regexMode

def validateAspRegex( mode, regex ):
   regexMode = aclListConfig.regexMode
   posixRegex = regexMode == 'regexModePosix'
   regexModeStr = "string" if posixRegex else "asn"

   helper = Tac.newInstance( "Routing::RouteMap::Helper" )
   if not ( ( posixRegex and helper.validatePosixRegex( regex ) ) or
            ( not posixRegex and helper.validateDfaRegex( regex ) ) ):
      mode.addError( "Invalid regular expression under regex-mode %s" %
                     regexModeStr )
      return False
   return True

def showBgpRegexpHelper( mode, regex, nbrAddr=None,
                         routeType=None, showPeerColumn=False,
                         ipv6=False, vrfName=DEFAULT_VRF ):

   if not validateAspRegex( mode, regex ):
      return False
   ipv6, routeType = showBgpRegexHelperValidateParams( mode, regex, nbrAddr,
                                                       routeType, ipv6 )
   result = _doShowIpBgpRoute( mode, ipv6=ipv6, peer=nbrAddr,
                               showPeerColumn=showPeerColumn,
                               filterType=routeType,
                               aspRegex=regex,
                               vrfName=vrfName )
   return result

def validatePosixRegex( mode, regex ):
   try:
      re.compile( regex )
      return True
   except re.error:
      mode.addError( invalidPosixRegexErrMsg )
      return False

def showBgpCommunityRegexpHelper( mode, regex, nbrAddr=None, routeType=None,
                                 ipv6=False, vrfName=DEFAULT_VRF ):
   if not validatePosixRegex( mode, regex ):
      return False
   ipv6, routeType = showBgpRegexHelperValidateParams( mode, regex, nbrAddr,
                                                       routeType, ipv6 )
   result = _doShowIpBgpRoute( mode, ipv6=ipv6, peer=nbrAddr,
                               filterType=routeType,
                               commRegex=regex,
                               vrfName=vrfName )
   return result

def showBgpCombinedRegexpHelper( mode, aspRegex, commRegex, nbrAddr=None,
                                 routeType=None, ipv6=False,
                                 vrfName=DEFAULT_VRF ):
   # Validate both regex's separately
   if aspRegex is not None and not validateAspRegex( mode, aspRegex ):
      return False
   if commRegex is not None and not validatePosixRegex( mode, commRegex ):
      return False
   ipv6, routeType = showBgpRegexHelperValidateParams( mode, '', nbrAddr,
                                                       routeType, ipv6 )
   if ( aspRegex and len( aspRegex ) > 256 ) or ( commRegex and
                                                  len( commRegex ) > 256 ):
      mode.addError( 'Regular expression exceeds the maximum allowed'
                     'length of 256 characters' )

   result = _doShowIpBgpRoute( mode, ipv6=ipv6, peer=nbrAddr,
                               filterType=routeType,
                               aspRegex=aspRegex,
                               commRegex=commRegex,
                               vrfName=vrfName )
   mode.addError( str( result ) )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsRoutes( mode, peerAddr, routeType,
                                routeFamilyAndType=None, detail=None,
                                routerId=None, vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr, highPerf=True,
                               familyFilterType=familyFilterType, 
                               filterType=routeType, vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsReceivedRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsReceivedRoutes( mode, peerAddr=None,
                                        filtered=None, routeFamilyAndType=None,
                                        detail=None, routerId=None, vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not peerAddr and not detail

   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               highPerf=True, familyFilterType=familyFilterType,
                               filterType=routeType, vrfName=vrfName )

   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsQueuedWithdrawals' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=queuedWithdrawalsVrfModel )
def doShowIpBgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                           routeFamilyAndType=None, vrfName=None ):
   # queued-withdrawals option to advertised-routes only supported in ArBgp
   return doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# Deprecated Command:
# "show ipv6 bgp neighbors <ip> [ ( ipv4 | ipv6 ) unicast ]
# ( received-routes [ filtered ] | routes | advertised-routes ) [ vrf <vrfName> ]"
#
# New Version:
# "show ipv6 bgp peers <ip> [ ( ipv4 | ipv6 ) unicast ]
# ( received-routes [ filtered ] | routes |
#   advertised-routes [queued-withdrawals] ) [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsRoutes( mode, peerAddr, routeType,
                                  routeFamilyAndType=None,
                                  detail=None, routerId=None, vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr, highPerf=True,
                               familyFilterType=familyFilterType, ipv6=True,
                               filterType=routeType, vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsReceivedRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsReceivedRoutes( mode, peerAddr=None,
                                          filtered=None, routeFamilyAndType=None,
                                          detail=None, routerId=None,
                                          vrfName=None ):
   familyFilterType = None
   if routeFamilyAndType:
      if routeFamilyAndType.get( 'addressFamily' ) == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif routeFamilyAndType.get( 'addressFamily' ) == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not peerAddr and not detail
   result = _doShowIpBgpRoute( mode, detail=detail, peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               familyFilterType=familyFilterType, ipv6=True,
                               filterType=routeType, vrfName=vrfName )

   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsQueuedWithdrawals' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=queuedWithdrawalsVrfModel )
def doShowIpv6BgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                             routeFamilyAndType=None, vrfName=None ):
   # queued-withdrawals option to advertised-routes only supported in ArBgp
   return doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { received-routes [ filtered ] | routes | 
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpNbrRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNbrRtsRegex( mode, nbrAddr, routeType, regex, routerId=None,
                            vrfName=None ):
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType, vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpBgpNbrRecvdRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNbrRecvdRtsRegex( mode, regex, nbrAddr=None, filtered=None,
                                 routerId=None, vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not nbrAddr
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType,
                                 showPeerColumn=showPeerColumn,
                                 vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# Deprecated command:
# "show ipv6 bgp neighbors <ip> { received-routes [ filtered ] | routes | 
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp peers <ip> { received-routes [ filtered ] | routes |
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpNbrRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNbrRtsRegex( mode, nbrAddr, routerId, routeType, regex,
                              vrfName=None ):
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType, ipv6=True, 
                                 vrfName=vrfName )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNbrRecvdRtsRegex' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNbrRecvdRtsRegex( mode, regex, nbrAddr=None, routerId=None,
                                   filtered=None, vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES

   showPeerColumn = not nbrAddr
   result = showBgpRegexpHelper( mode, regex, nbrAddr, routeType,
                                 showPeerColumn=showPeerColumn,
                                 ipv6=True, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
# as-path access-list <name> [vrf <vrfName>]"
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpNeighborsAsPathListRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsAsPathListRoutes( mode, addr, routeType, aspListName=None,
                                          detail=None, vrfName=None ):
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail,
                               filterType=routeType,
                               vrfName=vrfName, highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsAsPathListRecvdRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsAsPathListRecvdRoutes( mode, aspListName=None, addr=None,
                                               filtered=None, detail=None,
                                               vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not addr and not detail

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail,
                               filterType=routeType,
                               vrfName=vrfName )
   return result

# -------------------------------------------------------------------------------
# "show ipv6 bgp peers <ip> {routes|advertised-routes|
# received-routes [ filtered ] }
# as-path access-list <name> [vrf <vrfName>]"
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpNeighborsAsPathListRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsAsPathListRoutes( mode, addr, routeType, aspListName=None,
                                            detail=None, vrfName=None ):
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail, ipv6=True,
                               filterType=routeType,
                               vrfName=vrfName, highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsAsPathListRecvdRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsAsPathListRecvdRoutes( mode, aspListName=None, addr=None,
                                                 filtered=None, detail=None,
                                                 vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not addr and not detail

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail, ipv6=True,
                               showPeerColumn=showPeerColumn,
                               filterType=routeType,
                               vrfName=vrfName )
   return result

# -------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
# community < C > [ exact ] [ detail ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpNeighborsCommunityRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsCommunityRoutes( mode, addr, routeType, values,
                                         exact=None, detail=None, routerId=None,
                                         vrfName=None ):
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsCommunityRecvdRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsCommunityRecvdRoutes( mode, values, addr=None,
                                              filtered=None,
                                              exact=None, detail=None, 
                                              routerId=None, vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not addr and not detail

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail,
                               showPeerColumn=showPeerColumn,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# Deprecated command:
# "show ipv6 bgp neighbors <ip> {routes|advertised-routes|
# received-routes [ filtered ] }
# community <C> [ exact ] [ detail ] [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp peers <ip> {routes|advertised-routes|
# received-routes [ filtered ] }
# community <C> [ exact ] [ detail ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpNeighborsCommunityRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsCommunityRoutes( mode, addr, routeType, values,
                                         exact=None, detail=None, routerId=None,
                                         vrfName=None ):
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   elif routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail, ipv6=True,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsCommunityRecvdRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsCommunityRecvdRoutes( mode, values, addr=None,
                                                filtered=None,
                                                exact=None, detail=None,
                                                routerId=None, vrfName=None ):
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not addr and not detail

   result = _doShowIpBgpRoute( mode, peer=addr, detail=detail, ipv6=True,
                               showPeerColumn=showPeerColumn,
                               filterType=routeType, commValues=set( values ),
                               exact=exact, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
#  < prefix|addr > [ longer-prefixes ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpNeighborsPrefixAddr' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsPrefixAddr ( mode, peerAddr, routeType, pfxAddr,
                                     longerPrefixes=None, routerId=None,
                                     vrfName=None ):
   v6 = False
   if isinstance(pfxAddr, Tac.Type("Arnet::Ip6AddrWithMask")):
      v6 = True

   longerPrefixes = bool( longerPrefixes )
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   if routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES
   showPeerColumn = not peerAddr

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               longerPrefixes=longerPrefixes, 
                               filterType=routeType, ipv6=v6, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpBgpNeighborsPrefixAddrRecvdRts' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpNeighborsPrefixAddrRecvdRts( mode, pfxAddr, peerAddr=None,
                                             filtered=None,
                                             longerPrefixes=None, routerId=None,
                                             vrfName=None ):
   v6 = False
   if isinstance(pfxAddr, Tac.Type("Arnet::Ip6AddrWithMask")):
      v6 = True

   longerPrefixes = bool( longerPrefixes )
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not peerAddr

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), peer=peerAddr,
                               showPeerColumn=showPeerColumn,
                               longerPrefixes=longerPrefixes, 
                               filterType=routeType, ipv6=v6, vrfName=vrfName,
                               highPerf=True )
   return result

#-------------------------------------------------------------------------------
# Deprecated commands:
# "show ipv6 bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
#  < prefix|addr > [ longer-prefixes ] [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp neighbors <ip> { routes|advertised-routes|
# received-routes [ filtered ] }
#  < prefix|addr > [ longer-prefixes ] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpNeighborsPrefixAddr' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsPrefixAddr ( mode, peerAddr, routeType, pfxAddr,
                                     longerPrefixes=None, routerId=None,
                                     vrfName=None ):
   
   longerPrefixes = bool( longerPrefixes )
   if routeType == 'routes':
      routeType = BGP_NOT_SPECIFIED
   if routeType == 'advertised-routes':
      routeType = BGP_ADVERTISED_ROUTES

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), 
                               peer=peerAddr,
                               longerPrefixes=longerPrefixes, ipv6=True,
                               filterType=routeType, vrfName=vrfName,
                               highPerf=True )
   return result

@ArBgpShowOutput( 'doShowIpv6BgpNeighborsPrefixAddrRecvdRts' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpNeighborsPrefixAddrRecvdRts( mode, pfxAddr, peerAddr=None,
                              filtered=None, longerPrefixes=None, routerId=None,
                              vrfName=None ):

   longerPrefixes = bool( longerPrefixes )
   if filtered:
      routeType = BGP_RECV_FILTERED_ROUTES
   else:
      routeType = BGP_RECEIVED_ROUTES
   showPeerColumn = not peerAddr

   result = _doShowIpBgpRoute( mode, prefix=str( pfxAddr ), 
                               peer=peerAddr, showPeerColumn=showPeerColumn,
                               longerPrefixes=longerPrefixes, ipv6=True,
                               filterType=routeType, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> [ ( ipv4 unicast | ipv6 unicast | ipv4 multicast ) ]
#       ( routes | advertised-routes | received-routes [filtered] )
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNeighborsNlriAfRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowBgpNeighborsNlriAfRoutes( mode, peerAddr,
                                    routeFamilyAndType=None, aspListName=None,
                                    routeType=None, filtered=None,
                                    communityValuesAndExact=None,
                                    detail=None, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip>
#       ( routes | advertised-routes | received-routes [filtered] )
#       ( <ip> | <prefix> [longer-prefixes] )
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNeighborsNlriAfPrefixAddr' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowBgpNeighborsNlriAfPrefixAddr( mode, peerAddr,
                                        routeType=None, filtered=None,
                                        pfxAddr=None, longerPrefixes=None,
                                        communityValuesAndExact=None,
                                        detail=None, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show ip bgp neighbors [<peerip>] update errors [vrf <vrfName>]"
#-------------------------------------------------------------------------------

class ShowIpBgpNeighborsUpdateErrorsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip bgp neighbors [ ADDRESS ] update errors [ VRF ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'bgp': ShowTokens.bgpAfterShow,
      'neighbors': ShowTokens.neighbors,
      'ADDRESS': IpAddrMatcher.IpAddrMatcher( 'Neighbor address' ),
      'update': 'BGP update information',
      'errors': ShowTokens.errors,
      'VRF': allVrfExprFactory
   }

   cliModel = "RoutingBgpShowCliHandler.updateErrorsVrfModel"
   handler = "RoutingBgpShowCliHandler.handlerShowIpBgpNeighborsUpdateErrorsCmd"

BasicCli.addShowCommandClass( ShowIpBgpNeighborsUpdateErrorsCmd )

#-------------------------------------------------------------------------------
# Deprecated command:
# "show ipv6 bgp neighbors [<peerip>] update errors [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp peers [<peerip>] update errors [vrf <vrfName>]"
#-------------------------------------------------------------------------------

# New Version
class ShowIp6BgpNeighborsUpdateErrorsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 bgp peers [ ADDRESS ] update errors [ VRF ]'
   data = {
      'ipv6': CliToken.Ipv6.ipv6MatcherForShow,
      'bgp': ShowTokens.bgpAfterShow,
      'peers': ShowTokens.peers,
      'ADDRESS': Ip6AddrMatcher.Ip6AddrMatcher( 'Neighbor address' ),
      'update': 'BGP update information',
      'errors': ShowTokens.errors,
      'VRF': allVrfExprFactory
   }

   cliModel = "RoutingBgpShowCliHandler.updateErrorsVrfModel"
   handler = "RoutingBgpShowCliHandler.handlerShowIp6BgpNeighborsUpdateErrorsCmd"

BasicCli.addShowCommandClass( ShowIp6BgpNeighborsUpdateErrorsCmd )

#-------------------------------------------------------------------------------
# "show ip bgp summary [<prefix>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
desc = "Per VRF BGP summary"
summaryVrfModel = generateVrfCliModel( BgpCliModels.BgpSummary, desc )

def _doShowIpBgpSummary( mode, afi, safi, prefix, vrfName ):
   cmd = 'MIO_DGET_BGP_SUMMARY'
   args = {}
   if afi is not None:
      args[ 'afi' ] = int( afi )
   if safi is not None:
      args[ 'safi' ] = safi
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if prefix:
      return doShowBgpOutputNotSupported( mode )

   try:
      return showRibCapiCommand()( mode, BgpCliModels.BgpSummary, cmd, args,
                                   highPerf=True, clientName='BGP' )
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

@ArBgpShowOutput( 'doShowIpBgpSummary' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=summaryVrfModel )
def doShowIpBgpSummary( mode, prefix=None, vrfName=None ):
   return _doShowIpBgpSummary( mode, AFI_IP, PA4MP_SAFI_UNI, prefix, vrfName )

#-------------------------------------------------------------------------------
# "show ipv6 bgp summary [<prefix>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpSummary' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=summaryVrfModel )
def doShowIpv6BgpSummary( mode, prefix=None, vrfName=None ):
   return _doShowIpBgpSummary( mode, AFI_IP6, PA4MP_SAFI_UNI, prefix, vrfName )

#-----------------------------------------------------------------------------------
# "show bgp
#     ( ( ipv4 unicast ) | ( ipv6 unicast ) | ( ipv4 multicast ) |
#       ( ipv6 multicast ) | ( ipv4 labeled-unicast ) | ( ipv6 labeled-unicast ) )
#     summary [vrf <vrfName>]
#-----------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpNlriAfSummary' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=summaryVrfModel )
def doShowBgpNlriAfSummary( mode, addressFamilyAndType, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#----------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] summary [vrf <vrfName>]
#----------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeSummary' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=summaryVrfModel )
def doShowBgpSrTeSummary( mode, addressFamily=None, vrfName=DEFAULT_VRF ):
   afi = 0
   if addressFamily == 'ipv4':
      afi = AFI_IP
   elif addressFamily == 'ipv6':
      afi = AFI_IP6
   if vrfName != DEFAULT_VRF:
      return doShowBgpOutputNotSupported( mode )
   return _doShowIpBgpSummary( mode, afi, PA4MP_SAFI_SR_TE,
                               prefix=None, vrfName=vrfName )

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] [ detail ] [vrf <vrfName>]"
# "show bgp sr-te endpoint <> color <> distinguisher <> [detail] [vrf <vrfName>]"
#---------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeRoutes( mode, addressFamily=None, endpoint=None, color=None,
                         distinguisher=None, detail=None, vrfName=DEFAULT_VRF ):
   flag = BGP_ROUTE_DETAIL if detail is not None else 0
   highPerf = True
   routeType = BGP_NOT_SPECIFIED
   familyFilterType = None
   prefix = None
   if endpoint is not None and color is not None and distinguisher is not None:
      prefix = str( distinguisher ) + '|' + str( color ) + '|' + str( endpoint )
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, prefix=prefix, detail=detail,
                               highPerf=highPerf, flag=flag, srTe=True,
                               filterType=routeType, vrfName=vrfName,
                               familyFilterType=familyFilterType )
   return result

#---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community [<com>] [exact] [detail] [vrf <name>]"
#---------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeRoutesCommunities' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeRoutesCommunities( mode, addressFamily=None, values=None,
                                    exact=None, detail=None,
                                    vrfName=DEFAULT_VRF ):
   familyFilterType = None
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, detail=detail, commValues=set( values ),
                               familyFilterType=familyFilterType,
                               exact=exact, srTe=True, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community-list [<community-list name>]
#       [exact] [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeRoutesCommunityList' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeRoutesCommunityList( mode, addressFamily=None, commListName=None,
                                      exact=None, detail=None, vrfName=None ):
   familyFilterType = None
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, detail=detail, commListName=commListName,
                               familyFilterType=familyFilterType,
                               exact=exact, srTe=True, vrfName=vrfName )
   return result

#-----------------------------------------------------------------------------
# "show bgp neighbors <neigh_addr> [ ipv4 | ipv6 ] sr-te
# (policies | received-policies | advertised-policies) [ detail ] [vrf <vrfName>]"
#-----------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpSrTeNeighborRoutes' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowBgpSrTeNeighborRoutes( mode, addressFamily=None, policyFilter=None,
                                 detail=None, peerAddr=None, vrfName=DEFAULT_VRF ):
   flag = BGP_ROUTE_DETAIL if detail is not None else 0
   routeType = BGP_NOT_SPECIFIED
   if policyFilter == 'received-policies':
      routeType = BGP_RECEIVED_ROUTES
   elif policyFilter == 'advertised-policies':
      routeType = BGP_ADVERTISED_ROUTES
   familyFilterType = None
   if addressFamily is not None:
      if addressFamily == 'ipv4':
         familyFilterType = BGP_V4_ROUTES
      elif addressFamily == 'ipv6':
         familyFilterType = BGP_V6_ROUTES
   result = _doShowIpBgpRoute( mode, peer=peerAddr, detail=detail,
                               highPerf=True, flag=flag, srTe=True,
                               filterType=routeType, vrfName=vrfName,
                               familyFilterType=familyFilterType )
   return result

#--------------------------------------------------------------------
# The "show bgp [ipv4|ipv6] access-list
#     [<aclName>]
#     [summary] [dynamic] [detail]" command
#--------------------------------------------------------------------
def showBgpAcl( mode, addressFamily=None, name=None ):
   aclFilter = None
   if addressFamily == 'ipv4':
      aclFilter = 'ip'
   elif addressFamily == 'ipv6':
      aclFilter = 'ipv6'

   return AclCli.showServiceAcl( mode, aclCpConfig, aclStatus, aclCheckpoint,
                                 aclFilter, name, serviceName='bgpsacl' )

#-------------------------------------------------------------------------------
# "show ip bgp regexp <regex> [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpRegexp' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpBgpRegexp( mode, regex, vrfName=None ):
   result = showBgpRegexpHelper( mode, regex, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show ipv6 bgp regexp <regex> [vrf <vrfName>]
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpv6BgpRegexp' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=routeSummaryVrfModel )
def doShowIpv6BgpRegexp( mode, regex, vrfName ):
   result = showBgpRegexpHelper( mode, regex, ipv6=True, vrfName=vrfName )
   return result

#-------------------------------------------------------------------------------
# "show bgp statistics [vrf <vrfName>]"
#-------------------------------------------------------------------------------
desc = "Per VRF BGP Statistics"
statisticsVrfModel = generateVrfCliModel( BgpCliModels.BgpStatistics, desc,
                                          baseModel=BgpCliModels.BgpStatisticsBase )

@ArBgpShowOutput( 'doShowBgpStatistics' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=statisticsVrfModel )
def doShowBgpStatistics( mode, vrfName=None ):
   args = {}
   cmd = "MIO_DGET_BGP_STATS"
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   try:
      model = showRibCapiCommand()( mode, BgpCliModels.BgpStatistics,
                                    cmd, args, clientName='BGP',
                                    highPerf=True )
      model._vrf = vrfName
      return model
   except ( EmptyResponseException(), AmiResponseException() ):
      return None

#-------------------------------------------------------------------------------
# "show bgp convergence [vrf <vrfName>]"
#-------------------------------------------------------------------------------
showBgpConvergenceVrfModel = generateVrfCliModel( 
   BgpCliModels.ShowBgpConvergenceVrfModel,
   "Per VRF BGP convergence information." )
@ArBgpShowOutput( 'doShowBgpConvergence' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames, cliModel=showBgpConvergenceVrfModel )
def doShowBgpConvergence( mode, vrfName=DEFAULT_VRF ):
   args = {}
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   cmd = "MIO_DGET_BGP_CONVERGENCE"

   if mode.session_.outputFormat_ == "text":
      args[ 'fastpath' ] = 1
      return showRibDamiCommand()( sys.stdout.fileno(), cmd, args, clientName='BGP' )
   else:
      try:
         args[ 'fastpath' ] = 0
         return showRibCapiCommand()( mode, BgpCliModels.ShowBgpConvergenceVrfModel,
                                      cmd, args, clientName='BGP',
                                      skipNullResponses=False )
      except ( EmptyResponseException(), AmiResponseException() ):
         return None

def isIpAddr( addr ):
   try:
      Arnet.IpGenAddr( addr )
   except ( ValueError, IndexError ):
      return False
   return True

bgpAfterClearIpMatcher = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc='Reset BGP peering sessions' )
clearPeerAnyMatcher = CliMatcher.KeywordMatcher( '*',
      helpdesc=( 'Clear all IPv4 and IPv6 peering sessions '
                 '(Use neighbor * to clear only IPv4 or IPv6 sessions)' ),
      value=lambda mode, match: None )
clearReasonMatcher = CliMatcher.KeywordMatcher( 'reason',
      helpdesc='Hard reset reason to be sent to the peers' )
resetMsgMatcher = CliMatcher.StringMatcher( helpname='MESSAGE',
      helpdesc='Shut down communication reset message' )

class PeerClearCliExpression( CliCommand.CliExpression ):
   expression = ( '* | ( ( IP_PEER | IP6_PEER | IP6_LL_PEER ) '
                  '[ router-id ROUTERID ] ) | ( peer-group GROUP )' )
   data = { 'GROUP': peergroupNameMatcher,
            'IP_PEER': ipv4PeerMatcher,
            'IP6_PEER': ipv6PeerMatcher,
            'IP6_LL_PEER': ipv6LlPeerMatcher,
            'router-id': ShowTokens.routerId,
            'ROUTERID': IpAddrMatcher.IpAddrMatcher( "Neighbor router ID" ),
            'peer-group': 'BGP peer group information',
            '*': clearPeerAnyMatcher
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      try:
         addr = args[ next( peer for peer in
                     ( 'IP_PEER', 'IP6_PEER', 'IP6_LL_PEER', 'GROUP' )
                     if peer in args ) ]
      except StopIteration:
         addr = None
      args[ 'PEER' ] = PeerConfigKey( addr ) if addr is not None else addr

cliShowBgpInstanceVpnSummaryVrfModel = generateVrfCliModel(
   BgpCliModels.ShowBgpInstanceVpnSummaryModel,
   "per VRF bgp instance information models." )

@ArBgpShowOutput( 'doShowBgpInstanceVpnSummary', arBgpModeOnly=True )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames,
               cliModel=cliShowBgpInstanceVpnSummaryVrfModel )
def doShowBgpInstanceVpnSummary( mode, vrfName=None ):
   pass

#-----------------------------------------------------------------------------------
# "clear ip bgp PEER (([vrf <vrfName>] [reason <message>]) |
#                    ([soft] [in|out] [errors] [vrf <vrfName>]))"
#  in privileged EXEC mode
#-----------------------------------------------------------------------------------

class ClearIpBgpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'clear [ ip | ipv6 ] bgp '
              '[ PEER ] '
              '[ ( [ VRF ] reason MESSAGE ) '
              '| ( [ soft ] [ DIRECTION ] [ STATISTICS ] [ VRF ] ) ]' )
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ip': CliToken.Ip.ipMatcherForClear,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'bgp': bgpAfterClearIpMatcher,
      'PEER': PeerClearCliExpression,
      'VRF': allVrfExprFactory,
      'reason': clearReasonMatcher,
      'MESSAGE': resetMsgMatcher,
      'soft': 'Do not reset the session',
      'DIRECTION': CliMatcher.EnumMatcher( {
         'in': 'Send a route refresh message',
         'out': 'Send all routes to the peer',
      } ),
      'STATISTICS': CliMatcher.EnumMatcher( {
         'errors': 'Reset the error statistics',
         'counters': 'Reset the peer counters',
      } ),
   }

   handler = "RoutingBgpShowCliHandler.handlerClearIpBgpCmd"

BasicCli.EnableMode.addCommandClass( ClearIpBgpCmd )

#-------------------------------------------------------------------------------
# "clear { ip | ipv6 } bgp neighbor [ * ] [ vrf <vrfName> ] reason [ MESSAGE ]"
#    in privileged EXEC mode      
#-------------------------------------------------------------------------------
class ClearIpBgpTransportCmd( CliCommand.CliCommandClass ):
   syntax = ( 'clear [ ip | ipv6 ] bgp neighbor '
              '[ * ] [ VRF ] [ reason MESSAGE ]' )
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ip': CliToken.Ip.ipMatcherForClear,
      'ipv6': CliToken.Ipv6.ipv6MatcherForClear,
      'bgp': bgpAfterClearIpMatcher,
      'neighbor': 'Clear based on transport address family',
      '*': ( 'Clear specified transport Address Family peering '
             'sessions (IPv4 or IPv6 accordingly)' ),
      'VRF': allVrfExprFactory,
      'reason': clearReasonMatcher,
      'MESSAGE': resetMsgMatcher
   }

   handler = "RoutingBgpShowCliHandler.handlerClearIpBgpTransportCmd"

BasicCli.EnableMode.addCommandClass( ClearIpBgpTransportCmd )


#-------------------------------------------------------------------------------
# "show bgp traffic-policy field-set mappings [vrf <vrfName>]"
#-------------------------------------------------------------------------------
trafficPolicyFieldSetMappingsModel = generateVrfCliModel(
   BgpCliModels.ShowTrafficPolicyFieldSetCommunityMappingsModel,
   "Per VRF BGP traffic-policy field-set mappings",
   uncheckedModel=True )

@ArBgpShowOutput( 'doShowBgpTrafficPolicyFieldSetMappings' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames,
                cliModel=trafficPolicyFieldSetMappingsModel )
def doShowBgpTrafficPolicyFieldSetMappings( mode, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show bgp vrf selection policy field-set mappings
#-------------------------------------------------------------------------------
vrfSelectionPolicyFieldSetMappingsModel = generateVrfCliModel(
   BgpCliModels.ShowVrfSelectionPolicyFieldSetCommunityMappingsModel,
   "BGP VRF selection policy field-set mappings",
   uncheckedModel=True )

@ArBgpShowOutput( 'doShowBgpVrfSelectionPolicyFieldSetMappings' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames,
                cliModel=vrfSelectionPolicyFieldSetMappingsModel )
def doShowBgpVrfSelectionPolicyFieldSetMappings( mode, vrfName=None ):
   doShowBgpOutputNotSupported( mode )

#-------------------------------------------------------------------------------
# "show ip(v6) bgp field-set <name> [ detail ] [ vrf <vrfName> ]"
# "show bgp ( ipv4 | ipv6 ) unicast field-set <name> [ detail ] [ vrf <vrfName> ]"
#-------------------------------------------------------------------------------

@ArBgpShowOutput( 'doShowIpBgpFieldSet' )
@VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames,
                cliModel=routeSummaryVrfModel )
def doShowIpBgpFieldSet( mode, **kwargs ):
   doShowBgpOutputNotSupported( mode )

# This class is used by the "show configuration consistency" command while
# gathering undefined references to route maps in the BGP cluster of config commands.
routeMapTypes = [ "Route map" ]

class RouteMapNameReferenceGatherer:
   """
   This class gather references to route maps.
   """
   @staticmethod
   def gather( feature ):
      references = set()
      gatherer = RouteMapNameReferenceGatherer
      references = gatherer.gatherReferencesInBgpConfigs( references )
      return references

   @staticmethod
   def gatherReferencesInBgpConfigs( references ):
      gatherer = RouteMapNameReferenceGatherer
      # Some route map name attributes are direct attributes of the Config class,
      # while others are nested in other classed, e.g. AggregateList. The functions
      # provided in this dict dictate how these route map attributes are gathered
      # per group of attributes.
      bgpConfigAttrGroups = {
         "routeInstallMap": gatherer.gatherDirectlyFromAttr,
         "aggregateList": gatherer.gatherFromAggregateList,
         "networkList": gatherer.gatherFromNetworkList,
         "redistributeConfig": gatherer.gatherFromRedistConfig,
         "defaultExport": gatherer.gatherFromDefaultExportCfg,
         "routeMapImport": gatherer.gatherFromNameColl,
         "routeMapExport": gatherer.gatherFromNameColl,
      }
      peerConfigAttrGroups = {
         "routeMapIn": gatherer.gatherDirectlyFromAttr,
         "routeMapOut": gatherer.gatherDirectlyFromAttr,
         "defaultOriginateRouteMap": gatherer.gatherDirectlyFromAttr,
         "grHelperRouteMap": gatherer.gatherDirectlyFromAttr,
      }

      configs = [ bgpConfig ]
      configs += [ bgpVrfConfigDir.vrfConfig[ vrf ]
                   for vrf in bgpVrfConfigDir.vrfConfig ]
      for config in configs:
         gatherer.gatherReferencesInConfig( config, bgpConfigAttrGroups,
                                            bgpConfigAttrsAfMap, references )
         for peer in config.neighborConfig:
            peerConfig = config.neighborConfig[ peer ]
            gatherer.gatherReferencesInConfig( peerConfig, peerConfigAttrGroups,
                                               peerConfigAttrsAfMap, references )

         # Attributes that are not found in bgpConfigAttrsAfMap are added here
         gatherer.gatherDirectlyFromAttr( config, "rpkiOvRouteMap", references )
         gatherer.gatherDirectlyFromAttr( config, "rpkiRedistOvRouteMap",
                                          references )
         grHelperRouteMap = config.grHelperRouteMap
         if grHelperRouteMap.isSet and grHelperRouteMap.value:
            references.add( grHelperRouteMap.value )
         gatherer.gatherFromNameColl( config, "nhResolutionRibProfileRouteMapConfig",
                                      references )

      return references

   @staticmethod
   def gatherReferencesInConfig( config, attrGroups, configAttrsAfMap, references ):
      for attrGroup, gatherRefs in six.iteritems( attrGroups ):
         attrNames = configAttrsAfMap[ attrGroup ]
         for attrName in six.itervalues( attrNames ):
            gatherRefs( config, attrName, references )

   @staticmethod
   def gatherDirectlyFromAttr( config, attrName, references ):
      # The attribute itself is the route map name
      mapName = getattr( config, attrName )
      if mapName:
         references.add( mapName )

   @staticmethod
   def gatherFromNameColl( config, attrName, references ):
      # The attribute containing the route map names is a collection of strings
      mapNameColl = getattr( config, attrName )
      for key in mapNameColl:
         mapName = mapNameColl[ key ]
         if mapName:
            references.add( mapName )

   @staticmethod
   def gatherFromDefaultExportCfg( config, attrName, references ):
      # The attribute containing the route map names is a collection of
      # Routing::Bgp::DefaultExportCfg objects
      defaultExportCfg = getattr( config, attrName )
      for vpnAfiSafi in defaultExportCfg:
         defaultExport = defaultExportCfg[ vpnAfiSafi ]
         routeMap = defaultExport.routeMap
         if routeMap:
            references.add( routeMap )

   @staticmethod
   def gatherFromAggregateList( config, attrName, references ):
      # The attribute containing the route map names is a collection of
      # Routing::Bgp::AggregateAddress objects
      aggregateListAttr = getattr( config, attrName )
      for address in aggregateListAttr:
         aggregateAddress = aggregateListAttr[ address ]
         matchMap = aggregateAddress.matchMap
         if matchMap:
            references.add( matchMap )
         attrMap = aggregateAddress.attrMap
         if attrMap:
            references.add( attrMap )

   @staticmethod
   def gatherFromNetworkList( config, attrName, references ):
      # The attribute containing the route map names is a collection of
      # Routing::Bgp::Network objects
      networkListAttr = getattr( config, attrName )
      for networkPrefix in networkListAttr:
         network = networkListAttr[ networkPrefix ]
         routeMap = network.routeMap
         if routeMap:
            references.add( routeMap )

   @staticmethod
   def gatherFromRedistConfig( config, attrName, references ):
      # The attribute containing the route map names is a collection of
      # Routing::Bgp::Redistribute objects
      redistConfig = getattr( config, attrName )
      for proto in redistConfig:
         redistribute = redistConfig[ proto ]
         routeMap = redistribute.routeMap
         if routeMap:
            references.add( routeMap )

UndefinedReferenceChecker.addReferenceGatherer( routeMapTypes,
                                                RouteMapNameReferenceGatherer )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bgpConfig
   global aclListConfig
   global bgpClearReq, bgpClearResp
   global bgpVrfClearRespDir, bgpVrfClearReqDir
   global bgpVrfConfigDir
   global bgpLuLfib
   global l3Config, l3ProtocolAgentModelStatus, defaultVrfProtocolStatus
   global aclCpConfig
   global aclCheckpoint
   global aclStatus
   global bgpTestControlReq

   gv.sysname = entityManager.sysname()
   gv.skipAgentCheck = 'BGP_SKIP_AGENT_LIVENESS_CHECK_IN_TEST' in os.environ
   bgpConfig = LazyMount.mount( entityManager, 'routing/bgp/config',
                                'Routing::Bgp::Config', 'w' )
   bgpClearReq = LazyMount.mount( entityManager, 'routing/bgp/clear/request',
                                  'Routing::Bgp::ClearRequestDir', 'w' )
   bgpClearResp = LazyMount.mount( entityManager, 'routing/bgp/clear/response',
                                   'Routing::Bgp::ClearResponseDir', 'r' )
   aclListConfig = LazyMount.mount( entityManager,
                                    'routing/acl/config',
                                    'Acl::AclListConfig', 'r' )
   bgpVrfClearReqDir = LazyMount.mount( entityManager,
                                        'routing/bgp/clear/vrf/request',
                                        'Routing::Bgp::VrfClearRequestDir', 'w' )
   bgpVrfClearRespDir = LazyMount.mount( entityManager,
                                         'routing/bgp/clear/vrf/response',
                                         'Tac::Dir', 'r' )
   bgpVrfConfigDir = LazyMount.mount( entityManager, 'routing/bgp/vrf/config',
                                      'Routing::Bgp::VrfConfigDir', 'w' )
   bgpLuLfib = SmashLazyMount.mount( entityManager, 'mpls/protoLfibInputDir/bgpLu',
                             'Mpls::LfibStatus',
                             SmashLazyMount.mountInfo( 'reader' ) )
   l3Config = LazyMount.mount( entityManager, 'l3/config',
                               'L3::Config', 'r' )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   l3ProtocolAgentModelStatus = \
         LazyMount.mount( entityManager, 'l3/status/protocolAgentModelStatus',
                          'L3::ProtocolAgentModelStatus', 'r' )
   defaultVrfProtocolStatus = \
         LazyMount.mount( entityManager,
                          Cell.path( "routing/defaultVrfProtocolStatus" ),
                          "Tac::Dir", "ri" )
   bgpTestControlReq = LazyMount.mount( entityManager,
                                        'routing/bgp/testControlRequest',
                                        'Routing::Bgp::TestControlRequestDir', 'w' )
