# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from BasicCli import addShowCommandClass
from CliCommand import (
      CliExpression,
   )
from CliMatcher import (
      IntegerMatcher,
   )
from CliPlugin.AclCli import (
      accessListKwMatcherForServiceAcl,
      ipAclNameExpression,
   )
import CliPlugin.AclCliModel
from CliPlugin.RoutingBgpCli import (
      peergroupNameMatcher,
      V4V6PeerKeyCliExpression,
   )
import CliPlugin.RoutingBgpShowCli
from CliPlugin.RoutingBgpShowCli import (
      allVrfExprFactory,
      peerGroupVrfModel,
      showBgpConvergenceVrfModel,
      statisticsVrfModel,
      trafficPolicyFieldSetMappingsModel,
      vrfSelectionPolicyFieldSetMappingsModel,
   )
from CliToken.Ip import ipMatcherForShow
from CliToken.RoutingBgpShowCliTokens import (
      AddrFamilyAfterShowBgp,
      bgpAfterShow,
      bgpConvergence,
      bgpInstance,
      hiddenKeyword,
      internal,
      internalDumpDuplicates,
      peerGroupAfterShow,
      statistics,
      bgpUpdateGroup,
      trafficPolicy,
      fieldSet,
      mappings,
      vpn,
      summary,
      vrfVsp,
      selectionVsp,
      policyVsp,
   )
from ShowCommand import ShowCliCommandClass
import six

# This file provides the CLI parser handling for BGP show commands that are
# at instance level, i.e., not specific to any one AFI/SAFI
# This file can also include "show [] bgp neighbor" style command
# that describes neighbor state in detail
# "show bgp [afi/safi] summary" style commands which show the neighbor
# status and number of prefixes received/accepted.

brib = hiddenKeyword( 'brib' )
countAfterBrib = hiddenKeyword( 'count' )
bribDetail = hiddenKeyword( 'detail' )
bribStash = hiddenKeyword( 'stash' )
events = hiddenKeyword( 'events' )
eventsTracking = hiddenKeyword( 'tracking' )
eventsStart = hiddenKeyword( 'start' )
eventsDetail = hiddenKeyword( 'detail' )
eventsAdvtThreshold = hiddenKeyword( 'advertise-threshold' )
eventsStop = hiddenKeyword( 'stop' )
eventsSummary = hiddenKeyword( 'summary' )
eventsIar = hiddenKeyword( 'iar' )
eventsIarHistory = hiddenKeyword( 'history' )

#-------------------------------------------------------------------------------
# "show bgp [ipv4|ipv6] access-list [<aclName>]
#     [summary] [dynamic] [detail]"
#-------------------------------------------------------------------------------
class ShowBgpAccessList( ShowCliCommandClass ):
   syntax = """show bgp [ AF_AFTER_BGP ] access-list [ ACL_NAME_OPTIONS ]"""
   data = {
         "bgp": bgpAfterShow,
         "AF_AFTER_BGP": AddrFamilyAfterShowBgp,
         "access-list": accessListKwMatcherForServiceAcl,
         "ACL_NAME_OPTIONS": ipAclNameExpression,
      }
   cliModel = CliPlugin.AclCliModel.AllAclList
   handler = "RoutingBgpShowCliInstanceHandler.handlerShowBgpAccessList"

addShowCommandClass( ShowBgpAccessList )

class InternalOrExpression( CliExpression ):
   expression = """(
         ( internal [dump-duplicates] ) |
         ( internal brib count [detail] ) |
         ( internal events iar history ) |
         ( internal brib count stash ) |
         ( internal events tracking start [ advertise-threshold
               RIB_ADD_THRESHOLD RIBOUT_ADV_THRESHOLD ] ) |
         ( internal events tracking stop ) |
         ( internal events tracking summary ) |
         ( internal events tracking detail )
      )"""
   data = {
         "internal": internal,
         "dump-duplicates": internalDumpDuplicates,
         "brib": brib,
         "count": countAfterBrib,
         "detail": bribDetail,
         "stash": bribStash,
         "events": events,
         "tracking": eventsTracking,
         "start": eventsStart,
         "advertise-threshold": eventsAdvtThreshold,
         "RIB_ADD_THRESHOLD": IntegerMatcher( 0, 4294967294,
            helpdesc='threshold value for RIB routes' ),
         "RIBOUT_ADV_THRESHOLD": IntegerMatcher( 0, 4294967294,
            helpdesc='threshold value for RibOut routes' ),
         "stop": eventsStop,
         "summary": eventsSummary,
         "iar": eventsIar,
         "history": eventsIarHistory,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      if "internal" not in args:
         return

      kwRename = {
            "internal": None,
            "dump-duplicates": None,
            "brib": None,
            "count": None,
            "detail": None,
            "stash": None,
            "events": None,
            "tracking": None,
            "start": None,
            "advertise-threshold": None,
            "RIB_ADD_THRESHOLD": "numRibRoutes",
            "RIBOUT_ADV_THRESHOLD": "numRibOutRoutes",
            "stop": None,
            "summary": None,
            "iar": None,
            "history": None,
         }
      result = {}
      for kw, newName in six.iteritems( kwRename ):
         newName = newName or kw
         if kw in args:
            result[ newName ] = args.pop( kw )

      args[ "internal" ] = result

#-------------------------------------------------------------------------------
# "show bgp instance
#     (hidden options) internal [dump-duplicates]
#     (hidden options) internal [brib count [detail]]
#     (hidden options) internal [events iar history]
#     (hidden options) internal [events tracking start [advertise-threshold
#                       <rib-add-threshold> <ribout-advertise-threshold>]]
#     (hidden options) internal [events tracking stop]
#     (hidden options) internal [events tracking summary]
#     (hidden options) internal [events tracking detail]
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpInstanceInternal( ShowCliCommandClass ):
   syntax = "show bgp instance INTERNAL_EXPR [ VRF ]"
   data = {
      "bgp": bgpAfterShow,
      "instance": bgpInstance,
      "INTERNAL_EXPR": InternalOrExpression,
      "VRF": allVrfExprFactory,
   }
   handler = "RoutingBgpShowCliInstanceHandler.handlerShowBgpInstanceInternal"

addShowCommandClass( ShowBgpInstanceInternal )

#-------------------------------------------------------------------------------
# "show bgp instance [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpInstance( ShowCliCommandClass ):
   syntax = "show bgp instance [ VRF ]"
   data = {
         "bgp": bgpAfterShow,
         "instance": bgpInstance,
         "VRF": allVrfExprFactory,
      }
   cliModel = "RoutingBgpShowCliHandler.cliShowBgpInstanceVrfModel"
   handler = "RoutingBgpShowCliInstanceHandler.handlerShowBgpInstance"

addShowCommandClass( ShowBgpInstance )

# -------------------------------------------------------------------------------
# "show bgp instance vpn summary [vrf <vrfName>]"
# -------------------------------------------------------------------------------
class ShowBgpInstanceVpnSummary( ShowCliCommandClass ):
   syntax = "show bgp instance vpn summary [ VRF ]"
   data = {
         "bgp": bgpAfterShow,
         "instance": bgpInstance,
         "vpn": vpn,
         "summary": summary,
         "VRF": allVrfExprFactory,
      }
   cliModel = CliPlugin.RoutingBgpShowCli.cliShowBgpInstanceVpnSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return CliPlugin.RoutingBgpShowCli.doShowBgpInstanceVpnSummary( mode,
            vrfName=args.get( "VRF" ) )

addShowCommandClass( ShowBgpInstanceVpnSummary )

#-------------------------------------------------------------------------------
# "show bgp convergence [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpConvergence( ShowCliCommandClass ):
   syntax = """show bgp convergence [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "convergence": bgpConvergence,
         "VRF": allVrfExprFactory,
      }
   cliModel = showBgpConvergenceVrfModel
   handler = "RoutingBgpShowCliInstanceHandler.handlerShowBgpConvergence"

addShowCommandClass( ShowBgpConvergence )

#-------------------------------------------------------------------------------
# "show [ip] bgp peer-group [peergroupName] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpPeerGroup( ShowCliCommandClass ):
   syntax = """show [ ip ] bgp peer-group [ PG_NAME ] [ VRF ]"""
   # "ip" is accepted for backward compatibility reason
   data = {
         "ip": ipMatcherForShow,
         "bgp": bgpAfterShow,
         "peer-group": peerGroupAfterShow,
         "PG_NAME": peergroupNameMatcher,
         "VRF": allVrfExprFactory,
      }
   cliModel = peerGroupVrfModel
   handler = "RoutingBgpShowCliInstanceHandler.handlerShowBgpPeerGroup"

addShowCommandClass( ShowBgpPeerGroup )

#-------------------------------------------------------------------------------
# "show bgp statistics [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpStatistics( ShowCliCommandClass ):
   syntax = """show bgp statistics [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "statistics": statistics,
         "VRF": allVrfExprFactory,
      }
   cliModel = statisticsVrfModel
   handler = "RoutingBgpShowCliInstanceHandler.handlerShowBgpStatistics"

addShowCommandClass( ShowBgpStatistics )

#-------------------------------------------------------------------------------
# "show bgp update-group
#     (index | A.B.C.D | A:B:C:D:E:F:G:H | LL_ADDR)
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpUpdateGroup( ShowCliCommandClass ):
   syntax = """show bgp update-group
               [ ( INDEX | PEER_ADDR ) ]
               [ VRF ]"""
   data = {
         "bgp": bgpAfterShow,
         "update-group": bgpUpdateGroup,
         "INDEX": IntegerMatcher( 1, 4294967295,
            helpdesc="Index of the update group" ),
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "VRF": allVrfExprFactory,
      }
   cliModel = "RoutingBgpShowCliHandler.updateGroupVrfModel"
   handler = "RoutingBgpShowCliInstanceHandler.handlerShowBgpUpdateGroup"

addShowCommandClass( ShowBgpUpdateGroup )

#-------------------------------------------------------------------------------
# "show bgp traffic-policy field-set mappings [vrf <vrfName>]"
#-------------------------------------------------------------------------------
class ShowBgpTrafficPolicyFieldSetMappings( ShowCliCommandClass ):
   syntax = """show bgp traffic-policy field-set mappings [ VRF ]"""
   data = {
      "bgp": bgpAfterShow,
      "traffic-policy": trafficPolicy,
      "field-set": fieldSet,
      "mappings": mappings,
      "VRF": allVrfExprFactory,
      }
   cliModel = trafficPolicyFieldSetMappingsModel
   handler = "RoutingBgpShowCliInstanceHandler." \
      "handlerShowBgpTrafficPolicyFieldSetMappings"

addShowCommandClass( ShowBgpTrafficPolicyFieldSetMappings )

#-------------------------------------------------------------------------------
# "show bgp vrf selection policy field-set mappings
#-------------------------------------------------------------------------------
class ShowBgpVrfSelectionPolicyFieldSetMappings( ShowCliCommandClass ):
   syntax = """show bgp vrf selection policy field-set mappings"""
   data = {
      "bgp": bgpAfterShow,
      "vrf": vrfVsp,
      "selection": selectionVsp,
      "policy": policyVsp,
      "field-set": fieldSet,
      "mappings": mappings,
      }
   cliModel = vrfSelectionPolicyFieldSetMappingsModel
   handler = "RoutingBgpShowCliInstanceHandler." \
      "handlerShowBgpVrfSelectionPolicyFieldSetMappings"

addShowCommandClass( ShowBgpVrfSelectionPolicyFieldSetMappings )
