# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
import CliParser
import CliToken
from CliPlugin.RouterGeneralCli import RouterGeneralMode
from CliPlugin.RouterGeneralCli import routerGeneralCleanupHook
import LazyMount
import Tracing

traceHandle = Tracing.Handle( 'RoutingHardwareCli' )
t5 = traceHandle.trace5 # Info

routingHwConfig = None
routingHwStatus = None

#-------------------------------------------------------------------------------
# "[no|default] hardware next-hop fast-failover" in "router general" mode
#-------------------------------------------------------------------------------
def hardwareFastFailoverSupportedGuard( mode, token ):
   if routingHwStatus.hwFastFailoverSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def toggleFastFailover( enabled=False ):
   routingHwConfig.hwFastFailoverEnabled = enabled

def enableFastFailover( mode ):
   toggleFastFailover( enabled=True )

def disableFastFailover( mode ):
   toggleFastFailover( enabled=False )

hardwareKwMatcher = CliToken.Hardware.hardwareForConfigMatcher
nextHopKwMatcher = CliMatcher.KeywordMatcher( 'next-hop',
                                              helpdesc="Next-hop configuration" )
fastFailoverKwMatcher = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher(
         'fast-failover',
         helpdesc='hardware accelerated next-hop failover' ),
      guard=hardwareFastFailoverSupportedGuard )

class HardwareFastFailoverCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware next-hop fast-failover'
   noOrDefaultSyntax = 'hardware next-hop fast-failover'
   data = {
      'hardware' : hardwareKwMatcher,
      'next-hop' : nextHopKwMatcher,
      'fast-failover' : fastFailoverKwMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      toggleFastFailover( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      toggleFastFailover( False )

RouterGeneralMode.addCommandClass( HardwareFastFailoverCmd )

t5( 'Registered commands - in RouterGeneralMode.addCommand' )

# cleanup hook for 'no router general'
def setDefaultRoutingHwConfig():
   toggleFastFailover( enabled=False )

def noOrDefaultRouterGeneralMode( mode=None ):
   setDefaultRoutingHwConfig()

routerGeneralCleanupHook.addExtension( noOrDefaultRouterGeneralMode )

def Plugin( entityManager ):
   global routingHwConfig
   global routingHwStatus

   routingHwConfig = LazyMount.mount( entityManager,
                                      "routing/hardware/config",
                                      "Routing::Hardware::Config", "w" )
   routingHwStatus = LazyMount.mount( entityManager,
                                      "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
