# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Arnet
import BasicCli
import CliMatcher
import CliToken
import LazyMount
import ShowCommand
import SharedMem, Smash, SmashLazyMount
import OspfConsts
from IpLibConsts import DEFAULT_VRF
from CliModel import cliPrinted
from CliPlugin.VrfCli import generateVrfCliModel
from CliPlugin import MplsModel
from CliPlugin.OspfSrCliModels import (
   OspfShowSrModel,
   OspfPrefixSegmentFlagsModel,
   OspfPrefixSegmentModel,
   OspfShowSrPrefixSegmentSummaryModel,
   OspfShowSrTunnelModel,
   OspfSrTunnelTableEntry,
   OspfGlobalBlockModel,
   OspfShowSrGlobalBlocksSummaryModel,
   OspfAdjacencySegmentModel,
   OspfAdjacencySegmentFlagsModel,
   OspfShowSrAdjacencySegmentSummaryModel,
)
from CliPlugin.TunnelCli import (
   getTunnelIndexFromId,
   getViaModel,
   readMountTunnelTable,
   TunnelTableIdentifier,
   tunnelIndexMatcher
)
from CliPlugin import IpAddrMatcher
from CliPlugin.TunnelFibCli import getFecFromIntfId
from CliPlugin.TunnelModels import sortKey
from CliPlugin.RoutingOspfCli import (
   ospfConfig,
   matcherOspfShow,
   generateOspfInstListCliModel,
   OspfVrfModelDecorator,
   prefixMatcher,
)
from CliPlugin.SegmentRoutingCliShowHelper import (
   _populateSRGBValues,
   _populateSrCommonHeader,
   _getReachabilityAlgorithm,
   _getSegmentRoutingGlobalBlocksPeersCount,
   _getSelfOriginatedSegmentStatistics,
)
import sys
import Tac
from TypeFuture import TacLazyType
import Toggles.OspfToggleLib
import Toggles.gatedToggleLib
import Tracing

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

mplsConfig = None
mplsStatus = None
srSysdbStatusDir = None
srSmashStatusDir = None
l3Config = None
ospfSrBindingTable = None
ospfStatusDir = None
ospfSrTunnelTable = None

CommonLibConsumerSm = TacLazyType( 'CommonLibSmash::CommonLibConsumerSm' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
FwdEqvClass = TacLazyType( 'Mpls::FwdEqvClass' )
IpGenPrefix = TacLazyType( 'Arnet::IpGenPrefix' )
LabelBindingTable = TacLazyType( 'Mpls::LabelBindingTable' )
LabelBindingTableColl = TacLazyType( 'Mpls::LabelBindingTableColl' )
MplsBindingsHelper = TacLazyType( 'Mpls::Cli::MplsBindingsHelper' )
RouterId = TacLazyType( 'Mpls::RouterId' )
SrAdjAllocationType = TacLazyType(
   "Routing::SegmentRoutingCli::SrAdjAllocationType" )

def getSrSysdbStatus( instanceId=None, vrfName=None ):
   return srSysdbStatusDir.get( 'default' )

def getSrSmashStatus( instanceId=None, vrfName=None ):
   return srSmashStatusDir

def ospfSrActive( mode, instanceId, vrfName ):
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   if not l3Config.protocolAgentModel == "multi-agent":
      mode.addWarning( 'OSPF Segment Routing is not supported in' +
                       ' the current protocol model (%s)' %
                       l3Config.protocolAgentModel )
      return False

   if not instConfig or not instConfig.enable:
      mode.addWarning( 'OSPF (Instance Id: %s) is not running' % instanceId )
      return False

   if not instConfig.srConfig or instConfig.srConfig.shutdown:
      mode.addWarning( 'OSPF (Instance Id: %s) Segment Routing has been ' \
            'administratively shutdown' % instanceId )
      return False

   if not mplsConfig.mplsRouting:
      mode.addWarning( 'MPLS routing is not enabled' )
      return False

   # Sanity check that smash/sysdb status is available. They should be
   # available if the above conditions are met - except a transient 
   # state where gated hasn't processed srConfig yet or if ospf instance
   # isn't up for example no interfaces etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   if not srSysdbStatus or not srSmashStatus or \
      srSysdbStatus.routerId == Arnet.IpAddr( '0.0.0.0' ):
      mode.addWarning( 'OSPF (Instance Id: %s) Segment Routing is not running' % \
                       instanceId )
      return False

   return True

def populateSrCommonModel( model, instConfig, srSysdbStatus ):
   model.instanceId = instConfig.instance
   _populateSrCommonHeader( model, instConfig.srConfig.dataPlane, srSysdbStatus ) 

def getSegmentRoutingsSummaryModel( instanceId=None, vrfName=None ):
   result = OspfShowSrModel()
  
   # SR should be enabled (check done in ospfSrActive), no need to sanity check
   # instConfig, srSysdbStatus etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   routerId = instConfig.routerId
   statusVrf = ospfStatusDir.get( instConfig.vrfName )
   if statusVrf is not None:
      instanceStatus = statusVrf.instanceStatus.get( instanceId )
      if instanceStatus is not None:
         routerId = instanceStatus.routerId

   populateSrCommonModel( result, instConfig, srSysdbStatus )
   _populateSRGBValues( result, srSysdbStatus )
   result.reachabilityAlgorithm = _getReachabilityAlgorithm()
   result.srPeerCount = _getSegmentRoutingGlobalBlocksPeersCount( srSmashStatus )
   result.selfOriginatedSegmentStatistics = _getSelfOriginatedSegmentStatistics(
      srSmashStatus, srSysdbStatus, str( routerId ), instanceId )
   result.mappingServer = \
         result.selfOriginatedSegmentStatistics.proxyNodeSidCount > 0

   return result

def _populatePrefixSegmentFlags( flagsObj, pfxSeg ):
   flagsObj.np = pfxSeg.flags.noPhp
   flagsObj.m = pfxSeg.flags.proxy
   flagsObj.e = pfxSeg.flags.explicitNull
   flagsObj.v = pfxSeg.sid.isValue
   flagsObj.l = pfxSeg.flags.isLocal

def _getProtectionMode( protection ):
   protectionStr = "unprotected" if protection == "disabled" else protection
   return protectionStr
def getSegmentRoutingPrefixSegmentSummaryModel( instanceId=None, vrfName=None,
                                                selfOrig=None ):
   # ISIS does not set/reset selfOriginatedPrefix flag if 'self-originated'
   # subcommand is executed, but we are setting the value regardless of the
   # command variant. Owing to this as well as the facts that prefix segment flags
   # are different for OSPF and ISIS, separate functions are used for populating
   # OSPF and ISIS prefix segment models
   result = OspfShowSrPrefixSegmentSummaryModel()

   # SR should be enabled (check done in ospfSrActive), no need to sanity check
   # instConfig, srSysdbStatus etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   populateSrCommonModel( result, instConfig, srSysdbStatus )
   result.nodeSidCount = 0
   result.proxyNodeSidCount = 0
   result.prefixSidCount = 0

   for pfx, entry in sorted( srSmashStatus.prefixSegment.items() ):
      if selfOrig and str( entry.rtrid ) != str( srSysdbStatus.igpRtrId ):
         continue
      item = OspfPrefixSegmentModel()
      item.selfOriginatedPrefix = str( entry.rtrid ) == str( srSysdbStatus.igpRtrId )
      item.prefix = str( pfx )
      item.segmentId = entry.sid.index
      if Toggles.gatedToggleLib.toggleOspfTilfaEnabled():
         item.protection = _getProtectionMode( entry.protectionMode )
      
      anycastSystemIdCount = len( entry.anycastSystemID )
      if anycastSystemIdCount > 0:
         for sysId in entry.anycastSystemID:
            item.routerIds.append( entry.anycastSystemID[ sysId ].stringValue )
      else:
         item.routerIds.append( entry.rtrid.stringValue )

      item.flags = OspfPrefixSegmentFlagsModel()
      _populatePrefixSegmentFlags( item.flags, entry )

      if entry.flags.proxy:
         item.segmentType = "Proxy-Node"
         result.proxyNodeSidCount += 1
      elif entry.flags.node:
         item.segmentType = "Node"
         result.nodeSidCount += 1
      else:
         item.segmentType = "Prefix"
         result.prefixSidCount += 1
      result.prefixSegments.append( item )

   return result

def getSegmentRoutingGlobalBlocksSummaryModel( instanceId=None, vrfName=None ):
   result = OspfShowSrGlobalBlocksSummaryModel()

   # SR should be enabled (check done in ospfSrActive), no need to sanity check
   # instConfig, srSysdbStatus etc
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srSmashStatus = getSrSmashStatus( instanceId, vrfName )
   instConfig = ospfConfig().instanceConfig.get( instanceId )

   populateSrCommonModel( result, instConfig, srSysdbStatus )
   result.srPeerCount = _getSegmentRoutingGlobalBlocksPeersCount( srSmashStatus )

   def _populateGlobalBlock( item, routerId, base, size ):
      item.routerId = routerId.stringValue
      item.base = base
      item.size = size

   ourGlobalBlock = OspfGlobalBlockModel()
   _populateGlobalBlock( ourGlobalBlock, srSysdbStatus.igpRtrId,
                         srSysdbStatus.labelRange.base,
                         srSysdbStatus.labelRange.size )
   result.globalBlocks.append( ourGlobalBlock )
   
   for routerId, entry in sorted( srSmashStatus.globalBlock.items() ):
      item = OspfGlobalBlockModel()
      _populateGlobalBlock( item, routerId, entry.base, entry.size )
      result.globalBlocks.append( item )

   return result

def getProtectionMode( protection ):
   protectionStr = "unprotected" if protection == "disabled" else protection
   return protectionStr

def populateAdjacencySegments( result, srSysdbStatus ):
   adjacencySegments = sorted( srSysdbStatus.adjacencySegment.items() )
   for _, entry in adjacencySegments:
      for nexthop in entry.nexthop:
         item = OspfAdjacencySegmentModel()
         item.ipAddress = str( nexthop.hop )
         item.localIntf = str( nexthop.intfId )
         item.sid = entry.sid.index
         # populate flags
         item.flags = OspfAdjacencySegmentFlagsModel()
         item.flags.b = entry.flags.backup
         item.flags.v = entry.flags.isValue
         item.flags.l = entry.flags.isLocal
         item.flags.g = entry.flags.isSet
         # We don't have persistence flag in smash (entry.flags)
         # so it will always be 0 in show command which is fine
         # for now as the show command is only dumping the self-originated
         # adjacency segments and those do not support persistence
         item.lan = False
         item.protection = getProtectionMode( entry.protectionMode )
         result.adjacencySegments.append( item )

def getSegmentRoutingAdjacencySegmentModel( instanceId=None, vrfName=None ):
   result = OspfShowSrAdjacencySegmentSummaryModel()
   instConfig = ospfConfig().instanceConfig.get( instanceId )
   srSysdbStatus = getSrSysdbStatus( instanceId, vrfName )
   srAdjSidAllocationModeEnumMapping = dict( [
      ( SrAdjAllocationType.srAdjacencyAllocationNone, "none" ),
      ( SrAdjAllocationType.srAdjacencyAllocationSrOnly, "srOnly" ),
      ( SrAdjAllocationType.srAdjacencyAllocationAll, "all" ) ] )

   populateSrCommonModel( result, instConfig, srSysdbStatus )

   # populate adjacency allocation mode and SID pool
   result.adjSidAllocationMode = srAdjSidAllocationModeEnumMapping[
      instConfig.srConfig.srAdjSegmentAlloc ]
   if instConfig.srConfig.srAdjSegmentAlloc != \
      SrAdjAllocationType.srAdjacencyAllocationNone:
      statusVrf = ospfStatusDir.get( instConfig.vrfName )
      if statusVrf is not None:
         instStatus = statusVrf.instanceStatus.get( instanceId )
         if instStatus is not None:
            result.adjSidPoolBase = instStatus.srAdjBase
            result.adjSidPoolSize = instStatus.srAdjRange

   # populate adjacency segments
   populateAdjacencySegments( result, srSysdbStatus )
   return result

# 'segment-routing' keyword for show commands
segmentRoutingKw = CliMatcher.KeywordMatcher( 'segment-routing',
                        helpdesc='Segment Routing Information' )
tunnelKw = CliMatcher.KeywordMatcher( 'tunnel',
                        helpdesc='Segment Routing tunnel Information' )
OspfShowSrDictModel = generateOspfInstListCliModel( OspfShowSrModel )
OspfShowSrVrfDictModel = generateVrfCliModel( OspfShowSrDictModel,
                                              'OSPF SR information for all VRFs' )


@OspfVrfModelDecorator( vrfModelListType=OspfShowSrVrfDictModel, 
                        instModelListType=OspfShowSrDictModel )
def showIpOspfSr( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   # Since SR is supported only for default vrf, only display that,
   # the show command anyways does not provide a vrf filter
   if not ospfSrActive( mode, instanceId, DEFAULT_VRF ):
      return None
   return getSegmentRoutingsSummaryModel( instanceId )
 

def showIpOspfSrTunnel( mode, args ):
   endpoint = args.pop( 'IPADDR', None )
   tunnelIndex = args.pop( 'TUNNEL-INDEX', None )
   if endpoint:
      endpoint = str( endpoint )
   return showOspfSrTunnelTable( tunnelIndex=tunnelIndex, endpoint=endpoint )

# -------------------------------------------------------------------------------
# show ip ospf segment-routing tunnel [ tunnel-index | tep ]
# -------------------------------------------------------------------------------
def addrMatchTep( endpoint=None, tunnel=None ):
   if not endpoint:
      return True
   if not tunnel:
      return False
   if endpoint == tunnel.tep.stringValue:
      return True
   return False

def getOspfSrTunnelTableEntryModel( tunnelId ):
   vias = []
   tunnelTableEntry = ospfSrTunnelTable.entry.get( tunnelId )

   if tunnelTableEntry:
      for via in tunnelTableEntry.via.values():
         viaModel = None

         labels = []
         for mplsStackIndex in reversed( range( via.labels.stackSize ) ):
            labels.append( str( via.labels.labelStack( mplsStackIndex ) ) )

         if FecIdIntfId.isFecIdIntfId( via.intfId ):
            fec = getFecFromIntfId( via.intfId )
            if fec is None:
               continue

            for fecVia in sorted( fec.via.values() ):
               addr = Tac.Value( "Arnet::IpGenAddr", str( fecVia.hop ) )
               intfId = fecVia.intfId
               viaModel = getViaModel( addr, intfId, labels )
               if viaModel:
                  vias.append( viaModel )
         else:
            viaModel = getViaModel( via.nexthop, via.intfId, labels )
            if viaModel:
               vias.append( viaModel )

      # Make via rendering order deterministic
      vias.sort( key=sortKey )
      return OspfSrTunnelTableEntry(
         endpoint=tunnelTableEntry.tep, vias=vias )
   return None

def showOspfSrTunnelTable( tunnelIndex=None, endpoint=None ):
   entries = {}

   for tunnelId, tunnel in ospfSrTunnelTable.entry.items():
      if addrMatchTep( endpoint=endpoint, tunnel=tunnel ):
         entryModel = getOspfSrTunnelTableEntryModel( tunnelId=tunnelId )
         if entryModel:
            currentIndex = getTunnelIndexFromId( tunnelId )
            if tunnelIndex is not None and tunnelIndex != currentIndex:
               continue
            entries[ getTunnelIndexFromId( tunnelId ) ] = entryModel

   return OspfShowSrTunnelModel( entries=entries )

OspfShowSrPfxSegDictModel = \
      generateOspfInstListCliModel( OspfShowSrPrefixSegmentSummaryModel )
OspfShowSrPfxSegVrfDictModel = \
      generateVrfCliModel( OspfShowSrPfxSegDictModel,
                           'OSPF SR prefix segments for all VRFs' )

@OspfVrfModelDecorator( vrfModelListType=OspfShowSrPfxSegVrfDictModel, 
                        instModelListType=OspfShowSrPfxSegDictModel )
def showIpOspfSrPfxSeg( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   selfOrig = 'self-originated' in args
   # Since SR is supported only for default vrf, only display that,
   # the show command anyways does not provide a vrf filter
   if not ospfSrActive( mode, instanceId, DEFAULT_VRF ):
      return None

   return getSegmentRoutingPrefixSegmentSummaryModel( instanceId=instanceId,
                                                      selfOrig=selfOrig )

OspfShowSrGlobalBlocksDictModel = \
      generateOspfInstListCliModel( OspfShowSrGlobalBlocksSummaryModel )
OspfShowSrGlobalBlocksVrfDictModel = \
      generateVrfCliModel( OspfShowSrGlobalBlocksDictModel,
                           'OSPF SR global blocks for all VRFs' )

@OspfVrfModelDecorator( vrfModelListType=OspfShowSrGlobalBlocksVrfDictModel, 
                        instModelListType=OspfShowSrGlobalBlocksDictModel )
def showIpOspfSrGlobalBlocks( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   # Since SR is supported only for default vrf, only display that,
   # the show command anyways does not provide a vrf filter
   if not ospfSrActive( mode, instanceId, DEFAULT_VRF ):
      return None

   return getSegmentRoutingGlobalBlocksSummaryModel( instanceId=instanceId )

def showIpOspfSrBindings( mode, args ):
   if 'PREFIX' in args:
      prefix = args.get( 'PREFIX' )
   else:
      prefix = Tac.Type( 'Arnet::Prefix' ).nullPrefix
   genPrefix = IpGenPrefix( prefix.stringValue )

   fwdEqvClass = FwdEqvClass()
   fwdEqvClass.prefix = genPrefix

   bindingTable = ospfSrBindingTable
   peerLbtColl = LabelBindingTableColl( "plbtc" )
   localLbt = LabelBindingTable( RouterId() )
   commonLibConsumerSm = CommonLibConsumerSm( bindingTable,
                                              localLbt,
                                              peerLbtColl,
                                              True )

   helper = MplsBindingsHelper( mplsStatus.force(), False )
   helper.populateFecBindingsFromTablesFiltered(
      commonLibConsumerSm.localLbt,
      commonLibConsumerSm.peerLbtColl,
      'detail' in args,
      fwdEqvClass )
   helper.populateLocalConflicts()

   sys.stdout.flush()
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   helper.render( fd, fmt )
   return cliPrinted( MplsModel.MplsBindingsModel )

OspfShowSrAdjSegDictModel = \
   generateOspfInstListCliModel( OspfShowSrAdjacencySegmentSummaryModel )
OspfShowSrAdjSegVrfDictModel = \
   generateVrfCliModel( OspfShowSrAdjSegDictModel,
                        'OSPF SR adjacency segments for all VRFs' )

@OspfVrfModelDecorator( vrfModelListType=OspfShowSrAdjSegVrfDictModel,
                        instModelListType=OspfShowSrAdjSegDictModel )
def showIpOspfSrAdjSeg( mode, args ):
   instanceId = args.get( 'INSTANCE_ID' )
   # Since SR is supported only for default vrf, only display that,
   # the show command anyways does not provide a vrf filter
   if not ospfSrActive( mode, instanceId, DEFAULT_VRF ):
      return None

   return getSegmentRoutingAdjacencySegmentModel( instanceId=instanceId )

class ShowSegmentRouting( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing'
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'ospf': matcherOspfShow,
         'segment-routing' : segmentRoutingKw,
   } 
   handler = showIpOspfSr
   cliModel = OspfShowSrVrfDictModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRouting )

endpointHelpString = 'Match this endpoint prefix'

class ShowSegmentRoutingTunnel( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing tunnel [ IPADDR | TUNNEL-INDEX ]'
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'ospf': matcherOspfShow,
         'segment-routing': segmentRoutingKw,
         'tunnel': tunnelKw,
         'IPADDR': IpAddrMatcher.IpPrefixMatcher( helpdesc=endpointHelpString ),
         'TUNNEL-INDEX': tunnelIndexMatcher,
   }
   handler = showIpOspfSrTunnel
   cliModel = OspfShowSrTunnelModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRoutingTunnel )

class ShowSegmentRoutingPrefixSegment( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing prefix-segments [ self-originated ]'
   data = {
         'ip': CliToken.Ip.ipMatcherForShow,
         'ospf': matcherOspfShow,
         'segment-routing' : segmentRoutingKw,
         'prefix-segments' : 'Prefix Segment Information',
         'self-originated' : 'Self-Originated segments',
   }
   handler = showIpOspfSrPfxSeg
   cliModel = OspfShowSrPfxSegVrfDictModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRoutingPrefixSegment )

class ShowSegmentRoutingGlobalBlocks( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing global-blocks'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'segment-routing' : segmentRoutingKw,
      'global-blocks' : 'Peer SRGBs',
   }
   handler = showIpOspfSrGlobalBlocks
   cliModel = OspfShowSrGlobalBlocksVrfDictModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRoutingGlobalBlocks )

#-------------------------------------------------------------------------------
# show ip ospf segment-routing bindings
#-------------------------------------------------------------------------------
class ShowSegmentRoutingBindings( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing bindings [ PREFIX ] [ detail ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'segment-routing' : segmentRoutingKw,
      'bindings': 'Label bindings',
      'PREFIX' : prefixMatcher,
      'detail' : 'Display information in detail'
   }
   handler = showIpOspfSrBindings
   cliModel = MplsModel.MplsBindingsModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRoutingBindings )

#-------------------------------------------------------------------------------
# show ip ospf segment-routing adjacency-segments
#-------------------------------------------------------------------------------
class ShowSegmentRoutingAdjSegment( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip ospf segment-routing adjacency-segments'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'ospf': matcherOspfShow,
      'segment-routing': segmentRoutingKw,
      'adjacency-segments': 'Adjacency Segment Information',
   }
   handler = showIpOspfSrAdjSeg
   cliModel = OspfShowSrAdjSegVrfDictModel

if Toggles.OspfToggleLib.toggleOspfSegmentRoutingEnabled() and \
   Toggles.OspfToggleLib.toggleOspfSegmentRoutingAdjEnabled():
   BasicCli.addShowCommandClass( ShowSegmentRoutingAdjSegment )

def Plugin( entityManager ):
   global mplsConfig
   global mplsStatus
   global srSysdbStatusDir
   global srSmashStatusDir
   global l3Config
   global ospfSrBindingTable
   global ospfStatusDir
   global ospfSrTunnelTable

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )

   mplsConfig = LazyMount.mount( entityManager, 'routing/mpls/config',
                                 'Mpls::Config', 'r' )
   mplsStatus = LazyMount.mount( entityManager, 'mpls/status',
                                 "Mpls::Api::Status", "r" )
   srSysdbStatusDir = LazyMount.mount( entityManager, 'segmentrouting/ospf',
                                       'Tac::Dir', 'ri' )
   srSmashStatusDir = SmashLazyMount.mount( entityManager,
                                            'segmentrouting/ospf/default',
                                            'Smash::SegmentRouting::Status',
                                            SmashLazyMount.mountInfo( 'reader' ) )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", 'r' )
   ospfStatusDir = LazyMount.mount( entityManager, OspfConsts.statusPath,
                                    OspfConsts.statusType, 'ri' )
   ospfSrBindingTable = shmemEm.doMount( "mpls/labelBindingTables/ospf",
                                         "CommonLibSmash::LabelBindingTable",
                                         Smash.mountInfo( 'keyshadow' ) )
   ospfSrTunnelTable = readMountTunnelTable( TunnelTableIdentifier.ospfSrTunnelTable,
                                             entityManager )
