# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, LazyMount
import ConfigMount
import CliParser, BasicCli
import CliCommand
import CliMatcher
import RibCliLib
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin import AclCli
from CliPlugin import AclCliModel
from CliPlugin import IntfCli
from CliPlugin import IraIpCli
from CliPlugin import IraIpIntfCli
from CliPlugin import IraVrfCli
from CliPlugin.RouteMapCli import mapNameMatcher
from CliPlugin.VrfCli import VrfExprFactory, ALL_VRF_NAME, DEFAULT_VRF
import CliToken.Ip
from CliToken.Router import routerMatcherForConfig
from CliToken.RouteMapCliTokens import CommonTokens as RouteMapMatchers
from ConfigConsistencyChecker import UndefinedReferenceChecker
import Arnet
from CliMode.Rip import RipMode
from socket import IPPROTO_UDP
import AclLib
import AclCliLib
import ShowCommand
import ReversibleSecretCli
from TypeFuture import TacLazyType

# pylint: disable-msg=R0201

VersionType = Tac.Type( 'Routing::Rip::RipVersion' )
AuthKey = TacLazyType( 'Routing::Rip::AuthKey' )
AuthMode = TacLazyType( 'Routing::Rip::AuthMode' )

ripConfig = None
routingHardwareStatusCommon = None
routingHardwareStatus = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
l3Config = None

def intfId( name ):
   return Tac.Value( "Arnet::IntfId", name )

#-------------------------------------------------------------------------------
# Guard function to prevent configuration on systems without routing support
#-------------------------------------------------------------------------------
def routingSupportedGuard( mode, token ):
   # pylint: disable-msg=C0321
   if routingHardwareStatus.routingSupported: return None
   return CliParser.guardNotThisPlatform

# The config interface mode commands are listed below. These commands are
# added to the RoutingProtocolIntfConfigModelet.
#
#  * IP "config-if" mode commands
#     - [no] rip v2 multicast disable

#-------------------------------------------------------------------------------
# router rip config mode. A new instance of this mode is created when the
# user enters "router rip".
#-------------------------------------------------------------------------------
class RouterRipMode( RipMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'RIP configuration'

   def __init__( self, parent, session, vrfName ):
      self.vrfName = vrfName
      RipMode.__init__( self, ( vrfName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.aclConfig_ = aclConfig
      self.aclCpConfig_ = aclCpConfig

   def _getInstanceConfig( self ):
      return ripConfig.instanceConfig.get( self.vrfName )

   def enableEcmp( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.ecmp = True

   def disableEcmp( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.ecmp = False

   def setShutdown( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.shutdown = True

   def noShutdown( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.shutdown = False

   def setNetwork( self, args ):
      prefix = args[ 'PREFIX' ]
      if prefix is None:
         self.addError( "Invalid network wildcard mask" )
         return
      instanceConfig = self._getInstanceConfig()
      prefix = Arnet.Prefix( prefix )
      instanceConfig.network[ prefix ] = True

   def noNetwork( self, args ):
      prefix = args[ 'PREFIX' ]
      if prefix is None:
         self.addError( "Invalid network wildcard mask" )
         return
      instanceConfig = self._getInstanceConfig()
      prefix = Arnet.Prefix( prefix )
      del instanceConfig.network[ prefix ]

   def distanceRip( self, args ):
      dist = args[ 'DIST' ]
      instanceConfig = self._getInstanceConfig()
      instanceConfig.preference = dist

   def noDistanceRip( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.preference = instanceConfig.preferenceDefault

   def metricRip( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.metric = args[ 'METRIC' ]

   def noMetricRip( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.metric = instanceConfig.metricDefault

   def setRedistribute( self, args ):
      instanceConfig = self._getInstanceConfig()
      proto, ospfType = args[ 'PROTOCOL' ]
      rmName = args.get( 'RMNAME', '' )

      if proto == 'protoBgpAggregate' and rmName:
         self.addWarning( 'The route-map argument is no longer supported for '
                          'aggregates and has no effect' )
      if proto in instanceConfig.redistributeConfig:
         del instanceConfig.redistributeConfig[ proto ]
      redistribute = Tac.Value( 'Routing::Rip::RipRedistribute',
                                proto, rmName, routeType=ospfType )
      instanceConfig.redistributeConfig.addMember( redistribute )

   def noRedistribute( self, args ):
      instanceConfig = self._getInstanceConfig()
      proto = args[ 'PROTOCOL' ][ 0 ]
      del instanceConfig.redistributeConfig[ proto ]

   def setDistributeList( self, args ):
      instanceConfig = self._getInstanceConfig()
      inbound = 'in' in args
      intf = args[ 'INTF' ]
      rmName = args.get( 'RMNAME', '' )
      config = instanceConfig.distListIn if inbound else instanceConfig.distListOut
      if CliCommand.isNoOrDefaultCmd( args ):
         if intf in config and config[ intf ].rmc == rmName:
            del config[ intf ]
      else:
         del config[ intf ]
         dist = Tac.Value( 'Routing::Rip::RipDistributeList', intf=intf, rmc=rmName )
         config.addMember( dist )

   # pylint: disable-msg=W0621
   def timersBasic( self, args ):
      updateTime = args[ 'UPDATE_TIME' ]
      expireTime = args[ 'EXPIRE_TIME' ]
      garbageTime = args[ 'GARBAGE_TIME' ]
      instanceConfig = self._getInstanceConfig()
      if expireTime <= updateTime:
         self.addError( "Expiration time should be greater than update interval." )
         return

      instanceConfig.updateTime = updateTime
      instanceConfig.expireTime = expireTime
      instanceConfig.garbageTime = garbageTime

   def noTimersBasic( self, args ):
      instanceConfig = self._getInstanceConfig()
      instanceConfig.updateTime = instanceConfig.updateTimeDefault
      instanceConfig.expireTime = instanceConfig.expireTimeDefault
      instanceConfig.garbageTime = instanceConfig.garbageTimeDefault

   def setServiceAcl( self, args ):
      AclCliLib.setServiceAcl( self, 'rip', IPPROTO_UDP, 
                               self.aclConfig_, self.aclCpConfig_,                   
                               args[ 'ACL' ], aclType='ip', vrfName=self.vrfName,
                               port=[ AclLib.getServByName( IPPROTO_UDP, 'rip' ) ] )

   def noServiceAcl( self, args=None ):
      aclName = args.get( 'ACL' ) if args else None
      AclCliLib.noServiceAcl( self, 'rip', self.aclConfig_, self.aclCpConfig_, 
                              aclName, aclType='ip', vrfName=self.vrfName )

#--------------------------------------------------------------------------------
# [ no | default ] router rip [ vrf VRF_NAME ]
#--------------------------------------------------------------------------------
MAX_INSTANCES_IN_VRF = 1

def gotoRouterRipMode( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   config = ripConfig
   instanceConfig = config.instanceConfig.get( vrfName )
   if not instanceConfig:
      IraIpCli.warnIfRoutingDisabled( mode,
            vrfName=vrfName if vrfName else DEFAULT_VRF )

   if instanceConfig is None:
      vrfCapability = routingHardwareStatusCommon.vrfCapability
      # This means that we haven't initialized the maxVrfs field
      # yet.  This happens when parsing the system startup config,
      # and since we don't know how many are supported we assume
      # that the config file was written correctly and accept
      # everything into Config
      maxRipInstances = vrfCapability.maxVrfs + 1
      if vrfCapability.maxVrfs != -1 and \
             len( config.instanceConfig ) >= maxRipInstances:
         # pylint: disable-next=consider-using-f-string
         mode.addError( 'More than %d RIP instance is not supported'
                        % maxRipInstances )
         return
      else:
         IraVrfCli.addAgentVrfEntry( vrfName, "Rip" )
         instanceConfig = config.instanceConfig.newMember( vrfName )
   childMode = mode.childMode( RouterRipMode, vrfName=instanceConfig.vrfName )
   mode.session_.gotoChildMode( childMode )

attributesNotNeedingCleanup = { 'ripAuth', 'distListOut', 'distListIn',
                                'redistributeConfig', 'trustedGw',
                                'sourceGw', 'network' }
def deleteRouterRipMode( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   if vrfName in ripConfig.instanceConfig:
      del ripConfig.instanceConfig[ vrfName ]
      IraVrfCli.removeAgentVrfEntry( vrfName, "Rip" )

   # remove rip service ACL configuration
   childMode = mode.childMode( RouterRipMode, vrfName=vrfName )
   childMode.noServiceAcl()
   
   entAttrs = [ i.name for i in ripConfig.tacType.attributeQ if i.writable ]
   cleanupAttrs = ( set( entAttrs ) - attributesNotNeedingCleanup )
   for attr in cleanupAttrs:
      # protect ourselves against new attributes added without a Default
      if hasattr( ripConfig, attr + 'Default' ):
         setattr( ripConfig, attr, getattr( ripConfig, attr + 'Default' ) )

class RouterRipCmd( CliCommand.CliCommandClass ):
   syntax = 'router rip [ VRF ]'
   noOrDefaultSyntax = 'router rip [ VRF ] ...'
   data = {
      'router' : routerMatcherForConfig,
      'rip' : CliCommand.guardedKeyword( 'rip',
         helpdesc='Routing Information Protocol', guard=routingSupportedGuard ),
      'VRF': VrfExprFactory( helpdesc='VRF name', inclDefaultVrf=True ),
   }

   handler = gotoRouterRipMode
   noOrDefaultHandler = deleteRouterRipMode

BasicCli.GlobalConfigMode.addCommandClass( RouterRipCmd )


#--------------------------------------------------------------------------------
# [ no | default ] ecmp
# in "router-rip" mode
#--------------------------------------------------------------------------------
class EcmpCmd( CliCommand.CliCommandClass ):
   syntax = 'ecmp'
   noOrDefaultSyntax = 'ecmp ...'
   data = {
      'ecmp' : 'Configure to enable equal-cost multipaths',
   }

   handler = RouterRipMode.enableEcmp
   noOrDefaultHandler = RouterRipMode.disableEcmp

RouterRipMode.addCommandClass( EcmpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
# in "router-rip" mode
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown' : 'Shut down RIP',
   }

   handler = RouterRipMode.setShutdown
   noHandler = RouterRipMode.noShutdown
   defaultHandler = RouterRipMode.setShutdown

RouterRipMode.addCommandClass( ShutdownCmd )


#--------------------------------------------------------------------------------
# [ no | default ] network PREFIX
#--------------------------------------------------------------------------------
class NetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'network PREFIX'
   noOrDefaultSyntax = 'network PREFIX ...'
   data = {
      'network' : 'Configure routing for a network',
      'PREFIX' : IpAddrMatcher.ipPrefixExpr( 'Network address', 'Network mask',
         'Prefix', overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
         maskKeyword=True, inverseMask=True )
   }

   handler = RouterRipMode.setNetwork
   noOrDefaultHandler = RouterRipMode.noNetwork

RouterRipMode.addCommandClass( NetworkCmd )

#--------------------------------------------------------------------------------
# [ no | default ] distance DIST
#--------------------------------------------------------------------------------
class DistanceCmd( CliCommand.CliCommandClass ):
   syntax = 'distance DIST'
   noOrDefaultSyntax = 'distance ...'
   data = {
      'distance' : 'Administrative distance configuration',
      'DIST' : CliMatcher.IntegerMatcher( 1, 255, helpdesc='Distance for routes' ),
   }

   handler = RouterRipMode.distanceRip
   noOrDefaultHandler = RouterRipMode.noDistanceRip

RouterRipMode.addCommandClass( DistanceCmd )

#-------------------------------------------------------------------------------
# "[no] metric default [number]" command, in "router-rip" mode.
#
# legacy:
# "[no] default-metric [number]" command, in "router-rip" mode.
#-------------------------------------------------------------------------------
class MetricDefaultMetricCmd( CliCommand.CliCommandClass ):
   syntax = '( ( metric default ) | default-metric ) METRIC'
   noOrDefaultSyntax = '( ( metric default ) | default-metric ) ...'
   data = {
      'metric' : 'Set metric for RIP',
      'default' : 'Set default value',
      'default-metric' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'default-metric',
         helpdesc='Set default-metric for RIP' ),
         deprecatedByCmd='metric default' ),
      'METRIC' : CliMatcher.IntegerMatcher( 0, 16,
         helpdesc='Default metric for routes' ),
   }

   handler = RouterRipMode.metricRip
   noOrDefaultHandler = RouterRipMode.noMetricRip

RouterRipMode.addCommandClass( MetricDefaultMetricCmd )

#---------------------------------------------------------------------------------
# [no] redistribute  <proto> route-map <routemap>" command, in "router-rip" mode.
#---------------------------------------------------------------------------------

# Does not contain "connected" route, which is handled differently
class RedistributedProtocolExpression( CliCommand.CliExpression ):
   expression = '''( bgp | static | aggregate | ( ospf [ match INT_EXT ] ) )'''
   data = {
       'bgp': 'BGP routes',
       'static': 'Static routes',
       'ospf': 'OSPF protocol',
       'match': 'Routes learned by the OSPF protocol',
       'aggregate': CliCommand.Node( CliMatcher.KeywordMatcher(
           'aggregate', helpdesc='Aggreate Routes' ), hidden=True ),
       'INT_EXT': CliMatcher.EnumMatcher( {
           'internal': 'OSPF routes learned from internal sources',
           'external': 'OSPF routes learned from external sources'
       } )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      ospfType = args.get( 'INT_EXT', '' )
      for proto in ( 'bgp', 'static', 'ospf' ):
         if proto in args:
            protoStr = 'proto' + proto.capitalize()
            if ospfType == 'external':
               protoStr += 'Ase'
            break
      else:
         protoStr = 'protoBgpAggregate'

      args[ 'PROTOCOL' ] = ( protoStr, ospfType )
         
# route-map is mandatory when the redistributed protocol is 'connected' (reason
# unknown)
class RedistributeCmd( CliCommand.CliCommandClass ):
   syntax = '''redistribute
               ( ( PROTOCOL [ route-map RMNAME ] )
               | ( connected route-map RMNAME ) )'''
   noOrDefaultSyntax = 'redistribute ( PROTOCOL | connected ) ...'
   data = {
       'redistribute': 'Redistribute routes in to RIP', 
       'PROTOCOL': RedistributedProtocolExpression,
       'route-map': RouteMapMatchers.routeMapApplication,
       'RMNAME': mapNameMatcher,
       'connected': 'Connected interface routes'
   }

   handler = RouterRipMode.setRedistribute
   noOrDefaultHandler = RouterRipMode.noRedistribute

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'connected' in args:
         args[ 'PROTOCOL' ] = ( 'protoDirect', '' )

RouterRipMode.addCommandClass( RedistributeCmd )

# ------------------------------------------------------------------------------
# "[no|default] distribute-list in route-map <rmap-name> [<intf-name>]"
# "[no|default] distribute-list out route-map <rmap-name> [<intf-name>]" 
# command in "router-rip" mode.
# ------------------------------------------------------------------------------
class DistributeListCmd( CliCommand.CliCommandClass ):
   syntax = '''distribute-list ( in | out ) route-map RMNAME [ INTF ]'''
   noOrDefaultSyntax = syntax
   data = {
       'distribute-list': 'Configure distribute-list filter',
       'in': 'Inbound distribute-list',
       'out': 'Outbound distribute-list',
       'route-map': RouteMapMatchers.routeMapApplication,
       'RMNAME': mapNameMatcher,
       'INTF': IntfCli.Intf.matcherWithRoutingProtoSupport
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      intf = args.get( 'INTF' )
      intfName = intf.name if intf is not None else ''
      args[ 'INTF' ] = intfId( intfName )

   handler = RouterRipMode.setDistributeList
   noOrDefaultHandler = RouterRipMode.setDistributeList

RouterRipMode.addCommandClass( DistributeListCmd )

#--------------------------------------------------------------------------------
# [ no | default ] timers basic UPDATE_TIME EXPIRE_TIME GARBAGE_TIME
#--------------------------------------------------------------------------------
class TimersBasicCmd( CliCommand.CliCommandClass ):
   syntax = 'timers [ basic ] UPDATE_TIME EXPIRE_TIME GARBAGE_TIME'
   noOrDefaultSyntax = 'timers [ basic ] ...'
   data = {
      'timers' : 'Set RIP timers in seconds',
      'basic' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'basic',
            helpdesc='Set RIP basic timers in seconds' ),
         deprecatedByCmd='timers' ),
      'UPDATE_TIME' : CliMatcher.IntegerMatcher( 5, 2147483647,
         helpdesc='Update interval' ),
      'EXPIRE_TIME' : CliMatcher.IntegerMatcher( 5, 2147483647,
         helpdesc='Expiration time of a route' ),
      'GARBAGE_TIME' : CliMatcher.IntegerMatcher( 5, 2147483647,
         helpdesc='Deletion time of a route after its expiry' ),
   }

   handler = RouterRipMode.timersBasic
   noOrDefaultHandler = RouterRipMode.noTimersBasic

RouterRipMode.addCommandClass( TimersBasicCmd )

#-------------------------------------------------------------------------------
# [ no | default ] ip access-group ACL [ in ]
# in "router-rip" mode
#-------------------------------------------------------------------------------
class IpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL [ in ]'
   noOrDefaultSyntax = 'ip access-group [ ACL ]'
   data = {
            'ip': AclCli.ipKwForServiceAclMatcher,
            'access-group': AclCli.accessGroupKwMatcher,
            'ACL': AclCli.ipAclNameMatcher,
            'in': AclCli.inKwMatcher
          }
   handler = RouterRipMode.setServiceAcl
   noOrDefaultHandler = RouterRipMode.noServiceAcl

RouterRipMode.addCommandClass( IpAccessGroupCmd )

#-------------------------------------------------------------------------------
# IntfConfig for rip configuration, is created, when one of its attributes is
# configured, and deleted, when all its attributes go back to defaults.
#-------------------------------------------------------------------------------
def _getOrCreateIntfConfig( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      intfConfig = config.intfConfig.newMember( intfName )
   return intfConfig
   
def _deleteIntfConfigIfAllAttributeHaveDefaults( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      return
   if ( ( intfConfig.metricIn == intfConfig.metricInDefault ) and
        ( intfConfig.version == intfConfig.versionDefault ) and
        ( intfConfig.metricOut == intfConfig.metricOutDefault ) and
        ( intfConfig.authMode == intfConfig.authModeDefault ) and
        ( not intfConfig.authKeys ) ):
      del config.intfConfig[ intfName ]

#------------------------------------------------------------------------------------
# "[no] rip v2 multicast disable" in config-if mode.
#
# legacy:
# "[no] ip rip v2-broadcast" in config-if mode.
#------------------------------------------------------------------------------------
def cmdRipVersionBroadcast( mode, args ):
   # pylint: disable-msg=E1120
   intfConfig = _getOrCreateIntfConfig( ripConfig, mode.intf.name )
   if intfConfig:
      intfConfig.version = VersionType.ripVersion2Broadcast

def cmdNoRipVersionBroadcast( mode, args ):
   # pylint: disable-msg=E1120
   intfConfig = _getOrCreateIntfConfig( ripConfig, mode.intf.name )
   if intfConfig:
      intfConfig.version = VersionType.ripVersion2Multicast
      _deleteIntfConfigIfAllAttributeHaveDefaults( ripConfig, \
         mode.intf.name )

nodeRip = CliCommand.guardedKeyword( 'rip', helpdesc='Routing Information Protocol',
      guard=routingSupportedGuard )

class IpRipV2BroadcastCmd( CliCommand.CliCommandClass ):
   syntax = 'ip rip v2-broadcast'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'rip' : nodeRip,
      'v2-broadcast' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'v2-broadcast',
         helpdesc='Use broadcast for RIP v2' ),
         deprecatedByCmd='rip v2 multicast disable' ),
   }

   handler = cmdRipVersionBroadcast
   noOrDefaultHandler = cmdNoRipVersionBroadcast

IraIpIntfCli.RoutingProtocolIntfConfigModelet.addCommandClass( IpRipV2BroadcastCmd )

class RipV2MulticastDisableCmd( CliCommand.CliCommandClass ):
   syntax = 'rip v2 multicast disable'
   noOrDefaultSyntax = syntax
   data = {
      'rip' : nodeRip,
      'v2' : 'RIP v2',
      'multicast' : 'Set multicast parameters',
      'disable' : 'Disable',
   }

   handler = cmdRipVersionBroadcast
   noOrDefaultHandler = cmdNoRipVersionBroadcast

IraIpIntfCli.RoutingProtocolIntfConfigModelet.addCommandClass(
      RipV2MulticastDisableCmd )

#------------------------------------------------------------------------------------
# "rip authentication algorithm md5 tx key-id <id> [rx-disabled]
#------------------------------------------------------------------------------------
authKw = CliMatcher.KeywordMatcher( 'authentication',
                                    helpdesc='Configure authentication for RIP' )
algoKw = CliMatcher.KeywordMatcher( 'algorithm',
                                    helpdesc='Configure authentication algorithm' )
md5Kw = CliMatcher.KeywordMatcher( 'md5',
                                   helpdesc="MD5 authentication" )
authKeyIdKw = CliMatcher.KeywordMatcher( 'key-id',
                                         helpdesc="authentication key-id" )
authKeyKw = CliMatcher.KeywordMatcher( 'key',
      helpdesc="authentication key" )

keyidMin = 0
KeyidMax = 255
keyIdMatcher = CliMatcher.IntegerMatcher(
   keyidMin,
   KeyidMax,
   helpdesc='Set key-id value' )

authPasswdMinLen = 1
authPasswdMaxLen = 16
# pylint: disable-next=consider-using-f-string
authPasswdPattern = '.{%d,%d}' % ( authPasswdMinLen, authPasswdMaxLen )
cleartextMatcher = CliMatcher.PatternMatcher( authPasswdPattern, helpname='WORD',
                   # pylint: disable-next=consider-using-f-string
                   helpdesc='Password (up to %d characters)' % authPasswdMaxLen )
type7key = CliCommand.Node( matcher=CliMatcher.PatternMatcher(
   r'.+', helpname='WORD',
   helpdesc='Encrypted password'), sensitive=True )

class RipAuthenticationModeCmd( CliCommand.CliCommandClass ):
   syntax = 'rip authentication algorithm md5 tx key-id KEYID [ rx-disabled ]'
   noOrDefaultSyntax = '''
      rip authentication algorithm md5 tx key-id [ KEYID ... ]'''

   data = {
      'rip' : nodeRip,
      'authentication' : authKw,
      'algorithm': algoKw,
      'md5' : md5Kw,
      'tx' : 'Configure transmit key-id',
      'key-id' : authKeyIdKw,
      'KEYID' : keyIdMatcher,
      'rx-disabled' : 'Disable all authentication checks on a received packet',
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ripConfig, mode.intf.name )
      intfConfig.authMode = AuthMode.md5
      intfConfig.authTxKeyid = args[ 'KEYID' ]
      rxDisabled = ( 'rx-disabled' in args ) or intfConfig.authRxDisabledDefault
      intfConfig.authRxDisabled = rxDisabled

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ripConfig, mode.intf.name )
      intfConfig.authMode = intfConfig.authModeDefault
      intfConfig.authTxKeyid = intfConfig.authTxKeyidDefault
      intfConfig.authRxDisabled = intfConfig.authRxDisabledDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( ripConfig, mode.intf.name )

IraIpIntfCli.RoutingProtocolIntfConfigModelet.addCommandClass(
   RipAuthenticationModeCmd )

#----------------------------------------------------------------------------------
# "rip authentication algorithm md5 key-id <id> key <key>"
#----------------------------------------------------------------------------------
def generateRipEncryptionKey( mode, args ):
   # pylint: disable-next=consider-using-f-string
   return '%s_md5_%d' % ( mode.intf.name, args[ 'ID' ] ) 

class RipAuthenticationKeyIdCmd( CliCommand.CliCommandClass ):
   syntax = 'rip authentication algorithm md5 key-id ID key AUTHKEY'
   noOrDefaultSyntax = 'rip authentication algorithm md5 key-id ID [key [AUTHKEY]]'
   data = {
      'rip' : nodeRip,
      'authentication' : authKw,
      'algorithm' : algoKw,
      'md5' : md5Kw,
      'key-id' : authKeyIdKw,
      'ID' : keyIdMatcher,
      'key' : authKeyKw,
      'AUTHKEY' : ReversibleSecretCli.ReversiblePasswordCliExpression(
                     cleartextMatcher=cleartextMatcher,
                     obfuscatedTextMatcher=type7key,
                     uniqueKeyGenerator=generateRipEncryptionKey, algorithm='DES' ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ripConfig, mode.intf.name )
      keyid = args.get( 'ID' ) 
      authKeyEntry = AuthKey( keyid )
      authKeyEntry.authKey = args[ 'AUTHKEY' ]
      intfConfig.authKeys.addMember( authKeyEntry )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( ripConfig, mode.intf.name )
      keyid = args.get( 'ID' )
      del intfConfig.authKeys[ keyid ]
      if not intfConfig.authKeys:
         _deleteIntfConfigIfAllAttributeHaveDefaults( ripConfig, mode.intf.name )

IraIpIntfCli.RoutingProtocolIntfConfigModelet.addCommandClass(
   RipAuthenticationKeyIdCmd )

def checkRipInstance( vrfName=DEFAULT_VRF ):
   return vrfName in ripConfig.instanceConfig

def runShowCmd( mode, verb, vrfName, replaceKernelIntfs=False ):
   cmd = "show ip rip "
   
   def printOutput( showCmd, vrf ):
      ret = RibCliLib.cliRibdShowCommand( mode, showCmd, clientName="RIP",
                                          output=False,
                                          replaceKernelIntfs=replaceKernelIntfs,
                                          vrf=vrf, l3Config=l3Config )
      if ret:
         # pylint: disable-next=consider-using-f-string
         print( 'RIP Router (VRF %s)' % vrf )
         print( ret )
   
   showCmd = cmd + verb
   if vrfName == ALL_VRF_NAME:
      for vrf in ripConfig.instanceConfig:
         printOutput( showCmd, vrf )
   else:
      printOutput( showCmd, vrfName )

#--------------------------------------------------------------------------------
# show ip rip neighbors [ vrf VRFNAME ]
#--------------------------------------------------------------------------------
matcherAfterShowRip = CliMatcher.KeywordMatcher( 'rip',
      helpdesc='Show RIP information' )
vrfExprFactory = VrfExprFactory(
      helpdesc='VRF name',
      inclDefaultVrf=True,
      inclAllVrf=True )

class IpRipNeighborsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip rip neighbors [ VRF ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'rip' : matcherAfterShowRip,
      'neighbors' : 'Show RIP neighbors',
      'VRF' : vrfExprFactory,
   }

   @staticmethod
   def handler( mode, args ):
      runShowCmd( mode, "gateway-summary", args.get( 'VRF', DEFAULT_VRF ) )

BasicCli.addShowCommandClass( IpRipNeighborsCmd )

#--------------------------------------------------------------------------------
# show ip rip database [ active | PREFIX ] [ vrf VRF_NAME ]
#--------------------------------------------------------------------------------
class IpRipDatabaseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip rip database [ active | PREFIX ] [ VRF ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'rip' : matcherAfterShowRip,
      'database' : 'Show RIP database',
      'active' : 'Show active routes',
      'PREFIX' : IpAddrMatcher.ipPrefixExpr( 'Network address', 'Network mask',
         'Prefix', overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
         maskKeyword=False ),
      'VRF' : vrfExprFactory,
   }

   @staticmethod
   def handler( mode, args ):
      cmd = 'database'
      if 'active' in args:
         cmd += ' active'
      elif 'PREFIX' in args:
         cmd += ' ' + str( args[ 'PREFIX' ] )
      vrf = args.get( 'VRF', DEFAULT_VRF )
      runShowCmd( mode, cmd, vrf, replaceKernelIntfs=True )

BasicCli.addShowCommandClass( IpRipDatabaseCmd )

#-------------------------------------------------------------------------------
# show ip rip access-list [ ACL_NAME ]
#-------------------------------------------------------------------------------
class IpRipAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip rip access-list [ ACL_NAME ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForShow,
      'rip' : matcherAfterShowRip,
      'access-list' : AclCli.accessListKwMatcherForServiceAcl,
      'ACL_NAME' : AclCli.ipAclNameExpression,
   }
   cliModel = AclCliModel.AllAclList

   @staticmethod
   def handler( mode, args ):
      name = args[ '<aclNameExpr>' ]
      return AclCli.showServiceAcl( mode,
                                    aclCpConfig,
                                    aclStatus,
                                    aclCheckpoint,
                                    'ip',
                                    name,
                                    serviceName='rip' )

BasicCli.addShowCommandClass( IpRipAccessListCmd )

#-------------------------------------------------------------------------------
# The RipIntf class is used to remove the Rip IntfConfig object when
# an interface is deleted. The Intf class will create a new instance
# of RipIntf and call destroy when the interface is deleted
# -------------------------------------------------------------------------------
class RipIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      name = self.intf_.name
      del ripConfig.intfConfig[ name ]

#--------------------------------------------------------------------------------
# clear ip rip access-list counters
#--------------------------------------------------------------------------------
class ClearIpRipAccessListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip rip access-list counters'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'ip' : CliToken.Ip.ipMatcherForClear,
      'rip' : 'Reset RIP state',
      'access-list' : AclCli.accessListMatcher,
      'counters' : 'Reset ACL counters',
   }

   @staticmethod
   def handler( mode, args ):
      # clear counters of all access-lists
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, "ip" )

BasicCli.EnableMode.addCommandClass( ClearIpRipAccessListCountersCmd )

class RipRouteMapReferenceGatherer:
   """
   This class gathers references to route maps in commands under
   the 'router rip' mode.
   """
   @staticmethod
   def gather( feature ):
      references = set()
      gatherer = RipRouteMapReferenceGatherer
      for vrfName in ripConfig.instanceConfig:
         instanceConfig = ripConfig.instanceConfig[ vrfName ]
         gatherer.gatherRefsInRedistributeConfig( instanceConfig, references )
         gatherer.gatherRefsInDistList( instanceConfig, references )
      return references

   @staticmethod
   def gatherRefsInRedistributeConfig( instanceConfig, references ):
      redistConfig = instanceConfig.redistributeConfig
      for proto in redistConfig:
         redistribute = redistConfig[ proto ]
         routeMap = redistribute.rmc
         if routeMap:
            references.add( routeMap )

   @staticmethod
   def gatherRefsInDistList( instanceConfig, references ):
      distListIn = instanceConfig.distListIn
      for intf in distListIn:
         distList = distListIn[ intf ]
         routeMap = distList.rmc
         if routeMap:
            references.add( routeMap )
      distListOut = instanceConfig.distListOut
      for intf in distListOut:
         distList = distListOut[ intf ]
         routeMap = distList.rmc
         if routeMap:
            references.add( routeMap )

UndefinedReferenceChecker.addReferenceGatherer(
      [ "Route map" ], RipRouteMapReferenceGatherer )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global ripConfig
   global routingHardwareStatusCommon
   global routingHardwareStatus
   global aclConfig
   global aclCpConfig
   global aclCheckpoint
   global aclStatus
   global l3Config
   ripConfig = ConfigMount.mount( entityManager, 'routing/rip/config',
                                  'Routing::Rip::Config', "w" )
   routingHardwareStatusCommon = LazyMount.mount(
      entityManager, 'routing/hardware/statuscommon',
      'Routing::Hardware::StatusCommon', 'r' )
   routingHardwareStatus = LazyMount.mount( entityManager, 'routing/hardware/status',
                                            'Routing::Hardware::Status', 'r' )
   IntfCli.Intf.registerDependentClass( RipIntf )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   l3Config = LazyMount.mount( entityManager, "l3/config", "L3::Config", 'r' )
