#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import Cell
import CliCommand
import CliGlobal
import CliMatcher
from CliMode.Rpki import (
      RpkiCacheMode,
      RpkiOriginValidationBaseMode,
      RpkiTransportTcpMode,
      RpkiTransportTlsMode,
)
from CliPlugin import TechSupportCli
from CliPlugin.IntfCli import Intf
# pylint: disable-next=consider-using-from-import
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
# pylint: disable-next=consider-using-from-import
import CliPlugin.IpAddrMatcher as IpAddrMatcher
from CliPlugin.RouteMapCli import mapNameMatcher
from CliPlugin.RoutingBgpCli import (
      RouterBgpBaseMode,
      RouterBgpSharedModelet,
      deleteRouterBgpMacVrfHook,
      PeerCliExpression,
)
from CliPlugin.Ssl import profileMatcher, sslMatcher
from CliPlugin.VrfCli import VrfExprFactory
from CliSavePlugin.RoutingBgpCliSave import showBgpConfigInstanceCallbacks
from CliSavePlugin.RpkiCliSave import (
      saveRpkiOriginValidationConfig,
      saveCacheConfig,
)
from CliToken import (
      RoutingBgp,
      RoutingBgpShowCliTokens,
      Clear,
)
import ConfigMount
from EosRpkiLib import getCacheMessage
import HostnameCli
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType
import LazyMount
import ShowCommand
import Tac
from TypeFuture import TacLazyType

# pkgdeps: rpm RpkiLib-lib

# Using CliGlobal to ensure that the variables are set only once, thus avoiding race
# conditions
gv = CliGlobal.CliGlobal( rpkiConfig=None,
                          rpkiStatus=None,
                          l3Config=None,
                          sslConfig=None,
                          roaTableStatusDir=None )

def getCacheNames( mode ):
   return list( gv.rpkiConfig.cacheConfig )

def getRoaTableNames( mode ):
   return list( gv.roaTableStatusDir.roaTableStatus )
rpkiKwMatcher = CliMatcher.KeywordMatcher(
   'rpki',
   helpdesc='Resource Public Key Infrastructure' )
rpkiCacheKwMatcher = CliMatcher.KeywordMatcher(
   'cache',
   helpdesc='Cache server instance' )
rpkiCacheNameMatcher = CliMatcher.DynamicNameMatcher(
   getCacheNames,
   'Name of cache server',
   pattern=r'.+',
   helpname='NAME' )
# Exclude 'counters' from the list of cache names, to prevent us from
# misinterpreting this token as a cache name in the show commands.
rpkiShowCacheNameMatcher = CliMatcher.DynamicNameMatcher(
   getCacheNames,
   'Name of cache server',
   pattern=r'(?!counters$).+',
   helpname='NAME' )
roaKwMatcher = CliMatcher.KeywordMatcher(
   'roa',
   helpdesc='Route Origin Authorizations (ROAs) obtained from configured cache '
   'server(s)' )
rpkiShowRoaTableNameMatcher = CliMatcher.DynamicNameMatcher(
   getRoaTableNames,
   'Name of ROA table',
   pattern=r'.+',
   helpname='TABLENAME' )

class RpkiCacheConfigMode( RpkiCacheMode, BasicCli.ConfigModeBase ):
   name = 'RPKI cache server configuration'

   def __init__( self, parent, session, cacheName ):
      RpkiCacheMode.__init__( self, cacheName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.rpkiCacheConfig = gv.rpkiConfig.cacheConfig.newMember( cacheName )
      if not self.rpkiCacheConfig.roaTableConfig:
         rpkiDefaults = Tac.Type( 'Rpki::RpkiDefaults' )
         roaTableConfig = Tac.newInstance( 'Rpki::CacheConfig::RoaTableConfig' )
         roaTableConfig.roaTableNames.add( rpkiDefaults.defaultRoaTableName )
         self.rpkiCacheConfig.roaTableConfig = roaTableConfig

#------------------------------------------------------------------------------------
# The "[ no | default ] rpki cache <name> command, in "config-router-bgp" mode.
#------------------------------------------------------------------------------------
class EnterRpkiCacheConfigMode( CliCommand.CliCommandClass ):
   syntax = 'rpki cache NAME'
   noOrDefaultSyntax = syntax
   data = {
      'rpki': rpkiKwMatcher,
      'cache': 'Configuration options related to RPKI cache server',
      'NAME': rpkiCacheNameMatcher
   }

   handler = "RpkiCliHandler.doEnterRpkiCacheConfigMode"

   noOrDefaultHandler = "RpkiCliHandler.noEnterRpkiCacheConfigMode"

#------------------------------------------------------------------------------------
# The "[ no | default ] host IP_OR_HOST [ vrf VRF ] [ port PORT ] command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiHostVrfPortCmd( CliCommand.CliCommandClass ):
   syntax = 'host IP_OR_HOST [ VRF ] [ port PORT ]'
   noOrDefaultSyntax = 'host ...'
   data = {
      'host': 'Configure host of cache server',
      'IP_OR_HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True,
                    helpdesc='IPv4 address or IPv6 address or fully qualified '
                             'domain name of cache server' ),
      'VRF': VrfExprFactory( helpdesc='Specify the VRF in which '
                                      'the cache server is configured' ),
      'port': 'Configure transport port',
      'PORT': CliMatcher.IntegerMatcher( 0, 65535,
                                         helpdesc='Cache server port number' )
   }

   handler = "RpkiCliHandler.doRpkiHostVrfPortCmd"

   noOrDefaultHandler = "RpkiCliHandler.noRpkiHostVrfPortCmd"

#------------------------------------------------------------------------------------
# The "[ no | default ] refresh-interval < 1-86400 seconds > command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiRefreshIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh-interval SECONDS'
   noOrDefaultSyntax = 'refresh-interval ...'
   data = {
      'refresh-interval': 'Configure refresh interval',
      'SECONDS': CliMatcher.IntegerMatcher( 1, 86400,
                                            helpdesc='Number of seconds between '
                                                     'cache server poll' )
   }

   handler = "RpkiCliHandler.doRpkiRefreshIntervalCmd"

   noOrDefaultHandler = handler

#------------------------------------------------------------------------------------
# The "[ no | default ] retry-interval < 1-7200 seconds > command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiRetryIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'retry-interval SECONDS'
   noOrDefaultSyntax = 'retry-interval ...'
   data = {
      'retry-interval': 'Configure retry interval',
      'SECONDS': CliMatcher.IntegerMatcher( 1, 7200,
                                            helpdesc='Number of seconds between '
                                                     'cache server poll since '
                                                     'error' )
   }

   handler = "RpkiCliHandler.doRpkiRetryIntervalCmd"

   noOrDefaultHandler = handler

#------------------------------------------------------------------------------------
# The "[ no | default ] expire-interval < 600-172800 seconds > command,
# in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiExpireIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'expire-interval SECONDS'
   noOrDefaultSyntax = 'expire-interval ...'
   data = {
      'expire-interval': 'Configure expire interval',
      'SECONDS': CliMatcher.IntegerMatcher( 600, 172800,
                                            helpdesc='Number of seconds to retain '
                                                     'data synced from cache server '
                                                     'after connection loss' )
   }

   handler = "RpkiCliHandler.doRpkiExpireIntervalCmd"

   noOrDefaultHandler = handler

#------------------------------------------------------------------------------------
# The "[ no | default ] preference < 0-10 > command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiPreferenceCmd( CliCommand.CliCommandClass ):
   syntax = 'preference VALUE'
   noOrDefaultSyntax = 'preference ...'
   data = {
      'preference': 'Configure cache server preference',
      'VALUE': CliMatcher.IntegerMatcher( 0, 10,
                                          helpdesc='Cache server preference. '
                                                   'Lower value means '
                                                   'higher preference.' )
   }

   handler = "RpkiCliHandler.doRpkiPreferenceCmd"

   noOrDefaultHandler = "RpkiCliHandler.noRpkiPreferenceCmd"

#------------------------------------------------------------------------------------
# The "[ no | default ] local-interface <INTF> command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'local-interface INTF'
   noOrDefaultSyntax = 'local-interface ...'
   data = {
      'local-interface': 'Specify a local source interface to connect to cache '
                         'server',
      'INTF': Intf.matcherWithIpSupport,
   }

   handler = "RpkiCliHandler.doRpkiLocalInterfaceCmd"

   noOrDefaultHandler = "RpkiCliHandler.noRpkiLocalInterfaceCmd"

#------------------------------------------------------------------------------------
# The "[ no | default ] roa table {<NAME>} command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class RpkiRoaTableCmd( CliCommand.CliCommandClass ):
   syntax = 'roa table { NAME }'
   noOrDefaultSyntax = 'roa table [ { NAME } ]'
   data = {
      'roa': roaKwMatcher,
      'table': 'Assign RPKI ROA tables to cache server',
      'NAME': CliMatcher.DynamicNameMatcher( getRoaTableNames,
                                             'Name of RPKI ROA table',
                                             pattern=r'.+', helpname='NAME' ),
   }

   handler = "RpkiCliHandler.doRpkiRoaTableCmd"

   noOrDefaultHandler = "RpkiCliHandler.noRpkiRoaTableCmd"

transportKwMatcher = CliMatcher.KeywordMatcher(
                        'transport',
                        helpdesc='Configure transport option' )

transportAuthenticationTypes = TacLazyType( 'Rpki::RpkiTransportAuthenticationType' )

class RpkiTransportTcpConfigMode( RpkiTransportTcpMode, BasicCli.ConfigModeBase ):
   name = 'RPKI cache tcp transport configuration'

   def __init__( self, parent, session ):
      RpkiTransportTcpMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.rpkiCacheConfig = parent.rpkiCacheConfig
      self.tcp = transportAuthenticationTypes.tcp

#------------------------------------------------------------------------------------
# The "[ no | default ] transport tcp command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class EnterRpkiTransportTcpConfigMode( CliCommand.CliCommandClass ):
   syntax = 'transport tcp'
   noOrDefaultSyntax = syntax
   data = {
      'transport': transportKwMatcher,
      'tcp': 'Unprotected TCP'
   }

   handler = "RpkiCliHandler.doEnterRpkiTransportTcpConfigMode"

   noOrDefaultHandler = "RpkiCliHandler.noEnterRpkiTransportTcpConfigMode"

#------------------------------------------------------------------------------------
# The "[ no | default ] tcp keepalive IDLE_TIME PROBE_INTERVAL PROBE_COUNT" command
# in "config-rpki-cache-transport-tcp" mode.
#------------------------------------------------------------------------------------
class TcpKeepaliveCommand( CliCommand.CliCommandClass ):
   syntax = 'tcp keepalive IDLE_TIME PROBE_INTERVAL PROBE_COUNT'
   noOrDefaultSyntax = 'tcp keepalive ...'
   data = {
      'tcp': 'TCP connection options',
      'keepalive': 'Specify the RPKI cache\'s TCP keepalive parameters',
      'IDLE_TIME': CliMatcher.IntegerMatcher( 1, 86400,
         helpdesc='Idle time (seconds) before TCP keepalive' ),
      'PROBE_INTERVAL': CliMatcher.IntegerMatcher( 1, 3600,
         helpdesc='Interval (seconds) between TCP keepalive probes' ),
      'PROBE_COUNT': CliMatcher.IntegerMatcher( 1, 1000,
         helpdesc='Number of keepalive probes before closing connection' )
   }

   handler = "RpkiCliHandler.doTcpKeepaliveCommand"

   noOrDefaultHandler = "RpkiCliHandler.noTcpKeepaliveCommand"

class RpkiTransportTlsConfigMode( RpkiTransportTlsMode, BasicCli.ConfigModeBase ):
   name = 'RPKI cache server transport TLS configuration'

   def __init__( self, parent, session ):
      RpkiTransportTlsMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.rpkiCacheConfig = parent.rpkiCacheConfig
      self.tls = transportAuthenticationTypes.tls

#------------------------------------------------------------------------------------
# The "[ no | default ] transport tls command, in "config-rpki-cache" mode.
#------------------------------------------------------------------------------------
class EnterRpkiTransportTlsConfigMode( CliCommand.CliCommandClass ):
   syntax = 'transport tls'
   noOrDefaultSyntax = syntax
   data = {
      'transport': transportKwMatcher,
      'tls': 'Transport Layer Security'
   }

   handler = "RpkiCliHandler.doEnterRpkiTransportTlsConfigMode"

   noOrDefaultHandler = "RpkiCliHandler.noEnterRpkiTransportTlsConfigMode"

#------------------------------------------------------------------------------------
# The "[ no | default ] ssl profile PROFILE_NAME command,
# in "config-rpki-cache-trans-tls" mode.
#------------------------------------------------------------------------------------
class RpkiTlsSslProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILE_NAME'
   noOrDefaultSyntax = 'ssl profile ...'
   data = {
      'ssl': sslMatcher,
      'profile': profileMatcher,
      'PROFILE_NAME': CliMatcher.DynamicNameMatcher( lambda mode:
                                                     gv.sslConfig.profileConfig,
                                                     helpdesc='Profile name' ),
   }

   handler = "RpkiCliHandler.doRpkiTlsSslProfileCmd"

   noOrDefaultHandler = "RpkiCliHandler.noRpkiTlsSslProfileCmd"

#------------------------------------------------------------------------------------
# The "show bgp rpki roa ( ipv4 | ipv6 ) [<prefix>]" command.
#------------------------------------------------------------------------------------
class RpkiShowRoas( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp rpki roa ( ipv4 [ PREFIX4 ] ) | ( ipv6 [ PREFIX6 ] )
               [ cache ( missing [ NAME ] ) | NAME ] [ table TABLENAME ]'''

   _prefixText = 'Only display ROAs that affect origin validation' \
                 ' of the specified prefix'
   data = {
      'bgp': RoutingBgpShowCliTokens.bgpAfterShow,
      'rpki': rpkiKwMatcher,
      'roa': roaKwMatcher,
      'ipv4': 'IPv4 unicast address family',
      'ipv6': 'IPv6 unicast address family',
      'PREFIX4': IpAddrMatcher.IpPrefixMatcher(
         _prefixText,
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'PREFIX6': Ip6AddrMatcher.Ip6PrefixMatcher(
         _prefixText,
         overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'cache': 'Filter ROAs by the presence in or absence from a cache',
      'missing': 'Display entries missing from any synced cache '
                 'or the specified cache',
      'NAME': rpkiShowCacheNameMatcher,
      'table': 'Filter ROAs by the presence in a particular ROA table',
      'TABLENAME': rpkiShowRoaTableNameMatcher,
   }
   cliModel = "RpkiCliModels.RpkiShowRoaModel"

   handler = "RpkiCliHandler.doRpkiShowRoas"

def generateBgpRpkiCacheExpr( counters=False ):
   class RpkiShowCacheExpression( CliCommand.CliExpression ):
      expression = '''bgp rpki cache [ NAME ]'''
      data = {
         'bgp': RoutingBgpShowCliTokens.bgpAfterShow,
         'rpki': rpkiKwMatcher,
         'cache': rpkiCacheKwMatcher,
         'NAME': rpkiShowCacheNameMatcher,
      }
      if counters:
         expression += ' counters'
         data[ 'counters' ] = "Cache server PDU counters"
   return RpkiShowCacheExpression

def generateBgpRpkiClearCacheExpr( counters=False ):
   class RpkiClearCacheExpression( CliCommand.CliExpression ):
      expression = '''bgp rpki cache ( all | NAME )'''
      data = {
         'bgp': 'Bgp',
         'rpki': rpkiKwMatcher,
         'cache': rpkiCacheKwMatcher,
         'all': 'Clear all cache server instances',
         'NAME': rpkiShowCacheNameMatcher,
      }
      if counters:
         expression += ' counters'
         data[ 'counters' ] = "Cache server PDU and error PDU counters"
   return RpkiClearCacheExpression

#------------------------------------------------------------------------------------
# The "show bgp rpki roa summary" command.
#------------------------------------------------------------------------------------
class RpkiShowRoaSummary( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp rpki roa summary '''
   data = {
      'bgp': RoutingBgpShowCliTokens.bgpAfterShow,
      'rpki': rpkiKwMatcher,
      'roa': roaKwMatcher,
      'summary': 'Summarized ROA information'
   }
   cliModel = "RpkiCliModels.RpkiShowRoaSummaryModel"

   handler = "RpkiCliHandler.doRpkiShowRoaSummary"

#------------------------------------------------------------------------------------
# The "show bgp rpki roa table summary" command.
#------------------------------------------------------------------------------------
class RpkiShowRoaTableSummary( ShowCommand.ShowCliCommandClass ):
   syntax = '''show bgp rpki roa table summary'''
   data = {
      'bgp': RoutingBgpShowCliTokens.bgpAfterShow,
      'rpki': rpkiKwMatcher,
      'roa': roaKwMatcher,
      'table': 'RPKI ROA table',
      'summary': 'Summarized ROA table information'
   }
   cliModel = "RpkiCliModels.RpkiShowRoaTableSummaryModel"

   handler = "RpkiCliHandler.doRpkiShowRoaTableSummary"

#------------------------------------------------------------------------------------
# The "show bgp rpki cache [<name>]" command.
#------------------------------------------------------------------------------------
class RpkiShowCache( ShowCommand.ShowCliCommandClass ):
   syntax = '''show BGP_RPKI_CACHE_NAME [ detail ]'''
   data = {
      'BGP_RPKI_CACHE_NAME': generateBgpRpkiCacheExpr(),
      'detail': 'Detailed view'
   }
   cliModel = "RpkiCliModels.RpkiShowCacheModel"

   handler = "RpkiCliHandler.doRpkiShowCache"

def counterAdapter( mode, args, argsList ):
   caches = []
   cacheName = args.get( 'NAME' )

   cacheId = TacLazyType( 'Rpki::CacheId' )

   if cacheName and gv.rpkiStatus.cacheStatus.get( cacheName ):
      caches.append( cacheId( cacheName ) )
   elif not cacheName:
      caches = [ cacheId( cacheName ) for cacheName in gv.rpkiStatus.cacheStatus ]

   args[ 'CACHES' ] = caches

#------------------------------------------------------------------------------------
# The "show bgp rpki cache [<name>] counters" command.
#-----------------------------------------------------------------------------------
class RpkiShowCachePduCounters( ShowCommand.ShowCliCommandClass ):
   syntax = '''show BGP_RPKI_CACHE_COUNTERS'''
   data = {
      'BGP_RPKI_CACHE_COUNTERS': generateBgpRpkiCacheExpr( counters=True ),
   }
   cliModel = "RpkiCliModels.RpkiShowCacheCounterModel"

   handler = "RpkiCliHandler.doRpkiShowCachePduCounters"

   adapter = counterAdapter

#------------------------------------------------------------------------------------
# The "show bgp rpki cache [<name>] counters errors" command.
#-----------------------------------------------------------------------------------
class RpkiShowCachePduCountersErrors( ShowCommand.ShowCliCommandClass ):
   syntax = '''show BGP_RPKI_CACHE_COUNTERS errors'''
   data = {
      'BGP_RPKI_CACHE_COUNTERS': generateBgpRpkiCacheExpr( counters=True ),
      'errors': 'Cache server PDU error statistics',
   }
   cliModel = "RpkiCliModels.RpkiShowCacheErrorCounterModel"

   handler = "RpkiCliHandler.doRpkiShowCachePduCountersErrors"

   adapter = counterAdapter

def clearAdapter( mode, args, argsList ):
   cacheName = args.get( 'NAME' )
   # clearing a specific cache
   if cacheName:

      # check if cache exists
      if cacheName not in gv.rpkiStatus.cacheStatus:
         mode.addMessage( getCacheMessage( cacheName ) )
         return
   # clearing all caches
   if not gv.rpkiStatus.cacheStatus:
      # no cache is configured
      mode.addMessage( getCacheMessage( None ) )
      return

#------------------------------------------------------------------------------------
# The "clear bgp rpki cache ( all | <name> ) command.
#------------------------------------------------------------------------------------
class RpkiClearCacheCmd( CliCommand.CliCommandClass ):
   syntax = 'clear BGP_RPKI_CACHE_NAME'
   data = {
      'clear': Clear.clearKwNode,
      'BGP_RPKI_CACHE_NAME': generateBgpRpkiClearCacheExpr(),
   }
   adapter = clearAdapter

   handler = "RpkiCliHandler.doRpkiClearCacheCmd"

#------------------------------------------------------------------------------------
# The "clear bgp rpki cache ( all | <name> ) counters" command.
#------------------------------------------------------------------------------------
class RpkiClearCacheCounterCmd( CliCommand.CliCommandClass ):
   syntax = 'clear BGP_RPKI_CACHE_NAME'
   data = {
      'clear': Clear.clearKwNode,
      'BGP_RPKI_CACHE_NAME': generateBgpRpkiClearCacheExpr( counters=True ),
   }
   adapter = clearAdapter

   handler = "RpkiCliHandler.doRpkiClearCacheCounterCmd"

# Clean-up hook for rpkiConfig if bgpConfig is unconfigured.
def deleteRpkiCacheConfigHook():
   gv.rpkiConfig.cacheConfig.clear()

class RpkiOriginValidationMode( RpkiOriginValidationBaseMode,
                                BasicCli.ConfigModeBase ):
   name = 'RPKI Origin Validation configuration'

   def __init__( self, parent, session ):
      RpkiOriginValidationBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.vrfName = DEFAULT_VRF

originValidationKwMatcher = CliMatcher.KeywordMatcher(
   'origin-validation',
   helpdesc='Prefix origin validation' )

#------------------------------------------------------------------------------------
#  "[ no | default ] rpki origin-validation" command, in "router-bgp" mode.
#------------------------------------------------------------------------------------
class EnterBgpBaseRpkiOriginValidationMode( CliCommand.CliCommandClass ):
   syntax = 'rpki origin-validation'
   noOrDefaultSyntax = syntax
   data = {
         'rpki': rpkiKwMatcher,
         'origin-validation': originValidationKwMatcher,
      }

   handler = "RpkiCliHandler.doEnterBgpBaseRpkiOriginValidationMode"

   noOrDefaultHandler = "RpkiCliHandler.noEnterBgpBaseRpkiOriginValidationMode"

RouterBgpBaseMode.addCommandClass( EnterBgpBaseRpkiOriginValidationMode )

ibgpKwMatcher = CliMatcher.KeywordMatcher(
   'ibgp',
   helpdesc='Configure options for iBGP neighbors' )
ebgpKwMatcher = CliMatcher.KeywordMatcher(
   'ebgp',
   helpdesc='Configure options for eBGP neighbors' )
localKwMatcher = CliMatcher.KeywordMatcher(
   'local',
   helpdesc='Use the ROA database' )
communityKwMatcher = CliMatcher.KeywordMatcher(
   'community',
   helpdesc='Use the received Origin Validation State community' )
preferCommunityKwMatcher = CliMatcher.KeywordMatcher(
   'prefer-community',
   helpdesc=( 'Use the received Origin Validation State community '
              'then the ROA database' ) )
disabledKwMatcher = CliMatcher.KeywordMatcher(
   'disabled',
   helpdesc='Disabled' )

#------------------------------------------------------------------------------------
#  "[ no | default ] ( ibgp | ebgp ) ( local | community | prefer-community )"
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiRecvOriginValidation( CliCommand.CliCommandClass ):
   syntax = '( ibgp | ebgp ) ( local | community | prefer-community )'
   noOrDefaultSyntax = '( ibgp | ebgp ) ...'
   data = {
         'ibgp': ibgpKwMatcher,
         'ebgp': ebgpKwMatcher,
         'local': localKwMatcher,
         'community': communityKwMatcher,
         'prefer-community': preferCommunityKwMatcher,
   }

   handler = "RpkiCliHandler.setOriginValidationMethod"

   noOrDefaultHandler = "RpkiCliHandler.noOriginValidationMethod"

validationKwMatcher = CliMatcher.KeywordMatcher(
   'validation',
   helpdesc='Validation configuration' )
routeMapKwMatcher = CliMatcher.KeywordMatcher(
   'route-map',
   helpdesc='Route map' )

#------------------------------------------------------------------------------------
#  "[ no | default ] validation route-map ROUTEMAP" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiValidationRoutemap( CliCommand.CliCommandClass ):
   syntax = 'validation route-map ROUTEMAP'
   noOrDefaultSyntax = 'validation route-map ...'
   data = {
         'validation': validationKwMatcher,
         'route-map': routeMapKwMatcher,
         'ROUTEMAP': mapNameMatcher,
   }

   handler = "RpkiCliHandler.setOriginValidationRoutemap"

   noOrDefaultHandler = "RpkiCliHandler.noOriginValidationRoutemap"

redistributedKwMatcher = CliMatcher.KeywordMatcher(
   'redistributed',
   helpdesc='Configure options for redistributed routes' )

#------------------------------------------------------------------------------------
#  "[ no | default ] redistributed local" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiRedistributedOriginValidation( CliCommand.CliCommandClass ):
   syntax = 'redistributed local'
   noOrDefaultSyntax = 'redistributed ...'
   data = {
         'redistributed': redistributedKwMatcher,
         'local': localKwMatcher,
   }

   handler = "RpkiCliHandler.setOriginValidationMethod"

   noOrDefaultHandler = "RpkiCliHandler.noOriginValidationMethod"

#------------------------------------------------------------------------------------
#  "[ no | default ] redistributed validation route-map ROUTEMAP" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiRedistributedValidationRoutemap( CliCommand.CliCommandClass ):
   syntax = 'redistributed validation route-map ROUTEMAP'
   noOrDefaultSyntax = 'redistributed validation route-map ...'
   data = {
         'redistributed': redistributedKwMatcher,
         'validation': validationKwMatcher,
         'route-map': routeMapKwMatcher,
         'ROUTEMAP': mapNameMatcher,
   }

   handler = "RpkiCliHandler.setOriginValidationRoutemap"

   noOrDefaultHandler = "RpkiCliHandler.noOriginValidationRoutemap"

sendKwMatcher = CliMatcher.KeywordMatcher(
   'send',
   helpdesc='Attach Origin Validation State community to advertised routes' )

#------------------------------------------------------------------------------------
#  "[ no | default ] ( ibgp | ebgp ) send" command
#  in "rpki-origin-validation" mode.
#------------------------------------------------------------------------------------
class RpkiCommunitySend( CliCommand.CliCommandClass ):
   syntax = '( ibgp | ebgp ) send'
   noOrDefaultSyntax = syntax
   data = {
         'ibgp': ibgpKwMatcher,
         'ebgp': ebgpKwMatcher,
         'send': sendKwMatcher,
   }

   handler = "RpkiCliHandler.setOriginValidationSend"

   noOrDefaultHandler = "RpkiCliHandler.noOriginValidationSend"

RpkiOriginValidationMode.addCommandClass( RpkiRecvOriginValidation )
RpkiOriginValidationMode.addCommandClass( RpkiValidationRoutemap )
RpkiOriginValidationMode.addCommandClass( RpkiRedistributedOriginValidation )
RpkiOriginValidationMode.addCommandClass( RpkiRedistributedValidationRoutemap )
RpkiOriginValidationMode.addCommandClass( RpkiCommunitySend )

# Per-neighbor commands

#------------------------------------------------------------------------------------
#  "[no | default ] neighbor <addr|peer-group> rpki origin-validation ( local |
#    community | prefer-community | disabled )" command in "router-bgp" mode.
#------------------------------------------------------------------------------------
class BgpNeighborRpkiRecvOriginValidation( CliCommand.CliCommandClass ):
   syntax = ( 'neighbor PEER rpki origin-validation ( local | community | '
              'prefer-community | disabled )' )
   noOrDefaultSyntax = ( 'neighbor PEER rpki origin-validation ...' )
   data = {
         'neighbor': RoutingBgp.neighbor,
         'PEER': PeerCliExpression,
         'rpki': rpkiKwMatcher,
         'origin-validation': originValidationKwMatcher,
         'local': localKwMatcher,
         'community': communityKwMatcher,
         'prefer-community': preferCommunityKwMatcher,
         'disabled': disabledKwMatcher,
   }

   handler = "RpkiCliHandler.doBgpNeighborRpkiRecvOriginValidation"

   noOrDefaultHandler = "RpkiCliHandler.noBgpNeighborRpkiRecvOriginValidation"

#------------------------------------------------------------------------------------
#  "[no|default] neighbor <addr|peer-group> rpki origin-validation send [disabled]"
#   command in "router-bgp" mode.
#------------------------------------------------------------------------------------
class BgpNeighborRpkiCommunitySend( CliCommand.CliCommandClass ):
   syntax = ( 'neighbor PEER rpki origin-validation send [ disabled ]' )
   noOrDefaultSyntax = syntax
   data = {
         'neighbor': RoutingBgp.neighbor,
         'PEER': PeerCliExpression,
         'rpki': rpkiKwMatcher,
         'origin-validation': originValidationKwMatcher,
         'send': sendKwMatcher,
         'disabled': disabledKwMatcher,
   }

   handler = "RpkiCliHandler.doBgpNeighborRpkiCommunitySend"

   noOrDefaultHandler = "RpkiCliHandler.noBgpNeighborRpkiCommunitySend"

#---------------------------------------------------------------------------
# Add rpki related commands to 'show tech-support'.
# These commands only apply to multi-agent mode when an RPKI cache is configured.
#--------------------------------------------------------------------------
def _showTechRpkiGuard():
   return ( gv.l3Config.protocolAgentModel == ProtocolAgentModelType.multiAgent and
            len( getCacheNames( None ) ) > 0 )

TechSupportCli.registerShowTechSupportCmd(
   '2020-01-05 12:45:00',
   cmds=[ 'show bgp rpki cache detail',
          'show bgp rpki cache counters',
          'show bgp rpki cache counters errors',
          'show bgp rpki roa summary' ],
   cmdsGuard=_showTechRpkiGuard,
   summaryCmds=[ 'show bgp rpki cache detail',
                 'show bgp rpki cache counters',
                 'show bgp rpki cache counters errors',
                 'show bgp rpki roa summary' ],
   summaryCmdsGuard=_showTechRpkiGuard )

def showBgpConfigRpki( bgpConfig ):
   '''Return a dictionary of submodes and config commands to include in the
   "show bgp configuration active" output.'''
   cmdDict = {}
   for cacheName in sorted( gv.rpkiConfig.cacheConfig ):
      cacheConfig = gv.rpkiConfig.cacheConfig[ cacheName ]
      cmds = saveCacheConfig( cacheName, cacheConfig )
      cmdDict[ f"rpki cache {cacheName}" ] = cmds

   cmds = saveRpkiOriginValidationConfig( bgpConfig )
   if cmds:
      cmdDict[ 'rpki origin-validation' ] = cmds

   return cmdDict

showBgpConfigInstanceCallbacks.append( showBgpConfigRpki )

RouterBgpBaseMode.addCommandClass( EnterRpkiCacheConfigMode )
deleteRouterBgpMacVrfHook.addExtension( deleteRpkiCacheConfigHook )
RpkiCacheConfigMode.addCommandClass( RpkiHostVrfPortCmd )
RpkiCacheConfigMode.addCommandClass( RpkiRefreshIntervalCmd )
RpkiCacheConfigMode.addCommandClass( RpkiRetryIntervalCmd )
RpkiCacheConfigMode.addCommandClass( RpkiExpireIntervalCmd )
RpkiCacheConfigMode.addCommandClass( RpkiPreferenceCmd )
RpkiCacheConfigMode.addCommandClass( RpkiLocalInterfaceCmd )
RpkiCacheConfigMode.addCommandClass( EnterRpkiTransportTcpConfigMode )
RpkiTransportTcpConfigMode.addCommandClass( TcpKeepaliveCommand )
RpkiCacheConfigMode.addCommandClass( EnterRpkiTransportTlsConfigMode )
RpkiTransportTlsConfigMode.addCommandClass( RpkiTlsSslProfileCmd )
RpkiTransportTlsConfigMode.addCommandClass( TcpKeepaliveCommand )
RpkiCacheConfigMode.addCommandClass( RpkiRoaTableCmd )
BasicCli.addShowCommandClass( RpkiShowRoaTableSummary )
BasicCli.addShowCommandClass( RpkiShowRoas )
BasicCli.addShowCommandClass( RpkiShowRoaSummary )
BasicCli.addShowCommandClass( RpkiShowCache )
BasicCli.addShowCommandClass( RpkiShowCachePduCounters )
BasicCli.addShowCommandClass( RpkiShowCachePduCountersErrors )
BasicCli.EnableMode.addCommandClass( RpkiClearCacheCmd )
BasicCli.EnableMode.addCommandClass( RpkiClearCacheCounterCmd )
RouterBgpSharedModelet.addCommandClass( BgpNeighborRpkiRecvOriginValidation )
RouterBgpSharedModelet.addCommandClass( BgpNeighborRpkiCommunitySend )

def Plugin( entityManager ):
   gv.rpkiConfig = ConfigMount.mount( entityManager, 'routing/rpki/cache/config',
                                   'Rpki::CacheConfigDir', 'w' )
   gv.rpkiStatus = LazyMount.mount( entityManager,
                                 Cell.path( 'routing/rpki/cache/status' ),
                                 'Rpki::CacheStatusDir', 'r' )
   gv.l3Config = LazyMount.mount( entityManager, 'l3/config',
                               'L3::Config', 'r' )
   gv.sslConfig = LazyMount.mount( entityManager, 'mgmt/security/ssl/config',
                                'Mgmt::Security::Ssl::Config', 'r' )
   gv.roaTableStatusDir = LazyMount.mount( entityManager,
                                        Cell.path( 'routing/rpki/roaTable/status' ),
                                        'Rpki::RoaTableStatusDir', 'r' )
