# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys

from AgentDirectory import agentIsRunning
import BasicCli
import CliCommand
from CliMatcher import (
      DynamicNameMatcher,
      KeywordMatcher,
   )
import CliParser
from CliPlugin import RsvpLerModel
import CliPlugin.TechSupportCli
from IpLibConsts import DEFAULT_VRF
import LazyMount
import SharkLazyMount
import ShowCommand
import Tac
from TypeFuture import TacLazyType

# Type Alias
FeatureId = TacLazyType( 'FlexCounters::FeatureId' )
FeatureState = TacLazyType( 'Ale::FlexCounter::ConfigState' )

RsvpLerLspGroupState = TacLazyType( "Rsvp::RsvpLerLspGroupState" )
RsvpLerLspRequestConfigId = TacLazyType( "Rsvp::RsvpLerLspRequestConfigId" )
RsvpLerPathSpecId = TacLazyType( "Rsvp::RsvpLerPathSpecId" )
RsvpLerTunnelState = TacLazyType( "Rsvp::RsvpLerTunnelState" )

# Tokens
teMatcherForShow = KeywordMatcher( "traffic-engineering",
      helpdesc="Traffic Engineering related information" )
rsvpMatcherForShow = KeywordMatcher( "rsvp", helpdesc="Show RSVP information" )

# Globals
fcFeatureConfigDir = None
mplsRoutingConfig = None
teConfig = None
routingVrfInfoDir = None
rsvpCliConfig = None
rsvpLerCliConfig = None
rsvpLerConfig = None
rsvpLerStatus = None
rsvpLerSharkStatus = None
rsvpLerTunnelHistory = None
rsvpLerSubTunnelHistory = None
sysname = None
mplsHwCapability = None
routingHardwareRouteStatus = None

# RsvpLer is supported only on mpls supported platforms that have hfec enabled
def rsvpLerSupportedGuard( mode, token ):
   if ( mplsHwCapability.mplsSupported and
        routingHardwareRouteStatus.hierarchicalFecsEnabled ):
      return None
   else:
      return CliParser.guardNotThisPlatform

# Helpers
def showCmdWarnings( mode ):
   """Warn if any relevant features are not enabled / running"""
   routingInfo = routingVrfInfoDir.get( DEFAULT_VRF )
   if not rsvpCliConfig.enabled:
      mode.addWarning( "RSVP is not enabled" )
   if not mplsRoutingConfig.mplsRouting:
      mode.addWarning( "MPLS routing is not enabled" )
   if not ( routingInfo and routingInfo.routing ):
      mode.addWarning( "IP routing is not enabled" )
   if not teConfig.enabled or teConfig.routerId == teConfig.routerIdDefault:
      mode.addWarning( "Traffic engineering is not enabled" )
   if not rsvpTunnelAgentRunning():
      mode.addWarning( "Agent RSVP is not running" )
   if not mplsAgentRunning():
      mode.addWarning( "Agent MPLS is not running" )

def rsvpTunnelAgentRunning():
   return agentIsRunning( sysname, 'RsvpTunnel' )

def mplsAgentRunning():
   return agentIsRunning( sysname, 'Mpls' )

def tunnelNameList( mode ):
   if not rsvpLerStatus.tunnelSpecColl:
      return []
   return [ tunnelSpecId.tunnelName for tunnelSpecId in
            rsvpLerStatus.tunnelSpecColl.tunnelSpec ]

def showTeRsvpTunnel( mode, args ):
   showCmdWarnings( mode )

   filters = args.get( 'FILTERS' )
   summary = "summary" in args
   lspDetails = "lsp" in args
   detail = "detail" in args
   history = "history" in args
   subTunnel = "sub-tunnel" in args

   tacTunnelFilter = Tac.Value( "Rsvp::Cli::ShowRsvpLerTunnelFilter" )

   if filters:
      # filters [('tunnelFilter', ('tunnelName', 'myTunnel')),
      #         ]
      for filterType, arg in filters:
         if filterType == 'tunnelFilter':
            filterAttr, filterValue = arg
            if filterAttr in [ 'tunnelName' ]:
               setattr( tacTunnelFilter, filterAttr, filterValue )

   socket = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   LazyMount.force( rsvpLerCliConfig )
   LazyMount.force( rsvpLerStatus )
   rsvpSharkStatusEntity = SharkLazyMount.force( rsvpLerSharkStatus )
   renderer = Tac.newInstance( "Rsvp::Cli::ShowRsvpLerTunnel", rsvpCliConfig,
                               rsvpLerCliConfig, rsvpLerStatus,
                               rsvpSharkStatusEntity )
   renderer.render( socket, fmt, tacTunnelFilter, summary, lspDetails, detail,
                    history, subTunnel )
   return RsvpLerModel.RsvpLerTunnels

tunnelFilterSharedObj = object()

class RsvpLerTunnelFilterArg( CliCommand.CliExpression ):
   expression = ( '{ '
                  '( name NAME ) '
                  '}' )
   data = {
      'name': CliCommand.singleKeyword( 'name', helpdesc='Filter by tunnel name',
         sharedMatchObj=tunnelFilterSharedObj ),
      'NAME': DynamicNameMatcher( tunnelNameList, 'Tunnel name',
         pattern=CliParser.namePattern ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      filters = []
      if 'name' in args and 'NAME' in args:
         filters.append( ( 'tunnelFilter', ( 'tunnelName',
                                             args[ 'NAME' ][ 0 ] ) ) )

      if filters:
         args[ 'FILTERS' ] = filters

class TeRsvpTunnelCmd( ShowCommand.ShowCliCommandClass ):
   syntax = (
      '''show traffic-engineering rsvp tunnel [ FILTERS ]
         [ ( [ sub-tunnel ] summary ) | lsp | { detail | history } ] ''' )
   data = {
         "traffic-engineering": teMatcherForShow,
         "rsvp": rsvpMatcherForShow,
         "tunnel": "RSVP tunnel information",
         "FILTERS": RsvpLerTunnelFilterArg,
         "lsp": "Show LSP details",
         "sub-tunnel": "RSVP sub tunnel information for the tunnel",
         "summary": "Show summarized information",
         "detail": CliCommand.singleKeyword(
            "detail", "More comprehensive output" ),
         "history": CliCommand.singleKeyword(
            "history", "Show history information" ),
         }
   handler = showTeRsvpTunnel
   cliModel = RsvpLerModel.RsvpLerTunnels

BasicCli.addShowCommandClass( TeRsvpTunnelCmd )

#------------------------------------------------------------------------------------
# show traffic-engineering rsvp
#------------------------------------------------------------------------------------
def showTeRsvp( mode, args ):
   showCmdWarnings( mode )

   socket = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   LazyMount.force( rsvpLerConfig )
   LazyMount.force( rsvpLerCliConfig )
   LazyMount.force( rsvpLerStatus )
   rsvpSharkStatusEntity = SharkLazyMount.force( rsvpLerSharkStatus )

   renderer = Tac.newInstance( "Rsvp::Cli::ShowRsvpLer", rsvpLerConfig,
                               rsvpLerCliConfig, rsvpLerStatus,
                               rsvpSharkStatusEntity, fcFeatureConfigDir )
   renderer.render( socket, fmt )
   return RsvpLerModel.RsvpLerState

class TeRsvpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( "show traffic-engineering rsvp" )
   data = {
         "traffic-engineering": teMatcherForShow,
         "rsvp": rsvpMatcherForShow,
         }
   handler = showTeRsvp
   cliModel = RsvpLerModel.RsvpLerState

BasicCli.addShowCommandClass( TeRsvpCmd )

# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmd(
   '2020-04-02 15:00:00',
   cmds=[ 'show traffic-engineering rsvp',
          'show traffic-engineering rsvp tunnel detail',
          'show traffic-engineering rsvp tunnel lsp',
          'show traffic-engineering rsvp tunnel history' ],
   cmdsGuard=lambda: mplsHwCapability.mplsSupported )

def Plugin( entityManager ):
   global sysname
   sysname = entityManager.sysname()

   global routingVrfInfoDir
   routingVrfInfoDir = LazyMount.mount( entityManager,
                                        "routing/vrf/routingInfo/status",
                                        "Tac::Dir", "ri" )

   global rsvpLerSharkStatus
   autoUnmount = True
   RsvpLerSharkStatus = Tac.Type( "Rsvp::RsvpLerSharkStatus" )
   rsvpLerSharkStatus = SharkLazyMount.mount( entityManager,
      RsvpLerSharkStatus.mountPath,
      "Rsvp::RsvpLerSharkStatus",
      SharkLazyMount.mountInfo( 'shadow' ),
      autoUnmount )

   global rsvpLerStatus
   RsvpLerSysdbStatus = Tac.Type( "Rsvp::RsvpLerSysdbStatus" )
   rsvpLerStatus = LazyMount.mount( entityManager, RsvpLerSysdbStatus.mountPath,
                                    "Rsvp::RsvpLerSysdbStatus", "rS" )

   global mplsRoutingConfig
   mplsRoutingConfig = LazyMount.mount( entityManager,
                                        "routing/mpls/config",
                                        "Mpls::Config", "r" )

   global teConfig
   teConfig = LazyMount.mount( entityManager,
                               "te/config",
                               "TrafficEngineering::Config", 'r' )

   global rsvpCliConfig
   RsvpCliConfig = Tac.Type( "Rsvp::RsvpCliConfig" )
   rsvpCliConfig = LazyMount.mount( entityManager,
                                    RsvpCliConfig.mountPath,
                                    "Rsvp::RsvpCliConfig", 'r' )

   global mplsHwCapability
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability", 'r' )

   global routingHardwareRouteStatus
   routingHardwareRouteStatus = LazyMount.mount( entityManager,
                                                 "routing/hardware/route/status",
                                                 "Routing::Hardware::RouteStatus",
                                                 'r' )

   global rsvpLerCliConfig
   RsvpLerCliConfig = Tac.Type( "Rsvp::RsvpLerCliConfig" )
   rsvpLerCliConfig = LazyMount.mount( entityManager,
                                       RsvpLerCliConfig.mountPath,
                                       "Rsvp::RsvpLerCliConfig", 'r' )

   global rsvpLerConfig
   RsvpLerConfig = Tac.Type( "Rsvp::RsvpLerConfig" )
   rsvpLerConfig = LazyMount.mount( entityManager,
                                    RsvpLerConfig.mountPath,
                                    "Rsvp::RsvpLerConfig", 'r' )

   global fcFeatureConfigDir
   fcFeatureConfigDir = LazyMount.mount( entityManager,
                                           'flexCounter/featureConfigDir/cliAgent',
                                           'Ale::FlexCounter::FeatureConfigDir',
                                           'r' )
