# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
import prettytable

from CliModel import (
   Bool,
   DeferredModel,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
import Arnet
from ArnetModel import (
      IpGenericAddress,
      IpGenericPrefix,
   )
from IntfModels import Interface
import Tac

# Aliasing types
SessionId = Int
LspId = Int
NeighborAddress = IpGenericAddress
MplsLabel = Int
BwPriority = Tac.Type( "TrafficEngineering::BwPriority" )

def roundBw( number, digits ):
   r = round( number, digits )
   if number > 0.0 and r == 0.0:
      # round up if a non-zero number is be rounded down to zero,
      # just so customers are not confused seeing zero bandwidth
      # when they know there are active reservations in the systems
      r = 10**( -digits )
   return r

def createIndent( indentLevel ):
   ''' Create 3 space indent.
   '''
   return "   " * indentLevel

class NeighborLspInfo( Model ):
   sessionId = SessionId( help="Session ID" )
   lspId = LspId( help="LSP ID" )
   sessionName = Str( help="Session name", optional=True )

class NeighborP2mpLspInfo( Model ):
   extendedTunnelId = IpGenericAddress( help="Extended tunnel ID" )
   lspId = LspId( help="LSP ID" )
   sessionName = Str( help="Session name", optional=True )

class NeighborBypassInfo( Model ):
   bypassTunnel = Enum( help="Bypass tunnel state",
                        values=( 'available', 'inUse', 'notAvailable',
                                 'notRequested' ) )
   bypassTunnelName = Str( help="Bypass tunnel name", optional=True )
   bypassNextHop = IpGenericAddress( help="Bypass tunnel next hop", optional=True )
   bypassLabel = MplsLabel( help="Bypass tunnel label", optional=True )

class RsvpNeighbor( Model ):
   upstreams = List( help="Upstream session LSP information",
                     valueType=NeighborLspInfo )
   downstreams = List( help="Downstream session LSP information",
                      valueType=NeighborLspInfo )
   backupUpstreams = List( help="Backup upstream session LSP information",
                           valueType=NeighborLspInfo )
   backupDownstreams = List( help="Backup downstream session LSP information",
                             valueType=NeighborLspInfo )

   p2mpUpstreams = List( help="Upstream P2MP session LSP information",
                         valueType=NeighborP2mpLspInfo )
   p2mpDownstreams = List( help="Downstream P2MP session LSP information",
                           valueType=NeighborP2mpLspInfo )
   p2mpBackupUpstreams = List(
      help="Backup upstream P2MP session LSP information",
      valueType=NeighborP2mpLspInfo )
   p2mpBackupDownstreams = List(
      help="Backup downstream P2MP session LSP information",
      valueType=NeighborP2mpLspInfo )

   neighborCreationTime = Float( help="UTC time when neighbor was created",
                                 optional=True )
   authentication = Enum( help="Cryptographic authentication",
                          values=( 'disabled', 'MD5' ),
                          optional=True )

   # Not set in summary mode
   lastHelloReceived = Float(
         optional=True,
         help="UTC time when the last RSVP hello message was received" )
   lastHelloSent = Float(
         optional=True,
         help="UTC time when the last RSVP hello message was sent" )

   # Bypass tunnel with link protection
   bypassTunnel = Enum( help="Bypass tunnel state",
                        values=( 'available', 'inUse', 'notAvailable',
                                 'notRequested' ) )
   bypassTunnelName = Str( help="Bypass tunnel name", optional=True )
   bypassNextHop = IpGenericAddress( help="Bypass tunnel next hop", optional=True )
   bypassLabel = MplsLabel( help="Bypass tunnel label", optional=True )

   # Bypass tunnels with node protection
   bypassInfoNode = Dict( help="A dictionary of bypass tunnel state for node "
                               "protection keyed by tunnel destination IP address",
                          keyType=IpGenericAddress, valueType=NeighborBypassInfo,
                          optional=True )

   # Graceful restart
   operGrState = Enum( help='Graceful restart state',
                       values=( 'unknown', 'unsupported', 'inactive', 'restart',
                                'recovery' ) )
   restartTime = Float( help='Graceful restart restart time advertised by the '
                             'neighbor in seconds', optional=True )
   recoveryTime = Float( help='Graceful restart recovery time advertised by the '
                              'neighbor in seconds', optional=True )

class RsvpNeighbors( DeferredModel ):
   '''Model for "show mpls rsvp neighbor" '''
   neighbors = Dict( help="Neighbor information",
                     keyType=NeighborAddress,
                     valueType=RsvpNeighbor )

class RsvpNeighborStatistics( Model ):
   upstreamSessionCount = Int( help="Number of upstream sessions" )
   upstreamLspCount = Int( help="Number of upstream LSPs" )
   downstreamSessionCount = Int( help="Number of downstream sessions" )
   downstreamLspCount = Int( help="Number of downstream LSPs" )
   backupUpstreamSessionCount = Int( help="Number of backup upstream sessions" )
   backupUpstreamLspCount = Int( help="Number of backup upstream LSPs" )
   backupDownstreamSessionCount = Int( help="Number of backup downstream sessions" )
   backupDownstreamLspCount = Int( help="Number of backup downstream LSPs" )

class RsvpNeighborSummary( DeferredModel ):
   '''Model for "show mpls rsvp neighbor summary" '''
   neighborSummary = Dict( help="Neighbor summary",
                           keyType=NeighborAddress,
                           valueType=RsvpNeighborStatistics )

class LspCount( Model ):
   total = Int( help="Number of LSPs" )
   operational = Int( help="Number of operational LSPs" )
   usingBypassTunnels = Int( help="Number of LSP using bypass tunnels" )
   ingress = Int( help="Number of ingress LSPs" )
   transit = Int( help="Number of transit LSPs" )
   egress = Int( help="Number of egress LSPs" )

class P2mpLspCount( Model ):
   total = Int( help="Number of P2MP LSPs", default=0 )
   operational = Int( help="Number of operational P2MP LSPs", default=0 )
   usingBypassTunnels = Int( help="Number of LSP using bypass tunnels", default=0 )
   ingress = Int( help="Number of ingress P2MP LSPs", default=0 )
   transit = Int( help="Number of transit P2MP LSPs", default=0 )
   egress = Int( help="Number of egress P2MP LSPs", default=0 )
   bud = Int( help="Number of bud P2MP LSPs", default=0 )

class P2mpSubLspCount( Model ):
   total = Int( help="Number of sub-LSPs", default=0 )
   operational = Int( help="Number of operational sub-LSPs", default=0 )
   usingBypassTunnels = Int( help="Number of LSP using bypass tunnels", default=0 )
   ingress = Int( help="Number of ingress sub-LSPs", default=0 )
   transit = Int( help="Number of transit sub-LSPs", default=0 )
   egress = Int( help="Number of egress sub-LSPs", default=0 )

class HitlessRestart( Model ):
   recovery = Float( help='Hitless Restart recovery value in seconds' )

class GrHelper( Model ):
   maximumAcceptedRestart = Float(
         help="Maximum accepted restart value from neighbors in seconds",
         optional=True )
   maximumAcceptedRecovery = Float(
         help="Maximum accepted recovery value from neighbors in seconds",
         optional=True )

class GrSpeaker( Model ):
   restart = Float( help='Graceful Restart restart value in seconds' )
   recovery = Float( help='Graceful Restart recovery value in seconds' )

class RsvpState( Model ):
   '''Model for 'show mpls rsvp' '''
   adminState = Enum( help="Is RSVP enabled by config",
                      values=( "enabled", "disabled" ) )
   operationalState = Enum( help="Operational state of RSVP",
                            values=( "up", "down" ) )
   refreshInterval = Float( help="Time interval in seconds between "
                                 "RSVP neighbor state refresh messages" )
   refreshReduction = Bool( help="Is bundling of RSVP neighbor messages enabled" )
   helloEnabled = Bool( help="Are hello messages enabled" )
   helloInterval = Float( help="Time interval in seconds between "
                               "RSVP hello messages. "
                               "Set when helloEnabled is True",
                          optional=True )
   helloMultiplier = Float( help="Number of missed hellos after which "
                                 "the neighbor is expired. "
                                 "Set when helloEnabled is True",
                            optional=True )
   fastReroute = Bool( help="Is fast re-route enabled" )
   fastRerouteMode = Enum( help="Fast re-route mode",
                           values=( "none", "linkProtection", "nodeProtection" ) )
   hierarchicalFecsEnabled = Bool( help="Are hierarchical FECs enabled" )
   reversionModeConfig = Enum( help="Configured FRR reversion mode",
                               values=( "global", "local" ), optional=True )
   reversionMode = Enum( help="Operational FRR reversion mode",
                         values=( "global", "local" ), optional=True )
   bypassOptimizationInterval = Int( help="Time interval in seconds between FRR "
                                          "bypass tunnel re-optimization attempts" )
   authentication = Enum( help="Cryptographic authentication",
                          values=( 'disabled', 'MD5' ) )
   srlgMode = Enum( help="SRLG mode",
                    values=( "srlgNone", "srlgLoose", "srlgStrict" ) )
   preemptionPeriod = Float( help="Preemption timer value in seconds" )
   mtuSignalingEnabled = Bool( help='Is MTU signaling enabled' )

   # P2P specific.
   sessionCount = Int( help="Number of sessions" )
   lspCount = Submodel( help="LSP related counts", valueType=LspCount )
   ingressCount = Int( help="Number of ingress sessions" )
   transitCount = Int( help="Number of transit sessions" )
   egressCount = Int( help="Number of egress sessions" )

   # P2MP specific.
   p2mpEnabled = Bool( help='Is P2MP enabled' )
   p2mpSessionCount = Int( help="Number of P2MP sessions", optional=True )
   p2mpLspCount = Submodel( help="P2MP LSP related counts",
                            valueType=P2mpLspCount, optional=True )
   p2mpSubLspCount = Submodel( help="P2MP sub-LSP related counts",
                               valueType=P2mpSubLspCount, optional=True )

   bypassTunnelCount = Int( help="Number of bypass tunnels" )
   neighborCount = Int( help="Number of neighbors" )
   interfaceCount = Int( help="Number of interfaces" )
   labelLocalTerminationMode = Enum( help='Label local termination mode',
                                     values=( "implicitNull", "explicitNull" ),
                                     optional=True )
   hitlessRestart = Submodel( help='Hitless restart configuration',
                              valueType=HitlessRestart, optional=True )
   grHelper = Submodel( help='Graceful restart helper configuration',
                        valueType=GrHelper, optional=True )
   grSpeaker = Submodel( help="Graceful restart speaker configuration",
                         valueType=GrSpeaker, optional=True )

   def render( self ):
      print( "Administrative state: %s" % self.adminState )
      print( "Operational state: %s" % self.operationalState )
      print( "Refresh interval: %d seconds" % self.refreshInterval )
      print( "Refresh reduction: %s" % (
            "enabled" if self.refreshReduction else "disabled" ) )
      print( "Hello messages: %s" % (
            'enabled' if self.helloEnabled else 'disabled' ) )
      if self.helloEnabled:
         print( "   Hello interval: %d seconds" % self.helloInterval )
         print( "   Hello multiplier: %d" % self.helloMultiplier )
      print( "Fast Re-Route: %s" % (
            "enabled" if self.fastReroute else "disabled" ) )
      frrModeStr = "none"
      if self.fastRerouteMode == 'nodeProtection':
         frrModeStr = "node protection"
      elif self.fastRerouteMode == 'linkProtection':
         frrModeStr = "link protection"
      print( "   Mode: %s" % frrModeStr )
      print( "   Hierarchical FECs: %s" % (
            "enabled" if self.hierarchicalFecsEnabled else "disabled" ) )
      # pylint: disable-next=singleton-comparison
      if self.fastReroute and self.reversionMode != None:
         if self.reversionMode == 'global' and self.reversionModeConfig == 'local':
            reversionModeStr = "global (implied by node protection)"
         else:
            reversionModeStr = self.reversionMode
         print( "   Reversion: %s" % reversionModeStr )
      print( "   Bypass tunnel optimization interval: %s seconds"
             % self.bypassOptimizationInterval )
      if self.authentication == 'disabled':
         authStr = 'disabled'
      else:
         authStr = "enabled (%s)" % self.authentication
      print( "Cryptographic authentication: %s" % authStr )
      d = {
         'srlgNone' : 'none',
         'srlgLoose' : 'loose',
         'srlgStrict' : 'strict',
      }
      if self.srlgMode in d:
         print( "SRLG mode: %s" % d[ self.srlgMode ] )
      if self.preemptionPeriod == 0:
         print( "Soft preemption: disabled" )
      else:
         print( "Soft preemption: enabled" )
         print( "   Preemption timer: %d seconds" % self.preemptionPeriod )
      print( 'MTU signaling: %s' % (
            'enabled' if self.mtuSignalingEnabled else 'disabled' ) )

      d = {
         'explicitNull' : 'explicit null',
         'implicitNull' : 'implicit null',
      }
      if self.labelLocalTerminationMode in d:
         print( "Label type for local termination: %s" % (
               d[ self.labelLocalTerminationMode ] ) )
      # Hitless restart state
      print( 'Hitless restart:',
             'enabled' if self.hitlessRestart else 'disabled' )
      if self.hitlessRestart:
         print( 'Hitless restart recovery timer:',
                int( self.hitlessRestart.recovery ),
                'seconds' )
      # Graceful restart state
      if self.grSpeaker:
         print( "Graceful restart: enabled" )
      elif self.grHelper:
         print( "Graceful restart: enabled (helper only)" )
      else:
         print( "Graceful restart: disabled" )
      # Graceful restart speaker values
      if self.grSpeaker:
         print( "Graceful restart restart timer: %d seconds"
                % self.grSpeaker.restart )
         print( "Graceful restart recovery timer: %d seconds"
                % self.grSpeaker.recovery )
      # Graceful restart helper values
      if self.grHelper:
         if self.grHelper.maximumAcceptedRestart:
            print( 'Graceful restart maximum accepted neighbor restart timer: '
                   '%d seconds' % self.grHelper.maximumAcceptedRestart )
         else:
            print( 'Graceful restart maximum accepted neighbor restart timer: none' )
         if self.grHelper.maximumAcceptedRecovery:
            print( 'Graceful restart maximum accepted neighbor recovery timer: '
                   '%d seconds' % self.grHelper.maximumAcceptedRecovery )
         else:
            print( 'Graceful restart maximum accepted neighbor recovery timer: '
                   'none' )

      countInfoIndent = createIndent( 1 )
      countSubInfoIndent = createIndent( 2 )

      print( 'P2P' )
      print( countInfoIndent + f"Number of sessions: { self.sessionCount }" )
      print( countSubInfoIndent + 'Ingress/Transit/Egress: '
             f'{ self.ingressCount }/{ self.transitCount }/{ self.egressCount }' )
      print( countInfoIndent + f"Number of LSPs: { self.lspCount.total }" )
      print( countSubInfoIndent + f"Operational: { self.lspCount.operational }" )
      print( countSubInfoIndent + 'Ingress/Transit/Egress: '
             f'{ self.lspCount.ingress }/'
             f'{ self.lspCount.transit }/'
             f'{ self.lspCount.egress }' )
      print( countSubInfoIndent +
             f"Currently using bypass tunnels: { self.lspCount.usingBypassTunnels }"
            )

      if not self.p2mpEnabled:
         print( 'P2MP: disabled' )
      else:
         print( 'P2MP' )
         print( countInfoIndent +
                f'Number of sessions: { self.p2mpSessionCount }' )

         print( countInfoIndent + f'Number of LSPs: { self.p2mpLspCount.total }' )
         print( countSubInfoIndent +
                f'Operational: { self.p2mpLspCount.operational }' )
         print( countSubInfoIndent + f'Ingress/Transit/Egress/Bud: '
                f'{ self.p2mpLspCount.ingress }/'
                f'{ self.p2mpLspCount.transit }/'
                f'{ self.p2mpLspCount.egress }/'
                f'{ self.p2mpLspCount.bud }' )
         print( countSubInfoIndent + f'Currently using bypass tunnels: '
                f'{ self.p2mpLspCount.usingBypassTunnels }' )

         print( countInfoIndent + f'Number of sub-LSPs: '
                f'{ self.p2mpSubLspCount.total }' )
         print( countSubInfoIndent +
                f'Operational: { self.p2mpSubLspCount.operational }' )
         print( countSubInfoIndent + f'Ingress/Transit/Egress: '
                f'{ self.p2mpSubLspCount.ingress }/'
                f'{ self.p2mpSubLspCount.transit }/'
                f'{ self.p2mpSubLspCount.egress }' )
         print( countSubInfoIndent + f'Currently using bypass tunnels: '
                f'{ self.p2mpSubLspCount.usingBypassTunnels }' )

      print( f"Number of bypass tunnels: { self.bypassTunnelCount }" )
      print( f"Number of neighbors: { self.neighborCount }" )
      print( f"Number of interfaces: { self.interfaceCount }" )

class LspNeighborInfo( Model ):
   address = NeighborAddress( help="Neighbor address" )

   # Not set in summary mode
   localInterface = Interface( help="Local interface", optional=True )
   lastRefreshReceived = Float(
         optional=True,
         help="UTC time when the last neighbor refresh was received"
         )
   lastRefreshSent = Float(
         optional=True,
         help="UTC time when the last neighbor refresh was sent" )

class RsvpEroHop( Model ):
   # Not set if hop is AS
   prefix = IpGenericPrefix( help="ERO hop prefix", optional=True )
   loose = Bool( help="ERO hop is a loose hop" )
   # Not set if hop is IP
   asNumber = Int( help="ERO hop AS number", optional=True )

class RsvpRroHop( Model ):
   address = IpGenericAddress( help="RRO hop address" )
   label = MplsLabel( help="RRO hop label", optional=True )
   nodeId = Bool( help="RRO hop is a node ID" )
   frrMode = Enum( help="RRO hop's fast re-route mode",
                   values=( "none", "linkProtection", "nodeProtection" ) )
   frrInUse = Bool( help="RRO hop actively uses fast re-route local protection" )

class RsvpHistoryEntry( Model ):
   timestamp = Float( help='Timestamp of the event' )
   event = Str( help="History event" )

class RsvpHistory( Model ):
   history = List( help="Recent history events", valueType=RsvpHistoryEntry )

class Lsp( Model ):
   __revision__ = 2
   state = Enum( help="Operational state",
                 values=( 'up', 'down', 'pathOnly' ) )
   # Not set in summary mode and on transit routers
   lspType = Enum( help="LSP purpose",
                   values=( 'primary', 'secondary', 'bypass' ), optional=True )
   pathState = Enum( help="Path state",
                     values=( 'up', 'down', 'receivedOnly', 'sentOnly' ) )
   resvState = Enum( help="Reservation state",
                     values=( 'up', 'down', 'receivedOnly', 'sentOnly' ) )

   lspId = Int( help="LSP ID", optional=True )

   # Not set in summary mode
   lspCreationTime = Float( help="UTC time when LSP was created", optional=True )

   # Not set in summary mode
   sourceAddress = IpGenericAddress( help="Source address", optional=True )

   bypassTunnel = Enum( help="Bypass tunnel state",
                        values=( 'available', 'inUse', 'notAvailable',
                                 'notRequested', 'egress' ) )

   # Not set in summary mode
   bypassTunnelName = Str( help="Bypass tunnel name", optional=True )
   bypassNextHop = IpGenericAddress( help="Bypass tunnel next hop", optional=True )
   bypassLabel = MplsLabel( help="Bypass tunnel label", optional=True )

   # Not set when the session is in ingressRole or in summary mode
   upstreamNeighbor = Submodel( help="Upstream neighbor information",
                                valueType=LspNeighborInfo,
                                optional=True )
   backupUpstreamNeighborAddress = NeighborAddress( optional=True,
         help="Upstream backup neighbor address" )

   # Not set when the session is in egressRole or in summary mode
   downstreamNeighbor = Submodel( help="Downstream neighbor information",
                                  valueType=LspNeighborInfo,
                                  optional=True )
   backupDownstreamNeighborAddress = NeighborAddress( optional=True,
         help="Downstream backup neighbor address" )

   # Not set when the session is in ingressRole or in summary mode
   localLabel = MplsLabel( help="Local MPLS label",
                           optional=True )
   # Not set when the session is in egressRole or in summary mode
   downstreamLabel = MplsLabel( help="Downstream MPLS label",
                                optional=True )
   # Not set in ingressRole, egressRole or in summary mode
   unprogrammed = Bool( help="LSP is unprogrammed in hardware",
                              optional=True )

   sessionName = Str( help="Head end session name", optional=True )

   # Not set in summary mode
   path = List( valueType=str, help="Explicit path", optional=True )

   bypassLastHopLabel = MplsLabel( help="Merge point label", optional=True )
   frrRequestedMode = Enum( help="Fast re-route mode requested",
                            values=( "none", "linkProtection", "nodeProtection" ),
                            optional=True )
   frrOperationalMode = Enum( help="Fast re-route mode operational",
                              values=( "none", "linkProtection", "nodeProtection" ),
                              optional=True )
   bandwidth = Float( help="Requested bandwidth in bps", optional=True )
   bandwidthState = Enum(
      help="Current status of bandwidth reservation",
      values=( "unknown", "pending", "failedAdmission", "failedPreemption",
               "preempted", "reserved" ),
      optional=True )
   bandwidthInterface = Interface( help="Bandwidth reservation interface",
                                   optional=True )
   # Only set in detail mode
   setupPriority = Int( help="Setup priority", optional=True )
   holdPriority = Int( help="Hold priority", optional=True )
   softPreemptionRequested = Bool( help="Soft preemption has been requested",
                                   optional=True )
   mtuSignalingEnabled = Bool( help="MTU signaling is enabled for this LSP",
                               optional=True )
   inMtu = Int( help="MTU received in the incoming path message", optional=True )
   outMtu = Int( help="MTU sent in the outgoing path message", optional=True )
   ero = List( help="Explicit route hops", valueType=RsvpEroHop, optional=True )
   usRro = List( help="Record route hops from upstream",
                 valueType=RsvpRroHop,
                 optional=True )
   dsRro = List( help="Record route hops from downstream",
                 valueType=RsvpRroHop,
                 optional=True )
   # Only set in history mode
   lspHistory = Submodel( valueType=RsvpHistory, help="SP history events",
                          optional=True )

class RsvpSession( Model ):
   __revision__ = 2
   destination = IpGenericAddress( help="Destination address" )
   # Not set in summary mode
   tunnelId = Int( help="Tunnel ID", optional=True )
   # Not set in summary mode
   extendedTunnelId = IpGenericAddress( help="Extended tunnel ID",
                                        optional=True )
   # Not set in summary mode
   state = Enum( help="Session operational state",
                 values=( 'up', 'down' ),
                 optional=True )
   sessionCreationTime = Float( help="UTC time when session was created",
                                optional=True )
   sessionRole = Enum( help="Role for Session",
                       values=( 'ingress', 'egress', 'transit', 'unknown' ) )
   lsps = Dict( help="Table of LSP information keyed by LSP ID",
                keyType=int, valueType=Lsp )
   # Only set in history mode
   sessionHistory = Submodel( valueType=RsvpHistory,
                              help="Session history events",
                              optional=True )

class RsvpP2mpUsNeighbor( Model ):
   localInterface = Interface( help="Local interface", optional=True )
   localLabel = MplsLabel( help="Local MPLS label", optional=True )
   backupNeighborAddress = NeighborAddress( help="Backup neighbor address",
                                            optional=True )
   backupLocalInterface = Interface( help="Backup local interface", optional=True )
   inMtu = Int( help="MTU received in the incoming path message", optional=True )

class RsvpP2mpDsNeighbor( Model ):
   localInterface = Interface( help="Local interface", optional=True )
   downstreamLabel = MplsLabel( help="Downstream MPLS label", optional=True )
   bypassTunnel = Enum( help="Bypass tunnel state",
                        values=( "inUse", "available", "notAvailable",
                                 "notRequested" ),
                        default="notRequested" )
   backupNeighborAddress = NeighborAddress( help="Backup neighbor address",
                                            optional=True )
   backupLocalInterface = Interface( help="Backup local interface", optional=True )
   bypassTunnelName = Str( help="Bypass tunnel name", optional=True )
   bypassNextHop = IpGenericAddress( help="Bypass tunnel next hop", optional=True )
   bypassLabel = MplsLabel( help="Bypass tunnel label", optional=True )
   bypassLastHopLabel = MplsLabel( help="Merge point label", optional=True )
   frrRequestedMode = Enum( help="Fast re-route mode requested",
                            values=( "none", "linkProtection", "nodeProtection" ),
                            default="none" )
   frrOperationalMode = Enum( help="Fast re-route mode operational",
                              values=( "none", "linkProtection", "nodeProtection" ),
                              default="none" )
   bandwidth = Float( help="Requested bandwidth in bps", optional=True )
   setupPriority = Int( help="Setup priority", optional=True )
   holdPriority = Int( help="Hold priority", optional=True )
   softPreemptionRequested = Bool( help="Soft preemption has been requested",
                                   optional=True )
   bandwidthState = Enum(
      help="Current status of bandwidth reservation",
      values=( "unknown", "pending", "failedAdmission", "failedPreemption",
               "preempted", "reserved" ),
      optional=True )
   outMtu = Int( help="MTU sent in the outgoing path message", optional=True )

class RsvpP2mpSubLsp( Model ):
   role = Enum( help="Role for sub-LSP",
                values=( 'unknown', 'ingress', 'egress', 'transit' ),
                default='unknown' )

   state = Enum( help="Sub-LSP operational state",
                 values=( 'up', 'down', 'pathOnly' ), default='down' )
   pathState = Enum( help="Path state",
                     values=( 'up', 'down', 'receivedOnly', 'sentOnly' ),
                     default='down' )
   resvState = Enum( help="Reservation state",
                     values=( 'up', 'down', 'receivedOnly', 'sentOnly' ),
                     default='down' )

   upstreamNeighbor = IpGenericAddress( help="Upstream neighbor address",
                                        optional=True )
   downstreamNeighbor = IpGenericAddress( help="Downstream neighbor address",
                                          optional=True )

   ero = List( help="Explicit route hops", valueType=RsvpEroHop )
   usRro = List( help="Record route hops from upstream", valueType=RsvpRroHop )
   dsRro = List( help="Record route hops from downstream", valueType=RsvpRroHop )

   lastUsRefreshReceived = Float(
      help="UTC time when the last upstream neighbor refresh was received",
      optional=True )
   lastDsRefreshReceived = Float(
      help="UTC time when the last downstream neighbor refresh was received",
      optional=True )
   lastUsRefreshSent = Float(
      help="UTC time when the last upstream neighbor refresh was sent",
      optional=True )
   lastDsRefreshSent = Float(
      help="UTC time when the last downstream neighbor refresh was sent",
      optional=True )

class RsvpP2mpSubGroup( Model ):
   sessionName = Str( help="Sub-group session name", optional=True )
   mtuSignalingEnabled = Bool( help="MTU signaling is enabled for this sub-group",
                               default=False )
   upstreamNeighbors = Dict(
      help="Table of upstream neighbor information keyed by neighbor IP address",
      keyType=IpGenericAddress, valueType=RsvpP2mpUsNeighbor )
   downstreamNeighbors = Dict(
      help="Table of downstream neighbor information keyed by neighbor IP address",
      keyType=IpGenericAddress, valueType=RsvpP2mpDsNeighbor )
   subLsps = Dict(
      help="Table of sub-LSP information keyed by destination IP address",
      keyType=IpGenericAddress, valueType=RsvpP2mpSubLsp )

class RsvpP2mpLsp( Model ):
   lspRole = Enum( help="Role for LSP",
                   values=( 'ingress', 'egress', 'transit', 'bud', 'unknown' ),
                   default='unknown' )
   state = Enum( help="LSP operational state", values=( 'up', 'down' ),
                 default='down' )
   lspCreationTime = Float( help="UTC time when LSP was created" )
   subGroups = Dict( help="Table of sub-group information keyed by group ID",
                     keyType=int, valueType=RsvpP2mpSubGroup )

class RsvpP2mpSession( Model ):
   sessionName = Str( help="Session name", optional=True )
   p2mpId = Int( help="P2MP ID" )
   tunnelId = Int( help="Tunnel ID" )
   extendedTunnelId = IpGenericAddress( help="Extended tunnel ID" )
   state = Enum( help="Session operational state", values=( 'up', 'down' ),
                 default='down' )
   sessionCreationTime = Float( help="UTC time when session was created" )
   lsps = Dict( help="Table of LSP information keyed by LSP ID",
                keyType=int, valueType=RsvpP2mpLsp )
   pmsiIntfId = Interface( help="Egress PMSI interface", optional=True )

class RsvpSessions( DeferredModel ):
   '''Model for "show mpls rsvp session" and
      "show mpls rsvp session summary"
      "show mpls rsvp session history"

   Contains entries for both P2P and P2MP sessions.
   '''
   __revision__ = 2
   sessions = Dict( help="Table of P2P session information keyed by session number",
                    keyType=int,
                    valueType=RsvpSession )
   p2mpSessions = Dict(
      help="Table of P2MP session information keyed by session ID string",
      keyType=str,
      valueType=RsvpP2mpSession )

class RsvpMessageCounter( Model ):
   counts = Dict( keyType=str, valueType=int,
                  help="Message counts keyed by RsvpMessageType("
                       "Path, PathTear, PathErr, Resv, ResvTear"
                       "Srefresh, Other, Errors )" )

   def countInc( self, mType, ct ):
      currentCt = self.counts.get( mType, 0 )
      self.counts[ mType ] = currentCt + ct

   def renderRow( self ):
      orderedMessageTypes = [ "Path", "PathTear", "PathErr",
                             "Resv", "ResvTear", "ResvErr",
                             "Srefresh", "Other", "Errors" ]
      return [ self.counts.get( mType, 0 ) for mType in orderedMessageTypes ]


class RsvpInterfaceCounter( Model ):
   rx = Submodel( help="Received message counts",
                  valueType=RsvpMessageCounter )
   tx = Submodel( help="Sent message counts",
                  valueType=RsvpMessageCounter )

class RsvpErrorCounter( Model ):
   rx = Int( help="Received error counts" )
   tx = Int( help="Sent error counts" )

def printTable( table ):
   print( table )

def formatTable( table ):
   table.border = False
   table.set_style( prettytable.PLAIN_COLUMNS )
   table.right_padding_width = 2
   for col in table.field_names:
      table.align[ col ] = 'l'

def bandwithToPercentage( totalReserved, totalBandwidth ):
   if totalBandwidth > 0.0:
      use = totalReserved / totalBandwidth * 100.0
      percentageUse = str( round( use, 1 ) )
   else:
      percentageUse = "N/A"
   return percentageUse

def sortIntf( intfs ):
   '''Sort interfaces, but the default IntfId goes last.
   '''
   intfs = Arnet.sortIntf( intfs )
   if intfs and not intfs[ 0 ]:
      intfs = intfs[ 1: ] + intfs[ :1 ]
   return intfs

class RsvpMessageCounters( Model ):
   '''Model for "show mpls rsvp counters" '''
   interfaces = Dict( help="Interface message counters",
                      keyType=Interface,
                      valueType=RsvpInterfaceCounter )
   errorCounts = Dict( help="Error counters",
                       keyType=str, valueType=RsvpErrorCounter )

   def intfModel( self, intfId ):
      model = self.interfaces.get( intfId )
      if model is None:
         model = RsvpInterfaceCounter()
         model.rx = RsvpMessageCounter()
         model.tx = RsvpMessageCounter()
         self.interfaces[ intfId ] = model

      return model

   def errorCountsModel( self, key ):
      model = self.errorCounts.get( key )
      if model is None:
         model = RsvpErrorCounter()
         model.rx = 0
         model.tx = 0
         self.errorCounts[ key ] = model
      return model

   def render( self ):
      # Interface counters
      headers = [ "Interface", "Path", "PathTear", "PathErr",
                  "Resv", "ResvTear", "ResvErr", "Srefresh",
                  "Other", "Errors" ]
      separator = [ "-" * len( col ) for col in headers ]

      rxTable = prettytable.PrettyTable( headers )
      rxTable.add_row( separator )
      txTable = prettytable.PrettyTable( headers )
      txTable.add_row( separator )

      for intf in sortIntf( self.interfaces ):
         counters = self.interfaces[ intf ]
         for table, model in [ ( rxTable, counters.rx ),
                               ( txTable, counters.tx ) ]:
            row = [ intf if intf else "Unknown" ]
            row.extend( model.renderRow() )
            table.add_row( row )

      print( "Received Messages:\n" )
      formatTable( rxTable )
      for header in headers:
         if header in [ "Interface" ]:
            continue
         rxTable.align[ header ] = 'r'
      printTable( rxTable )

      print( "\nSent Messages:\n" )
      formatTable( txTable )
      for header in headers:
         if header in [ "Interface" ]:
            continue
         txTable.align[ header ] = 'r'
      printTable( txTable )

      # Error counters
      from collections import OrderedDict # pylint: disable=import-outside-toplevel
      displayNames = OrderedDict( [
         # json value: text display value
         ( 'unknownError', 'Unknown error' ),
         ( 'rsvpDisabledError', 'RSVP disabled on intf' ),
         ( 'mplsDisabledError', 'MPLS disabled on intf' ),
         ( 'intfError', 'Interface error' ),
         ( 'arpError', 'ARP error' ),
         ( 'invalidPktFormatError', 'Invalid packet format' ),
         ( 'pktParserGenericFailureError', 'Packet parser error' ),
         ( 'pktParserBadMsgChecksumError', 'Message checksum error' ),
         ( 'pktParserIntegrityFailureError', 'Message integrity error' ),
         ( 'pktParserSubLspEroDecompressionError', 'P2MP ERO decompression error' ),
         ( 'pktParserSubLspRroDecompressionError', 'P2MP RRO decompression error' ),
         ( 'pktGeneratorGenericFailureError', 'Packet generator error' ),
         ( 'pktGeneratorAuthIndexNotSetError', 'Auth index missing' ),
         ( 'pktGeneratorNoSecretForAuthIndexError', 'No secret for auth index' ),
         ( 'pktGeneratorRroDroppedError', 'RRO dropped' ),
         ( 'pktGeneratorPktTooBigForMtuError', 'Packet too big for MTU' ),
         ( 'unknownNeighborError', 'Unknown neighbor' ),
         ( 'invalidSeqNoError', 'Invalid sequence number' ),
         ( 'pktParserUnknownObjClassError', 'Unknown object class error' ),
         ( 'pktParserUnknownObjCtypeError', 'Unknown object c-type error' ),
      ] )
      headers = [ "Error", "On receive", "On send" ]
      separator = [ "-" * len( col ) for col in headers ]

      errorTable = prettytable.PrettyTable( headers )
      errorTable.add_row( separator )
      for errorJsonName, errorDisplayName in displayNames.items():
         errorCounter = self.errorCounts.get( errorJsonName )
         if errorCounter is None:
            continue
         row = [ errorDisplayName, errorCounter.rx, errorCounter.tx ]
         errorTable.add_row( row )

      print( "\nError counters during message processing:\n" )
      formatTable( errorTable )
      for header in headers:
         if header in [ "Error" ]:
            continue
         errorTable.align[ header ] = 'r'
      printTable( errorTable )

class RsvpBandwidthIntfModel( Model ):
   reservedBandwidth = Dict(
         help="A map of priority to bandwidth reserved in bits per second",
         keyType=int,
         valueType=float )
   totalBandwidth = Float(
         help="Total bandwidth available to RSVP in bits per second" )

class RsvpBandwidthIntfSummaryModel( Model ):
   reservedBandwidth = Float( help="Bandwidth reserved in bits per second" )
   totalBandwidth = Float(
         help="Total bandwidth available to RSVP in bits per second" )

class RsvpBandwidthSummaryModel( Model ):
   interfaces = Dict( help="A mapping of interface to bandwidth summary",
                      keyType=Interface, valueType=RsvpBandwidthIntfSummaryModel )

   def render( self ):
      # Setup Table Headers
      header1 = [ "Interface",
                  "Reserved bandwidth",
                  "Percentage used",
                  "Total bandwidth" ]
      header2 = [ "         ",
                  "              Mbps",
                  "              %",
                  "           Mbps" ]
      separator = [ "-" * ( len( col ) + 1 ) for col in header2 ]

      table = prettytable.PrettyTable( header1 )
      table.field_names = header1
      table.add_row( header2 )
      table.add_row( separator )

      # Align Numeric values to the right
      formatTable( table )
      table.align[ "Reserved bandwidth" ] = 'r'
      table.align[ "Percentage used" ] = 'r'
      table.align[ "Total bandwidth" ] = 'r'

      # Add rows
      for intfId in sortIntf( self.interfaces ):
         intfModel = self.interfaces[ intfId ]
         reservedBandwidthMbps = roundBw( intfModel.reservedBandwidth / 1e6, 1 )
         percentageUse = bandwithToPercentage( intfModel.reservedBandwidth,
                                               intfModel.totalBandwidth )
         totalBandwidthMbps = roundBw( intfModel.totalBandwidth / 1e6, 1 )
         table.add_row( [ intfId,
                          reservedBandwidthMbps,
                          percentageUse,
                          totalBandwidthMbps ] )
      print( table )

class RsvpBandwidthModel( Model ):
   interfaces = Dict( help="Interface's Bandwidth information",
                      keyType=Interface, valueType=RsvpBandwidthIntfModel )

   def render( self ):
      header1 = [ "Interface", "Priority", "Bandwidth", "" ]
      header2 = [ "         ", "        ", "Mbps     ", "  % " ]
      separator = [ "-" * ( len( col ) + 1 ) for col in header2 ]

      table = prettytable.PrettyTable( header1 )
      table.add_row( header2 )
      table.add_row( separator )

      # Align Numeric values to the right
      formatTable( table )
      table.align[ "Bandwidth" ] = 'r'
      table.align[ "Priority" ] = 'r'
      table.align[ "" ] = 'r'

      # Populate Table
      # Add rows
      for intfId in sortIntf( self.interfaces ):
         intfModel = self.interfaces[ intfId ]
         totalReserved = 0.0
         for priority in range( BwPriority.min, BwPriority.max + 1 ):
            bw = intfModel.reservedBandwidth.get( priority, 0.0 )
            reservedBandwidthMbps = roundBw( bw / 1e6, 1 )
            totalReserved += bw
            percentageUse = bandwithToPercentage( bw, intfModel.totalBandwidth )

            if priority == 0:
               row = [ intfId ]
            else:
               row = [ "" ]

            row.extend( [ priority,
                          reservedBandwidthMbps,
                          percentageUse ] )
            table.add_row( row )
         # Add total row
         totalReservedMbps = roundBw( totalReserved / 1e6, 1 )
         percentageUse = bandwithToPercentage( totalReserved,
                                               intfModel.totalBandwidth )
         table.add_row( [ "",
                          "Total",
                          totalReservedMbps,
                          percentageUse ] )
         table.add_row( [ "" for _ in header1 ] )
      print( table )
