#!/usr/bin/env python3
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Toggles.RcfLibToggleLib import toggleRcfRtMembershipRouteTargetExtraEnabled

import BasicCli
from CliPlugin.ArBgpCli import (
   CommonTokens,
   ShowBgpDebugPolicyBase,
)
from CliPlugin.RouteMapCli import RtSooExtCommCliMatcher
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression
from CliPlugin.RoutingBgpShowCli import (
   summaryVrfModel,
)
import CliMatcher
import CliParser
import CliParserCommon
import CliToken.RoutingBgp as bgpTokens
from CliToken.RoutingBgpShowCliTokens import (
      bgpAfterShow,
      detail,
      neighbors,
      routeTypeMatcher,
      summary,
)
import ShowCommand

#-------------------------------------------------------------------------------
# show bgp rt-membership summary
#-------------------------------------------------------------------------------

rtMembershipAfterShowBgp = CliMatcher.KeywordMatcher( "rt-membership",
      helpdesc="Route Target Membership address family" )

class ShowBgpRtMembershipSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp rt-membership summary'
   data = {
         "bgp" : bgpAfterShow,
         "rt-membership" : rtMembershipAfterShowBgp,
         "summary" : summary,
   }
   cliModel = summaryVrfModel
   handler = "RtMembershipCliHandler.ShowBgpRtMembershipSummary_handler"

BasicCli.addShowCommandClass( ShowBgpRtMembershipSummary )

#-------------------------------------------------------------------------------
# show bgp rt-membership floodlist [<rt>]
#-------------------------------------------------------------------------------

routeTargetFloodlist = CliMatcher.KeywordMatcher( 'floodlist',
                                                  'Route Target floodlist' )

class ShowRtMembershipFloodlist( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp rt-membership floodlist [ ROUTE_TARGET ]'
   data = {
      'bgp' : CommonTokens.showBgp,
      'rt-membership' : CommonTokens.rtMembership,
      'floodlist' : routeTargetFloodlist,
      'ROUTE_TARGET' : RtSooExtCommCliMatcher( "Route Target" ),
   }

   cliModel = "RtMembershipCliModels.ShowRtMembershipFloodlistModel"
   handler = "RtMembershipCliHandler.ShowRtMembershipFloodlist_handler"

BasicCli.addShowCommandClass( ShowRtMembershipFloodlist )

#-------------------------------------------------------------------------------
# show bgp neighbors <nbr> rt-membership filter
#-------------------------------------------------------------------------------

routeTargetFilter = CliMatcher.KeywordMatcher( 'filter',
                                               'Route Target filter' )

class ShowNeighborRtMembershipFilter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp neighbors PEER rt-membership filter'
   data = {
      'bgp' : CommonTokens.showBgp,
      'neighbors' : CommonTokens.neighbors,
      'PEER': V4V6PeerKeyCliExpression,
      'rt-membership' : CommonTokens.rtMembership,
      'filter' : routeTargetFilter,
   }

   cliModel = "RtMembershipCliModels.ShowRtMembershipNeighborFilterModel"
   handler = "RtMembershipCliHandler.ShowNeighborRtMembershipFilter_handler"

BasicCli.addShowCommandClass( ShowNeighborRtMembershipFilter )

#------------------------------------------------------------------------------
#  show bgp rt-membership [ detail ]
#------------------------------------------------------------------------------

class ShowBgpRtMembership( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp rt-membership [ detail ]"
   data = {
         "bgp" : bgpAfterShow,
         "rt-membership" : rtMembershipAfterShowBgp,
         "detail" : detail,
   }
   cliModel = "RtMembershipCliModels.rtMembershipRoutesVrfModel"
   handler = "RtMembershipCliHandler.ShowBgpRtMembership_handler"

BasicCli.addShowCommandClass( ShowBgpRtMembership )

#------------------------------------------------------------------------------
#  show bgp rt-membership route-target <rt> [ detail ]
#------------------------------------------------------------------------------

class RtPrefixCliMatcher( RtSooExtCommCliMatcher ):
   # Route Target prefix matcher used only for RT membership constrain.
   # It will match any Route Target with or without the length specified as well as
   # the RT membership default route (0/0).
   def __init__( self, helpdesc, **kargs ):
      super().__init__( helpdesc=helpdesc, **kargs )
      self.defaultRt = '0/0'
      self.completion_ = CliParser.Completion(
         'ASN:nn[/0-64] or IP-Address:nn[/0-64] or 0xnnnnnnnnnnnnnnnn[/0-64]',
         helpdesc, False )

   def match( self, mode, context, token ):
      if token == self.defaultRt:
         return CliParserCommon.MatchResult( token.split( '/' ), token.split( '/' ) )
      length = None
      part = token.split( '/' )

      # If rt prefix has length specified, split the token to handle only the rt
      # with parent class.
      if len( part ) > 1 and part[ 1 ]:
         token = part[ 0 ]
         length = part[ 1 ]
         try:
            if int( length ) < 0 or int( length ) > 64:
               return CliParserCommon.noMatch
         except ValueError:
            return CliParserCommon.noMatch

      if len( part ) > 2:
         return CliParserCommon.noMatch

      res = None
      if token.startswith( '0x' ):
         if len( token ) != 18:
            return CliParserCommon.noMatch
         # Check if it's a valid hexa number.
         try:
            int( token, 16 )
         except ValueError:
            return CliParserCommon.noMatch
      else:
         res = super().match( mode, context, token )

      if res == CliParserCommon.noMatch:
         return CliParserCommon.noMatch
      if length:
         return CliParserCommon.MatchResult( [ token, length ], [ token, length ] )
      return CliParserCommon.MatchResult( [ token ], [ token ] )

routeTargetMatcher = CliMatcher.KeywordMatcher( 'route-target',
                                 'Match for Route Target or Route Target prefix' )

class ShowBgpPrefixRtMembership( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp rt-membership route-target ROUTE_TARGET [ detail ]"
   data = {
         "bgp" : bgpAfterShow,
         "rt-membership" : rtMembershipAfterShowBgp,
         "route-target" : routeTargetMatcher,
         "ROUTE_TARGET" : RtPrefixCliMatcher( "Route Target prefix" ),
         "detail" : detail,
   }
   cliModel = "RtMembershipCliModels.rtMembershipRoutesVrfModel"
   handler = "RtMembershipCliHandler.ShowBgpPrefixRtMembership_handler"

BasicCli.addShowCommandClass( ShowBgpPrefixRtMembership )

#------------------------------------------------------------------------------
#  show bgp rt-membership origin-as <asn> [ detail ]
#------------------------------------------------------------------------------

originAs = CliMatcher.KeywordMatcher( 'origin-as', 'Match for autonomous-system' )

class ShowBgpOriginAsRtMembership( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp rt-membership origin-as ASN [ detail ]"
   data = {
         "bgp" : bgpAfterShow,
         "rt-membership" : rtMembershipAfterShowBgp,
         "origin-as" : originAs,
         "ASN": bgpTokens.AsNumOptCliExpr,
         "detail" : detail,
   }
   cliModel = "RtMembershipCliModels.rtMembershipRoutesVrfModel"
   handler = "RtMembershipCliHandler.ShowBgpOriginAsRtMembership_handler"

BasicCli.addShowCommandClass( ShowBgpOriginAsRtMembership )

#------------------------------------------------------------------------------
#  show bgp neighbors <ip> rt-membership [ advertised-routes |
#  received-routes | routes ]
#------------------------------------------------------------------------------

class ShowBgpNeighborsRtMembership( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp neighbors PEER_ADDR rt-membership ROUTE_TYPE"
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "rt-membership" : rtMembershipAfterShowBgp,
         "ROUTE_TYPE" : routeTypeMatcher,
   }
   cliModel = "RtMembershipCliModels.rtMembershipRoutesVrfModel"
   handler = "RtMembershipCliHandler.ShowBgpNeighborsRtMembership_handler"

BasicCli.addShowCommandClass( ShowBgpNeighborsRtMembership )

class ShowBgpDebugPolicyRtMembershipBase( ShowBgpDebugPolicyBase ):
   data = ShowBgpDebugPolicyBase.data.copy()
   data.update( {
      "rt-membership": rtMembershipAfterShowBgp,
      "origin-as": originAs,
      "ASN": bgpTokens.AsNumOptCliExpr,
      "route-target": routeTargetMatcher,
      "ROUTE_TARGET": RtPrefixCliMatcher( "Route Target prefix" ),
   } )
   handler = "RtMembershipCliHandler.policyDebugRtMembershipHandler"

class ShowBgpDebugPolicyRtMembershipInbound( ShowBgpDebugPolicyRtMembershipBase ):
   syntax = 'show bgp debug policy ' \
            'inbound neighbor ( ADDR | ADDR6 | all ) ' \
            'rt-membership [ rcf RCF_FUNC_NAME ] ' \
            'origin-as ASN route-target ROUTE_TARGET'
   data = ShowBgpDebugPolicyRtMembershipBase.data.copy()

if toggleRcfRtMembershipRouteTargetExtraEnabled():
   BasicCli.addShowCommandClass( ShowBgpDebugPolicyRtMembershipInbound )

class ShowBgpDebugPolicyRtMembershipOutbound( ShowBgpDebugPolicyRtMembershipBase ):
   syntax = 'show bgp debug policy ' \
            'outbound neighbor ( ADDR | ADDR6 ) ' \
            'rt-membership [ rcf RCF_FUNC_NAME ] ' \
            'origin-as ASN route-target ROUTE_TARGET'
   data = ShowBgpDebugPolicyRtMembershipBase.data.copy()

if toggleRcfRtMembershipRouteTargetExtraEnabled():
   BasicCli.addShowCommandClass( ShowBgpDebugPolicyRtMembershipOutbound )
