# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
# The server probe package's CLI plugin
import BasicCliModes
import CliCommand
import CliMatcher
import CliToken.Monitor
import CliToken.Clear
import LazyMount # pylint: disable=E0401
import BasicCli
import ShowCommand
import Tac
from TypeFuture import TacLazyType
from CliMode.ServerProbe import MonitorServerBaseMode
from CliPlugin import ServerProbeModel, TechSupportCli
import ReversibleSecretCli

config = None
status = None
counterConfig = None
counterStatus = None
serverProbeService = TacLazyType( 'ServerProbe::ServerProbeService' )

maxProbePeriod = 1000
matcherProbe = CliMatcher.KeywordMatcher( 'probe',
      helpdesc='Configuration for probes sent to the server' )
matcherInterval = CliMatcher.IntegerMatcher( 1, maxProbePeriod,
      helpdesc='Server probe period in seconds' )
maxFailCount = ( 2**8 ) - 1
matcherProbeCount = CliMatcher.IntegerMatcher( 1, maxFailCount,
      helpdesc='Number of consecutive failed probes before a server is'
            ' marked as dead' )
matcherProbeState = CliMatcher.EnumMatcher( {
   'failure': 'Number of consecutive failed probes to mark a server as dead',
   } )
matcherPassword = ReversibleSecretCli.defaultReversiblePwdCliExpr
matcherUsername = CliMatcher.PatternMatcher( pattern='.+',
                     helpname='WORD',
                     helpdesc='Username to be used in access-request probes' )

# This enum MUST be the same as the one in ServerProbe::Protocol
# found in /src/Aaa/ServerProbeTypes.tac
matcherProtocol = CliMatcher.EnumMatcher( {
   'radius': 'Configurations to monitor RADIUS servers',
   } )

# Class for monitor-server mode
class MonitorServerMode( MonitorServerBaseMode, BasicCli.ConfigModeBase ):
   name = "Configuration commands to monitor configured servers"

   def __init__( self, parent, session, protocol ):
      self.protocol = protocol
      MonitorServerBaseMode.__init__( self, protocol )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def getProtocol( self ):
      # if self.protocol is None then this function is being called
      # in the wrong place
      return Tac.enumValue( "ServerProbe::Protocol", self.protocol )

# ------------------------------------------------------------
# switch(config)# monitor server <PROTOCOL_NAME>
# ------------------------------------------------------------
class MonitorServerCommand( CliCommand.CliCommandClass ):
   syntax = "monitor server PROTOCOL"
   noOrDefaultSyntax = syntax
   data = {
         'monitor': CliToken.Monitor.monitorMatcher,
         'server': 'Monitor server reachability',
         'PROTOCOL': matcherProtocol,
         }

   handler = "ServerProbeCliHandler.monitorServerModeCmd_handler"
   noOrDefaultHandler = "ServerProbeCliHandler." + \
                        "monitorServerModeCmd_noOrDefaultHandler"

BasicCliModes.GlobalConfigMode.addCommandClass( MonitorServerCommand )

# ------------------------------------------------------------
# switch(config-monitor-server-<PROTOCOL_NAME>)# service dot1x
# ------------------------------------------------------------
class Dot1xServerProbeCommand( CliCommand.CliCommandClass ):
   syntax = 'service dot1x'
   noOrDefaultSyntax = syntax
   data = {
         'service': 'Monitor servers for specific services',
         'dot1x': 'Monitor servers used for 802.1X authentication',
         }

   handler = "ServerProbeCliHandler.serviceDot1xCmd_defaultHandler"
   noOrDefaultHandler = "ServerProbeCliHandler.serviceDot1xCmd_noOrDefaultHandler"

MonitorServerMode.addCommandClass( Dot1xServerProbeCommand )

# ------------------------------------------------------------
# switch(config-monitor-server-<PROTOCOL_NAME>)# probe interval <INT> seconds
# ------------------------------------------------------------
class ProbeIntervalCommand( CliCommand.CliCommandClass ):
   syntax = 'probe interval INTERVAL seconds'
   noOrDefaultSyntax = 'probe interval ...'
   data = {
         'probe': matcherProbe,
         'interval': 'Configure probe interval for servers',
         'INTERVAL': matcherInterval,
         'seconds': 'Unit in seconds',
         }

   handler = "ServerProbeCliHandler.probeIntervalCmd_defaultHandler"
   noOrDefaultHandler = "ServerProbeCliHandler.probeIntervalCmd_noOrDefaultHandler"

MonitorServerMode.addCommandClass( ProbeIntervalCommand )

# ------------------------------------------------------------
#  switch(config-monitor-server-<PROTOCOL_NAME>)# probe threshold failure <1-256>
# ------------------------------------------------------------
class ProbeCounterCommand( CliCommand.CliCommandClass ):
   syntax = 'probe threshold PROBE_STATE COUNT'
   noOrDefaultSyntax = 'probe threshold PROBE_STATE ...'
   data = {
         'probe': matcherProbe,
         'threshold': 'Threshold to classify a server as dead or alive',
         'PROBE_STATE': matcherProbeState,
         'COUNT': matcherProbeCount,
         }

   handler = "ServerProbeCliHandler.probeFailureThresholdCmd_handler"
   noOrDefaultHandler = "ServerProbeCliHandler." + \
                        "probeFailureThresholdCmd_noOrDefaultHandler"

MonitorServerMode.addCommandClass( ProbeCounterCommand )

# ------------------------------------------------------------
#  switch(config-monitor-server-<PROTOCOL_NAME>)# probe method ...
# ------------------------------------------------------------
# TODO : If another protocol is introduced in serverProbe, this command needs
#        to be made explicit only when the protocol is RADIUS as 'status-server'
#        and 'access-request' as probe methods are not applicable to other protocols

class ProbeMethodCommand( CliCommand.CliCommandClass ):
   syntax = '''probe method ( status-server | \
         ( access-request username USERNAME password PASSWORD ) )'''
   noOrDefaultSyntax = 'probe method ...'

   data = {
         'probe': matcherProbe,
         'method': 'Configure the method used to probe the server',
         'status-server': 'Use RADIUS Status-Server packets as probes (default)',
         'access-request': 'Use RADIUS Access-Request packets as probes',
         'username':
         'Configure the username to be used in the access-request packets',
         'USERNAME': matcherUsername,
         'password':
         'Configure the password to be used in the access-request packets',
         'PASSWORD': matcherPassword,
         }

   handler = "ServerProbeCliHandler.probeMethodCmd_handler"
   noOrDefaultHandler = "ServerProbeCliHandler.probeMethodCmd_noOrDefaultHandler"

MonitorServerMode.addCommandClass( ProbeMethodCommand )

# ------------------------------------------------------------
#  switch> show monitor server <PROTOCOL> [detail]
# ------------------------------------------------------------
class ShowServerProbeCommand( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor server PROTOCOL [ detail ]'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'server': 'Monitor server reachability',
         'PROTOCOL': matcherProtocol,
         'detail': 'Display information in detail',
         }
   cliModel = ServerProbeModel.ShowServerProbe

   handler = "ServerProbeCliHandler.showMonitorServerCmd_handler"

BasicCli.addShowCommandClass( ShowServerProbeCommand )

# -------------------------------------------------------------------------------
# switch# clear monitor server <PROTOCOL> counters
# -------------------------------------------------------------------------------

class ClearProbeCounterCommand( CliCommand.CliCommandClass ):
   syntax = 'clear monitor server PROTOCOL counters'
   data = {
         'clear': CliToken.Clear.clearKwNode,
         'monitor': CliToken.Monitor.monitorMatcherForClear,
         'server': 'Monitor server reachability',
         'PROTOCOL': matcherProtocol,
         'counters': 'Server probe packet counters'
         }

   handler = "ServerProbeCliHandler.clearMonitorServerCountersCmd_handler"

BasicCli.EnableMode.addCommandClass( ClearProbeCounterCommand )

# -------------------------------------------------------------------------------
# Register commands to show tech-support
# -------------------------------------------------------------------------------

TechSupportCli.registerShowTechSupportCmd(
      '2023-04-27 16:15:30',
      cmds=[ 'show monitor server radius detail' ],
      cmdsGuard=lambda: status and "dot1x" in status.servicesEnabled
)

def Plugin( entityManager ):
   global config, status, counterStatus, counterConfig

   config = LazyMount.mount( entityManager, "serverProbe/config",
         "ServerProbe::Config", "w" )
   status = LazyMount.mount( entityManager, "serverProbe/allServerStatus",
         "ServerProbe::ServerProbeStatus", "r" )
   counterStatus = LazyMount.mount( entityManager, "serverProbe/counter/status",
         "ServerProbe::ServerProbeCounterStatus", "r" )
   counterConfig = LazyMount.mount( entityManager, "serverProbe/counter/config",
         "ServerProbe::ServerProbeCounterConfig", "w" )
