# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import List
from CliModel import Bool
from CliModel import Float
from CliModel import Submodel
from CliModel import Enum
import Tac
import Ark
import TableOutput

ProtocolNameMap = {
      "radius": "RADIUS",
}

ProbeMethodMap = {
      "statusServerPacket": "Status-Server packet",
      "accessRequestPacket": "Access-Request packet",
}

class ServerInfo( Model ):
   host = Str( help="IP address or hostname of the probed server" )
   port = Int( help="Server port number" )
   acctPort = Int( help="Accounting port number", optional=True )
   vrf = Str( help="The VRF over which this server is reachable", optional=True )
   alive = Bool( help="Server is alive" )
   lastProbeTime = Float( help="UTC time when the server was last probed" )
   lastResponseTime = Float(
         help="UTC time when receiving the last successful response " )
   lastStatusChangeTime = Float(
         help="UTC time when the server status was last changed" )
   failCount = Int( help="Failed probes since last successful response" )

class ServerProbeStats( Model ):
   serverInfo = Submodel( valueType=ServerInfo, help="Probed server information" )
   probesSent = Int( help="Number of probe packets sent to the server" )
   responseAccepted = Int( help="Number of valid responses accepted" )
   responseDropped = Int( help="Number of responses dropped" )
   lastClearTime = Float( help="UTC time when the counters were last cleared" )

class ShowServerProbe( Model ):
   probedServers = List( valueType=ServerProbeStats,
         help="List of all probed servers" )
   _detail = Bool(
         help="Whether to show probe information in detail or as a summary" )
   protocol = Str( help="Protocol of the probed server" )
   probeMethod = Enum( values=ProbeMethodMap,
                       help="Server probe method",
                       optional=True )

   def detailIs( self, detail ):
      self._detail = detail

   def _getVrfStr( self, vrf ):
      return " (VRF %s)" % vrf if vrf else ""

   def _getPortStr( self, port, acctPort ):
      if acctPort:
         portStr = "Authentication port: %d, Accounting port: %d" % ( port,
                                                                      acctPort )
      else:
         portStr = "Port: %d" % port
      return portStr

   def renderDetail( self ):
      for server in self.probedServers:
         info = server.serverInfo
         serverInfo = ( "\n{protocol} server: {host}{vrf}, "
                        "{port}\n"
                        "Status: {alive}, Last changed: {lastChange}\n"
                        "Last probed: {lastProbeTime}\n"
                        "Last successful probe: {lastResponse}, "
                        "Failed probes since then: {failCount}"
                      ).format(
                           protocol=ProtocolNameMap[ self.protocol ],
                           host=info.host,
                           vrf=self._getVrfStr( info.vrf ),
                           port=self._getPortStr( info.port, info.acctPort ),
                           alive="alive" if info.alive else "not alive",
                           lastChange=Ark.utcTimeRelativeToNowStr(
                              info.lastStatusChangeTime ),
                           lastProbeTime=Ark.utcTimeRelativeToNowStr(
                              info.lastProbeTime ),
                           lastResponse=Ark.utcTimeRelativeToNowStr(
                              info.lastResponseTime ),
                           failCount=info.failCount )
         print( serverInfo )
         print( "Messages sent:", server.probesSent )
         print( "Messages accepted:", server.responseAccepted )
         print( "Messages dropped:", server.responseDropped )
         print( "Last time counters were cleared: %s" %
                Ark.utcTimeRelativeToNowStr( server.lastClearTime ) )

   def renderSummary( self ):
      table = TableOutput.createTable(
            [ "%s Server" % ProtocolNameMap[ self.protocol ],
              "Messages Sent",
              "Messages Accepted",
              "Messages Dropped",
              "Status" ],
            tableWidth=120 )
      leftFormat = TableOutput.Format( justify='left' )
      leftFormat.padLimitIs( True )
      rightFormat = TableOutput.Format( justify='right' )
      rightFormat.padLimitIs( True )
      formatServer = leftFormat
      formatPktSent = rightFormat
      formatRespRecv = rightFormat
      formatRespDrop = rightFormat
      formatStatus = leftFormat
      table.formatColumns( formatServer, formatPktSent,
                           formatRespRecv, formatRespDrop,
                           formatStatus )
      for server in self.probedServers:
         hostStr = server.serverInfo.host + self._getVrfStr( server.serverInfo.vrf )
         statusStr = "alive" if server.serverInfo.alive else "not alive"
         table.newRow( hostStr, server.probesSent, server.responseAccepted,
                       server.responseDropped, statusStr )
      print( table.output() )

   def render( self ):
      # We will add more probe methods later
      if self.probeMethod:
         print( "Probe method: %s" % ProbeMethodMap[ self.probeMethod ] )
         if self._detail:
            self.renderDetail()
         else:
            self.renderSummary()
      else:
         print( "Probe method: Disabled" )
