# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import ShowCommand
import Toggles.SfeToggleLib as Toggle

from CliToken.Agent import (
   agentKwForShow,
   agentKwForConfig
)
from SfeUtils import FlowCacheWalkerThreadConst

# ------------------------------------------------------------------------------
# (config)# agent sfe threads flow cache scan INPUT_THREADS
# ------------------------------------------------------------------------------
sfeMatcher = CliMatcher.KeywordMatcher( 'sfe', 'Software Forwarding Engine' )
threadsMatcher = CliMatcher.KeywordMatcher( 'threads', 'threading model' )
flowMatcher = CliMatcher.KeywordMatcher( 'flow', 'Flow configuration' )
cacheMatcher = CliMatcher.KeywordMatcher( 'cache', 'Flow cache table' )
scanMatcher = CliMatcher.KeywordMatcher( 'scan', 'Scan through flow cache table' )
inputThreadsMatcher = CliMatcher.IntegerMatcher( FlowCacheWalkerThreadConst.min,
                           FlowCacheWalkerThreadConst.max,
                           helpdesc='Number of walker threads' )

class configThreadCmd( CliCommand.CliCommandClass ):
   syntax = 'agent sfe threads flow cache scan INPUT_THREADS'
   noOrDefaultSyntax = 'agent sfe threads flow cache scan ...'
   data = {
      'agent' : agentKwForConfig,
      'sfe' : sfeMatcher,
      'threads' : threadsMatcher,
      'flow' : flowMatcher,
      'cache' : cacheMatcher,
      'scan' : scanMatcher,
      'INPUT_THREADS' : inputThreadsMatcher,
   }
   hidden = False
   handler = 'SfeFlowCacheWalkerHandler.handlerConfig'
   noOrDefaultHandler = 'SfeFlowCacheWalkerHandler.handlerConfigDefault'

if Toggle.toggleSfeFctWalkerEnabled():
   BasicCliModes.GlobalConfigMode.addCommandClass( configThreadCmd )

# ---------------------------------------------------------------------------------
# show agent sfe threads flow cache scan counters
# ---------------------------------------------------------------------------------
countersMatcher = CliMatcher.KeywordMatcher( 'counters', 'Walker debug counters' )

class ShowAgentSfeThreadsFlowCacheScanCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent sfe threads flow cache scan counters'
   data = {
      'agent' : agentKwForShow,
      'sfe' : sfeMatcher,
      'threads' : threadsMatcher,
      'flow' : flowMatcher,
      'cache' : cacheMatcher,
      'scan' : scanMatcher,
      'counters' : countersMatcher
   }

   handler = 'SfeFlowCacheWalkerHandler.handlerShowCounters'
   cliModel = 'SfeFlowCacheWalkerModel.ModelShowCounters'

if Toggle.toggleSfeFctWalkerEnabled():
   BasicCli.addShowCommandClass( ShowAgentSfeThreadsFlowCacheScanCounters )

# ---------------------------------------------------------------------------------
# show agent sfe threads flow cache scan counters detail [thread <N>]
# ---------------------------------------------------------------------------------
detailMatcher = CliMatcher.KeywordMatcher( 'detail', 'Walker debug counters detail' )
threadMatcher = CliMatcher.KeywordMatcher( 'thread', 'thread number' )
threadIdMatcher = CliMatcher.IntegerMatcher( FlowCacheWalkerThreadConst.min,
                           FlowCacheWalkerThreadConst.max,
                           helpdesc='Walker thread id' )

class ShowAgentSfeThreadsFlowCacheScanCountersDetail(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent sfe threads flow cache scan counters detail ' + \
            '[thread THREAD_ID]'
   data = {
      'agent' : agentKwForShow,
      'sfe' : sfeMatcher,
      'threads' : threadsMatcher,
      'flow' : flowMatcher,
      'cache' : cacheMatcher,
      'scan' : scanMatcher,
      'counters' : countersMatcher,
      'detail' : detailMatcher,
      'thread' : threadMatcher,
      'THREAD_ID' : threadIdMatcher,
   }

   handler = 'SfeFlowCacheWalkerHandler.handlerShowCountersDetail'
   cliModel = 'SfeFlowCacheWalkerModel.ModelShowCountersDetail'

if Toggle.toggleSfeFctWalkerEnabled():
   BasicCli.addShowCommandClass( ShowAgentSfeThreadsFlowCacheScanCountersDetail )

# ---------------------------------------------------------------------------------
# show agent sfe threads flow cache scan status [thread <N>]
# ---------------------------------------------------------------------------------
statusMatcher = CliMatcher.KeywordMatcher( 'status', 'Walker status' )

class ShowAgentSfeThreadsFlowCacheScanStatus(
      ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent sfe threads flow cache scan status [thread THREAD_ID]'
   data = {
      'agent' : agentKwForShow,
      'sfe' : sfeMatcher,
      'threads' : threadsMatcher,
      'flow' : flowMatcher,
      'cache' : cacheMatcher,
      'scan' : scanMatcher,
      'status' : statusMatcher,
      'thread' : threadMatcher,
      'THREAD_ID' : threadIdMatcher,
   }

   handler = 'SfeFlowCacheWalkerHandler.handlerShowStatus'
   cliModel = 'SfeFlowCacheWalkerModel.ModelShowStatus'

if Toggle.toggleSfeFctWalkerEnabled():
   BasicCli.addShowCommandClass( ShowAgentSfeThreadsFlowCacheScanStatus )
