# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac

from collections import namedtuple
from TypeFuture import TacLazyType

CounterId = TacLazyType( 'SfeModulesApi::CounterId' )
DfwOffloadCounterId = TacLazyType( 'SfeModulesApi::DfwOffloadCounterId' )
IpfixWalkerCounterId = TacLazyType( 'SfeFlowTracker::IpfixWalkerCounterId' )
PurgeWalkerCounterId = TacLazyType( 'SfeModulesApi::PurgeWalkerCounterId' )
WalkerClientType = TacLazyType( 'SfeModulesApi::WalkerClientType' )

maxThreads = Tac.Value( 'Sfe::FlowCacheWalkerThreads' ).max

CTR = namedtuple( 'Counter', [ 'cliText', 'counterId' ] )

class CounterPrefix:
   general = 'General'
   flowOffload = 'Flow offload'
   ipfix = 'IPFIX'

class WalkerCounterLib:
   def __init__( self ):
      self.general = {
         'purgedCount' : CTR( 'purged count', PurgeWalkerCounterId.purgedCount )
      }
      self.generalStatus = {
         'scanCount' : CTR( 'scan count', PurgeWalkerCounterId.walkCount )
      }
      self.ipfix = {
         'exportCount' : CTR( 'export count', IpfixWalkerCounterId.exportCount ),
         'failedExportCount' : CTR( 'failed export count',
                                IpfixWalkerCounterId.failExportCount )
      }
      self.flowOffload = {
         'flushCount' : CTR( 'flush count', DfwOffloadCounterId.flushCount )
      }
      self.allCounters = {
         CounterPrefix.general : self.general,
         CounterPrefix.flowOffload : self.flowOffload,
         CounterPrefix.ipfix : self.ipfix,
      }

      self.client = {
         CounterPrefix.general : None,
         CounterPrefix.flowOffload : WalkerClientType.walkerDfwOffload,
         CounterPrefix.ipfix : WalkerClientType.walkerDfwIpfix,
      }

   def getConterId( self, cPrefix, cName ):
      if cPrefix == CounterPrefix.general:
         if cName in self.generalStatus:
            return self.generalStatus[ cName ].counterId
      return self.allCounters[ cPrefix ][ cName ].counterId

   def getClient( self, cPrefix ):
      return self.client[ cPrefix ]

   def getCounter( self,
                  sharkTable, # SfeModulesApi::WalkerCounterShark
                  cPrefix,
                  cName ):
      if not sharkTable.walkerCounter:
         return None

      counterId = self.getConterId( cPrefix, cName )
      if cPrefix != CounterPrefix.general:
         clientName = self.getClient( cPrefix )
         if not sharkTable.walkerCounter.walkerClientCounter:
            return None

         client = sharkTable.walkerCounter.walkerClientCounter.get( clientName )
         if not client:
            return None
         counters = client.counter
      else:
         # purgeWalker counter
         counters = sharkTable.walkerCounter.counter

      counter = counters.get( counterId )
      if not counter:
         return None

      return counter
